/***************************************************************************

  stream.h

  The stream management routines

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#ifndef __GBX_STREAM_H
#define __GBX_STREAM_H

#include "gbx_value.h"
#include "gbx_archive.h"

union STREAM;

typedef
  struct STREAM_CLASS {
    int (*open)(union STREAM *stream, const char *path, int mode, void *data);
    int (*close)(union STREAM *stream);
    int (*read)(union STREAM *stream, char *buffer, long len);
    int (*write)(union STREAM *stream, char *buffer, long len);
    int (*seek)(union STREAM *stream, long pos, int whence);
    int (*tell)(union STREAM *stream, long *pos);
    int (*flush)(union STREAM *stream);
    int (*eof)(union STREAM *stream);
    int (*lof)(union STREAM *stream, long *len);
    int (*handle)(union STREAM *stream);
    }
  STREAM_CLASS;

typedef
  struct {
    STREAM_CLASS *type;
    short mode;
    struct {
      unsigned swap:1;
      unsigned _reserved:15;
      } flag;
    }
  STREAM_COMMON;


typedef
  struct {
    STREAM_COMMON common;
    long _reserved[6];
    }
  STREAM_RESERVED;

typedef
  struct {
    STREAM_COMMON common;
    long size;
    long fd;
    char car;
    boolean has_car;
    }
  STREAM_DIRECT;

typedef
  struct {
    STREAM_COMMON common;
    long size;
    FILE *file;
    }
  STREAM_BUFFER;

typedef
  struct {
    STREAM_COMMON common;
    long size;
    void *addr;
    long pos;
    }
  STREAM_MEMORY;

typedef
  struct {
    STREAM_COMMON common;
    ARCHIVE *arch;
    long size;
    long start;
    long pos;
    }
  STREAM_ARCH;

typedef
  struct {
    STREAM_COMMON common;
    long fdr;
    long fdw;
    char car;
    boolean has_car;
    }
  STREAM_PROCESS;


typedef
  union STREAM {
    STREAM_CLASS *type;
    STREAM_COMMON common;
    STREAM_RESERVED _reserved;
    STREAM_DIRECT direct;
    STREAM_BUFFER buffer;
    STREAM_MEMORY memory;
    STREAM_ARCH arch;
    STREAM_PROCESS process;
    }
  STREAM;

enum {
  ST_READ = 1,
  ST_WRITE = 2,
  ST_READ_WRITE = 3,
  ST_MODE = 3,
  ST_APPEND = 4,
  ST_CREATE = 8,
  ST_ACCESS = 15,
  ST_DIRECT = 16,
  ST_LINE = 32,
  ST_WATCH = 64,
  ST_BIG = 128,
  ST_LITTLE = 256
  };

EXTERN long STREAM_eff_read;

#ifndef __STREAM_IMPL_C

EXTERN STREAM_CLASS STREAM_direct;
EXTERN STREAM_CLASS STREAM_buffer;
EXTERN STREAM_CLASS STREAM_memory;
EXTERN STREAM_CLASS STREAM_arch;
EXTERN STREAM_CLASS STREAM_process;
/*EXTERN STREAM_CLASS STREAM_null;*/

#else

#define DECLARE_STREAM(stream) \
PUBLIC STREAM_CLASS stream = \
{ \
  (void *)stream_open, \
  (void *)stream_close, \
  (void *)stream_read, \
  (void *)stream_write, \
  (void *)stream_seek, \
  (void *)stream_tell, \
  (void *)stream_flush, \
  (void *)stream_eof, \
  (void *)stream_lof, \
  (void *)stream_handle \
}

#endif


PUBLIC void STREAM_exit(void);

PUBLIC void STREAM_open(STREAM *stream, const char *path, int mode);
PUBLIC void STREAM_close(STREAM *stream);
PUBLIC void STREAM_write(STREAM *stream, void *addr, long len);
PUBLIC void STREAM_line_input(STREAM *stream, char **addr);
PUBLIC void STREAM_input(STREAM *stream, char **addr);
PUBLIC long STREAM_tell(STREAM *stream);
PUBLIC void STREAM_seek(STREAM *stream, long pos, int whence);
PUBLIC void STREAM_read(STREAM *stream, void *addr, long len);
PUBLIC void STREAM_read_type(STREAM *stream, TYPE type, VALUE *value, long len);
PUBLIC void STREAM_write(STREAM *stream, void *addr, long len);
PUBLIC void STREAM_write_type(STREAM *stream, TYPE type, VALUE *value, long len);
PUBLIC void STREAM_flush(STREAM *stream);
PUBLIC int STREAM_handle(STREAM *stream);

PUBLIC void STREAM_load(const char *path, char **buffer, long *len);

#define STREAM_eof(_stream) ((*((_stream)->type->eof))(_stream))
#define STREAM_lof(_stream, _len) ((*((_stream)->type->lof))(_stream, _len))

#define STREAM_something_was_read(_stream) (STREAM_eff_read > 0)

#endif
