/***************************************************************************

  stream.c

  The stream management routines

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __GBX_STREAM_C

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>

#include "gb_common.h"
#include "gb_common_buffer.h"
#include "gb_common_swap.h"
#include "gb_error.h"
#include "gbx_value.h"
#include "gb_limit.h"
#include "gbx_exec.h"
#include "gbx_project.h"
#include "gb_file.h"
#include "gambas.h"
#include "gbx_regexp.h"

#include "gbx_string.h"

#include "gbx_stream.h"

#if 0
PRIVATE long do_nothing(void)
{
  return 0;
}

PUBLIC STREAM_CLASS STREAM_null =
{
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing,
  (void *)do_nothing
};
#endif

PUBLIC long STREAM_eff_read;

PUBLIC void STREAM_open(STREAM *stream, const char *path, int mode)
{
  if (FILE_is_relative(path))
  {
    if (EXEC_arch || !FILE_exist_real(path))
    {
      stream->type = &STREAM_arch;
      goto _OPEN;
    }
    
    if ((mode & ST_ACCESS) != ST_READ)
      THROW(E_ACCESS);
  }

  if (mode & ST_DIRECT)
    stream->type = &STREAM_direct;
  else
    stream->type = &STREAM_buffer;

  stream->common.mode = mode;

_OPEN:

  if (((mode & ST_BIG) && !EXEC_big_endian)
      || ((mode & ST_LITTLE) && EXEC_big_endian))
    stream->common.flag.swap = TRUE;

  if ((*(stream->type->open))(stream, path, mode, NULL))
    THROW_SYSTEM(errno, path);
}


PUBLIC void STREAM_close(STREAM *stream)
{
  if (stream->type == NULL)
    return;

  (*(stream->type->close))(stream);
  stream->type = NULL;
}


PUBLIC void STREAM_flush(STREAM *stream)
{
  (*(stream->type->flush))(stream);
}


PUBLIC void STREAM_read(STREAM *stream, void *addr, long len)
{
  if ((*(stream->type->read))(stream, addr, len))
  {
    switch(errno)
    {
      case 0:
      case EAGAIN:
        THROW(E_EOF);
      case EIO:
        THROW(E_READ);
      default:
        THROW_SYSTEM(errno, NULL);
    }
  }
}


PUBLIC void STREAM_read_max(STREAM *stream, void *addr, long len)
{
  if ((*(stream->type->read))(stream, addr, len))
  {
    switch(errno)
    {
      case 0:
      case EAGAIN:
        break;
      case EIO:
        THROW(E_READ);
      default:
        THROW_SYSTEM(errno, NULL);
    }
  }
}


PUBLIC void STREAM_write(STREAM *stream, void *addr, long len)
{
  if ((*(stream->type->write))(stream, addr, len))
  {
    switch(errno)
    {
      case EIO:
        THROW(E_WRITE);
      default:
        THROW_SYSTEM(errno, NULL);
    }
  }
}


PUBLIC long STREAM_tell(STREAM *stream)
{
  long pos;

  if (stream->type->tell(stream, &pos))
    /*THROW(E_SEEK, ERROR_get());*/
    THROW_SYSTEM(errno, NULL);

  return pos;
}


PUBLIC void STREAM_seek(STREAM *stream, long pos, int whence)
{
  if (stream->type->seek(stream, pos, whence))
  {
    switch(errno)
    {
      case EINVAL:
        THROW(E_ARG);
      default:
        THROW_SYSTEM(errno, NULL);
    }
  }
}


PRIVATE void add_string(char **addr, long *len_str, const char *src, int len)
{
  STRING_extend(addr, *len_str + len);
  memcpy(&((*addr)[*len_str]), src, len);

  *len_str += len;
}


PRIVATE void input(STREAM *stream, char **addr, boolean line)
{
  int len = 0;
  long len_str = 0;
  char c;

  *addr = NULL;

  for(;;)
  {
    STREAM_read(stream, &c, 1);
    if ((line && (c == '\n'))
        || (!line && (c <= ' ')))
      break;

    COMMON_buffer[len++] = c;
    if (len >= COMMON_BUF_MAX)
    {
      add_string(addr, &len_str, COMMON_buffer, len);
      len = 0;
    }

    if (STREAM_eof(stream))
      break;
  }

  if (len > 0)
    add_string(addr, &len_str, COMMON_buffer, len);

  STRING_extend_end(addr);
}


PUBLIC void STREAM_line_input(STREAM *stream, char **addr)
{
  input(stream, addr, TRUE);
}


PUBLIC void STREAM_input(STREAM *stream, char **addr)
{
  input(stream, addr, FALSE);
}


PUBLIC void STREAM_read_type(STREAM *stream, TYPE type, VALUE *value, long len)
{
  unsigned char data[4];

  value->type = type;

  switch (type)
  {
    case T_BOOLEAN:

      STREAM_read(stream, data, 1);
      value->_integer.value = (*data != 0) ? (-1) : 0;
      break;

    case T_BYTE:

      STREAM_read(stream, data, 1);
      value->_integer.value = *data;
      break;

    case T_SHORT:

      STREAM_read(stream, data, sizeof(short));
      if (stream->common.flag.swap)
        SWAP_short((short *)data);
      value->_integer.value = (long)*((short *)data);
      break;

    case T_INTEGER:

      STREAM_read(stream, &value->_integer.value, sizeof(long));
      if (stream->common.flag.swap)
        SWAP_long(&value->_integer.value);
      break;

    case T_FLOAT:

      STREAM_read(stream, &value->_float.value, sizeof(double));
      if (stream->common.flag.swap)
        SWAP_double(&value->_float.value);
      break;

    case T_DATE:

      STREAM_read(stream, &value->_date.date, sizeof(long));
      STREAM_read(stream, &value->_date.time, sizeof(long));
      if (stream->common.flag.swap)
      {
        SWAP_long(&value->_date.date);
        SWAP_long(&value->_date.time);
      }
      break;

    case T_STRING:
    case T_CSTRING:

      if (len == 0)
      {
        STREAM_read(stream, data, 1);

        switch (*data >> 6)
        {
          case 0:
          case 1:
            len = *data;
            break;

          case 2:
            STREAM_read(stream, &data[1], 1);
            *data &= 0x3F;

            if (stream->common.flag.swap)
              SWAP_short((short *)data);

            len = *((short *)data);
            break;

          case 3:
            STREAM_read(stream, &data[1], 3);
            *data &= 0x3F;

            if (stream->common.flag.swap)
              SWAP_long((long *)data);

            len = *((long *)data);
            break;
        }
      }

      STRING_new_temp_value(value, NULL, labs(len));
      if (len > 0)
        STREAM_read(stream, value->_string.addr, len);
      else
      {
        len = (-len);
        STREAM_read_max(stream, value->_string.addr, len);
        if (STREAM_eff_read < len)
        {
          if (STREAM_eff_read == 0)
            value->type = T_NULL;
          else
          {
            STRING_extend(&value->_string.addr, STREAM_eff_read);
            value->_string.len = STREAM_eff_read;
          }
        }
      }

      break;

    default:

      THROW(E_TYPE, "Standard type", TYPE_get_name(type));
  }
}


PUBLIC void STREAM_write_type(STREAM *stream, TYPE type, VALUE *value, long len)
{
  unsigned char data[8];

  value->type = type;

  switch (type)
  {
    case T_BOOLEAN:

      *data = value->_integer.value ? 0xFF : 0;
      STREAM_write(stream, data, 1);
      break;

    case T_BYTE:

      *data = (unsigned char)value->_integer.value;
      STREAM_write(stream, data, 1);
      break;

    case T_SHORT:

      *((short *)data) = (short)value->_integer.value;
      if (stream->common.flag.swap)
        SWAP_short((short *)data);
      STREAM_write(stream, data, sizeof(short));
      break;

    case T_INTEGER:

      *((long *)data) = value->_integer.value;
      if (stream->common.flag.swap)
        SWAP_long((long *)data);
      STREAM_write(stream, data, sizeof(long));
      break;

    case T_FLOAT:

      *((double *)data) = value->_float.value;
      if (stream->common.flag.swap)
        SWAP_double((double *)data);
      STREAM_write(stream, data, sizeof(double));
      break;

    case T_DATE:

      *((long *)data) = value->_date.date;
      if (stream->common.flag.swap)
        SWAP_long((long *)data);
      STREAM_write(stream, data, sizeof(long));

      *((long *)data) = value->_date.time;
      if (stream->common.flag.swap)
        SWAP_long((long *)data);
      STREAM_write(stream, data, sizeof(long));

      break;

    case T_STRING:
    case T_CSTRING:

      if (len == 0)
      {
        len = value->_string.len;

        if (len < 0x80)
        {
          *data = (unsigned char)len;
          STREAM_write(stream, data, 1);
        }
        else if (len < 0x4000)
        {
          *((short *)data) = (short)len | 0x8000;
          if (stream->common.flag.swap)
            SWAP_short((short *)data);
          STREAM_write(stream, data, 2);
        }
        else
        {
          *((long *)data) = len | 0xC0000000;
          if (stream->common.flag.swap)
            SWAP_long((long *)data);
          STREAM_write(stream, data, 4);
        }
      }

      STREAM_write(stream, value->_string.addr + value->_string.start, Min(len, value->_string.len));

      *data = 0;

      while (len > value->_string.len)
      {
        STREAM_write(stream, data, 1);
        len--;
      }

      break;

    default:

      THROW(E_TYPE, "Standard type", TYPE_get_name(type));
  }
}


PUBLIC void STREAM_load(const char *path, char **buffer, long *len)
{
  STREAM stream;

  STREAM_open(&stream, path, ST_READ);
  STREAM_lof(&stream, len);

  ALLOC(buffer, *len, "STREAM_load");

  STREAM_read(&stream, *buffer, *len);
  STREAM_close(&stream);
}


PUBLIC int STREAM_handle(STREAM *stream)
{
  return (*stream->type->handle)(stream);
}
