/***************************************************************************

  CWindow.cpp

  The Window and Form classes

  (c) 2000-2003 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __CWINDOW_CPP

#include <qnamespace.h>
#include <qapplication.h>
#include <qmenubar.h>
#include <qframe.h>
#include <qabstractlayout.h>
#include <qsizepolicy.h>
#include <qtoolbar.h>
#include <qworkspace.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qbitmap.h>

#if QT_VERSION >= 0x030100
  #include <qeventloop.h>
#endif

#include "gambas.h"

#include "main.h"
#include "CWidget.h"
#include "CMenu.h"
#include "CWindow.h"

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>


//#define DEBUG_STATE

DECLARE_EVENT(EVENT_Open);
DECLARE_EVENT(EVENT_Close);
DECLARE_EVENT(EVENT_Activate);
DECLARE_EVENT(EVENT_Deactivate);
DECLARE_EVENT(EVENT_Move);
DECLARE_EVENT(EVENT_Resize);
DECLARE_EVENT(EVENT_Show);
DECLARE_EVENT(EVENT_Hide);

CWINDOW *CWINDOW_Main = NULL;
CWINDOW *CWINDOW_Current = NULL;

static GB_CLASS CLASS_Workspace = NULL;
static GB_CLASS CLASS_Container = NULL;


/***************************************************************************

  Window manager specific code

***************************************************************************/

#define MAX_WINDOW_STATE 8

static bool _atom_init = false;

static Atom _atom_net_wm_state;
static Atom _atom_net_wm_state_above;
static Atom _atom_net_wm_state_stays_on_top;
static Atom _atom_net_wm_state_skip_taskbar;

static Atom _window_state[MAX_WINDOW_STATE];
static int _window_state_count = 0;


static void init_atoms(Display *dpy)
{
  if (_atom_init)
    return;
  
  _atom_net_wm_state = XInternAtom(dpy, "_NET_WM_STATE", True);
  _atom_net_wm_state_above = XInternAtom(dpy, "_NET_WM_STATE_ABOVE", True);
  _atom_net_wm_state_stays_on_top = XInternAtom(dpy, "_NET_WM_STATE_STAYS_ON_TOP", True);  
  _atom_net_wm_state_skip_taskbar = XInternAtom(dpy, "_NET_WM_STATE_SKIP_TASKBAR", True);
  _atom_init = true;
}

static void load_window_state(Display *dpy, WId wid)
{
  Atom type;
  int format;
  unsigned long length, after;
  unsigned char *data;

  init_atoms(dpy);
    
  _window_state_count = 0;
  
  XGetWindowProperty(dpy, wid, _atom_net_wm_state, 0, 8,
    False, AnyPropertyType, &type, &format,
    &length, &after, &data);  
    
  if (length > 8)
    length = 8;
    
  _window_state_count = length;
  memcpy(_window_state, data, _window_state_count * sizeof(Atom));
  
  XFree(data);
}

static void save_window_state(Display *dpy, WId wid)
{
  XChangeProperty(dpy, wid, _atom_net_wm_state, XA_ATOM, 32, PropModeReplace,
      (unsigned char *)_window_state, _window_state_count);
}

static bool is_window_state(Atom state)
{
  int i;
  
  for (i = 0; i < _window_state_count; i++)
  {
    if (_window_state[i] == state)
      return true;
  }
  
  return false;
}

static void set_window_state(Atom state)
{
  if (is_window_state(state))
    return;
    
  if (_window_state_count == MAX_WINDOW_STATE)
  {
    qDebug("Too many properties in window");
    return;
  }
  
  _window_state[_window_state_count++] = state;
}
    
static void clear_window_state(Atom state)
{
  int i;

  for (i = 0; i < _window_state_count; i++)
  {
    if (_window_state[i] == state)
    {
      _window_state_count--;
    
      for (; i < _window_state_count; i++)
        _window_state[i] = _window_state[i + 1];
        
      return;
    }
  }
  
}


void CWINDOW_change_property(QWidget *w, Atom property, bool set)
{
  XEvent e;
  long mask = (SubstructureRedirectMask | SubstructureNotifyMask);

  if (!w->isTopLevel())
    return;
  
  if (w->isVisible())
  {
    e.xclient.type = ClientMessage;
    e.xclient.message_type = _atom_net_wm_state;
    e.xclient.display = w->x11Display();
    e.xclient.window = w->winId();
    e.xclient.format = 32;
    e.xclient.data.l[0] = set ? 1 : 0;
    e.xclient.data.l[1] = property;
    e.xclient.data.l[2] = 0;
    e.xclient.data.l[3] = 0;
    e.xclient.data.l[4] = 0;
  
    XSendEvent(w->x11Display(), qt_xrootwin(), False, mask, &e);
  }
  else
  {
    load_window_state(w->x11Display(), w->winId());
    
    if (set)
      set_window_state(property);
    else
      clear_window_state(property);
    
    save_window_state(w->x11Display(), w->winId());
  }
}
    

/***************************************************************************

  Window

***************************************************************************/

BEGIN_METHOD_VOID(CWINDOW_init)

  CLASS_Workspace = GB.FindClass("Workspace");
  CLASS_Container = GB.FindClass("Container");

END_METHOD


BEGIN_METHOD(CWINDOW_new, GB_OBJECT parent)

  //MyMainWindow *win = new MyMainWindow(window_main ? QMAINWINDOW(window_main) : NULL);
  MyMainWindow *win = 0;
  QFrame *frame = 0;
  QWidget *container;

  if (MISSING(parent))
  {
    //qDebug("CWINDOW_new: missing parent");
  
    //qDebug("CWINDOW_Main = %p", CWINDOW_Main);
    //qDebug("CWINDOW_new %p", THIS);
    win = new MyMainWindow(CWINDOW_Main ? (MyMainWindow *)QWIDGET(CWINDOW_Main) : 0);
    container = new MyContainer(win);
    THIS->widget.widget = win;
    THIS->embedded = false;
    THIS->window = true;

    CWIDGET_new(win, (void *)_object, NULL);
  }
  else if (GB.Is(VARG(parent), CLASS_Container))
  {
    frame = new MyEmbeddedWindow(CONTAINER(VARG(parent)));
    container = frame;
    THIS->widget.widget = frame;
    THIS->embedded = true;
    THIS->window = false;

    CWIDGET_new(frame, (void *)_object, NULL);
  }
  else if (GB.Is(VARG(parent), CLASS_Workspace))
  {
    //qDebug(">> CWINDOW_new: workspace");
    win = new MyMainWindow(QWIDGET(VARG(parent)));
    container = new MyContainer(win);
    THIS->widget.widget = win;
    //frame = new MyEmbeddedWindow(QWIDGET(VARG(parent)));
    //container = frame;
    //FRAME = frame;
    THIS->embedded = true;
    THIS->window = true;

    //CWIDGET_set_flag(THIS, WF_PARENT_GEOMETRY);
    CWIDGET_new(win, (void *)_object, NULL);
    
    //qDebug("<< CWINDOW_new: workspace");
  }
  else
  {
    GB.Error("The parent of a Window must be a Container or a Workspace");
    return;
  }

  THIS->container = container;
  container->setBackgroundOrigin(QWidget::WindowOrigin);
  
  if (frame)
  {
    frame->installEventFilter(&CWindow::manager);
  }
  else
  {
    win->setCentralWidget(container);

    win->setOpaqueMoving(true);
    win->installEventFilter(&CWindow::manager);

    if (!THIS->embedded)
    {
      /*if (CWindow::count >= 64)
      {
        GB.Error("Too many windows opened");
        return;
      }*/
      
      CWindow::dict.insert(_object, OBJECT(const CWINDOW));
      CWindow::count = CWindow::dict.count();
      
      //qDebug("CWindow::count = %d (%p %s)", CWindow::count, _object, THIS->embedded ? "E" : "W");

      if (CWINDOW_Main == 0)
      {
        //qDebug("CWINDOW_Main -> %p", THIS);
        CWINDOW_Main = THIS;
      }
    }
  }

END_METHOD


BEGIN_METHOD_VOID(CFORM_new)

  GB.Attach(_object, _object, "Form");
  
  if (THIS->embedded && !THIS->window)
    FRAME->show();

END_METHOD


BEGIN_METHOD_VOID(CFORM_main)

  CWINDOW *form;

  form = (CWINDOW *)GB.AutoCreate(GB.GetClass(NULL), 0);

  //GB.New((void **)&form, GB.GetClass(NULL), NULL, NULL);

  ((MyMainWindow *)form->widget.widget)->showActivate();

END_METHOD


BEGIN_METHOD(CFORM_load, GB_OBJECT parent)

  if (!MISSING(parent))
  {
    GB.Push(1, GB_T_OBJECT, VARG(parent));
    //qDebug("CFORM_load + parent");
  }

  GB.AutoCreate(GB.GetClass(NULL), MISSING(parent) ? 0 : 1);

END_METHOD


BEGIN_METHOD_VOID(CWINDOW_free)

  //qDebug("CWINDOW_free");

  if (THIS->menu)
    delete THIS->menu;

  //PICTURE_set(&(window->icon), 0);
  GB.StoreObject(NULL, (void **)&(THIS->icon));
  GB.StoreObject(NULL, (void **)&(THIS->mask));
  GB.Unref((void **)&THIS->focus);

  /*CALL_METHOD_VOID(CWIDGET_delete);*/

END_METHOD


BEGIN_METHOD_VOID(CWINDOW_next)

  CWINDOW *next;
  QPtrDictIterator<CWINDOW> *iter = ENUM(QPtrDictIterator<CWINDOW> *);

  if (iter == NULL)
  {
    iter = new QPtrDictIterator<CWINDOW>(CWindow::dict);
    ENUM(QPtrDictIterator<CWINDOW> *) = iter;
  }

  next = iter->current();

  if (next == NULL)
  {
    delete iter;
    //ENUM(QPtrDictIterator<CWINDOW>) = NULL;
    GB.StopEnum();
    return;
  }

  ++(*iter);
  GB.ReturnObject(next);

END_METHOD


BEGIN_PROPERTY(CWINDOW_count)

  GB.ReturnInteger(CWindow::dict.count());

END_PROPERTY


BEGIN_METHOD(CWINDOW_get, GB_INTEGER id)

  QWidget *wid = QWidget::find(VARG(id));

  //qDebug("id = %d wid = %p", PARAM(id), wid);

  if (wid != 0 && wid->isTopLevel())
  {
    //qDebug("-> %p", CWidget::getReal(wid));
    GB.ReturnObject(CWidget::getReal(wid));
  }
  else
  {
    //qDebug("-> %p", 0);
    GB.ReturnNull();
  }

END_METHOD


static bool do_close(CWINDOW *_object, long ret, bool destroyed = false)
{
  bool closed;

  if (CWIDGET_test_flag(THIS, WF_IN_CLOSE) || CWIDGET_test_flag(THIS, WF_CLOSED))
    return false;

  if (!THIS->window)
  {
    CWIDGET_set_flag(THIS, WF_IN_CLOSE);
    closed = !GB.Raise(THIS, EVENT_Close, 0);
    CWIDGET_clear_flag(THIS, WF_IN_CLOSE);
    if (destroyed)
    {
      CWIDGET_set_flag(THIS, WF_CLOSED);
    }
    else if (closed)
    {
      CWIDGET_set_flag(THIS, WF_CLOSED);
      WINDOW->hide();
      if (!CWIDGET_test_flag(_object, WF_PERSISTENT))
        CWIDGET_destroy((CWIDGET *)THIS);
    }
  }
  else
  { 
    if (WINDOW->isHidden())
    {
      QCloseEvent e;
      QApplication::sendEvent(WINDOW, &e);
      closed = e.isAccepted();
    }
    else
      closed = WINDOW->close();
  }

  if (closed)
    THIS->ret = ret;

  //qDebug("CWINDOW_close: ret = %d", THIS->ret);

  return (!closed);
}

BEGIN_METHOD(CWINDOW_close, GB_INTEGER ret)

  long ret = VARGOPT(ret, 0);

  GB.ReturnBoolean(do_close(THIS, ret));

END_METHOD


BEGIN_METHOD_VOID(CWINDOW_raise)

  if (!THIS->window)
  {
    if (!FRAME->isVisible())
      FRAME->show();
    FRAME->raise();
  }
  else
  {
    if (!WINDOW->isVisible())
      WINDOW->showActivate();
    else
      WINDOW->raise();
  }

END_METHOD


BEGIN_METHOD_VOID(CWINDOW_show)

  if (!THIS->window)
  {
    FRAME->raise();
    FRAME->show();
  }
  else
  {
    //if (CWINDOW_Current)
    //  GB.Error("A modal window is already displayed");
    //else
    if (CWINDOW_Current)
      WINDOW->showModal();
    else
      WINDOW->showActivate();
  }

END_METHOD


BEGIN_METHOD_VOID(CWINDOW_show_modal)

  THIS->ret = 0;

  if (!THIS->embedded)
    WINDOW->showModal();
  //qDebug("CWINDOW_show_modal: ret = %d", THIS->ret);
  GB.ReturnInteger(THIS->ret);

END_METHOD


BEGIN_PROPERTY(CWINDOW_modal)

  if (!THIS->embedded)
    GB.ReturnBoolean(WINDOW->isModal());
  else
    GB.ReturnBoolean(false);

END_PROPERTY

/*BEGIN_METHOD_VOID(CWINDOW_dialog)

  CWINDOW *win;

  GB.New((void **)&win, GB.GetClass(NULL), NULL, NULL);

  win->ret = 0;
  ((MyMainWindow *)win->widget.widget)->showModal();
  GB.ReturnInteger(win->ret);

END_METHOD*/


BEGIN_PROPERTY(CWINDOW_persistent)

  /*
  if (READ_PROPERTY)
    GB.ReturnBoolean(WIDGET->isPersistent());
  else
    WIDGET->setPersistent(PROPERTY(char) != 0);
  */

  if (!THIS->window)
  {
    if (READ_PROPERTY)
      GB.ReturnBoolean(true);
  }
  else
  {
    if (READ_PROPERTY)
      GB.ReturnBoolean(CWIDGET_test_flag(THIS, WF_PERSISTENT));
    else
    {
      if (VPROP(GB_BOOLEAN))
        CWIDGET_set_flag(THIS, WF_PERSISTENT);
      else
        CWIDGET_clear_flag(THIS, WF_PERSISTENT);
    }
  }

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_text)

  if (!THIS->window)
  {
    if (READ_PROPERTY)
      GB.ReturnNull();
  }
  else
  {
    if (READ_PROPERTY)
      GB.ReturnNewZeroString(TO_UTF8(WINDOW->caption()));
    else
      WINDOW->setCaption(QSTRING_PROP());
  }

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_menu_count)

  if (THIS->menu)
    GB.ReturnInteger(THIS->menu->count());
  else
    GB.ReturnInteger(0);

END_PROPERTY


BEGIN_METHOD_VOID(CWINDOW_menu_next)

  CWINDOW *window = OBJECT(CWINDOW);
  unsigned int index;

  if (window->menu == NULL)
  {
    GB.StopEnum();
    return;
  }

  index = ENUM(int);

  if (index >= window->menu->count())
  {
    GB.StopEnum();
    return;
  }

  GB.ReturnObject(window->menu->at(index));

  ENUM(int) = index + 1;

END_PROPERTY




/*

BEGIN_METHOD(CWINDOW_toolbar_add, CTOOLBAR *toolbar)

  QMainWindow *wid = WIDGET;
  QToolBar *tb = QTOOLBAR(PARAM(toolbar));

  qDebug("ToolBar H = %d", tb->height());
  wid->addToolBar(tb);
  qDebug("ToolBar H = %d", tb->height());

END_METHOD


BEGIN_METHOD(CWINDOW_toolbar_remove, CTOOLBAR *toolbar)

  QMainWindow *wid = WIDGET;

  wid->removeToolBar(QTOOLBAR(PARAM(toolbar)));

END_METHOD
*/


BEGIN_PROPERTY(CWINDOW_id)

  if (!THIS->window)
    GB.ReturnInteger(0);
  else
    GB.ReturnInteger((long)WINDOW->winId());

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_border)

  if (!THIS->window)
  {
    if (READ_PROPERTY)
      GB.ReturnInteger(0);
  }
  else
  {
    if (READ_PROPERTY)
      GB.ReturnInteger(WINDOW->getBorder());
    else
      WINDOW->setBorder(VPROP(GB_INTEGER));
  }

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_icon)

  if (!THIS->window)
  {
    if (READ_PROPERTY)
      GB.ReturnNull();
  }
  else
  {
    if (READ_PROPERTY)
      GB.ReturnObject(THIS->icon);
    else
      SET_PIXMAP(WINDOW->setIcon, &(THIS->icon), PROP(GB_OBJECT));
      //WIDGET->setIcon(PICTURE_set(&(THIS->icon), PROPERTY(CPICTURE *)));
  }

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_mask)

  if (THIS->embedded)
  {
    if (READ_PROPERTY)
      GB.ReturnNull();
  }
  else
  {
    if (READ_PROPERTY)
      GB.ReturnObject(THIS->mask);
    else
    {
      GB.StoreObject(PROP(GB_OBJECT), (void **)&THIS->mask);
      WINDOW->defineMask();
    }
  }

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_state)

  if (!THIS->window)
  {
    if (READ_PROPERTY)
      GB.ReturnInteger(0);
  }
  else
  {
    if (READ_PROPERTY)
      GB.ReturnInteger(WINDOW->getState());
    else
      WINDOW->setState(VPROP(GB_INTEGER));
  }

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_top_only)

  if (READ_PROPERTY)
  {
    GB.ReturnBoolean(THIS->topOnly);
    return;
  }
  
  if (THIS->embedded)
    return;
    
  if ((THIS->topOnly == 0) == (VPROP(GB_BOOLEAN) == 0))
    return;
  
  THIS->topOnly = VPROP(GB_BOOLEAN);
  WINDOW->setTopOnly(VPROP(GB_BOOLEAN));

END_PROPERTY


BEGIN_PROPERTY(CWINDOW_skip_taskbar)

  if (READ_PROPERTY)
  {
    GB.ReturnBoolean(THIS->skipTaskbar);
    return;
  }
  
  if (THIS->embedded)
    return;
    
  if ((THIS->skipTaskbar == 0) == (VPROP(GB_BOOLEAN) == 0))
    return;
  
  THIS->skipTaskbar = VPROP(GB_BOOLEAN);
  WINDOW->setSkipTaskbar(VPROP(GB_BOOLEAN));

END_PROPERTY


BEGIN_METHOD_VOID(CWINDOW_center)

  if (THIS->embedded)
    return;

  WINDOW->center(true);
    
END_METHOD


BEGIN_METHOD_VOID(CWINDOW_delete)

  //qDebug("CWINDOW_delete %p", THIS);
  
  do_close(THIS, 0);    

  if (THIS->window)
    CWIDGET_clear_flag(THIS, WF_PERSISTENT);
  
  CWIDGET_destroy((CWIDGET *)THIS);

END_METHOD


BEGIN_PROPERTY(CWINDOW_visible)

  if (READ_PROPERTY)
    GB.ReturnBoolean(!WINDOW->isHidden());
  else
  {
    if (VPROP(GB_BOOLEAN))
      WINDOW->showActivate();
    else
      WINDOW->hide();
  }

END_PROPERTY

/*
BEGIN_PROPERTY(CWINDOW_design)

  if (READ_PROPERTY)
  {
    GB.ReturnBoolean(CWIDGET_test_flag(_object, WF_WIN_DESIGN));
    return;
  }

  if (PROPERTY(char))
  {
    CWIDGET_set_flag(_object, WF_WIN_DESIGN);

  }
  else
  {
    CWIDGET_clear_flag(_object, WF_WIN_DESIGN);
  }

END_PROPERTY
*/

/*
BEGIN_PROPERTY(CWINDOW_workspace)

  CWINDOW *win = OBJECT(CWINDOW);

  if (READ_PROPERTY)
    GB.ReturnBoolean(win->container->isA("QWorkspace"));
  else
  {
    if (win->container->children() != 0)
    {
      GB.Error("Property is read-only");
      return;
    }

    delete win->container;

    if (PROPERTY(char))
      win->container = new QWorkspace(WIDGET);
    else
      win->container = new QFrame(WIDGET);
  }

END_PROPERTY
*/


/***************************************************************************

  Dclarations

***************************************************************************/


GB_DESC CWindowMenusDesc[] =
{
  GB_DECLARE(".WindowMenus", 0), GB_VIRTUAL_CLASS(),

  GB_METHOD("_next", "Menu", CWINDOW_menu_next, NULL),
  GB_PROPERTY_READ("Count", "i", CWINDOW_menu_count),

  GB_END_DECLARE
};

/*
GB_DESC CWindowToolBarsDesc[] =
{
  GB_DECLARE(".Window.ToolBars", 0), GB_VIRTUAL_CLASS(),

  GB_METHOD("Add", NULL, CWINDOW_toolbar_add, "'Toolbar'Toolbar;"),
  GB_METHOD("Remove", NULL, CWINDOW_toolbar_remove, "'Toolbar'Toolbar;"),

  GB_END_DECLARE
};
*/


GB_DESC CWindowDesc[] =
{
  GB_DECLARE("Window", sizeof(CWINDOW)), GB_INHERITS("Container"),

  GB_CONSTANT("None", "i", 0),
  GB_CONSTANT("Fixed", "i", 1),
  GB_CONSTANT("Resizable", "i", 2),

  GB_CONSTANT("Normal", "i", 0),
  GB_CONSTANT("Minimized", "i", 1),
  GB_CONSTANT("Maximized", "i", 2),
  GB_CONSTANT("FullScreen", "i", 3),

  GB_STATIC_METHOD("_init", NULL, CWINDOW_init, NULL),
  GB_METHOD("_new", NULL, CWINDOW_new, "[(Parent)Control;]"),
  GB_METHOD("_free", NULL, CWINDOW_free, NULL),

  //GB_STATIC_METHOD("Dialog", "i", CWINDOW_dialog, NULL),

  GB_METHOD("Close", "b", CWINDOW_close, "[(Return)i]"),
  GB_METHOD("Raise", NULL, CWINDOW_raise, NULL),
  GB_METHOD("Show", NULL, CWINDOW_show, NULL),
  GB_METHOD("ShowModal", "i", CWINDOW_show_modal, NULL),
  GB_METHOD("ShowDialog", "i", CWINDOW_show_modal, NULL),
  GB_METHOD("Center", NULL, CWINDOW_center, NULL),
  GB_PROPERTY_READ("Modal", "b", CWINDOW_modal),

  //GB_STATIC_METHOD("Dialog", "i", CWINDOW_dialog, NULL),

  GB_METHOD("Delete", NULL, CWINDOW_delete, NULL),

  /*
  GB_PROPERTY_READ("ScreenX", "i", CWINDOW_x),
  GB_PROPERTY_READ("ScreenY", "i", CWINDOW_y),
  GB_PROPERTY("X", "i", CWINDOW_x),
  GB_PROPERTY("Y", "i", CWINDOW_y),
  GB_PROPERTY("W", "i", CWINDOW_w),
  GB_PROPERTY("H", "i", CWINDOW_h),
  GB_PROPERTY("Left", "i", CWINDOW_x),
  GB_PROPERTY("Top", "i", CWINDOW_y),
  GB_PROPERTY("Width", "i", CWINDOW_w),
  GB_PROPERTY("Height", "i", CWINDOW_h),
  */

  GB_PROPERTY("Persistent", "b", CWINDOW_persistent),
  GB_PROPERTY("Text", "s", CWINDOW_text),
  GB_PROPERTY("Title", "s", CWINDOW_text),
  GB_PROPERTY("Caption", "s", CWINDOW_text),
  GB_PROPERTY("Icon", "Picture", CWINDOW_icon),
  GB_PROPERTY("Border", "i<Window,None,Fixed,Resizable>", CWINDOW_border),
  GB_PROPERTY("State", "i<Window,Normal,Minimized,Maximized,FullScreen>", CWINDOW_state),
  GB_PROPERTY("TopOnly", "b", CWINDOW_top_only),
  GB_PROPERTY("SkipTaskbar", "b", CWINDOW_skip_taskbar),
  GB_PROPERTY("Visible", "b", CWINDOW_visible),
  GB_PROPERTY("Arrangement", "i<Arrange>", CCONTAINER_arrangement),
  //GB_PROPERTY("AutoResize", "b", CCONTAINER_auto_resize),
  GB_PROPERTY("Padding", "i", CCONTAINER_padding),
  GB_PROPERTY("Spacing", "i", CCONTAINER_spacing),

  //GB_PROPERTY("Picture", "Picture", CWINDOW_picture),
  GB_PROPERTY("Mask", "Picture", CWINDOW_mask),

  GB_PROPERTY_SELF("Menus", ".WindowMenus"),

  GB_PROPERTY_READ("Id", "i", CWINDOW_id),
  GB_PROPERTY_READ("Handle", "i", CWINDOW_id),

  GB_CONSTANT("_Properties", "s", CWINDOW_PROPERTIES),
  GB_CONSTANT("_DefaultEvent", "s", "Open"),
  GB_CONSTANT("_Arrangement", "i", ARRANGE_FILL),

  //GB_EVENT("Delete", NULL, NULL, &EVENT_Destroy),
  GB_EVENT("Close", "b", NULL, &EVENT_Close),
  GB_EVENT("Open", NULL, NULL, &EVENT_Open),
  GB_EVENT("Activate", NULL, NULL, &EVENT_Activate),
  GB_EVENT("Deactivate", NULL, NULL, &EVENT_Deactivate),
  GB_EVENT("Move", NULL, NULL, &EVENT_Move),
  GB_EVENT("Resize", NULL, NULL, &EVENT_Resize),
  GB_EVENT("Show", NULL, NULL, &EVENT_Show),
  GB_EVENT("Hide", NULL, NULL, &EVENT_Hide),

  GB_END_DECLARE
};


GB_DESC CWindowsDesc[] =
{
  GB_DECLARE("Windows", 0), GB_VIRTUAL_CLASS(),

  GB_STATIC_METHOD("_next", "Window", CWINDOW_next, NULL),
  GB_STATIC_METHOD("_get", "Window", CWINDOW_get, "(Id)i"),
  GB_STATIC_PROPERTY_READ("Count", "i", CWINDOW_count),

  GB_END_DECLARE
};


GB_DESC CFormDesc[] =
{
  GB_DECLARE("Form", sizeof(CFORM)), GB_INHERITS("Window"),
  GB_AUTO_CREATABLE(),

  GB_STATIC_METHOD("Main", NULL, CFORM_main, NULL),
  GB_STATIC_METHOD("Load", NULL, CFORM_load, "[(Parent)Control;]"),
  GB_METHOD("_new", NULL, CFORM_new, NULL),

  GB_END_DECLARE
};


/***************************************************************************

  MyMainWindow

***************************************************************************/

#if QT_VERSION >= 0x030005
MyMainWindow::MyMainWindow(QWidget *parent) :
  QMainWindow::QMainWindow(parent, 0, Qt::WType_TopLevel | (parent ? 0 : Qt::WGroupLeader)) //, 0, Qt::WType_TopLevel) // | (parent ? copy->getWFlags() : 0)) // | Qt::WDestructiveClose)
#else
MyMainWindow::MyMainWindow(QWidget *parent) :
  QMainWindow::QMainWindow(0) //, 0, Qt::WType_TopLevel) // | (copy ? copy->getWFlags() : 0)) // | Qt::WDestructiveClose)
#endif
{
  db = 0;
  cb = 0;
  sg = 0;
  shown = false;
  border = BorderResizable;
  state = StateNormal;
  mustCenter = false;
  _activate = false;

  mdichild = parent && parent->isA("QWorkspace");
  if (mdichild)
    clearWFlags(WType_TopLevel);
    
  //qDebug("MyMainWindow: %p -> %p", this, parentWidget());
    
  /*if (!parent)
  {
    qDebug("GroupLeader!");
    setWFlags(WGroupLeader);
  }*/
    
  //qDebug("parent = %p", parentWidget());
}


MyMainWindow::~MyMainWindow()
{
  CWINDOW *ob = (CWINDOW *)CWidget::get(this);

  //qDebug("~MyMainWindow %p ob = %p", this, ob);

  if (sg)
    delete sg;

  if (ob == NULL)
  {
    qWarning("~MyMainWindow: ob == NULL");
    return;
  }

  //do_close(ob, 0, true);

  GB.Detach(ob);

  if (!mdichild)
  {
    if (ob->menu)
      CMenu::unrefChildren(ob->menu);

    CWindow::dict.remove(ob);

    //if (ob == window_main)
    //  window_main = NULL;

    CWindow::count = CWindow::dict.count();
    //qDebug("~MyMainWindow: CWindow::count = %d (%p %s)", CWindow::count, ob, ob->embedded ? "E" : "W");

    MAIN_check_quit();
  }

  //qDebug("~MyMainWindow %p (end)", this);
}


void MyMainWindow::setCancelButton(QPushButton *pb, bool on)
{
  if (on)
  {
    cb = pb;
  }
  else
  {
    if (pb == cb)
      cb = 0;
  }
}

void MyMainWindow::setDefaultButton(QPushButton *pb, bool on)
{
  //qDebug("setDefaultButton: %s %s", pb->text().latin1(), on ? "true" : "false");

  if (on)
  {
    if (db)
      db->setDefault(false);

    db = pb;
    db->setDefault(true);
  }
  else
  {
    if (pb == db)
    {
      db->setDefault(false);
      db = 0;
    }
  }
}


int MyMainWindow::getState()
{
  if (isHidden())
    return state;

  if (isMinimized())
    return StateMinimized;
  else if (isMaximized())
    return StateMaximized;
  #if QT_VERSION >= 0x030100
  else if (isFullScreen())
    return StateFullScreen;
  #endif
  else
    return StateNormal;
}

void MyMainWindow::setState(int s)
{
  #ifdef DEBUG_STATE
  qDebug("setState(%d)", s);
  #endif

  if (s < 0 || s > 3)
    return;

  if (!isHidden() && s == getState())
  {
    //qDebug("show");
    show();
    return;
  }

  state = s;

  if (shown)
  {
    switch(state)
    {
      case StateNormal:
        #ifdef DEBUG_STATE
        qDebug("showNormal()");
        #endif
        showNormal();
        break;

      case StateMinimized:
        #ifdef DEBUG_STATE
        qDebug("showMinimized()");
        #endif
        showMinimized();
        break;

      case StateMaximized:
        #ifdef DEBUG_STATE
        qDebug("showMaximized()");
        #endif
        showMaximized();
        break;

      case StateFullScreen:
        #ifdef DEBUG_STATE
        qDebug("showFullScreen()");
        #endif
        showFullScreen();
        break;

      default:
        #ifdef DEBUG_STATE
        qDebug("state = %d ??", state);
        #endif
        break;
    }

    #if QT_VERSION < 0x030102
    switch(state)
    {
      case StateNormal:
      case StateMaximized:
      case StateFullScreen:
        qDebug("XMapWindow");
        XMapWindow(x11Display(), winId());
        XFlush(x11Display());
        //XSetInputFocus(x11Display(), winId(), RevertToNone, CurrentTime);
        break;

      default:
        break;
    }
    #endif
  }
}


void MyMainWindow::showEvent(QShowEvent *e)
{
  if (_activate)
  {
    raise();
    setActiveWindow();
    _activate = false;
  }
}

void MyMainWindow::showActivate()
{
  CWIDGET *_object = CWidget::get(this);
  int st;
  
  //qDebug(">> Show %d %d %d %d", x(), y(), width(), height());

  if (CWIDGET_test_flag(THIS, WF_IN_CLOSE) || CWIDGET_test_flag(THIS, WF_IN_SHOW))
  {
    //qDebug("Showing form during a close event !");
    return;
  }
  
  //qDebug("showActivate %p", _object);
  
  CWIDGET_set_flag(THIS, WF_IN_SHOW);
  CWIDGET_clear_flag(THIS, WF_CLOSED);

  if (!THIS->embedded)
  {
    setTopOnly(THIS->topOnly);
    setSkipTaskbar(THIS->skipTaskbar);
  }
  
  if (!shown)
  {
    GB.Raise(THIS, EVENT_Open, 0);
  }

  st = getState();

  if (!shown)
  {
    shown = true;
    setState(st);
    //THIS->x = x();
    //THIS->y = y();
    
    /*if (THIS->window && THIS->embedded)
    {
      qDebug("CWorkspace: install event filter: %p %s", parentWidget(), parentWidget()->className());
      parentWidget()->installEventFilter(&CWorkspace::manager);
    }*/

    GB.Raise(THIS, EVENT_Move, 0);
    GB.Raise(THIS, EVENT_Resize, 0);
  }
  else
  {
    if (st == StateMinimized)
    {
      setState(StateNormal);
      _activate = true;
    }
    else if (!isVisible())
    {
      //qDebug("#1");
      show();
      //if (THIS->embedded)
      //  setFocus();
    }
    else
    {
      if (THIS->embedded)
      {
        //qDebug("#2");
        //setActiveWindow();
        //raise();
        //workspace->setFocus();
        setFocus();
        //show();
      }
      else
      {
        //qDebug("#3");
        raise();
        setActiveWindow();
      }
    }

  }

  defineMask();
  
  CWIDGET_clear_flag(THIS, WF_IN_SHOW);
  //qDebug("<< showActivate");

  //qDebug("showActivate: %p  parent = %p  x = %d  y = %d", this, parentWidget(), mapToGlobal(QPoint(0, 0)).x(), mapToGlobal(QPoint(0, 0)).y());  
}

void MyMainWindow::showModal(void)
{
  WFlags flags = getWFlags();
  CWIDGET *_object = CWidget::get(this);
  bool persistent = CWIDGET_test_flag(THIS, WF_PERSISTENT);
  QWidget *parent = parentWidget();
  QWidget *reparent;
  CWINDOW *save = CWINDOW_Current;
  QPoint p = pos();

  if (testWFlags(WShowModal))
    return;

  //setWFlags( WShowModal );

  //QPoint p((qApp->desktop()->width() - width()) / 2, (qApp->desktop()->height() - height()) / 2);

  //clearWFlags(WStyle_Minimize);
  //clearWFlags(WStyle_SysMenu);

  //reparent(parentWidget(), WType_Dialog | WShowModal | WStyle_DialogBorder | (getWFlags() & 0xffff0000 & ~WStyle_SysMenu), p);
  //reparent(parentWidget(), WShowModal | WStyle_NormalBorder | WStyle_Title | WStyle_ContextHelp | (getWFlags() & 0xffff0000), p);

  
  /*p.setX((qApp->desktop()->width() - width()) / 2);
  p.setY((qApp->desktop()->height() - height()) / 2);
  doReparent(CWINDOW_Current, getWFlags(), p);
  if (CWINDOW_Current)
    move(x() - parentWidget()->x(), y() - parentWidget()->y());*/

  //setWFlags(WType_Dialog | WShowModal | WStyle_DialogBorder | WStyle_Dialog);
  
  mustCenter = true;
  
  /*if (parentWidget())
  {
    p.setX(p.x() - parentWidget()->pos().x());
    p.setY(p.y() - parentWidget()->pos().y());
  }*/

  reparent = qApp->activeWindow();
  if (!reparent && CWINDOW_Main)
  {
    reparent = CWINDOW_Main->widget.widget;
    if (reparent == this)
      reparent = 0;
  }
  
  #if QT_VERSION >= 030300  
    doReparent(reparent, getWFlags() | WStyle_DialogBorder | WShowModal, p);
  #else
    doReparent(reparent, getWFlags() | WType_Dialog | WShowModal | WStyle_DialogBorder | WStyle_Dialog, p);
  #endif


  
  if (border == BorderResizable)
    setSizeGrip(true);

  //XSetTransientForHint(x11Display(), winId(), None );

  showActivate();

  CWINDOW_Current = THIS;
  qApp->eventLoop()->enterLoop();
  CWINDOW_Current = save;

  if (persistent)
  {
    setSizeGrip(false);
    clearWState(WShowModal);
    setWFlags(flags);
    doReparent(parent, flags, p);
  }

  /* Gets the customed destroy event immediately */
  //qApp->eventLoop()->processEvents(QEventLoop::ExcludeUserInput);
}

/*
void MyMainWindow::hide(void)
{
  CWIDGET *_object = CWidget::get(this);
  
  QMainWindow::hide();

  if (THIS->topOnly)
    setTopOnly(true);
}
*/

void MyMainWindow::setTopOnly(bool top)
{
  CWINDOW_change_property(this, _atom_net_wm_state_above, top);
  CWINDOW_change_property(this, _atom_net_wm_state_stays_on_top, top);
  
  //#if QT_VERSION < 0x030200
  //bool hasIcon;
  //#endif
  //QPixmap p;
  //WFlags flags;
  //QWidget *parent;

  
  #if 0
  flags = getWFlags();
  if (top)
  {
    flags |= WStyle_StaysOnTop;
    //flags |= WX11BypassWM;
    
    /*if (border == BorderNone)
      flags |= WType_Popup;
    else
      flags |= WType_Popup;*/
    //parent = 0;
  }
  else
  {
    flags &= ~WStyle_StaysOnTop;
    //flags &= ~WX11BypassWM;
    //flags &= ~WType_Popup;
    //parent = CWINDOW_Main ? CWINDOW_Main->widget.widget : 0;
  }

  // #if QT_VERSION < 0x030200
  /* QT bug */
  hasIcon = icon() != 0;
  if (hasIcon)
    p = *icon();
  //# endif

  //qDebug("setTopOnly");
  reparent(parentWidget(), flags, pos());

  //#if QT_VERSION < 0x030200
  if (hasIcon)
    setIcon(p);
  //# endif
  #endif
}

void MyMainWindow::setSkipTaskbar(bool show)
{
  CWINDOW_change_property(this, _atom_net_wm_state_skip_taskbar, show);
  
  /*
  load_window_state(x11Display(), winId());
  if (show)
    set_window_state(_atom_net_wm_state_skip_taskbar);
  else
    clear_window_state(_atom_net_wm_state_skip_taskbar);
  save_window_state(x11Display(), winId());
  */
}

/*
QSize MyMainWindow::sizeHint() const
{
  //if (fixed)
  return QSize(width(), height());

  //return QFrame::sizeHint();
}
*/

void MyMainWindow::moveSizeGrip()
{
  CWINDOW *window;
  QWidget *cont;

  if (sg == 0)
    return;

  window = (CWINDOW *)CWidget::get(this);
  cont = window->container;

  sg->move(cont->rect().bottomRight() - sg->rect().bottomRight());
}

void MyMainWindow::setSizeGrip(bool on)
{
  if (on == (sg != 0))
    return;

  if (!on)
  {
    delete sg;
    sg = 0;
  }
  else //if (!parentWidget())
  {
    sg = new QSizeGrip(((CWINDOW *)CWidget::get(this))->container);
    sg->adjustSize();
    moveSizeGrip();
    sg->lower();
    sg->show();
  }
}

void MyMainWindow::setBorder(int b)
{
  int f;

  if (b == border || b < 0 || b > 2)
    return;

  if (b == BorderNone)
  {
    //clearWFlags(Qt::WStyle_NormalBorder);
    //clearWFlags(Qt::WStyle_DialogBorder);
    //setWFlags(Qt::WStyle_NoBorderEx);
    //reparent(parentWidget(), getWFlags(), pos());

    f = WStyle_Customize | WStyle_NoBorderEx | getWFlags(); // & 0xffff0000);
    
    /*if (f & WStyle_StaysOnTop)
      f |= WType_Popup;
    else
      f &= ~WType_Popup;*/
      
    if (!mdichild)
      f |= WType_TopLevel;

    doReparent(parentWidget(), f, pos());

    border = b;
    return;
  }


  if (border == BorderNone)
    doReparent(parentWidget(), (mdichild ? 0 : WType_TopLevel) | (getWFlags() /*& 0xffff0000*/), QPoint(0,0) );

  if (b == BorderFixed)
  {
    layout()->setResizeMode(QLayout::FreeResize);
    setMinimumSize(width(), height());
    setMaximumSize(width(), height());
  }
  else
  {
    setMinimumSize(0, 0);
    setMaximumSize(QWIDGETSIZE_MAX, QWIDGETSIZE_MAX);
    layout()->setResizeMode(QLayout::Minimum);
  }

  border = b;
}

/*
bool MyMainWindow::isFixed(void)
{
  QWidget *cont = centralWidget();

  return (!(cont->minimumWidth() == 0 && cont->minimumHeight() == 0));
}
*/

void MyMainWindow::paintUnclip(bool on)
{
  if (on)
    setWFlags(Qt::WPaintUnclipped);
  else
    clearWFlags(Qt::WPaintUnclipped);
}


void MyMainWindow::moveEvent(QMoveEvent *e)
{
  CWIDGET *_object = CWidget::getReal(this);

  //qDebug("Move");
  
  QMainWindow::moveEvent(e);

  //qDebug("Move (pos %d %d) (oldPos %d %d)", e->pos().x(), e->pos().y(), e->oldPos().x(), e->oldPos().y());
  //qDebug("     (geom %d %d) (fgeom %d %d)", geometry().x(), geometry().y(), frameGeometry().x(), frameGeometry().y());
  //qDebug("     Visible = %s Hidden = %s", (isVisible() ? "Yes" : "No"), (isHidden() ? "Yes" : "No"));
  //qDebug("     Flags = 0x%s State = 0x%s", QString::number(getWFlags(), 16).latin1(), QString::number(getWState(), 16).latin1());

  //if (CWIDGET_test_flag(ob, WF_IGNORE_MOVE))
  if (!testWFlags(Qt::WStyle_NoBorderEx))
  {
    if (geometry().x() == frameGeometry().x() && geometry().y() == frameGeometry().y())
    {
      //qDebug("...Ignore");
      return;
    }
  }

  if (!isHidden())
  {
    THIS->x = x();
    THIS->y = y();
    //qDebug("moveEvent: x= %d y = %d", x(), y());
  }

  //qDebug("moveEvent %ld %ld isHidden:%s shown:%s ", THIS->x, THIS->y, isHidden() ? "1" : "0", shown ? "1" : "0");
  
  if (shown)
    GB.Raise(THIS, EVENT_Move, 0);
}

/*
static void post_resize_event(CWINDOW *_object)
{
  qDebug("post resize: %d %d", THIS->w, THIS->h);
  WINDOW->resize(THIS->w, THIS->h);
  GB.Unref((void **)&_object);  
}
*/

void MyMainWindow::resizeEvent(QResizeEvent *e)
{
  CWINDOW *_object = (CWINDOW *)CWidget::getReal(this);
  //int w, h;

  //qDebug("Resize");
  
  //qDebug("Resize %p: %d %d <- %d %d", _object, e->size().width(), e->size().height(), e->oldSize().width(), e->oldSize().height());
  
  QMainWindow::resizeEvent(e);
  
  //if (e->size() == e->oldSize())
  //{
  //  qDebug("...Ignore");
  //  return;
  //}

  if (sg)
    moveSizeGrip();

  if (!isHidden())
  {
    updateGeometry();
    THIS->w = THIS->container->width();
    THIS->h = THIS->container->height();
    //qDebug("THIS->w = %d  THIS->h = %d", THIS->w, THIS->h);
  }

  //qDebug("resizeEvent %ld %ld isHidden:%s shown:%s ", THIS->w, THIS->h, isHidden() ? "1" : "0", shown ? "1" : "0");
  //qDebug("THIS->h = %ld  THIS->container->height() = %ld  height() = %ld", THIS->h, THIS->container->height(), height());
    
  if (shown)
  {
    /*w = THIS->w;
    h = THIS->h;*/
    GB.Raise(THIS, EVENT_Resize, 0);
    /*if (w != THIS->w || h != THIS->h)
    {
      GB.Ref(THIS);
      GB.Post((void (*)())post_resize_event, (long)THIS);
    }*/
  }
  
}


void MyMainWindow::keyPressEvent(QKeyEvent *e)
{
  QPushButton *test = 0;

  if (e->state() == 0 || ( e->state() & Keypad && e->key() == Key_Enter ))
  {
    switch (e->key())
    {
      case Key_Enter:
      case Key_Return:

        test = defaultButton();
        break;

      case Key_Escape:

        test = cancelButton();
        break;
    }

    if (test && CWidget::get(test))
    {
      if (test->isVisible() && test->isEnabled())
      {
        test->animateClick();
        return;
      }

      /*
      QObjectList *list = queryList( "QPushButton" );
      QObjectListIt it( *list );
      QPushButton *pb;

      while ((pb = (QPushButton*)it.current()))
      {
        if (pb == test)
        {
          delete list;
          if (pb->isVisible() && pb->isEnabled())
            pb->animateClick();
          return;
        }
        ++it;
      }

      delete list;
      */
    }
  }

  e->ignore();
}


static bool closeAll()
{
  CWINDOW *win;
  QPtrDictIterator<CWINDOW> iter(CWindow::dict);
  
  //qDebug("CLOSE ALL");
  
  for(;;)
  {
    win = iter.current();
    if (!win)
      break;
    if (win != CWINDOW_Main && do_close(win, 0))
    {
      //qDebug("ABORTED %p", win);
      return true;
    }
    ++iter;
  }
  
  return false;
}

static void deleteAll()
{
  CWINDOW *win;
  QPtrDictIterator<CWINDOW> iter(CWindow::dict);
  
  //qDebug("DELETE ALL");
  
  for(;;)
  {
    win = iter.current();
    if (!win)
      break;
    
    ++iter;
    
    if (win != CWINDOW_Main)
    {
      //CWIDGET_set_flag(win, WF_CLOSED);
      //qDebug("post DELETE to %p", win);
      //qApp->postEvent(win->widget.widget, new QEvent(EVENT_CLOSE));
      CWIDGET_destroy((CWIDGET *)win);
      //GB.Post((void *)deleteAfter, win
      //delete win;
    }
  }
  
  //qApp->eventLoop()->processEvents(QEventLoop::AllEvents);
}

void MyMainWindow::closeEvent(QCloseEvent *e)
{
  CWINDOW *_object = (CWINDOW *)CWidget::get(this);
  bool cancel;

  e->ignore();

  //if (qApp->loopLevel() != THIS->level)
  //  return;

  //if (_object == CWINDOW_Main && qApp->loopLevel() > 1)
  //  return;

  //qDebug("closeEvent: level = %d", qApp->loopLevel());

  if (MAIN_in_wait)
    goto IGNORE;

  if (CWINDOW_Current && (THIS != CWINDOW_Current))
    goto IGNORE;

  if (!THIS)
  {
    qWarning("closeEvent: THIS == NULL");
    goto IGNORE;
  }
  
  CWIDGET_set_flag(_object, WF_IN_CLOSE);
  cancel = GB.Raise(_object, EVENT_Close, 0);
  CWIDGET_clear_flag(_object, WF_IN_CLOSE);

  if (!cancel && THIS == CWINDOW_Main)
  {
    if (closeAll())
      cancel = true;
  }

  CWIDGET_set_flag(THIS, WF_CLOSED);
  
  if (!CWIDGET_test_flag(_object, WF_PERSISTENT))
  {

    if (cancel)
      goto IGNORE;

    if (CWINDOW_Main == THIS)
    {
      deleteAll();
      CWINDOW_Main = 0;
    }
      
    CWIDGET_destroy((CWIDGET *)THIS);
  }

  //qDebug("Accept !");
  e->accept();

  if (testWFlags(Qt::WShowModal))
  {
    //qDebug("exit_loop");
    qApp->eventLoop()->exitLoop();
  }

  return;

IGNORE:

  CWIDGET_clear_flag(THIS, WF_CLOSED);
  e->ignore();
}



bool MyMainWindow::isPersistent(void)
{
  return !testWFlags(WDestructiveClose);
}


void MyMainWindow::setPersistent(bool pers)
{
  if (!pers)
    setWFlags(WDestructiveClose);
  else
    clearWFlags(WDestructiveClose);
}


void MyMainWindow::defineMask()
{
  CWINDOW *_object = (CWINDOW *)CWidget::get(this);
  QPixmap *p;
  //QBitmap b;

  if (!THIS->mask)
  {
    clearMask();
    THIS->container->setErasePixmap(0);
  }
  else
  {
    p = THIS->mask->pixmap;
  
    THIS->container->setErasePixmap(*p);
  
    if (p->hasAlpha())
    {
      //b = *(p->mask());
      setMask(*(p->mask()));
      //XShapeCombineMask( x11Display(), winId(), ShapeBounding, 0, 0, b.handle(), ShapeSet );
    }
    else
      clearMask();
  }
  
  //doReparent(parentWidget(), getWFlags(), pos());
}


void MyMainWindow::doReparent(QWidget *parent, WFlags f, const QPoint &pos)
{
  bool hasIcon;
  QPixmap p;
  CWINDOW *_object = (CWINDOW *)CWidget::get(this);
  //qDebug("reparent %p to %p", this, parent);
  
  hasIcon = icon() != 0;
  if (hasIcon)
    p = *icon();
  
  reparent(parent, f, pos);
  setTopOnly(THIS->topOnly);
  setSkipTaskbar(THIS->skipTaskbar);
  
  if (hasIcon)
    setIcon(p);
  //qDebug("new parent = %p", parentWidget());
}


void MyMainWindow::center(bool force = false)
{
  QPoint p;
  
  if (!force && !mustCenter)
    return;
    
  mustCenter = false;
        
  p.setX((qApp->desktop()->width() - width()) / 2);
  p.setY((qApp->desktop()->height() - height()) / 2);
  
  move(p);
}

/***************************************************************************

  CWindow

***************************************************************************/

CWindow CWindow::manager;
int CWindow::count = 0;
QPtrDict<CWINDOW> CWindow::dict;

static void post_activate_event(void *ob)
{
  GB.Raise(ob, EVENT_Activate, 0);
  GB.Unref(&ob);
}

static void post_deactivate_event(void *ob)
{
  GB.Raise(ob, EVENT_Deactivate, 0);
  GB.Unref(&ob);
}

bool CWindow::eventFilter(QObject *o, QEvent *e)
{
  CWINDOW *ob;
  //bool cancel;

  ob = (CWINDOW *)CWidget::get(o);
  if (ob != NULL)
  {
    if (e->type() == QEvent::WindowActivate)
    {
      if (ob->window)
      {
        //qDebug("Activate: CWINDOW_Current = %p  ob = %p", CWINDOW_Current, ob);
        #if QT_VERSION >= 0x030100
        if ((ob == CWINDOW_Current) || (qApp->eventLoop()->loopLevel() <= 1))
        #else
        if ((ob == CWINDOW_Current) || (qApp->loopLevel() <= 1))
        #endif
        //GB.Raise(ob, EVENT_Activate, 0);
        
        if (GB.CanRaise(ob, EVENT_Activate))
        {
          GB.Ref(ob);
          GB.Post((void (*)())post_activate_event, (long)ob);
        }
      }
    }
    else if (e->type() == QEvent::WindowDeactivate)
    {
      if (ob->window)
      {
        //qDebug("Deactivate: CWINDOW_Current = %p  ob = %p", CWINDOW_Current, ob);
        #if QT_VERSION >= 0x030100
        if ((ob == CWINDOW_Current) || (qApp->eventLoop()->loopLevel() <= 1))
        #else
        if ((ob == CWINDOW_Current) || (qApp->loopLevel() <= 1))
        #endif
        //GB.Raise(ob, EVENT_Deactivate, 0);
        
        //((MyMainWindow *)o)->setLastFocus(qApp->focusWidget());
        
        if (GB.CanRaise(ob, EVENT_Deactivate))
        {
          GB.Ref(ob);
          GB.Post((void (*)())post_deactivate_event, (long)ob);
        }
      }
    }
    else if (e->type() == QEvent::Show)
    {
      MyMainWindow *w = (MyMainWindow *)o;
      
      if (ob->window)
        w->center();
      
      GB.Raise(ob, EVENT_Show, 0);
      if (ob->focus)
      {
        ob->focus->widget->setFocus();
        GB.Unref((void **)&ob->focus);
        ob->focus = NULL;
      }
    }
    else if (e->type() == QEvent::Hide)
    {
      GB.Raise(ob, EVENT_Hide, 0);
    }
    /*else if (e->type() == EVENT_CLOSE)
    {
      qDebug("EVENT_CLOSE %p", ob);
      s o;
        
      return true;
    }*/
  }

  return QObject::eventFilter(o, e);    // standard event processing
}



/***************************************************************************

  CWorkspace

***************************************************************************/

CWorkspace CWorkspace::manager;

bool CWorkspace::eventFilter(QObject *o, QEvent *e)
{
  CWINDOW *ob;
  QWidget *w;

  if (e->type() == QEvent::Move)
  {
    qDebug("CWorkspace: got move on %p %s", o, o->className());
    w = (QWidget *)qt_find_obj_child(o, "MyMainWindow", "");
    if (w)
    {
      ob = (CWINDOW *)CWidget::get(w);
      if (ob != NULL)
      {
        qDebug("CWorkspace: move");
        ((MyMainWindow *)w)->moveEvent((QMoveEvent *)e);
      }
    }
  }

  return QObject::eventFilter(o, e);    // standard event processing
}



/***************************************************************************

  MyEmbeddedWindow

***************************************************************************/


MyEmbeddedWindow::MyEmbeddedWindow(QWidget *parent) :
  MyContainer(parent)
{
  shown = false;
}


MyEmbeddedWindow::~MyEmbeddedWindow()
{
  void *ob = CWidget::get(this);
  do_close((CWINDOW *)ob, 0, true);
}


static void post_show_event(void *_object)
{
  //qDebug("post_show_event %p", WINDOW);
  GB.Raise(THIS, EVENT_Open, 0);
  GB.Raise(THIS, EVENT_Move, 0);
  GB.Raise(THIS, EVENT_Resize, 0);
  GB.Unref(&_object);
}


void MyEmbeddedWindow::show()
{
  //qDebug("MyEmbeddedWindow::show: %p", this);

  MyContainer::show();
}


void MyEmbeddedWindow::showEvent(QShowEvent *e)
{
  CWIDGET *ob = CWidget::getReal(this);
  
  //qDebug("MyEmbeddedWindow::showEvent: %p", this);
  
  MyContainer::showEvent(e);
  
  if (!shown)
  {
    //qDebug("MyEmbeddedWindow::showEvent: post_show_event");
    GB.Ref(ob);
    GB.Post((void (*)())post_show_event, (long)ob);
    shown = true;
  }
}


void MyEmbeddedWindow::resizeEvent(QResizeEvent *e)
{
  //qDebug("MyEmbeddedWindow::resizeEvent: %p (%d x %d)", this, e->size().width(), e->size().height());
  
  MyContainer::resizeEvent(e);
  
  if (shown)
    GB.Raise(CWidget::getReal(this), EVENT_Resize, 0);
  
  //qDebug("--> (%d %d) %s", this->width(), this->height(), shown ? "1" : "0");
}


void MyEmbeddedWindow::moveEvent(QMoveEvent *e)
{
  CWIDGET *ob = CWidget::getReal(this);

  //qDebug("embedded Move %p (%d %d)", this, e->pos().x(), e->pos().y());
  
  MyContainer::moveEvent(e);

  if (shown)
    GB.Raise(ob, EVENT_Move, 0);
}
