/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 * Lucas Beeler <lucas@yorba.org>
 */

#include "media-monitor.h"
#include "media-collection.h"
#include "media-source.h"

#include <QDir>
#include <QElapsedTimer>
#include <QFileInfo>
#include <QSet>
#include <QString>

/*!
 * \brief MediaMonitor::MediaMonitor
 */
MediaMonitor::MediaMonitor(QObject *parent)
    : QObject(parent),
      m_workerThread(this)
{
    m_worker = new MediaMonitorWorker();
    m_worker->moveToThread(&m_workerThread);
    QObject::connect(&m_workerThread, SIGNAL(finished()),
                     m_worker, SLOT(deleteLater()));

    QObject::connect(m_worker, SIGNAL(mediaItemAdded(QString)),
                     this, SIGNAL(mediaItemAdded(QString)), Qt::QueuedConnection);
    QObject::connect(m_worker, SIGNAL(mediaItemRemoved(qint64)),
                     this, SIGNAL(mediaItemRemoved(qint64)), Qt::QueuedConnection);

    m_workerThread.start(QThread::LowPriority);
}

/*!
 * \brief MediaMonitor::~MediaMonitor
 */
MediaMonitor::~MediaMonitor()
{
    m_workerThread.quit();
    m_workerThread.wait();
}

/*!
 * \brief MediaMonitor::startMonitoring starts monitoring the given directories
 * new and delted files
 * \param targetDirectories
 */
void MediaMonitor::startMonitoring(const QStringList &targetDirectories)
{
    QMetaObject::invokeMethod(m_worker, "startMonitoring", Qt::QueuedConnection,
                              Q_ARG(QStringList, targetDirectories));
}

/*!
 * \brief MediaMonitor::checkConsitency checks the given datastructure, if it is
 * in sync with the file system (files got added, deleted meanwhile)
 * \param mediaCollection
 */
void MediaMonitor::checkConsitency(const MediaCollection *mediaCollection)
{
    m_worker->setMediaCollection(mediaCollection);
    QMetaObject::invokeMethod(m_worker, "checkConsitency", Qt::QueuedConnection);
}

/*!
 * \brief MediaMonitor::setMonitoringOnHold when true, all actions based on file
 * activity are delayed until it's set back to true
 * \param onHold
 */
void MediaMonitor::setMonitoringOnHold(bool onHold)
{
    m_worker->setMonitoringOnHold(onHold);
}


/*!
 * \brief MediaMonitor::MediaMonitor
 */
MediaMonitorWorker::MediaMonitorWorker(QObject *parent)
    : QObject(parent),
      m_targetDirectories(),
      m_watcher(this),
      m_manifest(),
      m_fileActivityTimer(this),
      m_onHold(false)
{
    QObject::connect(&m_watcher, SIGNAL(directoryChanged(const QString&)), this,
                     SLOT(onDirectoryEvent(const QString&)));

    m_fileActivityTimer.setSingleShot(true);
    m_fileActivityTimer.setInterval(100);
    QObject::connect(&m_fileActivityTimer, SIGNAL(timeout()), this,
                     SLOT(onFileActivityCeased()));
}

/*!
 * \brief MediaMonitor::~MediaMonitor
 */
MediaMonitorWorker::~MediaMonitorWorker()
{
}

/*!
 * \brief MediaMonitorWorker::setMediaCollection
 * \param mediaCollection
 */
void MediaMonitorWorker::setMediaCollection(const MediaCollection *mediaCollection)
{
    m_mediaCollection = mediaCollection;
}

/*!
 * \brief MediaMonitorWorker::setMonitoringOnHold
 * \param onHold
 */
void MediaMonitorWorker::setMonitoringOnHold(bool onHold)
{
    m_onHold = onHold;
}

/*!
 * \brief MediaMonitor::startMonitoring
 * \param targetDirectories
 */
void MediaMonitorWorker::startMonitoring(const QStringList &targetDirectories)
{
    QStringList newDirectories;
    foreach (const QString& dir, targetDirectories) {
        if (!m_targetDirectories.contains(dir))
            newDirectories.append(dir);
    }
    m_targetDirectories += newDirectories;
    m_manifest = getManifest(m_targetDirectories);
    m_watcher.addPaths(newDirectories);
}

/*!
 * \brief MediaMonitorWorker::checkConsitency
 * \param mediaCollection
 */
void MediaMonitorWorker::checkConsitency()
{
    checkForRemovedMedias();
    checkForNewMedias();
}

/*!
 * \brief MediaMonitor::onDirectoryEvent
 * \param eventSource
 */
void MediaMonitorWorker::onDirectoryEvent(const QString& eventSource)
{
    m_fileActivityTimer.start();
}

/*!
 * \brief MediaMonitor::onFileActivityCeased
 */
void MediaMonitorWorker::onFileActivityCeased()
{
    if (m_onHold) {
        m_fileActivityTimer.start();
        return;
    }

    QStringList new_manifest = getManifest(m_targetDirectories);

    QStringList added = subtractManifest(new_manifest, m_manifest);
    for (int i = 0; i < added.size(); i++)
        emit mediaItemAdded(added.at(i));

    QStringList removed = subtractManifest(m_manifest, new_manifest);
    for (int i = 0; i < removed.size(); i++) {
        QFileInfo file(removed.at(i));
        const MediaSource *media = m_mediaCollection->mediaFromFileinfo(file);
        if (media)
            emit mediaItemRemoved(media->id());
    }

    m_manifest = new_manifest;
}

/*!
 * \brief MediaMonitor::getManifest
 * \param dir
 * \return
 */
QStringList MediaMonitorWorker::getManifest(const QStringList &dirs)
{
    QStringList allFiles;
    foreach (const QString &dirName, dirs) {
        QDir dir(dirName);
        QStringList fileList = dir.entryList(QDir::Files, QDir::Time);
        foreach (const QString &fileName, fileList) {
            const QFileInfo fi(dirName + QDir::separator() + fileName);
            allFiles.append(fi.absoluteFilePath());
        }
    }
    return allFiles;
}

/*!
 * \brief MediaMonitor::subtractManifest
 * \param m1
 * \param m2
 * \return
 */
QStringList MediaMonitorWorker::subtractManifest(const QStringList& m1,
                                            const QStringList& m2)
{
    QSet<QString> result = QSet<QString>::fromList(m1);
    result.subtract(QSet<QString>::fromList(m2));
    return QStringList(result.toList());
}

/*!
 * \brief MediaMonitorWorker::checkForNewMedias checks for files in the filesystem
 * that are not in the datastructure
 * \param mediaCollection
 */
void MediaMonitorWorker::checkForNewMedias()
{
    foreach (const QString& file, m_manifest) {
        if (!m_mediaCollection->containsFile(file))
            emit mediaItemAdded(file);
    }
}

/*!
 * \brief MediaMonitorWorker::checkForRemovedMedias checks if there are files in
 * the datastructure, but not in the file system
 */
void MediaMonitorWorker::checkForRemovedMedias()
{
    const QList<DataObject*> medias = m_mediaCollection->getAll();
    foreach (const DataObject* obj, medias) {
        const MediaSource *media = qobject_cast<const MediaSource*>(obj);
        Q_ASSERT(media);
        if (!m_manifest.contains(media->file().absoluteFilePath()))
            emit mediaItemRemoved(media->id());
    }
}
