/*
 * Galaxium Messenger
 * Copyright (C) 2008 Ben Motmans <ben.motmans@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.IO;
using System.Xml;
using System.Collections.Generic;

using Galaxium.Core;
using Anculus.Core;

namespace Galaxium.Protocol
{
	public sealed class IndexedConversationLog : IConversationLog
	{
		private IndexedConversationLogMetaData _metaData;
		private List<IndexedConversationLogArchive> _archives;
		
		private string _directory;
		private string _logName;
		
		private bool _keepAlive;
		
		internal IndexedConversationLog (string directory, string logName, bool keepAlive)
		{
			_directory = directory;
			_logName = logName;
			_keepAlive = keepAlive;
			
			_archives = new List<IndexedConversationLogArchive> ();
			_metaData = new IndexedConversationLogMetaData (this, Path.Combine (directory, logName + ".index"));
		}
		
		public bool KeepAlive
		{
			get { return _keepAlive; }
			set { _keepAlive = value; }
		}
		
		internal List<IndexedConversationLogArchive> Archives
		{
			get { return _archives; }
		}
		
		internal IndexedConversationLogMetaData MetaData
		{
			get { return _metaData; }
		}
		
		internal string Directory
		{
			get { return _directory; }
		}
		
		internal string LogName
		{
			get { return _logName; }
		}

		public void LogMessage (ITextMessage msg)
		{
			IndexedConversationLogArchive archive = GetActiveArchive ();
			archive.LogMessage (msg);
		}

		public void LogEvent (DateTime timestamp, string description)
		{
			IndexedConversationLogArchive archive = GetActiveArchive ();
			archive.LogEvent (timestamp, description);
		}
		
		public void Close ()
		{
			foreach (IndexedConversationLogArchive archive in _archives)
				archive.Close ();
		}
		
		public IEnumerable<ConversationLogEntry> GetNLastEntries (int n)
		{
			if (n <= 0)
				throw new ArgumentOutOfRangeException ("n", "The number of entries must be > 0.");

			int count = 0;

			List<ConversationLogEntry> entries = new List<ConversationLogEntry> ();
			for (int i = (_archives.Count - 1); i >= 0; i--) {
				IndexedConversationLogArchive archive = _archives[i];

				foreach (ConversationLogEntry entry in archive.GetNLastEntries (n - count)) {
					entries.Add (entry);
					count++;
					
					if (count == n)
						return entries;
				}
			}
			
			return entries;
		}

		public IEnumerable<ConversationLogEntry> SearchAll (string keyword)
		{
			if (String.IsNullOrEmpty (keyword))
				throw new ArgumentException ("'keyword' can't be null or an empty string.");
			
			int n = _archives.Count;
			List<ConversationLogEntry> entries = new List<ConversationLogEntry> ();
			for (int i=0; i<n; i++) {
				IndexedConversationLogArchive archive = _archives[i];
				
				entries.AddRange (archive.SearchAll (keyword));
			}
			return entries;
		}

		public ConversationLogEntry Search (string keyword)
		{
			if (String.IsNullOrEmpty (keyword))
				throw new ArgumentException ("'keyword' can't be null or an empty string.");

			int n = _archives.Count;
			for (int i=0; i<n; i++) {
				IndexedConversationLogArchive archive = _archives[i];
				
				ConversationLogEntry entry = archive.Search (keyword);
				if (entry != null)
					return entry;
			}
			
			return null;
		}

		public ConversationLogEntry SearchNext (string keyword, ConversationLogEntry entry)
		{
			if (String.IsNullOrEmpty (keyword))
				throw new ArgumentException ("'keyword' can't be null or an empty string.");
			if (entry == null)
				throw new ArgumentNullException ("entry");
			
			int n = _archives.Count;
			int start = entry.ArchiveIndex;
			
			for (int i=start; i<n; i++) {
				IndexedConversationLogArchive archive = _archives[i];
				
				ConversationLogEntry next = archive.SearchNext (keyword, entry);
				if (next != null)
					return next;
			}
			
			return null;
		}
		
		private IndexedConversationLogArchive GetActiveArchive ()
		{
			if (_archives.Count == 0)
				return CreateArchive ();
			
			IndexedConversationLogArchive last = _archives[_archives.Count - 1];
			if (last.IsMaximumSizeReached ())
				return CreateArchive ();
			return last;
		}
		
		private IndexedConversationLogArchive CreateArchive ()
		{
			int archiveIndex = _archives.Count;
			
			string filename = null;
			string absoluteFilename = null;
			
			bool exists = true;
			while (exists) {
				filename = String.Concat (_logName, FileUtility.GetRandomFileName (), ".log");
				absoluteFilename = Path.Combine (_directory, filename);
				
				exists = File.Exists (absoluteFilename);
			}
			
			IndexedConversationLogArchive archive = new IndexedConversationLogArchive (this, absoluteFilename, filename, archiveIndex);
			AddArchive (archive);
			return archive;
		}
		
		private void AddArchive (IndexedConversationLogArchive archive)
		{
			_archives.Add (archive);

			if (_archives.Count > _metaData.LogArchiveCount)
				_archives.RemoveAt (0);
			
			_metaData.WriteIndex ();
		}
		
		internal void DeleteLogArchives (string error)
		{
			foreach (string file in System.IO.Directory.GetFiles (_directory, _logName + "*")) {
				Log.Warn ("Deleting log archive '{0}'. ({1})", file, error);
				File.Delete (file);
			}
			_archives.Clear ();
		}
	}
}