/*
 * Galaxium Messenger
 * Copyright (C) 2007 Ben Motmans <ben.motmans@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Threading;
using System.Collections.Generic;

using Mono.Addins;
using Galaxium.Core;

namespace Galaxium.Client
{
	public static class SongPlayingUtility
	{
		//TODO: use WeakReference + byref variable for monitor start request
		//if a monitor is null, remove it and recheck the need for the polling thread
		public static event EventHandler<SongInformationEventArgs> SongChanged;
		
		private static IList<ISongInformationBackend> _backends;
		private static IList<object> _monitors;
		
		private static Thread _thread;
		
		static SongPlayingUtility ()
		{
			_monitors = new List<object> ();
			_backends = new List<ISongInformationBackend> ();

			foreach (SongInformationBackendExtension ext in AddinManager.GetExtensionNodes ("/Galaxium/Backends/SongInformation")) {
				ISongInformationBackend backend = ext.SongInformationBackend;
				if (backend != null) {
					backend.SongChanged += BackendSongChanged;
					_backends.Add (backend);
				}
			}
		}

		public static void StartMonitorRequest (object identifier)
		{
			ThrowUtility.ThrowIfNull ("identifier", identifier);

			if (!_monitors.Contains (identifier)) {
				_monitors.Add (identifier);

				if (_monitors.Count == 1)
					StartListening ();
			}
		}

		public static void StopMonitorRequest (object identifier)
		{
			ThrowUtility.ThrowIfNull ("identifier", identifier);

			_monitors.Remove (identifier);
			if (_monitors.Count == 0)
				StopListening ();
		}

		private static ISongInformationBackend FindActiveSongInformationBackend ()
		{
			foreach (ISongInformationBackend backend in _backends) {
				if (backend.IsAvailable ())
					return backend;
			}
			return null;
		}
		
		private static void StartListening ()
		{
			_thread = new Thread (new ThreadStart (StartListeningAsync));
			_thread.Priority = ThreadPriority.Lowest;
			_thread.IsBackground = true;
			_thread.Start ();
		}
		
		private static void StartListeningAsync ()
		{
			bool backendActive = false;
			ISongInformationBackend backend = null;
			
			while (true) {
				while (!backendActive) {
					backend = FindActiveSongInformationBackend ();
					backendActive = backend != null;
					
					Thread.Sleep (60000);
				}
				
				while (backendActive) {
					if (backend.RequiresPolling) {
						SongInformation info = backend.GetCurrentSong ();
						if (info != null) {
							ThreadUtility.SyncDispatch (new VoidDelegate (delegate
							{
								SongInformationEventArgs args = new SongInformationEventArgs (info);
								BackendSongChanged (backend, args);
							}));
						}
					}
					
					backendActive = backend.IsAvailable ();
					Thread.Sleep (30000);
				}
			}
		}
		
		private static void BackendSongChanged (object sender, SongInformationEventArgs args)
		{
			if (SongChanged != null)
				SongChanged (sender, args);
		}
		
		private static void StopListening ()
		{
			if (_thread != null) {
				try {
					_thread.Abort ();
				} catch {
				} finally {
					_thread = null;
				}
			}
		}
	}
}