/* GADMIN-DHCPD - An easy to use GTK+ frontend for ISC DHCPD.
 * Copyright (C) 2004 - 2009 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "gettext.h"
#include "allocate.h"
#include "functions.h"
#include "failover_functions.h"

extern char DHCPD_CONF_BUF[1024];



/* Does a pool with this name exist */
int failover_name_exists(gchar *name)
{
    FILE *fp;
    char *line;
    long conf_size = 0;
    int retval = 0;
    gchar *failover_name = g_strdup_printf("\"%s\"", name);

    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
        printf("Couldnt find: %s\n", DHCPD_CONF_BUF);

	g_free(failover_name);
        return retval;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( strstr(line, "subnet") && strstr(line, "netmask") && strstr(line, "{") )
	  break;
    
	if( strstr(line, "failover") && strstr(line, "peer")
	&& strstr(line, failover_name) )
	{
	    retval = 1;
	    break;	
	} 
    }

    g_free(failover_name);

    return retval;
}



/* Does a failover declaration and corresponding pool exist in this scope */
int failover_exists(gchar *nic, gchar *subnet, gchar *netmask)
{
    FILE *fp;
    char *line, *buffy;
    long conf_size = 0;
    int found_scope = 0;
    int found_failover_peer = 0;
    int found_failover_decl = 0;
    int retval = 0;

    gchar *scope_line1 = g_strdup_printf("subnet %s netmask %s\n", subnet, netmask);
    gchar *scope_line2 = g_strdup_printf("subnet %s netmask %s{\n", subnet, netmask);
    gchar *scope_line3 = g_strdup_printf("subnet %s netmask %s {\n", subnet, netmask);
    gchar *nic_line = g_strdup_printf("interface %s;\n", nic);


    /* Find the correct scope */
    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
        printf("Couldnt find: %s\n", DHCPD_CONF_BUF);

	g_free(scope_line1);
	g_free(scope_line2);
	g_free(scope_line3);
	g_free(nic_line);

        return retval;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);
    buffy = allocate(1024);

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( (strstr(line, scope_line1) || strstr(line, scope_line2) 
	|| strstr(line, scope_line3) ) && ! strstr(line, "#") )
	{
	    while(fgets(line, conf_size, fp)!=NULL)
	    {
		if( strstr(line, nic_line) && ! strstr(line, "#") )
		{
		    found_scope = 1;
		    break;
		}
	    }
	}
	if( found_scope )
	  break;
    }

    /* Locate the correct failover peer and save the name */
    if( found_scope && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	/* Break at scope end */
	if( strstr(line, "}") && ! strstr(line, "#") && ! strstr(line, "pool") )
	  break;

	if( strlen(line)>1000 )
	  continue;

	remove_semicolon(line);

	if( strstr(line, "failover peer") && ! strstr(line, "#") )
	{
    	    sscanf(line, "%*s %*s %s", buffy);
	    found_failover_peer = 1;
	    break;
	}
    }

    rewind(fp);
    
    /* We have found the name of the failover declaration, scroll to it and save its first line */
    if( found_failover_peer && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	/* Found the failover declaration */
	if( strstr(line, "failover") && strstr(line, "peer") && strstr(line, buffy) )
	{
	    found_failover_decl = 1;
	    break;	
	}
    }

    
    if( found_failover_peer && found_failover_decl )
      retval = 1;

    g_free(scope_line1);
    g_free(scope_line2);
    g_free(scope_line3);
    g_free(nic_line);
    
    free(buffy);
    free(line);

    return retval;
}



/* Return the entire failover declaration */
gchar * get_entire_failover(gchar *nic, gchar *subnet, gchar *netmask)
{
    FILE *fp;
    char *line, *buffy, *failover;
    long conf_size = 0;
    int found_scope = 0;
    int found_failover_peer = 0;
    int found_failover_decl = 0;
    int i=0, z=0;

    gchar *entire_failover = NULL;

    gchar *scope_line1 = g_strdup_printf("subnet %s netmask %s\n", subnet, netmask);
    gchar *scope_line2 = g_strdup_printf("subnet %s netmask %s{\n", subnet, netmask);
    gchar *scope_line3 = g_strdup_printf("subnet %s netmask %s {\n", subnet, netmask);
    gchar *nic_line = g_strdup_printf("interface %s;\n", nic);

    /* Find the correct scope */
    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
        printf("Couldnt open dhcpd.conf here:\n%s\n", DHCPD_CONF_BUF);

        return entire_failover;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);
    buffy = allocate(1024);
    failover = allocate(4096);

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( (strstr(line, scope_line1) || strstr(line, scope_line2) 
	|| strstr(line, scope_line3) ) && ! strstr(line, "#") )
	{
	    while(fgets(line, conf_size, fp)!=NULL)
	    {
		if( strstr(line, nic_line) && ! strstr(line, "#") )
		{
		    found_scope = 1;
		    break;
		}
	    }
	}
	if( found_scope )
	  break;
    }

    /* Locate the correct failover peer and save the name */
    if( found_scope && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	/* Break at scope end */
	if( strstr(line, "}") && ! strstr(line, "#") && ! strstr(line, "pool") )
	  break;

	if( strlen(line)>1000 )
	  continue;

	remove_semicolon(line);

	if( strstr(line, "failover peer") && ! strstr(line, "#") )
	{
	    /* Find the beginning of the pool name */
	    for(i=0; line[i]!='\0'; i++)
	       if( line[i]=='"')
	         break;

	    i++;
	    snprintf(buffy, 1000, "%s", &line[i]);

	    /* Find the end */
	    for(z=i; line[z]!='\0'; z++)
	       if( line[z]=='"')
	         break;

	    /* Cut it at the end */
	    buffy[z-i]='\0';

	    found_failover_peer = 1;
	    break;
	}
    }

    rewind(fp);
    
    /* We have found the name of the failover declaration, scroll to it and save its first line */
    if( found_failover_peer && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	/* Found the failover declaration */
	if( strstr(line, "failover") && strstr(line, "peer") && strstr(line, buffy) )
	{
	    found_failover_decl = 1;
	    strcat(failover, line);
	    break;	
	}
    }

    /* Save the rest of the failover */
    if( found_failover_peer && found_failover_decl && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( strstr(line, "}") )
	  break;

	strcat(failover, line);
    }
    fclose(fp);

    
    if( found_failover_peer && found_failover_decl )
      entire_failover = g_strdup_printf("%s}\n", failover);
    
    free(buffy);
    free(failover);
    free(line);

    return entire_failover;
}



/* "What" is any option in the failover decl such as: peer address, port etc  */
gchar * get_failover(gchar *nic, gchar *subnet, gchar *netmask, char what[1000])
{
    FILE *fp;
    char *line, *buffy, *option;
    long conf_size = 0;
    int found_scope = 0;
    int found_failover_peer = 0;
    int found_failover_decl = 0;
    int i=0, z=0;

    gchar *failover_peer = NULL;
    gchar *ret_entire;

    gchar *scope_line1 = g_strdup_printf("subnet %s netmask %s\n", subnet, netmask);
    gchar *scope_line2 = g_strdup_printf("subnet %s netmask %s{\n", subnet, netmask);
    gchar *scope_line3 = g_strdup_printf("subnet %s netmask %s {\n", subnet, netmask);
    gchar *nic_line = g_strdup_printf("interface %s;\n", nic);


    /* Return the entire failover declaration */
    if( strstr(what, "return-entire-declaration") )
    {
	ret_entire = get_entire_failover(nic, subnet, netmask);

	g_free(scope_line1);
	g_free(scope_line2);
	g_free(scope_line3);
	g_free(nic_line);

	return ret_entire;    
    }
    

    /* Find the correct scope */
    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
	g_free(scope_line1);
	g_free(scope_line2);
	g_free(scope_line3);
	g_free(nic_line);

        return failover_peer;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);
    buffy = allocate(1024);
    option = allocate(1024);

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( (strstr(line, scope_line1) || strstr(line, scope_line2) 
	|| strstr(line, scope_line3) ) && ! strstr(line, "#") )
	{
	    while(fgets(line, conf_size, fp)!=NULL)
	    {
		if( strstr(line, nic_line) && ! strstr(line, "#") )
		{
		    found_scope = 1;
		    break;
		}
	    }
	}
	if( found_scope )
	  break;
    }

    /* Locate the pool name in the scope and save it */
    if( found_scope && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	/* Break at scope end */
	if( strstr(line, "}") && ! strstr(line, "#") && ! strstr(line, "pool") )
	  break;

	if( strlen(line)>1000 )
	  continue;

	remove_semicolon(line);

	if( strstr(line, "failover peer") && ! strstr(line, "{") && ! strstr(line, "#") )
	{
	    /* Find the beginning of the pool name */
	    for(i=0; line[i]!='\0'; i++)
	       if( line[i]=='"')
	         break;

	    i++;
	    snprintf(buffy, 1000, "%s", &line[i]);

	    /* Find the end */
	    for(z=i; line[z]!='\0'; z++)
	       if( line[z]=='"')
	         break;

	    /* Cut it at the end */
	    buffy[z-i]='\0';

	    found_failover_peer = 1;

	    /* Return the declared failover name */
	    if( strstr(what, "return-failover-name") )
	    {
		snprintf(option, 1000, "%s", &buffy[0]);

		g_free(scope_line1);
		g_free(scope_line2);
		g_free(scope_line3);
		g_free(nic_line);

		failover_peer = g_strdup_printf("%s", option);

		free(option);
		free(buffy);
		free(line);
		fclose(fp);

		return failover_peer;
	    }

	    break;
	}
    }

    rewind(fp);
    
    /* We have found the failover declarations name, scroll to it */
    if( found_failover_peer && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	/* Found the failover declaration */
	if( strstr(line, "failover") && strstr(line, "peer") && strstr(line, buffy) )
	{
	    found_failover_decl = 1;
	    break;	
	}
    }

    /* We are at the failover declaration, now we want its options */
    if( found_failover_decl && conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( strstr(line, "}") && ! strstr(line, "#") )
	  break;

	if( strstr(line, what) )
	{
	    if( strstr(what, "primary;") )
	    {
    		sscanf(line, "%s", buffy);
		snprintf(option, 1000, "%s", buffy);
		break;
	    }

	    if( strstr(what, "secondary;") )
	    {
    		sscanf(line, "%s", buffy);
		snprintf(option, 1000, "%s", buffy);
		break;
	    }

	    if( strstr(what, "address") && ! strstr(what, "peer address") )
	    {
    		sscanf(line, "%*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "port") && ! strstr(what, "peer port") )
	    {
    		sscanf(line, "%*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "peer address") )
	    {
    		sscanf(line, "%*s %*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "peer port") )
	    {
    		sscanf(line, "%*s %*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "max-response-delay") )
	    {
    		sscanf(line, "%*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "max-unacked-updates") )
	    {
    		sscanf(line, "%*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "load balance max seconds") )
	    {
    		sscanf(line, "%*s %*s %*s %*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "mclt") )
	    {
    		sscanf(line, "%*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    if( strstr(what, "split") )
	    {
    		sscanf(line, "%*s %s", buffy);
		snprintf(option, 1000, "%s", buffy);
		if( option[strlen(option)-1]==';' )
		  option[strlen(option)-1]='\0';
		break;
	    }

	    /* Add more here if needed... */
	}
    }
    fclose(fp);
    free(line);
    free(buffy);

    failover_peer = g_strdup_printf("%s", option);
    free(option);

    return failover_peer;
}
