/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: FirewallDialog.cpp,v 1.42 2007/04/14 00:18:43 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"
#include "platforms.h"

#include "FWBTree.h"
#include "FirewallDialog.h"
#include "ObjectManipulator.h"
#include "DialogFactory.h"
#include "FWWindow.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/Management.h"
#include "fwbuilder/FWException.h"
#include "fwbuilder/Resources.h"

#include <vector>
#include <map>

#include <qlineedit.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qtextedit.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qiconview.h>
#include <qtabwidget.h>
#include <qlabel.h>
#include <qtimer.h>

#include <iostream>

using namespace std;
using namespace libfwbuilder;

void FirewallDialog::loadFWObject(FWObject *o)
{
    obj=o;
    Firewall *s = dynamic_cast<Firewall*>(obj);
    assert(s!=NULL);

    init=true;

    fillLibraries(libs,obj);

/* fill in platform */
    setPlatform(platform, obj->getStr("platform").c_str() );

    fillVersion();

/* fill in host OS  */
    setHostOS(hostOS, obj->getStr("host_OS").c_str() );

/* ---------------- */

    Management *mgmt=s->getManagementObject();
    assert(mgmt!=NULL);

//    FWOptions  *opt =s->getOptionsObject();

    obj_name->setText( QString::fromUtf8(s->getName().c_str()) );
//    snmpCommunity->setText( mgmt->getSNMPManagement()->getReadCommunity().c_str() );

    comment->setTextFormat(QTextEdit::PlainText);

    comment->setText( QString::fromUtf8(s->getComment().c_str()) );

    inactive->setChecked(s->getInactive());

    //apply->setEnabled( false );


    obj_name->setEnabled(!o->isReadOnly());
    setDisabledPalette(obj_name);

    libs->setEnabled(!o->isReadOnly());
    setDisabledPalette(libs);

    platform->setEnabled(!o->isReadOnly());
    setDisabledPalette(platform);

    version->setEnabled(!o->isReadOnly());
    setDisabledPalette(version);

    hostOS->setEnabled(!o->isReadOnly());
    setDisabledPalette(hostOS);

    fwAdvanced->setEnabled(!o->isReadOnly());
    setDisabledPalette(fwAdvanced);

    osAdvanced->setEnabled(!o->isReadOnly());
    setDisabledPalette(osAdvanced);

    comment->setReadOnly(o->isReadOnly());
    setDisabledPalette(comment);

//    snmpCommunity->setEnabled(!o->isReadOnly());
//    setDisabledPalette(snmpCommunity);

    inactive->setEnabled(!o->isReadOnly());
    setDisabledPalette(inactive);
    

    init=false;
}

/* fill in version */
void FirewallDialog::fillVersion()
{
    version->clear();

    list<QStringPair> vl=getVersionsForPlatform( readPlatform(platform) );
    QString            v=obj->getStr("version").c_str();
    int cp=0;
    for (list<QStringPair>::iterator i1=vl.begin(); i1!=vl.end(); i1++,cp++)
    {
        if (fwbdebug)
            qDebug(QString("Adding version %1").arg(i1->second));

        version->insertItem( i1->second );
        if ( v == i1->first ) version->setCurrentItem( cp );
    }
}

void FirewallDialog::saveVersion()
{
    QString pl = readPlatform(platform);

    list<QStringPair> vl=getVersionsForPlatform( pl.latin1() );
    QString           v = version->currentText();
    list<QStringPair>::iterator li =
        std::find_if(vl.begin(),vl.end(),findSecondInQStringPair(v));
    if (li!=vl.end())
        obj->setStr("version", li->first.latin1() );

#if 0
    int cp=0;
    for (list<QStringPair>::iterator i1=vl.begin(); i1!=vl.end(); i1++,cp++)
    {
        if ( v == i1.data() )
        {
            obj->setStr("version", i1.key().latin1() );
            break;
        }
    }
#endif
}

void FirewallDialog::platformChanged()
{
    fillVersion();
    changed();

    QString so=Resources::platform_res[readPlatform(platform).latin1()]->getResourceStr("/FWBuilderResources/Target/supported_os").c_str();
    if (so.isEmpty()) return;

    QString ho=so.section(",",0);
    setHostOS( hostOS, ho.latin1() );

    QString pl = readPlatform(platform);
    fwAdvanced->setEnabled( pl!="unknown" );
}

void FirewallDialog::hostOSChanged()
{
    changed();
    QString ho = readHostOS(hostOS);
    osAdvanced->setEnabled( ho!="unknown_os" );
}

void FirewallDialog::changed()
{
    //apply->setEnabled( true );
    emit changed_sign();
}

void FirewallDialog::validate(bool *res)
{
    *res=true;
    if (!isTreeReadWrite(this,obj)) { *res=false; return; }
    if (!validateName(this,obj,obj_name->text())) { *res=false; return; }
}

void FirewallDialog::isChanged(bool *res)
{
    if (fwbdebug)
        qDebug("FirewallDialog::isChanged");
    //*res=(!init && apply->isEnabled());
}

void FirewallDialog::libChanged()
{
    changed();
}

void FirewallDialog::applyChanges()
{
    Firewall *s = dynamic_cast<Firewall*>(obj);
    Management *mgmt=s->getManagementObject();
    assert(mgmt!=NULL);

//    FWOptions  *opt =s->getOptionsObject();

    assert(s!=NULL);

    string oldname=obj->getName();
    string newname=string(obj_name->text().utf8());
    string oldplatform=obj->getStr("platform");

    string oldVer=obj->getStr("version");

    obj->setName( newname );
    obj->setComment( string(comment->text().utf8()) );
//    mgmt->getSNMPManagement()->setReadCommunity( snmpCommunity->text().latin1() );

    string pl = readPlatform(platform).latin1();
    obj->setStr("platform", pl );

    obj->setStr("host_OS", readHostOS(hostOS).latin1() );
    
    s->setInactive(inactive->isChecked());
    
    saveVersion();

    string newVer=obj->getStr("version");
        
    om->updateObjName(obj,QString::fromUtf8(oldname.c_str()));

    if (oldplatform!=pl || oldname!=newname || oldVer!=newVer)
    {
        if (fwbdebug)
            qDebug("FirewallDialog::applyChanges() scheduling call to reopenFirewall()");
        //mw->reopenFirewall();
        //QTimer::singleShot( 0, mw, SLOT(reopenFirewall()) );
        mw->scheduleRuleSetRedraw();
    }

    if (oldplatform!=pl)
    {
        if (fwbdebug)
            qDebug("FirewallDialog::applyChanges() platform has changed - clear option 'compiler'");
        Firewall *s = Firewall::cast(obj);
        assert(s!=NULL);
        FWOptions  *opt =s->getOptionsObject();
        opt->setStr("compiler","");
    }

    init=true;

/* move to another lib if we have to */
    if (! FWBTree::isSystem(obj) && libs->currentText() != QString(obj->getLibrary()->getName().c_str()))
        om->moveObject(libs->currentText(), obj);

    init=false;

    //apply->setEnabled( false );
    om->updateLastModifiedTimestampForAllFirewalls(s);

}

void FirewallDialog::discardChanges()
{
    loadFWObject(obj);
}

void FirewallDialog::openFWDialog()
{
    /*if (apply->isEnabled())*/ applyChanges();

    if (obj->getStr("version").empty()) saveVersion();

    try
    {
        QWidget *w = DialogFactory::createFWDialog(this,obj);
        if (w==NULL)   return;   // some dialogs may not be implemented yet
        QDialog *d=dynamic_cast<QDialog*>(w);
        assert(d!=NULL);

        d->exec();   // modal dialog, dialog saves data into the object
    }
    catch (FWException &ex)
    {
        QMessageBox::critical(
            this,"Firewall Builder",
            tr("FWBuilder API error: %1").arg(ex.toString().c_str()),
            tr("&Continue"), QString::null,QString::null,
            0, 1 );
        return;
    }
}


void FirewallDialog::openOSDialog()
{
    /*if (apply->isEnabled())*/ applyChanges();

    try
    {
        QWidget *w = DialogFactory::createOSDialog(this,obj);
        if (w==NULL)   return;   // some dialogs may not be implemented yet
        QDialog *d=dynamic_cast<QDialog*>(w);
        assert(d!=NULL);

        d->exec();   // modal dialog, dialog saves data into the object
    }
    catch (FWException &ex)
    {
        QMessageBox::critical(
            this,"Firewall Builder",
            tr("FWBuilder API error: %1").arg(ex.toString().c_str()),
            tr("&Continue"), QString::null,QString::null,
            0, 1 );
        return;
    }
}

/* ObjectEditor class connects its slot to this signal and does all
 * the verification for us, then accepts (or not) the event. So we do
 * nothing here and defer all the processing to ObjectEditor
 */
void FirewallDialog::closeEvent(QCloseEvent *e)
{
    emit close_sign(e);

}

