//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: PDESystem.hpp,v 1.3 2003/09/21 14:34:18 delpinux Exp $

#ifndef PDE_SYSTEM_HPP
#define PDE_SYSTEM_HPP

#include <vector>

#include <PDEProblem.hpp>
#include <ReferenceCounting.hpp>

/*!
  \class PDESystem

  This class implements systems of PDEs.

  \author Stphane Del Pino
*/

#include <Problem.hpp>

class PDESystem
  : public Problem
{
private:
  //! The list of PDE Problems.
  std::vector<ConstReferenceCounting<PDEProblem> > __pdeProblems;

public:
  void add(ReferenceCounting<PDEProblem> pdeProblem)
  {
    __pdeProblems.push_back(pdeProblem);
  }

  //! read only access to the ith problem.
  const PDEProblem& operator [] (const size_t i) const
  {
    assert(i<__pdeProblems.size());
    return *(__pdeProblems[i]);
  }

  const UserFunction& secondMember(const size_t& i) const
  {
    return (*(*__pdeProblems[i]).pde().secondMember());
  }

  const BoundaryConditionSet& boundaryConditionSet(const size_t& i) const
  {
    return (*__pdeProblems[i]).boundaryConditionSet();
  }

  ReferenceCounting<Problem> operator * (ConstReferenceCounting<UserFunction> u) const
  {
    PDESystem* newPDESystem = new PDESystem(*this);

    for (size_t i=0; i<this->numberOfEquations(); ++i) {
      (*newPDESystem).__pdeProblems[i] = (*__pdeProblems[i]) * u;
    }
    return newPDESystem;
  }

  const size_t numberOfUnknown() const
  {
    return __pdeProblems.size();
  }

  //! Read only access to the number of equations.
  const size_t numberOfEquations() const
  {
    return __pdeProblems.size();
  }

  //! Writes the system.
  friend std::ostream& operator << (std::ostream& os,
				    const PDESystem& pdeS)
  {
    for(size_t i=0; i<pdeS.numberOfEquations(); ++i) {
      os << *(pdeS.__pdeProblems[i]) << '\n';
    }
    return os;
  }

  //! Construction of the PDESystem.
  PDESystem ()
    : Problem(Problem::pdeProblem, 0)
  {
    ;
  }

  //! Construction of the PDESystem.
  PDESystem (const size_t& dimension)
    : Problem(Problem::pdeProblem, 0)
  {
    __pdeProblems.reserve(dimension);
  }

  /** 
   * Copy constructor
   * 
   * @param p a given problem
   * 
   */
  PDESystem(const PDESystem& p)
    : Problem(p),
      __pdeProblems(p.__pdeProblems)
  {
  }

  //! destructor
  ~PDESystem ()
  {
    ;
  }
};

#endif // PDE_SYSTEM_HPP

