//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FunctionExpression.hpp,v 1.18 2005/11/20 13:21:59 delpinux Exp $

#ifndef FUNCTION_EXPRESSION_HPP
#define FUNCTION_EXPRESSION_HPP

#include <Expression.hpp>
#include <RealExpression.hpp>
#include <Variable.hpp>

#include <TinyVector.hpp>
#include <FileDescriptor.hpp>

#include <EmbededFunctions.hpp>
#include <UserFunction.hpp>

#include <ErrorHandler.hpp>

#include <FEMFunction.hpp>

#include <DiscretizationType.hpp>
#include <Normal.hpp>

class Scene;

#include <vector>
#include <list>
#include <cmath>

class FunctionExpression
  : public Expression
{
public:
  enum FunctionType {
    functionValue,
    constant,
    normalComponent,
    fem,
    dx,
    meshElementsReferences,
    meshVerticesReferences,
    composed,
    variable,
    binaryOperator,
    unaryOperator,
    convection,
    characteristic,
    meshCharacteristic,
    domainCharacteristic,
    read,
    boolean
  };
private:
  //! Type of function
  FunctionType __type;

  //! is true if the function is unknown
  bool __unknown;

public:
  virtual bool hasBoundaryExpression() const
  {
    return false;
  }

  const FunctionExpression::FunctionType& type() const
  {
    return __type;
  }

  /*!
    Returns the value of the expression.
  */
  virtual ReferenceCounting<FunctionExpression> value() = 0;

  virtual real_t value(const real_t& x,
		       const real_t& y,
		       const real_t& z) const = 0;

  FunctionExpression(const FunctionExpression& e)
    : Expression(e),
      __type(e.__type),
      __unknown(e.__unknown)
  {
    ;
  }

  FunctionExpression(FunctionExpression::FunctionType type)
    : Expression(Expression::function),
      __type(type),
      __unknown(false)
  {
    ;
  }

  virtual ~FunctionExpression()
  {
    ;
  }
};


class FunctionExpressionValue
  : public FunctionExpression
{
private:
  std::ostream& put(std::ostream& os) const
  {
    switch(__functionExpressionValue) {
    case FunctionExpressionValue::x: {
      os << 'x';
      break;
    }
    case FunctionExpressionValue::y: {
      os << 'y';
      break;
    }
    case FunctionExpressionValue::z: {
      os << 'z';
      break;
    }
    default: {
      os << '(' << __FILE__ << ':' << __LINE__ << ":UNDEFINED FUNCTION)";
    }
    }
    return os;
  }

public:
  enum Value {
    x,
    y,
    z
  };

private:
  const FunctionExpressionValue::Value __functionExpressionValue;

public:
  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    switch(__functionExpressionValue) {
    case FunctionExpressionValue::x: {
      return x;
    }
    case FunctionExpressionValue::y: {
      return y;
    }
    case FunctionExpressionValue::z: {
      return z;
    }
    default: {
      throw ErrorHandler(__FILE__,__LINE__,
			 "undefined function",
			 ErrorHandler::unexpected);
    }
    }
    return 0;
  }

  void execute()
  {
    ;
  }

  FunctionExpressionValue(const FunctionExpressionValue::Value& v)
    : FunctionExpression(FunctionExpression::functionValue),
      __functionExpressionValue(v)
  {
    ;
  }

  FunctionExpressionValue(const FunctionExpressionValue& f)
    : FunctionExpression(f),
      __functionExpressionValue(f.__functionExpressionValue)
  {
    ;
  }

  ~FunctionExpressionValue()
  {
    ;
  }
};


class FunctionExpressionConstant
  : public FunctionExpression
{
private:
  ReferenceCounting<RealExpression> __realExpression;

  mutable real_t __realValue;

  std::ostream& put(std::ostream& os) const
  {
    os << __realValue;
    return os;
  }

public:
  ReferenceCounting<FunctionExpression> value()
  {
    ReferenceCounting<RealExpression> r
      = new RealExpressionValue(__realValue);
    ReferenceCounting<FunctionExpression> f
      = new FunctionExpressionConstant(r);

    return f;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    return __realValue;
  }

  void execute()
  {
    (*__realExpression).execute();
    __realValue = (*__realExpression).realValue();
  }

  FunctionExpressionConstant(ReferenceCounting<RealExpression> r)
    : FunctionExpression(FunctionExpression::constant),
      __realExpression(r),
      __realValue(0)
  {
    ;
  }

  FunctionExpressionConstant(const FunctionExpressionConstant& f)
    : FunctionExpression(f),
      __realExpression(f.__realExpression),
      __realValue(f.__realValue)
  {
    ;
  }

  ~FunctionExpressionConstant()
  {
    ;
  }
};

class FunctionExpressionComposed
  : public FunctionExpression
{
private:
  ReferenceCounting<FunctionExpression> __function;

  ReferenceCounting<FunctionExpression> __functionExpressionX;
  ReferenceCounting<FunctionExpression> __functionExpressionY;
  ReferenceCounting<FunctionExpression> __functionExpressionZ;

  std::ostream& put(std::ostream& os) const
  {
    os << (*__function)
       << '(' << *__functionExpressionX
       << ',' << *__functionExpressionY
       << ',' << *__functionExpressionZ << ')';
    return os;
  }

public:
  bool hasBoundaryExpression() const
  {
    return ((*__functionExpressionX).hasBoundaryExpression() or
	    (*__functionExpressionY).hasBoundaryExpression() or
	    (*__functionExpressionZ).hasBoundaryExpression());
  }

  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    const real_t X = (*__functionExpressionX).value(x,y,z);
    const real_t Y = (*__functionExpressionY).value(x,y,z);
    const real_t Z = (*__functionExpressionZ).value(x,y,z);
    return (*__function).value(X,Y,Z);
  }

  void execute()
  {
    (*__function).execute();
    (*__functionExpressionX).execute();
    (*__functionExpressionY).execute();
    (*__functionExpressionZ).execute();
  }

  FunctionExpressionComposed(ReferenceCounting<FunctionExpression> f,
			     ReferenceCounting<FunctionExpression> X,
			     ReferenceCounting<FunctionExpression> Y,
			     ReferenceCounting<FunctionExpression> Z)
    : FunctionExpression(FunctionExpression::composed),
      __function(f),
      __functionExpressionX(X),
      __functionExpressionY(Y),
      __functionExpressionZ(Z)
  {
    ;
  }

  FunctionExpressionComposed(const FunctionExpressionComposed& f)
    : FunctionExpression(f),
      __function(f.__function),
      __functionExpressionX(f.__functionExpressionX),
      __functionExpressionY(f.__functionExpressionY),
      __functionExpressionZ(f.__functionExpressionZ)
  {
    ;
  }

  ~FunctionExpressionComposed()
  {
    ;
  }
};


class FunctionExpressionNormalComponent
  : public FunctionExpression
{
private:
  Normal::ComponentType __normalType;

  ReferenceCounting<Normal> __normal;

  std::ostream& put(std::ostream& os) const
  {
    switch(__normalType) {
    case Normal::undefined: {
      os << "[undefined normal]";
      break;
    }
    case Normal::x: {
      os << "nx";
      break;
    }
    case Normal::y: {
      os << "ny";
      break;
    }
    case Normal::z: {
      os << "nz";
      break;
    }
    }
    return os;
  }

public:
  bool hasBoundaryExpression() const
  {
    return true;
  }

  void execute();

  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  FunctionExpressionNormalComponent(const FunctionExpressionNormalComponent& f)
    : FunctionExpression(f),
      __normalType(f.__normalType),
      __normal(f.__normal)
  {
    ;
  }

  FunctionExpressionNormalComponent(Normal::ComponentType componentType)
    : FunctionExpression(FunctionExpression::normalComponent),
      __normalType(componentType),
      __normal(0)
  {
    ;
  }

  ~FunctionExpressionNormalComponent()
  {
    ;
  }
};

class FunctionExpressionDx
  : public FunctionExpression
{
public:
  enum Direction {
    x,
    y,
    z
  };

private:
  bool hasBoundaryExpression() const
  {
    return (*__function).hasBoundaryExpression();
  }

  //! xi
  FunctionExpressionDx::Direction __direction;

  //! The function to derive
  ReferenceCounting<FunctionExpression> __function;

  std::ostream& put(std::ostream& os) const
  {
    switch (__direction) {
    case (x): {
      os << "dx(";
      break;
    }
    case (y): {
      os << "dy(";
      break;
    }
    case (z): {
      os << "dz(";
      break;
    }
    }
    os << *__function << ')';
    return os;
  }

public:
  ReferenceCounting<FunctionExpression> value();

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionDx(ReferenceCounting<FunctionExpression> function,
		       FunctionExpressionDx::Direction d);

  FunctionExpressionDx(const FunctionExpressionDx& r);

  ~FunctionExpressionDx();
};

class FunctionExpressionFEM
  : public FunctionExpression
{
private:
  const DiscretizationType::Type __discretizationType;

  ReferenceCounting<MeshExpression>  __mesh;

  ReferenceCounting<FEMFunctionBase> __femFunction;

  ReferenceCounting<FunctionExpression> __function;

  std::ostream& put(std::ostream& os) const
  {
    os << "FEM Function";
    return os;
  }

public:
  ReferenceCounting<MeshExpression> mesh() const;

  const FEMFunctionBase& femFunction() const;

  FEMFunctionBase& femFunction();

  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  real_t dx(const real_t& x,
	    const real_t& y,
	    const real_t& z,
	    const FunctionExpressionDx::Direction& d) const;

  real_t dx(const TinyVector<3>& X,
	    const FunctionExpressionDx::Direction& d) const
  {
    return dx(X[0], X[1], X[2], d);
  }

  void execute();

  FunctionExpressionFEM(ReferenceCounting<MeshExpression> mesh,
			ReferenceCounting<FunctionExpression> r,
			const DiscretizationType::Type& femType);

  FunctionExpressionFEM(ReferenceCounting<MeshExpression> mesh,
			ReferenceCounting<FEMFunctionBase> f);

  FunctionExpressionFEM(const FunctionExpressionFEM& r);

  ~FunctionExpressionFEM();
};

class FunctionExpressionVariable
  : public FunctionExpression
{
private:
  std::ostream& put(std::ostream& os) const
  {
    os << (*__functionVariable).name() << ':';
    switch(__discretizationType) {
    case DiscretizationType::LagrangianFEM1: {
      os << "P1";
      break;
    }
    case DiscretizationType::LagrangianFEM2: {
      os << "P2";
      break;
    }
    default: {
      throw ErrorHandler(__FILE__,__LINE__,
			 "not implemented",
			 ErrorHandler::unexpected);
    }
    }
    if (StreamCenter::instance().getDebugLevel() > 3) {
      os << '{' << (*__functionExpression) << '}';
    }
    return os;
  }

  ReferenceCounting<FunctionVariable> __functionVariable;
  ReferenceCounting<FunctionExpression> __functionExpression;

  DiscretizationType::Type __discretizationType;

public:
  bool hasBoundaryExpression() const
  {
    return (*__functionExpression).hasBoundaryExpression();
  }

  void subscribe()
  {
    (*__functionVariable).subscribe();
  }

  void unsubscribe()
  {
    (*__functionVariable).unsubscribe();
  }

  ReferenceCounting<FunctionExpression> value();

  const DiscretizationType::Type& discretizationType() const
  {
    return __discretizationType;
  }

  const ReferenceCounting<FunctionVariable> variable() const;

  ReferenceCounting<FunctionVariable> variable();

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionVariable(ReferenceCounting<FunctionVariable> r,
			     const DiscretizationType::Type& type = DiscretizationType::LagrangianFEM1);

  FunctionExpressionVariable(const FunctionExpressionVariable& e);

  ~FunctionExpressionVariable();
};


template <typename BinaryOperator>
class FunctionExpressionBinaryOperator
  : public FunctionExpression
{
public:
  std::ostream& put(std::ostream& os) const
  {
    os << '(' << (*__f1Value) << __b << (*__f2Value) << ')';
    return os;
  }
private:
  //! The operator
  BinaryOperator __b;

  //! Functions used by the binary operation.
  ReferenceCounting<FunctionExpression> __f1;
  ReferenceCounting<FunctionExpression> __f2;

  //! The "value" of the functions
  ReferenceCounting<FunctionExpression> __f1Value;
  ReferenceCounting<FunctionExpression> __f2Value;

public:
  bool hasBoundaryExpression() const
  {
    return (*__f1Value).hasBoundaryExpression() or (*__f2Value).hasBoundaryExpression();
  }

  ReferenceCounting<FunctionExpression> value()
  {
    ReferenceCounting<FunctionExpression> b
      = new FunctionExpressionBinaryOperator<BinaryOperator>((*__f1Value).value(),
							     (*__f2Value).value());
    return b;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    real_t d1 = (*__f1Value).value(x,y,z);
    real_t d2 = (*__f2Value).value(x,y,z);
    return __b(d1,d2);
  }

  void execute()
  {
    __f1Value = __f1;
    __f2Value = __f2;
    (*__f1Value).execute();
    (*__f2Value).execute();
  }

  FunctionExpressionBinaryOperator(ReferenceCounting<FunctionExpression> f1,
				   ReferenceCounting<FunctionExpression> f2)
    : FunctionExpression(FunctionExpression::binaryOperator),
      __f1(f1),
      __f2(f2)
  {
    ;
  }

  FunctionExpressionBinaryOperator(const FunctionExpressionBinaryOperator<BinaryOperator>& f)
    : FunctionExpression(f),
      __f1(f.__f1),
      __f2(f.__f2),
      __f1Value(f.__f1Value),
      __f2Value(f.__f2Value)
  {
    ;
  }

  ~FunctionExpressionBinaryOperator()
  {
    ;
  }
};


template <typename UnaryOperator>
class FunctionExpressionUnaryOperator
  : public FunctionExpression
{
private:
  UnaryOperator __u;
  ReferenceCounting<FunctionExpression> __r;

  std::ostream& put(std::ostream& os) const
  {
    os << __u << '(' << (*__r) << ')';
    return os;
  }

public:
  bool hasBoundaryExpression() const
  {
    return (*__r).hasBoundaryExpression();
  }

  ReferenceCounting<FunctionExpression> value()
  {
    ReferenceCounting<FunctionExpression> u
      = new FunctionExpressionUnaryOperator<UnaryOperator>((*__r).value());
    return u;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    real_t d = (*__r).value(x,y,z);
    return __u(d);
  }

  void execute()
  {
    (*__r).execute();
  }

  FunctionExpressionUnaryOperator(ReferenceCounting<FunctionExpression> r)
    : FunctionExpression(FunctionExpression::unaryOperator),
      __r(r)
  {
    ;
  }

  FunctionExpressionUnaryOperator(const FunctionExpressionUnaryOperator<UnaryOperator>& f)
    : FunctionExpression(f),
      __r(f.__r)
  {
    ;
  }

  ~FunctionExpressionUnaryOperator()
  {
    ;
  }
};

class FunctionExpressionConvection
  : public FunctionExpression
{
private:
  //! true while function has not been evaluated
  bool __isToEvaluate;

  //! Time step value
  real_t __timeStepValue;

  //! The first component of the convection field
  ReferenceCounting<FunctionExpression> __v0Function;

  //! The second component of the convection field
  ReferenceCounting<FunctionExpression> __v1Function;

  //! The third component of the convection field
  ReferenceCounting<FunctionExpression> __v2Function;

  //! The time step
  ReferenceCounting<RealExpression> __timeStep;

  //! The convected function
  ReferenceCounting<FunctionExpression> __convectedFunction;

  //! The obtained function
  ReferenceCounting<MeshExpression> __mesh;

  //! The convection operator
  ReferenceCounting<UserFunction> __convectionOperator;

  //! converted functions
  ReferenceCounting<UserFunction> __v0UserFunction;
  ReferenceCounting<UserFunction> __v1UserFunction;
  ReferenceCounting<UserFunction> __v2UserFunction;
  ReferenceCounting<UserFunction> __convectedUserFunction;

  std::ostream& put(std::ostream& os) const
  {
    os << "convect(" <<*__convectedFunction << ')';
    return os;
  }

public:
  bool hasBoundaryExpression() const
  {
    return ((*__v0Function).hasBoundaryExpression() or
	    (*__v1Function).hasBoundaryExpression() or
	    (*__v2Function).hasBoundaryExpression() or
	    (*__convectedFunction).hasBoundaryExpression());
  }

  ReferenceCounting<FunctionExpression> value();

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionConvection(ReferenceCounting<FunctionExpression> v0,
			       ReferenceCounting<FunctionExpression> v1,
			       ReferenceCounting<FunctionExpression> v2,
			       ReferenceCounting<RealExpression> dt,
			       ReferenceCounting<FunctionExpression> phi);

  FunctionExpressionConvection(const FunctionExpressionConvection& r);

  ~FunctionExpressionConvection();
};

class Vector3Expression;
class Object;
class FunctionExpressionCharacteristic
  : public FunctionExpression
{
private:
  ReferenceCounting<Vector3Expression> __reference;
  /*!
    \todo Should use a ReferenceCounter (if the scene changes)
    \bug Cannot use ReferenceCounter (the objects are given by references ...)
   */
  std::list<Object* > __objects;

  ReferenceCounting<Scene> __scene;

  std::ostream& put(std::ostream& os) const;

public:
  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionCharacteristic(ReferenceCounting<Vector3Expression> ref);

  FunctionExpressionCharacteristic(const FunctionExpressionCharacteristic& re);

  ~FunctionExpressionCharacteristic();
};

class Structured3DMesh;
class FunctionExpressionMeshCharacteristic
  : public FunctionExpression
{
private:
  ReferenceCounting<Mesh> __mesh;
  ReferenceCounting<MeshExpression> __meshExpression;

  std::ostream& put(std::ostream& os) const;

public:
  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionMeshCharacteristic(ReferenceCounting<MeshExpression> ref);

  FunctionExpressionMeshCharacteristic(const FunctionExpressionMeshCharacteristic& re);

  ~FunctionExpressionMeshCharacteristic();
};

class FunctionExpressionMeshReferences
  : public FunctionExpression
{
public:
  class ReferencesSet
  {
    friend class FunctionExpressionMeshReferences;

  public:
    typedef std::vector<std::pair<ReferenceCounting<RealExpression>,
				  ReferenceCounting<FunctionExpression> > >
    FunctionReferences;

  private:
    FunctionReferences __functionReferences;

  public:
    FunctionReferences& functionReferences()
    {
      return __functionReferences;
    }

    bool hasBoundaryExpression() const
    {
      bool found = false;
      for (FunctionReferences::const_iterator i = __functionReferences.begin();
	   i != __functionReferences.end(); ++i) {
	found = (*(*i).second).hasBoundaryExpression();
      }
      return found;
    }

    friend std::ostream& operator << (std::ostream& os, const ReferencesSet& r)
    {
      FunctionReferences::const_iterator i = r.__functionReferences.begin();
      if (i != r.__functionReferences.end())
	os << (*(*i).first) << ':' << (*(*i).second);
      i++;
      for (; i != r.__functionReferences.end(); ++i) {
	os << ',' << (*(*i).first) << ':' << (*(*i).second);
      }
      return os;
    }

    void execute()
    {
      for (FunctionReferences::iterator i = __functionReferences.begin();
	   i != __functionReferences.end(); ++i) {
	RealExpression* r = (*i).first;
	r->execute();
	(*(*i).second).execute();
      }
    }

    void add(ReferenceCounting<RealExpression> ref,
	     ReferenceCounting<FunctionExpression> function)
    {
      __functionReferences.push_back(std::make_pair(ref, function));
    }

    ReferencesSet(ReferenceCounting<RealExpression> ref,
		  ReferenceCounting<FunctionExpression> function)
    {
      __functionReferences.push_back(std::make_pair(ref, function));
    }

    ReferencesSet(const ReferencesSet& f)
      : __functionReferences(f.__functionReferences)
    {
      ;
    }

    ~ReferencesSet()
    {
      ;
    }
  };

protected:
  ReferenceCounting<MeshExpression> __mesh;
  ReferenceCounting<ReferencesSet> __referenceSet;

  std::map<size_t, ReferenceCounting<FunctionExpression> > __functionReferences;

public:
  FunctionExpressionMeshReferences(ReferenceCounting<MeshExpression> mesh,
				   ReferenceCounting<ReferencesSet> ref,
				   FunctionExpression::FunctionType t);

  FunctionExpressionMeshReferences(const FunctionExpressionMeshReferences& f);

  virtual ~FunctionExpressionMeshReferences();
};

class FunctionExpressionMeshElementsReferences
  : public FunctionExpressionMeshReferences
{
private:
  std::ostream& put(std::ostream& os) const;

  template <typename MeshType>
  real_t __evaluate(const real_t& x,
		    const real_t& y,
		    const real_t& z) const;
public:
  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionMeshElementsReferences(ReferenceCounting<MeshExpression> mesh,
					   ReferenceCounting<ReferencesSet> ref);

  FunctionExpressionMeshElementsReferences(const FunctionExpressionMeshElementsReferences& f);

  ~FunctionExpressionMeshElementsReferences();
};

class FunctionExpressionMeshVerticesReferences
  : public FunctionExpressionMeshReferences
{
private:
  ReferenceCounting<FunctionExpressionFEM> __femFunction;

  std::ostream& put(std::ostream& os) const;

public:
  ReferenceCounting<FunctionExpression> value();

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionMeshVerticesReferences(ReferenceCounting<MeshExpression> mesh,
					   ReferenceCounting<ReferencesSet> ref);

  FunctionExpressionMeshVerticesReferences(const FunctionExpressionMeshVerticesReferences& f);

  ~FunctionExpressionMeshVerticesReferences();
};

class DomainExpression;
class FunctionExpressionDomainCharacteristic
  : public FunctionExpression
{
private:
  ReferenceCounting<DomainExpression> __domain;

  std::ostream& put(std::ostream& os) const;

public:
  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionDomainCharacteristic(ReferenceCounting<DomainExpression> D);

  FunctionExpressionDomainCharacteristic(const FunctionExpressionDomainCharacteristic& re);

  ~FunctionExpressionDomainCharacteristic();
};



class FunctionExpressionRead
  : public FunctionExpression
{
private:
  std::ostream& put(std::ostream& os) const;

  ReferenceCounting<FileDescriptor> __fileDescriptor;

  ReferenceCounting<FunctionExpressionFEM> __femFunction;

  ReferenceCounting<StringExpression> __fileName;

  ReferenceCounting<MeshExpression>  __mesh;

public:
  ReferenceCounting<FunctionExpression> value();

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  void execute();

  FunctionExpressionRead(ReferenceCounting<FileDescriptor> descriptor,
			 ReferenceCounting<StringExpression> fileName,
			 ReferenceCounting<MeshExpression> mesh);

  FunctionExpressionRead(const FunctionExpressionRead& f);

  ~FunctionExpressionRead();
};

class FunctionExpressionBoolean
  : public FunctionExpression
{
public:
  FunctionExpressionBoolean()
    : FunctionExpression(FunctionExpression::boolean)
  {
    ;
  }

  FunctionExpressionBoolean(const FunctionExpressionBoolean& feb)
    : FunctionExpression(feb)
  {
    ;
  }

  virtual bool boolean(const real_t& x,
		       const real_t& y,
		       const real_t& z) const = 0;

  virtual ~FunctionExpressionBoolean()
  {
    ;
  }
};

class BooleanExpression;
class FunctionExpressionBooleanValue
  : public FunctionExpressionBoolean
{
private:
  ReferenceCounting<BooleanExpression> __booleanExpression;

  std::ostream& put(std::ostream & os) const
  {
    os << "Boolean";
    return os;
  }
public:
  void execute();

  ReferenceCounting<FunctionExpression> value();

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  bool boolean(const real_t& x,
	       const real_t& y,
	       const real_t& z) const;

  FunctionExpressionBooleanValue(ReferenceCounting<BooleanExpression> boolExp);

  FunctionExpressionBooleanValue(const FunctionExpressionBooleanValue& f);

  ~FunctionExpressionBooleanValue();
};

template <bool(*B)(const real_t x, const real_t y)>
class FunctionExpressionBooleanCompare
  : public FunctionExpressionBoolean
{
private:
  std::ostream& put(std::ostream & os) const
  {
    os << (*__f);
    operatorName<B>(os);
    os << (*__g);
    return os;
  }

  ReferenceCounting<FunctionExpression> __f;
  ReferenceCounting<FunctionExpression> __g;
public:
  bool hasBoundaryExpression() const
  {
    return ((*__f).hasBoundaryExpression() or (*__g).hasBoundaryExpression());
  }

  void execute()
  {
    (*__f).execute();
    (*__g).execute();
  }

  bool boolean(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    return B((*__f).value(x,y,z), (*__g).value(x,y,z));
  }

  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    return (B((*__f).value(x,y,z), (*__g).value(x,y,z))?1:0);
  }

  FunctionExpressionBooleanCompare(ReferenceCounting<FunctionExpression> f,
				   ReferenceCounting<FunctionExpression> g)
    : __f(f),
      __g(g)
  {
    ;
  }

  FunctionExpressionBooleanCompare(const FunctionExpressionBooleanCompare<B>& f)
    : __f(f.__f),
      __g(f.__g)
  {
    ;
  }

  ~FunctionExpressionBooleanCompare()
  {
    ;
  }
};

template <bool(*B)(const bool a, const bool b)>
class FunctionExpressionBooleanBinaryOperator
  : public FunctionExpressionBoolean
{
private:
  std::ostream& put(std::ostream & os) const
  {
    os << '(' << (*__r1) << ')';
    binaryBooleanName<B>(os);
    os << '('  << (*__r2) << ')';
    return os;
  }

  ReferenceCounting<FunctionExpressionBoolean> __r1;
  ReferenceCounting<FunctionExpressionBoolean> __r2;

public:
  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    return (boolean(x,y,z))?1.:0.;
  }

  bool boolean(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    return B((*__r1).boolean(x,y,z), (*__r2).boolean(x,y,z));
  }

  void execute()
  {
    (*__r1).execute();
    (*__r2).execute();
  }

  FunctionExpressionBooleanBinaryOperator(ReferenceCounting<FunctionExpressionBoolean> r1,
					  ReferenceCounting<FunctionExpressionBoolean> r2)
    : __r1(r1),
      __r2(r2)
  {
    ;
  }

  FunctionExpressionBooleanBinaryOperator(const FunctionExpressionBooleanBinaryOperator<B>& e)
    : __r1(e.__r1),
      __r2(e.__r2)
  {
    ;
  }

  ~FunctionExpressionBooleanBinaryOperator()
  {
    ;
  }
};

template <bool (*F)(const bool x)>
class FunctionExpressionBooleanUnaryOperator
  : public FunctionExpressionBoolean
{
private:

  std::ostream& put(std::ostream & os) const
  {
    unaryBooleanName<F>(os);
    os << '(' << (*__r) << ')';
    return os;
  }

  ReferenceCounting<FunctionExpressionBoolean> __r;

public:

  ReferenceCounting<FunctionExpression> value()
  {
    return this;
  }

  real_t value(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    return (1-(*__r).value(x,y,z));
  }

  bool boolean(const real_t& x,
	       const real_t& y,
	       const real_t& z) const
  {
    return F((*__r).boolean(x,y,z));
  }

  void execute()
  {
    (*__r).execute();
  }

  FunctionExpressionBooleanUnaryOperator(ReferenceCounting<FunctionExpressionBoolean> r)
    : __r(r)
  {
    ;
  }

  FunctionExpressionBooleanUnaryOperator(const FunctionExpressionBooleanUnaryOperator<F>& e)
    : __r(e.__r)
  {
    ;
  }

  ~FunctionExpressionBooleanUnaryOperator()
  {
    ;
  }
};

#endif // FUNCTION_EXPRESSION_HPP
