//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FiniteElementMethod.cpp,v 1.8 2004/12/31 14:00:47 delpinux Exp $

#include <FiniteElementMethod.hpp>

#include <PDESolution.hpp>

#include <PDEProblem.hpp>

#include <Structured3DMesh.hpp>
#include <MeshOfHexahedra.hpp>
#include <MeshOfTetrahedra.hpp>

#include <FEMDiscretization.hpp>

#include <BoundaryConditionDiscretizationFEM.hpp>

#include <map>
#include <list>

#include <KrylovSolver.hpp>

#include <MatrixManagement.hpp>

#include <SparseMatrix.hpp>

#include <Timer.hpp>

#include <ErrorHandler.hpp>

template <typename MeshType>
void FiniteElementMethod::__discretize()
{
  MemoryManager MM;

  bool performAssembling =MM.ReserveMatrix(__A,
					   problem().numberOfUnknown(),
					   __degreeOfFreedomSet.size());
  ffout(2) << "reserve \n";

  MM.ReserveVector(__b,
		   problem().numberOfUnknown(),
		   __degreeOfFreedomSet.size());

  ffout(2) << "Discretizing the PDE Problem ... ";

  ReferenceCounting<FEMDiscretization<MeshType> > FEM
    = new FEMDiscretization<MeshType>(problem(),
				      dynamic_cast<MeshType&>(mesh()),
				      *__A,*__b, __degreeOfFreedomSet);

  if (performAssembling) {
    (*FEM).assembleMatrix();
  } else {
    ffout(2) << "keeping previous operator discretization\n";
  }

  (*FEM).assembleSecondMember();

  ffout(2) << "done\n";

  ffout(2) << "Discretizing Boundary Conditions\n";

  BoundaryConditionDiscretizationFEM<MeshType>* bcd
    = new BoundaryConditionDiscretizationFEM<MeshType>(problem(),
						       dynamic_cast<MeshType&>(mesh()),
						       __degreeOfFreedomSet);
  bcd->associatesMeshesToBoundaryConditions();
  ReferenceCounting<BoundaryConditionDiscretization> bcDiscretization = bcd;

  ffout(2) << "Second Member Modification\n";

  (*bcDiscretization).setSecondMember(__A,__b);


  ffout(2) << "Matrix Modification\n";

  (*bcDiscretization).setMatrix(__A,__b);

  ffout(2) << "done\n";

  if ((*__A).type() == BaseMatrix::doubleHashedMatrix) {
    Timer t;
    t.start();

    SparseMatrix* aa
      = new SparseMatrix(static_cast<DoubleHashedMatrix&>(*__A));
    __A = aa; // now use sparse matrix

    t.stop();
    ffout(2) << "Matrix copy: " << t << '\n';
  }
}


void FiniteElementMethod::Discretize (ConstReferenceCounting<Problem> Pb)
{
  __problem = Pb;


  switch (mesh().type()) {
  case Mesh::cartesianHexahedraMesh: {
    this->__discretize<Structured3DMesh>();
    break;
  }
  case Mesh::hexahedraMesh: {
    this->__discretize<MeshOfHexahedra>();
    break;
  }
  case Mesh::tetrahedraMesh: {
    this->__discretize<MeshOfTetrahedra>();
    break;
  }
  default: {
    throw ErrorHandler(__FILE__, __LINE__,
		       "mesh type cannot be used for computation",
		       ErrorHandler::normal);
  }
  }
}

void FiniteElementMethod::Compute (Solution& U)
{
  PDESolution& u = static_cast<PDESolution&>(U);

  ffout(2) << "Solving the linear system\n";

  KrylovSolver K(*__A, *__b, __degreeOfFreedomSet);

  K.solve(u.values(), problem());
}

