//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: EditorWindow.cpp,v 1.8 2005/02/02 21:44:27 delpinux Exp $

#include <EditorWindow.hpp>

#include <qimage.h>
#include <qpixmap.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtextedit.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qstatusbar.h>
#include <qmessagebox.h>
#include <qprinter.h>
#include <qapplication.h>
#include <qaccel.h>
#include <qtextstream.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qwhatsthis.h>
#include <qaction.h>
#include <qsimplerichtext.h>

#include <filesave.xpm>
#include <fileopen.xpm>
#include <fileprint.xpm>

#include <FFSyntaxHighlighter.hpp>

#include <FFThread.hpp>

#include <sstream>

EditorWindow::EditorWindow()
    : QMainWindow(0, "example editor main window", WDestructiveClose)
{
    printer = new QPrinter(QPrinter::HighResolution);

    QAction * fileNewAction;
    QAction * fileOpenAction;
    QAction * fileSaveAction, * fileSaveAsAction, * filePrintAction;
    QAction * fileCloseAction, * fileQuitAction;

    fileNewAction = new QAction("&New", CTRL+Key_N, this, "new");
    connect(fileNewAction, SIGNAL(activated()), this,
             SLOT(newFile()));

    fileOpenAction = new QAction(QPixmap(fileopen), "&Open...",
                                  CTRL+Key_O, this, "open");
    connect(fileOpenAction, SIGNAL(activated()), this, SLOT(choose()));

    const char * fileOpenText = "<p><img source=\"fileopen\"> "
                     "Click this button to open a <it>new file</it>. <br>"
                     "You can also select the <b>Open</b> command "
                     "from the <b>File</b> menu.</p>";
    QMimeSourceFactory::defaultFactory()->setPixmap("fileopen",
                          fileOpenAction->iconSet().pixmap());
    fileOpenAction->setWhatsThis(fileOpenText);

    fileSaveAction = new QAction(QPixmap(filesave),
                                  "&Save", CTRL+Key_S, this, "save");
    connect(fileSaveAction, SIGNAL(activated()), this, SLOT(save()));

    const char * fileSaveText = "<p>Click this button to save the file you "
                     "are editing. You will be prompted for a file name.\n"
                     "You can also select the <b>Save</b> command "
                     "from the <b>File</b> menu.</p>";
    fileSaveAction->setWhatsThis(fileSaveText);

    fileSaveAsAction = new QAction("Save File As", "Save &As...", 0,  this,
                                    "save as");
    connect(fileSaveAsAction, SIGNAL(activated()), this,
             SLOT(saveAs()));
    fileSaveAsAction->setWhatsThis(fileSaveText);

    filePrintAction = new QAction("Print File", QPixmap(fileprint),
                                   "&Print...", CTRL+Key_P, this, "print");
    connect(filePrintAction, SIGNAL(activated()), this,
             SLOT(print()));

    const char * filePrintText = "Click this button to print the file you "
                     "are editing.\n You can also select the Print "
                     "command from the File menu.";
    filePrintAction->setWhatsThis(filePrintText);

    fileCloseAction = new QAction("Close", "&Close", CTRL+Key_W, this,
                                   "close");
    connect(fileCloseAction, SIGNAL(activated()), this,
             SLOT(close()));

    fileQuitAction = new QAction("Quit", "&Quit", CTRL+Key_Q, this,
                                  "quit");
    connect(fileQuitAction, SIGNAL(activated()), qApp,
             SLOT(closeAllWindows()));

    // populate a tool bar with some actions

    QToolBar * fileTools = new QToolBar(this, "file operations");
    fileTools->setLabel("File Operations");
    fileOpenAction->addTo(fileTools);
    fileSaveAction->addTo(fileTools);
    filePrintAction->addTo(fileTools);
    (void)QWhatsThis::whatsThisButton(fileTools);


    // populate a menu with all actions

    QPopupMenu * file = new QPopupMenu(this);
    menuBar()->insertItem("&File", file);
    fileNewAction->addTo(file);
    fileOpenAction->addTo(file);
    fileSaveAction->addTo(file);
    fileSaveAsAction->addTo(file);
    file->insertSeparator();
    filePrintAction->addTo(file);
    file->insertSeparator();
    fileCloseAction->addTo(file);
    fileQuitAction->addTo(file);

    // adds the action menu
    QPopupMenu * action = new QPopupMenu(this);
    menuBar()->insertItem("&Action", action);
    action->insertItem("&Run", this, SLOT(run()), CTRL+Key_R);
    action->insertItem("&Pause", this, SLOT(pause()), CTRL+Key_P);
    action->insertItem("&Resume", this, SLOT(resume()));
    action->insertItem("&Stop", this, SLOT(stop()));

    menuBar()->insertSeparator();
    
    // add a help menu

    QPopupMenu * help = new QPopupMenu(this);
    menuBar()->insertItem("&Help", help);
    help->insertItem("&About", this, SLOT(aboutff3d()), Key_F1);
    help->insertItem("About &Qt", this, SLOT(aboutQt()));
    help->insertItem("About &Vtk", this, SLOT(aboutVtk()));
    help->insertSeparator();
    help->insertItem("What's &This", this, SLOT(whatsThis()),
                      SHIFT+Key_F1);


    // create and define the central widget

    e = new QTextEdit(this, "editor");
    
    e->setFocus();
    e->setFont(QFont("Courier",10));
    FFSyntaxHighlighter* syntaxHighlighter = new FFSyntaxHighlighter(e);

    setCentralWidget(e);
    statusBar()->message("Wellcome to FreeFEM3D", 2000);

    resize(450, 600);
}


EditorWindow::~EditorWindow()
{
    delete printer;
}



void EditorWindow::newFile()
{
    EditorWindow *ed = new EditorWindow;
    ed->show();
}

void EditorWindow::choose()
{
    QString fn = QFileDialog::getOpenFileName(QString::null, QString::null,
					       this);
    if (!fn.isEmpty())
	load(fn);
    else
	statusBar()->message("Loading aborted", 2000);
}


void EditorWindow::load(const QString &fileName)
{
    QFile f(fileName);
    if (!f.open(IO_ReadOnly))
	return;

    QTextStream ts(&f);
    e->setText(ts.read());
    e->setModified(FALSE);
    setCaption(fileName);
    statusBar()->message("Loaded document " + fileName, 2000);
}


void EditorWindow::save()
{
    if (filename.isEmpty()) {
	saveAs();
	return;
    }

    QString text = e->text();
    QFile f(filename);
    if (!f.open(IO_WriteOnly)) {
	statusBar()->message(QString("Could not write to %1").arg(filename),
			      2000);
	return;
    }

    QTextStream t(&f);
    t << text;
    f.close();

    e->setModified(FALSE);

    setCaption(filename);

    statusBar()->message(QString("File %1 saved").arg(filename), 2000);
}


void EditorWindow::saveAs()
{
    QString fn = QFileDialog::getSaveFileName(QString::null, QString::null,
					       this);
    if (!fn.isEmpty()) {
	filename = fn;
	save();
    } else {
	statusBar()->message("Saving aborted", 2000);
    }
}


void EditorWindow::print()
{
    printer->setFullPage(TRUE);
    if (printer->setup(this)) {		// printer dialog
	statusBar()->message("Printing...");
	QPainter p;
	if(!p.begin(printer)) {               // paint on printer
	    statusBar()->message("Printing aborted", 2000);
	    return;
	}

	QPaintDeviceMetrics metrics(p.device());
	int dpiy = metrics.logicalDpiY();
	int margin = (int) ((2/2.54)*dpiy); // 2 cm margins
	QRect body(margin, margin, metrics.width() - 2*margin, metrics.height() - 2*margin);
	QSimpleRichText richText(QStyleSheet::convertFromPlainText(e->text()),
				  QFont(),
				  e->context(),
				  e->styleSheet(),
				  e->mimeSourceFactory(),
				  body.height());
	richText.setWidth(&p, body.width());
  	QRect view(body);
	int page = 1;
	do {
	    richText.draw(&p, body.left(), body.top(), view, colorGroup());
	    view.moveBy(0, body.height());
	    p.translate(0, -body.height());
	    p.drawText(view.right() - p.fontMetrics().width(QString::number(page)),
			view.bottom() + p.fontMetrics().ascent() + 5, QString::number(page));
	    if (view.top()  >= richText.height())
		break;
	    printer->newPage();
	    page++;
	} while (TRUE);

	statusBar()->message("Printing completed", 2000);
    } else {
	statusBar()->message("Printing aborted", 2000);
    }
}

void EditorWindow::closeEvent(QCloseEvent* ce)
{
    if (!e->isModified()) {
	ce->accept();
	return;
    }

    switch(QMessageBox::information(this, "Qt Editor Example",
				      "The document has been changed since "
				      "the last save.",
				      "Save Now", "Cancel", "Leave Anyway",
				      0, 1)) {
    case 0:
	save();
	ce->accept();
	break;
    case 1:
    default: // just for sanity
	ce->ignore();
	break;
    case 2:
	ce->accept();
	break;
    }
}


void EditorWindow::run()
{
  std::stringstream text;
  text << e->text() << std::ends;

  FFThread* thread =new FFThread(text);
  thread->launch();
}

void EditorWindow::pause()
{
  fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
}
void EditorWindow::resume()
{
  fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
}

void EditorWindow::stop()
{
  fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
}


void EditorWindow::aboutff3d()
{
    QMessageBox::about(this, "FreeFEM3D",
			"FreeFEM3D, blablabla");
}


void EditorWindow::aboutQt()
{
    QMessageBox::aboutQt(this, "Qt blabla");
}

void EditorWindow::aboutVtk()
{
    QMessageBox::about(this, "Vtk", "Vtk blabla");
}
