/*	
 *		File		: main.c
 *							
 *		Title		: Flink email monitoring applet for GNOME 2	
 *										
 *		Description	: Main source file, all	base functions here	
 *										
 *		Authors		: Marcus Leyman, 2002 <marcus@leyman.nu>	
 *										
 *		License		: GPL						
 *										
 *		Date		: 27 May 2002
 *		
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"

/********************************************************
 * Function:	flink_gerror_handler		        *
 *							*
 * Description:	Creates error dialog with err->message	*								
 * 							*
 * Arguments: 	Flink *flink, For parent window		*
 *		GError *err, For error message		*
 *							*
 * Returns: 	nothing					*
 ********************************************************/
void
flink_gerror_handler (Flink *flink, 
		     GError *err)
{
  GtkWidget *dialog;	

  dialog = gtk_message_dialog_new (GTK_WINDOW (flink->applet),
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_CLOSE,
				   _("An error has occured: %s"),
				   err->message);

  g_signal_connect (G_OBJECT (dialog), 
		    "response",
		    G_CALLBACK (gtk_widget_destroy),
		    NULL);

  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);

  gtk_widget_show (dialog);

  g_error_free (err);
}

/********************************************************
 * Function: 	flink_orientation_change		*
 *							*
 * Description: Sets Flink structs orientation value if *
 *		it changes.				*
 *							*
 * Arguments: 	GtkWidget *w, 				*
 *		gint arg, 				*
 *		gpointer data, Should be a Flink struct	*
 *							*
 * Returns: 	nothing					*
 ********************************************************/
static void
flink_orientation_change (GtkWidget *w, 
			 gint arg, 
			 gpointer data) 
{
  Flink *flink = (Flink *) data;
  flink->orientation = panel_applet_get_orient(PANEL_APPLET(flink->applet));
}

/********************************************************
 * Function: 	flink_menu_position			*
 *							*
 * Description:	positions the menu relative to applet 	*
 *		and screen		                *
 *							*
 * Arguments:	GtkMenu *b_menu, the menu (wow)		*
 *		gint *x, pointer to x-position to set	*
 *		gint *y, pointer to y-position to set	*
 *		gboolean *push_in,			*
 *		gpointer data, should be Flink struct	*
 *							*
 * Returns: 	nothing				        *
 ********************************************************/
void
flink_menu_position	(GtkMenu *b_menu, 
			gint *x, 
			gint *y, 
			gboolean *push_in, 
			gpointer data)
{
  Flink 			*flink = (Flink *) data;
  GtkRequisition  	req;
  gint            	width, height;
  gint			ox, oy;
	
  gtk_widget_size_request (GTK_WIDGET(b_menu), &req);
	
  gdk_window_get_origin (flink->status_frame->window, &ox, &oy);
  gdk_drawable_get_size (flink->status_frame->window, &width, &height);

  *x = ox;
  *y = oy;
	
  switch (flink->orientation) {
  case PANEL_APPLET_ORIENT_DOWN:
    *x += (width - req.width) / 2;
    *y += height;
    break;
		
  case PANEL_APPLET_ORIENT_UP:
    *x += (width - req.width) / 2;
    *y -= req.height;
    break;
		
  case PANEL_APPLET_ORIENT_LEFT:
    *x -= req.width;
    *y += (height - req.height) / 2;			
    break;
		
  case PANEL_APPLET_ORIENT_RIGHT:
    *x += width;
    *y += (height - req.height) / 2;			
    break;
		
  default:
    break;
  }
	
  if (*x < 0) {
    *x = 0;
  }
  if (*y < 0) {
    *y = 0;
  }	
}

/********************************************************
 * Function:	flink_popdown_menu		        *
 *							*
 * Description:	closes down status menu			*
 *							*
 * Arguments:	Flink *flink,				*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_popdown_menu (Flink *flink) 
{
  if (flink->popup_menu) {
    gdk_pointer_ungrab (GDK_CURRENT_TIME);		
    gtk_widget_destroy(GTK_WIDGET(flink->popup_menu));
    gtk_frame_set_shadow_type(GTK_FRAME(flink->status_frame), GTK_SHADOW_NONE);
    flink->popup_menu = NULL;
  }
}	

/********************************************************
 * Function:	flink_status_nofocus			*
 *							*
 * Description:	tells above to close down status menu	*
 *							*
 * Arguments:	GtkWidget *w, 			        *
 *		gpointer user_data, Flink structure	*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_status_nofocus 	(GtkWidget *w, 
			 gpointer user_data)
{
  Flink *flink = (Flink *) user_data;
  if (flink->popup_menu)
    flink_popdown_menu(flink);
}

/********************************************************
 * Function:	flink_bell				*
 *							*
 * Description:	Runs bells, programs etc. if specified	*
 *		in config		                *
 *							*
 * Arguments:	none					*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_bell ()
{
  GConfClient	*client;
	
  client = gconf_client_get_default();
	
  if (gconf_client_get_bool(client, "/apps/flink/general/beep", NULL)) {
    gdk_beep();
  }
	
  if (gconf_client_get_bool(client, "/apps/flink/general/sound", NULL)) {
    gchar	*soundfile;
    soundfile = gconf_client_get_string(client,
					"/apps/flink/general/soundfile",
					NULL);
    if (soundfile)
      gnome_sound_play(soundfile);
		
    g_free(soundfile);
  }
	
  if (gconf_client_get_bool(client, "/apps/flink/general/program_autorun", NULL)) {
    gchar	*programfile;
    programfile = gconf_client_get_string(client,
					  "/apps/flink/general/programfile",
					  NULL);
    if(programfile)
      gnome_execute_shell ("~/", programfile);
  }
}

/********************************************************
 * Function:	flink_new_mail			        *
 *							*
 * Description:	updates status label, by notifyer	*
 *							*
 * Arguments:	GConfClient *client, 			*
 *		guint cnxn_id, notifyer connection id	*
 *		GConfEntry *entry, total_mail		*
 *		gpointer user_data, Flink structure	*
 *						     	*
 * Returns:	nothing					*
 ********************************************************/
void
flink_new_mail	(GConfClient *client, 
		guint cnxn_id,
		GConfEntry *entry,
		gpointer user_data)
{
  gint 	mail;
  Flink *flink = (Flink *) user_data;
	
  mail = gconf_value_get_int(entry->value);
	
  gtk_label_set_text(GTK_LABEL(flink->status_indicator),
		     g_strdup_printf("%d", mail));
  if (mail > 0) {
    gtk_widget_hide(GTK_WIDGET(flink->status_image_nomail));
    gtk_widget_show(GTK_WIDGET(flink->status_image_mail));
  } else {
    gtk_widget_hide(GTK_WIDGET(flink->status_image_mail));
    gtk_widget_show(GTK_WIDGET(flink->status_image_nomail));
  }
}

/********************************************************
 * Function:	flink_newmail_key_changed		*
 *							*
 * Description:	runs flink_mail_set_total_key		*
 *							*
 * Arguments:	GConfClient *client, 			*
 *		guint cnxn_id, notifyer connection id	*
 *		GConfEntry *entry, total_mail		*
 *		gpointer user_data, Flink structure	*
 *							*
 * Returns:	nothing					*
 ********************************************************/

void
flink_newmail_key_changed	(GConfClient *client, 
				 guint cnxn_id,
				 GConfEntry *entry,
				 gpointer user_data)
{
  flink_mail_set_total_key();
}

/********************************************************
 * Function:	flink_popup_menu			*
 *							*
 * Description:	pops up status menu			*
 *							*
 * Arguments:	Flink *flink, 				*
 *		GdkEventButton *eb,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void 
flink_popup_menu	(Flink *flink, 
			 GdkEventButton *eb,
			 GdkEventKey *ek) 
{
  GConfClient	*client;
  GtkWidget	*checkall_menuitem;
  GtkWidget	*limiter_menuitem;
  gint		current;
  gint		accounts;
	
  client = gconf_client_get_default();
	
  /* FIXME: proper error handling */
	
  accounts = gconf_client_get_int(client,
				  "/apps/flink/accounts/accounts",
				  NULL);
	
  flink->popup_menu = gtk_menu_new();
	
  if(accounts == 0) {
    /* Previously we had an item doing nothing */
    GtkWidget	*no_accs_item;
    no_accs_item = gtk_menu_item_new_with_label(_("Configure Flink"));
    gtk_widget_show(no_accs_item);
    gtk_menu_append(flink->popup_menu, no_accs_item);
    /* now we want it to run config instead */
    g_signal_connect(GTK_OBJECT(no_accs_item),
		     "button_press_event",
		     GTK_SIGNAL_FUNC(flink_properties_clicked),
		     NULL);
  } else {

    for (current = 1; current <= accounts; current++) {
      GtkWidget	*menuitem[accounts];
      gchar		*title, *title_key, *mail_key;
      gint		mail;
			
      title_key = g_strdup_printf("/apps/flink/accounts/%d_title", current);
      mail_key = g_strdup_printf("/apps/flink/accounts/%d_newmail", current);
			
      title = gconf_client_get_string(client, title_key, NULL);
      mail = gconf_client_get_int(client, mail_key, NULL);

      switch(mail) {
      case -1: /* could not connect to host */
	title = g_strdup_printf(_("%s (no connection)"), title);
	mail = 0;
	break;		
      case -2: /* wrong username or password */
	title = g_strdup_printf(_("%s (wrong username or password)"), title);
	mail = 0;
	break;
      case -3: /* no such path */
	title = g_strdup_printf(_("%s (incorrect path)"), title);
	mail = 0;
	break;
      case -4: /* could not open mailfile */
	title = g_strdup_printf(_("%s (error opening mbox file)"), title);
	mail = 0;
	break;
      case -5: /* other error */
	title = g_strdup_printf(_("%s (an error has occured)"), title);
	mail = 0;
	break;
      case -6: /* must provide foldername */
	title = g_strdup_printf(_("%s (incorrect foldername)"), title);
	mail = 0;
	break;
      default:
	break;
      }
			
      menuitem[current] = gtk_menu_item_new_with_label(g_strdup_printf("%d, %s", mail, title));
      gtk_widget_show(menuitem[current]);
      gtk_menu_append(flink->popup_menu, menuitem[current]);
      g_signal_connect(GTK_OBJECT(menuitem[current]), 
		       "button_press_event",
		       GTK_SIGNAL_FUNC(flink_mail_check_account),
		       (gpointer) current);
    }
  }
	
  if (accounts > 1) {
    checkall_menuitem = gtk_menu_item_new_with_label(_("Check all accounts"));
    limiter_menuitem = gtk_menu_item_new_with_label(NULL);
    gtk_widget_show(checkall_menuitem);
    gtk_widget_show(limiter_menuitem);
    gtk_menu_append(flink->popup_menu, limiter_menuitem);
    gtk_menu_append(flink->popup_menu, checkall_menuitem);
		
    g_signal_connect(GTK_OBJECT(checkall_menuitem),
		     "button_press_event",
		     GTK_SIGNAL_FUNC(flink_mail_check_all_accounts),
		     NULL);
  }
  if (eb) {
    gtk_menu_popup(GTK_MENU(flink->popup_menu), NULL, NULL, flink_menu_position, flink, eb->button, eb->time);
  } else {
    gtk_menu_popup(GTK_MENU(flink->popup_menu), NULL, NULL, flink_menu_position, flink, ek->keyval, ek->time);
  }
  g_signal_connect(GTK_OBJECT(flink->popup_menu), "hide",
		   GTK_SIGNAL_FUNC(flink_status_nofocus), (gpointer) flink);
}

/********************************************************
 * Function:	flink_status_clicked			*
 *							*
 * Description:	When status button is clicked		*
 *							*
 * Arguments:	GtkWidget *widget,			*
 *		GdkEventButton *eb,			*
 *		Flink *flink,				*
 *							*
 * Returns:	TRUE if success, else FALSE		*
 ********************************************************/

static gboolean
flink_status_clicked	(GtkWidget * widget, 
			 GdkEventButton * eb, 
			 Flink *flink)
{
  if (eb->button != 1) {

    return FALSE; 
  }
  if (!flink->popup_menu) {
    flink_popup_menu(flink, eb, NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(flink->status_frame), GTK_SHADOW_IN);
  } else {
    flink_popdown_menu(flink);
  }
  return TRUE; 
}

/********************************************************
 * Function:	flink_status_clicked			*
 *							*
 * Description:	When status button is clicked		*
 *							*
 * Arguments:	GtkWidget *widget,			*
 *		GdkEventKey *ek,			*
 *		Flink *flink,				*
 *							*
 * Returns:	TRUE if success, else FALSE		*
 ********************************************************/
static gboolean
flink_status_keypress	(GtkWidget * widget,
			 GdkEventKey *ek,
			 Flink *flink)
{

  switch(ek->keyval) {
  case GDK_Escape:
    /*popdown if up*/
    if (!flink->popup_menu) {
      return FALSE;
    } else {
      flink_popdown_menu(flink);
      return TRUE;
    }
    
  case GDK_KP_Enter:
  case GDK_ISO_Enter:
  case GDK_3270_Enter:
  case GDK_Return:
  case GDK_space:
  case GDK_KP_Space:
    /*popup if down*/
    if (!flink->popup_menu) {
      flink_popup_menu(flink, NULL, ek);
      gtk_frame_set_shadow_type(GTK_FRAME(flink->status_frame), GTK_SHADOW_IN);
      return TRUE;
    } else {
      return FALSE;
    }

  default:
    break;
  }
    return FALSE;
}

/********************************************************
 * Function:	flink_realize_config			*
 *							*
 * Description:	starts poll timeouts and notifys	*
 *							*
 * Arguments:	Flink *flink				*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_realize_config	(GConfClient *cli,
			 guint cnxn_id,
			 GConfEntry *entry,	
			 gpointer user_data)

{
  GConfClient	*client;
  gint		accounts;
  gint		current = 0;
  guint		ref;
	
  client = gconf_client_get_default();
        
  accounts = gconf_client_get_int(client,
				  "/apps/flink/accounts/accounts",
				  NULL);

  for (current = 1; current <= accounts; current++) {
    gchar		*timeout_path;
    gint		timeout;		

    timeout_path = g_strdup_printf("/apps/flink/accounts/%d_interval", current);
    timeout = gconf_client_get_int(client, 
				   timeout_path,
				   NULL);
    if(timeout != 0) {
      ref = gtk_timeout_add(((guint) timeout * 60000),
			    flink_timeout_reached,
			    (gpointer) current);
    }
    gconf_client_notify_add(client,
			    g_strdup_printf("/apps/flink/accounts/%d_newmail", current),
			    flink_newmail_key_changed,
			    NULL,
			    NULL,
			    NULL);
		
    gconf_client_notify_add(client,
			    timeout_path,
			    flink_timeout_changed,
			    (gpointer) ref,
			    NULL,
			    NULL);
  }
}

/********************************************************
 * Function:	flink_create_applet			*
 *							*
 * Description:	Creates the applet widgets		*
 *							*
 * Arguments:	PanelApplet *applet, Applet to fill	*
 *							*
 * Returns:	Completed Flink structure		*
 ********************************************************/
Flink *
flink_create_applet	(PanelApplet *applet) 
{
  Flink		*flink;
  GConfClient	*client;
  gchar		*mail_image;	
  gchar		*nomail_image;
  gboolean	retval;
  GError		*err = NULL;
				
  flink = g_new0(Flink, 1);
		
  flink->orientation = panel_applet_get_orient(applet);
	
  /* Set up configuration stuff */
  panel_applet_add_preferences (applet, "/schemas/apps/flink/prefs", NULL);
	
  client = gconf_client_get_default ();

  gconf_client_add_dir (client, "/apps/flink",
			GCONF_CLIENT_PRELOAD_NONE, NULL);

  /* fetch gconf keys */
  mail_image = gconf_client_get_string(client, "/apps/flink/general/mailimage", &err);
  nomail_image = gconf_client_get_string(client, "/apps/flink/general/nomailimage", &err);

  if (err) 
    flink_gerror_handler(flink, err);

  /* FIXME this path should automatically be generated of course noone knows what strange paths people have*/
  if (!mail_image) {
    mail_image = g_strdup_printf("%s/flink/flink_small_mail.png", GNOME_ICONDIR);
    gconf_client_set_string(client, 
			    "/apps/flink/general/mailimage",
			    mail_image,
			    &err);
  }
  if (!nomail_image) {
    nomail_image = g_strdup_printf("%s/flink/flink_small_nomail.png", GNOME_ICONDIR);
    gconf_client_set_string(client,
			    "/apps/flink/general/nomailimage",
			    nomail_image,
			    &err);
  }
        
  /* Create the base applet */
  flink->applet = GTK_WIDGET(applet);
	
  flink->status_table = gtk_table_new(1,2,FALSE);
	
  flink->status_frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(flink->status_frame), GTK_SHADOW_NONE);
	
  flink->status_indicator = gtk_label_new("0");
	
  flink->status_image_mail = gtk_image_new_from_file(mail_image);
  flink->status_image_nomail = gtk_image_new_from_file(nomail_image);
	
  gtk_table_attach_defaults(GTK_TABLE(flink->status_table), 
			    flink->status_image_mail,
			    0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(flink->status_table), 
			    flink->status_image_nomail,
			    0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(flink->status_table), 
			    flink->status_indicator,
			    1, 2, 0, 1);

  /* some signal handling */
  g_signal_connect(GTK_OBJECT(flink->applet), "change_orient",
		   GTK_SIGNAL_FUNC(flink_orientation_change), (gpointer) flink);	
  g_signal_connect(GTK_OBJECT(flink->applet), "button_press_event",
		   GTK_SIGNAL_FUNC(flink_status_clicked), flink);
  g_signal_connect(GTK_OBJECT(flink->applet), "key-press-event",
		   GTK_SIGNAL_FUNC(flink_status_keypress), flink);

  gconf_client_set_int(client,
		       "/apps/flink/general/total_mail",
		       0,
		       NULL);
  gconf_client_notify_add(client,
			  "/apps/flink/general/total_mail",
			  flink_new_mail,
			  (gpointer) flink,
			  NULL,
			  NULL);
  gconf_client_notify_add(client,
			  "/apps/flink/accounts/accounts",
			  flink_realize_config,
			  (gpointer) flink,
			  NULL,
			  NULL);
                          
  gtk_container_add(GTK_CONTAINER(flink->status_frame), flink->status_table);
  gtk_container_add(GTK_CONTAINER(flink->applet), flink->status_frame);
	
  flink_mail_set_total_key();
	
  flink_realize_config(NULL,(guint) NULL,NULL,NULL);
	
  return flink;
}

/********************************************************
 * Function:	flink_properties_clicked		*
 *							*
 * Description:	when propertes menuitem is selected	*
 *							*
 * Arguments:	none					*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void 
flink_properties_clicked	() 
{
  GtkWidget 	*prefsdialog;
  prefsdialog = create_prefs_dialog();
  gtk_widget_show(prefsdialog);
}

/********************************************************
 * Function:	flink_about				*
 *							*
 * Description:	About dialog				*
 *							*
 * Arguments:	none					*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void 
flink_about	() 
{
  GtkWidget		*about_box;
  static const gchar 	*authors[] = 
    {"Marcus Leyman <marcus@leyman.nu>", NULL};

  about_box = gnome_about_new("Flink",
			      "1.1.0",
			      "Copyright (c) Marcus Leyman 2002",
			      _("Checks (multiple) Mailboxes at given interval"),
			      authors,
			      NULL,
			      NULL,
			      NULL);
  gtk_widget_show(about_box);
}

/********************************************************
 * Function:     flink_help                              *
 *                                                       *
 * Description:  Shows application help                  *
 *                                                       *
 * Arguments:    none                                    *
 *                                                       *
 * Returns:      nothing                                 *
 ********************************************************/
void
flink_help	()
{
  const gchar 	*helpfile = "";
  GError	*error;
  gboolean	success;
  success = gnome_help_display(helpfile,
			       NULL,
			       &error);
}

/********************************************************
 * Function:	flink_applet_menu_verbs			*
 *							*
 * Description:	rightclick menu items			*
 *							*
 * Arguments:						*
 *							*
 * Returns:	BonoboUIVerb				*
 ********************************************************/
static const BonoboUIVerb 
flink_applet_menu_verbs	[] = 
  {
    BONOBO_UI_UNSAFE_VERB ("FlinkProperties", flink_properties_clicked),
    BONOBO_UI_UNSAFE_VERB ("FlinkHelp", flink_help),
    BONOBO_UI_UNSAFE_VERB ("FlinkAbout", flink_about),
    BONOBO_UI_VERB_END
  };

/********************************************************
 * Function:	flink_applet_fill			*
 *							*
 * Description:	calls functions to fill applet		*
 *							*
 * Arguments:	PanelApplet *applet, applet to fill	*
 *							*
 * Returns:	TRUE, simply :)				*
 ********************************************************/
static gboolean
flink_applet_fill	(PanelApplet *applet) 
{
  Flink *flink;
	
  flink = flink_create_applet(applet);
		
  panel_applet_setup_menu_from_file (applet,
				     NULL,
				     "GNOME_Flink.xml",
				     NULL,
				     flink_applet_menu_verbs,
				     flink);
  gtk_widget_show_all(flink->applet);
  /*  flink_realize_config(NULL); */
  return TRUE;
}

/********************************************************
 * Function:	flink_factory				*
 *							*
 * Description:	Flink trying to act Warhol		*
 *							*
 * Arguments:	PanelApplet *applet, applet to fill	*
 *		const gchar *iid, oafid			*
 *		gpointer data,				*
 *							*
 * Returns:	TRUE if success				*
 ********************************************************/
static gboolean
flink_factory	(PanelApplet *applet, 
		 const gchar *iid, 
		 gpointer data) 
{

  gboolean retval = FALSE;

  if (!strcmp (iid, "OAFIID:GNOME_Flink"))
    retval = flink_applet_fill(applet);
	
  return retval;
}

/********************************************************
 * Function:	PANEL_APPLET_BONOBO_FACTORY		*
 *							*
 * Description:	magic					*
 ********************************************************/
PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_Flink_Factory", 
			     PANEL_TYPE_APPLET, 
			     "Flink mailchecker",
			     "0", 
			     flink_factory, 
			     NULL)

     
