/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

 Module:       FGKinemat.cpp
 Author:       Tony Peden, for flight control system authored by Jon S. Berndt
 Date started: 12/02/01

 ------------- Copyright (C) 2000 Anthony K. Peden -------------

 This program is free software; you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option) any later
 version.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 details.

 You should have received a copy of the GNU General Public License along with
 this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 Place - Suite 330, Boston, MA  02111-1307, USA.

 Further information about the GNU General Public License can also be found on
 the world wide web at http://www.gnu.org.

FUNCTIONAL DESCRIPTION
--------------------------------------------------------------------------------

HISTORY
--------------------------------------------------------------------------------

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
COMMENTS, REFERENCES,  and NOTES
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
INCLUDES
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

#include <math.h>
#include <float.h>

#include "FGKinemat.h"

namespace JSBSim {

static const char *IdSrc = "$Id: FGKinemat.cpp,v 1.8 2004/03/14 14:57:07 ehofman Exp $";
static const char *IdHdr = ID_FLAPS;

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
CLASS IMPLEMENTATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

FGKinemat::FGKinemat(FGFCS* fcs, FGConfigFile* AC_cfg) : FGFCSComponent(fcs),
                                                         AC_cfg(AC_cfg)
{
  string token,sOutputIdx;
  double tmpDetent;
  double tmpTime;

  Detents.clear();
  TransitionTimes.clear();

  OutputPct=0;
  DoScale = true;

  Type = AC_cfg->GetValue("TYPE");
  Name = AC_cfg->GetValue("NAME");
  AC_cfg->GetNextConfigLine();

  while ((token = AC_cfg->GetValue()) != string("/COMPONENT")) {
    *AC_cfg >> token;
    if (token == "INPUT") {
      token = AC_cfg->GetValue("INPUT");
      if( InputNodes.size() > 0 ) {
        cerr << "Kinemat can only accept one input" << endl;
      } else  {
        *AC_cfg >> token;
        InputNodes.push_back( resolveSymbol(token) );
      }

    } else if ( token == "DETENTS" ) {
      *AC_cfg >> NumDetents;
      if (NumDetents < 2) {
        cerr << "Kinemat must have at least 2 DETENTS" << endl;
      }
      for (int i=0;i<NumDetents;i++) {
        *AC_cfg >> tmpDetent;
        *AC_cfg >> tmpTime;
        Detents.push_back(tmpDetent);
        TransitionTimes.push_back(tmpTime);
      }
    } else if (token == "OUTPUT") {

      IsOutput = true;
      *AC_cfg >> sOutputIdx;
      OutputNode = PropertyManager->GetNode(sOutputIdx, true);
    } else if (token == "NOSCALE") {

      DoScale = false;
    }
  }
  FGFCSComponent::bind();
  treenode->Tie("output-norm", this, &FGKinemat::GetOutputPct );

  Debug(0);
}

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

FGKinemat::~FGKinemat()
{
  Debug(1);
}

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

bool FGKinemat::Run(void )
{
  double dt = fcs->GetState()->Getdt();

  Input = InputNodes[0]->getDoubleValue();

  if (DoScale)  Input *= Detents[NumDetents-1];

  Output = OutputNode->getDoubleValue();

  if (Input < Detents[0])
    Input = Detents[0];
  else if (Detents[NumDetents-1] < Input)
    Input = Detents[NumDetents-1];

  // Process all detent intervals the movement traverses until either the
  // final value is reached or the time interval has finished.
  while (0.0 < dt && Input != Output) {

    // Find the area where Output is in
    int ind;
    for (ind = 1; (Input < Output) ? Detents[ind] < Output : Detents[ind] <= Output ; ++ind)
      if (NumDetents <= ind)
        break;

    // A transition time of 0.0 means an infinite rate.
    // The output is reached in one step
    if (TransitionTimes[ind] <= 0.0) {
      Output = Input;
      break;
    } else {
      // Compute the rate in this area
      double Rate = (Detents[ind] - Detents[ind-1])/TransitionTimes[ind];
      // Compute the maximum input value inside this area
      double ThisInput = Input;
      if (ThisInput < Detents[ind-1])   ThisInput = Detents[ind-1];
      if (Detents[ind] < ThisInput)     ThisInput = Detents[ind];
      // Compute the time to reach the value in ThisInput
      double ThisDt = fabs((ThisInput-Output)/Rate);
      // and clip to the timestep size
      if (dt < ThisDt) ThisDt = dt;
      dt -= ThisDt;
      // Do the output calculation
      if (Output < Input)
        Output += ThisDt*Rate;
      else
        Output -= ThisDt*Rate;
    }
  }

  OutputPct = (Output-Detents[0])/(Detents[NumDetents-1]-Detents[0]);

  if (IsOutput) SetOutput();

  return true;
}

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//    The bitmasked value choices are as follows:
//    unset: In this case (the default) JSBSim would only print
//       out the normally expected messages, essentially echoing
//       the config files as they are read. If the environment
//       variable is not set, debug_lvl is set to 1 internally
//    0: This requests JSBSim not to output any messages
//       whatsoever.
//    1: This value explicity requests the normal JSBSim
//       startup messages
//    2: This value asks for a message to be printed out when
//       a class is instantiated
//    4: When this value is set, a message is displayed when a
//       FGModel object executes its Run() method
//    8: When this value is set, various runtime state variables
//       are printed out periodically
//    16: When set various parameters are sanity checked and
//       a message is printed out when they go out of bounds

void FGKinemat::Debug(int from)
{
  if (debug_lvl <= 0) return;

  if (debug_lvl & 1) { // Standard console startup message output
    if (from == 0) { // Constructor
      cout << "      INPUT: " << InputNodes[0]->getName() << endl;
      cout << "      DETENTS: " << NumDetents << endl;
      for (int i=0;i<NumDetents;i++) {
        cout << "        " << Detents[i] << " " << TransitionTimes[i] << endl;
      }
      if (IsOutput) cout << "      OUTPUT: " << OutputNode->getName() << endl;
      if (!DoScale) cout << "      NOSCALE" << endl;
    }
  }
  if (debug_lvl & 2 ) { // Instantiation/Destruction notification
    if (from == 0) cout << "Instantiated: FGKinemat" << endl;
    if (from == 1) cout << "Destroyed:    FGKinemat" << endl;
  }
  if (debug_lvl & 4 ) { // Run() method entry print for FGModel-derived objects
  }
  if (debug_lvl & 8 ) { // Runtime state variables
  }
  if (debug_lvl & 16) { // Sanity checking
  }
  if (debug_lvl & 64) {
    if (from == 0) { // Constructor
      cout << IdSrc << endl;
      cout << IdHdr << endl;
    }
  }
}
}
