/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

 Header:       FGRotation.h
 Author:       Jon Berndt
 Date started: 12/02/98

 ------------- Copyright (C) 1999  Jon S. Berndt (jsb@hal-pc.org) -------------

 This program is free software; you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option) any later
 version.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 details.

 You should have received a copy of the GNU General Public License along with
 this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 Place - Suite 330, Boston, MA  02111-1307, USA.

 Further information about the GNU General Public License can also be found on
 the world wide web at http://www.gnu.org.

HISTORY
--------------------------------------------------------------------------------
12/02/98   JSB   Created

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
SENTRY
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

#ifndef FGROTATION_H
#define FGROTATION_H

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
INCLUDES
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

#ifdef FGFS
#  include <simgear/compiler.h>
#  include <math.h>
#else
#  if defined (sgi) && !defined(__GNUC__)
#    include <math.h>
#  else
#    include <cmath>
#  endif
#endif

#include "FGModel.h"
#include "FGColumnVector3.h"

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
DEFINITIONS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

#define ID_ROTATION "$Id: FGRotation.h,v 1.5 2004/03/14 14:57:07 ehofman Exp $"

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
FORWARD DECLARATIONS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

namespace JSBSim {

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
CLASS DOCUMENTATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/** Models the rotational portion of the Equations of Motion.
    Note: The order of rotations used in this class corresponds to a 3-2-1 sequence,
    or Y-P-R, or Z-Y-X, if you prefer.
    @see Cooke, Zyda, Pratt, and McGhee, "NPSNET: Flight Simulation Dynamic Modeling
    Using Quaternions", Presence, Vol. 1, No. 4, pp. 404-420  Naval Postgraduate
    School, January 1994
    @see D. M. Henderson, "Euler Angles, Quaternions, and Transformation Matrices",
    JSC 12960, July 1977
    @see Richard E. McFarland, "A Standard Kinematic Model for Flight Simulation at
    NASA-Ames", NASA CR-2497, January 1975
    @see Barnes W. McCormick, "Aerodynamics, Aeronautics, and Flight Mechanics",
    Wiley & Sons, 1979 ISBN 0-471-03032-5
    @see Bernard Etkin, "Dynamics of Flight, Stability and Control", Wiley & Sons,
    1982 ISBN 0-471-08936-2
*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
CLASS DECLARATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

class FGRotation : public FGModel
{
public:
  FGRotation(FGFDMExec*);
  ~FGRotation();

  bool Run(void);

  inline FGColumnVector3& GetPQR(void) {return vPQR;}
  inline double GetPQR(int axis) const {return vPQR(axis);}
  inline FGColumnVector3& GetAeroPQR(void) {return vAeroPQR;}
  inline double GetAeroPQR(int axis) const {return vAeroPQR(axis);}
  inline FGColumnVector3& GetPQRdot(void) {return vPQRdot;}
  inline double GetPQRdot(int idx) const {return vPQRdot(idx);}
  inline FGColumnVector3& GetEuler(void) {return vEuler;}
  inline double GetEuler(int axis) const {return vEuler(axis);}
  inline FGColumnVector3& GetEulerRates(void) { return vEulerRates; }
  inline double GetEulerRates(int axis) const { return vEulerRates(axis); }
  inline void SetPQR(FGColumnVector3 tt) {vPQR = tt;}
  inline void SetPQR(double p, double q, double r) {vPQR(eP)=p;
                                                    vPQR(eQ)=q;
                                                    vPQR(eR)=r;}
  inline void SetAeroPQR(FGColumnVector3 tt) {vAeroPQR = tt;}
  inline void SetAeroPQR(double p, double q, double r) {vAeroPQR(eP)=p;
                                                        vAeroPQR(eQ)=q;
                                                        vAeroPQR(eR)=r;}
  inline void SetEuler(FGColumnVector3 tt) {vEuler = tt;}

  inline double Getphi(void) const {return vEuler(1);}
  inline double Gettht(void) const {return vEuler(2);}
  inline double Getpsi(void) const {return vEuler(3);}

  inline double GetCosphi(void) const {return cPhi;}
  inline double GetCostht(void) const {return cTht;}
  inline double GetCospsi(void) const {return cPsi;}

  inline double GetSinphi(void) const {return sPhi;}
  inline double GetSintht(void) const {return sTht;}
  inline double GetSinpsi(void) const {return sPsi;}

  void bind(void);
  void unbind(void);

private:
  FGColumnVector3 vPQR;
  FGColumnVector3 vAeroPQR;
  FGColumnVector3 vPQRdot;
  FGColumnVector3 vPQRdot_prev[4];
  FGColumnVector3 vMoments;
  FGColumnVector3 vEuler;
  FGColumnVector3 vEulerRates;

  double cTht,sTht;
  double cPhi,sPhi;
  double cPsi,sPsi;

  double dt;

  void GetState(void);

  void Debug(int from);
};
}
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
#endif

