/** \file builtin_help.c

    Functions for printing usage information of builtin commands. This
	file is automatically generated from the file builtin_help.hdr and
	various help files in the doc_src directory.
*/

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <wchar.h>
#include <sys/types.h>

#include "util.h"
#include "common.h"
#include "halloc_util.h"

/**
	Hashtable storing the help text
*/
static hash_table_t tbl;

static void builtin_help_init();

char *builtin_help_get( const wchar_t *cmd )
{
	builtin_help_init();

	return (char *)hash_get( &tbl, (void *)cmd );
}

/**
	Initialize help hash table. Don't invoke it manually, 
	it is called by builtin_help_get automatically.
*/
static void builtin_help_init()
{
	static int is_help_init = 0;
	if( is_help_init )
		return;
	is_help_init=1;
	halloc_register_function( global_context, (void (*)(void *))&hash_destroy, &tbl );
	hash_init( &tbl, &hash_wcs_func, &hash_wcs_cmp );
	hash_put( &tbl, L"source",
		"\n"
		"   Description\n"
		"       Evaluates the commands of the specified file in the cu"
		"rrent shell. This\n"
		"       is different from starting a new process to perform th"
		"e commands (i.e.\n"
		"       fish < FILENAME) since the commands will be evaluated "
		"by the current\n"
		"       shell, which means that changes in environment variabl"
		"es, etc., will\n"
		"       remain.\n"
		"\n"
		"   Example\n"
		"       . \x7e" "/.fish\n"
		"\n"
		"       causes fish to reread its initialization file.\n"
		"\n"
		"" );

	hash_put( &tbl, L"and",
		"   Synopsis\n"
		"       COMMAND1; and COMMAND2\n"
		"\n"
		"   Description\n"
		"       The and builtin is used to execute a command if the cu"
		"rrent exit status\n"
		"       (as set by the last previous command) is 0.\n"
		"\n"
		"       The and command does not change the current exit statu"
		"s.\n"
		"\n"
		"       The exit status of the last foreground command to exit"
		" can always be\n"
		"       accessed using the $status variable.\n"
		"\n"
		"   Example\n"
		"       The following code runs the make command to build a pr"
		"ogram, if the\n"
		"       build succceds, the program is installed. If either st"
		"ep fails, make\n"
		"       clean is run, which removes the files created by the b"
		"uild process\n"
		"\n"
		"       make; and make install; or make clean\n"
		"\n"
		"" );

	hash_put( &tbl, L"begin",
		"   Synopsis\n"
		"       begin; [COMMANDS...;] end\n"
		"\n"
		"   Description\n"
		"       The begin builtin is used to create a new block of cod"
		"e. The block is\n"
		"       unconditionally executed. begin; ...; end is equivalen"
		"t to if true;\n"
		"       ...; end. The begin command is used to group any numbe"
		"r of commands\n"
		"       into a block. The reason for doing so is usually eithe"
		"r to introduce a\n"
		"       new variable scope, to redirect the input or output of"
		" a set of\n"
		"       commands as a group, or to specify precedence when usi"
		"ng the\n"
		"       conditional commands like and.\n"
		"\n"
		"       The begin command does not change the current exit sta"
		"tus.\n"
		"\n"
		"   Example\n"
		"       The following code sets a number of variables inside o"
		"f a block scope.\n"
		"       Since the variables are set inside the block and have "
		"local scope, they\n"
		"       will be automatically deleted when the block ends.\n"
		"\n"
		"       begin\n"
		"\t    set -x PIRATE Yarrr\n"
		"\t    ...\n"
		"       end\n"
		"       # This will not output anything, since PIRATE went out"
		" of scope at the end of\n"
		"       # the block and was killed\n"
		"       echo $PIRATE\n"
		"\n"
		"       In the following code, all output is redirected to the"
		" file out.html.\n"
		"\n"
		"       begin\n"
		"\t    echo $xml_header\n"
		"\t    echo $html_header\n"
		"\t    if test -e $file\n"
		"\t\t ...\n"
		"\t    end\n"
		"\t    ...\n"
		"\n"
		"       end > out.html\n"
		"\n"
		"" );

	hash_put( &tbl, L"bg",
		"   Synopsis\n"
		"       bg [PID...]\n"
		"\n"
		"   Description\n"
		"       Sends the specified jobs to the background. A backgrou"
		"nd job is\n"
		"       executed simultaneously with fish, and does not have a"
		"ccess to the\n"
		"       keyboard. If no job is specified, the last job to be u"
		"sed is put in the\n"
		"       background. If PID is specified, the jobs with the spe"
		"cified group ids\n"
		"       are put in the background.\n"
		"\n"
		"       The PID of the desired process is usually found by usi"
		"ng process\n"
		"       globbing.\n"
		"\n"
		"   Example\n"
		"       bg %0 will put the job with job id 0 in the background"
		".\n"
		"\n"
		"" );

	hash_put( &tbl, L"bind",
		"   Synopsis\n"
		"       bind [OPTIONS] [BINDINGS...]\n"
		"\n"
		"       The bind builtin causes fish to add the readline style"
		" bindings\n"
		"       specified by BINDINGS to the list of key bindings, as "
		"if they appeared\n"
		"       in your \x7e" "/.fish_inputrc file.\n"
		"\n"
		"       For more information on the syntax keyboard bindings, "
		"use man readline\n"
		"       to access the readline documentation. The availiable c"
		"ommands are\n"
		"       listed in the Command Line Editor section of the fish "
		"manual - but you\n"
		"       may also use any fish command! To write such commands,"
		" see the\n"
		"       commandline builtin. It\xe2" "\x80" "\x99" "s good pra"
		"ctice to put the code into a function\n"
		"       -b and bind to the function name.\n"
		"\n"
		"   Description\n"
		"       \xc2" "\xb7" " -M MODE or --set-mode=MODE sets the cur"
		"rent input mode to MODE.\n"
		"\n"
		"   Example\n"
		"       bind -M vi changes to the vi input mode\n"
		"\n"
		"       bind \xe2" "\x80" "\x99" "\xe2" "\x80" "\x99" "\\M-j\xe2" ""
		"\x80" "\x99" ": jobs\xe2" "\x80" "\x99" " Binds the jobs comm"
		"and to the Alt-j keyboard\n"
		"       shortcut\n"
		"\n"
		"" );

	hash_put( &tbl, L"block",
		"   Synopsis\n"
		"       block [OPTIONS...]\n"
		"\n"
		"   Description\n"
		"       \xc2" "\xb7" " -l or --local Release the block at the "
		"end of the currently innermost\n"
		"\t block scope\n"
		"\n"
		"       \xc2" "\xb7" " -g or --global Never automatically rele"
		"ase the lock\n"
		"\n"
		"       \xc2" "\xb7" " -e or --erase Release global block\n"
		"\n"
		"   Example\n"
		"       block -g\n"
		"       #Do something that should not be interrupted\n"
		"       block -e\n"
		"\n"
		"" );

	hash_put( &tbl, L"break",
		"   Synopsis\n"
		"       LOOP_CONSTRUCT; [COMMANDS...] break; [COMMANDS...] end"
		"\n"
		"\n"
		"   Description\n"
		"       The break builtin is used to halt a currently running "
		"loop, such as a\n"
		"       for loop or a while loop. It is usually added inside o"
		"f a conditional\n"
		"       block such as an if statement or a switch statement.\n"
		"\n"
		"   Example\n"
		"       The following code searches all .c files for smurfs, a"
		"nd halts at the\n"
		"       first occurrence.\n"
		"\n"
		"       for i in *.c\n"
		"\t   if grep smurf $i\n"
		"\t       echo Smurfs are present in $i\n"
		"\t       break\n"
		"\t   end\n"
		"       end\n"
		"\n"
		"" );

	hash_put( &tbl, L"builtin",
		"   Synopsis\n"
		"       builtin BUILTINNAME [OPTIONS...]\n"
		"\n"
		"   Description\n"
		"       \xc2" "\xb7" " -n or --names List the names of all def"
		"ined builtins\n"
		"\n"
		"       Prefixing a command with the word \xe2" "\x80" "\x99" ""
		"builtin\xe2" "\x80" "\x99" " forces fish to ignore any\n"
		"       aliases with the same name.\n"
		"\n"
		"   Example\n"
		"       builtin jobs\n"
		"\n"
		"       causes fish to execute the jobs builtin, even if a fun"
		"ction named jobs\n"
		"       exists.\n"
		"\n"
		"" );

	hash_put( &tbl, L"case",
		"   Synopsis\n"
		"       switch VALUE; [case [WILDCARD...]; [COMMANDS...]; ...]"
		" end\n"
		"\n"
		"   Description\n"
		"       The switch statement is used to perform one of several"
		" blocks of\n"
		"       commands depending on whether a specified value equals"
		" one of several\n"
		"       wildcarded values. The case statement is used together"
		" with the switch\n"
		"       statement in order to determine which block should be "
		"performed.\n"
		"\n"
		"       Each case command is given one or more parameter. The "
		"first case\n"
		"       command with a parameter that matches the string speci"
		"fied in the\n"
		"       switch command will be evaluated. case parameters may "
		"contain\n"
		"       wildcards. These need to be escaped or quoted in order"
		" to avoid regular\n"
		"       wildcard expansion using filenames.\n"
		"\n"
		"       Note that fish does not fall through on case statement"
		"s. Though the\n"
		"       syntax may look a bit like C switch statements, it beh"
		"aves more like\n"
		"       the case stamantes of traditional shells.\n"
		"\n"
		"       Also note that command substitutions in a case stateme"
		"nt will be\n"
		"       evaluated even if it\xe2" "\x80" "\x99" "s body is not"
		" taken. This may seem\n"
		"       counterintuitive at first, but it is unavoidable, sinc"
		"e it would be\n"
		"       impossible to know if a case command will evaluate to "
		"true before all\n"
		"       forms of parameter expansion have been performed for t"
		"he case command.\n"
		"\n"
		"   Example\n"
		"       If the variable $animal contains the name of an animal"
		", the following\n"
		"       code would attempt to classify it:\n"
		"\n"
		"       switch $animal\n"
		"\t   case cat\n"
		"\t       echo evil\n"
		"\t   case wolf dog human moose dolphin whale\n"
		"\t       echo mammal\n"
		"\t   case duck goose albatross\n"
		"\t       echo bird\n"
		"\t   case shark trout stingray\n"
		"\t       echo fish\n"
		"\t   case \xe2" "\x80" "\x99" "*\xe2" "\x80" "\x99" "\n"
		"\t       echo I have no idea what a $animal is\n"
		"       end\n"
		"\n"
		"       If the above code was run with $animal set to whale, t"
		"he output would\n"
		"       be mammal.\n"
		"\n"
		"" );

	hash_put( &tbl, L"cd",
		"   Synopsis\n"
		"       cd [DIRECTORY]\n"
		"\n"
		"   Description Changes the current\n"
		"       directory. If DIRECTORY is supplied it will become the"
		" new directory.\n"
		"       If DIRECTORY is a relative path, the paths found in th"
		"e CDPATH\n"
		"       environment variable array will be tried as prefixes f"
		"or the specified\n"
		"       path. If CDPATH is not set, it is assumed to be \xe2" ""
		"\x80" "\x99" ".\xe2" "\x80" "\x99" ". If DIRECTORY is\n"
		"       not specified, $HOME will be the new directory.\n"
		"\n"
		"" );

	hash_put( &tbl, L"command",
		"   Synopsis\n"
		"       command COMMANDNAME [OPTIONS...]\n"
		"\n"
		"   Description\n"
		"       prefixing a command with the word \xe2" "\x80" "\x99" ""
		"command\xe2" "\x80" "\x99" " forces fish to ignore any\n"
		"       aliases or builtins with the same name.\n"
		"\n"
		"   Example\n"
		"       command ls\n"
		"\n"
		"       causes fish to execute the ls program, even if there e"
		"xists a \xe2" "\x80" "\x99" "ls\xe2" "\x80" "\x99" "\n"
		"       alias.\n"
		"\n"
		"" );

	hash_put( &tbl, L"commandline",
		"   Synopsis\n"
		"       commandline [OPTIONS] [CMD]\n"
		"\n"
		"   Description\n"
		"       \xc2" "\xb7" " CMD is the new value of the commandline"
		". If unspecified, the current\n"
		"\t value of the commandline is written to standard output.\n"
		"\n"
		"       The following switches change what the commandline bui"
		"ltin does\n"
		"\n"
		"       \xc2" "\xb7" " -C or --cursor set or get the current c"
		"ursor position, not the\n"
		"\t contents of the buffer. If no argument is given, the curre"
		"nt cursor\n"
		"\t position is printed, otherwise the argument is interpreted"
		" as the new\n"
		"\t cursor position.\n"
		"\n"
		"       \xc2" "\xb7" " -f or --function inject readline functi"
		"ons into the reader. This\n"
		"\t option can not be combined with any other option. It will "
		"cause any\n"
		"\t additional arguments to be interpreted as readline functio"
		"ns, and\n"
		"\t these functions will be injected into the reader, so that "
		"they will\n"
		"\t be returned to the reader before any additional actual key"
		"presses are\n"
		"\t read.\n"
		"\n"
		"       The following switches change the way commandline upda"
		"tes the\n"
		"       commandline buffer\n"
		"\n"
		"       \xc2" "\xb7" " -a or --append do not remove the curren"
		"t commandline, append the\n"
		"\t specified string at the end of it\n"
		"\n"
		"       \xc2" "\xb7" " -i or --insert do not remove the curren"
		"t commandline, insert the\n"
		"\t specified string at the current cursor position\n"
		"\n"
		"       \xc2" "\xb7" " -r or --replace remove the current comm"
		"andline and replace it with\n"
		"\t the specified string (default)\n"
		"\n"
		"       The following switches change what part of the command"
		"line is printed\n"
		"       or updated\n"
		"\n"
		"       \xc2" "\xb7" " -b or --current-buffer select the entir"
		"e buffer (default)\n"
		"\n"
		"       \xc2" "\xb7" " -j or --current-job select the current "
		"job\n"
		"\n"
		"       \xc2" "\xb7" " -p or --current-process select the curr"
		"ent process\n"
		"\n"
		"       \xc2" "\xb7" " -t or --current-token select the curren"
		"t token.\n"
		"\n"
		"       The following switch changes the way commandline print"
		"s the current\n"
		"       commandline buffer\n"
		"\n"
		"       \xc2" "\xb7" " -c or --cut-at-cursor only print select"
		"ion up until the current\n"
		"\t cursor position\n"
		"\n"
		"       \xc2" "\xb7" " -o or --tokenize tokenize the selection"
		" and print one string-type\n"
		"\t token per line\n"
		"\n"
		"       If commandline is called during a call to complete a g"
		"iven string using\n"
		"       complete -C STRING, commandline will consider the spec"
		"ified string to\n"
		"       be the current contents of the commandline.\n"
		"\n"
		"   Example\n"
		"       commandline -j $history[3]\n"
		"\n"
		"       replaces the job under the cursor with the third item "
		"from the\n"
		"       commandline history.\n"
		"\n"
		"" );

	hash_put( &tbl, L"complete",
		"   Synopsis\n"
		"       complete (-c\x7c" "--command\x7c" "-p\x7c" "--path) CO"
		"MMAND [(-s\x7c" "--short-option)\n"
		"       SHORT_OPTION] [(-l\x7c" "--long-option\x7c" "-o\x7c" ""
		"--old-option) LONG_OPTION\n"
		"       [(-a\x7c" "\x7c" "--arguments) OPTION_ARGUMENTS] [(-d\x7c" ""
		"--description) DESCRIPTION]\n"
		"\n"
		"   Description\n"
		"       For an introduction to how to specify completions, see"
		" the section\n"
		"       Writing your own completions of the fish manual.\n"
		"\n"
		"       \xc2" "\xb7" " COMMAND is the name of the command for "
		"which to add a completion\n"
		"\n"
		"       \xc2" "\xb7" " SHORT_OPTION is a one character option "
		"for the command\n"
		"\n"
		"       \xc2" "\xb7" " LONG_OPTION is a multi character option"
		" for the command\n"
		"\n"
		"       \xc2" "\xb7" " OPTION_ARGUMENTS is parameter containin"
		"g a space-separated list of\n"
		"\t possible option-arguments, which may contain subshells\n"
		"\n"
		"       \xc2" "\xb7" " DESCRIPTION is a description of what th"
		"e option and/or option\n"
		"\t arguments do\n"
		"\n"
		"       \xc2" "\xb7" " -C STRING or --do-complete=STRING makes"
		" complete try to find all\n"
		"\t possible completions for the specified string\n"
		"\n"
		"       \xc2" "\xb7" " -e or --erase implies that the specifie"
		"d completion should be deleted\n"
		"\n"
		"       \xc2" "\xb7" " -f or --no-files specifies that the opt"
		"ion specified by this\n"
		"\t completion may not be followed by a filename\n"
		"\n"
		"       \xc2" "\xb7" " -n or --condition specifies a shell com"
		"mand that must return 0 if the\n"
		"\t completion is to be used. This makes it possible to specif"
		"y\n"
		"\t completions that should only be used in some cases.\n"
		"\n"
		"       \xc2" "\xb7" " -o or --old-option implies that the com"
		"mand uses old long style\n"
		"\t options with only one dash\n"
		"\n"
		"       \xc2" "\xb7" " -p or --path implies that the string CO"
		"MMAND is the full path of the\n"
		"\t command\n"
		"\n"
		"       \xc2" "\xb7" " -r or --require-parameter specifies tha"
		"t the option specified by this\n"
		"\t completion always must have an option argument, i.e. may n"
		"ot be\n"
		"\t followed by another option\n"
		"\n"
		"       \xc2" "\xb7" " -u or --unauthorative implies that ther"
		"e may be more options than the\n"
		"\t ones specified, and that fish should not assume that optio"
		"ns not\n"
		"\t listed are spelling errors\n"
		"\n"
		"       \xc2" "\xb7" " -x or --exclusive implies both -r and -"
		"f\n"
		"\n"
		"       Command specific tab-completions in fish are based on "
		"the notion of\n"
		"       options and arguments. An option is a parameter which "
		"begins with a\n"
		"       hyphen, such as \xe2" "\x80" "\x99" "-h\xe2" "\x80" "\x99" ""
		", \xe2" "\x80" "\x99" "-help\xe2" "\x80" "\x99" " or \xe2" "\x80" ""
		"\x99" "--help\xe2" "\x80" "\x99" ". Arguments are parameters\n"
		"       that do not begin with a hyphen. Fish recognizes three"
		" styles of\n"
		"       options, the same styles as the GNU version of the get"
		"opt library.\n"
		"       These styles are:\n"
		"\n"
		"       \xc2" "\xb7" " Short options, like \xe2" "\x80" "\x99" ""
		"-a\xe2" "\x80" "\x99" ". Short options are a single character"
		" long,\n"
		"\t are preceded by a single hyphen and may be grouped togethe"
		"r (like\n"
		"\t \xe2" "\x80" "\x99" "-la\xe2" "\x80" "\x99" ", which is eq"
		"uivalent to \xe2" "\x80" "\x99" "-l -a\xe2" "\x80" "\x99" ")."
		" Option arguments may be\n"
		"\t specified in the following parameter (\xe2" "\x80" "\x99" ""
		"-w 32\xe2" "\x80" "\x99" ") or by appending the\n"
		"\t option with the value (\xe2" "\x80" "\x99" "-w32\xe2" "\x80" ""
		"\x99" ").\n"
		"\n"
		"       \xc2" "\xb7" " Old style long options, like \xe2" "\x80" ""
		"\x99" "-Wall\xe2" "\x80" "\x99" ". Old style long options can"
		" be\n"
		"\t more than one character long, are preceded by a single hyp"
		"hen and may\n"
		"\t not be grouped together. Option arguments are specified in"
		" the\n"
		"\t following parameter (\xe2" "\x80" "\x99" "-ao null\xe2" "\x80" ""
		"\x99" ").\n"
		"\n"
		"       \xc2" "\xb7" " GNU style long options, like \xe2" "\x80" ""
		"\x99" "--colors\xe2" "\x80" "\x99" ". GNU style long options "
		"can\n"
		"\t be more than one character long, are preceded by two hyphe"
		"ns, and may\n"
		"\t not be grouped together. Option arguments may be specified"
		" in the\n"
		"\t following parameter (\xe2" "\x80" "\x99" "--quoting-style "
		"shell\xe2" "\x80" "\x99" ") or by appending the\n"
		"\t option with a \xe2" "\x80" "\x99" "=\xe2" "\x80" "\x99" " "
		"and the value (\xe2" "\x80" "\x99" "--quoting-style=shell\xe2" ""
		"\x80" "\x99" "). GNU style\n"
		"\t long options may be abbreviated so long as the abbreviatio"
		"n is unique\n"
		"\t (\xe2" "\x80" "\x99" "--h\xe2" "\x80" "\x99" " is equivale"
		"nt to \xe2" "\x80" "\x99" "--help\xe2" "\x80" "\x99" " if hel"
		"p is the only long option\n"
		"\t beginning with an \xe2" "\x80" "\x99" "h\xe2" "\x80" "\x99" ""
		").\n"
		"\n"
		"       The options for specifying command name, command path,"
		" or command\n"
		"       switches may all be used multiple times to specify mul"
		"tiple commands\n"
		"       which have the same completion or multiple switches ac"
		"cepted by a\n"
		"       command.\n"
		"\n"
		"       When erasing completions, it is possible to either era"
		"se all\n"
		"       completions for a specific command by specifying compl"
		"ete -e -c\n"
		"       COMMAND, or by specifying a specific completion option"
		" to delete by\n"
		"       specifying either a long, short or old style option.\n"
		"\n"
		"   Example\n"
		"       The short style option -o for the gcc command requires"
		" that a file\n"
		"       follows it. This can be done using writing complete -c"
		" gcc -s o -r.\n"
		"\n"
		"       The short style option -d for the grep command require"
		"s that one of the\n"
		"       strings \xe2" "\x80" "\x99" "read\xe2" "\x80" "\x99" ""
		", \xe2" "\x80" "\x99" "skip\xe2" "\x80" "\x99" " or \xe2" "\x80" ""
		"\x99" "recurse\xe2" "\x80" "\x99" " is used. This can be spec"
		"ified\n"
		"       writing complete -c grep -s d -x -a \xe2" "\x80" "\x99" ""
		"read skip recurse\xe2" "\x80" "\x99" ".\n"
		"\n"
		"       The su command takes any username as an argument. User"
		"names are given\n"
		"       as the first colon-separated field in the file /etc/pa"
		"sswd. This can be\n"
		"       specified as: complete -x -c su -d \xe2" "\x80" "\x99" ""
		"Username\xe2" "\x80" "\x99" " -a \xe2" "\x80" "\x99" "(cat /e"
		"tc/passwd\x7c" "cut\n"
		"       -d : -f 1)\xe2" "\x80" "\x99" " .\n"
		"\n"
		"       The rpm command has several different modes. If the -e"
		" or --erase flag\n"
		"       has been specified, rpm should delete one or more pack"
		"ages, in which\n"
		"       case several switches related to deleting packages are"
		" valid, like the\n"
		"       nodeps switch.\n"
		"\n"
		"       This can be written as:\n"
		"\n"
		"       complete -c rpm -n \xe2" "\x80" "\x99" "__fish_contain"
		"s_opt -s e erase\xe2" "\x80" "\x99" " -l nodeps -d \xe2" "\x80" ""
		"\x99" "Don\xe2" "\x80" "\x99" "t\n"
		"       check dependencies\xe2" "\x80" "\x99" "\n"
		"\n"
		"       where __fish_contains_opt is a function that checks th"
		"e commandline\n"
		"       buffer for the presence of a specified set of options."
		"\n"
		"\n"
		"" );

	hash_put( &tbl, L"continue",
		"\n"
		"   Synopsis\n"
		"       LOOP_CONSTRUCT; [COMMANDS...;] continue; [COMMANDS...;"
		"] end\n"
		"\n"
		"   Description\n"
		"       The continue builtin is used to skip the current lap o"
		"f the innermost\n"
		"       currently running loop, such as a for loop or a while "
		"loop. It is\n"
		"       usually added inside of a conditional block such as an"
		" if statement or\n"
		"       a switch statement.\n"
		"\n"
		"   Example\n"
		"       The following code removes all tmp files without smurf"
		"s.\n"
		"\n"
		"       for i in *.tmp\n"
		"\t   if grep smurf $i\n"
		"\t       continue\n"
		"\t   end\n"
		"\t   rm $i\n"
		"       end\n"
		"\n"
		"" );

	hash_put( &tbl, L"else",
		"   Synopsis\n"
		"       if CONDITION; COMMANDS_TRUE...; [else; COMMANDS_FALSE."
		"..;] end\n"
		"\n"
		"   Description\n"
		"       if will execute the command CONDITION. If the conditio"
		"n\xe2" "\x80" "\x99" "s exit status\n"
		"       is 0, the commands COMMANDS_TRUE will execute. If it i"
		"s not 0 and else\n"
		"       is given, COMMANDS_FALSE will be executed. Hint: use b"
		"egin; ...; end\n"
		"       for complex conditions.\n"
		"\n"
		"   Example\n"
		"       The command if test -f foo.txt; echo foo.txt exists; e"
		"lse; echo foo.txt\n"
		"       does not exist; end will print foo.txt exists if the f"
		"ile foo.txt\n"
		"       exists and is a regular file, otherwise it will print "
		"foo.txt does not\n"
		"       exist.\n"
		"\n"
		"" );

	hash_put( &tbl, L"end",
		"   Synopsis\n"
		"       begin; [COMMANDS...] end\n"
		"       if CONDITION; COMMANDS_TRUE...; [else; COMMANDS_FALSE."
		"..;] end\n"
		"       while CONDITION; COMMANDS...; end\n"
		"       for VARNAME in [VALUES...]; COMMANDS...; end\n"
		"       switch VALUE; [case [WILDCARD...]; [COMMANDS...]; ...]"
		" end\n"
		"\n"
		"   Description\n"
		"       end ends a block of commands. For more information, re"
		"ad the\n"
		"       documentation for the block constructs, such as if, fo"
		"r and while.\n"
		"\n"
		"       The end command does not change the current exit statu"
		"s.\n"
		"\n"
		"" );

	hash_put( &tbl, L"eval",
		"   Synopsis\n"
		"       eval [COMMANDS...]\n"
		"\n"
		"   Description\n"
		"       The eval builtin causes fish to evaluate the specified"
		" parameters as a\n"
		"       command. If more than one parameter is specified, all "
		"parameters will\n"
		"       be joined using a space character as a separator.\n"
		"\n"
		"   Example\n"
		"       set cmd ls\n"
		"       eval $cmd\n"
		"\n"
		"       will call the ls command.\n"
		"\n"
		"" );

	hash_put( &tbl, L"exec",
		"   Synopsis\n"
		"       exec COMMAND [OPTIONS...]\n"
		"\n"
		"   Description\n"
		"       The exec builtin is used to replace the currently runn"
		"ing shells\n"
		"       process image with a new command. On successful comple"
		"tion, exec never\n"
		"       returns. exec can not be used inside a pipeline.\n"
		"\n"
		"   Example\n"
		"       exec emacs starts up the emacs text editor. When emacs"
		" exits, the\n"
		"       session will terminate.\n"
		"\n"
		"" );

	hash_put( &tbl, L"exit",
		"   Synopsis\n"
		"       exit [STATUS]\n"
		"\n"
		"   Description\n"
		"       The exit builtin causes fish to exit. If STATUS is sup"
		"plied, it will be\n"
		"       converted to an integer and used as the exit code. Oth"
		"erwise the exit\n"
		"       code will be that of the last command executed.\n"
		"\n"
		"       If exit is called while sourcing a file (using the . b"
		"uiltin) the rest\n"
		"       of the file will be skipped, but the shell itself will"
		" not exit.\n"
		"\n"
		"" );

	hash_put( &tbl, L"fg",
		"   Synopsis\n"
		"       fg [PID]\n"
		"\n"
		"   Description\n"
		"       Sends the specified job to the foreground. While a for"
		"eground job is\n"
		"       executed, fish is suspended. If no job is specified, t"
		"he last job to be\n"
		"       used is put in the foreground. If PID is specified, th"
		"e job with the\n"
		"       specified group id is put in the foreground.\n"
		"\n"
		"       The PID of the desired process is usually found by usi"
		"ng process\n"
		"       globbing.\n"
		"\n"
		"   Example\n"
		"       fg %0 will put the job with job id 0 in the foreground"
		".\n"
		"\n"
		"" );

	hash_put( &tbl, L"for",
		"   Synopsis\n"
		"       for VARNAME in [VALUES...]; COMMANDS...; end\n"
		"\n"
		"   Description\n"
		"       for is a loop construct. It will perform the commands "
		"specified by\n"
		"       COMMANDS multiple times. Each time the environment var"
		"iable specified\n"
		"       by VARNAME is assigned a new value from VALUES. If VAL"
		"UES is empty,\n"
		"       COMMANDS will not be executed at all.\n"
		"\n"
		"   Example\n"
		"       The command\n"
		"\n"
		"       for i in foo bar baz; echo $i; end\n"
		"\n"
		"       would output:\n"
		"\n"
		"       foo\n"
		"       bar\n"
		"       baz\n"
		"\n"
		"" );

	hash_put( &tbl, L"function",
		"   Synopsis\n"
		"       function [OPTIONS] NAME; BODY; end\n"
		"\n"
		"   Description\n"
		"       \xc2" "\xb7" " -d DESCRIPTION or --description=DESCRIP"
		"TION is a description of what\n"
		"\t the function does, suitable as a completion description\n"
		"\n"
		"       \xc2" "\xb7" " -j PID or  --on-job-exit PID tells fish"
		" to run this function when the\n"
		"\t job with group id PID exits. Instead of PID, the string \xe2" ""
		"\x80" "\x99" "caller\xe2" "\x80" "\x99" " can\n"
		"\t be specified. This is only legal when in a command substit"
		"ution, and\n"
		"\t will result in the handler being triggered by the exit of "
		"the job\n"
		"\t which created this command substitution.\n"
		"\n"
		"       \xc2" "\xb7" " -p PID or  --on-process-exit PID tells "
		"fish to run this function when\n"
		"\t the fish child process with process id PID exits\n"
		"\n"
		"       \xc2" "\xb7" " -s or --on-signal SIGSPEC tells fish to"
		" run this function when the\n"
		"\t signal SIGSPEC is delivered. SIGSPEC can be a signal numbe"
		"r, or the\n"
		"\t signal name, such as SIGHUP (or just HUP)\n"
		"\n"
		"       \xc2" "\xb7" " -v or --on-variable VARIABLE_NAME tells"
		" fish to run this function\n"
		"\t when the variable VARIABLE_NAME changes value\n"
		"\n"
		"       This builtin command is used to create a new function."
		" A function is a\n"
		"       list of commands that will be executed when the name o"
		"f the function is\n"
		"       entered. The function\n"
		"\n"
		"       function hi\n"
		"\t    echo hello\n"
		"       end\n"
		"\n"
		"       will write hello whenever the user enters hi.\n"
		"\n"
		"       If the user enters any additional arguments after the "
		"function, they\n"
		"       are inserted into the environment variable array argv."
		"\n"
		"\n"
		"   Example\n"
		"       function ll\n"
		"\t    ls -l $argv\n"
		"       end\n"
		"\n"
		"       will run the ls command, using the -l option, while pa"
		"ssing on any\n"
		"       additional files and switches to ls.\n"
		"\n"
		"       function mkdir -d \xe2" "\x80" "\x99" "Create a direct"
		"ory and set CWD\xe2" "\x80" "\x99" "\n"
		"\t    mkdir $argv\n"
		"\t    if test $status = 0\n"
		"\t\t switch $argv[(count $argv)]\n"
		"\t\t      case \xe2" "\x80" "\x99" "-*\xe2" "\x80" "\x99" "\n"
		"\n"
		"\t\t      case \xe2" "\x80" "\x99" "*\xe2" "\x80" "\x99" "\n"
		"\t\t\t   cd $argv[(count $argv)]\n"
		"\t\t\t   return\n"
		"\t\t end\n"
		"\t    end\n"
		"       end\n"
		"\n"
		"       will run the mkdir command, and if it is successful, c"
		"hange the current\n"
		"       working directory to the one just created.\n"
		"\n"
		"" );

	hash_put( &tbl, L"functions",
		"   Synopsis\n"
		"       functions [-e] FUNCTIONS...\n"
		"\n"
		"   Description\n"
		"       This builtin command is used to print or erase functio"
		"ns.\n"
		"\n"
		"       \xc2" "\xb7" " -a or --all list all functions, even th"
		"ose whose name start with an\n"
		"\t underscore.\n"
		"\n"
		"       \xc2" "\xb7" " -d DESCRIPTION or --description=DESCRIP"
		"TION change the description of\n"
		"\t this function\n"
		"\n"
		"       \xc2" "\xb7" " -e or --erase causes the specified func"
		"tions to be erased.\n"
		"\n"
		"       \xc2" "\xb7" " -h or --help display a help message and"
		" exit\n"
		"\n"
		"       \xc2" "\xb7" " -n or --names list only the names of al"
		"l defined functions, not their\n"
		"\t definition\n"
		"\n"
		"       \xc2" "\xb7" " -q or --query test if the specified fun"
		"ctions exist. Does not output\n"
		"\t anything, but the builtins exit status is the number of fu"
		"nctions\n"
		"\t specified that were not defined.\n"
		"\n"
		"       The default behavior of functions when called with no "
		"arguments, is to\n"
		"       print the names and definitions of all defined functio"
		"ns. If any non-\n"
		"       switch parameters are given, only the definition of th"
		"e specified\n"
		"       functions are printed.\n"
		"\n"
		"       Automatically loaded functions can not be removed usin"
		"g functions -e.\n"
		"       Either remove the definition file or change the $fish_"
		"function_path\n"
		"       variable to remove autoloaded functions.\n"
		"\n"
		"       The exit status of the functions builtin is the number"
		" functions\n"
		"       specified in the argument list that do not exist.\n"
		"\n"
		"" );

	hash_put( &tbl, L"if",
		"   Synopsis\n"
		"       if CONDITION; COMMANDS_TRUE...; [else; COMMANDS_FALSE."
		"..;] end\n"
		"\n"
		"   Description\n"
		"       if will execute the command CONDITION. If the conditio"
		"n\xe2" "\x80" "\x99" "s exit status\n"
		"       is 0, the commands COMMANDS_TRUE will execute. If the "
		"exit status is\n"
		"       not 0 and else is given, COMMANDS_FALSE will be execut"
		"ed.\n"
		"\n"
		"       In order to use the exit status of mutiple commands as"
		" the condition of\n"
		"       an if block, use begin; ...; end and the short circut "
		"commands and and\n"
		"       or.\n"
		"\n"
		"       The exit status of the last foreground command to exit"
		" can always be\n"
		"       accessed using the $status variable.\n"
		"\n"
		"   Example\n"
		"       if test -f foo.txt\n"
		"\t    echo foo.txt exists\n"
		"       else\n"
		"\t    echo foo.txt does not exist\n"
		"       end\n"
		"\n"
		"\twill print foo.txt exists if the file foo.txt exists and is"
		" a regular\n"
		"       file, otherwise it will print foo.txt does not exist.\n"
		"\n"
		"" );

	hash_put( &tbl, L"jobs",
		"   jobs-synopsis\n"
		"       jobs [OPTIONS] [PID]\n"
		"\n"
		"   Description\n"
		"       The jobs builtin causes fish to print a list of the cu"
		"rrently running\n"
		"       jobs and their status.\n"
		"\n"
		"       jobs accepts the following switches:\n"
		"\n"
		"       \xc2" "\xb7" " -c or --command print the command name "
		"for each process in jobs\n"
		"\n"
		"       \xc2" "\xb7" " -g or --group only print the group id o"
		"f each job\n"
		"\n"
		"       \xc2" "\xb7" " -h or --help display a help message and"
		" exit\n"
		"\n"
		"       \xc2" "\xb7" " -l or --last only the last job to be st"
		"arted is printed\n"
		"\n"
		"       \xc2" "\xb7" " -p or --pid print the process id for ea"
		"ch process in all jobs\n"
		"\n"
		"       On systems that supports this feature, jobs will print"
		" the CPU usage of\n"
		"       each job since the last command was executed. The CPU "
		"usage is\n"
		"       expressed as a percentage of full CPU activity. Note t"
		"hat on\n"
		"       multiprocessor systems, the total activity may be more"
		" than 100%.\n"
		"\n"
		"" );

	hash_put( &tbl, L"not",
		"   Synopsis\n"
		"       not COMMAND [OPTIONS...]\n"
		"\n"
		"   Description\n"
		"       The not builtin is used to negate the exit status of a"
		"nother command.\n"
		"\n"
		"   Example\n"
		"       The following code reports an error and exits if no fi"
		"le named spoon\n"
		"       can be found.\n"
		"\n"
		"       if not test -f spoon\n"
		"\t    echo There is no spoon\n"
		"\t    exit 1\n"
		"       end\n"
		"\n"
		"" );

	hash_put( &tbl, L"or",
		"   Synopsis\n"
		"       COMMAND1; or COMMAND2\n"
		"\n"
		"   Description\n"
		"       The or builtin is used to execute a command if the cur"
		"rent exit status\n"
		"       (as set by the last previous command) is not 0.\n"
		"\n"
		"       The or command does not change the current exit status"
		".\n"
		"\n"
		"       The exit status of the last foreground command to exit"
		" can always be\n"
		"       accessed using the $status variable.\n"
		"\n"
		"   Example\n"
		"       The following code runs the make command to build a pr"
		"ogram, if the\n"
		"       build succceds, the program is installed. If either st"
		"ep fails, make\n"
		"       clean is run, which removes the files created by the b"
		"uild process\n"
		"\n"
		"       make; and make install; or make clean\n"
		"\n"
		"" );

	hash_put( &tbl, L"random",
		"   Synopsis\n"
		"       random [SEED]\n"
		"\n"
		"   Description\n"
		"       The random command is used to generate a random number"
		" in the interval\n"
		"       0<=N<32767. If an argument is given, it is used to see"
		"d the random\n"
		"       number generator. This can be useful for debugging pur"
		"poses, where it\n"
		"       can be desirable to get the same random number sequenc"
		"e multiple times.\n"
		"       If the random number generator is called without first"
		" seeding it, the\n"
		"       current time will be used as the seed.\n"
		"\n"
		"   Example\n"
		"       The following code will count down from a random numbe"
		"r to 1:\n"
		"\n"
		"       for i in (seq (random) -1 1)\n"
		"\t    echo $i\n"
		"\t    sleep\n"
		"       end\n"
		"\n"
		"" );

	hash_put( &tbl, L"return",
		"   Synopsis\n"
		"       function NAME; [COMMANDS...;] return [STATUS]; [COMMAN"
		"DS...;] end\n"
		"\n"
		"   Description\n"
		"       The return builtin is used to halt a currently running"
		" function. It is\n"
		"       usually added inside of a conditional block such as an"
		" if statement or\n"
		"       a switch statement to conditionally stop the executing"
		" function and\n"
		"       return to the caller, but it can also be used to speci"
		"fy the exit\n"
		"       status of a function.\n"
		"\n"
		"       \xc2" "\xb7" " STATUS is the return status of the func"
		"tion. If unspecified, the\n"
		"\t status is unchanged.\n"
		"\n"
		"   Example\n"
		"       The following code is an implementation of the false c"
		"ommand as a fish\n"
		"       function\n"
		"\n"
		"       function false\n"
		"\t    return 1\n"
		"       end\n"
		"\n"
		"" );

	hash_put( &tbl, L"read",
		"   Synopsis\n"
		"       read [OPTIONS] [VARIABLES...]\n"
		"\n"
		"   Description\n"
		"       The read builtin causes fish to read one line from sta"
		"ndard input and\n"
		"       store the result in one or more environment variables."
		"\n"
		"\n"
		"       \xc2" "\xb7" " -c CMD or --command=CMD specifies that "
		"the initial string in the\n"
		"\t interactive mode command buffer should be CMD.\n"
		"\n"
		"       \xc2" "\xb7" " -e or --export specifies that the varia"
		"bles will be exported to\n"
		"\t subshells.\n"
		"\n"
		"       \xc2" "\xb7" " -g or --global specifies that the varia"
		"bles will be made global.\n"
		"\n"
		"       \xc2" "\xb7" " -p PROMPT_CMD or --prompt=PROMPT_CMD sp"
		"ecifies that the output of the\n"
		"\t shell command PROMPT_CMD should be used as the prompt for "
		"the\n"
		"\t interactive mode prompt. The default prompt command is set"
		"_color\n"
		"\t green; echo read; set_color normal; echo \xe2" "\x80" "\x99" ""
		"> \xe2" "\x80" "\x99" ".\n"
		"\n"
		"       \xc2" "\xb7" " -u or --unexport causes the specified e"
		"nvironment not to be exported\n"
		"\t to child processes\n"
		"\n"
		"       \xc2" "\xb7" " -U or --universal causes the specified "
		"environment variable to be\n"
		"\t made universal. If this option is supplied, the variable w"
		"ill be\n"
		"\t shared between all the current users fish instances on the"
		" current\n"
		"\t computer, and will be preserved across restarts of the she"
		"ll.\n"
		"\n"
		"       \xc2" "\xb7" " -x or --export causes the specified env"
		"ironment variable to be\n"
		"\t exported to child processes\n"
		"\n"
		"       Read starts by reading a single line of input from std"
		"in, the line is\n"
		"       then tokenized using the IFS environment variable. Eac"
		"h variable\n"
		"       specified in VARIABLES is then assigned one tokenized "
		"string element.\n"
		"       If there are more tokens than variables, the complete "
		"remainder is\n"
		"       assigned to the last variable.\n"
		"\n"
		"   Example\n"
		"       echo hello\x7c" "read foo\n"
		"\n"
		"       Will cause the variable $foo to be assigned the value "
		"hello.\n"
		"\n"
		"" );

	hash_put( &tbl, L"set",
		"   Synopsis\n"
		"       set [SCOPE_OPTIONS]\n"
		"       set [OPTIONS] VARIABLE_NAME VALUES...\n"
		"       set [OPTIONS] VARIABLE_NAME[INDICES]... VALUES...\n"
		"       set (-q \x7c" " --query) [SCOPE_OPTIONS] VARIABLE_NAME"
		"S...\n"
		"       set (-e \x7c" " --erase) [SCOPE_OPTIONS] VARIABLE_NAME"
		"\n"
		"       set (-e \x7c" " --erase) [SCOPE_OPTIONS] VARIABLE_NAME"
		"[INDICES]...\n"
		"\n"
		"       The set builtin causes fish to assign the variable VAR"
		"IABLE_NAME the\n"
		"       values VALUES....\n"
		"\n"
		"   Description\n"
		"       \xc2" "\xb7" " -e or --erase causes the specified envi"
		"ronment variable to be erased\n"
		"\n"
		"       \xc2" "\xb7" " -l or --local forces the specified envi"
		"ronment variable to be given a\n"
		"\t scope that is local to the current block, even if a variab"
		"le with the\n"
		"\t given name exists and is non-local\n"
		"\n"
		"       \xc2" "\xb7" " -g or --global causes the specified env"
		"ironment variable to be given\n"
		"\t a global scope. Non-global variables disappear when the bl"
		"ock they\n"
		"\t belong to ends\n"
		"\n"
		"       \xc2" "\xb7" " -U or --universal causes the specified "
		"environment variable to be\n"
		"\t given a universal scope. If this option is supplied, the v"
		"ariable\n"
		"\t will be shared between all the current users fish instance"
		"s on the\n"
		"\t current computer, and will be preserved across restarts of"
		" the shell.\n"
		"\n"
		"       \xc2" "\xb7" " -n or --names List only the names of al"
		"l defined variables, not their\n"
		"\t value\n"
		"\n"
		"       \xc2" "\xb7" " -q or --query test if the specified var"
		"iable names are defined. Does\n"
		"\t not output anything, but the builtins exit status is the n"
		"umber of\n"
		"\t variables specified that were not defined.\n"
		"\n"
		"       \xc2" "\xb7" " -u or --unexport causes the specified e"
		"nvironment not to be exported\n"
		"\t to child processes\n"
		"\n"
		"       \xc2" "\xb7" " -x or --export causes the specified env"
		"ironment variable to be\n"
		"\t exported to child processes\n"
		"\n"
		"       If set is called with no arguments, the names and valu"
		"es of all\n"
		"       environment variables are printed. If some of the scop"
		"e or export flags\n"
		"       have been given, only the variables matching the speci"
		"fied scope are\n"
		"       printed.\n"
		"\n"
		"       If a variable is set to more than one value, the varia"
		"ble will be an\n"
		"       array with the specified elements. If a variable is se"
		"t to zero\n"
		"       elements, it will become an array with zero elements.\n"
		"\n"
		"       If the variable name is one or more array elements, su"
		"ch as PATH[1 3\n"
		"       7], only those array elements specified will be change"
		"d. When array\n"
		"       indices are specified to set, multiple arguments may b"
		"e used to specify\n"
		"       additional indexes, e.g. set PATH[1] PATH[4] /bin /sbi"
		"n. If you specify\n"
		"       a negative index when expanding or assigning to an arr"
		"ay variable, the\n"
		"       index will be calculated from the end of the array. Fo"
		"r example, the\n"
		"       index -1 means the last index of an array.\n"
		"\n"
		"       The scoping rules when creating or updating a variable"
		" are:\n"
		"\n"
		"       1.  If a variable is explicitly set to either universa"
		"l, global or\n"
		"\t   local, that setting will be honored. If a variable of th"
		"e same name\n"
		"\t   exists in a different scope, that variable will not be c"
		"hanged.\n"
		"\n"
		"       2.  If a variable is not explicitly set to be either u"
		"niversal, global\n"
		"\t   or local, but has been previously defined, the previos v"
		"ariable\n"
		"\t   scope is used.\n"
		"\n"
		"       3.  If a variable is not explicitly set to be either u"
		"niversal, global\n"
		"\t   or local and has never before been defined, the variable"
		" will be\n"
		"\t   local to the currently executing functions. If no functi"
		"on is\n"
		"\t   executing, the variable will be global.\n"
		"\n"
		"       The exporting rules when creating or updating a variab"
		"le are identical\n"
		"       to the scoping rules for variables:\n"
		"\n"
		"       1.  If a variable is explicitly set to either be expor"
		"ted or not\n"
		"\t   exported, that setting will be honored.\n"
		"\n"
		"       2.  If a variable is not explicitly set to be exported"
		" or not exported,\n"
		"\t   but has been previously defined, the previous exporting "
		"rule for\n"
		"\t   the variable is kept.\n"
		"\n"
		"       3.  If a variable is not explicitly set to be either g"
		"lobal or local\n"
		"\t   and has never before been defined, the variable will not"
		" be\n"
		"\t   exported.\n"
		"\n"
		"       In query mode, the scope to be examined can be specifi"
		"ed.\n"
		"\n"
		"       In erase mode, if variable indices are specified, only"
		" the specified\n"
		"       slices of the array variable will be erased. When eras"
		"ing an entire\n"
		"       variable (i.e. no slicing), the scope of the variable "
		"to be erased can\n"
		"       be specified. That way, a global variable can be erase"
		"d even if a local\n"
		"       variable with the same name exists. Scope can not be s"
		"pecified when\n"
		"       erasing a slice of an array. The innermost scope is al"
		"ways used.\n"
		"\n"
		"       The set command requires all switch arguments to come "
		"before any non-\n"
		"       switch arguments. For example, set flags -l will have "
		"the effect of\n"
		"       setting the value of the variable flags to \xe2" "\x80" ""
		"\x99" "-l\xe2" "\x80" "\x99" ", not making the\n"
		"       variable local.\n"
		"\n"
		"       In assignment mode, set exits with an exit status of z"
		"ero it the\n"
		"       variable assignments where sucessfully performed, with"
		" a non-zero exit\n"
		"       status otherwise. In query mode, the exit status is th"
		"e number of\n"
		"       variables that where not found. In erase mode, set exi"
		"ts with a zero\n"
		"       exit status in case of success, with a non-zero exit s"
		"tatus if the\n"
		"       commandline was invalid, if the variable was write-pro"
		"tected or if the\n"
		"       variable did not exist.\n"
		"\n"
		"   Example\n"
		"       set -xg will print all global, exported variables.\n"
		"\n"
		"       set foo hi sets the value of the variable foo to be hi"
		".\n"
		"\n"
		"       set -e smurf removes the variable smurf.\n"
		"\n"
		"       set PATH[4] \x7e" "/bin changes the fourth element of "
		"the PATH array to \x7e" "/bin\n"
		"\n"
		"" );

	hash_put( &tbl, L"status",
		"   Synopsis\n"
		"       status [OPTION]\n"
		"\n"
		"   Description\n"
		"       \xc2" "\xb7" " -c or --is-command-substitution returns"
		" 0 if fish is currently\n"
		"\t executing a command substitution\n"
		"\n"
		"       \xc2" "\xb7" " -b or --is-block returns 0 if fish is c"
		"urrently executing a block of\n"
		"\t code\n"
		"\n"
		"       \xc2" "\xb7" " -i or --is-interactive returns 0 if fis"
		"h is interactive,\n"
		"\t i.e.connected to a keyboard\n"
		"\n"
		"       \xc2" "\xb7" " -l or --is-login returns 0 if fish is a"
		" login shell, i.e. if fish\n"
		"\t should perform login tasks such as setting up the PATH.\n"
		"\n"
		"" );

	hash_put( &tbl, L"switch",
		"   Synopsis\n"
		"       switch VALUE; [case [WILDCARD...]; [COMMANDS...]; ...]"
		" end\n"
		"\n"
		"   Description\n"
		"       The switch statement is used to perform one of several"
		" blocks of\n"
		"       commands depending on whether a specified value equals"
		" one of several\n"
		"       wildcarded values. The case statement is used together"
		" with the switch\n"
		"       statement in order to determine which block should be "
		"performed.\n"
		"\n"
		"       Each case command is given one or more parameter. The "
		"first case\n"
		"       command with a parameter that matches the string speci"
		"fied in the\n"
		"       switch command will be evaluated. case parameters may "
		"contain\n"
		"       wildcards. These need to be escaped or quoted in order"
		" to avoid regular\n"
		"       wildcard expansion using filenames.\n"
		"\n"
		"       Note that fish does not fall through on case statement"
		"s. Though the\n"
		"       syntax may look a bit like C switch statements, it beh"
		"aves more like\n"
		"       the case stamantes of traditional shells.\n"
		"\n"
		"       Also note that command substitutions in a case stateme"
		"nt will be\n"
		"       evaluated even if it\xe2" "\x80" "\x99" "s body is not"
		" taken. This may seem\n"
		"       counterintuitive at first, but it is unavoidable, sinc"
		"e it would be\n"
		"       impossible to know if a case command will evaluate to "
		"true before all\n"
		"       forms of parameter expansion have been performed for t"
		"he case command.\n"
		"\n"
		"   Example\n"
		"       If the variable $animal contains the name of an animal"
		", the following\n"
		"       code would attempt to classify it:\n"
		"\n"
		"       switch $animal\n"
		"\t   case cat\n"
		"\t       echo evil\n"
		"\t   case wolf dog human moose dolphin whale\n"
		"\t       echo mammal\n"
		"\t   case duck goose albatross\n"
		"\t       echo bird\n"
		"\t   case shark trout stingray\n"
		"\t       echo fish\n"
		"\t   case \xe2" "\x80" "\x99" "*\xe2" "\x80" "\x99" "\n"
		"\t       echo I have no idea what a $animal is\n"
		"       end\n"
		"\n"
		"       If the above code was run with $animal set to whale, t"
		"he output would\n"
		"       be mammal.\n"
		"\n"
		"" );

	hash_put( &tbl, L"ulimit",
		"   Synopsis\n"
		"       ulimit [OPTIONS] [LIMIT]\n"
		"\n"
		"   Description\n"
		"       The ulimit builtin provides control over the resources"
		" available to the\n"
		"       shell and to processes started by it. The -H and -S op"
		"tions specify\n"
		"       that the hard or soft limit is set for the given resou"
		"rce. A hard limit\n"
		"       cannot be increased once it is set; a soft limit may b"
		"e increased up to\n"
		"       the value of the hard limit. If neither -H nor -S is s"
		"pecified, both\n"
		"       the soft and hard limits are set. The value of limit c"
		"an be a number in\n"
		"       the unit specified for the resource or one of the spec"
		"ial values hard,\n"
		"       soft, or unlimited, which stand for the current hard l"
		"imit, the current\n"
		"       soft limit, and no limit, respectively. If limit is om"
		"itted, the\n"
		"       current value of the soft limit of the resource is pri"
		"nted, unless the\n"
		"       -H option is given. When more than one resource is spe"
		"cified, the limit\n"
		"       name and unit are printed before the value. Other opti"
		"ons are\n"
		"       interpreted as follows:\n"
		"\n"
		"       \xc2" "\xb7" " -a or --all Set or get all current limi"
		"ts\n"
		"\n"
		"       \xc2" "\xb7" " -c or --core-size The maximum size of c"
		"ore files created\n"
		"\n"
		"       \xc2" "\xb7" " -d or --data-size The maximum size of a"
		" process\xe2" "\x80" "\x99" "s data segment\n"
		"\n"
		"       \xc2" "\xb7" " -f or --file-size The maximum size of f"
		"iles created by the shell\n"
		"\n"
		"       \xc2" "\xb7" " -l or --lock-size The maximum size that"
		" may be locked into memory\n"
		"\n"
		"       \xc2" "\xb7" " -m or --resident-set-size The maximum r"
		"esident set size\n"
		"\n"
		"       \xc2" "\xb7" " -n or --file-descriptor-count The maxim"
		"um number of open file\n"
		"\t descriptors (most systems do not allow this value to be se"
		"t)\n"
		"\n"
		"       \xc2" "\xb7" " -s or --stack-size The maximum stack si"
		"ze\n"
		"\n"
		"       \xc2" "\xb7" " -t or --cpu-time The maximum amount of "
		"cpu time in seconds\n"
		"\n"
		"       \xc2" "\xb7" " -u or --process-count The maximum numbe"
		"r of processes available to a\n"
		"\t single user\n"
		"\n"
		"       \xc2" "\xb7" " -v or --virtual-memory-size The maximum"
		" amount of virtual memory\n"
		"\t available to the shell. If supported by OS.\n"
		"\n"
		"       If limit is given, it is the new value of the specifie"
		"d resource. If no\n"
		"       option is given, then -f is assumed. Values are in kil"
		"obytes, except\n"
		"       for -t, which is in seconds and -n and -u, which are u"
		"nscaled values.\n"
		"       The return status is 0 unless an invalid option or arg"
		"ument is\n"
		"       supplied, or an error occurs while setting a new limit"
		".\n"
		"\n"
		"       The fish implementation of ulimit should behave identi"
		"cally to the\n"
		"       implementation in bash, except for these differences:\n"
		"\n"
		"       \xc2" "\xb7" " Fish ulimit supports GNU-style long opt"
		"ions for all switches\n"
		"\n"
		"       \xc2" "\xb7" " Fish ulimit does not support the -p opt"
		"ion for getting the pipe size.\n"
		"\t The bash implementation consists of a compile-time check t"
		"hat\n"
		"\t empirically guesses this number by writing to a pipe and w"
		"aiting for\n"
		"\t SIGPIPE.\n"
		"\n"
		"       \xc2" "\xb7" " Fish ulimit does not support getting th"
		"e values of multiple limits in\n"
		"\t one command, except by using the -a switch\n"
		"\n"
		"   Example\n"
		"       ulimit -Hs 64\n"
		"\n"
		"       would set the hard stack size limit to 64 kB:\n"
		"\n"
		"" );

	hash_put( &tbl, L"while",
		"   Synopsis\n"
		"       while CONDITION; COMMANDS...; end\n"
		"\n"
		"   Description\n"
		"       The while builtin causes fish to continually execute C"
		"ONDITION and\n"
		"       execute COMMANDS as long as CONDITION returned with st"
		"atus 0. If\n"
		"       CONDITION is false on the first time, COMMANDS will no"
		"t be executed at\n"
		"       all. Hints: use begin; ...; end for complex conditions"
		"; more complex\n"
		"       control can be achieved with while true containing a b"
		"reak.\n"
		"\n"
		"   Example\n"
		"       while test -f foo.txt; echo file exists; sleep 10; end"
		"\n"
		"\n"
		"       causes fish to print the line \xe2" "\x80" "\x99" "fil"
		"e exists\xe2" "\x80" "\x99" " at 10 second intervals as\n"
		"       long as the file foo.txt exists.\n"
		"\n"
		"" );

	hash_put( &tbl, L"count",
		"   Synopsis\n"
		"       count $VARIABLE\n"
		"\n"
		"   Description\n"
		"       count prints the number of arguments that were passed "
		"to it. This is\n"
		"       usually used to find out how many elements an environm"
		"ent variable\n"
		"       array contains, but this is not the only potential usa"
		"ge for the count\n"
		"       command.\n"
		"\n"
		"       The count command does not accept any options, not eve"
		"n \xe2" "\x80" "\x99" "-h\xe2" "\x80" "\x99" ". This way\n"
		"       the user does not have to worry about an array contain"
		"ing elements such\n"
		"       as dashes. fish performs a special check when invoking"
		" the count\n"
		"       program, and if the user uses a help option, this help"
		" page is\n"
		"       displayed, but if a help option is contained inside of"
		" a variable or is\n"
		"       the result of expansion, it will be passed on to the c"
		"ount program.\n"
		"\n"
		"       Count exits with a non-zero exit status if no argument"
		"s where passed to\n"
		"       it, with zero otherwise.\n"
		"\n"
		"   Example\n"
		"       count $PATH\n"
		"\n"
		"       returns the number of directories in the users PATH va"
		"riable.\n"
		"\n"
		"       count *.txt\n"
		"\n"
		"       returns the number of files in the current working dir"
		"ectory ending\n"
		"       with the suffix \xe2" "\x80" "\x99" ".txt\xe2" "\x80" ""
		"\x99" ".\n"
		"\n"
		"" );

}
