/* -*-C-*- */

/* $Id: lex.l,v 1.90 2007/04/03 19:03:02 nicm Exp $ */

/*
 * Copyright (c) 2006 Nicholas Marriott <nicm@users.sourceforge.net>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

%option yylineno

%{
#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "fdm.h"
#include "y.tab.h"

extern int 			 yylex(void);
extern __dead printflike1 void	 yyerror(const char *, ...);

char 				*read_str(char, int);

/*
 * For include files. Because of yacc read-ahead include works in a weird way:
 * the include var is parsed by lex and causes a buffer switch to the new file
 * yacc then carries on with this as if it was inline, until EOF is reached
 * and lex calls yywrap() which switches back to the old file. If this seems
 * like a horrible hack to you, that's because it is.
 */
struct fileent {
	FILE			*yyin;
	int		 	 yylineno;
	YY_BUFFER_STATE		 yybuffer;
	char			*curfile;
};
ARRAY_DECL(, struct fileent *)	 filestack;
char				*curfile;

/* Redefine ECHO to cause an error. */
#define ECHO yyerror("syntax error");

%}

%%

[0-9]+ {
	const char	*errstr;

        yylval.number = strtonum(yytext, 0, LLONG_MAX, &errstr);
	if (errstr != NULL)
		yyerror("number is %s", errstr);

        return (NUMBER);
}
\' {
	yylval.string = read_str('\'', 0);
        return (STRING);
}
\" {
	yylval.string = read_str('"', 1);
        return (STRING);
}
include[ \t]*\" {
	struct replpath	 rp;
	char		*s;

	if (parse_tags == NULL) {
		strb_create(&parse_tags);
		default_tags(&parse_tags, NULL, NULL);
	}
	
	rp.str = read_str('"', 1);
	s = replacepath(&rp, parse_tags, NULL, NULL);
	xfree(rp.str);

	include_start(s);

	return (INCLUDE);
}
include[ \t]*\' {
	struct replpath	 rp;
	char		*s;

	if (parse_tags == NULL) {
		strb_create(&parse_tags);
		default_tags(&parse_tags, NULL, NULL);
	}
	
	rp.str = read_str('\'', 0);
	s = replacepath(&rp, parse_tags, NULL, NULL);
	xfree(rp.str);

	include_start(s);

	return (INCLUDE);
}
\$[A-Za-z][A-Za-z0-9_]* {
        yylval.string = xstrdup(yytext);
	return (STRMACRO);
}
\$\{[A-Za-z][A-Za-z0-9_]*\} {
        yylval.string = xstrdup(yytext);
	return (STRMACROB);
}
\%[A-Za-z][A-Za-z0-9_]* {
        yylval.string = xstrdup(yytext);
	return (NUMMACRO);
}
\%\{[A-Za-z][A-Za-z0-9_]*\} {
        yylval.string = xstrdup(yytext);
	return (NUMMACROB);
}
(byte|bytes|b|B) return (TOKBYTES);
(day|days) return (TOKDAYS);
(default-user|defuser) return (TOKDEFUSER);
(delete-oversized|deltoobig) return (TOKDELTOOBIG);
(gigabyte|gigabytes|g|G|gb|GB) return (TOKGIGABYTES);
(hour|hours) return (TOKHOURS);
(kilobyte|kilobytes|k|K|kb|KB) return (TOKKILOBYTES);
(lock-types|locktypes) return (TOKLOCKTYPES);
(maximum-size|maxsize) return (TOKMAXSIZE);
(megabyte|megabytes|m|M|mb|MB) return (TOKMEGABYTES);
(minute|minutes) return (TOKMINUTES);
(month|months) return (TOKMONTHS);
(second|seconds) return (TOKSECONDS);
(week|weeks) return (TOKWEEKS);
(year|years) return (TOKYEARS);
account	return (TOKACCOUNT);
accounts return (TOKACCOUNTS);
action return (TOKACTION);
actions return (TOKACTIONS);
add-from return (TOKADDFROM);
add-header return (TOKADDHEADER);
age return (TOKAGE);
all return (TOKALL);
allow-multiple return (TOKALLOWMANY);
and return (TOKAND);
any-name return (TOKANYNAME);
any-size return (TOKANYSIZE);
any-type return (TOKANYTYPE);
append return (TOKAPPEND);
append-string return (TOKAPPENDSTRING);
attachment return (TOKATTACHMENT);
body return (TOKBODY);
cache return (TOKCACHE);
case return (TOKCASE);
compress return (TOKCOMPRESS);
continue return (TOKCONTINUE);
count return (TOKCOUNT);
disabled return (TOKDISABLED);
domain return (TOKDOMAIN);
domains return (TOKDOMAINS);
dotlock return (LCKDOTLOCK);
drop return (TOKDROP);
exec return (TOKEXEC);
fcntl return (LCKFCNTL);
file-group return (TOKFILEGROUP);
file-umask return (TOKFILEUMASK);
flock return (LCKFLOCK);
folder return (TOKFOLDER);
from-headers return (TOKFROMHEADERS);
group return (TOKGROUP);
groups return (TOKGROUPS);
header return (TOKHEADER);
headers return (TOKHEADERS);
imap return (TOKIMAP);
imaps return (TOKIMAPS);
in return (TOKIN);
invalid return (TOKINVALID);
keep return (TOKKEEP);
lock-file return (TOKLOCKFILE);
maildir return (TOKMAILDIR);
maildirs return (TOKMAILDIRS);
match return (TOKMATCH);
matched return (TOKMATCHED);
mbox return (TOKMBOX);
nntp return (TOKNNTP);
no-received return (TOKNORECEIVED);
none return (TOKNONE);
not return (TOKNOT);
or return (TOKOR);
pass return (TOKPASS);
pipe return (TOKPIPE);
pop3 return (TOKPOP3);
pop3s return (TOKPOP3S);
port return (TOKPORT);
proxy return (TOKPROXY);
purge-after return (TOKPURGEAFTER);
queue-high return (TOKQUEUEHIGH);
queue-low return (TOKQUEUELOW);
remove-header return (TOKREMOVEHEADER);
returns return (TOKRETURNS);
rewrite return (TOKREWRITE);
server return (TOKSERVER);
set return (TOKSET);
size return (TOKSIZE);
smtp return (TOKSMTP);
stdin return (TOKSTDIN);
stdout return (TOKSTDOUT);
string return (TOKSTRING);
tag return (TOKTAG);
tagged return (TOKTAGGED);
timeout return (TOKTIMEOUT);
to return (TOKTO);
total-size return (TOKTOTALSIZE);
unmatched return (TOKUNMATCHED);
unmatched-mail return (TOKIMPLACT);
user return (TOKUSER);
users return (TOKUSERS);
value return (TOKVALUE);
write return (TOKWRITE);
= return ('=');
== return (TOKEQ);
!= return (TOKNE);
\+ return ('+');
\( return ('(');
\) return (')');
\, return (',');
\< return ('<');
\> return ('>');
\{ return ('{');
\} return ('}');
\#.*\n /* ignore comments */;
\n /* ignore end of line */;
[ \t]+ /* ignore whitespace */;

%%

char *
read_str(char endc, int esc)
{
	int		 done = 0, ch, valid;
	size_t		 pos = 0, len, off, slen;
	char	         name[MAXNAMESIZE], *s, *buf;
	struct macro	*macro;

	len = 24;
        buf = xmalloc(len + 1);

        while (!done) {
		ch = input();
		if (ch == endc) {
                        done = 1;
                        continue;
		}
                switch (ch) {
		case 0:
		case EOF:
			yyerror("unterminated string");
                case '\\':
			if (!esc)
				break;
                        switch (ch = input()) {
			case 0:
			case EOF:
				yyerror("unterminated string");
                        case 'r':
                                ch = '\r';
                                break;
                        case 'n':
                                ch = '\n';
                                break;
                        case 't':
                                ch = '\t';
                                break;
                        }
                        break;
		case '$':
		case '%':
			if (!esc)
				break;

			name[0] = ch;
			off = 1;

			/* first character */
			ch = input();
			if (ch == 0 || ch == EOF)
				yyerror("unterminated string");
			if (ch != '{') {
				unput(ch);
				ch = name[0];
				break;
			}

			/* remaining characters */
			valid = 1;
			do {
				ch = input();
				if (ch == 0 || ch == EOF)
					yyerror("unterminated string");
				if (ch == '}')
					break;
				if (!ismacro(ch))
					valid = 0;
				name[off++] = ch;
			} while (off < (sizeof name));
			if (ch != '}')
				yyerror("missing } or macro name too long");
			name[off] = '\0';
			if (!valid || !ismacrofirst(name[1]))
				yyerror("invalid macro name: %s", name);

			if ((macro = find_macro(name)) == NULL)
				yyerror("undefined macro: %s", name);

			if (macro->type == MACRO_NUMBER)
 				xasprintf(&s, "%lld", macro->value.num);
			else
				s = macro->value.str;
			slen = strlen(s);

			ENSURE_FOR(buf, len, pos, slen + 1);
			memcpy(buf + pos, s, slen);
			pos += slen;

			if (macro->type == MACRO_NUMBER)
				xfree(s);
			continue;
                }

                buf[pos] = ch;
                pos++;
                ENSURE_SIZE(buf, len, pos);
        }

        buf[pos] = '\0';

	return (buf);
}

void
include_start(char *file)
{
	char		*path;
	struct fileent	*top;

	if (*file == '\0')
		yyerror("invalid include file");

	top = xmalloc(sizeof *top);
	top->yyin = yyin;
	top->yylineno = yylineno;	
	top->yybuffer = YY_CURRENT_BUFFER;
	top->curfile = curfile;
	ARRAY_ADD(&filestack, top, struct fileent *);

	yyin = fopen(file, "r");
	if (yyin == NULL) {
		xasprintf(&path, "%s/%s", xdirname(conf.conf_file), file);
		if (access(path, R_OK) != 0)
			yyerror("%s: %s", path, strerror(errno));
		yyin = fopen(path, "r");
		if (yyin == NULL)
			yyerror("%s: %s", path, strerror(errno));
		curfile = path;
		xfree(file);
	} else
		curfile = file;

	log_debug2("including file %s", curfile);
	yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));
	yylineno = 1;
}

int
include_finish(void)
{
	struct fileent	*top;

	if (ARRAY_EMPTY(&filestack)) {
		ARRAY_FREE(&filestack);
		return (1);
	}

	log_debug2("finished file %s", curfile);
	yy_delete_buffer(YY_CURRENT_BUFFER);

	top = ARRAY_LAST(&filestack, struct fileent *);

	yyin = top->yyin;
	yylineno = top->yylineno;
	yy_switch_to_buffer(top->yybuffer);

	xfree(curfile);
	curfile = top->curfile;

	xfree(top);
	ARRAY_TRUNC(&filestack, 1, struct fileent *);

	return (0);
}
