/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "lib/inspector.h"

#include <assert.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>

static const int MAX_TESTS = 100;

#define faum_log faum_log_
static void faum_log_(
        enum faum_log_level level,
        const char *type,
        const char *name,
        const char *fmt,
        ...
)
{
	va_list args;
	char buffer[1024];
	char *buf = buffer;

	switch (level) {
	case FAUM_LOG_FATAL: strcpy(buf, "FATAL:"); break;
	case FAUM_LOG_CRITICAL: strcpy(buf, "CRITICAL:"); break;
	case FAUM_LOG_ERROR: strcpy(buf, "ERROR:"); break;
	case FAUM_LOG_WARNING: strcpy(buf, "WARNING:"); break;
	case FAUM_LOG_INFO: strcpy(buf, "INFO:"); break;
	case FAUM_LOG_DEBUG: strcpy(buf, "DEBUG:"); break;
	default: assert(0);
	}
	strcat(buf, " ");
	strcat(buf, type);
	if (name[0] != '\0') {
		strcat(buf, " ");
		strcat(buf, name);
	}
	strcat(buf, ": ");

	va_start(args, fmt);
	vsprintf(buf + strlen(buf), fmt, args);
	va_end(args);

	fputs(buffer, stderr);
}

#include "lib/inspector.c"

static void insert_text(void *ctx, const char *text, int len)
{
	if (len == -1) {
		len = strlen(text);
	}
	printf("%.*s", len, text);
	fflush(stdout);
}

typedef struct {
	const char *name;
	uint32_t reg0;
	uint16_t reg1;
	uint8_t reg2;
	int32_t reg3;
	int16_t reg4;
	int8_t reg5;
	int reg6;
} comp_type_0;

static size_t print_status(system_node *node, char *buf, size_t len)
{
       comp_type_0 *comp = (comp_type_0 *)node->opaque;
       return snprintf(buf, len,
                       "comp_type_0 status: name=%s",
                       comp->name);
}

typedef struct {
	const char *str;
	int len;
	insp_result expect;
	insp_result result;
	breakpoint_data data;
} bp_testcase;

static comp_type_0 comp0;
static comp_type_0 comp1;
static system_node* comp0_node;
static system_node* comp1_node;

/**
 * @brief fill_bp_testcases
 * @param tests
 * @param n
 * @return number of valid tests
 */
static int fill_bp_testcases(bp_testcase *tests, int n) {
	//static const int n_tests = 5;
	bp_testcase *test;
	unsigned i = 0;

	test = &tests[i++];
	test->str = "3 < 5";
	test->len = -1;
	test->expect = LIT_COMPARE_LIT;

	test = &tests[i++];
	test->str = "3 <";
	test->len = -1;
	test->expect = SYNTAX_ERROR;

	test = &tests[i++];
	test->str = "3";
	test->len = -1;
	test->expect = SYNTAX_ERROR;

	test = &tests[i++];
	test->str = "foobar";
	test->len = -1;
	test->expect = SYNTAX_ERROR;

	test = &tests[i++];
	test->str = "reg0 = 3";
	test->len = -1;
	test->expect = ILLEGAL_COMPARATOR;

	test = &tests[i++];
	test->str = "foobar == 3";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "foobar == 3 & a != b";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "reg0 == 3 & a != b";
	test->len = -1;
	test->expect = ILLEGAL_CONJUNCTION;

	test = &tests[i++];
	test->str = "reg0 == 0";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_EQ);
	test->data.used_nodes[0] = comp0_node;
	test->data.used_fields[0] = &comp0_node->fields[0];

	test = &tests[i++];
	test->str = "reg0 != 0";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_NEQ);
	test->data.used_nodes[0] = comp0_node;
	test->data.used_fields[0] = &comp0_node->fields[0];

	test = &tests[i++];
	test->str = "reg0 < 0";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_LT);
	test->data.used_nodes[0] = comp0_node;
	test->data.used_fields[0] = &comp0_node->fields[0];

	test = &tests[i++];
	test->str = "reg0 > 0";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_GT);
	test->data.used_nodes[0] = comp0_node;
	test->data.used_fields[0] = &comp0_node->fields[0];

	test = &tests[i++];
	test->str = "reg0 <= 0";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_LE);
	test->data.used_nodes[0] = comp0_node;
	test->data.used_fields[0] = &comp0_node->fields[0];

	test = &tests[i++];
	test->str = "reg0 >= 0";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_GE);
	test->data.used_nodes[0] = comp0_node;
	test->data.used_fields[0] = &comp0_node->fields[0];

	test = &tests[i++];
	test->str = "/pc1/comp1/reg1 == 0";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT16_T, DATA_TYPE_UINT16_T, COMPARATOR_EQ);
	test->data.used_nodes[0] = comp1_node;
	test->data.used_fields[0] = &comp1_node->fields[1];

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 == 3 && a != b";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 == 3 && 4 <= reg2";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 2;
	test->data.n_used_nodes = 2;
	test->data.n_used_fields = 2;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_EQ);
	test->data.test_cbs[1] = get_compare_func(DATA_TYPE_UINT8_T, DATA_TYPE_UINT8_T, COMPARATOR_LE);
	test->data.used_nodes[0] = comp1_node;
	test->data.used_nodes[1] = comp0_node;
	test->data.used_fields[0] = &comp1_node->fields[0];
	test->data.used_fields[1] = &comp0_node->fields[2];
	*((uint32_t*)((uint8_t*)test->data.literal_data+0)) = (uint32_t)3;
	*((uint8_t*)((uint8_t*)test->data.literal_data+4)) = (uint8_t)4;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 <> 3";
	test->len = -1;
	test->expect = ILLEGAL_COMPARATOR;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg1 != /pc0/motherboard/comp0/reg1";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 2;
	test->data.n_used_fields = 2;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT16_T, DATA_TYPE_UINT16_T, COMPARATOR_NEQ);
	test->data.used_nodes[0] = comp1_node;
	test->data.used_nodes[1] = comp0_node;
	test->data.used_fields[0] = &comp1_node->fields[1];
	test->data.used_fields[1] = &comp0_node->fields[1];

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0/ == -3";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "/pc1/comp1 == -3";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "/pc1/comp1/ == -3";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "/pc1/comp_/reg0 == -3";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0/foobar == -3";
	test->len = -1;
	test->expect = NO_SUCH_FIELD;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 == -3";
	test->len = -1;
	test->expect = SIGN_COMPARE_UNSIGN;

	test = &tests[i++];
	test->str = "3 == 0x3";
	test->len = -1;
	test->expect = LIT_COMPARE_LIT;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 == 0xff";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_UINT32_T, DATA_TYPE_UINT32_T, COMPARATOR_EQ);
	test->data.used_nodes[0] = comp1_node;
	test->data.used_fields[0] = &comp1_node->fields[0];
	*((uint32_t*)((uint8_t*)test->data.literal_data+0)) = (uint32_t)0xff;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 == 0xfffffffff";
	test->len = -1;
	test->expect = LITAREAL_OVERFLOW;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg5 == 128";
	test->len = -1;
	test->expect = LITAREAL_OVERFLOW;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg5 == -129";
	test->len = -1;
	test->expect = LITAREAL_OVERFLOW;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg5 == 127";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_INT8_T, DATA_TYPE_INT8_T, COMPARATOR_EQ);
	test->data.used_nodes[0] = comp1_node;
	test->data.used_fields[0] = &comp1_node->fields[5];
	*((int8_t*)((uint8_t*)test->data.literal_data+0)) = (int8_t)127;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg5 == -128";
	test->len = -1;
	test->expect = SUCCESS;
	test->data.n_conditions = 1;
	test->data.n_used_nodes = 1;
	test->data.n_used_fields = 1;
	test->data.test_cbs[0] = get_compare_func(DATA_TYPE_INT8_T, DATA_TYPE_INT8_T, COMPARATOR_EQ);
	test->data.used_nodes[0] = comp1_node;
	test->data.used_fields[0] = &comp1_node->fields[5];
	*((int8_t*)((uint8_t*)test->data.literal_data+0)) = (int8_t)-128;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg6 == #0";
	test->len = -1;
	test->expect = SIGN_COMPARE_UNSIGN;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg2 == #0";
	test->len = -1;
	test->expect = NO_SUCH_COUNTER;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg2 == #1";
	test->len = -1;
	test->expect = NO_SUCH_COUNTER;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg2 == #-1";
	test->len = -1;
	test->expect = SYNTAX_ERROR;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 == 0xffffx";
	test->len = -1;
	test->expect = SYNTAX_ERROR;

	test = &tests[i++];
	test->str = "/pc1/comp1/reg0 == 0xxf";
	test->len = -1;
	test->expect = SYNTAX_ERROR;

	assert(i <= MAX_TESTS);

	return i;
}

static void evaluate_bp_tests(inspector_context *ctx, bp_testcase *tests, int n) {
	int n_success = 0;
	int n_fail = 0;
	int i = 0;

	for (i=0; i<n; i++) {
		bp_testcase *test = &tests[i];
		breakpoint_data bp;
		int fail = 0;
		int len = test->len;
		int formatted_len = 0;
		memset(&bp, 0, sizeof(breakpoint_data));

		if (test->len == -1) {
			len = strlen(test->str);
		}

		char cmd_formatted[len+1];
		formatted_len = format_breakpoint_argument(test->str,
		                                           len,
		                                           cmd_formatted);

		test->result = parse_breakpoint3(ctx, cmd_formatted, formatted_len, &bp);

		if (test->expect != test->result) {
			printf("eval bp %d: expect=%s result=%s\n", i, error_msg(test->expect), error_msg(test->result));
			fail = 1;
		} else if (test->result == SUCCESS) {
			if (test->data.n_conditions != bp.n_conditions) {
				fail = 1;
				printf("eval bp %d: n_conditions differ: expect=%d result=%d\n", i, test->data.n_conditions, bp.n_conditions);
			}
			if (test->data.n_used_nodes != bp.n_used_nodes) {
				fail = 1;
				printf("eval bp %d: n_used_nodes differ: expect=%d result=%d\n", i, test->data.n_used_nodes, bp.n_used_nodes);
			}
			if (test->data.n_used_fields != bp.n_used_fields) {
				fail = 1;
				printf("eval bp %d: n_used_fields differ: expect=%d result=%d\n", i, test->data.n_used_fields, bp.n_used_fields);
			}
			if (0 != memcmp(test->data.test_cbs, bp.test_cbs, sizeof(bp.test_cbs))) {
				fail = 1;
				printf("eval bp %d: test_cbs differs\n", i);
			}
			if (0 != memcmp(test->data.literal_data, bp.literal_data, sizeof(bp.literal_data))) {
				fail = 1;
				printf("eval bp %d: literal_data differs\n", i);
			}
			if (0 != memcmp(test->data.used_nodes, bp.used_nodes, sizeof(bp.used_nodes))) {
				fail = 1;
				printf("eval bp %d: used_nodes differs\n", i);
			}
			if (0 != memcmp(test->data.used_fields, bp.used_fields, sizeof(bp.used_fields))) {
				fail = 1;
				printf("eval bp %d: used_fields differs\n", i);
			}
		}
		if (fail != 0) {
			n_fail++;
		} else {
			n_success++;
		}
	}
	printf("eval: success=%d fail=%d\n", n_success, n_fail);
	fflush(stdout);
}

static void test_parse_ul() {
	unsigned long val;
	insp_result ul_result;

	val = 0;
	ul_result = parse_ul("", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SYNTAX_ERROR), error_msg(ul_result));
	val = 0;
	ul_result = parse_ul("0", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SUCCESS), error_msg(ul_result));
	val = 0;
	ul_result = parse_ul("100", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SUCCESS), error_msg(ul_result));
	val = 0;
	ul_result = parse_ul("-100", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SYNTAX_ERROR), error_msg(ul_result));
	val = 0;
	ul_result = parse_ul("0xf", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SUCCESS), error_msg(ul_result));
	val = 0;
	ul_result = parse_ul("0xffffffff", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SUCCESS), error_msg(ul_result));
	val = 0;
	ul_result = parse_ul("foobar", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SYNTAX_ERROR), error_msg(ul_result));
	val = 0;
	ul_result = parse_ul("a1", -1, &val);
	printf("parse_ul: val=%lu\texpect=%s result=%s\n", val, error_msg(SYNTAX_ERROR), error_msg(ul_result));
}

static void print_comp_info(system_node *node) {
	printf("%s: active breakpoints=%d\n", node->name, node->has_active_breakpoints);
	if (node->has_active_breakpoints) {
		int i;
		for (i=0; i<node->num_fields; i++) {
			system_node_field *f = &node->fields[i];
			if (f->is_used_in_breakpoints) {
				printf("field=%s is used\n", f->name);
			}
		}
	}
	fflush(stdout);
}

static void print_bp_info(breakpoint_data *bp) {
	uint16_t i;
	printf("\tn_conditions=%d\n", bp->n_conditions);
	for (i=0; i<bp->n_used_nodes; i++) {
		printf("\tuse node %s\n", bp->used_nodes[i]->name);
	}
	for (i=0; i<bp->n_used_fields; i++) {
		printf("\tuse field %s\n", bp->used_fields[i]->name);
	}
	fflush(stdout);
}

static void print_breakpoints(inspector_context *ctx) {
	unsigned i;

	for (i = 0; i<ctx->n_cbreakpoints; i++) {
		counter_breakpoint *cbp = &ctx->cbreakpoints[i];
		printf("#%d:\n", cbp->id);
		print_bp_info(&cbp->data);
	}

	for (i = 0; i<ctx->n_tbreakpoints; i++) {
		trigger_breakpoint *tbp = &ctx->tbreakpoints[i];
		printf("bp%d:\n", tbp->id);
		print_bp_info(&tbp->data);
	}
}

int main(int argc, char *argv[]) {
	inspector_context *ctx;
	console_interface console;

	comp0.name = ":pc0:motherboard:comp0";
	comp1.name = ":pc1:comp1";

	system_node_tree_init();

	comp0_node = system_comp_register(comp0.name, &print_status, &comp0);
	system_node_register_field(comp0_node, "reg0",
	                           offsetof(comp_type_0, reg0),
	                           DATA_TYPE_UINT32_T);
	system_node_register_field(comp0_node, "reg1",
	                           offsetof(comp_type_0, reg1),
	                           DATA_TYPE_UINT16_T);
	system_node_register_field(comp0_node, "reg2",
	                           offsetof(comp_type_0, reg2),
	                           DATA_TYPE_UINT8_T);
	system_node_register_field(comp0_node, "reg3",
	                           offsetof(comp_type_0, reg3),
	                           DATA_TYPE_INT32_T);
	system_node_register_field(comp0_node, "reg4",
	                           offsetof(comp_type_0, reg4),
	                           DATA_TYPE_INT16_T);
	system_node_register_field(comp0_node, "reg5",
	                           offsetof(comp_type_0, reg5),
	                           DATA_TYPE_INT8_T);
	system_node_register_field(comp0_node, "reg6",
	                           offsetof(comp_type_0, reg6),
	                           DATA_TYPE_INT);

	comp1_node = system_comp_register(comp1.name, &print_status, &comp1);
	system_node_register_field(comp1_node, "reg0",
	                           offsetof(comp_type_0, reg0),
	                           DATA_TYPE_UINT32_T);
	system_node_register_field(comp1_node, "reg1",
	                           offsetof(comp_type_0, reg1),
	                           DATA_TYPE_UINT16_T);
	system_node_register_field(comp1_node, "reg2",
	                           offsetof(comp_type_0, reg2),
	                           DATA_TYPE_UINT8_T);
	system_node_register_field(comp1_node, "reg3",
	                           offsetof(comp_type_0, reg3),
	                           DATA_TYPE_INT32_T);
	system_node_register_field(comp1_node, "reg4",
	                           offsetof(comp_type_0, reg4),
	                           DATA_TYPE_INT16_T);
	system_node_register_field(comp1_node, "reg5",
	                           offsetof(comp_type_0, reg5),
	                           DATA_TYPE_INT8_T);
	system_node_register_field(comp1_node, "reg6",
	                           offsetof(comp_type_0, reg6),
	                           DATA_TYPE_INT);


	ctx = inspector_context_new();

	console.insert_text_cb = &insert_text;
	console.get_prompt_text_cb = &console_get_prompt_text_default;
	console.pop_from_prompt_cb = &console_pop_from_prompt_default;
	console.push_to_prompt_cb = &console_push_to_prompt_default;

	inspector_set_console(ctx, &console);
	inspector_set_global_inspector(ctx);

	inspector_exec_command(ctx, "cd pc0", -1);
	inspector_exec_command(ctx, "cd motherboard", -1);
	inspector_exec_command(ctx, "cd comp0", -1);
	inspector_exec_command(ctx, "state", -1);

	bp_testcase bp_tests[MAX_TESTS];
	memset(&bp_tests[0], 0, MAX_TESTS * sizeof(bp_testcase));
	int n_tests = fill_bp_testcases(bp_tests, MAX_TESTS);

	evaluate_bp_tests(ctx, bp_tests, n_tests);

	test_parse_ul();

	inspector_exec_command(ctx, "b trigger reg0 != 0", -1);

	inspector_exec_command(ctx, "b trigger reg1 != 1", -1);

	inspector_exec_command(ctx, "b trigger reg2 != 2", -1);

	inspector_exec_command(ctx, "b trigger reg3 != 3", -1);

	inspector_exec_command(ctx, "b trigger reg0 != 4", -1);

	inspector_exec_command(ctx, "b count reg0 != 5", -1);

	inspector_exec_command(ctx, "b trigger #0 != 6", -1);

	inspector_exec_command(ctx, "b count reg5 < -1", -1);

	inspector_exec_command(ctx, "b trigger #1 >= 0xffff", -1);

	inspector_exec_command(ctx, "b show", -1);

	inspector_exec_command(ctx, "b del", -1);
	printf("expected invalid argument\n");

	inspector_exec_command(ctx, "b del #200", -1);
	printf("expected No such breakpoint.\n");

	inspector_exec_command(ctx, "b del #-200", -1);
	printf("expected invalid argument\n");

	inspector_exec_command(ctx, "b del bp200", -1);
	printf("expected invalid argument\n");

	inspector_exec_command(ctx, "b del #foo", -1);
	printf("expected invalid argument\n");

	inspector_exec_command(ctx, "b del bpfoo", -1);
	printf("expected invalid argument\n");

	inspector_exec_command(ctx, "b del #0", -1);

	inspector_exec_command(ctx, "b del bp3", -1);

	inspector_exec_command(ctx, "b show", -1);

	inspector_exec_command(ctx, "b del bp5", -1);

	inspector_exec_command(ctx, "b show", -1);

	inspector_exec_command(ctx, "b del bp2", -1);

	inspector_exec_command(ctx, "b show", -1);

	inspector_exec_command(ctx, "b del bp4", -1);

	inspector_exec_command(ctx, "b show", -1);

	inspector_exec_command(ctx, "b del #2", -1);

	inspector_exec_command(ctx, "b del #0", -1);

	inspector_exec_command(ctx, "b show", -1);

	inspector_exec_command(ctx, "b disable bp6", -1);

	inspector_exec_command(ctx, "b show", -1);

	inspector_exec_command(ctx, "b enable bp6", -1);

	inspector_exec_command(ctx, "b show", -1);

	print_comp_info(comp0_node);
	print_comp_info(comp1_node);

	print_breakpoints(ctx);
	inspector_exec_command(ctx, "b trigger 1</pc1/comp1/reg4", -1);
	inspector_exec_command(ctx, "b show", -1);
	print_breakpoints(ctx);

	print_comp_info(comp0_node);
	print_comp_info(comp1_node);

	fflush(stdout);
	return 0;
}
