/*
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "system.h"
#include "glue-main.h"
#include "glue-gui.h"
#include "glue-log.h"
#include "string_tools.h"

#include "system.inc1"
#include "system.inc3"
#include "system.inc5"

static void *cpssp = &cpssp;

static const struct {
    const char *type;
    void *(*create)(char *);
    void (*destroy)(void *);
    void (*suspend)(void *, FILE *);
    void (*resume)(void *, FILE *);
    void *(*merge)(void *, void *);
    void (*split)(void *);
} sig_type[] = {
#include "system.inc2"
#include "system.inc4"
};

static const struct {
    const char *type;
    const char *generic[50];
    const char *port[4000];
    enum system_gender gender[4000];
    void *(*create)(const char *name, ...);
    void (*destroy)(void *);
    void (*suspend)(void *, FILE *);
    void (*resume)(void *, FILE *);
    void *(*gui_create)(unsigned int page, const char *name, ...);
    void (*gui_destroy)(void *);
    void (*gui_suspend)(void *, FILE *);
    void (*gui_resume)(void *, FILE *);
    void *(*aui_create)(const char *name, ...);
    void (*aui_destroy)(void *);
    void (*aui_suspend)(void *, FILE *);
    void (*aui_resume)(void *, FILE *);
} comp_type[] = {
#include "system.inc6"
};

/* Generic GUI pages. */
static struct {
    unsigned int id;
    char *name;
} page[64];
static unsigned int page_count = 0;

/* Generic signals. */
static struct {
    unsigned int id;
    char *type;
    char *name;

    void *ptr;
} sig[100000];

/* Generic components. */
static struct {
    char *type;
    char *name;
    unsigned int node;
    unsigned int page;
    unsigned int id;

    void *ptr;
    void *aui_ptr;
    void *gui_ptr;
} comp[10000];

/* Generic generics. */
static struct {
    unsigned int comp_id;
    char *type;
    char *name;
    char *value;
} generic[10000];
static unsigned int generic_count = 0;

/* Generic connections. */
static struct {
    unsigned int comp_id;
    char *port;
    enum system_gender gender;
    unsigned int sig_id;
    void *sig_ptr;

    void *merge;
    int peer;
} conn[100000];

static const char *system_name[64];
static unsigned int system_level;

static unsigned int
system_id(void)
{
    static unsigned int id = 0;

    return id++;
}

void
system_name_push(const char *name)
{
    assert(name);

    assert(system_level < sizeof(system_name) / sizeof(system_name[0]));
    system_name[system_level++] = name;
}

void
system_name_pop(void)
{
    assert(1 <= system_level);
    system_level--;
}

const char *
system_path(void)
{
    static char path[1024];
    unsigned int level;

    assert(1 <= system_level);

    assert(strlen(system_name[0]) + 1 <= sizeof(path));
    strcpy(path, system_name[0]);
    for (level = 1; level < system_level; level++) {
	assert(strlen(path) + 1 + strlen(system_name[level]) + 1
	       <= sizeof(path));
	strcat(path, ":");
	strcat(path, system_name[level]);
    }

    return path;
}

int
system_page_create(const char *name)
{
    unsigned int page_id;
    unsigned int pi;

    page_id = system_id();

    for (pi = 0; pi < page_count; pi++) {
	if (strcasecmp(page[pi].name, name) == 0) {
	    /* Found */
	    assert(0); /* FIXME */
	    return -1;
	}
    }

    page[page_count].id = page_id;
    page[page_count].name = strdup(name);
    assert(page[page_count].name);

    gui_page_add(page_id, page[page_count].name);

    return page_id;
}

int
system_page_destroy(unsigned int page_id)
{
    /* FIXME */

    return 0;
}

int
system_page_lookup(const char *name)
{
    unsigned int pi;

    for (pi = 0; ; pi++) {
	if (pi == sizeof(page) / sizeof(page[0])) {
	    /* Page not found. */
	    return -1;
	}

	if (strcasecmp(page[pi].name, name) == 0) {
	    break;
	}
    }

    return page[pi].id;
}

int
system_sig_create(const char *type, const char *name)
{
    unsigned int sig_id;
    unsigned int si;

    sig_id = system_id();

    for (si = 0; si < sizeof(sig) / sizeof(sig[0]); si++) {
	if (! sig[si].type
	    || ! sig[si].name) {
	    continue;
	}
	if (strcasecmp(sig[si].name, name) == 0) {
	    /* Found */
	    assert(0);
	    return -1;
	}
    }
    for (si = 0; ; si++) {
	if (si == sizeof(sig) / sizeof(sig[0])) {
	    /* Table Full */
	    assert(0);
	    return -1;
	}
	if (! sig[si].name) {
	    break;
	}
    }

    sig[si].id = sig_id;

    sig[si].type = strdup(type);
    assert(sig[si].type);

    sig[si].name = strdup(name);
    assert(sig[si].name);

    sig[si].ptr = NULL;

    return sig_id;
}

void *
system_sig_get(unsigned int sig_id)
{
    unsigned int si;

    /* Lookup signal. */
    for (si = 0; ; si++) {
	if (si == sizeof(sig) / sizeof(sig[0])) {
	    /* Not found. */
	    /* FIXME */
	    assert(0);
	}
	if (! sig[si].type
	    || ! sig[si].name) {
	    continue;
	}
	if (sig[si].id == sig_id) {
	    /* Found */
	    break;
	}
    }

    assert(sig[si].id == sig_id);
    assert(sig[si].type);
    assert(sig[si].name);

    if (! sig[si].ptr) {
	unsigned int j;

	for (j = 0; ; j++) {
	    if (j == sizeof(sig_type) / sizeof(sig_type[0])) {
		/* FIXME */
		fprintf(stderr, "Type: %s.\n", sig[si].type);
		assert(0);
	    }
	    if (strcasecmp(sig[si].type, sig_type[j].type) == 0) {
		sig[si].ptr = sig_type[j].create(sig[si].name);
		break;
	    }
	}
    }

    return sig[si].ptr;
}

int
system_sig_unget(void *ptr)
{
    unsigned int si;
    unsigned int j;

    for (si = 0; ; si++) {
	if (si == sizeof(sig) / sizeof(sig[0])) {
	    /* Signal not found. */
	    assert(0); /* FIXME */
	}
	if (! sig[si].type
	    || ! sig[si].name) {
	    continue;
	}
	if (sig[si].ptr == ptr) {
	    break;
	}
    }

    for (j = 0; ; j++) {
	if (j == sizeof(sig_type) / sizeof(sig_type[0])) {
	    /* FIXME */
	    assert(0);
	}
	if (strcasecmp(sig[si].type, sig_type[j].type) == 0) {
	    sig_type[j].destroy(sig[si].ptr);
	    break;
	}
    }

    sig[si].ptr = NULL;

    return 0;
}

void
system_sig_destroy(unsigned int sig_id)
{
    unsigned int si;

    /* Lookup signal. */
    for (si = 0; ; si++) {
	if (si == sizeof(sig) / sizeof(sig[0])) {
	    /* Not found. */
	    return; /* FIXME */
	}
	if (! sig[si].type
	    || ! sig[si].name) {
	    continue;
	}
	if (sig[si].id == sig_id) {
	    /* Found. */
	    break;
	}
    }

    if (sig[si].ptr) {
	/* Hack - FIXME */
	system_sig_unget(sig[si].ptr);
    }

    free(sig[si].type);
    sig[si].type = NULL;

    free(sig[si].name);
    sig[si].name = NULL;
}

int
system_sig_info(unsigned int sig_id, char *type, char *name)
{
    unsigned int si;

    /* Lookup signal. */
    for (si = 0; ; si++) {
	if (si == sizeof(sig) / sizeof(sig[0])) {
	    /* Not found. */
	    return -1;
	}
	if (! sig[si].type
	    || ! sig[si].name) {
	    continue;
	}
	if (sig[si].id == sig_id) {
	    /* Found. */
	    break;
	}
    }

    strcpy(type, sig[si].type);
    strcpy(name, sig[si].name);

    return 0;
}

int
system_comp_create(
		   const char *type,
		   const char *name,
		   unsigned int node,
		   unsigned int page
		  )
{
    int comp_id;
    unsigned int ci;

    if (strchr(name, '/')) {
	/* Name mustn't contain '/'. */
	fprintf(stderr, "%s: WARNING: %s: %s.\n", progname,
		name, "Bad name");
	return -1;
    }

    comp_id = system_id();

    for (ci = 0; ci < sizeof(comp) / sizeof(comp[0]); ci++) {
	if (! comp[ci].name) {
	    continue;
	}
	if (strcasecmp(comp[ci].name, name) == 0) {
	    /* Found */
	    fprintf(stderr, "ERROR: %s: exists already.\n", name);
	    assert(0); /* FIXME */
	    return -1;
	}
    }
    for (ci = 0; ; ci++) {
	if (ci == sizeof(comp) / sizeof(comp[0])) {
	    /* Table Full */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (! comp[ci].name) {
	    break;
	}
    }

    comp[ci].id = comp_id;

    comp[ci].type = strdup(type);
    assert(comp[ci].type);

    comp[ci].name = strdup(name);
    assert(comp[ci].name);

    comp[ci].node = node;
    comp[ci].page = page;


    return comp_id;
}

int
system_comp_generic_set(
			unsigned int comp_id,
			const char *t,
			const char *n,
			const char *v
		       )
{
    char *type;
    char *name;
    char *value;
    unsigned int i;

    /* check if generic with that name is already present */
    for (i = 0; i < generic_count; i++) {
	if (generic[i].comp_id != comp_id) {
	    continue;
	}

	if (strcasecmp(generic[i].name, n) == 0) {
	    faum_log(FAUM_LOG_WARNING, "system", __func__,
		     "Generic %s of component %d already set!\n",
		     n, comp_id);
	}
    }

    generic[generic_count].comp_id = comp_id;

    assert(t);
    type = strdup(t);
    assert(type);
    assert(n);
    name = strdup(n);
    assert(name);
    generic[generic_count].type = type;
    generic[generic_count].name = name;

    if (v) {
	value = strdup(v);
	assert(value);
    } else {
	value = NULL;
    }
    generic[generic_count].value = value;

    generic_count++;

    return 0;
}

int
system_comp_port_connect(
			 unsigned int comp_id,
			 const char *port,
			 unsigned int sig_id
			)
{
    unsigned int conn_id;
    unsigned int ci;
    unsigned int si;

    for (ci = 0; ; ci ++) {
	if (ci == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (comp[ci].id == comp_id) {
	    /* Component found. */
	    break;
	}
    }
    for (si = 0; ; si++) {
	if (si == sizeof(sig) / sizeof(sig[0])) {
	    /* Signal not found. */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (sig[si].id == sig_id) {
	    /* Signal found. */
	    break;
	}
    }

    for (conn_id = 0; conn_id < sizeof(conn) / sizeof(conn[0]); conn_id++) {
	if (! conn[conn_id].port) {
	    continue;
	}

	if (conn[conn_id].comp_id == comp_id
	    && strcasecmp(conn[conn_id].port, port) == 0) {
	    /* Port already defined. */
	    assert(0); /* FIXME */
	    return -1;
	}
    }

    /* Lookup free connection entry. */
    for (conn_id = 0; ; conn_id++) {
	if (conn_id == sizeof(conn) / sizeof(conn[0])) {
	    /* Table full. */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (! conn[conn_id].port) {
	    break;
	}
    }

    conn[conn_id].comp_id = comp_id;
    conn[conn_id].port = strdup(port);
    assert(conn[conn_id].port);
    conn[conn_id].sig_id = sig_id;
    conn[conn_id].sig_ptr = system_sig_get(sig_id);

    conn[conn_id].merge = NULL;
    conn[conn_id].peer = -1;

    return 0;
}

static const char *
generic_get(unsigned int comp_id, const char *name)
{
    unsigned int i;

    for (i = 0; ; i++) {
	if (i == generic_count) {
	    fprintf(stderr, "WARNING: Generic %s not set for component!\n",
		    name);
	    return NULL;
	}
	if (generic[i].comp_id == comp_id
	    && strcasecmp(generic[i].name, name) == 0) {
	    return generic[i].value;
	}
    }
}

static void *
sig_lookup(unsigned int comp_id, const char *port, enum system_gender gender)
{
    unsigned int conn_id;

    for (conn_id = 0; ; conn_id++) {
	if (conn_id == sizeof(conn) / sizeof(conn[0])) {
	    assert(0); /* FIXME */
	}
	assert(conn_id < sizeof(conn) / sizeof(conn[0]));

	if (! conn[conn_id].port) {
	    continue;
	}

	if (conn[conn_id].comp_id == comp_id
	    && strcasecmp(conn[conn_id].port, port) == 0) {
	    if (gender != SYSTEM_GENDER_NEUTRAL) {
		assert(conn[conn_id].gender == 0
		       || conn[conn_id].gender == gender);
		conn[conn_id].gender = gender;
	    }
	    return conn[conn_id].sig_ptr;
	}
    }
}

int
system_comp_init(unsigned int comp_id)
{
    unsigned int ci;
    unsigned int i;

    /* Lookup component. */
    for (ci = 0; ; ci++) {
	if (ci == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (comp[ci].id == comp_id
	    && comp[ci].type
	    && comp[ci].name) {
	    /* Component found. */
	    break;
	}
    }

    for (i = 0; ; i++) {
	if (i == sizeof(comp_type) / sizeof(comp_type[0])) {
	    fprintf(stderr, "%s\n", comp[ci].type);
	    assert(0); /* FIXME */
	}
	if (strcasecmp(comp[ci].type, comp_type[i].type) == 0) {
	    const void *ptr[4000];
	    unsigned int nptrs;
	    unsigned int j;

	    nptrs = 0;
	    for (j = 0; comp_type[i].generic[j]; j++) {
		assert(nptrs < sizeof(ptr) / sizeof(ptr[0]));
		ptr[nptrs++] = generic_get(comp_id, comp_type[i].generic[j]);
	    }
	    for (j = 0; comp_type[i].port[j]; j++) {
		assert(nptrs < sizeof(ptr) / sizeof(ptr[0]));
		ptr[nptrs++] = sig_lookup(comp_id, comp_type[i].port[j], comp_type[i].gender[j]);
	    }
	    while (nptrs < sizeof(ptr) / sizeof(ptr[0])) {
		ptr[nptrs++] = NULL;
	    }

	    if (comp_type[i].create) {
		comp[ci].ptr = comp_type[i].create(comp[ci].name,
						   ptr[0], ptr[1], ptr[2], ptr[3], ptr[4],
						   ptr[5], ptr[6], ptr[7], ptr[8], ptr[9],
						   ptr[10], ptr[11], ptr[12], ptr[13], ptr[14],
						   ptr[15], ptr[16], ptr[17], ptr[18], ptr[19],
						   ptr[20], ptr[21], ptr[22], ptr[23], ptr[24],
						   ptr[25], ptr[26], ptr[27], ptr[28], ptr[29],
						   ptr[30], ptr[31], ptr[32], ptr[33], ptr[34],
						   ptr[35], ptr[36], ptr[37], ptr[38], ptr[39],
						   ptr[40], ptr[41], ptr[42], ptr[43], ptr[44],
						   ptr[45], ptr[46], ptr[47], ptr[48], ptr[49],
						   ptr[50], ptr[51], ptr[52], ptr[53], ptr[54],
						   ptr[55], ptr[56], ptr[57], ptr[58], ptr[59],
						   ptr[60], ptr[61], ptr[62], ptr[63], ptr[64],
						   ptr[65], ptr[66], ptr[67], ptr[68], ptr[69],
						   ptr[70], ptr[71], ptr[72], ptr[73], ptr[74],
						   ptr[75], ptr[76], ptr[77], ptr[78], ptr[79],
						   ptr[80], ptr[81], ptr[82], ptr[83], ptr[84],
						   ptr[85], ptr[86], ptr[87], ptr[88], ptr[89],
						   ptr[90], ptr[91], ptr[92], ptr[93], ptr[94],
						   ptr[95], ptr[96], ptr[97], ptr[98], ptr[99],
						   ptr[100], ptr[101], ptr[102], ptr[103], ptr[104],
						   ptr[105], ptr[106], ptr[107], ptr[108], ptr[109],
						   ptr[110], ptr[111], ptr[112], ptr[113], ptr[114],
						   ptr[115], ptr[116], ptr[117], ptr[118], ptr[119],
						   ptr[120], ptr[121], ptr[122], ptr[123], ptr[124],
						   ptr[125], ptr[126], ptr[127], ptr[128], ptr[129],
						   ptr[130], ptr[131], ptr[132], ptr[133], ptr[134],
						   ptr[135], ptr[136], ptr[137], ptr[138], ptr[139],
						   ptr[140], ptr[141], ptr[142], ptr[143], ptr[144],
						   ptr[145], ptr[146], ptr[147], ptr[148], ptr[149],
						   ptr[150], ptr[151], ptr[152], ptr[153], ptr[154],
						   ptr[155], ptr[156], ptr[157], ptr[158], ptr[159],
						   ptr[160], ptr[161], ptr[162], ptr[163], ptr[164],
						   ptr[165], ptr[166], ptr[167], ptr[168], ptr[169],
						   ptr[170], ptr[171], ptr[172], ptr[173], ptr[174],
						   ptr[175], ptr[176], ptr[177], ptr[178], ptr[179],
						   ptr[180], ptr[181], ptr[182], ptr[183], ptr[184],
						   ptr[185], ptr[186], ptr[187], ptr[188], ptr[189],
						   ptr[190], ptr[191], ptr[192], ptr[193], ptr[194],
						   ptr[195], ptr[196], ptr[197], ptr[198], ptr[199],
						   ptr[200], ptr[201], ptr[202], ptr[203], ptr[204],
						   ptr[205], ptr[206], ptr[207], ptr[208], ptr[209],
						   ptr[210], ptr[211], ptr[212], ptr[213], ptr[214],
						   ptr[215], ptr[216], ptr[217], ptr[218], ptr[219],
						   ptr[220], ptr[221], ptr[222], ptr[223], ptr[224],
						   ptr[225], ptr[226], ptr[227], ptr[228], ptr[229],
						   ptr[230], ptr[231], ptr[232], ptr[233], ptr[234],
						   ptr[235], ptr[236], ptr[237], ptr[238], ptr[239],
						   ptr[240], ptr[241], ptr[242], ptr[243], ptr[244],
						   ptr[245], ptr[246], ptr[247], ptr[248], ptr[249],
						   ptr[250], ptr[251], ptr[252], ptr[253], ptr[254],
						   ptr[255], ptr[256], ptr[257], ptr[258], ptr[259],
						   ptr[260], ptr[261], ptr[262], ptr[263], ptr[264],
						   ptr[265], ptr[266], ptr[267], ptr[268], ptr[269],
						   ptr[270], ptr[271], ptr[272], ptr[273], ptr[274],
						   ptr[275], ptr[276], ptr[277], ptr[278], ptr[279],
						   ptr[280], ptr[281], ptr[282], ptr[283], ptr[284],
						   ptr[285], ptr[286], ptr[287], ptr[288], ptr[289],
						   ptr[290], ptr[291], ptr[292], ptr[293], ptr[294],
						   ptr[295], ptr[296], ptr[297], ptr[298], ptr[299],
						   ptr[300], ptr[301], ptr[302], ptr[303], ptr[304],
						   ptr[305], ptr[306], ptr[307], ptr[308], ptr[309],
						   ptr[310], ptr[311], ptr[312], ptr[313], ptr[314],
						   ptr[315], ptr[316], ptr[317], ptr[318], ptr[319],
						   ptr[320], ptr[321], ptr[322], ptr[323], ptr[324],
						   ptr[325], ptr[326], ptr[327], ptr[328], ptr[329],
						   ptr[330], ptr[331], ptr[332], ptr[333], ptr[334],
						   ptr[335], ptr[336], ptr[337], ptr[338], ptr[339],
						   ptr[340], ptr[341], ptr[342], ptr[343], ptr[344],
						   ptr[345], ptr[346], ptr[347], ptr[348], ptr[349],
						   ptr[350], ptr[351], ptr[352], ptr[353], ptr[354],
						   ptr[355], ptr[356], ptr[357], ptr[358], ptr[359],
						   ptr[360], ptr[361], ptr[362], ptr[363], ptr[364],
						   ptr[365], ptr[366], ptr[367], ptr[368], ptr[369],
						   ptr[370], ptr[371], ptr[372], ptr[373], ptr[374],
						   ptr[375], ptr[376], ptr[377], ptr[378], ptr[379],
						   ptr[380], ptr[381], ptr[382], ptr[383], ptr[384],
						   ptr[385], ptr[386], ptr[387], ptr[388], ptr[389],
						   ptr[390], ptr[391], ptr[392], ptr[393], ptr[394],
						   ptr[395], ptr[396], ptr[397], ptr[398], ptr[399],
						   ptr[400], ptr[401], ptr[402], ptr[403], ptr[404],
						   ptr[405], ptr[406], ptr[407], ptr[408], ptr[409],
						   ptr[410], ptr[411], ptr[412], ptr[413], ptr[414],
						   ptr[415], ptr[416], ptr[417], ptr[418], ptr[419],
						   ptr[420], ptr[421], ptr[422], ptr[423], ptr[424],
						   ptr[425], ptr[426], ptr[427], ptr[428], ptr[429],
						   ptr[430], ptr[431], ptr[432], ptr[433], ptr[434],
						   ptr[435], ptr[436], ptr[437], ptr[438], ptr[439],
						   ptr[440], ptr[441], ptr[442], ptr[443], ptr[444],
						   ptr[445], ptr[446], ptr[447], ptr[448], ptr[449],
						   ptr[450], ptr[451], ptr[452], ptr[453], ptr[454],
						   ptr[455], ptr[456], ptr[457], ptr[458], ptr[459],
						   ptr[460], ptr[461], ptr[462], ptr[463], ptr[464],
						   ptr[465], ptr[466], ptr[467], ptr[468], ptr[469],
						   ptr[470], ptr[471], ptr[472], ptr[473], ptr[474],
						   ptr[475], ptr[476], ptr[477], ptr[478], ptr[479],
						   ptr[480], ptr[481], ptr[482], ptr[483], ptr[484],
						   ptr[485], ptr[486], ptr[487], ptr[488], ptr[489],
						   ptr[490], ptr[491], ptr[492], ptr[493], ptr[494],
						   ptr[495], ptr[496], ptr[497], ptr[498], ptr[499],
						   ptr[500], ptr[501], ptr[502], ptr[503], ptr[504],
						   ptr[505], ptr[506], ptr[507], ptr[508], ptr[509],
						   ptr[510], ptr[511], ptr[512], ptr[513], ptr[514],
						   ptr[515], ptr[516], ptr[517], ptr[518], ptr[519],
						   ptr[520], ptr[521], ptr[522], ptr[523], ptr[524],
						   ptr[525], ptr[526], ptr[527], ptr[528], ptr[529],
						   ptr[530], ptr[531], ptr[532], ptr[533], ptr[534],
						   ptr[535], ptr[536], ptr[537], ptr[538], ptr[539],
						   ptr[540], ptr[541], ptr[542], ptr[543], ptr[544],
						   ptr[545], ptr[546], ptr[547], ptr[548], ptr[549],
						   ptr[550], ptr[551], ptr[552], ptr[553], ptr[554],
						   ptr[555], ptr[556], ptr[557], ptr[558], ptr[559],
						   ptr[560], ptr[561], ptr[562], ptr[563], ptr[564],
						   ptr[565], ptr[566], ptr[567], ptr[568], ptr[569],
						   ptr[570], ptr[571], ptr[572], ptr[573], ptr[574],
						   ptr[575], ptr[576], ptr[577], ptr[578], ptr[579],
						   ptr[580], ptr[581], ptr[582], ptr[583], ptr[584],
						   ptr[585], ptr[586], ptr[587], ptr[588], ptr[589],
						   ptr[590], ptr[591], ptr[592], ptr[593], ptr[594],
						   ptr[595], ptr[596], ptr[597], ptr[598], ptr[599],
						   ptr[600], ptr[601], ptr[602], ptr[603], ptr[604],
						   ptr[605], ptr[606], ptr[607], ptr[608], ptr[609],
						   ptr[610], ptr[611], ptr[612], ptr[613], ptr[614],
						   ptr[615], ptr[616], ptr[617], ptr[618], ptr[619],
						   ptr[620], ptr[621], ptr[622], ptr[623], ptr[624],
						   ptr[625], ptr[626], ptr[627], ptr[628], ptr[629],
						   ptr[630], ptr[631], ptr[632], ptr[633], ptr[634],
						   ptr[635], ptr[636], ptr[637], ptr[638], ptr[639],
						   ptr[640], ptr[641], ptr[642], ptr[643], ptr[644],
						   ptr[645], ptr[646], ptr[647], ptr[648], ptr[649],
						   ptr[650], ptr[651], ptr[652], ptr[653], ptr[654],
						   ptr[655], ptr[656], ptr[657], ptr[658], ptr[659],
						   ptr[660], ptr[661], ptr[662], ptr[663], ptr[664],
						   ptr[665], ptr[666], ptr[667], ptr[668], ptr[669],
						   ptr[670], ptr[671], ptr[672], ptr[673], ptr[674],
						   ptr[675], ptr[676], ptr[677], ptr[678], ptr[679],
						   ptr[680], ptr[681], ptr[682], ptr[683], ptr[684],
						   ptr[685], ptr[686], ptr[687], ptr[688], ptr[689],
						   ptr[690], ptr[691], ptr[692], ptr[693], ptr[694],
						   ptr[695], ptr[696], ptr[697], ptr[698], ptr[699],
						   ptr[700], ptr[701], ptr[702], ptr[703], ptr[704],
						   ptr[705], ptr[706], ptr[707], ptr[708], ptr[709],
						   ptr[710], ptr[711], ptr[712], ptr[713], ptr[714],
						   ptr[715], ptr[716], ptr[717], ptr[718], ptr[719],
						   ptr[720], ptr[721], ptr[722], ptr[723], ptr[724],
						   ptr[725], ptr[726], ptr[727], ptr[728], ptr[729],
						   ptr[730], ptr[731], ptr[732], ptr[733], ptr[734],
						   ptr[735], ptr[736], ptr[737], ptr[738], ptr[739],
						   ptr[740], ptr[741], ptr[742], ptr[743], ptr[744],
						   ptr[745], ptr[746], ptr[747], ptr[748], ptr[749],
						   ptr[750], ptr[751], ptr[752], ptr[753], ptr[754],
						   ptr[755], ptr[756], ptr[757], ptr[758], ptr[759],
						   ptr[760], ptr[761], ptr[762], ptr[763], ptr[764],
						   ptr[765], ptr[766], ptr[767], ptr[768], ptr[769],
						   ptr[770], ptr[771], ptr[772], ptr[773], ptr[774],
						   ptr[775], ptr[776], ptr[777], ptr[778], ptr[779],
						   ptr[780], ptr[781], ptr[782], ptr[783], ptr[784],
						   ptr[785], ptr[786], ptr[787], ptr[788], ptr[789],
						   ptr[790], ptr[791], ptr[792], ptr[793], ptr[794],
						   ptr[795], ptr[796], ptr[797], ptr[798], ptr[799],
						   ptr[800], ptr[801], ptr[802], ptr[803], ptr[804],
						   ptr[805], ptr[806], ptr[807], ptr[808], ptr[809],
						   ptr[810], ptr[811], ptr[812], ptr[813], ptr[814],
						   ptr[815], ptr[816], ptr[817], ptr[818], ptr[819],
						   ptr[820], ptr[821], ptr[822], ptr[823], ptr[824],
						   ptr[825], ptr[826], ptr[827], ptr[828], ptr[829],
						   ptr[830], ptr[831], ptr[832], ptr[833], ptr[834],
						   ptr[835], ptr[836], ptr[837], ptr[838], ptr[839],
						   ptr[840], ptr[841], ptr[842], ptr[843], ptr[844],
						   ptr[845], ptr[846], ptr[847], ptr[848], ptr[849],
						   ptr[850], ptr[851], ptr[852], ptr[853], ptr[854],
						   ptr[855], ptr[856], ptr[857], ptr[858], ptr[859],
						   ptr[860], ptr[861], ptr[862], ptr[863], ptr[864],
						   ptr[865], ptr[866], ptr[867], ptr[868], ptr[869],
						   ptr[870], ptr[871], ptr[872], ptr[873], ptr[874],
						   ptr[875], ptr[876], ptr[877], ptr[878], ptr[879],
						   ptr[880], ptr[881], ptr[882], ptr[883], ptr[884],
						   ptr[885], ptr[886], ptr[887], ptr[888], ptr[889],
						   ptr[890], ptr[891], ptr[892], ptr[893], ptr[894],
						   ptr[895], ptr[896], ptr[897], ptr[898], ptr[899],
						   ptr[900], ptr[901], ptr[902], ptr[903], ptr[904],
						   ptr[905], ptr[906], ptr[907], ptr[908], ptr[909],
						   ptr[910], ptr[911], ptr[912], ptr[913], ptr[914],
						   ptr[915], ptr[916], ptr[917], ptr[918], ptr[919],
						   ptr[920], ptr[921], ptr[922], ptr[923], ptr[924],
						   ptr[925], ptr[926], ptr[927], ptr[928], ptr[929],
						   ptr[930], ptr[931], ptr[932], ptr[933], ptr[934],
						   ptr[935], ptr[936], ptr[937], ptr[938], ptr[939],
						   ptr[940], ptr[941], ptr[942], ptr[943], ptr[944],
						   ptr[945], ptr[946], ptr[947], ptr[948], ptr[949],
						   ptr[950], ptr[951], ptr[952], ptr[953], ptr[954],
						   ptr[955], ptr[956], ptr[957], ptr[958], ptr[959],
						   ptr[960], ptr[961], ptr[962], ptr[963], ptr[964],
						   ptr[965], ptr[966], ptr[967], ptr[968], ptr[969],
						   ptr[970], ptr[971], ptr[972], ptr[973], ptr[974],
						   ptr[975], ptr[976], ptr[977], ptr[978], ptr[979],
						   ptr[980], ptr[981], ptr[982], ptr[983], ptr[984],
						   ptr[985], ptr[986], ptr[987], ptr[988], ptr[989],
						   ptr[990], ptr[991], ptr[992], ptr[993], ptr[994],
						   ptr[995], ptr[996], ptr[997], ptr[998], ptr[999],
						   ptr[1000], ptr[1001], ptr[1002], ptr[1003], ptr[1004],
						   ptr[1005], ptr[1006], ptr[1007], ptr[1008], ptr[1009],
						   ptr[1010], ptr[1011], ptr[1012], ptr[1013], ptr[1014],
						   ptr[1015], ptr[1016], ptr[1017], ptr[1018], ptr[1019],
						   ptr[1020], ptr[1021], ptr[1022], ptr[1023], ptr[1024],
						   ptr[1025], ptr[1026], ptr[1027], ptr[1028], ptr[1029],
						   ptr[1030], ptr[1031], ptr[1032], ptr[1033], ptr[1034],
						   ptr[1035], ptr[1036], ptr[1037], ptr[1038], ptr[1039],
						   ptr[1040], ptr[1041], ptr[1042], ptr[1043], ptr[1044],
						   ptr[1045], ptr[1046], ptr[1047], ptr[1048], ptr[1049],
						   ptr[1050], ptr[1051], ptr[1052], ptr[1053], ptr[1054],
						   ptr[1055], ptr[1056], ptr[1057], ptr[1058], ptr[1059],
						   ptr[1060], ptr[1061], ptr[1062], ptr[1063], ptr[1064],
						   ptr[1065], ptr[1066], ptr[1067], ptr[1068], ptr[1069],
						   ptr[1070], ptr[1071], ptr[1072], ptr[1073], ptr[1074],
						   ptr[1075], ptr[1076], ptr[1077], ptr[1078], ptr[1079],
						   ptr[1080], ptr[1081], ptr[1082], ptr[1083], ptr[1084],
						   ptr[1085], ptr[1086], ptr[1087], ptr[1088], ptr[1089],
						   ptr[1090], ptr[1091], ptr[1092], ptr[1093], ptr[1094],
						   ptr[1095], ptr[1096], ptr[1097], ptr[1098], ptr[1099],
						   ptr[1100], ptr[1101], ptr[1102], ptr[1103], ptr[1104],
						   ptr[1105], ptr[1106], ptr[1107], ptr[1108], ptr[1109],
						   ptr[1110], ptr[1111], ptr[1112], ptr[1113], ptr[1114],
						   ptr[1115], ptr[1116], ptr[1117], ptr[1118], ptr[1119],
						   ptr[1120], ptr[1121], ptr[1122], ptr[1123], ptr[1124],
						   ptr[1125], ptr[1126], ptr[1127], ptr[1128], ptr[1129],
						   ptr[1130], ptr[1131], ptr[1132], ptr[1133], ptr[1134],
						   ptr[1135], ptr[1136], ptr[1137], ptr[1138], ptr[1139],
						   ptr[1140], ptr[1141], ptr[1142], ptr[1143], ptr[1144],
						   ptr[1145], ptr[1146], ptr[1147], ptr[1148], ptr[1149],
						   ptr[1150], ptr[1151], ptr[1152], ptr[1153], ptr[1154],
						   ptr[1155], ptr[1156], ptr[1157], ptr[1158], ptr[1159],
						   ptr[1160], ptr[1161], ptr[1162], ptr[1163], ptr[1164],
						   ptr[1165], ptr[1166], ptr[1167], ptr[1168], ptr[1169],
						   ptr[1170], ptr[1171], ptr[1172], ptr[1173], ptr[1174],
						   ptr[1175], ptr[1176], ptr[1177], ptr[1178], ptr[1179],
						   ptr[1180], ptr[1181], ptr[1182], ptr[1183], ptr[1184],
						   ptr[1185], ptr[1186], ptr[1187], ptr[1188], ptr[1189],
						   ptr[1190], ptr[1191], ptr[1192], ptr[1193], ptr[1194],
						   ptr[1195], ptr[1196], ptr[1197], ptr[1198], ptr[1199],
						   ptr[1200], ptr[1201], ptr[1202], ptr[1203], ptr[1204],
						   ptr[1205], ptr[1206], ptr[1207], ptr[1208], ptr[1209],
						   ptr[1210], ptr[1211], ptr[1212], ptr[1213], ptr[1214],
						   ptr[1215], ptr[1216], ptr[1217], ptr[1218], ptr[1219],
						   ptr[1220], ptr[1221], ptr[1222], ptr[1223], ptr[1224],
						   ptr[1225], ptr[1226], ptr[1227], ptr[1228], ptr[1229],
						   ptr[1230], ptr[1231], ptr[1232], ptr[1233], ptr[1234],
						   ptr[1235], ptr[1236], ptr[1237], ptr[1238], ptr[1239],
						   ptr[1240], ptr[1241], ptr[1242], ptr[1243], ptr[1244],
						   ptr[1245], ptr[1246], ptr[1247], ptr[1248], ptr[1249],
						   ptr[1250], ptr[1251], ptr[1252], ptr[1253], ptr[1254],
						   ptr[1255], ptr[1256], ptr[1257], ptr[1258], ptr[1259],
						   ptr[1260], ptr[1261], ptr[1262], ptr[1263], ptr[1264],
						   ptr[1265], ptr[1266], ptr[1267], ptr[1268], ptr[1269],
						   ptr[1270], ptr[1271], ptr[1272], ptr[1273], ptr[1274],
						   ptr[1275], ptr[1276], ptr[1277], ptr[1278], ptr[1279],
						   ptr[1280], ptr[1281], ptr[1282], ptr[1283], ptr[1284],
						   ptr[1285], ptr[1286], ptr[1287], ptr[1288], ptr[1289],
						   ptr[1290], ptr[1291], ptr[1292], ptr[1293], ptr[1294],
						   ptr[1295], ptr[1296], ptr[1297], ptr[1298], ptr[1299],
						   ptr[1300], ptr[1301], ptr[1302], ptr[1303], ptr[1304],
						   ptr[1305], ptr[1306], ptr[1307], ptr[1308], ptr[1309],
						   ptr[1310], ptr[1311], ptr[1312], ptr[1313], ptr[1314],
						   ptr[1315], ptr[1316], ptr[1317], ptr[1318], ptr[1319],
						   ptr[1320], ptr[1321], ptr[1322], ptr[1323], ptr[1324],
						   ptr[1325], ptr[1326], ptr[1327], ptr[1328], ptr[1329],
						   ptr[1330], ptr[1331], ptr[1332], ptr[1333], ptr[1334],
						   ptr[1335], ptr[1336], ptr[1337], ptr[1338], ptr[1339],
						   ptr[1340], ptr[1341], ptr[1342], ptr[1343], ptr[1344],
						   ptr[1345], ptr[1346], ptr[1347], ptr[1348], ptr[1349],
						   ptr[1350], ptr[1351], ptr[1352], ptr[1353], ptr[1354],
						   ptr[1355], ptr[1356], ptr[1357], ptr[1358], ptr[1359],
						   ptr[1360], ptr[1361], ptr[1362], ptr[1363], ptr[1364],
						   ptr[1365], ptr[1366], ptr[1367], ptr[1368], ptr[1369],
						   ptr[1370], ptr[1371], ptr[1372], ptr[1373], ptr[1374],
						   ptr[1375], ptr[1376], ptr[1377], ptr[1378], ptr[1379],
						   ptr[1380], ptr[1381], ptr[1382], ptr[1383], ptr[1384],
						   ptr[1385], ptr[1386], ptr[1387], ptr[1388], ptr[1389],
						   ptr[1390], ptr[1391], ptr[1392], ptr[1393], ptr[1394],
						   ptr[1395], ptr[1396], ptr[1397], ptr[1398], ptr[1399],
						   ptr[1400], ptr[1401], ptr[1402], ptr[1403], ptr[1404],
						   ptr[1405], ptr[1406], ptr[1407], ptr[1408], ptr[1409],
						   ptr[1410], ptr[1411], ptr[1412], ptr[1413], ptr[1414],
						   ptr[1415], ptr[1416], ptr[1417], ptr[1418], ptr[1419],
						   ptr[1420], ptr[1421], ptr[1422], ptr[1423], ptr[1424],
						   ptr[1425], ptr[1426], ptr[1427], ptr[1428], ptr[1429],
						   ptr[1430], ptr[1431], ptr[1432], ptr[1433], ptr[1434],
						   ptr[1435], ptr[1436], ptr[1437], ptr[1438], ptr[1439],
						   ptr[1440], ptr[1441], ptr[1442], ptr[1443], ptr[1444],
						   ptr[1445], ptr[1446], ptr[1447], ptr[1448], ptr[1449],
						   ptr[1450], ptr[1451], ptr[1452], ptr[1453], ptr[1454],
						   ptr[1455], ptr[1456], ptr[1457], ptr[1458], ptr[1459],
						   ptr[1460], ptr[1461], ptr[1462], ptr[1463], ptr[1464],
						   ptr[1465], ptr[1466], ptr[1467], ptr[1468], ptr[1469],
						   ptr[1470], ptr[1471], ptr[1472], ptr[1473], ptr[1474],
						   ptr[1475], ptr[1476], ptr[1477], ptr[1478], ptr[1479],
						   ptr[1480], ptr[1481], ptr[1482], ptr[1483], ptr[1484],
						   ptr[1485], ptr[1486], ptr[1487], ptr[1488], ptr[1489],
						   ptr[1490], ptr[1491], ptr[1492], ptr[1493], ptr[1494],
						   ptr[1495], ptr[1496], ptr[1497], ptr[1498], ptr[1499],
						   ptr[1500], ptr[1501], ptr[1502], ptr[1503], ptr[1504],
						   ptr[1505], ptr[1506], ptr[1507], ptr[1508], ptr[1509],
						   ptr[1510], ptr[1511], ptr[1512], ptr[1513], ptr[1514],
						   ptr[1515], ptr[1516], ptr[1517], ptr[1518], ptr[1519],
						   ptr[1520], ptr[1521], ptr[1522], ptr[1523], ptr[1524],
						   ptr[1525], ptr[1526], ptr[1527], ptr[1528], ptr[1529],
						   ptr[1530], ptr[1531], ptr[1532], ptr[1533], ptr[1534],
						   ptr[1535], ptr[1536], ptr[1537], ptr[1538], ptr[1539],
						   ptr[1540], ptr[1541], ptr[1542], ptr[1543], ptr[1544],
						   ptr[1545], ptr[1546], ptr[1547], ptr[1548], ptr[1549],
						   ptr[1550], ptr[1551], ptr[1552], ptr[1553], ptr[1554],
						   ptr[1555], ptr[1556], ptr[1557], ptr[1558], ptr[1559],
						   ptr[1560], ptr[1561], ptr[1562], ptr[1563], ptr[1564],
						   ptr[1565], ptr[1566], ptr[1567], ptr[1568], ptr[1569],
						   ptr[1570], ptr[1571], ptr[1572], ptr[1573], ptr[1574],
						   ptr[1575], ptr[1576], ptr[1577], ptr[1578], ptr[1579],
						   ptr[1580], ptr[1581], ptr[1582], ptr[1583], ptr[1584],
						   ptr[1585], ptr[1586], ptr[1587], ptr[1588], ptr[1589],
						   ptr[1590], ptr[1591], ptr[1592], ptr[1593], ptr[1594],
						   ptr[1595], ptr[1596], ptr[1597], ptr[1598], ptr[1599],
						   ptr[1600], ptr[1601], ptr[1602], ptr[1603], ptr[1604],
						   ptr[1605], ptr[1606], ptr[1607], ptr[1608], ptr[1609],
						   ptr[1610], ptr[1611], ptr[1612], ptr[1613], ptr[1614],
						   ptr[1615], ptr[1616], ptr[1617], ptr[1618], ptr[1619],
						   ptr[1620], ptr[1621], ptr[1622], ptr[1623], ptr[1624],
						   ptr[1625], ptr[1626], ptr[1627], ptr[1628], ptr[1629],
						   ptr[1630], ptr[1631], ptr[1632], ptr[1633], ptr[1634],
						   ptr[1635], ptr[1636], ptr[1637], ptr[1638], ptr[1639],
						   ptr[1640], ptr[1641], ptr[1642], ptr[1643], ptr[1644],
						   ptr[1645], ptr[1646], ptr[1647], ptr[1648], ptr[1649],
						   ptr[1650], ptr[1651], ptr[1652], ptr[1653], ptr[1654],
						   ptr[1655], ptr[1656], ptr[1657], ptr[1658], ptr[1659],
						   ptr[1660], ptr[1661], ptr[1662], ptr[1663], ptr[1664],
						   ptr[1665], ptr[1666], ptr[1667], ptr[1668], ptr[1669],
						   ptr[1670], ptr[1671], ptr[1672], ptr[1673], ptr[1674],
						   ptr[1675], ptr[1676], ptr[1677], ptr[1678], ptr[1679],
						   ptr[1680], ptr[1681], ptr[1682], ptr[1683], ptr[1684],
						   ptr[1685], ptr[1686], ptr[1687], ptr[1688], ptr[1689],
						   ptr[1690], ptr[1691], ptr[1692], ptr[1693], ptr[1694],
						   ptr[1695], ptr[1696], ptr[1697], ptr[1698], ptr[1699],
						   ptr[1700], ptr[1701], ptr[1702], ptr[1703], ptr[1704],
						   ptr[1705], ptr[1706], ptr[1707], ptr[1708], ptr[1709],
						   ptr[1710], ptr[1711], ptr[1712], ptr[1713], ptr[1714],
						   ptr[1715], ptr[1716], ptr[1717], ptr[1718], ptr[1719],
						   ptr[1720], ptr[1721], ptr[1722], ptr[1723], ptr[1724],
						   ptr[1725], ptr[1726], ptr[1727], ptr[1728], ptr[1729],
						   ptr[1730], ptr[1731], ptr[1732], ptr[1733], ptr[1734],
						   ptr[1735], ptr[1736], ptr[1737], ptr[1738], ptr[1739],
						   ptr[1740], ptr[1741], ptr[1742], ptr[1743], ptr[1744],
						   ptr[1745], ptr[1746], ptr[1747], ptr[1748], ptr[1749],
						   ptr[1750], ptr[1751], ptr[1752], ptr[1753], ptr[1754],
						   ptr[1755], ptr[1756], ptr[1757], ptr[1758], ptr[1759],
						   ptr[1760], ptr[1761], ptr[1762], ptr[1763], ptr[1764],
						   ptr[1765], ptr[1766], ptr[1767], ptr[1768], ptr[1769],
						   ptr[1770], ptr[1771], ptr[1772], ptr[1773], ptr[1774],
						   ptr[1775], ptr[1776], ptr[1777], ptr[1778], ptr[1779],
						   ptr[1780], ptr[1781], ptr[1782], ptr[1783], ptr[1784],
						   ptr[1785], ptr[1786], ptr[1787], ptr[1788], ptr[1789],
						   ptr[1790], ptr[1791], ptr[1792], ptr[1793], ptr[1794],
						   ptr[1795], ptr[1796], ptr[1797], ptr[1798], ptr[1799],
						   ptr[1800], ptr[1801], ptr[1802], ptr[1803], ptr[1804],
						   ptr[1805], ptr[1806], ptr[1807], ptr[1808], ptr[1809],
						   ptr[1810], ptr[1811], ptr[1812], ptr[1813], ptr[1814],
						   ptr[1815], ptr[1816], ptr[1817], ptr[1818], ptr[1819],
						   ptr[1820], ptr[1821], ptr[1822], ptr[1823], ptr[1824],
						   ptr[1825], ptr[1826], ptr[1827], ptr[1828], ptr[1829],
						   ptr[1830], ptr[1831], ptr[1832], ptr[1833], ptr[1834],
						   ptr[1835], ptr[1836], ptr[1837], ptr[1838], ptr[1839],
						   ptr[1840], ptr[1841], ptr[1842], ptr[1843], ptr[1844],
						   ptr[1845], ptr[1846], ptr[1847], ptr[1848], ptr[1849],
						   ptr[1850], ptr[1851], ptr[1852], ptr[1853], ptr[1854],
						   ptr[1855], ptr[1856], ptr[1857], ptr[1858], ptr[1859],
						   ptr[1860], ptr[1861], ptr[1862], ptr[1863], ptr[1864],
						   ptr[1865], ptr[1866], ptr[1867], ptr[1868], ptr[1869],
						   ptr[1870], ptr[1871], ptr[1872], ptr[1873], ptr[1874],
						   ptr[1875], ptr[1876], ptr[1877], ptr[1878], ptr[1879],
						   ptr[1880], ptr[1881], ptr[1882], ptr[1883], ptr[1884],
						   ptr[1885], ptr[1886], ptr[1887], ptr[1888], ptr[1889],
						   ptr[1890], ptr[1891], ptr[1892], ptr[1893], ptr[1894],
						   ptr[1895], ptr[1896], ptr[1897], ptr[1898], ptr[1899],
						   ptr[1900], ptr[1901], ptr[1902], ptr[1903], ptr[1904],
						   ptr[1905], ptr[1906], ptr[1907], ptr[1908], ptr[1909],
						   ptr[1910], ptr[1911], ptr[1912], ptr[1913], ptr[1914],
						   ptr[1915], ptr[1916], ptr[1917], ptr[1918], ptr[1919],
						   ptr[1920], ptr[1921], ptr[1922], ptr[1923], ptr[1924],
						   ptr[1925], ptr[1926], ptr[1927], ptr[1928], ptr[1929],
						   ptr[1930], ptr[1931], ptr[1932], ptr[1933], ptr[1934],
						   ptr[1935], ptr[1936], ptr[1937], ptr[1938], ptr[1939],
						   ptr[1940], ptr[1941], ptr[1942], ptr[1943], ptr[1944],
						   ptr[1945], ptr[1946], ptr[1947], ptr[1948], ptr[1949],
						   ptr[1950], ptr[1951], ptr[1952], ptr[1953], ptr[1954],
						   ptr[1955], ptr[1956], ptr[1957], ptr[1958], ptr[1959],
						   ptr[1960], ptr[1961], ptr[1962], ptr[1963], ptr[1964],
						   ptr[1965], ptr[1966], ptr[1967], ptr[1968], ptr[1969],
						   ptr[1970], ptr[1971], ptr[1972], ptr[1973], ptr[1974],
						   ptr[1975], ptr[1976], ptr[1977], ptr[1978], ptr[1979],
						   ptr[1980], ptr[1981], ptr[1982], ptr[1983], ptr[1984],
						   ptr[1985], ptr[1986], ptr[1987], ptr[1988], ptr[1989],
						   ptr[1990], ptr[1991], ptr[1992], ptr[1993], ptr[1994],
						   ptr[1995], ptr[1996], ptr[1997], ptr[1998], ptr[1999],
						   ptr[2000], ptr[2001], ptr[2002], ptr[2003], ptr[2004],
						   ptr[2005], ptr[2006], ptr[2007], ptr[2008], ptr[2009],
						   ptr[2010], ptr[2011], ptr[2012], ptr[2013], ptr[2014],
						   ptr[2015], ptr[2016], ptr[2017], ptr[2018], ptr[2019],
						   ptr[2020], ptr[2021], ptr[2022], ptr[2023], ptr[2024],
						   ptr[2025], ptr[2026], ptr[2027], ptr[2028], ptr[2029],
						   ptr[2030], ptr[2031], ptr[2032], ptr[2033], ptr[2034],
						   ptr[2035], ptr[2036], ptr[2037], ptr[2038], ptr[2039],
						   ptr[2040], ptr[2041], ptr[2042], ptr[2043], ptr[2044],
						   ptr[2045], ptr[2046], ptr[2047], ptr[2048], ptr[2049],
						   ptr[2050], ptr[2051], ptr[2052], ptr[2053], ptr[2054],
						   ptr[2055], ptr[2056], ptr[2057], ptr[2058], ptr[2059],
						   ptr[2060], ptr[2061], ptr[2062], ptr[2063], ptr[2064],
						   ptr[2065], ptr[2066], ptr[2067], ptr[2068], ptr[2069],
						   ptr[2070], ptr[2071], ptr[2072], ptr[2073], ptr[2074],
						   ptr[2075], ptr[2076], ptr[2077], ptr[2078], ptr[2079],
						   ptr[2080], ptr[2081], ptr[2082], ptr[2083], ptr[2084],
						   ptr[2085], ptr[2086], ptr[2087], ptr[2088], ptr[2089],
						   ptr[2090], ptr[2091], ptr[2092], ptr[2093], ptr[2094],
						   ptr[2095], ptr[2096], ptr[2097], ptr[2098], ptr[2099],
						   ptr[2100], ptr[2101], ptr[2102], ptr[2103], ptr[2104],
						   ptr[2105], ptr[2106], ptr[2107], ptr[2108], ptr[2109],
						   ptr[2110], ptr[2111], ptr[2112], ptr[2113], ptr[2114],
						   ptr[2115], ptr[2116], ptr[2117], ptr[2118], ptr[2119],
						   ptr[2120], ptr[2121], ptr[2122], ptr[2123], ptr[2124],
						   ptr[2125], ptr[2126], ptr[2127], ptr[2128], ptr[2129],
						   ptr[2130], ptr[2131], ptr[2132], ptr[2133], ptr[2134],
						   ptr[2135], ptr[2136], ptr[2137], ptr[2138], ptr[2139],
						   ptr[2140], ptr[2141], ptr[2142], ptr[2143], ptr[2144],
						   ptr[2145], ptr[2146], ptr[2147], ptr[2148], ptr[2149],
						   ptr[2150], ptr[2151], ptr[2152], ptr[2153], ptr[2154],
						   ptr[2155], ptr[2156], ptr[2157], ptr[2158], ptr[2159],
						   ptr[2160], ptr[2161], ptr[2162], ptr[2163], ptr[2164],
						   ptr[2165], ptr[2166], ptr[2167], ptr[2168], ptr[2169],
						   ptr[2170], ptr[2171], ptr[2172], ptr[2173], ptr[2174],
						   ptr[2175], ptr[2176], ptr[2177], ptr[2178], ptr[2179],
						   ptr[2180], ptr[2181], ptr[2182], ptr[2183], ptr[2184],
						   ptr[2185], ptr[2186], ptr[2187], ptr[2188], ptr[2189],
						   ptr[2190], ptr[2191], ptr[2192], ptr[2193], ptr[2194],
						   ptr[2195], ptr[2196], ptr[2197], ptr[2198], ptr[2199],
						   ptr[2200], ptr[2201], ptr[2202], ptr[2203], ptr[2204],
						   ptr[2205], ptr[2206], ptr[2207], ptr[2208], ptr[2209],
						   ptr[2210], ptr[2211], ptr[2212], ptr[2213], ptr[2214],
						   ptr[2215], ptr[2216], ptr[2217], ptr[2218], ptr[2219],
						   ptr[2220], ptr[2221], ptr[2222], ptr[2223], ptr[2224],
						   ptr[2225], ptr[2226], ptr[2227], ptr[2228], ptr[2229],
						   ptr[2230], ptr[2231], ptr[2232], ptr[2233], ptr[2234],
						   ptr[2235], ptr[2236], ptr[2237], ptr[2238], ptr[2239],
						   ptr[2240], ptr[2241], ptr[2242], ptr[2243], ptr[2244],
						   ptr[2245], ptr[2246], ptr[2247], ptr[2248], ptr[2249],
						   ptr[2250], ptr[2251], ptr[2252], ptr[2253], ptr[2254],
						   ptr[2255], ptr[2256], ptr[2257], ptr[2258], ptr[2259],
						   ptr[2260], ptr[2261], ptr[2262], ptr[2263], ptr[2264],
						   ptr[2265], ptr[2266], ptr[2267], ptr[2268], ptr[2269],
						   ptr[2270], ptr[2271], ptr[2272], ptr[2273], ptr[2274],
						   ptr[2275], ptr[2276], ptr[2277], ptr[2278], ptr[2279],
						   ptr[2280], ptr[2281], ptr[2282], ptr[2283], ptr[2284],
						   ptr[2285], ptr[2286], ptr[2287], ptr[2288], ptr[2289],
						   ptr[2290], ptr[2291], ptr[2292], ptr[2293], ptr[2294],
						   ptr[2295], ptr[2296], ptr[2297], ptr[2298], ptr[2299],
						   ptr[2300], ptr[2301], ptr[2302], ptr[2303], ptr[2304],
						   ptr[2305], ptr[2306], ptr[2307], ptr[2308], ptr[2309],
						   ptr[2310], ptr[2311], ptr[2312], ptr[2313], ptr[2314],
						   ptr[2315], ptr[2316], ptr[2317], ptr[2318], ptr[2319],
						   ptr[2320], ptr[2321], ptr[2322], ptr[2323], ptr[2324],
						   ptr[2325], ptr[2326], ptr[2327], ptr[2328], ptr[2329],
						   ptr[2330], ptr[2331], ptr[2332], ptr[2333], ptr[2334],
						   ptr[2335], ptr[2336], ptr[2337], ptr[2338], ptr[2339],
						   ptr[2340], ptr[2341], ptr[2342], ptr[2343], ptr[2344],
						   ptr[2345], ptr[2346], ptr[2347], ptr[2348], ptr[2349],
						   ptr[2350], ptr[2351], ptr[2352], ptr[2353], ptr[2354],
						   ptr[2355], ptr[2356], ptr[2357], ptr[2358], ptr[2359],
						   ptr[2360], ptr[2361], ptr[2362], ptr[2363], ptr[2364],
						   ptr[2365], ptr[2366], ptr[2367], ptr[2368], ptr[2369],
						   ptr[2370], ptr[2371], ptr[2372], ptr[2373], ptr[2374],
						   ptr[2375], ptr[2376], ptr[2377], ptr[2378], ptr[2379],
						   ptr[2380], ptr[2381], ptr[2382], ptr[2383], ptr[2384],
						   ptr[2385], ptr[2386], ptr[2387], ptr[2388], ptr[2389],
						   ptr[2390], ptr[2391], ptr[2392], ptr[2393], ptr[2394],
						   ptr[2395], ptr[2396], ptr[2397], ptr[2398], ptr[2399],
						   ptr[2400], ptr[2401], ptr[2402], ptr[2403], ptr[2404],
						   ptr[2405], ptr[2406], ptr[2407], ptr[2408], ptr[2409],
						   ptr[2410], ptr[2411], ptr[2412], ptr[2413], ptr[2414],
						   ptr[2415], ptr[2416], ptr[2417], ptr[2418], ptr[2419],
						   ptr[2420], ptr[2421], ptr[2422], ptr[2423], ptr[2424],
						   ptr[2425], ptr[2426], ptr[2427], ptr[2428], ptr[2429],
						   ptr[2430], ptr[2431], ptr[2432], ptr[2433], ptr[2434],
						   ptr[2435], ptr[2436], ptr[2437], ptr[2438], ptr[2439],
						   ptr[2440], ptr[2441], ptr[2442], ptr[2443], ptr[2444],
						   ptr[2445], ptr[2446], ptr[2447], ptr[2448], ptr[2449],
						   ptr[2450], ptr[2451], ptr[2452], ptr[2453], ptr[2454],
						   ptr[2455], ptr[2456], ptr[2457], ptr[2458], ptr[2459],
						   ptr[2460], ptr[2461], ptr[2462], ptr[2463], ptr[2464],
						   ptr[2465], ptr[2466], ptr[2467], ptr[2468], ptr[2469],
						   ptr[2470], ptr[2471], ptr[2472], ptr[2473], ptr[2474],
						   ptr[2475], ptr[2476], ptr[2477], ptr[2478], ptr[2479],
						   ptr[2480], ptr[2481], ptr[2482], ptr[2483], ptr[2484],
						   ptr[2485], ptr[2486], ptr[2487], ptr[2488], ptr[2489],
						   ptr[2490], ptr[2491], ptr[2492], ptr[2493], ptr[2494],
						   ptr[2495], ptr[2496], ptr[2497], ptr[2498], ptr[2499],
						   ptr[2500], ptr[2501], ptr[2502], ptr[2503], ptr[2504],
						   ptr[2505], ptr[2506], ptr[2507], ptr[2508], ptr[2509],
						   ptr[2510], ptr[2511], ptr[2512], ptr[2513], ptr[2514],
						   ptr[2515], ptr[2516], ptr[2517], ptr[2518], ptr[2519],
						   ptr[2520], ptr[2521], ptr[2522], ptr[2523], ptr[2524],
						   ptr[2525], ptr[2526], ptr[2527], ptr[2528], ptr[2529],
						   ptr[2530], ptr[2531], ptr[2532], ptr[2533], ptr[2534],
						   ptr[2535], ptr[2536], ptr[2537], ptr[2538], ptr[2539],
						   ptr[2540], ptr[2541], ptr[2542], ptr[2543], ptr[2544],
						   ptr[2545], ptr[2546], ptr[2547], ptr[2548], ptr[2549],
						   ptr[2550], ptr[2551], ptr[2552], ptr[2553], ptr[2554],
						   ptr[2555], ptr[2556], ptr[2557], ptr[2558], ptr[2559],
						   ptr[2560], ptr[2561], ptr[2562], ptr[2563], ptr[2564],
						   ptr[2565], ptr[2566], ptr[2567], ptr[2568], ptr[2569],
						   ptr[2570], ptr[2571], ptr[2572], ptr[2573], ptr[2574],
						   ptr[2575], ptr[2576], ptr[2577], ptr[2578], ptr[2579],
						   ptr[2580], ptr[2581], ptr[2582], ptr[2583], ptr[2584],
						   ptr[2585], ptr[2586], ptr[2587], ptr[2588], ptr[2589],
						   ptr[2590], ptr[2591], ptr[2592], ptr[2593], ptr[2594],
						   ptr[2595], ptr[2596], ptr[2597], ptr[2598], ptr[2599],
						   ptr[2600], ptr[2601], ptr[2602], ptr[2603], ptr[2604],
						   ptr[2605], ptr[2606], ptr[2607], ptr[2608], ptr[2609],
						   ptr[2610], ptr[2611], ptr[2612], ptr[2613], ptr[2614],
						   ptr[2615], ptr[2616], ptr[2617], ptr[2618], ptr[2619],
						   ptr[2620], ptr[2621], ptr[2622], ptr[2623], ptr[2624],
						   ptr[2625], ptr[2626], ptr[2627], ptr[2628], ptr[2629],
						   ptr[2630], ptr[2631], ptr[2632], ptr[2633], ptr[2634],
						   ptr[2635], ptr[2636], ptr[2637], ptr[2638], ptr[2639],
						   ptr[2640], ptr[2641], ptr[2642], ptr[2643], ptr[2644],
						   ptr[2645], ptr[2646], ptr[2647], ptr[2648], ptr[2649],
						   ptr[2650], ptr[2651], ptr[2652], ptr[2653], ptr[2654],
						   ptr[2655], ptr[2656], ptr[2657], ptr[2658], ptr[2659],
						   ptr[2660], ptr[2661], ptr[2662], ptr[2663], ptr[2664],
						   ptr[2665], ptr[2666], ptr[2667], ptr[2668], ptr[2669],
						   ptr[2670], ptr[2671], ptr[2672], ptr[2673], ptr[2674],
						   ptr[2675], ptr[2676], ptr[2677], ptr[2678], ptr[2679],
						   ptr[2680], ptr[2681], ptr[2682], ptr[2683], ptr[2684],
						   ptr[2685], ptr[2686], ptr[2687], ptr[2688], ptr[2689],
						   ptr[2690], ptr[2691], ptr[2692], ptr[2693], ptr[2694],
						   ptr[2695], ptr[2696], ptr[2697], ptr[2698], ptr[2699],
						   ptr[2700], ptr[2701], ptr[2702], ptr[2703], ptr[2704],
						   ptr[2705], ptr[2706], ptr[2707], ptr[2708], ptr[2709],
						   ptr[2710], ptr[2711], ptr[2712], ptr[2713], ptr[2714],
						   ptr[2715], ptr[2716], ptr[2717], ptr[2718], ptr[2719],
						   ptr[2720], ptr[2721], ptr[2722], ptr[2723], ptr[2724],
						   ptr[2725], ptr[2726], ptr[2727], ptr[2728], ptr[2729],
						   ptr[2730], ptr[2731], ptr[2732], ptr[2733], ptr[2734],
						   ptr[2735], ptr[2736], ptr[2737], ptr[2738], ptr[2739],
						   ptr[2740], ptr[2741], ptr[2742], ptr[2743], ptr[2744],
						   ptr[2745], ptr[2746], ptr[2747], ptr[2748], ptr[2749],
						   ptr[2750], ptr[2751], ptr[2752], ptr[2753], ptr[2754],
						   ptr[2755], ptr[2756], ptr[2757], ptr[2758], ptr[2759],
						   ptr[2760], ptr[2761], ptr[2762], ptr[2763], ptr[2764],
						   ptr[2765], ptr[2766], ptr[2767], ptr[2768], ptr[2769],
						   ptr[2770], ptr[2771], ptr[2772], ptr[2773], ptr[2774],
						   ptr[2775], ptr[2776], ptr[2777], ptr[2778], ptr[2779],
						   ptr[2780], ptr[2781], ptr[2782], ptr[2783], ptr[2784],
						   ptr[2785], ptr[2786], ptr[2787], ptr[2788], ptr[2789],
						   ptr[2790], ptr[2791], ptr[2792], ptr[2793], ptr[2794],
						   ptr[2795], ptr[2796], ptr[2797], ptr[2798], ptr[2799],
						   ptr[2800], ptr[2801], ptr[2802], ptr[2803], ptr[2804],
						   ptr[2805], ptr[2806], ptr[2807], ptr[2808], ptr[2809],
						   ptr[2810], ptr[2811], ptr[2812], ptr[2813], ptr[2814],
						   ptr[2815], ptr[2816], ptr[2817], ptr[2818], ptr[2819],
						   ptr[2820], ptr[2821], ptr[2822], ptr[2823], ptr[2824],
						   ptr[2825], ptr[2826], ptr[2827], ptr[2828], ptr[2829],
						   ptr[2830], ptr[2831], ptr[2832], ptr[2833], ptr[2834],
						   ptr[2835], ptr[2836], ptr[2837], ptr[2838], ptr[2839],
						   ptr[2840], ptr[2841], ptr[2842], ptr[2843], ptr[2844],
						   ptr[2845], ptr[2846], ptr[2847], ptr[2848], ptr[2849],
						   ptr[2850], ptr[2851], ptr[2852], ptr[2853], ptr[2854],
						   ptr[2855], ptr[2856], ptr[2857], ptr[2858], ptr[2859],
						   ptr[2860], ptr[2861], ptr[2862], ptr[2863], ptr[2864],
						   ptr[2865], ptr[2866], ptr[2867], ptr[2868], ptr[2869],
						   ptr[2870], ptr[2871], ptr[2872], ptr[2873], ptr[2874],
						   ptr[2875], ptr[2876], ptr[2877], ptr[2878], ptr[2879],
						   ptr[2880], ptr[2881], ptr[2882], ptr[2883], ptr[2884],
						   ptr[2885], ptr[2886], ptr[2887], ptr[2888], ptr[2889],
						   ptr[2890], ptr[2891], ptr[2892], ptr[2893], ptr[2894],
						   ptr[2895], ptr[2896], ptr[2897], ptr[2898], ptr[2899],
						   ptr[2900], ptr[2901], ptr[2902], ptr[2903], ptr[2904],
						   ptr[2905], ptr[2906], ptr[2907], ptr[2908], ptr[2909],
						   ptr[2910], ptr[2911], ptr[2912], ptr[2913], ptr[2914],
						   ptr[2915], ptr[2916], ptr[2917], ptr[2918], ptr[2919],
						   ptr[2920], ptr[2921], ptr[2922], ptr[2923], ptr[2924],
						   ptr[2925], ptr[2926], ptr[2927], ptr[2928], ptr[2929],
						   ptr[2930], ptr[2931], ptr[2932], ptr[2933], ptr[2934],
						   ptr[2935], ptr[2936], ptr[2937], ptr[2938], ptr[2939],
						   ptr[2940], ptr[2941], ptr[2942], ptr[2943], ptr[2944],
						   ptr[2945], ptr[2946], ptr[2947], ptr[2948], ptr[2949],
						   ptr[2950], ptr[2951], ptr[2952], ptr[2953], ptr[2954],
						   ptr[2955], ptr[2956], ptr[2957], ptr[2958], ptr[2959],
						   ptr[2960], ptr[2961], ptr[2962], ptr[2963], ptr[2964],
						   ptr[2965], ptr[2966], ptr[2967], ptr[2968], ptr[2969],
						   ptr[2970], ptr[2971], ptr[2972], ptr[2973], ptr[2974],
						   ptr[2975], ptr[2976], ptr[2977], ptr[2978], ptr[2979],
						   ptr[2980], ptr[2981], ptr[2982], ptr[2983], ptr[2984],
						   ptr[2985], ptr[2986], ptr[2987], ptr[2988], ptr[2989],
						   ptr[2990], ptr[2991], ptr[2992], ptr[2993], ptr[2994],
						   ptr[2995], ptr[2996], ptr[2997], ptr[2998], ptr[2999],
						   ptr[3000], ptr[3001], ptr[3002], ptr[3003], ptr[3004],
						   ptr[3005], ptr[3006], ptr[3007], ptr[3008], ptr[3009],
						   ptr[3010], ptr[3011], ptr[3012], ptr[3013], ptr[3014],
						   ptr[3015], ptr[3016], ptr[3017], ptr[3018], ptr[3019],
						   ptr[3020], ptr[3021], ptr[3022], ptr[3023], ptr[3024],
						   ptr[3025], ptr[3026], ptr[3027], ptr[3028], ptr[3029],
						   ptr[3030], ptr[3031], ptr[3032], ptr[3033], ptr[3034],
						   ptr[3035], ptr[3036], ptr[3037], ptr[3038], ptr[3039],
						   ptr[3040], ptr[3041], ptr[3042], ptr[3043], ptr[3044],
						   ptr[3045], ptr[3046], ptr[3047], ptr[3048], ptr[3049],
						   ptr[3050], ptr[3051], ptr[3052], ptr[3053], ptr[3054],
						   ptr[3055], ptr[3056], ptr[3057], ptr[3058], ptr[3059],
						   ptr[3060], ptr[3061], ptr[3062], ptr[3063], ptr[3064],
						   ptr[3065], ptr[3066], ptr[3067], ptr[3068], ptr[3069],
						   ptr[3070], ptr[3071], ptr[3072], ptr[3073], ptr[3074],
						   ptr[3075], ptr[3076], ptr[3077], ptr[3078], ptr[3079],
						   ptr[3080], ptr[3081], ptr[3082], ptr[3083], ptr[3084],
						   ptr[3085], ptr[3086], ptr[3087], ptr[3088], ptr[3089],
						   ptr[3090], ptr[3091], ptr[3092], ptr[3093], ptr[3094],
						   ptr[3095], ptr[3096], ptr[3097], ptr[3098], ptr[3099],
						   ptr[3100], ptr[3101], ptr[3102], ptr[3103], ptr[3104],
						   ptr[3105], ptr[3106], ptr[3107], ptr[3108], ptr[3109],
						   ptr[3110], ptr[3111], ptr[3112], ptr[3113], ptr[3114],
						   ptr[3115], ptr[3116], ptr[3117], ptr[3118], ptr[3119],
						   ptr[3120], ptr[3121], ptr[3122], ptr[3123], ptr[3124],
						   ptr[3125], ptr[3126], ptr[3127], ptr[3128], ptr[3129],
						   ptr[3130], ptr[3131], ptr[3132], ptr[3133], ptr[3134],
						   ptr[3135], ptr[3136], ptr[3137], ptr[3138], ptr[3139],
						   ptr[3140], ptr[3141], ptr[3142], ptr[3143], ptr[3144],
						   ptr[3145], ptr[3146], ptr[3147], ptr[3148], ptr[3149],
						   ptr[3150], ptr[3151], ptr[3152], ptr[3153], ptr[3154],
						   ptr[3155], ptr[3156], ptr[3157], ptr[3158], ptr[3159],
						   ptr[3160], ptr[3161], ptr[3162], ptr[3163], ptr[3164],
						   ptr[3165], ptr[3166], ptr[3167], ptr[3168], ptr[3169],
						   ptr[3170], ptr[3171], ptr[3172], ptr[3173], ptr[3174],
						   ptr[3175], ptr[3176], ptr[3177], ptr[3178], ptr[3179],
						   ptr[3180], ptr[3181], ptr[3182], ptr[3183], ptr[3184],
						   ptr[3185], ptr[3186], ptr[3187], ptr[3188], ptr[3189],
						   ptr[3190], ptr[3191], ptr[3192], ptr[3193], ptr[3194],
						   ptr[3195], ptr[3196], ptr[3197], ptr[3198], ptr[3199],
						   ptr[3200], ptr[3201], ptr[3202], ptr[3203], ptr[3204],
						   ptr[3205], ptr[3206], ptr[3207], ptr[3208], ptr[3209],
						   ptr[3210], ptr[3211], ptr[3212], ptr[3213], ptr[3214],
						   ptr[3215], ptr[3216], ptr[3217], ptr[3218], ptr[3219],
						   ptr[3220], ptr[3221], ptr[3222], ptr[3223], ptr[3224],
						   ptr[3225], ptr[3226], ptr[3227], ptr[3228], ptr[3229],
						   ptr[3230], ptr[3231], ptr[3232], ptr[3233], ptr[3234],
						   ptr[3235], ptr[3236], ptr[3237], ptr[3238], ptr[3239],
						   ptr[3240], ptr[3241], ptr[3242], ptr[3243], ptr[3244],
						   ptr[3245], ptr[3246], ptr[3247], ptr[3248], ptr[3249],
						   ptr[3250], ptr[3251], ptr[3252], ptr[3253], ptr[3254],
						   ptr[3255], ptr[3256], ptr[3257], ptr[3258], ptr[3259],
						   ptr[3260], ptr[3261], ptr[3262], ptr[3263], ptr[3264],
						   ptr[3265], ptr[3266], ptr[3267], ptr[3268], ptr[3269],
						   ptr[3270], ptr[3271], ptr[3272], ptr[3273], ptr[3274],
						   ptr[3275], ptr[3276], ptr[3277], ptr[3278], ptr[3279],
						   ptr[3280], ptr[3281], ptr[3282], ptr[3283], ptr[3284],
						   ptr[3285], ptr[3286], ptr[3287], ptr[3288], ptr[3289],
						   ptr[3290], ptr[3291], ptr[3292], ptr[3293], ptr[3294],
						   ptr[3295], ptr[3296], ptr[3297], ptr[3298], ptr[3299],
						   ptr[3300], ptr[3301], ptr[3302], ptr[3303], ptr[3304],
						   ptr[3305], ptr[3306], ptr[3307], ptr[3308], ptr[3309],
						   ptr[3310], ptr[3311], ptr[3312], ptr[3313], ptr[3314],
						   ptr[3315], ptr[3316], ptr[3317], ptr[3318], ptr[3319],
						   ptr[3320], ptr[3321], ptr[3322], ptr[3323], ptr[3324],
						   ptr[3325], ptr[3326], ptr[3327], ptr[3328], ptr[3329],
						   ptr[3330], ptr[3331], ptr[3332], ptr[3333], ptr[3334],
						   ptr[3335], ptr[3336], ptr[3337], ptr[3338], ptr[3339],
						   ptr[3340], ptr[3341], ptr[3342], ptr[3343], ptr[3344],
						   ptr[3345], ptr[3346], ptr[3347], ptr[3348], ptr[3349],
						   ptr[3350], ptr[3351], ptr[3352], ptr[3353], ptr[3354],
						   ptr[3355], ptr[3356], ptr[3357], ptr[3358], ptr[3359],
						   ptr[3360], ptr[3361], ptr[3362], ptr[3363], ptr[3364],
						   ptr[3365], ptr[3366], ptr[3367], ptr[3368], ptr[3369],
						   ptr[3370], ptr[3371], ptr[3372], ptr[3373], ptr[3374],
						   ptr[3375], ptr[3376], ptr[3377], ptr[3378], ptr[3379],
						   ptr[3380], ptr[3381], ptr[3382], ptr[3383], ptr[3384],
						   ptr[3385], ptr[3386], ptr[3387], ptr[3388], ptr[3389],
						   ptr[3390], ptr[3391], ptr[3392], ptr[3393], ptr[3394],
						   ptr[3395], ptr[3396], ptr[3397], ptr[3398], ptr[3399],
						   ptr[3400], ptr[3401], ptr[3402], ptr[3403], ptr[3404],
						   ptr[3405], ptr[3406], ptr[3407], ptr[3408], ptr[3409],
						   ptr[3410], ptr[3411], ptr[3412], ptr[3413], ptr[3414],
						   ptr[3415], ptr[3416], ptr[3417], ptr[3418], ptr[3419],
						   ptr[3420], ptr[3421], ptr[3422], ptr[3423], ptr[3424],
						   ptr[3425], ptr[3426], ptr[3427], ptr[3428], ptr[3429],
						   ptr[3430], ptr[3431], ptr[3432], ptr[3433], ptr[3434],
						   ptr[3435], ptr[3436], ptr[3437], ptr[3438], ptr[3439],
						   ptr[3440], ptr[3441], ptr[3442], ptr[3443], ptr[3444],
						   ptr[3445], ptr[3446], ptr[3447], ptr[3448], ptr[3449],
						   ptr[3450], ptr[3451], ptr[3452], ptr[3453], ptr[3454],
						   ptr[3455], ptr[3456], ptr[3457], ptr[3458], ptr[3459],
						   ptr[3460], ptr[3461], ptr[3462], ptr[3463], ptr[3464],
						   ptr[3465], ptr[3466], ptr[3467], ptr[3468], ptr[3469],
						   ptr[3470], ptr[3471], ptr[3472], ptr[3473], ptr[3474],
						   ptr[3475], ptr[3476], ptr[3477], ptr[3478], ptr[3479],
						   ptr[3480], ptr[3481], ptr[3482], ptr[3483], ptr[3484],
						   ptr[3485], ptr[3486], ptr[3487], ptr[3488], ptr[3489],
						   ptr[3490], ptr[3491], ptr[3492], ptr[3493], ptr[3494],
						   ptr[3495], ptr[3496], ptr[3497], ptr[3498], ptr[3499],
						   ptr[3500], ptr[3501], ptr[3502], ptr[3503], ptr[3504],
						   ptr[3505], ptr[3506], ptr[3507], ptr[3508], ptr[3509],
						   ptr[3510], ptr[3511], ptr[3512], ptr[3513], ptr[3514],
						   ptr[3515], ptr[3516], ptr[3517], ptr[3518], ptr[3519],
						   ptr[3520], ptr[3521], ptr[3522], ptr[3523], ptr[3524],
						   ptr[3525], ptr[3526], ptr[3527], ptr[3528], ptr[3529],
						   ptr[3530], ptr[3531], ptr[3532], ptr[3533], ptr[3534],
						   ptr[3535], ptr[3536], ptr[3537], ptr[3538], ptr[3539],
						   ptr[3540], ptr[3541], ptr[3542], ptr[3543], ptr[3544],
						   ptr[3545], ptr[3546], ptr[3547], ptr[3548], ptr[3549],
						   ptr[3550], ptr[3551], ptr[3552], ptr[3553], ptr[3554],
						   ptr[3555], ptr[3556], ptr[3557], ptr[3558], ptr[3559],
						   ptr[3560], ptr[3561], ptr[3562], ptr[3563], ptr[3564],
						   ptr[3565], ptr[3566], ptr[3567], ptr[3568], ptr[3569],
						   ptr[3570], ptr[3571], ptr[3572], ptr[3573], ptr[3574],
						   ptr[3575], ptr[3576], ptr[3577], ptr[3578], ptr[3579],
						   ptr[3580], ptr[3581], ptr[3582], ptr[3583], ptr[3584],
						   ptr[3585], ptr[3586], ptr[3587], ptr[3588], ptr[3589],
						   ptr[3590], ptr[3591], ptr[3592], ptr[3593], ptr[3594],
						   ptr[3595], ptr[3596], ptr[3597], ptr[3598], ptr[3599],
						   ptr[3600], ptr[3601], ptr[3602], ptr[3603], ptr[3604],
						   ptr[3605], ptr[3606], ptr[3607], ptr[3608], ptr[3609],
						   ptr[3610], ptr[3611], ptr[3612], ptr[3613], ptr[3614],
						   ptr[3615], ptr[3616], ptr[3617], ptr[3618], ptr[3619],
						   ptr[3620], ptr[3621], ptr[3622], ptr[3623], ptr[3624],
						   ptr[3625], ptr[3626], ptr[3627], ptr[3628], ptr[3629],
						   ptr[3630], ptr[3631], ptr[3632], ptr[3633], ptr[3634],
						   ptr[3635], ptr[3636], ptr[3637], ptr[3638], ptr[3639],
						   ptr[3640], ptr[3641], ptr[3642], ptr[3643], ptr[3644],
						   ptr[3645], ptr[3646], ptr[3647], ptr[3648], ptr[3649],
						   ptr[3650], ptr[3651], ptr[3652], ptr[3653], ptr[3654],
						   ptr[3655], ptr[3656], ptr[3657], ptr[3658], ptr[3659],
						   ptr[3660], ptr[3661], ptr[3662], ptr[3663], ptr[3664],
						   ptr[3665], ptr[3666], ptr[3667], ptr[3668], ptr[3669],
						   ptr[3670], ptr[3671], ptr[3672], ptr[3673], ptr[3674],
						   ptr[3675], ptr[3676], ptr[3677], ptr[3678], ptr[3679],
						   ptr[3680], ptr[3681], ptr[3682], ptr[3683], ptr[3684],
						   ptr[3685], ptr[3686], ptr[3687], ptr[3688], ptr[3689],
						   ptr[3690], ptr[3691], ptr[3692], ptr[3693], ptr[3694],
						   ptr[3695], ptr[3696], ptr[3697], ptr[3698], ptr[3699],
						   ptr[3700], ptr[3701], ptr[3702], ptr[3703], ptr[3704],
						   ptr[3705], ptr[3706], ptr[3707], ptr[3708], ptr[3709],
						   ptr[3710], ptr[3711], ptr[3712], ptr[3713], ptr[3714],
						   ptr[3715], ptr[3716], ptr[3717], ptr[3718], ptr[3719],
						   ptr[3720], ptr[3721], ptr[3722], ptr[3723], ptr[3724],
						   ptr[3725], ptr[3726], ptr[3727], ptr[3728], ptr[3729],
						   ptr[3730], ptr[3731], ptr[3732], ptr[3733], ptr[3734],
						   ptr[3735], ptr[3736], ptr[3737], ptr[3738], ptr[3739],
						   ptr[3740], ptr[3741], ptr[3742], ptr[3743], ptr[3744],
						   ptr[3745], ptr[3746], ptr[3747], ptr[3748], ptr[3749],
						   ptr[3750], ptr[3751], ptr[3752], ptr[3753], ptr[3754],
						   ptr[3755], ptr[3756], ptr[3757], ptr[3758], ptr[3759],
						   ptr[3760], ptr[3761], ptr[3762], ptr[3763], ptr[3764],
						   ptr[3765], ptr[3766], ptr[3767], ptr[3768], ptr[3769],
						   ptr[3770], ptr[3771], ptr[3772], ptr[3773], ptr[3774],
						   ptr[3775], ptr[3776], ptr[3777], ptr[3778], ptr[3779],
						   ptr[3780], ptr[3781], ptr[3782], ptr[3783], ptr[3784],
						   ptr[3785], ptr[3786], ptr[3787], ptr[3788], ptr[3789],
						   ptr[3790], ptr[3791], ptr[3792], ptr[3793], ptr[3794],
						   ptr[3795], ptr[3796], ptr[3797], ptr[3798], ptr[3799],
						   ptr[3800], ptr[3801], ptr[3802], ptr[3803], ptr[3804],
						   ptr[3805], ptr[3806], ptr[3807], ptr[3808], ptr[3809],
						   ptr[3810], ptr[3811], ptr[3812], ptr[3813], ptr[3814],
						   ptr[3815], ptr[3816], ptr[3817], ptr[3818], ptr[3819],
						   ptr[3820], ptr[3821], ptr[3822], ptr[3823], ptr[3824],
						   ptr[3825], ptr[3826], ptr[3827], ptr[3828], ptr[3829],
						   ptr[3830], ptr[3831], ptr[3832], ptr[3833], ptr[3834],
						   ptr[3835], ptr[3836], ptr[3837], ptr[3838], ptr[3839],
						   ptr[3840], ptr[3841], ptr[3842], ptr[3843], ptr[3844],
						   ptr[3845], ptr[3846], ptr[3847], ptr[3848], ptr[3849],
						   ptr[3850], ptr[3851], ptr[3852], ptr[3853], ptr[3854],
						   ptr[3855], ptr[3856], ptr[3857], ptr[3858], ptr[3859],
						   ptr[3860], ptr[3861], ptr[3862], ptr[3863], ptr[3864],
						   ptr[3865], ptr[3866], ptr[3867], ptr[3868], ptr[3869],
						   ptr[3870], ptr[3871], ptr[3872], ptr[3873], ptr[3874],
						   ptr[3875], ptr[3876], ptr[3877], ptr[3878], ptr[3879],
						   ptr[3880], ptr[3881], ptr[3882], ptr[3883], ptr[3884],
						   ptr[3885], ptr[3886], ptr[3887], ptr[3888], ptr[3889],
						   ptr[3890], ptr[3891], ptr[3892], ptr[3893], ptr[3894],
						   ptr[3895], ptr[3896], ptr[3897], ptr[3898], ptr[3899],
						   ptr[3900], ptr[3901], ptr[3902], ptr[3903], ptr[3904],
						   ptr[3905], ptr[3906], ptr[3907], ptr[3908], ptr[3909],
						   ptr[3910], ptr[3911], ptr[3912], ptr[3913], ptr[3914],
						   ptr[3915], ptr[3916], ptr[3917], ptr[3918], ptr[3919],
						   ptr[3920], ptr[3921], ptr[3922], ptr[3923], ptr[3924],
						   ptr[3925], ptr[3926], ptr[3927], ptr[3928], ptr[3929],
						   ptr[3930], ptr[3931], ptr[3932], ptr[3933], ptr[3934],
						   ptr[3935], ptr[3936], ptr[3937], ptr[3938], ptr[3939],
						   ptr[3940], ptr[3941], ptr[3942], ptr[3943], ptr[3944],
						   ptr[3945], ptr[3946], ptr[3947], ptr[3948], ptr[3949],
						   ptr[3950], ptr[3951], ptr[3952], ptr[3953], ptr[3954],
						   ptr[3955], ptr[3956], ptr[3957], ptr[3958], ptr[3959],
						   ptr[3960], ptr[3961], ptr[3962], ptr[3963], ptr[3964],
						   ptr[3965], ptr[3966], ptr[3967], ptr[3968], ptr[3969],
						   ptr[3970], ptr[3971], ptr[3972], ptr[3973], ptr[3974],
						   ptr[3975], ptr[3976], ptr[3977], ptr[3978], ptr[3979],
						   ptr[3980], ptr[3981], ptr[3982], ptr[3983], ptr[3984],
						   ptr[3985], ptr[3986], ptr[3987], ptr[3988], ptr[3989],
						   ptr[3990], ptr[3991], ptr[3992], ptr[3993], ptr[3994],
						   ptr[3995], ptr[3996], ptr[3997], ptr[3998], ptr[3999]);
	    } else {
		comp[ci].ptr = NULL;
	    }
	    if (comp_type[i].gui_create) {
		comp[ci].gui_ptr = comp_type[i].gui_create(comp[ci].page, comp[ci].name,
							   ptr[0], ptr[1], ptr[2], ptr[3], ptr[4],
							   ptr[5], ptr[6], ptr[7], ptr[8], ptr[9],
							   ptr[10], ptr[11], ptr[12], ptr[13], ptr[14],
							   ptr[15], ptr[16], ptr[17], ptr[18], ptr[19],
							   ptr[20], ptr[21], ptr[22], ptr[23], ptr[24],
							   ptr[25], ptr[26], ptr[27], ptr[28], ptr[29],
							   ptr[30], ptr[31], ptr[32], ptr[33], ptr[34],
							   ptr[35], ptr[36], ptr[37], ptr[38], ptr[39],
							   ptr[40], ptr[41], ptr[42], ptr[43], ptr[44],
							   ptr[45], ptr[46], ptr[47], ptr[48], ptr[49],
							   ptr[50], ptr[51], ptr[52], ptr[53], ptr[54],
							   ptr[55], ptr[56], ptr[57], ptr[58], ptr[59],
							   ptr[60], ptr[61], ptr[62], ptr[63], ptr[64],
							   ptr[65], ptr[66], ptr[67], ptr[68], ptr[69],
							   ptr[70], ptr[71], ptr[72], ptr[73], ptr[74],
							   ptr[75], ptr[76], ptr[77], ptr[78], ptr[79],
							   ptr[80], ptr[81], ptr[82], ptr[83], ptr[84],
							   ptr[85], ptr[86], ptr[87], ptr[88], ptr[89],
							   ptr[90], ptr[91], ptr[92], ptr[93], ptr[94],
							   ptr[95], ptr[96], ptr[97], ptr[98], ptr[99],
							   ptr[100], ptr[101], ptr[102], ptr[103], ptr[104],
							   ptr[105], ptr[106], ptr[107], ptr[108], ptr[109],
							   ptr[110], ptr[111], ptr[112], ptr[113], ptr[114],
							   ptr[115], ptr[116], ptr[117], ptr[118], ptr[119],
							   ptr[120], ptr[121], ptr[122], ptr[123], ptr[124],
							   ptr[125], ptr[126], ptr[127], ptr[128], ptr[129],
							   ptr[130], ptr[131], ptr[132], ptr[133], ptr[134],
							   ptr[135], ptr[136], ptr[137], ptr[138], ptr[139],
							   ptr[140], ptr[141], ptr[142], ptr[143], ptr[144],
							   ptr[145], ptr[146], ptr[147], ptr[148], ptr[149],
							   ptr[150], ptr[151], ptr[152], ptr[153], ptr[154],
							   ptr[155], ptr[156], ptr[157], ptr[158], ptr[159],
							   ptr[160], ptr[161], ptr[162], ptr[163], ptr[164],
							   ptr[165], ptr[166], ptr[167], ptr[168], ptr[169],
							   ptr[170], ptr[171], ptr[172], ptr[173], ptr[174],
							   ptr[175], ptr[176], ptr[177], ptr[178], ptr[179],
							   ptr[180], ptr[181], ptr[182], ptr[183], ptr[184],
							   ptr[185], ptr[186], ptr[187], ptr[188], ptr[189],
							   ptr[190], ptr[191], ptr[192], ptr[193], ptr[194],
							   ptr[195], ptr[196], ptr[197], ptr[198], ptr[199],
							   ptr[200], ptr[201], ptr[202], ptr[203], ptr[204],
							   ptr[205], ptr[206], ptr[207], ptr[208], ptr[209],
							   ptr[210], ptr[211], ptr[212], ptr[213], ptr[214],
							   ptr[215], ptr[216], ptr[217], ptr[218], ptr[219],
							   ptr[220], ptr[221], ptr[222], ptr[223], ptr[224],
							   ptr[225], ptr[226], ptr[227], ptr[228], ptr[229],
							   ptr[230], ptr[231], ptr[232], ptr[233], ptr[234],
							   ptr[235], ptr[236], ptr[237], ptr[238], ptr[239],
							   ptr[240], ptr[241], ptr[242], ptr[243], ptr[244],
							   ptr[245], ptr[246], ptr[247], ptr[248], ptr[249],
							   ptr[250], ptr[251], ptr[252], ptr[253], ptr[254],
							   ptr[255], ptr[256], ptr[257], ptr[258], ptr[259],
							   ptr[260], ptr[261], ptr[262], ptr[263], ptr[264],
							   ptr[265], ptr[266], ptr[267], ptr[268], ptr[269],
							   ptr[270], ptr[271], ptr[272], ptr[273], ptr[274],
							   ptr[275], ptr[276], ptr[277], ptr[278], ptr[279],
							   ptr[280], ptr[281], ptr[282], ptr[283], ptr[284],
							   ptr[285], ptr[286], ptr[287], ptr[288], ptr[289],
							   ptr[290], ptr[291], ptr[292], ptr[293], ptr[294],
							   ptr[295], ptr[296], ptr[297], ptr[298], ptr[299],
							   ptr[300], ptr[301], ptr[302], ptr[303], ptr[304],
							   ptr[305], ptr[306], ptr[307], ptr[308], ptr[309],
							   ptr[310], ptr[311], ptr[312], ptr[313], ptr[314],
							   ptr[315], ptr[316], ptr[317], ptr[318], ptr[319],
							   ptr[320], ptr[321], ptr[322], ptr[323], ptr[324],
							   ptr[325], ptr[326], ptr[327], ptr[328], ptr[329],
							   ptr[330], ptr[331], ptr[332], ptr[333], ptr[334],
							   ptr[335], ptr[336], ptr[337], ptr[338], ptr[339],
							   ptr[340], ptr[341], ptr[342], ptr[343], ptr[344],
							   ptr[345], ptr[346], ptr[347], ptr[348], ptr[349],
							   ptr[350], ptr[351], ptr[352], ptr[353], ptr[354],
							   ptr[355], ptr[356], ptr[357], ptr[358], ptr[359],
							   ptr[360], ptr[361], ptr[362], ptr[363], ptr[364],
							   ptr[365], ptr[366], ptr[367], ptr[368], ptr[369],
							   ptr[370], ptr[371], ptr[372], ptr[373], ptr[374],
							   ptr[375], ptr[376], ptr[377], ptr[378], ptr[379],
							   ptr[380], ptr[381], ptr[382], ptr[383], ptr[384],
							   ptr[385], ptr[386], ptr[387], ptr[388], ptr[389],
							   ptr[390], ptr[391], ptr[392], ptr[393], ptr[394],
							   ptr[395], ptr[396], ptr[397], ptr[398], ptr[399],
							   ptr[400], ptr[401], ptr[402], ptr[403], ptr[404],
							   ptr[405], ptr[406], ptr[407], ptr[408], ptr[409],
							   ptr[410], ptr[411], ptr[412], ptr[413], ptr[414],
							   ptr[415], ptr[416], ptr[417], ptr[418], ptr[419],
							   ptr[420], ptr[421], ptr[422], ptr[423], ptr[424],
							   ptr[425], ptr[426], ptr[427], ptr[428], ptr[429],
							   ptr[430], ptr[431], ptr[432], ptr[433], ptr[434],
							   ptr[435], ptr[436], ptr[437], ptr[438], ptr[439],
							   ptr[440], ptr[441], ptr[442], ptr[443], ptr[444],
							   ptr[445], ptr[446], ptr[447], ptr[448], ptr[449],
							   ptr[450], ptr[451], ptr[452], ptr[453], ptr[454],
							   ptr[455], ptr[456], ptr[457], ptr[458], ptr[459],
							   ptr[460], ptr[461], ptr[462], ptr[463], ptr[464],
							   ptr[465], ptr[466], ptr[467], ptr[468], ptr[469],
							   ptr[470], ptr[471], ptr[472], ptr[473], ptr[474],
							   ptr[475], ptr[476], ptr[477], ptr[478], ptr[479],
							   ptr[480], ptr[481], ptr[482], ptr[483], ptr[484],
							   ptr[485], ptr[486], ptr[487], ptr[488], ptr[489],
							   ptr[490], ptr[491], ptr[492], ptr[493], ptr[494],
							   ptr[495], ptr[496], ptr[497], ptr[498], ptr[499],
							   ptr[500], ptr[501], ptr[502], ptr[503], ptr[504],
							   ptr[505], ptr[506], ptr[507], ptr[508], ptr[509],
							   ptr[510], ptr[511], ptr[512], ptr[513], ptr[514],
							   ptr[515], ptr[516], ptr[517], ptr[518], ptr[519],
							   ptr[520], ptr[521], ptr[522], ptr[523], ptr[524],
							   ptr[525], ptr[526], ptr[527], ptr[528], ptr[529],
							   ptr[530], ptr[531], ptr[532], ptr[533], ptr[534],
							   ptr[535], ptr[536], ptr[537], ptr[538], ptr[539],
							   ptr[540], ptr[541], ptr[542], ptr[543], ptr[544],
							   ptr[545], ptr[546], ptr[547], ptr[548], ptr[549],
							   ptr[550], ptr[551], ptr[552], ptr[553], ptr[554],
							   ptr[555], ptr[556], ptr[557], ptr[558], ptr[559],
							   ptr[560], ptr[561], ptr[562], ptr[563], ptr[564],
							   ptr[565], ptr[566], ptr[567], ptr[568], ptr[569],
							   ptr[570], ptr[571], ptr[572], ptr[573], ptr[574],
							   ptr[575], ptr[576], ptr[577], ptr[578], ptr[579],
							   ptr[580], ptr[581], ptr[582], ptr[583], ptr[584],
							   ptr[585], ptr[586], ptr[587], ptr[588], ptr[589],
							   ptr[590], ptr[591], ptr[592], ptr[593], ptr[594],
							   ptr[595], ptr[596], ptr[597], ptr[598], ptr[599],
							   ptr[600], ptr[601], ptr[602], ptr[603], ptr[604],
							   ptr[605], ptr[606], ptr[607], ptr[608], ptr[609],
							   ptr[610], ptr[611], ptr[612], ptr[613], ptr[614],
							   ptr[615], ptr[616], ptr[617], ptr[618], ptr[619],
							   ptr[620], ptr[621], ptr[622], ptr[623], ptr[624],
							   ptr[625], ptr[626], ptr[627], ptr[628], ptr[629],
							   ptr[630], ptr[631], ptr[632], ptr[633], ptr[634],
							   ptr[635], ptr[636], ptr[637], ptr[638], ptr[639],
							   ptr[640], ptr[641], ptr[642], ptr[643], ptr[644],
							   ptr[645], ptr[646], ptr[647], ptr[648], ptr[649],
							   ptr[650], ptr[651], ptr[652], ptr[653], ptr[654],
							   ptr[655], ptr[656], ptr[657], ptr[658], ptr[659],
							   ptr[660], ptr[661], ptr[662], ptr[663], ptr[664],
							   ptr[665], ptr[666], ptr[667], ptr[668], ptr[669],
							   ptr[670], ptr[671], ptr[672], ptr[673], ptr[674],
							   ptr[675], ptr[676], ptr[677], ptr[678], ptr[679],
							   ptr[680], ptr[681], ptr[682], ptr[683], ptr[684],
							   ptr[685], ptr[686], ptr[687], ptr[688], ptr[689],
							   ptr[690], ptr[691], ptr[692], ptr[693], ptr[694],
							   ptr[695], ptr[696], ptr[697], ptr[698], ptr[699],
							   ptr[700], ptr[701], ptr[702], ptr[703], ptr[704],
							   ptr[705], ptr[706], ptr[707], ptr[708], ptr[709],
							   ptr[710], ptr[711], ptr[712], ptr[713], ptr[714],
							   ptr[715], ptr[716], ptr[717], ptr[718], ptr[719],
							   ptr[720], ptr[721], ptr[722], ptr[723], ptr[724],
							   ptr[725], ptr[726], ptr[727], ptr[728], ptr[729],
							   ptr[730], ptr[731], ptr[732], ptr[733], ptr[734],
							   ptr[735], ptr[736], ptr[737], ptr[738], ptr[739],
							   ptr[740], ptr[741], ptr[742], ptr[743], ptr[744],
							   ptr[745], ptr[746], ptr[747], ptr[748], ptr[749],
							   ptr[750], ptr[751], ptr[752], ptr[753], ptr[754],
							   ptr[755], ptr[756], ptr[757], ptr[758], ptr[759],
							   ptr[760], ptr[761], ptr[762], ptr[763], ptr[764],
							   ptr[765], ptr[766], ptr[767], ptr[768], ptr[769],
							   ptr[770], ptr[771], ptr[772], ptr[773], ptr[774],
							   ptr[775], ptr[776], ptr[777], ptr[778], ptr[779],
							   ptr[780], ptr[781], ptr[782], ptr[783], ptr[784],
							   ptr[785], ptr[786], ptr[787], ptr[788], ptr[789],
							   ptr[790], ptr[791], ptr[792], ptr[793], ptr[794],
							   ptr[795], ptr[796], ptr[797], ptr[798], ptr[799],
							   ptr[800], ptr[801], ptr[802], ptr[803], ptr[804],
							   ptr[805], ptr[806], ptr[807], ptr[808], ptr[809],
							   ptr[810], ptr[811], ptr[812], ptr[813], ptr[814],
							   ptr[815], ptr[816], ptr[817], ptr[818], ptr[819],
							   ptr[820], ptr[821], ptr[822], ptr[823], ptr[824],
							   ptr[825], ptr[826], ptr[827], ptr[828], ptr[829],
							   ptr[830], ptr[831], ptr[832], ptr[833], ptr[834],
							   ptr[835], ptr[836], ptr[837], ptr[838], ptr[839],
							   ptr[840], ptr[841], ptr[842], ptr[843], ptr[844],
							   ptr[845], ptr[846], ptr[847], ptr[848], ptr[849],
							   ptr[850], ptr[851], ptr[852], ptr[853], ptr[854],
							   ptr[855], ptr[856], ptr[857], ptr[858], ptr[859],
							   ptr[860], ptr[861], ptr[862], ptr[863], ptr[864],
							   ptr[865], ptr[866], ptr[867], ptr[868], ptr[869],
							   ptr[870], ptr[871], ptr[872], ptr[873], ptr[874],
							   ptr[875], ptr[876], ptr[877], ptr[878], ptr[879],
							   ptr[880], ptr[881], ptr[882], ptr[883], ptr[884],
							   ptr[885], ptr[886], ptr[887], ptr[888], ptr[889],
							   ptr[890], ptr[891], ptr[892], ptr[893], ptr[894],
							   ptr[895], ptr[896], ptr[897], ptr[898], ptr[899],
							   ptr[900], ptr[901], ptr[902], ptr[903], ptr[904],
							   ptr[905], ptr[906], ptr[907], ptr[908], ptr[909],
							   ptr[910], ptr[911], ptr[912], ptr[913], ptr[914],
							   ptr[915], ptr[916], ptr[917], ptr[918], ptr[919],
							   ptr[920], ptr[921], ptr[922], ptr[923], ptr[924],
							   ptr[925], ptr[926], ptr[927], ptr[928], ptr[929],
							   ptr[930], ptr[931], ptr[932], ptr[933], ptr[934],
							   ptr[935], ptr[936], ptr[937], ptr[938], ptr[939],
							   ptr[940], ptr[941], ptr[942], ptr[943], ptr[944],
							   ptr[945], ptr[946], ptr[947], ptr[948], ptr[949],
							   ptr[950], ptr[951], ptr[952], ptr[953], ptr[954],
							   ptr[955], ptr[956], ptr[957], ptr[958], ptr[959],
							   ptr[960], ptr[961], ptr[962], ptr[963], ptr[964],
							   ptr[965], ptr[966], ptr[967], ptr[968], ptr[969],
							   ptr[970], ptr[971], ptr[972], ptr[973], ptr[974],
							   ptr[975], ptr[976], ptr[977], ptr[978], ptr[979],
							   ptr[980], ptr[981], ptr[982], ptr[983], ptr[984],
							   ptr[985], ptr[986], ptr[987], ptr[988], ptr[989],
							   ptr[990], ptr[991], ptr[992], ptr[993], ptr[994],
							   ptr[995], ptr[996], ptr[997], ptr[998], ptr[999],
							   ptr[1000], ptr[1001], ptr[1002], ptr[1003], ptr[1004],
							   ptr[1005], ptr[1006], ptr[1007], ptr[1008], ptr[1009],
							   ptr[1010], ptr[1011], ptr[1012], ptr[1013], ptr[1014],
							   ptr[1015], ptr[1016], ptr[1017], ptr[1018], ptr[1019],
							   ptr[1020], ptr[1021], ptr[1022], ptr[1023], ptr[1024],
							   ptr[1025], ptr[1026], ptr[1027], ptr[1028], ptr[1029],
							   ptr[1030], ptr[1031], ptr[1032], ptr[1033], ptr[1034],
							   ptr[1035], ptr[1036], ptr[1037], ptr[1038], ptr[1039],
							   ptr[1040], ptr[1041], ptr[1042], ptr[1043], ptr[1044],
							   ptr[1045], ptr[1046], ptr[1047], ptr[1048], ptr[1049],
							   ptr[1050], ptr[1051], ptr[1052], ptr[1053], ptr[1054],
							   ptr[1055], ptr[1056], ptr[1057], ptr[1058], ptr[1059],
							   ptr[1060], ptr[1061], ptr[1062], ptr[1063], ptr[1064],
							   ptr[1065], ptr[1066], ptr[1067], ptr[1068], ptr[1069],
							   ptr[1070], ptr[1071], ptr[1072], ptr[1073], ptr[1074],
							   ptr[1075], ptr[1076], ptr[1077], ptr[1078], ptr[1079],
							   ptr[1080], ptr[1081], ptr[1082], ptr[1083], ptr[1084],
							   ptr[1085], ptr[1086], ptr[1087], ptr[1088], ptr[1089],
							   ptr[1090], ptr[1091], ptr[1092], ptr[1093], ptr[1094],
							   ptr[1095], ptr[1096], ptr[1097], ptr[1098], ptr[1099],
							   ptr[1100], ptr[1101], ptr[1102], ptr[1103], ptr[1104],
							   ptr[1105], ptr[1106], ptr[1107], ptr[1108], ptr[1109],
							   ptr[1110], ptr[1111], ptr[1112], ptr[1113], ptr[1114],
							   ptr[1115], ptr[1116], ptr[1117], ptr[1118], ptr[1119],
							   ptr[1120], ptr[1121], ptr[1122], ptr[1123], ptr[1124],
							   ptr[1125], ptr[1126], ptr[1127], ptr[1128], ptr[1129],
							   ptr[1130], ptr[1131], ptr[1132], ptr[1133], ptr[1134],
							   ptr[1135], ptr[1136], ptr[1137], ptr[1138], ptr[1139],
							   ptr[1140], ptr[1141], ptr[1142], ptr[1143], ptr[1144],
							   ptr[1145], ptr[1146], ptr[1147], ptr[1148], ptr[1149],
							   ptr[1150], ptr[1151], ptr[1152], ptr[1153], ptr[1154],
							   ptr[1155], ptr[1156], ptr[1157], ptr[1158], ptr[1159],
							   ptr[1160], ptr[1161], ptr[1162], ptr[1163], ptr[1164],
							   ptr[1165], ptr[1166], ptr[1167], ptr[1168], ptr[1169],
							   ptr[1170], ptr[1171], ptr[1172], ptr[1173], ptr[1174],
							   ptr[1175], ptr[1176], ptr[1177], ptr[1178], ptr[1179],
							   ptr[1180], ptr[1181], ptr[1182], ptr[1183], ptr[1184],
							   ptr[1185], ptr[1186], ptr[1187], ptr[1188], ptr[1189],
							   ptr[1190], ptr[1191], ptr[1192], ptr[1193], ptr[1194],
							   ptr[1195], ptr[1196], ptr[1197], ptr[1198], ptr[1199],
							   ptr[1200], ptr[1201], ptr[1202], ptr[1203], ptr[1204],
							   ptr[1205], ptr[1206], ptr[1207], ptr[1208], ptr[1209],
							   ptr[1210], ptr[1211], ptr[1212], ptr[1213], ptr[1214],
							   ptr[1215], ptr[1216], ptr[1217], ptr[1218], ptr[1219],
							   ptr[1220], ptr[1221], ptr[1222], ptr[1223], ptr[1224],
							   ptr[1225], ptr[1226], ptr[1227], ptr[1228], ptr[1229],
							   ptr[1230], ptr[1231], ptr[1232], ptr[1233], ptr[1234],
							   ptr[1235], ptr[1236], ptr[1237], ptr[1238], ptr[1239],
							   ptr[1240], ptr[1241], ptr[1242], ptr[1243], ptr[1244],
							   ptr[1245], ptr[1246], ptr[1247], ptr[1248], ptr[1249],
							   ptr[1250], ptr[1251], ptr[1252], ptr[1253], ptr[1254],
							   ptr[1255], ptr[1256], ptr[1257], ptr[1258], ptr[1259],
							   ptr[1260], ptr[1261], ptr[1262], ptr[1263], ptr[1264],
							   ptr[1265], ptr[1266], ptr[1267], ptr[1268], ptr[1269],
							   ptr[1270], ptr[1271], ptr[1272], ptr[1273], ptr[1274],
							   ptr[1275], ptr[1276], ptr[1277], ptr[1278], ptr[1279],
							   ptr[1280], ptr[1281], ptr[1282], ptr[1283], ptr[1284],
							   ptr[1285], ptr[1286], ptr[1287], ptr[1288], ptr[1289],
							   ptr[1290], ptr[1291], ptr[1292], ptr[1293], ptr[1294],
							   ptr[1295], ptr[1296], ptr[1297], ptr[1298], ptr[1299],
							   ptr[1300], ptr[1301], ptr[1302], ptr[1303], ptr[1304],
							   ptr[1305], ptr[1306], ptr[1307], ptr[1308], ptr[1309],
							   ptr[1310], ptr[1311], ptr[1312], ptr[1313], ptr[1314],
							   ptr[1315], ptr[1316], ptr[1317], ptr[1318], ptr[1319],
							   ptr[1320], ptr[1321], ptr[1322], ptr[1323], ptr[1324],
							   ptr[1325], ptr[1326], ptr[1327], ptr[1328], ptr[1329],
							   ptr[1330], ptr[1331], ptr[1332], ptr[1333], ptr[1334],
							   ptr[1335], ptr[1336], ptr[1337], ptr[1338], ptr[1339],
							   ptr[1340], ptr[1341], ptr[1342], ptr[1343], ptr[1344],
							   ptr[1345], ptr[1346], ptr[1347], ptr[1348], ptr[1349],
							   ptr[1350], ptr[1351], ptr[1352], ptr[1353], ptr[1354],
							   ptr[1355], ptr[1356], ptr[1357], ptr[1358], ptr[1359],
							   ptr[1360], ptr[1361], ptr[1362], ptr[1363], ptr[1364],
							   ptr[1365], ptr[1366], ptr[1367], ptr[1368], ptr[1369],
							   ptr[1370], ptr[1371], ptr[1372], ptr[1373], ptr[1374],
							   ptr[1375], ptr[1376], ptr[1377], ptr[1378], ptr[1379],
							   ptr[1380], ptr[1381], ptr[1382], ptr[1383], ptr[1384],
							   ptr[1385], ptr[1386], ptr[1387], ptr[1388], ptr[1389],
							   ptr[1390], ptr[1391], ptr[1392], ptr[1393], ptr[1394],
							   ptr[1395], ptr[1396], ptr[1397], ptr[1398], ptr[1399],
							   ptr[1400], ptr[1401], ptr[1402], ptr[1403], ptr[1404],
							   ptr[1405], ptr[1406], ptr[1407], ptr[1408], ptr[1409],
							   ptr[1410], ptr[1411], ptr[1412], ptr[1413], ptr[1414],
							   ptr[1415], ptr[1416], ptr[1417], ptr[1418], ptr[1419],
							   ptr[1420], ptr[1421], ptr[1422], ptr[1423], ptr[1424],
							   ptr[1425], ptr[1426], ptr[1427], ptr[1428], ptr[1429],
							   ptr[1430], ptr[1431], ptr[1432], ptr[1433], ptr[1434],
							   ptr[1435], ptr[1436], ptr[1437], ptr[1438], ptr[1439],
							   ptr[1440], ptr[1441], ptr[1442], ptr[1443], ptr[1444],
							   ptr[1445], ptr[1446], ptr[1447], ptr[1448], ptr[1449],
							   ptr[1450], ptr[1451], ptr[1452], ptr[1453], ptr[1454],
							   ptr[1455], ptr[1456], ptr[1457], ptr[1458], ptr[1459],
							   ptr[1460], ptr[1461], ptr[1462], ptr[1463], ptr[1464],
							   ptr[1465], ptr[1466], ptr[1467], ptr[1468], ptr[1469],
							   ptr[1470], ptr[1471], ptr[1472], ptr[1473], ptr[1474],
							   ptr[1475], ptr[1476], ptr[1477], ptr[1478], ptr[1479],
							   ptr[1480], ptr[1481], ptr[1482], ptr[1483], ptr[1484],
							   ptr[1485], ptr[1486], ptr[1487], ptr[1488], ptr[1489],
							   ptr[1490], ptr[1491], ptr[1492], ptr[1493], ptr[1494],
							   ptr[1495], ptr[1496], ptr[1497], ptr[1498], ptr[1499],
							   ptr[1500], ptr[1501], ptr[1502], ptr[1503], ptr[1504],
							   ptr[1505], ptr[1506], ptr[1507], ptr[1508], ptr[1509],
							   ptr[1510], ptr[1511], ptr[1512], ptr[1513], ptr[1514],
							   ptr[1515], ptr[1516], ptr[1517], ptr[1518], ptr[1519],
							   ptr[1520], ptr[1521], ptr[1522], ptr[1523], ptr[1524],
							   ptr[1525], ptr[1526], ptr[1527], ptr[1528], ptr[1529],
							   ptr[1530], ptr[1531], ptr[1532], ptr[1533], ptr[1534],
							   ptr[1535], ptr[1536], ptr[1537], ptr[1538], ptr[1539],
							   ptr[1540], ptr[1541], ptr[1542], ptr[1543], ptr[1544],
							   ptr[1545], ptr[1546], ptr[1547], ptr[1548], ptr[1549],
							   ptr[1550], ptr[1551], ptr[1552], ptr[1553], ptr[1554],
							   ptr[1555], ptr[1556], ptr[1557], ptr[1558], ptr[1559],
							   ptr[1560], ptr[1561], ptr[1562], ptr[1563], ptr[1564],
							   ptr[1565], ptr[1566], ptr[1567], ptr[1568], ptr[1569],
							   ptr[1570], ptr[1571], ptr[1572], ptr[1573], ptr[1574],
							   ptr[1575], ptr[1576], ptr[1577], ptr[1578], ptr[1579],
							   ptr[1580], ptr[1581], ptr[1582], ptr[1583], ptr[1584],
							   ptr[1585], ptr[1586], ptr[1587], ptr[1588], ptr[1589],
							   ptr[1590], ptr[1591], ptr[1592], ptr[1593], ptr[1594],
							   ptr[1595], ptr[1596], ptr[1597], ptr[1598], ptr[1599],
							   ptr[1600], ptr[1601], ptr[1602], ptr[1603], ptr[1604],
							   ptr[1605], ptr[1606], ptr[1607], ptr[1608], ptr[1609],
							   ptr[1610], ptr[1611], ptr[1612], ptr[1613], ptr[1614],
							   ptr[1615], ptr[1616], ptr[1617], ptr[1618], ptr[1619],
							   ptr[1620], ptr[1621], ptr[1622], ptr[1623], ptr[1624],
							   ptr[1625], ptr[1626], ptr[1627], ptr[1628], ptr[1629],
							   ptr[1630], ptr[1631], ptr[1632], ptr[1633], ptr[1634],
							   ptr[1635], ptr[1636], ptr[1637], ptr[1638], ptr[1639],
							   ptr[1640], ptr[1641], ptr[1642], ptr[1643], ptr[1644],
							   ptr[1645], ptr[1646], ptr[1647], ptr[1648], ptr[1649],
							   ptr[1650], ptr[1651], ptr[1652], ptr[1653], ptr[1654],
							   ptr[1655], ptr[1656], ptr[1657], ptr[1658], ptr[1659],
							   ptr[1660], ptr[1661], ptr[1662], ptr[1663], ptr[1664],
							   ptr[1665], ptr[1666], ptr[1667], ptr[1668], ptr[1669],
							   ptr[1670], ptr[1671], ptr[1672], ptr[1673], ptr[1674],
							   ptr[1675], ptr[1676], ptr[1677], ptr[1678], ptr[1679],
							   ptr[1680], ptr[1681], ptr[1682], ptr[1683], ptr[1684],
							   ptr[1685], ptr[1686], ptr[1687], ptr[1688], ptr[1689],
							   ptr[1690], ptr[1691], ptr[1692], ptr[1693], ptr[1694],
							   ptr[1695], ptr[1696], ptr[1697], ptr[1698], ptr[1699],
							   ptr[1700], ptr[1701], ptr[1702], ptr[1703], ptr[1704],
							   ptr[1705], ptr[1706], ptr[1707], ptr[1708], ptr[1709],
							   ptr[1710], ptr[1711], ptr[1712], ptr[1713], ptr[1714],
							   ptr[1715], ptr[1716], ptr[1717], ptr[1718], ptr[1719],
							   ptr[1720], ptr[1721], ptr[1722], ptr[1723], ptr[1724],
							   ptr[1725], ptr[1726], ptr[1727], ptr[1728], ptr[1729],
							   ptr[1730], ptr[1731], ptr[1732], ptr[1733], ptr[1734],
							   ptr[1735], ptr[1736], ptr[1737], ptr[1738], ptr[1739],
							   ptr[1740], ptr[1741], ptr[1742], ptr[1743], ptr[1744],
							   ptr[1745], ptr[1746], ptr[1747], ptr[1748], ptr[1749],
							   ptr[1750], ptr[1751], ptr[1752], ptr[1753], ptr[1754],
							   ptr[1755], ptr[1756], ptr[1757], ptr[1758], ptr[1759],
							   ptr[1760], ptr[1761], ptr[1762], ptr[1763], ptr[1764],
							   ptr[1765], ptr[1766], ptr[1767], ptr[1768], ptr[1769],
							   ptr[1770], ptr[1771], ptr[1772], ptr[1773], ptr[1774],
							   ptr[1775], ptr[1776], ptr[1777], ptr[1778], ptr[1779],
							   ptr[1780], ptr[1781], ptr[1782], ptr[1783], ptr[1784],
							   ptr[1785], ptr[1786], ptr[1787], ptr[1788], ptr[1789],
							   ptr[1790], ptr[1791], ptr[1792], ptr[1793], ptr[1794],
							   ptr[1795], ptr[1796], ptr[1797], ptr[1798], ptr[1799],
							   ptr[1800], ptr[1801], ptr[1802], ptr[1803], ptr[1804],
							   ptr[1805], ptr[1806], ptr[1807], ptr[1808], ptr[1809],
							   ptr[1810], ptr[1811], ptr[1812], ptr[1813], ptr[1814],
							   ptr[1815], ptr[1816], ptr[1817], ptr[1818], ptr[1819],
							   ptr[1820], ptr[1821], ptr[1822], ptr[1823], ptr[1824],
							   ptr[1825], ptr[1826], ptr[1827], ptr[1828], ptr[1829],
							   ptr[1830], ptr[1831], ptr[1832], ptr[1833], ptr[1834],
							   ptr[1835], ptr[1836], ptr[1837], ptr[1838], ptr[1839],
							   ptr[1840], ptr[1841], ptr[1842], ptr[1843], ptr[1844],
							   ptr[1845], ptr[1846], ptr[1847], ptr[1848], ptr[1849],
							   ptr[1850], ptr[1851], ptr[1852], ptr[1853], ptr[1854],
							   ptr[1855], ptr[1856], ptr[1857], ptr[1858], ptr[1859],
							   ptr[1860], ptr[1861], ptr[1862], ptr[1863], ptr[1864],
							   ptr[1865], ptr[1866], ptr[1867], ptr[1868], ptr[1869],
							   ptr[1870], ptr[1871], ptr[1872], ptr[1873], ptr[1874],
							   ptr[1875], ptr[1876], ptr[1877], ptr[1878], ptr[1879],
							   ptr[1880], ptr[1881], ptr[1882], ptr[1883], ptr[1884],
							   ptr[1885], ptr[1886], ptr[1887], ptr[1888], ptr[1889],
							   ptr[1890], ptr[1891], ptr[1892], ptr[1893], ptr[1894],
							   ptr[1895], ptr[1896], ptr[1897], ptr[1898], ptr[1899],
							   ptr[1900], ptr[1901], ptr[1902], ptr[1903], ptr[1904],
							   ptr[1905], ptr[1906], ptr[1907], ptr[1908], ptr[1909],
							   ptr[1910], ptr[1911], ptr[1912], ptr[1913], ptr[1914],
							   ptr[1915], ptr[1916], ptr[1917], ptr[1918], ptr[1919],
							   ptr[1920], ptr[1921], ptr[1922], ptr[1923], ptr[1924],
							   ptr[1925], ptr[1926], ptr[1927], ptr[1928], ptr[1929],
							   ptr[1930], ptr[1931], ptr[1932], ptr[1933], ptr[1934],
							   ptr[1935], ptr[1936], ptr[1937], ptr[1938], ptr[1939],
							   ptr[1940], ptr[1941], ptr[1942], ptr[1943], ptr[1944],
							   ptr[1945], ptr[1946], ptr[1947], ptr[1948], ptr[1949],
							   ptr[1950], ptr[1951], ptr[1952], ptr[1953], ptr[1954],
							   ptr[1955], ptr[1956], ptr[1957], ptr[1958], ptr[1959],
							   ptr[1960], ptr[1961], ptr[1962], ptr[1963], ptr[1964],
							   ptr[1965], ptr[1966], ptr[1967], ptr[1968], ptr[1969],
							   ptr[1970], ptr[1971], ptr[1972], ptr[1973], ptr[1974],
							   ptr[1975], ptr[1976], ptr[1977], ptr[1978], ptr[1979],
							   ptr[1980], ptr[1981], ptr[1982], ptr[1983], ptr[1984],
							   ptr[1985], ptr[1986], ptr[1987], ptr[1988], ptr[1989],
							   ptr[1990], ptr[1991], ptr[1992], ptr[1993], ptr[1994],
							   ptr[1995], ptr[1996], ptr[1997], ptr[1998], ptr[1999],
							   ptr[2000], ptr[2001], ptr[2002], ptr[2003], ptr[2004],
							   ptr[2005], ptr[2006], ptr[2007], ptr[2008], ptr[2009],
							   ptr[2010], ptr[2011], ptr[2012], ptr[2013], ptr[2014],
							   ptr[2015], ptr[2016], ptr[2017], ptr[2018], ptr[2019],
							   ptr[2020], ptr[2021], ptr[2022], ptr[2023], ptr[2024],
							   ptr[2025], ptr[2026], ptr[2027], ptr[2028], ptr[2029],
							   ptr[2030], ptr[2031], ptr[2032], ptr[2033], ptr[2034],
							   ptr[2035], ptr[2036], ptr[2037], ptr[2038], ptr[2039],
							   ptr[2040], ptr[2041], ptr[2042], ptr[2043], ptr[2044],
							   ptr[2045], ptr[2046], ptr[2047], ptr[2048], ptr[2049],
							   ptr[2050], ptr[2051], ptr[2052], ptr[2053], ptr[2054],
							   ptr[2055], ptr[2056], ptr[2057], ptr[2058], ptr[2059],
							   ptr[2060], ptr[2061], ptr[2062], ptr[2063], ptr[2064],
							   ptr[2065], ptr[2066], ptr[2067], ptr[2068], ptr[2069],
							   ptr[2070], ptr[2071], ptr[2072], ptr[2073], ptr[2074],
							   ptr[2075], ptr[2076], ptr[2077], ptr[2078], ptr[2079],
							   ptr[2080], ptr[2081], ptr[2082], ptr[2083], ptr[2084],
							   ptr[2085], ptr[2086], ptr[2087], ptr[2088], ptr[2089],
							   ptr[2090], ptr[2091], ptr[2092], ptr[2093], ptr[2094],
							   ptr[2095], ptr[2096], ptr[2097], ptr[2098], ptr[2099],
							   ptr[2100], ptr[2101], ptr[2102], ptr[2103], ptr[2104],
							   ptr[2105], ptr[2106], ptr[2107], ptr[2108], ptr[2109],
							   ptr[2110], ptr[2111], ptr[2112], ptr[2113], ptr[2114],
							   ptr[2115], ptr[2116], ptr[2117], ptr[2118], ptr[2119],
							   ptr[2120], ptr[2121], ptr[2122], ptr[2123], ptr[2124],
							   ptr[2125], ptr[2126], ptr[2127], ptr[2128], ptr[2129],
							   ptr[2130], ptr[2131], ptr[2132], ptr[2133], ptr[2134],
							   ptr[2135], ptr[2136], ptr[2137], ptr[2138], ptr[2139],
							   ptr[2140], ptr[2141], ptr[2142], ptr[2143], ptr[2144],
							   ptr[2145], ptr[2146], ptr[2147], ptr[2148], ptr[2149],
							   ptr[2150], ptr[2151], ptr[2152], ptr[2153], ptr[2154],
							   ptr[2155], ptr[2156], ptr[2157], ptr[2158], ptr[2159],
							   ptr[2160], ptr[2161], ptr[2162], ptr[2163], ptr[2164],
							   ptr[2165], ptr[2166], ptr[2167], ptr[2168], ptr[2169],
							   ptr[2170], ptr[2171], ptr[2172], ptr[2173], ptr[2174],
							   ptr[2175], ptr[2176], ptr[2177], ptr[2178], ptr[2179],
							   ptr[2180], ptr[2181], ptr[2182], ptr[2183], ptr[2184],
							   ptr[2185], ptr[2186], ptr[2187], ptr[2188], ptr[2189],
							   ptr[2190], ptr[2191], ptr[2192], ptr[2193], ptr[2194],
							   ptr[2195], ptr[2196], ptr[2197], ptr[2198], ptr[2199],
							   ptr[2200], ptr[2201], ptr[2202], ptr[2203], ptr[2204],
							   ptr[2205], ptr[2206], ptr[2207], ptr[2208], ptr[2209],
							   ptr[2210], ptr[2211], ptr[2212], ptr[2213], ptr[2214],
							   ptr[2215], ptr[2216], ptr[2217], ptr[2218], ptr[2219],
							   ptr[2220], ptr[2221], ptr[2222], ptr[2223], ptr[2224],
							   ptr[2225], ptr[2226], ptr[2227], ptr[2228], ptr[2229],
							   ptr[2230], ptr[2231], ptr[2232], ptr[2233], ptr[2234],
							   ptr[2235], ptr[2236], ptr[2237], ptr[2238], ptr[2239],
							   ptr[2240], ptr[2241], ptr[2242], ptr[2243], ptr[2244],
							   ptr[2245], ptr[2246], ptr[2247], ptr[2248], ptr[2249],
							   ptr[2250], ptr[2251], ptr[2252], ptr[2253], ptr[2254],
							   ptr[2255], ptr[2256], ptr[2257], ptr[2258], ptr[2259],
							   ptr[2260], ptr[2261], ptr[2262], ptr[2263], ptr[2264],
							   ptr[2265], ptr[2266], ptr[2267], ptr[2268], ptr[2269],
							   ptr[2270], ptr[2271], ptr[2272], ptr[2273], ptr[2274],
							   ptr[2275], ptr[2276], ptr[2277], ptr[2278], ptr[2279],
							   ptr[2280], ptr[2281], ptr[2282], ptr[2283], ptr[2284],
							   ptr[2285], ptr[2286], ptr[2287], ptr[2288], ptr[2289],
							   ptr[2290], ptr[2291], ptr[2292], ptr[2293], ptr[2294],
							   ptr[2295], ptr[2296], ptr[2297], ptr[2298], ptr[2299],
							   ptr[2300], ptr[2301], ptr[2302], ptr[2303], ptr[2304],
							   ptr[2305], ptr[2306], ptr[2307], ptr[2308], ptr[2309],
							   ptr[2310], ptr[2311], ptr[2312], ptr[2313], ptr[2314],
							   ptr[2315], ptr[2316], ptr[2317], ptr[2318], ptr[2319],
							   ptr[2320], ptr[2321], ptr[2322], ptr[2323], ptr[2324],
							   ptr[2325], ptr[2326], ptr[2327], ptr[2328], ptr[2329],
							   ptr[2330], ptr[2331], ptr[2332], ptr[2333], ptr[2334],
							   ptr[2335], ptr[2336], ptr[2337], ptr[2338], ptr[2339],
							   ptr[2340], ptr[2341], ptr[2342], ptr[2343], ptr[2344],
							   ptr[2345], ptr[2346], ptr[2347], ptr[2348], ptr[2349],
							   ptr[2350], ptr[2351], ptr[2352], ptr[2353], ptr[2354],
							   ptr[2355], ptr[2356], ptr[2357], ptr[2358], ptr[2359],
							   ptr[2360], ptr[2361], ptr[2362], ptr[2363], ptr[2364],
							   ptr[2365], ptr[2366], ptr[2367], ptr[2368], ptr[2369],
							   ptr[2370], ptr[2371], ptr[2372], ptr[2373], ptr[2374],
							   ptr[2375], ptr[2376], ptr[2377], ptr[2378], ptr[2379],
							   ptr[2380], ptr[2381], ptr[2382], ptr[2383], ptr[2384],
							   ptr[2385], ptr[2386], ptr[2387], ptr[2388], ptr[2389],
							   ptr[2390], ptr[2391], ptr[2392], ptr[2393], ptr[2394],
							   ptr[2395], ptr[2396], ptr[2397], ptr[2398], ptr[2399],
							   ptr[2400], ptr[2401], ptr[2402], ptr[2403], ptr[2404],
							   ptr[2405], ptr[2406], ptr[2407], ptr[2408], ptr[2409],
							   ptr[2410], ptr[2411], ptr[2412], ptr[2413], ptr[2414],
							   ptr[2415], ptr[2416], ptr[2417], ptr[2418], ptr[2419],
							   ptr[2420], ptr[2421], ptr[2422], ptr[2423], ptr[2424],
							   ptr[2425], ptr[2426], ptr[2427], ptr[2428], ptr[2429],
							   ptr[2430], ptr[2431], ptr[2432], ptr[2433], ptr[2434],
							   ptr[2435], ptr[2436], ptr[2437], ptr[2438], ptr[2439],
							   ptr[2440], ptr[2441], ptr[2442], ptr[2443], ptr[2444],
							   ptr[2445], ptr[2446], ptr[2447], ptr[2448], ptr[2449],
							   ptr[2450], ptr[2451], ptr[2452], ptr[2453], ptr[2454],
							   ptr[2455], ptr[2456], ptr[2457], ptr[2458], ptr[2459],
							   ptr[2460], ptr[2461], ptr[2462], ptr[2463], ptr[2464],
							   ptr[2465], ptr[2466], ptr[2467], ptr[2468], ptr[2469],
							   ptr[2470], ptr[2471], ptr[2472], ptr[2473], ptr[2474],
							   ptr[2475], ptr[2476], ptr[2477], ptr[2478], ptr[2479],
							   ptr[2480], ptr[2481], ptr[2482], ptr[2483], ptr[2484],
							   ptr[2485], ptr[2486], ptr[2487], ptr[2488], ptr[2489],
							   ptr[2490], ptr[2491], ptr[2492], ptr[2493], ptr[2494],
							   ptr[2495], ptr[2496], ptr[2497], ptr[2498], ptr[2499],
							   ptr[2500], ptr[2501], ptr[2502], ptr[2503], ptr[2504],
							   ptr[2505], ptr[2506], ptr[2507], ptr[2508], ptr[2509],
							   ptr[2510], ptr[2511], ptr[2512], ptr[2513], ptr[2514],
							   ptr[2515], ptr[2516], ptr[2517], ptr[2518], ptr[2519],
							   ptr[2520], ptr[2521], ptr[2522], ptr[2523], ptr[2524],
							   ptr[2525], ptr[2526], ptr[2527], ptr[2528], ptr[2529],
							   ptr[2530], ptr[2531], ptr[2532], ptr[2533], ptr[2534],
							   ptr[2535], ptr[2536], ptr[2537], ptr[2538], ptr[2539],
							   ptr[2540], ptr[2541], ptr[2542], ptr[2543], ptr[2544],
							   ptr[2545], ptr[2546], ptr[2547], ptr[2548], ptr[2549],
							   ptr[2550], ptr[2551], ptr[2552], ptr[2553], ptr[2554],
							   ptr[2555], ptr[2556], ptr[2557], ptr[2558], ptr[2559],
							   ptr[2560], ptr[2561], ptr[2562], ptr[2563], ptr[2564],
							   ptr[2565], ptr[2566], ptr[2567], ptr[2568], ptr[2569],
							   ptr[2570], ptr[2571], ptr[2572], ptr[2573], ptr[2574],
							   ptr[2575], ptr[2576], ptr[2577], ptr[2578], ptr[2579],
							   ptr[2580], ptr[2581], ptr[2582], ptr[2583], ptr[2584],
							   ptr[2585], ptr[2586], ptr[2587], ptr[2588], ptr[2589],
							   ptr[2590], ptr[2591], ptr[2592], ptr[2593], ptr[2594],
							   ptr[2595], ptr[2596], ptr[2597], ptr[2598], ptr[2599],
							   ptr[2600], ptr[2601], ptr[2602], ptr[2603], ptr[2604],
							   ptr[2605], ptr[2606], ptr[2607], ptr[2608], ptr[2609],
							   ptr[2610], ptr[2611], ptr[2612], ptr[2613], ptr[2614],
							   ptr[2615], ptr[2616], ptr[2617], ptr[2618], ptr[2619],
							   ptr[2620], ptr[2621], ptr[2622], ptr[2623], ptr[2624],
							   ptr[2625], ptr[2626], ptr[2627], ptr[2628], ptr[2629],
							   ptr[2630], ptr[2631], ptr[2632], ptr[2633], ptr[2634],
							   ptr[2635], ptr[2636], ptr[2637], ptr[2638], ptr[2639],
							   ptr[2640], ptr[2641], ptr[2642], ptr[2643], ptr[2644],
							   ptr[2645], ptr[2646], ptr[2647], ptr[2648], ptr[2649],
							   ptr[2650], ptr[2651], ptr[2652], ptr[2653], ptr[2654],
							   ptr[2655], ptr[2656], ptr[2657], ptr[2658], ptr[2659],
							   ptr[2660], ptr[2661], ptr[2662], ptr[2663], ptr[2664],
							   ptr[2665], ptr[2666], ptr[2667], ptr[2668], ptr[2669],
							   ptr[2670], ptr[2671], ptr[2672], ptr[2673], ptr[2674],
							   ptr[2675], ptr[2676], ptr[2677], ptr[2678], ptr[2679],
							   ptr[2680], ptr[2681], ptr[2682], ptr[2683], ptr[2684],
							   ptr[2685], ptr[2686], ptr[2687], ptr[2688], ptr[2689],
							   ptr[2690], ptr[2691], ptr[2692], ptr[2693], ptr[2694],
							   ptr[2695], ptr[2696], ptr[2697], ptr[2698], ptr[2699],
							   ptr[2700], ptr[2701], ptr[2702], ptr[2703], ptr[2704],
							   ptr[2705], ptr[2706], ptr[2707], ptr[2708], ptr[2709],
							   ptr[2710], ptr[2711], ptr[2712], ptr[2713], ptr[2714],
							   ptr[2715], ptr[2716], ptr[2717], ptr[2718], ptr[2719],
							   ptr[2720], ptr[2721], ptr[2722], ptr[2723], ptr[2724],
							   ptr[2725], ptr[2726], ptr[2727], ptr[2728], ptr[2729],
							   ptr[2730], ptr[2731], ptr[2732], ptr[2733], ptr[2734],
							   ptr[2735], ptr[2736], ptr[2737], ptr[2738], ptr[2739],
							   ptr[2740], ptr[2741], ptr[2742], ptr[2743], ptr[2744],
							   ptr[2745], ptr[2746], ptr[2747], ptr[2748], ptr[2749],
							   ptr[2750], ptr[2751], ptr[2752], ptr[2753], ptr[2754],
							   ptr[2755], ptr[2756], ptr[2757], ptr[2758], ptr[2759],
							   ptr[2760], ptr[2761], ptr[2762], ptr[2763], ptr[2764],
							   ptr[2765], ptr[2766], ptr[2767], ptr[2768], ptr[2769],
							   ptr[2770], ptr[2771], ptr[2772], ptr[2773], ptr[2774],
							   ptr[2775], ptr[2776], ptr[2777], ptr[2778], ptr[2779],
							   ptr[2780], ptr[2781], ptr[2782], ptr[2783], ptr[2784],
							   ptr[2785], ptr[2786], ptr[2787], ptr[2788], ptr[2789],
							   ptr[2790], ptr[2791], ptr[2792], ptr[2793], ptr[2794],
							   ptr[2795], ptr[2796], ptr[2797], ptr[2798], ptr[2799],
							   ptr[2800], ptr[2801], ptr[2802], ptr[2803], ptr[2804],
							   ptr[2805], ptr[2806], ptr[2807], ptr[2808], ptr[2809],
							   ptr[2810], ptr[2811], ptr[2812], ptr[2813], ptr[2814],
							   ptr[2815], ptr[2816], ptr[2817], ptr[2818], ptr[2819],
							   ptr[2820], ptr[2821], ptr[2822], ptr[2823], ptr[2824],
							   ptr[2825], ptr[2826], ptr[2827], ptr[2828], ptr[2829],
							   ptr[2830], ptr[2831], ptr[2832], ptr[2833], ptr[2834],
							   ptr[2835], ptr[2836], ptr[2837], ptr[2838], ptr[2839],
							   ptr[2840], ptr[2841], ptr[2842], ptr[2843], ptr[2844],
							   ptr[2845], ptr[2846], ptr[2847], ptr[2848], ptr[2849],
							   ptr[2850], ptr[2851], ptr[2852], ptr[2853], ptr[2854],
							   ptr[2855], ptr[2856], ptr[2857], ptr[2858], ptr[2859],
							   ptr[2860], ptr[2861], ptr[2862], ptr[2863], ptr[2864],
							   ptr[2865], ptr[2866], ptr[2867], ptr[2868], ptr[2869],
							   ptr[2870], ptr[2871], ptr[2872], ptr[2873], ptr[2874],
							   ptr[2875], ptr[2876], ptr[2877], ptr[2878], ptr[2879],
							   ptr[2880], ptr[2881], ptr[2882], ptr[2883], ptr[2884],
							   ptr[2885], ptr[2886], ptr[2887], ptr[2888], ptr[2889],
							   ptr[2890], ptr[2891], ptr[2892], ptr[2893], ptr[2894],
							   ptr[2895], ptr[2896], ptr[2897], ptr[2898], ptr[2899],
							   ptr[2900], ptr[2901], ptr[2902], ptr[2903], ptr[2904],
							   ptr[2905], ptr[2906], ptr[2907], ptr[2908], ptr[2909],
							   ptr[2910], ptr[2911], ptr[2912], ptr[2913], ptr[2914],
							   ptr[2915], ptr[2916], ptr[2917], ptr[2918], ptr[2919],
							   ptr[2920], ptr[2921], ptr[2922], ptr[2923], ptr[2924],
							   ptr[2925], ptr[2926], ptr[2927], ptr[2928], ptr[2929],
							   ptr[2930], ptr[2931], ptr[2932], ptr[2933], ptr[2934],
							   ptr[2935], ptr[2936], ptr[2937], ptr[2938], ptr[2939],
							   ptr[2940], ptr[2941], ptr[2942], ptr[2943], ptr[2944],
							   ptr[2945], ptr[2946], ptr[2947], ptr[2948], ptr[2949],
							   ptr[2950], ptr[2951], ptr[2952], ptr[2953], ptr[2954],
							   ptr[2955], ptr[2956], ptr[2957], ptr[2958], ptr[2959],
							   ptr[2960], ptr[2961], ptr[2962], ptr[2963], ptr[2964],
							   ptr[2965], ptr[2966], ptr[2967], ptr[2968], ptr[2969],
							   ptr[2970], ptr[2971], ptr[2972], ptr[2973], ptr[2974],
							   ptr[2975], ptr[2976], ptr[2977], ptr[2978], ptr[2979],
							   ptr[2980], ptr[2981], ptr[2982], ptr[2983], ptr[2984],
							   ptr[2985], ptr[2986], ptr[2987], ptr[2988], ptr[2989],
							   ptr[2990], ptr[2991], ptr[2992], ptr[2993], ptr[2994],
							   ptr[2995], ptr[2996], ptr[2997], ptr[2998], ptr[2999],
							   ptr[3000], ptr[3001], ptr[3002], ptr[3003], ptr[3004],
							   ptr[3005], ptr[3006], ptr[3007], ptr[3008], ptr[3009],
							   ptr[3010], ptr[3011], ptr[3012], ptr[3013], ptr[3014],
							   ptr[3015], ptr[3016], ptr[3017], ptr[3018], ptr[3019],
							   ptr[3020], ptr[3021], ptr[3022], ptr[3023], ptr[3024],
							   ptr[3025], ptr[3026], ptr[3027], ptr[3028], ptr[3029],
							   ptr[3030], ptr[3031], ptr[3032], ptr[3033], ptr[3034],
							   ptr[3035], ptr[3036], ptr[3037], ptr[3038], ptr[3039],
							   ptr[3040], ptr[3041], ptr[3042], ptr[3043], ptr[3044],
							   ptr[3045], ptr[3046], ptr[3047], ptr[3048], ptr[3049],
							   ptr[3050], ptr[3051], ptr[3052], ptr[3053], ptr[3054],
							   ptr[3055], ptr[3056], ptr[3057], ptr[3058], ptr[3059],
							   ptr[3060], ptr[3061], ptr[3062], ptr[3063], ptr[3064],
							   ptr[3065], ptr[3066], ptr[3067], ptr[3068], ptr[3069],
							   ptr[3070], ptr[3071], ptr[3072], ptr[3073], ptr[3074],
							   ptr[3075], ptr[3076], ptr[3077], ptr[3078], ptr[3079],
							   ptr[3080], ptr[3081], ptr[3082], ptr[3083], ptr[3084],
							   ptr[3085], ptr[3086], ptr[3087], ptr[3088], ptr[3089],
							   ptr[3090], ptr[3091], ptr[3092], ptr[3093], ptr[3094],
							   ptr[3095], ptr[3096], ptr[3097], ptr[3098], ptr[3099],
							   ptr[3100], ptr[3101], ptr[3102], ptr[3103], ptr[3104],
							   ptr[3105], ptr[3106], ptr[3107], ptr[3108], ptr[3109],
							   ptr[3110], ptr[3111], ptr[3112], ptr[3113], ptr[3114],
							   ptr[3115], ptr[3116], ptr[3117], ptr[3118], ptr[3119],
							   ptr[3120], ptr[3121], ptr[3122], ptr[3123], ptr[3124],
							   ptr[3125], ptr[3126], ptr[3127], ptr[3128], ptr[3129],
							   ptr[3130], ptr[3131], ptr[3132], ptr[3133], ptr[3134],
							   ptr[3135], ptr[3136], ptr[3137], ptr[3138], ptr[3139],
							   ptr[3140], ptr[3141], ptr[3142], ptr[3143], ptr[3144],
							   ptr[3145], ptr[3146], ptr[3147], ptr[3148], ptr[3149],
							   ptr[3150], ptr[3151], ptr[3152], ptr[3153], ptr[3154],
							   ptr[3155], ptr[3156], ptr[3157], ptr[3158], ptr[3159],
							   ptr[3160], ptr[3161], ptr[3162], ptr[3163], ptr[3164],
							   ptr[3165], ptr[3166], ptr[3167], ptr[3168], ptr[3169],
							   ptr[3170], ptr[3171], ptr[3172], ptr[3173], ptr[3174],
							   ptr[3175], ptr[3176], ptr[3177], ptr[3178], ptr[3179],
							   ptr[3180], ptr[3181], ptr[3182], ptr[3183], ptr[3184],
							   ptr[3185], ptr[3186], ptr[3187], ptr[3188], ptr[3189],
							   ptr[3190], ptr[3191], ptr[3192], ptr[3193], ptr[3194],
							   ptr[3195], ptr[3196], ptr[3197], ptr[3198], ptr[3199],
							   ptr[3200], ptr[3201], ptr[3202], ptr[3203], ptr[3204],
							   ptr[3205], ptr[3206], ptr[3207], ptr[3208], ptr[3209],
							   ptr[3210], ptr[3211], ptr[3212], ptr[3213], ptr[3214],
							   ptr[3215], ptr[3216], ptr[3217], ptr[3218], ptr[3219],
							   ptr[3220], ptr[3221], ptr[3222], ptr[3223], ptr[3224],
							   ptr[3225], ptr[3226], ptr[3227], ptr[3228], ptr[3229],
							   ptr[3230], ptr[3231], ptr[3232], ptr[3233], ptr[3234],
							   ptr[3235], ptr[3236], ptr[3237], ptr[3238], ptr[3239],
							   ptr[3240], ptr[3241], ptr[3242], ptr[3243], ptr[3244],
							   ptr[3245], ptr[3246], ptr[3247], ptr[3248], ptr[3249],
							   ptr[3250], ptr[3251], ptr[3252], ptr[3253], ptr[3254],
							   ptr[3255], ptr[3256], ptr[3257], ptr[3258], ptr[3259],
							   ptr[3260], ptr[3261], ptr[3262], ptr[3263], ptr[3264],
							   ptr[3265], ptr[3266], ptr[3267], ptr[3268], ptr[3269],
							   ptr[3270], ptr[3271], ptr[3272], ptr[3273], ptr[3274],
							   ptr[3275], ptr[3276], ptr[3277], ptr[3278], ptr[3279],
							   ptr[3280], ptr[3281], ptr[3282], ptr[3283], ptr[3284],
							   ptr[3285], ptr[3286], ptr[3287], ptr[3288], ptr[3289],
							   ptr[3290], ptr[3291], ptr[3292], ptr[3293], ptr[3294],
							   ptr[3295], ptr[3296], ptr[3297], ptr[3298], ptr[3299],
							   ptr[3300], ptr[3301], ptr[3302], ptr[3303], ptr[3304],
							   ptr[3305], ptr[3306], ptr[3307], ptr[3308], ptr[3309],
							   ptr[3310], ptr[3311], ptr[3312], ptr[3313], ptr[3314],
							   ptr[3315], ptr[3316], ptr[3317], ptr[3318], ptr[3319],
							   ptr[3320], ptr[3321], ptr[3322], ptr[3323], ptr[3324],
							   ptr[3325], ptr[3326], ptr[3327], ptr[3328], ptr[3329],
							   ptr[3330], ptr[3331], ptr[3332], ptr[3333], ptr[3334],
							   ptr[3335], ptr[3336], ptr[3337], ptr[3338], ptr[3339],
							   ptr[3340], ptr[3341], ptr[3342], ptr[3343], ptr[3344],
							   ptr[3345], ptr[3346], ptr[3347], ptr[3348], ptr[3349],
							   ptr[3350], ptr[3351], ptr[3352], ptr[3353], ptr[3354],
							   ptr[3355], ptr[3356], ptr[3357], ptr[3358], ptr[3359],
							   ptr[3360], ptr[3361], ptr[3362], ptr[3363], ptr[3364],
							   ptr[3365], ptr[3366], ptr[3367], ptr[3368], ptr[3369],
							   ptr[3370], ptr[3371], ptr[3372], ptr[3373], ptr[3374],
							   ptr[3375], ptr[3376], ptr[3377], ptr[3378], ptr[3379],
							   ptr[3380], ptr[3381], ptr[3382], ptr[3383], ptr[3384],
							   ptr[3385], ptr[3386], ptr[3387], ptr[3388], ptr[3389],
							   ptr[3390], ptr[3391], ptr[3392], ptr[3393], ptr[3394],
							   ptr[3395], ptr[3396], ptr[3397], ptr[3398], ptr[3399],
							   ptr[3400], ptr[3401], ptr[3402], ptr[3403], ptr[3404],
							   ptr[3405], ptr[3406], ptr[3407], ptr[3408], ptr[3409],
							   ptr[3410], ptr[3411], ptr[3412], ptr[3413], ptr[3414],
							   ptr[3415], ptr[3416], ptr[3417], ptr[3418], ptr[3419],
							   ptr[3420], ptr[3421], ptr[3422], ptr[3423], ptr[3424],
							   ptr[3425], ptr[3426], ptr[3427], ptr[3428], ptr[3429],
							   ptr[3430], ptr[3431], ptr[3432], ptr[3433], ptr[3434],
							   ptr[3435], ptr[3436], ptr[3437], ptr[3438], ptr[3439],
							   ptr[3440], ptr[3441], ptr[3442], ptr[3443], ptr[3444],
							   ptr[3445], ptr[3446], ptr[3447], ptr[3448], ptr[3449],
							   ptr[3450], ptr[3451], ptr[3452], ptr[3453], ptr[3454],
							   ptr[3455], ptr[3456], ptr[3457], ptr[3458], ptr[3459],
							   ptr[3460], ptr[3461], ptr[3462], ptr[3463], ptr[3464],
							   ptr[3465], ptr[3466], ptr[3467], ptr[3468], ptr[3469],
							   ptr[3470], ptr[3471], ptr[3472], ptr[3473], ptr[3474],
							   ptr[3475], ptr[3476], ptr[3477], ptr[3478], ptr[3479],
							   ptr[3480], ptr[3481], ptr[3482], ptr[3483], ptr[3484],
							   ptr[3485], ptr[3486], ptr[3487], ptr[3488], ptr[3489],
							   ptr[3490], ptr[3491], ptr[3492], ptr[3493], ptr[3494],
							   ptr[3495], ptr[3496], ptr[3497], ptr[3498], ptr[3499],
							   ptr[3500], ptr[3501], ptr[3502], ptr[3503], ptr[3504],
							   ptr[3505], ptr[3506], ptr[3507], ptr[3508], ptr[3509],
							   ptr[3510], ptr[3511], ptr[3512], ptr[3513], ptr[3514],
							   ptr[3515], ptr[3516], ptr[3517], ptr[3518], ptr[3519],
							   ptr[3520], ptr[3521], ptr[3522], ptr[3523], ptr[3524],
							   ptr[3525], ptr[3526], ptr[3527], ptr[3528], ptr[3529],
							   ptr[3530], ptr[3531], ptr[3532], ptr[3533], ptr[3534],
							   ptr[3535], ptr[3536], ptr[3537], ptr[3538], ptr[3539],
							   ptr[3540], ptr[3541], ptr[3542], ptr[3543], ptr[3544],
							   ptr[3545], ptr[3546], ptr[3547], ptr[3548], ptr[3549],
							   ptr[3550], ptr[3551], ptr[3552], ptr[3553], ptr[3554],
							   ptr[3555], ptr[3556], ptr[3557], ptr[3558], ptr[3559],
							   ptr[3560], ptr[3561], ptr[3562], ptr[3563], ptr[3564],
							   ptr[3565], ptr[3566], ptr[3567], ptr[3568], ptr[3569],
							   ptr[3570], ptr[3571], ptr[3572], ptr[3573], ptr[3574],
							   ptr[3575], ptr[3576], ptr[3577], ptr[3578], ptr[3579],
							   ptr[3580], ptr[3581], ptr[3582], ptr[3583], ptr[3584],
							   ptr[3585], ptr[3586], ptr[3587], ptr[3588], ptr[3589],
							   ptr[3590], ptr[3591], ptr[3592], ptr[3593], ptr[3594],
							   ptr[3595], ptr[3596], ptr[3597], ptr[3598], ptr[3599],
							   ptr[3600], ptr[3601], ptr[3602], ptr[3603], ptr[3604],
							   ptr[3605], ptr[3606], ptr[3607], ptr[3608], ptr[3609],
							   ptr[3610], ptr[3611], ptr[3612], ptr[3613], ptr[3614],
							   ptr[3615], ptr[3616], ptr[3617], ptr[3618], ptr[3619],
							   ptr[3620], ptr[3621], ptr[3622], ptr[3623], ptr[3624],
							   ptr[3625], ptr[3626], ptr[3627], ptr[3628], ptr[3629],
							   ptr[3630], ptr[3631], ptr[3632], ptr[3633], ptr[3634],
							   ptr[3635], ptr[3636], ptr[3637], ptr[3638], ptr[3639],
							   ptr[3640], ptr[3641], ptr[3642], ptr[3643], ptr[3644],
							   ptr[3645], ptr[3646], ptr[3647], ptr[3648], ptr[3649],
							   ptr[3650], ptr[3651], ptr[3652], ptr[3653], ptr[3654],
							   ptr[3655], ptr[3656], ptr[3657], ptr[3658], ptr[3659],
							   ptr[3660], ptr[3661], ptr[3662], ptr[3663], ptr[3664],
							   ptr[3665], ptr[3666], ptr[3667], ptr[3668], ptr[3669],
							   ptr[3670], ptr[3671], ptr[3672], ptr[3673], ptr[3674],
							   ptr[3675], ptr[3676], ptr[3677], ptr[3678], ptr[3679],
							   ptr[3680], ptr[3681], ptr[3682], ptr[3683], ptr[3684],
							   ptr[3685], ptr[3686], ptr[3687], ptr[3688], ptr[3689],
							   ptr[3690], ptr[3691], ptr[3692], ptr[3693], ptr[3694],
							   ptr[3695], ptr[3696], ptr[3697], ptr[3698], ptr[3699],
							   ptr[3700], ptr[3701], ptr[3702], ptr[3703], ptr[3704],
							   ptr[3705], ptr[3706], ptr[3707], ptr[3708], ptr[3709],
							   ptr[3710], ptr[3711], ptr[3712], ptr[3713], ptr[3714],
							   ptr[3715], ptr[3716], ptr[3717], ptr[3718], ptr[3719],
							   ptr[3720], ptr[3721], ptr[3722], ptr[3723], ptr[3724],
							   ptr[3725], ptr[3726], ptr[3727], ptr[3728], ptr[3729],
							   ptr[3730], ptr[3731], ptr[3732], ptr[3733], ptr[3734],
							   ptr[3735], ptr[3736], ptr[3737], ptr[3738], ptr[3739],
							   ptr[3740], ptr[3741], ptr[3742], ptr[3743], ptr[3744],
							   ptr[3745], ptr[3746], ptr[3747], ptr[3748], ptr[3749],
							   ptr[3750], ptr[3751], ptr[3752], ptr[3753], ptr[3754],
							   ptr[3755], ptr[3756], ptr[3757], ptr[3758], ptr[3759],
							   ptr[3760], ptr[3761], ptr[3762], ptr[3763], ptr[3764],
							   ptr[3765], ptr[3766], ptr[3767], ptr[3768], ptr[3769],
							   ptr[3770], ptr[3771], ptr[3772], ptr[3773], ptr[3774],
							   ptr[3775], ptr[3776], ptr[3777], ptr[3778], ptr[3779],
							   ptr[3780], ptr[3781], ptr[3782], ptr[3783], ptr[3784],
							   ptr[3785], ptr[3786], ptr[3787], ptr[3788], ptr[3789],
							   ptr[3790], ptr[3791], ptr[3792], ptr[3793], ptr[3794],
							   ptr[3795], ptr[3796], ptr[3797], ptr[3798], ptr[3799],
							   ptr[3800], ptr[3801], ptr[3802], ptr[3803], ptr[3804],
							   ptr[3805], ptr[3806], ptr[3807], ptr[3808], ptr[3809],
							   ptr[3810], ptr[3811], ptr[3812], ptr[3813], ptr[3814],
							   ptr[3815], ptr[3816], ptr[3817], ptr[3818], ptr[3819],
							   ptr[3820], ptr[3821], ptr[3822], ptr[3823], ptr[3824],
							   ptr[3825], ptr[3826], ptr[3827], ptr[3828], ptr[3829],
							   ptr[3830], ptr[3831], ptr[3832], ptr[3833], ptr[3834],
							   ptr[3835], ptr[3836], ptr[3837], ptr[3838], ptr[3839],
							   ptr[3840], ptr[3841], ptr[3842], ptr[3843], ptr[3844],
							   ptr[3845], ptr[3846], ptr[3847], ptr[3848], ptr[3849],
							   ptr[3850], ptr[3851], ptr[3852], ptr[3853], ptr[3854],
							   ptr[3855], ptr[3856], ptr[3857], ptr[3858], ptr[3859],
							   ptr[3860], ptr[3861], ptr[3862], ptr[3863], ptr[3864],
							   ptr[3865], ptr[3866], ptr[3867], ptr[3868], ptr[3869],
							   ptr[3870], ptr[3871], ptr[3872], ptr[3873], ptr[3874],
							   ptr[3875], ptr[3876], ptr[3877], ptr[3878], ptr[3879],
							   ptr[3880], ptr[3881], ptr[3882], ptr[3883], ptr[3884],
							   ptr[3885], ptr[3886], ptr[3887], ptr[3888], ptr[3889],
							   ptr[3890], ptr[3891], ptr[3892], ptr[3893], ptr[3894],
							   ptr[3895], ptr[3896], ptr[3897], ptr[3898], ptr[3899],
							   ptr[3900], ptr[3901], ptr[3902], ptr[3903], ptr[3904],
							   ptr[3905], ptr[3906], ptr[3907], ptr[3908], ptr[3909],
							   ptr[3910], ptr[3911], ptr[3912], ptr[3913], ptr[3914],
							   ptr[3915], ptr[3916], ptr[3917], ptr[3918], ptr[3919],
							   ptr[3920], ptr[3921], ptr[3922], ptr[3923], ptr[3924],
							   ptr[3925], ptr[3926], ptr[3927], ptr[3928], ptr[3929],
							   ptr[3930], ptr[3931], ptr[3932], ptr[3933], ptr[3934],
							   ptr[3935], ptr[3936], ptr[3937], ptr[3938], ptr[3939],
							   ptr[3940], ptr[3941], ptr[3942], ptr[3943], ptr[3944],
							   ptr[3945], ptr[3946], ptr[3947], ptr[3948], ptr[3949],
							   ptr[3950], ptr[3951], ptr[3952], ptr[3953], ptr[3954],
							   ptr[3955], ptr[3956], ptr[3957], ptr[3958], ptr[3959],
							   ptr[3960], ptr[3961], ptr[3962], ptr[3963], ptr[3964],
							   ptr[3965], ptr[3966], ptr[3967], ptr[3968], ptr[3969],
							   ptr[3970], ptr[3971], ptr[3972], ptr[3973], ptr[3974],
							   ptr[3975], ptr[3976], ptr[3977], ptr[3978], ptr[3979],
							   ptr[3980], ptr[3981], ptr[3982], ptr[3983], ptr[3984],
							   ptr[3985], ptr[3986], ptr[3987], ptr[3988], ptr[3989],
							   ptr[3990], ptr[3991], ptr[3992], ptr[3993], ptr[3994],
							   ptr[3995], ptr[3996], ptr[3997], ptr[3998], ptr[3999]);
	    } else {
		comp[ci].gui_ptr = NULL;
	    }
	    if (comp_type[i].aui_create) {
		comp[ci].aui_ptr = comp_type[i].aui_create(comp[ci].name,
							   ptr[0], ptr[1], ptr[2], ptr[3], ptr[4],
							   ptr[5], ptr[6], ptr[7], ptr[8], ptr[9],
							   ptr[10], ptr[11], ptr[12], ptr[13], ptr[14],
							   ptr[15], ptr[16], ptr[17], ptr[18], ptr[19],
							   ptr[20], ptr[21], ptr[22], ptr[23], ptr[24],
							   ptr[25], ptr[26], ptr[27], ptr[28], ptr[29],
							   ptr[30], ptr[31], ptr[32], ptr[33], ptr[34],
							   ptr[35], ptr[36], ptr[37], ptr[38], ptr[39],
							   ptr[40], ptr[41], ptr[42], ptr[43], ptr[44],
							   ptr[45], ptr[46], ptr[47], ptr[48], ptr[49],
							   ptr[50], ptr[51], ptr[52], ptr[53], ptr[54],
							   ptr[55], ptr[56], ptr[57], ptr[58], ptr[59],
							   ptr[60], ptr[61], ptr[62], ptr[63], ptr[64],
							   ptr[65], ptr[66], ptr[67], ptr[68], ptr[69],
							   ptr[70], ptr[71], ptr[72], ptr[73], ptr[74],
							   ptr[75], ptr[76], ptr[77], ptr[78], ptr[79],
							   ptr[80], ptr[81], ptr[82], ptr[83], ptr[84],
							   ptr[85], ptr[86], ptr[87], ptr[88], ptr[89],
							   ptr[90], ptr[91], ptr[92], ptr[93], ptr[94],
							   ptr[95], ptr[96], ptr[97], ptr[98], ptr[99],
							   ptr[100], ptr[101], ptr[102], ptr[103], ptr[104],
							   ptr[105], ptr[106], ptr[107], ptr[108], ptr[109],
							   ptr[110], ptr[111], ptr[112], ptr[113], ptr[114],
							   ptr[115], ptr[116], ptr[117], ptr[118], ptr[119],
							   ptr[120], ptr[121], ptr[122], ptr[123], ptr[124],
							   ptr[125], ptr[126], ptr[127], ptr[128], ptr[129],
							   ptr[130], ptr[131], ptr[132], ptr[133], ptr[134],
							   ptr[135], ptr[136], ptr[137], ptr[138], ptr[139],
							   ptr[140], ptr[141], ptr[142], ptr[143], ptr[144],
							   ptr[145], ptr[146], ptr[147], ptr[148], ptr[149],
							   ptr[150], ptr[151], ptr[152], ptr[153], ptr[154],
							   ptr[155], ptr[156], ptr[157], ptr[158], ptr[159],
							   ptr[160], ptr[161], ptr[162], ptr[163], ptr[164],
							   ptr[165], ptr[166], ptr[167], ptr[168], ptr[169],
							   ptr[170], ptr[171], ptr[172], ptr[173], ptr[174],
							   ptr[175], ptr[176], ptr[177], ptr[178], ptr[179],
							   ptr[180], ptr[181], ptr[182], ptr[183], ptr[184],
							   ptr[185], ptr[186], ptr[187], ptr[188], ptr[189],
							   ptr[190], ptr[191], ptr[192], ptr[193], ptr[194],
							   ptr[195], ptr[196], ptr[197], ptr[198], ptr[199],
							   ptr[200], ptr[201], ptr[202], ptr[203], ptr[204],
							   ptr[205], ptr[206], ptr[207], ptr[208], ptr[209],
							   ptr[210], ptr[211], ptr[212], ptr[213], ptr[214],
							   ptr[215], ptr[216], ptr[217], ptr[218], ptr[219],
							   ptr[220], ptr[221], ptr[222], ptr[223], ptr[224],
							   ptr[225], ptr[226], ptr[227], ptr[228], ptr[229],
							   ptr[230], ptr[231], ptr[232], ptr[233], ptr[234],
							   ptr[235], ptr[236], ptr[237], ptr[238], ptr[239],
							   ptr[240], ptr[241], ptr[242], ptr[243], ptr[244],
							   ptr[245], ptr[246], ptr[247], ptr[248], ptr[249],
							   ptr[250], ptr[251], ptr[252], ptr[253], ptr[254],
							   ptr[255], ptr[256], ptr[257], ptr[258], ptr[259],
							   ptr[260], ptr[261], ptr[262], ptr[263], ptr[264],
							   ptr[265], ptr[266], ptr[267], ptr[268], ptr[269],
							   ptr[270], ptr[271], ptr[272], ptr[273], ptr[274],
							   ptr[275], ptr[276], ptr[277], ptr[278], ptr[279],
							   ptr[280], ptr[281], ptr[282], ptr[283], ptr[284],
							   ptr[285], ptr[286], ptr[287], ptr[288], ptr[289],
							   ptr[290], ptr[291], ptr[292], ptr[293], ptr[294],
							   ptr[295], ptr[296], ptr[297], ptr[298], ptr[299],
							   ptr[300], ptr[301], ptr[302], ptr[303], ptr[304],
							   ptr[305], ptr[306], ptr[307], ptr[308], ptr[309],
							   ptr[310], ptr[311], ptr[312], ptr[313], ptr[314],
							   ptr[315], ptr[316], ptr[317], ptr[318], ptr[319],
							   ptr[320], ptr[321], ptr[322], ptr[323], ptr[324],
							   ptr[325], ptr[326], ptr[327], ptr[328], ptr[329],
							   ptr[330], ptr[331], ptr[332], ptr[333], ptr[334],
							   ptr[335], ptr[336], ptr[337], ptr[338], ptr[339],
							   ptr[340], ptr[341], ptr[342], ptr[343], ptr[344],
							   ptr[345], ptr[346], ptr[347], ptr[348], ptr[349],
							   ptr[350], ptr[351], ptr[352], ptr[353], ptr[354],
							   ptr[355], ptr[356], ptr[357], ptr[358], ptr[359],
							   ptr[360], ptr[361], ptr[362], ptr[363], ptr[364],
							   ptr[365], ptr[366], ptr[367], ptr[368], ptr[369],
							   ptr[370], ptr[371], ptr[372], ptr[373], ptr[374],
							   ptr[375], ptr[376], ptr[377], ptr[378], ptr[379],
							   ptr[380], ptr[381], ptr[382], ptr[383], ptr[384],
							   ptr[385], ptr[386], ptr[387], ptr[388], ptr[389],
							   ptr[390], ptr[391], ptr[392], ptr[393], ptr[394],
							   ptr[395], ptr[396], ptr[397], ptr[398], ptr[399],
							   ptr[400], ptr[401], ptr[402], ptr[403], ptr[404],
							   ptr[405], ptr[406], ptr[407], ptr[408], ptr[409],
							   ptr[410], ptr[411], ptr[412], ptr[413], ptr[414],
							   ptr[415], ptr[416], ptr[417], ptr[418], ptr[419],
							   ptr[420], ptr[421], ptr[422], ptr[423], ptr[424],
							   ptr[425], ptr[426], ptr[427], ptr[428], ptr[429],
							   ptr[430], ptr[431], ptr[432], ptr[433], ptr[434],
							   ptr[435], ptr[436], ptr[437], ptr[438], ptr[439],
							   ptr[440], ptr[441], ptr[442], ptr[443], ptr[444],
							   ptr[445], ptr[446], ptr[447], ptr[448], ptr[449],
							   ptr[450], ptr[451], ptr[452], ptr[453], ptr[454],
							   ptr[455], ptr[456], ptr[457], ptr[458], ptr[459],
							   ptr[460], ptr[461], ptr[462], ptr[463], ptr[464],
							   ptr[465], ptr[466], ptr[467], ptr[468], ptr[469],
							   ptr[470], ptr[471], ptr[472], ptr[473], ptr[474],
							   ptr[475], ptr[476], ptr[477], ptr[478], ptr[479],
							   ptr[480], ptr[481], ptr[482], ptr[483], ptr[484],
							   ptr[485], ptr[486], ptr[487], ptr[488], ptr[489],
							   ptr[490], ptr[491], ptr[492], ptr[493], ptr[494],
							   ptr[495], ptr[496], ptr[497], ptr[498], ptr[499],
							   ptr[500], ptr[501], ptr[502], ptr[503], ptr[504],
							   ptr[505], ptr[506], ptr[507], ptr[508], ptr[509],
							   ptr[510], ptr[511], ptr[512], ptr[513], ptr[514],
							   ptr[515], ptr[516], ptr[517], ptr[518], ptr[519],
							   ptr[520], ptr[521], ptr[522], ptr[523], ptr[524],
							   ptr[525], ptr[526], ptr[527], ptr[528], ptr[529],
							   ptr[530], ptr[531], ptr[532], ptr[533], ptr[534],
							   ptr[535], ptr[536], ptr[537], ptr[538], ptr[539],
							   ptr[540], ptr[541], ptr[542], ptr[543], ptr[544],
							   ptr[545], ptr[546], ptr[547], ptr[548], ptr[549],
							   ptr[550], ptr[551], ptr[552], ptr[553], ptr[554],
							   ptr[555], ptr[556], ptr[557], ptr[558], ptr[559],
							   ptr[560], ptr[561], ptr[562], ptr[563], ptr[564],
							   ptr[565], ptr[566], ptr[567], ptr[568], ptr[569],
							   ptr[570], ptr[571], ptr[572], ptr[573], ptr[574],
							   ptr[575], ptr[576], ptr[577], ptr[578], ptr[579],
							   ptr[580], ptr[581], ptr[582], ptr[583], ptr[584],
							   ptr[585], ptr[586], ptr[587], ptr[588], ptr[589],
							   ptr[590], ptr[591], ptr[592], ptr[593], ptr[594],
							   ptr[595], ptr[596], ptr[597], ptr[598], ptr[599],
							   ptr[600], ptr[601], ptr[602], ptr[603], ptr[604],
							   ptr[605], ptr[606], ptr[607], ptr[608], ptr[609],
							   ptr[610], ptr[611], ptr[612], ptr[613], ptr[614],
							   ptr[615], ptr[616], ptr[617], ptr[618], ptr[619],
							   ptr[620], ptr[621], ptr[622], ptr[623], ptr[624],
							   ptr[625], ptr[626], ptr[627], ptr[628], ptr[629],
							   ptr[630], ptr[631], ptr[632], ptr[633], ptr[634],
							   ptr[635], ptr[636], ptr[637], ptr[638], ptr[639],
							   ptr[640], ptr[641], ptr[642], ptr[643], ptr[644],
							   ptr[645], ptr[646], ptr[647], ptr[648], ptr[649],
							   ptr[650], ptr[651], ptr[652], ptr[653], ptr[654],
							   ptr[655], ptr[656], ptr[657], ptr[658], ptr[659],
							   ptr[660], ptr[661], ptr[662], ptr[663], ptr[664],
							   ptr[665], ptr[666], ptr[667], ptr[668], ptr[669],
							   ptr[670], ptr[671], ptr[672], ptr[673], ptr[674],
							   ptr[675], ptr[676], ptr[677], ptr[678], ptr[679],
							   ptr[680], ptr[681], ptr[682], ptr[683], ptr[684],
							   ptr[685], ptr[686], ptr[687], ptr[688], ptr[689],
							   ptr[690], ptr[691], ptr[692], ptr[693], ptr[694],
							   ptr[695], ptr[696], ptr[697], ptr[698], ptr[699],
							   ptr[700], ptr[701], ptr[702], ptr[703], ptr[704],
							   ptr[705], ptr[706], ptr[707], ptr[708], ptr[709],
							   ptr[710], ptr[711], ptr[712], ptr[713], ptr[714],
							   ptr[715], ptr[716], ptr[717], ptr[718], ptr[719],
							   ptr[720], ptr[721], ptr[722], ptr[723], ptr[724],
							   ptr[725], ptr[726], ptr[727], ptr[728], ptr[729],
							   ptr[730], ptr[731], ptr[732], ptr[733], ptr[734],
							   ptr[735], ptr[736], ptr[737], ptr[738], ptr[739],
							   ptr[740], ptr[741], ptr[742], ptr[743], ptr[744],
							   ptr[745], ptr[746], ptr[747], ptr[748], ptr[749],
							   ptr[750], ptr[751], ptr[752], ptr[753], ptr[754],
							   ptr[755], ptr[756], ptr[757], ptr[758], ptr[759],
							   ptr[760], ptr[761], ptr[762], ptr[763], ptr[764],
							   ptr[765], ptr[766], ptr[767], ptr[768], ptr[769],
							   ptr[770], ptr[771], ptr[772], ptr[773], ptr[774],
							   ptr[775], ptr[776], ptr[777], ptr[778], ptr[779],
							   ptr[780], ptr[781], ptr[782], ptr[783], ptr[784],
							   ptr[785], ptr[786], ptr[787], ptr[788], ptr[789],
							   ptr[790], ptr[791], ptr[792], ptr[793], ptr[794],
							   ptr[795], ptr[796], ptr[797], ptr[798], ptr[799],
							   ptr[800], ptr[801], ptr[802], ptr[803], ptr[804],
							   ptr[805], ptr[806], ptr[807], ptr[808], ptr[809],
							   ptr[810], ptr[811], ptr[812], ptr[813], ptr[814],
							   ptr[815], ptr[816], ptr[817], ptr[818], ptr[819],
							   ptr[820], ptr[821], ptr[822], ptr[823], ptr[824],
							   ptr[825], ptr[826], ptr[827], ptr[828], ptr[829],
							   ptr[830], ptr[831], ptr[832], ptr[833], ptr[834],
							   ptr[835], ptr[836], ptr[837], ptr[838], ptr[839],
							   ptr[840], ptr[841], ptr[842], ptr[843], ptr[844],
							   ptr[845], ptr[846], ptr[847], ptr[848], ptr[849],
							   ptr[850], ptr[851], ptr[852], ptr[853], ptr[854],
							   ptr[855], ptr[856], ptr[857], ptr[858], ptr[859],
							   ptr[860], ptr[861], ptr[862], ptr[863], ptr[864],
							   ptr[865], ptr[866], ptr[867], ptr[868], ptr[869],
							   ptr[870], ptr[871], ptr[872], ptr[873], ptr[874],
							   ptr[875], ptr[876], ptr[877], ptr[878], ptr[879],
							   ptr[880], ptr[881], ptr[882], ptr[883], ptr[884],
							   ptr[885], ptr[886], ptr[887], ptr[888], ptr[889],
							   ptr[890], ptr[891], ptr[892], ptr[893], ptr[894],
							   ptr[895], ptr[896], ptr[897], ptr[898], ptr[899],
							   ptr[900], ptr[901], ptr[902], ptr[903], ptr[904],
							   ptr[905], ptr[906], ptr[907], ptr[908], ptr[909],
							   ptr[910], ptr[911], ptr[912], ptr[913], ptr[914],
							   ptr[915], ptr[916], ptr[917], ptr[918], ptr[919],
							   ptr[920], ptr[921], ptr[922], ptr[923], ptr[924],
							   ptr[925], ptr[926], ptr[927], ptr[928], ptr[929],
							   ptr[930], ptr[931], ptr[932], ptr[933], ptr[934],
							   ptr[935], ptr[936], ptr[937], ptr[938], ptr[939],
							   ptr[940], ptr[941], ptr[942], ptr[943], ptr[944],
							   ptr[945], ptr[946], ptr[947], ptr[948], ptr[949],
							   ptr[950], ptr[951], ptr[952], ptr[953], ptr[954],
							   ptr[955], ptr[956], ptr[957], ptr[958], ptr[959],
							   ptr[960], ptr[961], ptr[962], ptr[963], ptr[964],
							   ptr[965], ptr[966], ptr[967], ptr[968], ptr[969],
							   ptr[970], ptr[971], ptr[972], ptr[973], ptr[974],
							   ptr[975], ptr[976], ptr[977], ptr[978], ptr[979],
							   ptr[980], ptr[981], ptr[982], ptr[983], ptr[984],
							   ptr[985], ptr[986], ptr[987], ptr[988], ptr[989],
							   ptr[990], ptr[991], ptr[992], ptr[993], ptr[994],
							   ptr[995], ptr[996], ptr[997], ptr[998], ptr[999],
							   ptr[1000], ptr[1001], ptr[1002], ptr[1003], ptr[1004],
							   ptr[1005], ptr[1006], ptr[1007], ptr[1008], ptr[1009],
							   ptr[1010], ptr[1011], ptr[1012], ptr[1013], ptr[1014],
							   ptr[1015], ptr[1016], ptr[1017], ptr[1018], ptr[1019],
							   ptr[1020], ptr[1021], ptr[1022], ptr[1023], ptr[1024],
							   ptr[1025], ptr[1026], ptr[1027], ptr[1028], ptr[1029],
							   ptr[1030], ptr[1031], ptr[1032], ptr[1033], ptr[1034],
							   ptr[1035], ptr[1036], ptr[1037], ptr[1038], ptr[1039],
							   ptr[1040], ptr[1041], ptr[1042], ptr[1043], ptr[1044],
							   ptr[1045], ptr[1046], ptr[1047], ptr[1048], ptr[1049],
							   ptr[1050], ptr[1051], ptr[1052], ptr[1053], ptr[1054],
							   ptr[1055], ptr[1056], ptr[1057], ptr[1058], ptr[1059],
							   ptr[1060], ptr[1061], ptr[1062], ptr[1063], ptr[1064],
							   ptr[1065], ptr[1066], ptr[1067], ptr[1068], ptr[1069],
							   ptr[1070], ptr[1071], ptr[1072], ptr[1073], ptr[1074],
							   ptr[1075], ptr[1076], ptr[1077], ptr[1078], ptr[1079],
							   ptr[1080], ptr[1081], ptr[1082], ptr[1083], ptr[1084],
							   ptr[1085], ptr[1086], ptr[1087], ptr[1088], ptr[1089],
							   ptr[1090], ptr[1091], ptr[1092], ptr[1093], ptr[1094],
							   ptr[1095], ptr[1096], ptr[1097], ptr[1098], ptr[1099],
							   ptr[1100], ptr[1101], ptr[1102], ptr[1103], ptr[1104],
							   ptr[1105], ptr[1106], ptr[1107], ptr[1108], ptr[1109],
							   ptr[1110], ptr[1111], ptr[1112], ptr[1113], ptr[1114],
							   ptr[1115], ptr[1116], ptr[1117], ptr[1118], ptr[1119],
							   ptr[1120], ptr[1121], ptr[1122], ptr[1123], ptr[1124],
							   ptr[1125], ptr[1126], ptr[1127], ptr[1128], ptr[1129],
							   ptr[1130], ptr[1131], ptr[1132], ptr[1133], ptr[1134],
							   ptr[1135], ptr[1136], ptr[1137], ptr[1138], ptr[1139],
							   ptr[1140], ptr[1141], ptr[1142], ptr[1143], ptr[1144],
							   ptr[1145], ptr[1146], ptr[1147], ptr[1148], ptr[1149],
							   ptr[1150], ptr[1151], ptr[1152], ptr[1153], ptr[1154],
							   ptr[1155], ptr[1156], ptr[1157], ptr[1158], ptr[1159],
							   ptr[1160], ptr[1161], ptr[1162], ptr[1163], ptr[1164],
							   ptr[1165], ptr[1166], ptr[1167], ptr[1168], ptr[1169],
							   ptr[1170], ptr[1171], ptr[1172], ptr[1173], ptr[1174],
							   ptr[1175], ptr[1176], ptr[1177], ptr[1178], ptr[1179],
							   ptr[1180], ptr[1181], ptr[1182], ptr[1183], ptr[1184],
							   ptr[1185], ptr[1186], ptr[1187], ptr[1188], ptr[1189],
							   ptr[1190], ptr[1191], ptr[1192], ptr[1193], ptr[1194],
							   ptr[1195], ptr[1196], ptr[1197], ptr[1198], ptr[1199],
							   ptr[1200], ptr[1201], ptr[1202], ptr[1203], ptr[1204],
							   ptr[1205], ptr[1206], ptr[1207], ptr[1208], ptr[1209],
							   ptr[1210], ptr[1211], ptr[1212], ptr[1213], ptr[1214],
							   ptr[1215], ptr[1216], ptr[1217], ptr[1218], ptr[1219],
							   ptr[1220], ptr[1221], ptr[1222], ptr[1223], ptr[1224],
							   ptr[1225], ptr[1226], ptr[1227], ptr[1228], ptr[1229],
							   ptr[1230], ptr[1231], ptr[1232], ptr[1233], ptr[1234],
							   ptr[1235], ptr[1236], ptr[1237], ptr[1238], ptr[1239],
							   ptr[1240], ptr[1241], ptr[1242], ptr[1243], ptr[1244],
							   ptr[1245], ptr[1246], ptr[1247], ptr[1248], ptr[1249],
							   ptr[1250], ptr[1251], ptr[1252], ptr[1253], ptr[1254],
							   ptr[1255], ptr[1256], ptr[1257], ptr[1258], ptr[1259],
							   ptr[1260], ptr[1261], ptr[1262], ptr[1263], ptr[1264],
							   ptr[1265], ptr[1266], ptr[1267], ptr[1268], ptr[1269],
							   ptr[1270], ptr[1271], ptr[1272], ptr[1273], ptr[1274],
							   ptr[1275], ptr[1276], ptr[1277], ptr[1278], ptr[1279],
							   ptr[1280], ptr[1281], ptr[1282], ptr[1283], ptr[1284],
							   ptr[1285], ptr[1286], ptr[1287], ptr[1288], ptr[1289],
							   ptr[1290], ptr[1291], ptr[1292], ptr[1293], ptr[1294],
							   ptr[1295], ptr[1296], ptr[1297], ptr[1298], ptr[1299],
							   ptr[1300], ptr[1301], ptr[1302], ptr[1303], ptr[1304],
							   ptr[1305], ptr[1306], ptr[1307], ptr[1308], ptr[1309],
							   ptr[1310], ptr[1311], ptr[1312], ptr[1313], ptr[1314],
							   ptr[1315], ptr[1316], ptr[1317], ptr[1318], ptr[1319],
							   ptr[1320], ptr[1321], ptr[1322], ptr[1323], ptr[1324],
							   ptr[1325], ptr[1326], ptr[1327], ptr[1328], ptr[1329],
							   ptr[1330], ptr[1331], ptr[1332], ptr[1333], ptr[1334],
							   ptr[1335], ptr[1336], ptr[1337], ptr[1338], ptr[1339],
							   ptr[1340], ptr[1341], ptr[1342], ptr[1343], ptr[1344],
							   ptr[1345], ptr[1346], ptr[1347], ptr[1348], ptr[1349],
							   ptr[1350], ptr[1351], ptr[1352], ptr[1353], ptr[1354],
							   ptr[1355], ptr[1356], ptr[1357], ptr[1358], ptr[1359],
							   ptr[1360], ptr[1361], ptr[1362], ptr[1363], ptr[1364],
							   ptr[1365], ptr[1366], ptr[1367], ptr[1368], ptr[1369],
							   ptr[1370], ptr[1371], ptr[1372], ptr[1373], ptr[1374],
							   ptr[1375], ptr[1376], ptr[1377], ptr[1378], ptr[1379],
							   ptr[1380], ptr[1381], ptr[1382], ptr[1383], ptr[1384],
							   ptr[1385], ptr[1386], ptr[1387], ptr[1388], ptr[1389],
							   ptr[1390], ptr[1391], ptr[1392], ptr[1393], ptr[1394],
							   ptr[1395], ptr[1396], ptr[1397], ptr[1398], ptr[1399],
							   ptr[1400], ptr[1401], ptr[1402], ptr[1403], ptr[1404],
							   ptr[1405], ptr[1406], ptr[1407], ptr[1408], ptr[1409],
							   ptr[1410], ptr[1411], ptr[1412], ptr[1413], ptr[1414],
							   ptr[1415], ptr[1416], ptr[1417], ptr[1418], ptr[1419],
							   ptr[1420], ptr[1421], ptr[1422], ptr[1423], ptr[1424],
							   ptr[1425], ptr[1426], ptr[1427], ptr[1428], ptr[1429],
							   ptr[1430], ptr[1431], ptr[1432], ptr[1433], ptr[1434],
							   ptr[1435], ptr[1436], ptr[1437], ptr[1438], ptr[1439],
							   ptr[1440], ptr[1441], ptr[1442], ptr[1443], ptr[1444],
							   ptr[1445], ptr[1446], ptr[1447], ptr[1448], ptr[1449],
							   ptr[1450], ptr[1451], ptr[1452], ptr[1453], ptr[1454],
							   ptr[1455], ptr[1456], ptr[1457], ptr[1458], ptr[1459],
							   ptr[1460], ptr[1461], ptr[1462], ptr[1463], ptr[1464],
							   ptr[1465], ptr[1466], ptr[1467], ptr[1468], ptr[1469],
							   ptr[1470], ptr[1471], ptr[1472], ptr[1473], ptr[1474],
							   ptr[1475], ptr[1476], ptr[1477], ptr[1478], ptr[1479],
							   ptr[1480], ptr[1481], ptr[1482], ptr[1483], ptr[1484],
							   ptr[1485], ptr[1486], ptr[1487], ptr[1488], ptr[1489],
							   ptr[1490], ptr[1491], ptr[1492], ptr[1493], ptr[1494],
							   ptr[1495], ptr[1496], ptr[1497], ptr[1498], ptr[1499],
							   ptr[1500], ptr[1501], ptr[1502], ptr[1503], ptr[1504],
							   ptr[1505], ptr[1506], ptr[1507], ptr[1508], ptr[1509],
							   ptr[1510], ptr[1511], ptr[1512], ptr[1513], ptr[1514],
							   ptr[1515], ptr[1516], ptr[1517], ptr[1518], ptr[1519],
							   ptr[1520], ptr[1521], ptr[1522], ptr[1523], ptr[1524],
							   ptr[1525], ptr[1526], ptr[1527], ptr[1528], ptr[1529],
							   ptr[1530], ptr[1531], ptr[1532], ptr[1533], ptr[1534],
							   ptr[1535], ptr[1536], ptr[1537], ptr[1538], ptr[1539],
							   ptr[1540], ptr[1541], ptr[1542], ptr[1543], ptr[1544],
							   ptr[1545], ptr[1546], ptr[1547], ptr[1548], ptr[1549],
							   ptr[1550], ptr[1551], ptr[1552], ptr[1553], ptr[1554],
							   ptr[1555], ptr[1556], ptr[1557], ptr[1558], ptr[1559],
							   ptr[1560], ptr[1561], ptr[1562], ptr[1563], ptr[1564],
							   ptr[1565], ptr[1566], ptr[1567], ptr[1568], ptr[1569],
							   ptr[1570], ptr[1571], ptr[1572], ptr[1573], ptr[1574],
							   ptr[1575], ptr[1576], ptr[1577], ptr[1578], ptr[1579],
							   ptr[1580], ptr[1581], ptr[1582], ptr[1583], ptr[1584],
							   ptr[1585], ptr[1586], ptr[1587], ptr[1588], ptr[1589],
							   ptr[1590], ptr[1591], ptr[1592], ptr[1593], ptr[1594],
							   ptr[1595], ptr[1596], ptr[1597], ptr[1598], ptr[1599],
							   ptr[1600], ptr[1601], ptr[1602], ptr[1603], ptr[1604],
							   ptr[1605], ptr[1606], ptr[1607], ptr[1608], ptr[1609],
							   ptr[1610], ptr[1611], ptr[1612], ptr[1613], ptr[1614],
							   ptr[1615], ptr[1616], ptr[1617], ptr[1618], ptr[1619],
							   ptr[1620], ptr[1621], ptr[1622], ptr[1623], ptr[1624],
							   ptr[1625], ptr[1626], ptr[1627], ptr[1628], ptr[1629],
							   ptr[1630], ptr[1631], ptr[1632], ptr[1633], ptr[1634],
							   ptr[1635], ptr[1636], ptr[1637], ptr[1638], ptr[1639],
							   ptr[1640], ptr[1641], ptr[1642], ptr[1643], ptr[1644],
							   ptr[1645], ptr[1646], ptr[1647], ptr[1648], ptr[1649],
							   ptr[1650], ptr[1651], ptr[1652], ptr[1653], ptr[1654],
							   ptr[1655], ptr[1656], ptr[1657], ptr[1658], ptr[1659],
							   ptr[1660], ptr[1661], ptr[1662], ptr[1663], ptr[1664],
							   ptr[1665], ptr[1666], ptr[1667], ptr[1668], ptr[1669],
							   ptr[1670], ptr[1671], ptr[1672], ptr[1673], ptr[1674],
							   ptr[1675], ptr[1676], ptr[1677], ptr[1678], ptr[1679],
							   ptr[1680], ptr[1681], ptr[1682], ptr[1683], ptr[1684],
							   ptr[1685], ptr[1686], ptr[1687], ptr[1688], ptr[1689],
							   ptr[1690], ptr[1691], ptr[1692], ptr[1693], ptr[1694],
							   ptr[1695], ptr[1696], ptr[1697], ptr[1698], ptr[1699],
							   ptr[1700], ptr[1701], ptr[1702], ptr[1703], ptr[1704],
							   ptr[1705], ptr[1706], ptr[1707], ptr[1708], ptr[1709],
							   ptr[1710], ptr[1711], ptr[1712], ptr[1713], ptr[1714],
							   ptr[1715], ptr[1716], ptr[1717], ptr[1718], ptr[1719],
							   ptr[1720], ptr[1721], ptr[1722], ptr[1723], ptr[1724],
							   ptr[1725], ptr[1726], ptr[1727], ptr[1728], ptr[1729],
							   ptr[1730], ptr[1731], ptr[1732], ptr[1733], ptr[1734],
							   ptr[1735], ptr[1736], ptr[1737], ptr[1738], ptr[1739],
							   ptr[1740], ptr[1741], ptr[1742], ptr[1743], ptr[1744],
							   ptr[1745], ptr[1746], ptr[1747], ptr[1748], ptr[1749],
							   ptr[1750], ptr[1751], ptr[1752], ptr[1753], ptr[1754],
							   ptr[1755], ptr[1756], ptr[1757], ptr[1758], ptr[1759],
							   ptr[1760], ptr[1761], ptr[1762], ptr[1763], ptr[1764],
							   ptr[1765], ptr[1766], ptr[1767], ptr[1768], ptr[1769],
							   ptr[1770], ptr[1771], ptr[1772], ptr[1773], ptr[1774],
							   ptr[1775], ptr[1776], ptr[1777], ptr[1778], ptr[1779],
							   ptr[1780], ptr[1781], ptr[1782], ptr[1783], ptr[1784],
							   ptr[1785], ptr[1786], ptr[1787], ptr[1788], ptr[1789],
							   ptr[1790], ptr[1791], ptr[1792], ptr[1793], ptr[1794],
							   ptr[1795], ptr[1796], ptr[1797], ptr[1798], ptr[1799],
							   ptr[1800], ptr[1801], ptr[1802], ptr[1803], ptr[1804],
							   ptr[1805], ptr[1806], ptr[1807], ptr[1808], ptr[1809],
							   ptr[1810], ptr[1811], ptr[1812], ptr[1813], ptr[1814],
							   ptr[1815], ptr[1816], ptr[1817], ptr[1818], ptr[1819],
							   ptr[1820], ptr[1821], ptr[1822], ptr[1823], ptr[1824],
							   ptr[1825], ptr[1826], ptr[1827], ptr[1828], ptr[1829],
							   ptr[1830], ptr[1831], ptr[1832], ptr[1833], ptr[1834],
							   ptr[1835], ptr[1836], ptr[1837], ptr[1838], ptr[1839],
							   ptr[1840], ptr[1841], ptr[1842], ptr[1843], ptr[1844],
							   ptr[1845], ptr[1846], ptr[1847], ptr[1848], ptr[1849],
							   ptr[1850], ptr[1851], ptr[1852], ptr[1853], ptr[1854],
							   ptr[1855], ptr[1856], ptr[1857], ptr[1858], ptr[1859],
							   ptr[1860], ptr[1861], ptr[1862], ptr[1863], ptr[1864],
							   ptr[1865], ptr[1866], ptr[1867], ptr[1868], ptr[1869],
							   ptr[1870], ptr[1871], ptr[1872], ptr[1873], ptr[1874],
							   ptr[1875], ptr[1876], ptr[1877], ptr[1878], ptr[1879],
							   ptr[1880], ptr[1881], ptr[1882], ptr[1883], ptr[1884],
							   ptr[1885], ptr[1886], ptr[1887], ptr[1888], ptr[1889],
							   ptr[1890], ptr[1891], ptr[1892], ptr[1893], ptr[1894],
							   ptr[1895], ptr[1896], ptr[1897], ptr[1898], ptr[1899],
							   ptr[1900], ptr[1901], ptr[1902], ptr[1903], ptr[1904],
							   ptr[1905], ptr[1906], ptr[1907], ptr[1908], ptr[1909],
							   ptr[1910], ptr[1911], ptr[1912], ptr[1913], ptr[1914],
							   ptr[1915], ptr[1916], ptr[1917], ptr[1918], ptr[1919],
							   ptr[1920], ptr[1921], ptr[1922], ptr[1923], ptr[1924],
							   ptr[1925], ptr[1926], ptr[1927], ptr[1928], ptr[1929],
							   ptr[1930], ptr[1931], ptr[1932], ptr[1933], ptr[1934],
							   ptr[1935], ptr[1936], ptr[1937], ptr[1938], ptr[1939],
							   ptr[1940], ptr[1941], ptr[1942], ptr[1943], ptr[1944],
							   ptr[1945], ptr[1946], ptr[1947], ptr[1948], ptr[1949],
							   ptr[1950], ptr[1951], ptr[1952], ptr[1953], ptr[1954],
							   ptr[1955], ptr[1956], ptr[1957], ptr[1958], ptr[1959],
							   ptr[1960], ptr[1961], ptr[1962], ptr[1963], ptr[1964],
							   ptr[1965], ptr[1966], ptr[1967], ptr[1968], ptr[1969],
							   ptr[1970], ptr[1971], ptr[1972], ptr[1973], ptr[1974],
							   ptr[1975], ptr[1976], ptr[1977], ptr[1978], ptr[1979],
							   ptr[1980], ptr[1981], ptr[1982], ptr[1983], ptr[1984],
							   ptr[1985], ptr[1986], ptr[1987], ptr[1988], ptr[1989],
							   ptr[1990], ptr[1991], ptr[1992], ptr[1993], ptr[1994],
							   ptr[1995], ptr[1996], ptr[1997], ptr[1998], ptr[1999],
							   ptr[2000], ptr[2001], ptr[2002], ptr[2003], ptr[2004],
							   ptr[2005], ptr[2006], ptr[2007], ptr[2008], ptr[2009],
							   ptr[2010], ptr[2011], ptr[2012], ptr[2013], ptr[2014],
							   ptr[2015], ptr[2016], ptr[2017], ptr[2018], ptr[2019],
							   ptr[2020], ptr[2021], ptr[2022], ptr[2023], ptr[2024],
							   ptr[2025], ptr[2026], ptr[2027], ptr[2028], ptr[2029],
							   ptr[2030], ptr[2031], ptr[2032], ptr[2033], ptr[2034],
							   ptr[2035], ptr[2036], ptr[2037], ptr[2038], ptr[2039],
							   ptr[2040], ptr[2041], ptr[2042], ptr[2043], ptr[2044],
							   ptr[2045], ptr[2046], ptr[2047], ptr[2048], ptr[2049],
							   ptr[2050], ptr[2051], ptr[2052], ptr[2053], ptr[2054],
							   ptr[2055], ptr[2056], ptr[2057], ptr[2058], ptr[2059],
							   ptr[2060], ptr[2061], ptr[2062], ptr[2063], ptr[2064],
							   ptr[2065], ptr[2066], ptr[2067], ptr[2068], ptr[2069],
							   ptr[2070], ptr[2071], ptr[2072], ptr[2073], ptr[2074],
							   ptr[2075], ptr[2076], ptr[2077], ptr[2078], ptr[2079],
							   ptr[2080], ptr[2081], ptr[2082], ptr[2083], ptr[2084],
							   ptr[2085], ptr[2086], ptr[2087], ptr[2088], ptr[2089],
							   ptr[2090], ptr[2091], ptr[2092], ptr[2093], ptr[2094],
							   ptr[2095], ptr[2096], ptr[2097], ptr[2098], ptr[2099],
							   ptr[2100], ptr[2101], ptr[2102], ptr[2103], ptr[2104],
							   ptr[2105], ptr[2106], ptr[2107], ptr[2108], ptr[2109],
							   ptr[2110], ptr[2111], ptr[2112], ptr[2113], ptr[2114],
							   ptr[2115], ptr[2116], ptr[2117], ptr[2118], ptr[2119],
							   ptr[2120], ptr[2121], ptr[2122], ptr[2123], ptr[2124],
							   ptr[2125], ptr[2126], ptr[2127], ptr[2128], ptr[2129],
							   ptr[2130], ptr[2131], ptr[2132], ptr[2133], ptr[2134],
							   ptr[2135], ptr[2136], ptr[2137], ptr[2138], ptr[2139],
							   ptr[2140], ptr[2141], ptr[2142], ptr[2143], ptr[2144],
							   ptr[2145], ptr[2146], ptr[2147], ptr[2148], ptr[2149],
							   ptr[2150], ptr[2151], ptr[2152], ptr[2153], ptr[2154],
							   ptr[2155], ptr[2156], ptr[2157], ptr[2158], ptr[2159],
							   ptr[2160], ptr[2161], ptr[2162], ptr[2163], ptr[2164],
							   ptr[2165], ptr[2166], ptr[2167], ptr[2168], ptr[2169],
							   ptr[2170], ptr[2171], ptr[2172], ptr[2173], ptr[2174],
							   ptr[2175], ptr[2176], ptr[2177], ptr[2178], ptr[2179],
							   ptr[2180], ptr[2181], ptr[2182], ptr[2183], ptr[2184],
							   ptr[2185], ptr[2186], ptr[2187], ptr[2188], ptr[2189],
							   ptr[2190], ptr[2191], ptr[2192], ptr[2193], ptr[2194],
							   ptr[2195], ptr[2196], ptr[2197], ptr[2198], ptr[2199],
							   ptr[2200], ptr[2201], ptr[2202], ptr[2203], ptr[2204],
							   ptr[2205], ptr[2206], ptr[2207], ptr[2208], ptr[2209],
							   ptr[2210], ptr[2211], ptr[2212], ptr[2213], ptr[2214],
							   ptr[2215], ptr[2216], ptr[2217], ptr[2218], ptr[2219],
							   ptr[2220], ptr[2221], ptr[2222], ptr[2223], ptr[2224],
							   ptr[2225], ptr[2226], ptr[2227], ptr[2228], ptr[2229],
							   ptr[2230], ptr[2231], ptr[2232], ptr[2233], ptr[2234],
							   ptr[2235], ptr[2236], ptr[2237], ptr[2238], ptr[2239],
							   ptr[2240], ptr[2241], ptr[2242], ptr[2243], ptr[2244],
							   ptr[2245], ptr[2246], ptr[2247], ptr[2248], ptr[2249],
							   ptr[2250], ptr[2251], ptr[2252], ptr[2253], ptr[2254],
							   ptr[2255], ptr[2256], ptr[2257], ptr[2258], ptr[2259],
							   ptr[2260], ptr[2261], ptr[2262], ptr[2263], ptr[2264],
							   ptr[2265], ptr[2266], ptr[2267], ptr[2268], ptr[2269],
							   ptr[2270], ptr[2271], ptr[2272], ptr[2273], ptr[2274],
							   ptr[2275], ptr[2276], ptr[2277], ptr[2278], ptr[2279],
							   ptr[2280], ptr[2281], ptr[2282], ptr[2283], ptr[2284],
							   ptr[2285], ptr[2286], ptr[2287], ptr[2288], ptr[2289],
							   ptr[2290], ptr[2291], ptr[2292], ptr[2293], ptr[2294],
							   ptr[2295], ptr[2296], ptr[2297], ptr[2298], ptr[2299],
							   ptr[2300], ptr[2301], ptr[2302], ptr[2303], ptr[2304],
							   ptr[2305], ptr[2306], ptr[2307], ptr[2308], ptr[2309],
							   ptr[2310], ptr[2311], ptr[2312], ptr[2313], ptr[2314],
							   ptr[2315], ptr[2316], ptr[2317], ptr[2318], ptr[2319],
							   ptr[2320], ptr[2321], ptr[2322], ptr[2323], ptr[2324],
							   ptr[2325], ptr[2326], ptr[2327], ptr[2328], ptr[2329],
							   ptr[2330], ptr[2331], ptr[2332], ptr[2333], ptr[2334],
							   ptr[2335], ptr[2336], ptr[2337], ptr[2338], ptr[2339],
							   ptr[2340], ptr[2341], ptr[2342], ptr[2343], ptr[2344],
							   ptr[2345], ptr[2346], ptr[2347], ptr[2348], ptr[2349],
							   ptr[2350], ptr[2351], ptr[2352], ptr[2353], ptr[2354],
							   ptr[2355], ptr[2356], ptr[2357], ptr[2358], ptr[2359],
							   ptr[2360], ptr[2361], ptr[2362], ptr[2363], ptr[2364],
							   ptr[2365], ptr[2366], ptr[2367], ptr[2368], ptr[2369],
							   ptr[2370], ptr[2371], ptr[2372], ptr[2373], ptr[2374],
							   ptr[2375], ptr[2376], ptr[2377], ptr[2378], ptr[2379],
							   ptr[2380], ptr[2381], ptr[2382], ptr[2383], ptr[2384],
							   ptr[2385], ptr[2386], ptr[2387], ptr[2388], ptr[2389],
							   ptr[2390], ptr[2391], ptr[2392], ptr[2393], ptr[2394],
							   ptr[2395], ptr[2396], ptr[2397], ptr[2398], ptr[2399],
							   ptr[2400], ptr[2401], ptr[2402], ptr[2403], ptr[2404],
							   ptr[2405], ptr[2406], ptr[2407], ptr[2408], ptr[2409],
							   ptr[2410], ptr[2411], ptr[2412], ptr[2413], ptr[2414],
							   ptr[2415], ptr[2416], ptr[2417], ptr[2418], ptr[2419],
							   ptr[2420], ptr[2421], ptr[2422], ptr[2423], ptr[2424],
							   ptr[2425], ptr[2426], ptr[2427], ptr[2428], ptr[2429],
							   ptr[2430], ptr[2431], ptr[2432], ptr[2433], ptr[2434],
							   ptr[2435], ptr[2436], ptr[2437], ptr[2438], ptr[2439],
							   ptr[2440], ptr[2441], ptr[2442], ptr[2443], ptr[2444],
							   ptr[2445], ptr[2446], ptr[2447], ptr[2448], ptr[2449],
							   ptr[2450], ptr[2451], ptr[2452], ptr[2453], ptr[2454],
							   ptr[2455], ptr[2456], ptr[2457], ptr[2458], ptr[2459],
							   ptr[2460], ptr[2461], ptr[2462], ptr[2463], ptr[2464],
							   ptr[2465], ptr[2466], ptr[2467], ptr[2468], ptr[2469],
							   ptr[2470], ptr[2471], ptr[2472], ptr[2473], ptr[2474],
							   ptr[2475], ptr[2476], ptr[2477], ptr[2478], ptr[2479],
							   ptr[2480], ptr[2481], ptr[2482], ptr[2483], ptr[2484],
							   ptr[2485], ptr[2486], ptr[2487], ptr[2488], ptr[2489],
							   ptr[2490], ptr[2491], ptr[2492], ptr[2493], ptr[2494],
							   ptr[2495], ptr[2496], ptr[2497], ptr[2498], ptr[2499],
							   ptr[2500], ptr[2501], ptr[2502], ptr[2503], ptr[2504],
							   ptr[2505], ptr[2506], ptr[2507], ptr[2508], ptr[2509],
							   ptr[2510], ptr[2511], ptr[2512], ptr[2513], ptr[2514],
							   ptr[2515], ptr[2516], ptr[2517], ptr[2518], ptr[2519],
							   ptr[2520], ptr[2521], ptr[2522], ptr[2523], ptr[2524],
							   ptr[2525], ptr[2526], ptr[2527], ptr[2528], ptr[2529],
							   ptr[2530], ptr[2531], ptr[2532], ptr[2533], ptr[2534],
							   ptr[2535], ptr[2536], ptr[2537], ptr[2538], ptr[2539],
							   ptr[2540], ptr[2541], ptr[2542], ptr[2543], ptr[2544],
							   ptr[2545], ptr[2546], ptr[2547], ptr[2548], ptr[2549],
							   ptr[2550], ptr[2551], ptr[2552], ptr[2553], ptr[2554],
							   ptr[2555], ptr[2556], ptr[2557], ptr[2558], ptr[2559],
							   ptr[2560], ptr[2561], ptr[2562], ptr[2563], ptr[2564],
							   ptr[2565], ptr[2566], ptr[2567], ptr[2568], ptr[2569],
							   ptr[2570], ptr[2571], ptr[2572], ptr[2573], ptr[2574],
							   ptr[2575], ptr[2576], ptr[2577], ptr[2578], ptr[2579],
							   ptr[2580], ptr[2581], ptr[2582], ptr[2583], ptr[2584],
							   ptr[2585], ptr[2586], ptr[2587], ptr[2588], ptr[2589],
							   ptr[2590], ptr[2591], ptr[2592], ptr[2593], ptr[2594],
							   ptr[2595], ptr[2596], ptr[2597], ptr[2598], ptr[2599],
							   ptr[2600], ptr[2601], ptr[2602], ptr[2603], ptr[2604],
							   ptr[2605], ptr[2606], ptr[2607], ptr[2608], ptr[2609],
							   ptr[2610], ptr[2611], ptr[2612], ptr[2613], ptr[2614],
							   ptr[2615], ptr[2616], ptr[2617], ptr[2618], ptr[2619],
							   ptr[2620], ptr[2621], ptr[2622], ptr[2623], ptr[2624],
							   ptr[2625], ptr[2626], ptr[2627], ptr[2628], ptr[2629],
							   ptr[2630], ptr[2631], ptr[2632], ptr[2633], ptr[2634],
							   ptr[2635], ptr[2636], ptr[2637], ptr[2638], ptr[2639],
							   ptr[2640], ptr[2641], ptr[2642], ptr[2643], ptr[2644],
							   ptr[2645], ptr[2646], ptr[2647], ptr[2648], ptr[2649],
							   ptr[2650], ptr[2651], ptr[2652], ptr[2653], ptr[2654],
							   ptr[2655], ptr[2656], ptr[2657], ptr[2658], ptr[2659],
							   ptr[2660], ptr[2661], ptr[2662], ptr[2663], ptr[2664],
							   ptr[2665], ptr[2666], ptr[2667], ptr[2668], ptr[2669],
							   ptr[2670], ptr[2671], ptr[2672], ptr[2673], ptr[2674],
							   ptr[2675], ptr[2676], ptr[2677], ptr[2678], ptr[2679],
							   ptr[2680], ptr[2681], ptr[2682], ptr[2683], ptr[2684],
							   ptr[2685], ptr[2686], ptr[2687], ptr[2688], ptr[2689],
							   ptr[2690], ptr[2691], ptr[2692], ptr[2693], ptr[2694],
							   ptr[2695], ptr[2696], ptr[2697], ptr[2698], ptr[2699],
							   ptr[2700], ptr[2701], ptr[2702], ptr[2703], ptr[2704],
							   ptr[2705], ptr[2706], ptr[2707], ptr[2708], ptr[2709],
							   ptr[2710], ptr[2711], ptr[2712], ptr[2713], ptr[2714],
							   ptr[2715], ptr[2716], ptr[2717], ptr[2718], ptr[2719],
							   ptr[2720], ptr[2721], ptr[2722], ptr[2723], ptr[2724],
							   ptr[2725], ptr[2726], ptr[2727], ptr[2728], ptr[2729],
							   ptr[2730], ptr[2731], ptr[2732], ptr[2733], ptr[2734],
							   ptr[2735], ptr[2736], ptr[2737], ptr[2738], ptr[2739],
							   ptr[2740], ptr[2741], ptr[2742], ptr[2743], ptr[2744],
							   ptr[2745], ptr[2746], ptr[2747], ptr[2748], ptr[2749],
							   ptr[2750], ptr[2751], ptr[2752], ptr[2753], ptr[2754],
							   ptr[2755], ptr[2756], ptr[2757], ptr[2758], ptr[2759],
							   ptr[2760], ptr[2761], ptr[2762], ptr[2763], ptr[2764],
							   ptr[2765], ptr[2766], ptr[2767], ptr[2768], ptr[2769],
							   ptr[2770], ptr[2771], ptr[2772], ptr[2773], ptr[2774],
							   ptr[2775], ptr[2776], ptr[2777], ptr[2778], ptr[2779],
							   ptr[2780], ptr[2781], ptr[2782], ptr[2783], ptr[2784],
							   ptr[2785], ptr[2786], ptr[2787], ptr[2788], ptr[2789],
							   ptr[2790], ptr[2791], ptr[2792], ptr[2793], ptr[2794],
							   ptr[2795], ptr[2796], ptr[2797], ptr[2798], ptr[2799],
							   ptr[2800], ptr[2801], ptr[2802], ptr[2803], ptr[2804],
							   ptr[2805], ptr[2806], ptr[2807], ptr[2808], ptr[2809],
							   ptr[2810], ptr[2811], ptr[2812], ptr[2813], ptr[2814],
							   ptr[2815], ptr[2816], ptr[2817], ptr[2818], ptr[2819],
							   ptr[2820], ptr[2821], ptr[2822], ptr[2823], ptr[2824],
							   ptr[2825], ptr[2826], ptr[2827], ptr[2828], ptr[2829],
							   ptr[2830], ptr[2831], ptr[2832], ptr[2833], ptr[2834],
							   ptr[2835], ptr[2836], ptr[2837], ptr[2838], ptr[2839],
							   ptr[2840], ptr[2841], ptr[2842], ptr[2843], ptr[2844],
							   ptr[2845], ptr[2846], ptr[2847], ptr[2848], ptr[2849],
							   ptr[2850], ptr[2851], ptr[2852], ptr[2853], ptr[2854],
							   ptr[2855], ptr[2856], ptr[2857], ptr[2858], ptr[2859],
							   ptr[2860], ptr[2861], ptr[2862], ptr[2863], ptr[2864],
							   ptr[2865], ptr[2866], ptr[2867], ptr[2868], ptr[2869],
							   ptr[2870], ptr[2871], ptr[2872], ptr[2873], ptr[2874],
							   ptr[2875], ptr[2876], ptr[2877], ptr[2878], ptr[2879],
							   ptr[2880], ptr[2881], ptr[2882], ptr[2883], ptr[2884],
							   ptr[2885], ptr[2886], ptr[2887], ptr[2888], ptr[2889],
							   ptr[2890], ptr[2891], ptr[2892], ptr[2893], ptr[2894],
							   ptr[2895], ptr[2896], ptr[2897], ptr[2898], ptr[2899],
							   ptr[2900], ptr[2901], ptr[2902], ptr[2903], ptr[2904],
							   ptr[2905], ptr[2906], ptr[2907], ptr[2908], ptr[2909],
							   ptr[2910], ptr[2911], ptr[2912], ptr[2913], ptr[2914],
							   ptr[2915], ptr[2916], ptr[2917], ptr[2918], ptr[2919],
							   ptr[2920], ptr[2921], ptr[2922], ptr[2923], ptr[2924],
							   ptr[2925], ptr[2926], ptr[2927], ptr[2928], ptr[2929],
							   ptr[2930], ptr[2931], ptr[2932], ptr[2933], ptr[2934],
							   ptr[2935], ptr[2936], ptr[2937], ptr[2938], ptr[2939],
							   ptr[2940], ptr[2941], ptr[2942], ptr[2943], ptr[2944],
							   ptr[2945], ptr[2946], ptr[2947], ptr[2948], ptr[2949],
							   ptr[2950], ptr[2951], ptr[2952], ptr[2953], ptr[2954],
							   ptr[2955], ptr[2956], ptr[2957], ptr[2958], ptr[2959],
							   ptr[2960], ptr[2961], ptr[2962], ptr[2963], ptr[2964],
							   ptr[2965], ptr[2966], ptr[2967], ptr[2968], ptr[2969],
							   ptr[2970], ptr[2971], ptr[2972], ptr[2973], ptr[2974],
							   ptr[2975], ptr[2976], ptr[2977], ptr[2978], ptr[2979],
							   ptr[2980], ptr[2981], ptr[2982], ptr[2983], ptr[2984],
							   ptr[2985], ptr[2986], ptr[2987], ptr[2988], ptr[2989],
							   ptr[2990], ptr[2991], ptr[2992], ptr[2993], ptr[2994],
							   ptr[2995], ptr[2996], ptr[2997], ptr[2998], ptr[2999],
							   ptr[3000], ptr[3001], ptr[3002], ptr[3003], ptr[3004],
							   ptr[3005], ptr[3006], ptr[3007], ptr[3008], ptr[3009],
							   ptr[3010], ptr[3011], ptr[3012], ptr[3013], ptr[3014],
							   ptr[3015], ptr[3016], ptr[3017], ptr[3018], ptr[3019],
							   ptr[3020], ptr[3021], ptr[3022], ptr[3023], ptr[3024],
							   ptr[3025], ptr[3026], ptr[3027], ptr[3028], ptr[3029],
							   ptr[3030], ptr[3031], ptr[3032], ptr[3033], ptr[3034],
							   ptr[3035], ptr[3036], ptr[3037], ptr[3038], ptr[3039],
							   ptr[3040], ptr[3041], ptr[3042], ptr[3043], ptr[3044],
							   ptr[3045], ptr[3046], ptr[3047], ptr[3048], ptr[3049],
							   ptr[3050], ptr[3051], ptr[3052], ptr[3053], ptr[3054],
							   ptr[3055], ptr[3056], ptr[3057], ptr[3058], ptr[3059],
							   ptr[3060], ptr[3061], ptr[3062], ptr[3063], ptr[3064],
							   ptr[3065], ptr[3066], ptr[3067], ptr[3068], ptr[3069],
							   ptr[3070], ptr[3071], ptr[3072], ptr[3073], ptr[3074],
							   ptr[3075], ptr[3076], ptr[3077], ptr[3078], ptr[3079],
							   ptr[3080], ptr[3081], ptr[3082], ptr[3083], ptr[3084],
							   ptr[3085], ptr[3086], ptr[3087], ptr[3088], ptr[3089],
							   ptr[3090], ptr[3091], ptr[3092], ptr[3093], ptr[3094],
							   ptr[3095], ptr[3096], ptr[3097], ptr[3098], ptr[3099],
							   ptr[3100], ptr[3101], ptr[3102], ptr[3103], ptr[3104],
							   ptr[3105], ptr[3106], ptr[3107], ptr[3108], ptr[3109],
							   ptr[3110], ptr[3111], ptr[3112], ptr[3113], ptr[3114],
							   ptr[3115], ptr[3116], ptr[3117], ptr[3118], ptr[3119],
							   ptr[3120], ptr[3121], ptr[3122], ptr[3123], ptr[3124],
							   ptr[3125], ptr[3126], ptr[3127], ptr[3128], ptr[3129],
							   ptr[3130], ptr[3131], ptr[3132], ptr[3133], ptr[3134],
							   ptr[3135], ptr[3136], ptr[3137], ptr[3138], ptr[3139],
							   ptr[3140], ptr[3141], ptr[3142], ptr[3143], ptr[3144],
							   ptr[3145], ptr[3146], ptr[3147], ptr[3148], ptr[3149],
							   ptr[3150], ptr[3151], ptr[3152], ptr[3153], ptr[3154],
							   ptr[3155], ptr[3156], ptr[3157], ptr[3158], ptr[3159],
							   ptr[3160], ptr[3161], ptr[3162], ptr[3163], ptr[3164],
							   ptr[3165], ptr[3166], ptr[3167], ptr[3168], ptr[3169],
							   ptr[3170], ptr[3171], ptr[3172], ptr[3173], ptr[3174],
							   ptr[3175], ptr[3176], ptr[3177], ptr[3178], ptr[3179],
							   ptr[3180], ptr[3181], ptr[3182], ptr[3183], ptr[3184],
							   ptr[3185], ptr[3186], ptr[3187], ptr[3188], ptr[3189],
							   ptr[3190], ptr[3191], ptr[3192], ptr[3193], ptr[3194],
							   ptr[3195], ptr[3196], ptr[3197], ptr[3198], ptr[3199],
							   ptr[3200], ptr[3201], ptr[3202], ptr[3203], ptr[3204],
							   ptr[3205], ptr[3206], ptr[3207], ptr[3208], ptr[3209],
							   ptr[3210], ptr[3211], ptr[3212], ptr[3213], ptr[3214],
							   ptr[3215], ptr[3216], ptr[3217], ptr[3218], ptr[3219],
							   ptr[3220], ptr[3221], ptr[3222], ptr[3223], ptr[3224],
							   ptr[3225], ptr[3226], ptr[3227], ptr[3228], ptr[3229],
							   ptr[3230], ptr[3231], ptr[3232], ptr[3233], ptr[3234],
							   ptr[3235], ptr[3236], ptr[3237], ptr[3238], ptr[3239],
							   ptr[3240], ptr[3241], ptr[3242], ptr[3243], ptr[3244],
							   ptr[3245], ptr[3246], ptr[3247], ptr[3248], ptr[3249],
							   ptr[3250], ptr[3251], ptr[3252], ptr[3253], ptr[3254],
							   ptr[3255], ptr[3256], ptr[3257], ptr[3258], ptr[3259],
							   ptr[3260], ptr[3261], ptr[3262], ptr[3263], ptr[3264],
							   ptr[3265], ptr[3266], ptr[3267], ptr[3268], ptr[3269],
							   ptr[3270], ptr[3271], ptr[3272], ptr[3273], ptr[3274],
							   ptr[3275], ptr[3276], ptr[3277], ptr[3278], ptr[3279],
							   ptr[3280], ptr[3281], ptr[3282], ptr[3283], ptr[3284],
							   ptr[3285], ptr[3286], ptr[3287], ptr[3288], ptr[3289],
							   ptr[3290], ptr[3291], ptr[3292], ptr[3293], ptr[3294],
							   ptr[3295], ptr[3296], ptr[3297], ptr[3298], ptr[3299],
							   ptr[3300], ptr[3301], ptr[3302], ptr[3303], ptr[3304],
							   ptr[3305], ptr[3306], ptr[3307], ptr[3308], ptr[3309],
							   ptr[3310], ptr[3311], ptr[3312], ptr[3313], ptr[3314],
							   ptr[3315], ptr[3316], ptr[3317], ptr[3318], ptr[3319],
							   ptr[3320], ptr[3321], ptr[3322], ptr[3323], ptr[3324],
							   ptr[3325], ptr[3326], ptr[3327], ptr[3328], ptr[3329],
							   ptr[3330], ptr[3331], ptr[3332], ptr[3333], ptr[3334],
							   ptr[3335], ptr[3336], ptr[3337], ptr[3338], ptr[3339],
							   ptr[3340], ptr[3341], ptr[3342], ptr[3343], ptr[3344],
							   ptr[3345], ptr[3346], ptr[3347], ptr[3348], ptr[3349],
							   ptr[3350], ptr[3351], ptr[3352], ptr[3353], ptr[3354],
							   ptr[3355], ptr[3356], ptr[3357], ptr[3358], ptr[3359],
							   ptr[3360], ptr[3361], ptr[3362], ptr[3363], ptr[3364],
							   ptr[3365], ptr[3366], ptr[3367], ptr[3368], ptr[3369],
							   ptr[3370], ptr[3371], ptr[3372], ptr[3373], ptr[3374],
							   ptr[3375], ptr[3376], ptr[3377], ptr[3378], ptr[3379],
							   ptr[3380], ptr[3381], ptr[3382], ptr[3383], ptr[3384],
							   ptr[3385], ptr[3386], ptr[3387], ptr[3388], ptr[3389],
							   ptr[3390], ptr[3391], ptr[3392], ptr[3393], ptr[3394],
							   ptr[3395], ptr[3396], ptr[3397], ptr[3398], ptr[3399],
							   ptr[3400], ptr[3401], ptr[3402], ptr[3403], ptr[3404],
							   ptr[3405], ptr[3406], ptr[3407], ptr[3408], ptr[3409],
							   ptr[3410], ptr[3411], ptr[3412], ptr[3413], ptr[3414],
							   ptr[3415], ptr[3416], ptr[3417], ptr[3418], ptr[3419],
							   ptr[3420], ptr[3421], ptr[3422], ptr[3423], ptr[3424],
							   ptr[3425], ptr[3426], ptr[3427], ptr[3428], ptr[3429],
							   ptr[3430], ptr[3431], ptr[3432], ptr[3433], ptr[3434],
							   ptr[3435], ptr[3436], ptr[3437], ptr[3438], ptr[3439],
							   ptr[3440], ptr[3441], ptr[3442], ptr[3443], ptr[3444],
							   ptr[3445], ptr[3446], ptr[3447], ptr[3448], ptr[3449],
							   ptr[3450], ptr[3451], ptr[3452], ptr[3453], ptr[3454],
							   ptr[3455], ptr[3456], ptr[3457], ptr[3458], ptr[3459],
							   ptr[3460], ptr[3461], ptr[3462], ptr[3463], ptr[3464],
							   ptr[3465], ptr[3466], ptr[3467], ptr[3468], ptr[3469],
							   ptr[3470], ptr[3471], ptr[3472], ptr[3473], ptr[3474],
							   ptr[3475], ptr[3476], ptr[3477], ptr[3478], ptr[3479],
							   ptr[3480], ptr[3481], ptr[3482], ptr[3483], ptr[3484],
							   ptr[3485], ptr[3486], ptr[3487], ptr[3488], ptr[3489],
							   ptr[3490], ptr[3491], ptr[3492], ptr[3493], ptr[3494],
							   ptr[3495], ptr[3496], ptr[3497], ptr[3498], ptr[3499],
							   ptr[3500], ptr[3501], ptr[3502], ptr[3503], ptr[3504],
							   ptr[3505], ptr[3506], ptr[3507], ptr[3508], ptr[3509],
							   ptr[3510], ptr[3511], ptr[3512], ptr[3513], ptr[3514],
							   ptr[3515], ptr[3516], ptr[3517], ptr[3518], ptr[3519],
							   ptr[3520], ptr[3521], ptr[3522], ptr[3523], ptr[3524],
							   ptr[3525], ptr[3526], ptr[3527], ptr[3528], ptr[3529],
							   ptr[3530], ptr[3531], ptr[3532], ptr[3533], ptr[3534],
							   ptr[3535], ptr[3536], ptr[3537], ptr[3538], ptr[3539],
							   ptr[3540], ptr[3541], ptr[3542], ptr[3543], ptr[3544],
							   ptr[3545], ptr[3546], ptr[3547], ptr[3548], ptr[3549],
							   ptr[3550], ptr[3551], ptr[3552], ptr[3553], ptr[3554],
							   ptr[3555], ptr[3556], ptr[3557], ptr[3558], ptr[3559],
							   ptr[3560], ptr[3561], ptr[3562], ptr[3563], ptr[3564],
							   ptr[3565], ptr[3566], ptr[3567], ptr[3568], ptr[3569],
							   ptr[3570], ptr[3571], ptr[3572], ptr[3573], ptr[3574],
							   ptr[3575], ptr[3576], ptr[3577], ptr[3578], ptr[3579],
							   ptr[3580], ptr[3581], ptr[3582], ptr[3583], ptr[3584],
							   ptr[3585], ptr[3586], ptr[3587], ptr[3588], ptr[3589],
							   ptr[3590], ptr[3591], ptr[3592], ptr[3593], ptr[3594],
							   ptr[3595], ptr[3596], ptr[3597], ptr[3598], ptr[3599],
							   ptr[3600], ptr[3601], ptr[3602], ptr[3603], ptr[3604],
							   ptr[3605], ptr[3606], ptr[3607], ptr[3608], ptr[3609],
							   ptr[3610], ptr[3611], ptr[3612], ptr[3613], ptr[3614],
							   ptr[3615], ptr[3616], ptr[3617], ptr[3618], ptr[3619],
							   ptr[3620], ptr[3621], ptr[3622], ptr[3623], ptr[3624],
							   ptr[3625], ptr[3626], ptr[3627], ptr[3628], ptr[3629],
							   ptr[3630], ptr[3631], ptr[3632], ptr[3633], ptr[3634],
							   ptr[3635], ptr[3636], ptr[3637], ptr[3638], ptr[3639],
							   ptr[3640], ptr[3641], ptr[3642], ptr[3643], ptr[3644],
							   ptr[3645], ptr[3646], ptr[3647], ptr[3648], ptr[3649],
							   ptr[3650], ptr[3651], ptr[3652], ptr[3653], ptr[3654],
							   ptr[3655], ptr[3656], ptr[3657], ptr[3658], ptr[3659],
							   ptr[3660], ptr[3661], ptr[3662], ptr[3663], ptr[3664],
							   ptr[3665], ptr[3666], ptr[3667], ptr[3668], ptr[3669],
							   ptr[3670], ptr[3671], ptr[3672], ptr[3673], ptr[3674],
							   ptr[3675], ptr[3676], ptr[3677], ptr[3678], ptr[3679],
							   ptr[3680], ptr[3681], ptr[3682], ptr[3683], ptr[3684],
							   ptr[3685], ptr[3686], ptr[3687], ptr[3688], ptr[3689],
							   ptr[3690], ptr[3691], ptr[3692], ptr[3693], ptr[3694],
							   ptr[3695], ptr[3696], ptr[3697], ptr[3698], ptr[3699],
							   ptr[3700], ptr[3701], ptr[3702], ptr[3703], ptr[3704],
							   ptr[3705], ptr[3706], ptr[3707], ptr[3708], ptr[3709],
							   ptr[3710], ptr[3711], ptr[3712], ptr[3713], ptr[3714],
							   ptr[3715], ptr[3716], ptr[3717], ptr[3718], ptr[3719],
							   ptr[3720], ptr[3721], ptr[3722], ptr[3723], ptr[3724],
							   ptr[3725], ptr[3726], ptr[3727], ptr[3728], ptr[3729],
							   ptr[3730], ptr[3731], ptr[3732], ptr[3733], ptr[3734],
							   ptr[3735], ptr[3736], ptr[3737], ptr[3738], ptr[3739],
							   ptr[3740], ptr[3741], ptr[3742], ptr[3743], ptr[3744],
							   ptr[3745], ptr[3746], ptr[3747], ptr[3748], ptr[3749],
							   ptr[3750], ptr[3751], ptr[3752], ptr[3753], ptr[3754],
							   ptr[3755], ptr[3756], ptr[3757], ptr[3758], ptr[3759],
							   ptr[3760], ptr[3761], ptr[3762], ptr[3763], ptr[3764],
							   ptr[3765], ptr[3766], ptr[3767], ptr[3768], ptr[3769],
							   ptr[3770], ptr[3771], ptr[3772], ptr[3773], ptr[3774],
							   ptr[3775], ptr[3776], ptr[3777], ptr[3778], ptr[3779],
							   ptr[3780], ptr[3781], ptr[3782], ptr[3783], ptr[3784],
							   ptr[3785], ptr[3786], ptr[3787], ptr[3788], ptr[3789],
							   ptr[3790], ptr[3791], ptr[3792], ptr[3793], ptr[3794],
							   ptr[3795], ptr[3796], ptr[3797], ptr[3798], ptr[3799],
							   ptr[3800], ptr[3801], ptr[3802], ptr[3803], ptr[3804],
							   ptr[3805], ptr[3806], ptr[3807], ptr[3808], ptr[3809],
							   ptr[3810], ptr[3811], ptr[3812], ptr[3813], ptr[3814],
							   ptr[3815], ptr[3816], ptr[3817], ptr[3818], ptr[3819],
							   ptr[3820], ptr[3821], ptr[3822], ptr[3823], ptr[3824],
							   ptr[3825], ptr[3826], ptr[3827], ptr[3828], ptr[3829],
							   ptr[3830], ptr[3831], ptr[3832], ptr[3833], ptr[3834],
							   ptr[3835], ptr[3836], ptr[3837], ptr[3838], ptr[3839],
							   ptr[3840], ptr[3841], ptr[3842], ptr[3843], ptr[3844],
							   ptr[3845], ptr[3846], ptr[3847], ptr[3848], ptr[3849],
							   ptr[3850], ptr[3851], ptr[3852], ptr[3853], ptr[3854],
							   ptr[3855], ptr[3856], ptr[3857], ptr[3858], ptr[3859],
							   ptr[3860], ptr[3861], ptr[3862], ptr[3863], ptr[3864],
							   ptr[3865], ptr[3866], ptr[3867], ptr[3868], ptr[3869],
							   ptr[3870], ptr[3871], ptr[3872], ptr[3873], ptr[3874],
							   ptr[3875], ptr[3876], ptr[3877], ptr[3878], ptr[3879],
							   ptr[3880], ptr[3881], ptr[3882], ptr[3883], ptr[3884],
							   ptr[3885], ptr[3886], ptr[3887], ptr[3888], ptr[3889],
							   ptr[3890], ptr[3891], ptr[3892], ptr[3893], ptr[3894],
							   ptr[3895], ptr[3896], ptr[3897], ptr[3898], ptr[3899],
							   ptr[3900], ptr[3901], ptr[3902], ptr[3903], ptr[3904],
							   ptr[3905], ptr[3906], ptr[3907], ptr[3908], ptr[3909],
							   ptr[3910], ptr[3911], ptr[3912], ptr[3913], ptr[3914],
							   ptr[3915], ptr[3916], ptr[3917], ptr[3918], ptr[3919],
							   ptr[3920], ptr[3921], ptr[3922], ptr[3923], ptr[3924],
							   ptr[3925], ptr[3926], ptr[3927], ptr[3928], ptr[3929],
							   ptr[3930], ptr[3931], ptr[3932], ptr[3933], ptr[3934],
							   ptr[3935], ptr[3936], ptr[3937], ptr[3938], ptr[3939],
							   ptr[3940], ptr[3941], ptr[3942], ptr[3943], ptr[3944],
							   ptr[3945], ptr[3946], ptr[3947], ptr[3948], ptr[3949],
							   ptr[3950], ptr[3951], ptr[3952], ptr[3953], ptr[3954],
							   ptr[3955], ptr[3956], ptr[3957], ptr[3958], ptr[3959],
							   ptr[3960], ptr[3961], ptr[3962], ptr[3963], ptr[3964],
							   ptr[3965], ptr[3966], ptr[3967], ptr[3968], ptr[3969],
							   ptr[3970], ptr[3971], ptr[3972], ptr[3973], ptr[3974],
							   ptr[3975], ptr[3976], ptr[3977], ptr[3978], ptr[3979],
							   ptr[3980], ptr[3981], ptr[3982], ptr[3983], ptr[3984],
							   ptr[3985], ptr[3986], ptr[3987], ptr[3988], ptr[3989],
							   ptr[3990], ptr[3991], ptr[3992], ptr[3993], ptr[3994],
							   ptr[3995], ptr[3996], ptr[3997], ptr[3998], ptr[3999]);
	    } else {
		comp[ci].aui_ptr = NULL;
	    }
	    break;
	}
    }

    if (strcasecmp(comp[ci].type, "power_supply") == 0) {
	sig_boolean_connect_out(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 0);
	sig_boolean_connect_out(sig_lookup(comp_id, "mech_power_switch", 0),
				cpssp, 0);
	sig_boolean_set(sig_lookup(comp_id, "power_230v", 0),
			cpssp, 1);
	sig_boolean_set(sig_lookup(comp_id, "mech_power_switch", 0),
			cpssp, 1);

    } else if (strcasecmp(comp[ci].type, "power_supply_at") == 0) {
	sig_boolean_connect_out(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 0);
	sig_boolean_connect_out(sig_lookup(comp_id, "mech_power_switch", 0),
				cpssp, 0);
	sig_boolean_set(sig_lookup(comp_id, "power_230v", 0),
			cpssp, 1);

    } else if (strcasecmp(comp[ci].type, "monitor") == 0) {
	sig_boolean_connect_out(sig_lookup(comp_id, "power_230v", 0),
				cpssp, 1);
	sig_boolean_set(sig_lookup(comp_id, "power_230v", 0),
			cpssp, 1);

    } else if (strcasecmp(comp[ci].type, "telephone_isp") == 0) {
	sig_boolean_connect_out(sig_lookup(comp_id, "switch", 0), cpssp, 0);
	sig_boolean_set(sig_lookup(comp_id, "switch", 0), cpssp, 1);

    } else if (strcasecmp(comp[ci].type, "telephone_switch") == 0) {
	sig_boolean_connect_out(sig_lookup(comp_id, "switch", 0), cpssp, 0);
	sig_boolean_set(sig_lookup(comp_id, "switch", 0), cpssp, 1);
    }

    return 0;
}

int
system_comp_exit(unsigned int comp_id)
{
    unsigned int ci;

    for (ci = 0; ; ci++) {
	if (ci == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (! comp[ci].type
	    || ! comp[ci].name) {
	    continue;
	}
	if (comp[ci].id == comp_id) {
	    /* Component found. */
	    break;
	}
    }

    /* Nothing to do, jet. */

    return 0;
}

int
system_comp_destroy(unsigned int comp_id)
{
    unsigned int ci;
    unsigned int i;

    for (ci = 0; ; ci++) {
	if (ci == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (comp[ci].id == comp_id
	    && comp[ci].type
	    && comp[ci].name) {
	    /* Component found. */
	    break;
	}
    }

    for (i = 0; ; i++) {
	if (i == sizeof(comp_type) / sizeof(comp_type[0])) {
	    fprintf(stderr, "%s\n", comp[ci].type);
	    assert(0); /* FIXME */
	}
	if (strcasecmp(comp[ci].type, comp_type[i].type) == 0) {
	    if (comp_type[i].destroy) {
		comp_type[i].destroy(comp[ci].ptr);
	    }
	    if (comp_type[i].gui_destroy) {
		comp_type[i].gui_destroy(comp[ci].gui_ptr);
	    }
	    if (comp_type[i].aui_destroy) {
		comp_type[i].aui_destroy(comp[ci].aui_ptr);
	    }
	    break;
	}
    }

    free(comp[ci].type);
    comp[ci].type = NULL;

    free(comp[ci].name);
    comp[ci].name = NULL;

    return 0;
}

int
system_comp_info(
		 unsigned int comp_id,
		 char *type,
		 char *name,
		 unsigned int *nodep,
		 unsigned int *pagep
		)
{
    unsigned int ci;

    for (ci = 0; ; ci++) {
	if (ci == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    assert(0); /* FIXME */
	    return -1;
	}
	if (! comp[ci].type
	    || ! comp[ci].name) {
	    continue;
	}
	if (comp[ci].id == comp_id) {
	    /* Component found. */
	    break;
	}
    }

    strcpy(type, comp[ci].type);
    strcpy(name, comp[ci].name);
    *nodep = comp[ci].node;
    *pagep = comp[ci].page;

    return 0;
}

int
system_comp_generic_info(
			 unsigned int comp_id,
			 unsigned int generic_id,
			 char *type,
			 char *name,
			 char *value
			)
{
    unsigned int id;

    for (id = 0; ; id++) {
	if (id == sizeof(generic) / sizeof(generic[0])) {
	    /* Not found. */
	    return -1;
	}
	if (! generic[id].name
	    || ! generic[id].value) {
	    continue;
	}
	if (generic[id].comp_id == comp_id) {
	    if (generic_id == 0) {
		strcpy(type, generic[id].type);
		strcpy(name, generic[id].name);
		strcpy(value, generic[id].value);
		return 0;
	    } else {
		generic_id--;
	    }
	}
    }
}

int
system_comp_port_info(
		      unsigned int comp_id,
		      unsigned int port_id,
		      char *port,
		      unsigned int *sigidp
		     )
{
    unsigned int id;

    for (id = 0; ; id++) {
	if (id == sizeof(conn) / sizeof(conn[0])) {
	    /* Not found. */
	    return -1;
	}
	if (! conn[id].port) {
	    continue;
	}
	if (conn[id].comp_id == comp_id) {
	    if (port_id == 0) {
		strcpy(port, conn[id].port);
		*sigidp = conn[id].sig_id;
		return 0;
	    } else {
		port_id--;
	    }
	}
    }
}

int
system_arch_create(
		   const char *type,
		   const char *name,
		   unsigned int node,
		   unsigned int page
		  )
{
    return system_comp_create(type, name, node, page);
}

int
system_arch_generic_set(
			unsigned int id,
			const char *t,
			const char *n,
			const char *v
		       )
{
    return system_comp_generic_set(id, t, n, v);
}

int
system_arch_port_connect(
			 unsigned int arch_id,
			 const char *port,
			 unsigned int sig_id
			)
{
    return system_comp_port_connect(arch_id, port, sig_id);
}

int
system_arch_init(unsigned int id)
{
    return system_comp_init(id);
}

int
system_arch_exit(unsigned int id)
{
    return system_comp_exit(id);
}

int
system_arch_destroy(unsigned int id)
{
    return system_comp_destroy(id);
}

void
system_comp_connect(unsigned int comp_id, const char *port, unsigned int sig_id)
{
    struct sig_manage *sig_manage;
    void *sig_ptr;

    sig_manage = sig_lookup(comp_id, "manage", 0);
    assert(sig_manage);

    sig_ptr = system_sig_get(sig_id);
    assert(sig_ptr);

    sig_manage_connect_port(sig_manage, port, sig_ptr);

    /* FIXME */
    // ret = system_sig_unget(sig_ptr);
    // assert(0 <= ret);
}

void
system_comp_disconnect(unsigned int comp_id, const char *port)
{
    struct sig_manage *sig_manage;

    sig_manage = sig_lookup(comp_id, "manage", 0);
    assert(sig_manage);

    sig_manage_disconnect_port(sig_manage, port);
}

int
system_comp_lookup(const char *name)
{
    unsigned int id;

    for (id = 0; id < sizeof(comp) / sizeof(comp[0]); id++) {
	if (comp[id].name == NULL) {
	    /* past last entry */
	    return -1;
	}

	if (strcasecmp(comp[id].name, name) == 0) {
	    return comp[id].id;
	}
    }

    return -1;
}

int
system_sig_lookup(const char *name)
{
    unsigned int si;

    for (si = 0; ; si++) {
	if (si == sizeof(sig) / sizeof(sig[0])) {
	    /* Signal not found. */
	    return -1;
	}

	if (sig[si].name
	    && strcasecmp(sig[si].name, name) == 0) {
	    break;
	}
    }

    return sig[si].id;
}

int
system_port_lookup(
		   const char *type,
		   enum system_gender gender,
		   unsigned int n,
		   const char **comp_name,
		   const char **port_name
		  )
{
    unsigned int conn_id;
    unsigned int comp_id;
    unsigned int sig_id;

    for (conn_id = 0; ; conn_id++) {
	if (conn_id == sizeof(conn) / sizeof(conn[0])) {
	    return -1;
	}
	if (! conn[conn_id].port) {
	    /* Empty slot. */
	    continue;
	}
	if (conn[conn_id].gender != gender) {
	    /* Wrong gender. */
	    continue;
	}
	for (sig_id = 0; ; sig_id++) {
	    assert(sig_id < sizeof(sig) / sizeof(sig[0]));
	    if (! sig[sig_id].type
		|| ! sig[sig_id].name) {
		continue;
	    }
	    if (sig[sig_id].id == conn[conn_id].sig_id) {
		break;
	    }
	}
	if (strcasecmp(sig[sig_id].type, type) != 0) {
	    /* Wrong type. */
	    continue;
	}
	for (comp_id = 0; ; comp_id++) {
	    assert(comp_id < sizeof(comp) / sizeof(comp[0]));
	    if (! comp[comp_id].type
		|| ! comp[comp_id].name) {
		continue;
	    }
	    if (comp[comp_id].id == conn[conn_id].comp_id) {
		break;
	    }
	}

	if (n == 0) {
	    *comp_name = comp[comp_id].name;
	    *port_name = conn[conn_id].port;
	    return 1;
	} else {
	    n--;
	}
    }
}

int
system_port_connect(
		    const char *comp_name0,
		    const char *port_name0,
		    const char *comp_name1,
		    const char *port_name1
		   )
{
    unsigned int comp_id0;
    unsigned int comp_id1;
    unsigned int conn_id0;
    unsigned int conn_id1;
    unsigned int sig_id0;
    unsigned int sig_id1;
    unsigned int type_id;

    /*
     * Lookup component 0.
     */
    for (comp_id0 = 0; ; comp_id0++) {
	if (comp_id0 == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    return -1;
	}
	if (! comp[comp_id0].type
	    || ! comp[comp_id0].name) {
	    /* Empty slot. */
	    continue;
	}
	if (strcasecmp(comp[comp_id0].name, comp_name0) == 0) {
	    /* Component found. */
	    break;
	}
    }

    /*
     * Lookup component 1.
     */
    for (comp_id1 = 0; ; comp_id1++) {
	if (comp_id1 == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    return -1;
	}
	if (! comp[comp_id1].type
	    || ! comp[comp_id1].name) {
	    /* Empty slot. */
	    continue;
	}
	if (strcasecmp(comp[comp_id1].name, comp_name1) == 0) {
	    /* Component found. */
	    break;
	}
    }

    /*
     * Lookup port 0.
     */
    for (conn_id0 = 0; ; conn_id0++) {
	if (conn_id0 == sizeof(conn) / sizeof(conn[0])) {
	    /* Connection not found. */
	    return -1;
	}
	if (! conn[conn_id0].port) {
	    /* Empty slot. */
	    continue;
	}
	if (conn[conn_id0].comp_id == comp[comp_id0].id
	    && strcasecmp(conn[conn_id0].port, port_name0) == 0) {
	    /* Connection found. */
	    break;
	}
    }

    /*
     * Lookup port 1.
     */
    for (conn_id1 = 0; ; conn_id1++) {
	if (conn_id1 == sizeof(conn) / sizeof(conn[0])) {
	    /* Connection not found. */
	    return -1;
	}
	if (! conn[conn_id1].port) {
	    /* Empty slot. */
	    continue;
	}
	if (conn[conn_id1].comp_id == comp[comp_id1].id
	    && strcasecmp(conn[conn_id1].port, port_name1) == 0) {
	    /* Connection found. */
	    break;
	}
    }

    /*
     * Lookup signal 0.
     */
    for (sig_id0 = 0; ; sig_id0++) {
	if (sig_id0 == sizeof(sig) / sizeof(sig[0])) {
	    /* Signal not found. */
	    assert(0); /* Mustn't happen. */
	}
	if (! sig[sig_id0].type
	    || ! sig[sig_id0].name) {
	    /* Empty slot. */
	    continue;
	}
	if (sig[sig_id0].id == conn[conn_id0].sig_id) {
	    /* Signal found. */
	    break;
	}
    }

    /*
     * Lookup signal 1.
     */
    for (sig_id1 = 0; ; sig_id1++) {
	if (sig_id1 == sizeof(sig) / sizeof(sig[0])) {
	    /* Signal not found. */
	    assert(0); /* Mustn't happen. */
	}
	if (! sig[sig_id1].type
	    || ! sig[sig_id1].name) {
	    /* Empty slot. */
	    continue;
	}
	if (sig[sig_id1].id == conn[conn_id1].sig_id) {
	    /* Signal found. */
	    break;
	}
    }

    /*
     * Lookup signal type.
     */
    for (type_id = 0; ; type_id++) {
	if (type_id == sizeof(sig_type) / sizeof(sig_type[0])) {
	    /* Signal type not found. */
	    assert(0); /* Mustn't happen. */
	}
	if (strcasecmp(sig_type[type_id].type, sig[sig_id0].type) == 0) {
	    /* Signal type found. */
	    break;
	}
    }

    /*
     * Check if signals connectable.
     */
    if (conn[conn_id0].peer != -1
	|| conn[conn_id1].peer != -1) {
	/* Port already connected. */
	return -1;
    }
    if (strcasecmp(sig[sig_id0].type, sig[sig_id1].type) != 0) {
	/* Signals of different type. */
	return -1;
    }
    if ((conn[conn_id0].gender | conn[conn_id1].gender) != 0
	&& (conn[conn_id0].gender | conn[conn_id1].gender) != 3) {
	/* Wrong genders. */
	return -1;
    }

    /*
     * Connect signals.
     */
    assert(conn[conn_id0].sig_ptr); /* FIXME */
    assert(conn[conn_id1].sig_ptr); /* FIXME */
    assert(! conn[conn_id0].merge);
    assert(! conn[conn_id1].merge);
    assert(conn[conn_id0].peer == -1);
    assert(conn[conn_id1].peer == -1);

    conn[conn_id0].peer = conn_id1;
    conn[conn_id1].peer = conn_id0;

    conn[conn_id0].merge =
	conn[conn_id1].merge = (*sig_type[type_id].merge)(
							  conn[conn_id0].sig_ptr, conn[conn_id1].sig_ptr);

    return 0;
}

int
system_port_disconnect(
		       const char *comp_name0,
		       const char *port_name0
		      )
{
    unsigned int comp_id0;
    unsigned int conn_id0;
    unsigned int conn_id1;
    unsigned int sig_id0;
    unsigned int type_id;

    /*
     * Lookup component 0.
     */
    for (comp_id0 = 0; ; comp_id0++) {
	if (comp_id0 == sizeof(comp) / sizeof(comp[0])) {
	    /* Component not found. */
	    return -1;
	}
	if (! comp[comp_id0].type
	    || ! comp[comp_id0].name) {
	    /* Empty slot. */
	    continue;
	}
	if (strcasecmp(comp[comp_id0].name, comp_name0) == 0) {
	    /* Component found. */
	    break;
	}
    }

    /*
     * Lookup port 0.
     */
    for (conn_id0 = 0; ; conn_id0++) {
	if (conn_id0 == sizeof(conn) / sizeof(conn[0])) {
	    /* Connection not found. */
	    return -1;
	}
	if (! conn[conn_id0].port) {
	    /* Empty slot. */
	    continue;
	}
	if (conn[conn_id0].comp_id == comp[comp_id0].id
	    && strcasecmp(conn[conn_id0].port, port_name0) == 0) {
	    /* Connection found. */
	    break;
	}
    }

    /*
     * Lookup signal 0.
     */
    for (sig_id0 = 0; ; sig_id0++) {
	if (sig_id0 == sizeof(sig) / sizeof(sig[0])) {
	    /* Signal not found. */
	    assert(0); /* Mustn't happen. */
	}
	if (! sig[sig_id0].type
	    || ! sig[sig_id0].name) {
	    /* Empty slot. */
	    continue;
	}
	if (sig[sig_id0].id == conn[conn_id0].sig_id) {
	    /* Signal found. */
	    break;
	}
    }

    /*
     * Lookup signal type.
     */
    for (type_id = 0; ; type_id++) {
	if (type_id == sizeof(sig_type) / sizeof(sig_type[0])) {
	    /* Signal type not found. */
	    assert(0); /* Mustn't happen. */
	}
	if (strcasecmp(sig_type[type_id].type, sig[sig_id0].type) == 0) {
	    /* Signal type found. */
	    break;
	}
    }

    /*
     * Checks if signal disconnectable.
     */
    if (conn[conn_id0].peer == -1) {
	/* Not connected. */
	return -1;
    }

    /*
     * Disconnect.
     */
    conn_id1 = conn[conn_id0].peer;
    assert(conn[conn_id1].peer == conn_id0);

    assert(conn_id0 < sizeof(conn) / sizeof(conn[0]));
    assert(conn_id1 < sizeof(conn) / sizeof(conn[0]));
    assert(conn[conn_id0].merge);
    assert(conn[conn_id1].merge);

    (*sig_type[type_id].split)(conn[conn_id0].merge);

    conn[conn_id0].merge = NULL;
    conn[conn_id0].peer = -1;
    conn[conn_id1].merge = NULL;
    conn[conn_id1].peer = -1;

    return 0;
}

void 
suspend_components(FILE *fComp)
{
    int i, j;
    char compname[100];
    int component_found;	

    for (i = 0; i < sizeof(comp)/sizeof(comp[0]) ; i++) {

	if (!comp[i].name){
	    continue;
	}

	component_found = 0;

	for (j = 0; j < sizeof(comp_type)/sizeof(comp_type[0]); j++) {

	    if (strcasecmp(comp[i].type, comp_type[j].type) == 0) {
		strcpy(compname, comp[i].name);
		fwrite(compname, sizeof(compname), 1, fComp);
		memset(&compname[0], 0, sizeof(compname));

		if (comp_type[j].suspend) {
		    fprintf(stderr, "\t\tSuspend %s at %p\n", comp_type[j].type, comp[i].ptr);
		    comp_type[j].suspend(comp[i].ptr, fComp);
		}

		if (comp_type[j].gui_suspend) {
		    fprintf(stderr, "\t\tSuspend GUI of %s at %p\n", comp_type[j].type, comp[i].gui_ptr);
		    comp_type[j].gui_suspend(comp[i].gui_ptr, fComp);
		}

		if (comp_type[j].aui_suspend) {
		    fprintf(stderr, "\t\tSuspend AUI of %s at %p\n", comp_type[j].type, comp[i].aui_ptr);
		    comp_type[j].aui_suspend(comp[i].aui_ptr, fComp);
		}

		component_found = 1;
		break;
	    }	
	}

	if (!component_found){
	    fprintf(stderr, "Type %s of component %s not found... aborting\n", comp[i].type, comp[i].name);
	    assert(0);
	}
    }

    strcpy(compname, "SUSP_END");
    fwrite(compname, sizeof(compname), 1, fComp);
}

void 
resume_components(FILE *fComp)
{
    int i, j;
    int read_chars;
    char compname[100];	

    while (1)
    {
	read_chars = fread(compname, sizeof(compname), 1, fComp);
	if (strcasecmp("SUSP_END", compname) == 0) {
	    break;
	}

	for (i = 0; i < sizeof(comp)/sizeof(comp[0]); i++) {

	    if (strcasecmp(comp[i].name, compname) == 0) {

		for (j = 0; j < (sizeof(comp_type)/sizeof(comp_type[0])); j++) {

		    if (strcasecmp(comp[i].type, comp_type[j].type) == 0) {

			if (comp_type[j].resume) {

			    fprintf(stderr, "\t\tResume %s at %p\n", comp_type[j].type, comp[i].ptr);

			    comp_type[j].resume(comp[i].ptr, fComp);
			}
			if (comp_type[j].gui_resume) {

			    fprintf(stderr, "\t\tResume GUI of %s at %p\n", comp_type[j].type, comp[i].gui_ptr);

			    comp_type[j].gui_resume(comp[i].gui_ptr, fComp);
			}
			if (comp_type[j].aui_resume) {

			    fprintf(stderr, "\t\tResume AUI of %s at %p\n", comp_type[j].type, comp[i].aui_ptr);

			    comp_type[j].aui_resume(comp[i].aui_ptr, fComp);
			}
			break;
		    }
		}
		break;
	    }
	}
    }
}

void 
suspend_signals(FILE *fSig)
{
    int i, j;
    char signame[100];
    int signal_found;	

    for (i = 0; i < sizeof(sig)/sizeof(sig[0]) ; i++) {

	if (!sig[i].name){
	    continue;
	}

	signal_found = 0;

	for (j = 0; j < sizeof(sig_type)/sizeof(sig_type[0]); j++) {

	    if (strcasecmp(sig[i].type, sig_type[j].type) == 0) {
		strcpy(signame, sig[i].name);
		fwrite(signame, sizeof(signame), 1, fSig);
		memset(&signame[0], 0, sizeof(signame));

		if (sig_type[j].suspend) {
		    fprintf(stderr, "\t\tSuspend %s at %p\n", sig_type[j].type, sig[i].ptr);
		    sig_type[j].suspend(sig[i].ptr, fSig);
		}				
		signal_found = 1;
		break;
	    }	
	}

	if (!signal_found){
	    fprintf(stderr, "Type %s of signal %s not found... aborting\n", sig[i].type, sig[i].name);
	    assert(0);
	}
    }

    strcpy(signame, "SUSP_END");
    fwrite(signame, sizeof(signame), 1, fSig);
}

void
resume_signals(FILE *fSig)
{
    int i, j;
    int read_chars;
    char signame[100];	

    while (1)
    {
	read_chars = fread(signame, sizeof(signame), 1, fSig);
	if (strcasecmp("SUSP_END", signame) == 0) {
	    break;
	}

	for (i = 0; i < sizeof(sig)/sizeof(sig[0]); i++) {

	    if (strcasecmp(sig[i].name, signame) == 0) {

		for (j = 0; j < (sizeof(sig_type)/sizeof(sig_type[0])); j++) {

		    if (strcasecmp(sig[i].type, sig_type[j].type) == 0) {

			if (sig_type[j].resume) {

			    fprintf(stderr, "\t\tResume %s at %p\n", sig_type[j].type, sig[i].ptr);

			    sig_type[j].resume(sig[i].ptr, fSig);
			}
			break;
		    }
		}
		break;
	    }
	}
    }
}

void 
suspend_connections(FILE *fConn)
{
    int i;
    size_t writecheck;

    for (i = 0; i < sizeof(conn)/sizeof(conn[0]) ; i++) {

	writecheck = fwrite(&conn[i].comp_id, sizeof(conn[i].comp_id), 1, fConn);
	writecheck += fwrite(&conn[i].sig_id, sizeof(conn[i].sig_id), 1, fConn);
	writecheck += fwrite(&conn[i].gender, sizeof(conn[i].gender), 1, fConn);
	writecheck += fwrite(&conn[i].peer, sizeof(conn[i].peer), 1, fConn);

	if (writecheck != 4){
	    fprintf(stderr, "fwrite in suspend_connections failed\n");
	    return;
	}
    }
}

void 
resume_connections(FILE *fConn)
{
    int i, j;
    size_t readcheck;
    unsigned int comp_id;
    unsigned int sig_id;
    enum system_gender gender;
    int peer;


    for (i = 0; i < sizeof(conn)/sizeof(conn[0]) ; i++) {

	readcheck = fread(&comp_id, sizeof(conn[0].comp_id), 1, fConn);
	readcheck += fread(&sig_id, sizeof(conn[0].sig_id), 1, fConn);
	readcheck += fread(&gender, sizeof(conn[0].gender), 1, fConn);
	readcheck += fread(&peer, sizeof(conn[0].peer), 1, fConn);

	if (readcheck != 4){
	    fprintf(stderr, "fread in resume_connections failed\n");
	    return;
	}



	if (peer != conn[i].peer && sig[sig_id].type){
	    assert(conn[i].merge == 0);

	    for (j = 0; j < (sizeof(sig_type)/sizeof(sig_type[0])); j++) {

		if (strcasecmp(sig[sig_id].type, sig_type[j].type) == 0) {

		    fprintf(stderr, "\t\tMerge %s \n", sig_type[j].type);

		    conn[i].peer = peer;
		    conn[i].merge = (*sig_type[j].merge)(
							 conn[i].sig_ptr, conn[peer].sig_ptr);

		    conn[peer].peer = i;
		    conn[peer].merge = conn[i].merge;
		    break;
		}
	    }
	}

	if (comp_id != conn[i].comp_id ||
	    sig_id != conn[i].sig_id ||
	    gender != conn[i].gender ||
	    peer != conn[i].peer){

	    fprintf(stderr, "Connection difference detected\n\n");
	    fprintf(stderr, "comp_id = %u conn[%d].comp_id = %u\n", comp_id, i, conn[i].comp_id);
	    fprintf(stderr, "sig_id = %u conn[%d].sig_id = %u\n", sig_id, i, conn[i].sig_id);
	    fprintf(stderr, "gender = %d conn[%d].gender = %d\n", gender, i, conn[i].gender);
	    fprintf(stderr, "peer = %d conn[%d].peer = %d\n", peer, i, conn[i].peer);
	    fprintf(stderr, "conn[%d].port = %s\n", i, conn[i].port);
	    fprintf(stderr, "conn[%d].sig_ptr = %p\n", i, conn[i].sig_ptr);
	    fprintf(stderr, "conn[%d].merge = %p\n", i, conn[i].merge);
	}
    }
}

void
system_suspend(void)
{
    FILE *fComp;
    FILE *fSig;
    FILE *fConn;
    FILE *fSched;

    fComp = fopen("./Components.suspend", "wb");
    fSig = fopen("./Signals.suspend", "wb");
    fConn = fopen("./Connections.suspend", "wb");
    fSched = fopen("./Schedules.suspend", "wb");

    if (fComp == NULL){
	fprintf(stderr, "\nCreating Components.suspend failed\n");
	return;
    }
    if (fSig == NULL){
	fprintf(stderr, "\nCreating Signals.suspend failed\n");
	return;
    }
    if (fConn == NULL){
	fprintf(stderr, "\nCreating Connections.suspend failed\n");
	return;
    }
    if (fSched == NULL){
	fprintf(stderr, "\nCreating Schedules.suspend failed\n");
	return;
    }

    suspend_components(fComp);
    suspend_signals(fSig);
    suspend_connections(fConn);
    suspend_schedules(fSched);

    fclose(fComp);
    fclose(fSig);
    fclose(fConn);
    fclose(fSched);
}

void
system_resume(void)
{
    FILE *fComp;
    FILE *fSig;
    FILE *fConn;
    FILE *fSched;

    fComp = fopen("./Components.suspend", "rb");
    fSig = fopen("./Signals.suspend", "rb");
    fConn = fopen("./Connections.suspend", "rb");
    fSched = fopen("./Schedules.suspend", "rb");

    if (fComp == NULL){
	fprintf(stderr, "\nOpening Components.suspend failed\n");
	return;
    }

    if (fSig == NULL){
	fprintf(stderr, "\nOpening Signals.suspend failed\n");
	return;
    }
    if (fConn == NULL){
	fprintf(stderr, "\nOpening Connections.suspend failed\n");
	return;
    }

    if (fSched == NULL){
	fprintf(stderr, "\nOpening Schedules.suspend failed\n");
	return;
    }

    resume_components(fComp);
    fprintf(stderr, "\nResume components complete\n");
    resume_signals(fSig);
    fprintf(stderr, "\nResume signals complete\n");
    resume_connections(fConn);
    fprintf(stderr, "\nResume connections complete\n");
    resume_schedules(fSched);
    fprintf(stderr, "\nResume schedules complete\n");
    fprintf(stderr, "\nResume complete\n");

    fclose(fComp);
    fclose(fSig);
    fclose(fConn);
    fclose(fSched);
}
