/*
 * Copyright (C) 2003-2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue-main.h"
#include "glue-shm.h"
#include "glue-suspend.h"

#include "chip_gen_sound.h"

#define COMP_(x) chip_gen_sound_ ## x


struct cpssp {
	struct sig_std_logic *port_rc_time_0;
	struct sig_std_logic *port_rc_time_1;
	struct sig_std_logic *port_rc_time_2;
	struct sig_std_logic *port_rc_time_3;
	struct sig_std_logic *port_volume_0;
	struct sig_std_logic *port_volume_1;
	struct sig_std_logic *port_volume_2;
	struct sig_std_logic *port_volume_3;
	struct sig_std_logic *port_ton_out;
	/* bit 0-2: note
	 * bit 3: currently unused
	 * bit 4-5: octave
	 * bit 6: output on/off
	 * bit 7: generator on/off
	 * bit 8-9: volume
	 */
	unsigned int state_ton_in;
	/* boolean value of output */
	unsigned int state_ton_out;
};

static const unsigned long long note_intervals[] = {
	TIME_HZ / 132.0, /* C0 */
	TIME_HZ / 148.5, /* D0 */
	TIME_HZ / 165.0, /* E0 */
	TIME_HZ / 176.0, /* F0 */
	TIME_HZ / 198.0, /* G0 */
	TIME_HZ / 220.0, /* A0 */
	TIME_HZ / 247.5, /* H0 */
	TIME_HZ / 247.5  /* H0 */
};

static void
COMP_(reset)(struct cpssp *cpssp)
{
	cpssp->state_ton_in = 0;
	cpssp->state_ton_out = 0;
}

static void
COMP_(rstn_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	switch (val) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L:
		COMP_(reset)(cpssp);
		break;

	default:
		break;
	}
}

static unsigned long long
COMP_(calculate_delay)(const struct cpssp *cpssp)
{
	unsigned long long delay;

	delay = note_intervals[cpssp->state_ton_in & 7];
	delay >>= ((cpssp->state_ton_in >> 4) & 3) + 1;
	assert(0 < delay);
	return delay;
}

static int
COMP_(output_enabled)(const struct cpssp *cpssp)
{
	return cpssp->state_ton_in & (1 << 6);
}

static int
COMP_(generator_enabled)(const struct cpssp *cpssp)
{
	return cpssp->state_ton_in & (1 << 7);
}

static void
COMP_(tick)(void *_cpssp)
{
	unsigned int out_val;
	unsigned long long delay;
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	cpssp->state_ton_out = ! cpssp->state_ton_out;

	if (cpssp->state_ton_out) {
		out_val = SIG_STD_LOGIC_1;
	} else {
		out_val = SIG_STD_LOGIC_0;
	}
		
	if (COMP_(output_enabled)(cpssp)) {
		sig_std_logic_set(cpssp->port_ton_out, cpssp, out_val);
	}

	delay = COMP_(calculate_delay)(cpssp);

	if (COMP_(generator_enabled)(cpssp)) {
		time_call_after(delay, COMP_(tick), cpssp);
	}
}

static void
COMP_(vol_out_update)(struct cpssp *cpssp)
{
	const unsigned int volume = (cpssp->state_ton_in >> 8) & 3;

	if (volume == 0) {
		sig_std_logic_set(cpssp->port_volume_0, cpssp, SIG_STD_LOGIC_0);
	} else {
		sig_std_logic_set(cpssp->port_volume_0, cpssp, SIG_STD_LOGIC_Z);
	}

	if (volume == 1) {
		sig_std_logic_set(cpssp->port_volume_1, cpssp, SIG_STD_LOGIC_0);
	} else {
		sig_std_logic_set(cpssp->port_volume_1, cpssp, SIG_STD_LOGIC_Z);
	}

	if (volume == 2) {
		sig_std_logic_set(cpssp->port_volume_2, cpssp, SIG_STD_LOGIC_0);
	} else {
		sig_std_logic_set(cpssp->port_volume_2, cpssp, SIG_STD_LOGIC_Z);
	}

	if (volume == 3) {
		sig_std_logic_set(cpssp->port_volume_3, cpssp, SIG_STD_LOGIC_0);
	} else {
		sig_std_logic_set(cpssp->port_volume_3, cpssp, SIG_STD_LOGIC_Z);
	}
}

static void
COMP_(note_in_change)(
	struct cpssp *cpssp, 
	unsigned int new_state,
	unsigned int bit_pos
)
{
	unsigned long long delay;

	switch (new_state) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L:
		cpssp->state_ton_in &= ~(1 << bit_pos);
		break;

	case SIG_STD_LOGIC_Z:
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_1:
		cpssp->state_ton_in |= 1 << bit_pos;
		break;

	default:
		/* do nothing */
		return;
	}

	delay = COMP_(calculate_delay)(cpssp);

	if (COMP_(generator_enabled)(cpssp)) {
		time_call_delete(COMP_(tick), cpssp);
		time_call_after(delay, COMP_(tick), cpssp);
	}

	COMP_(vol_out_update)(cpssp);
}

static void
COMP_(ton_in_0_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 0);
}

static void
COMP_(ton_in_1_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 1);
}

static void
COMP_(ton_in_2_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 2);
}

static void
COMP_(ton_in_3_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 3);
}

static void
COMP_(ton_in_4_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 4);
}

static void
COMP_(ton_in_5_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 5);
}

static void
COMP_(ton_in_6_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 6);
}

static void
COMP_(ton_in_7_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 7);
}

static void
COMP_(ton_in_8_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 8);
}

static void
COMP_(ton_in_9_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 9);
}

static void
COMP_(ton_in_10_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 10);
}

static void
COMP_(ton_in_11_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 11);
}

static void
COMP_(ton_in_12_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 12);
}

static void
COMP_(ton_in_13_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 13);
}

static void
COMP_(ton_in_14_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 14);
}

static void
COMP_(ton_in_15_set)(void *_cpssp, unsigned int val)
{
	COMP_(note_in_change)((struct cpssp *)_cpssp, val, 15);
}

void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_ton_in_0,
	struct sig_std_logic *port_ton_in_1,
	struct sig_std_logic *port_ton_in_2,
	struct sig_std_logic *port_ton_in_3,
	struct sig_std_logic *port_ton_in_4,
	struct sig_std_logic *port_ton_in_5,
	struct sig_std_logic *port_ton_in_6,
	struct sig_std_logic *port_ton_in_7,
	struct sig_std_logic *port_ton_in_8,
	struct sig_std_logic *port_ton_in_9,
	struct sig_std_logic *port_ton_in_10,
	struct sig_std_logic *port_ton_in_11,
	struct sig_std_logic *port_ton_in_12,
	struct sig_std_logic *port_ton_in_13,
	struct sig_std_logic *port_ton_in_14,
	struct sig_std_logic *port_ton_in_15,
	struct sig_std_logic *port_pci_clock,
	struct sig_std_logic *port_rstn,
	struct sig_std_logic *port_rc_time_0,
	struct sig_std_logic *port_rc_time_1,
	struct sig_std_logic *port_rc_time_2,
	struct sig_std_logic *port_rc_time_3,
	struct sig_std_logic *port_volume_0,
	struct sig_std_logic *port_volume_1,
	struct sig_std_logic *port_volume_2,
	struct sig_std_logic *port_volume_3,
	struct sig_std_logic *port_ton_out
)
{
	static const struct sig_std_logic_funcs ton_in_0_funcs = {
		.std_logic_set = COMP_(ton_in_0_set)
	};
	static const struct sig_std_logic_funcs ton_in_1_funcs = {
		.std_logic_set = COMP_(ton_in_1_set)
	};
	static const struct sig_std_logic_funcs ton_in_2_funcs = {
		.std_logic_set = COMP_(ton_in_2_set)
	};
	static const struct sig_std_logic_funcs ton_in_3_funcs = {
		.std_logic_set = COMP_(ton_in_3_set)
	};
	static const struct sig_std_logic_funcs ton_in_4_funcs = {
		.std_logic_set = COMP_(ton_in_4_set)
	};
	static const struct sig_std_logic_funcs ton_in_5_funcs = {
		.std_logic_set = COMP_(ton_in_5_set)
	};
	static const struct sig_std_logic_funcs ton_in_6_funcs = {
		.std_logic_set = COMP_(ton_in_6_set)
	};
	static const struct sig_std_logic_funcs ton_in_7_funcs = {
		.std_logic_set = COMP_(ton_in_7_set)
	};
	static const struct sig_std_logic_funcs ton_in_8_funcs = {
		.std_logic_set = COMP_(ton_in_8_set)
	};
	static const struct sig_std_logic_funcs ton_in_9_funcs = {
		.std_logic_set = COMP_(ton_in_9_set)
	};
	static const struct sig_std_logic_funcs ton_in_10_funcs = {
		.std_logic_set = COMP_(ton_in_10_set)
	};
	static const struct sig_std_logic_funcs ton_in_11_funcs = {
		.std_logic_set = COMP_(ton_in_11_set)
	};
	static const struct sig_std_logic_funcs ton_in_12_funcs = {
		.std_logic_set = COMP_(ton_in_12_set)
	};
	static const struct sig_std_logic_funcs ton_in_13_funcs = {
		.std_logic_set = COMP_(ton_in_13_set)
	};
	static const struct sig_std_logic_funcs ton_in_14_funcs = {
		.std_logic_set = COMP_(ton_in_14_set)
	};
	static const struct sig_std_logic_funcs ton_in_15_funcs = {
		.std_logic_set = COMP_(ton_in_15_set)
	};
#if 0 /* not needed in C implementation. */
	static const struct sig_std_logic_funcs pci_clock_funcs = {
		/* FIXME */
	};
#endif
	static const struct sig_std_logic_funcs rstn_funcs = {
		.std_logic_set = COMP_(rstn_set)
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->state_ton_out = 0;

	/* Call */
	/* Out */
	cpssp->port_rc_time_0 = port_rc_time_0;
	sig_std_logic_connect_out(port_rc_time_0, cpssp, 0);

	cpssp->port_rc_time_1 = port_rc_time_1;
	sig_std_logic_connect_out(port_rc_time_1, cpssp, 0);

	cpssp->port_rc_time_2 = port_rc_time_2;
	sig_std_logic_connect_out(port_rc_time_2, cpssp, 0);

	cpssp->port_rc_time_3 = port_rc_time_3;
	sig_std_logic_connect_out(port_rc_time_3, cpssp, 0);

	cpssp->port_volume_0 = port_volume_0;
	sig_std_logic_connect_out(port_volume_0, cpssp, 0);

	cpssp->port_volume_1 = port_volume_1;
	sig_std_logic_connect_out(port_volume_1, cpssp, 0);

	cpssp->port_volume_2 = port_volume_2;
	sig_std_logic_connect_out(port_volume_2, cpssp, 0);

	cpssp->port_volume_3 = port_volume_3;
	sig_std_logic_connect_out(port_volume_3, cpssp, 0);

	cpssp->port_ton_out = port_ton_out;
	sig_std_logic_connect_out(port_ton_out, cpssp, 0);

	/* In */
	cpssp->state_ton_in = 0;
	sig_std_logic_connect_in(port_ton_in_0, cpssp, &ton_in_0_funcs);
	sig_std_logic_connect_in(port_ton_in_1, cpssp, &ton_in_1_funcs);
	sig_std_logic_connect_in(port_ton_in_2, cpssp, &ton_in_2_funcs);
	sig_std_logic_connect_in(port_ton_in_3, cpssp, &ton_in_3_funcs);
	sig_std_logic_connect_in(port_ton_in_4, cpssp, &ton_in_4_funcs);
	sig_std_logic_connect_in(port_ton_in_5, cpssp, &ton_in_5_funcs);
	sig_std_logic_connect_in(port_ton_in_6, cpssp, &ton_in_6_funcs);
	sig_std_logic_connect_in(port_ton_in_7, cpssp, &ton_in_7_funcs);
	sig_std_logic_connect_in(port_ton_in_8, cpssp, &ton_in_8_funcs);
	sig_std_logic_connect_in(port_ton_in_9, cpssp, &ton_in_9_funcs);
	sig_std_logic_connect_in(port_ton_in_10, cpssp, &ton_in_10_funcs);
	sig_std_logic_connect_in(port_ton_in_11, cpssp, &ton_in_11_funcs);
	sig_std_logic_connect_in(port_ton_in_12, cpssp, &ton_in_12_funcs);
	sig_std_logic_connect_in(port_ton_in_13, cpssp, &ton_in_13_funcs);
	sig_std_logic_connect_in(port_ton_in_14, cpssp, &ton_in_14_funcs);
	sig_std_logic_connect_in(port_ton_in_15, cpssp, &ton_in_15_funcs);

#if 0 /* unused in C implementation */
	sig_std_logic_connect_in(port_pci_clock, cpssp, &pci_clock_funcs);
#endif
	sig_std_logic_connect_in(port_rstn, cpssp, &rstn_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */
	time_call_delete(COMP_(tick), cpssp);

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fp);
}
