/*
 * Parts derived from QEMU sources.
 * Modified for FAUmachine by Volkmar Sieh.
 *  
 *  Copyright (c) 2005-2016 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */

#ifndef __CPU_STATE_H_INCLUDED
#define __CPU_STATE_H_INCLUDED

#include "qemu/libcpu.h"

#include "arch_gen_cpu_x86.h"

#define INCLUDE
#include "arch_gen_cpu_x86_mmu.c"
#undef INCLUDE

#ifndef CONFIG_CPU_CACHE_LINE_SIZE
#define CACHE_LINE_SIZE	(256)
#else
#define CACHE_LINE_SIZE	CONFIG_CPU_CACHE_LINE_SIZE
#endif
#define CACHE_LINE_MASK	(~(CACHE_LINE_SIZE - 1))

#define TARGET_PAGE_BITS 12
#define TARGET_PAGE_SIZE (1 << TARGET_PAGE_BITS)
#define TARGET_PAGE_MASK ~(TARGET_PAGE_SIZE - 1)


#if 80486 <= CONFIG_CPU && CONFIG_CPU_LM_SUPPORT
/* 64bit data support */
#define data_t	int64_t
#define udata_t	uint64_t
#elif 80386 <= CONFIG_CPU
/* 32bit data support */
#define data_t	int32_t
#define udata_t	uint32_t
#else
/* 16bit data support */
#define data_t	int16_t
#define udata_t	uint16_t
#endif

#if 80486 <= CONFIG_CPU && CONFIG_CPU_LM_SUPPORT
/* 64bit virtual address support */
#define vaddr_t uint64_t
#else
#define vaddr_t uint32_t
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_PAE_SUPPORT
/* 36bit physical address support */
#define paddr_t uint64_t
#else
#define paddr_t uint32_t
#endif
typedef char *Haddr;

#define TB_JMP_CACHE_BITS 12
#define TB_JMP_CACHE_SIZE (1 << TB_JMP_CACHE_BITS)

/*
 * Constants
 */
#define CODE_GEN_MAX_SIZE	65536
#define CODE_GEN_ALIGN		16 /* must be >= of the size of a icache line */

#define CODE_GEN_HASH_BITS	15
#define CODE_GEN_PHYS_HASH_BITS	15

#define CODE_GEN_HASH_SIZE      (1 << CODE_GEN_HASH_BITS)
#define CODE_GEN_PHYS_HASH_SIZE (1 << CODE_GEN_PHYS_HASH_BITS)

/* threshold to flush the translated code buffer */
#define CODE_GEN_BUFFER_MAX_SIZE (CODE_GEN_BUFFER_SIZE - CODE_GEN_MAX_SIZE)

#define SMC_BITMAP_USE_THRESHOLD 10

/* estimated block size for TB allocation */
/* XXX: use a per code average code fragment size and modulate it
 *    according to the host CPU */
#define CODE_GEN_AVG_BLOCK_SIZE 128

#define CODE_GEN_MAX_BLOCKS    (CODE_GEN_BUFFER_SIZE / CODE_GEN_AVG_BLOCK_SIZE)

#if defined(__powerpc__)
#define USE_DIRECT_JUMP
#endif
#if (defined(__i386__) || defined(__x86_64__)) && !defined(_WIN32)
#define USE_DIRECT_JUMP
#endif

typedef struct TranslationBlock {
	target_ulong pc;	/* simulated PC corresponding to this block
				   (EIP + CS base) */
	target_ulong cs_base;	/* CS base for this block */
	unsigned int flags;	/* flags defining in which context the code
				   was generated */
	uint16_t size;		/* size of target code for this block
				   (1 <= size <= TARGET_PAGE_SIZE) */
	uint16_t cflags;	/* compile flags */
#define CF_TB_FP_USED	0x0001	/* fp ops are used in TB */
#define CF_FP_USED	0x0002	/* fp ops are used in TB or in a chained TB */
#define CF_SINGLE_INSN	0x0004	/* compile only a single instruction */

	uint8_t *tc_ptr;	/* pointer to the translated code */
	struct TranslationBlock *hash_next; /* next matching tb for virtual
					       address */
	/* next matching tb for physical address. */
	struct TranslationBlock *phys_hash_next;
	/* first and second physical page containing code. The lower bit
	   of the pointer tells the index in page_next[] */
	struct TranslationBlock *page_next[2];
	target_ulong page_addr[2];

	/* the following data are used to directly call another TB from
	   the code of this one. */
	uint16_t tb_next_offset[2]; /* offset of original jump target */
#ifdef USE_DIRECT_JUMP
	uint16_t tb_jmp_offset[4]; /* offset of jump instruction */
#else
	uint8_t *tb_next[2]; /* address of jump generated code */
#endif
	/* list of TBs jumping to this one. This is a circular list using
	   the two least significant bits of the pointers to tell what is
	   the next pointer: 0 = jmp_next[0], 1 = jmp_next[1], 2 = jmp_first */
	struct TranslationBlock *jmp_next[2];
	struct TranslationBlock *jmp_first;
} TranslationBlock;

typedef struct PageDesc {
	/* list of TBs intersecting this ram page */
	TranslationBlock *first_tb;
	/*
	 * in order to optimize self modifying code, we count the number
	 * of lookups we do to a given page to use a bitmap
	 */
	unsigned int code_write_count;
	uint8_t *code_bitmap;
} PageDesc;

#ifdef __i386__
#define AREG0 "ebp"
#endif
#ifdef __x86_64__
#define AREG0 "rbp"
#endif

#if TARGET_LONG_BITS <= HOST_LONG_BITS
/* Host registers can be used to store target registers. */

#ifdef __i386__
#define AREG1 "ebx"
// #define AREG2 "esi" /* Otherwise we'll get internal compiler errors. */
#define AREG3 "edi"
#endif
#ifdef __x86_64__
#if !defined(DARWIN)
#define AREG1 "rbx"
#define AREG2 "r12"
#define AREG3 "r13"
#if 0 /* These two lines are commented out without comment in current QEMU */
#define AREG4 "r14"
#define AREG5 "r15"
#endif
#endif
#endif

#endif /* TARGET_LONG_BITS <= HOST_LONG_BITS */

#ifdef FAST

#ifdef AREG1
register target_ulong _T0 asm(AREG1);
#define T0(cpssp)	_T0
#else
#define T0(cpssp)	(cpssp->t0)
#endif

#ifdef AREG2
register target_ulong _T1 asm(AREG2);
#define T1(cpssp)	_T1
#else
#define T1(cpssp)	(cpssp->t1)
#endif

#ifdef AREG3
register target_ulong _T2 asm(AREG3);
#define T2(cpssp)	_T2
#else
#define T2(cpssp)	(cpssp->t2)
#endif

#define A0(cpssp)	T2(cpssp)

#endif /* FAST */

struct cpssp {
#define STATE

#include "arch_gen_cpu_x86_core.c"
#include "arch_gen_cpu_x86_fpu.c"
#include "arch_gen_cpu_x86_align.c"

	/* Signals */
#if defined(CONFIG_CPU_SOCKET_ISA)
	struct sig_isa_bus *port_bus;
#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	struct sig_host_bus *port_bus;
#elif defined(CONFIG_CPU_SOCKET_HOST775)
	struct sig_host775_bus *port_bus;
#endif
	struct sig_std_logic *sig_n_ferr;
	int state_n_ignne;
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	struct sig_icc_bus *icc_bus;
#endif

#define NAME		apic
#define NAME_(x)	CONC(CHIP, apic, x)
#include "arch_gen_cpu_x86_apic.c"
#undef NAME_
#undef NAME

#if defined CONFIG_CPU_L2_SIZE
#define CONFIG_CACHE2_ASSOC	CONFIG_CPU_L2_ASSOC
#define CONFIG_CACHE2_SIZE	CONFIG_CPU_L2_SIZE
#define CONFIG_CACHE2_LINE_SIZE	CONFIG_CPU_CACHE_LINE_SIZE
#endif
#define NAME		cache2
#define NAME_(x)	CONC(CHIP, cache2, x)
#include "arch_gen_cpu_x86_cache2.c"
#undef NAME_
#undef NAME
#if defined CONFIG_CPU_L2_SIZE
#undef CONFIG_CACHE2_LINE_SIZE
#undef CONFIG_CACHE2_SIZE
#undef CONFIG_CACHE2_ASSOC
#endif

#define NAME		mtrr
#define NAME_(x)	CONC(CHIP, mtrr, x)
#include "arch_gen_cpu_x86_mtrr.c"
#undef NAME_
#undef NAME

#define NAME		a20gate
#define NAME_(x)	CONC(CHIP, a20gate, x)
#include "arch_gen_cpu_x86_a20gate.c"
#undef NAME_
#undef NAME

#define NAME		mmui
#define NAME_(x)	CONC(CHIP, mmui, x)
#ifdef CONFIG_CPU_TLBI_SIZE
#define CONFIG_MMU_TLB_ASSOC	CONFIG_CPU_TLBI_ASSOC
#define CONFIG_MMU_TLB_SIZE	CONFIG_CPU_TLBI_SIZE
#elif defined(CONFIG_CPU_TLBI4M_SIZE) && defined(CONFIG_CPU_TLBI4K_SIZE)
#define CONFIG_MMU_TLB4K_ASSOC	CONFIG_CPU_TLBI4K_ASSOC
#define CONFIG_MMU_TLB4K_SIZE	CONFIG_CPU_TLBI4K_SIZE
#define CONFIG_MMU_TLB4M_ASSOC	CONFIG_CPU_TLBI4M_ASSOC
#define CONFIG_MMU_TLB4M_SIZE	CONFIG_CPU_TLBI4M_SIZE
#endif
#include "arch_gen_cpu_x86_mmu.c"
#undef CONFIG_MMU_TLB4M_SIZE
#undef CONFIG_MMU_TLB4M_ASSOC
#undef CONFIG_MMU_TLB4K_SIZE
#undef CONFIG_MMU_TLB4K_ASSOC
#undef CONFIG_MMU_TLB_SIZE
#undef CONFIG_MMU_TLB_ASSOC
#undef NAME_
#undef NAME

#define NAME		mmud
#define NAME_(x)	CONC(CHIP, mmud, x)
#ifdef CONFIG_CPU_TLBD_SIZE
#define CONFIG_MMU_TLB_ASSOC	CONFIG_CPU_TLBD_ASSOC
#define CONFIG_MMU_TLB_SIZE	CONFIG_CPU_TLBD_SIZE
#elif defined(CONFIG_CPU_TLBD4M_SIZE) && defined(CONFIG_CPU_TLBD4K_SIZE)
#define CONFIG_MMU_TLB4K_ASSOC	CONFIG_CPU_TLBD4K_ASSOC
#define CONFIG_MMU_TLB4K_SIZE	CONFIG_CPU_TLBD4K_SIZE
#define CONFIG_MMU_TLB4M_ASSOC	CONFIG_CPU_TLBD4M_ASSOC
#define CONFIG_MMU_TLB4M_SIZE	CONFIG_CPU_TLBD4M_SIZE
#endif
#include "arch_gen_cpu_x86_mmu.c"
#undef CONFIG_MMU_TLB4M_SIZE
#undef CONFIG_MMU_TLB4M_ASSOC
#undef CONFIG_MMU_TLB4K_SIZE
#undef CONFIG_MMU_TLB4K_ASSOC
#undef CONFIG_MMU_TLB_SIZE
#undef CONFIG_MMU_TLB_ASSOC
#undef NAME_
#undef NAME

#define NAME		cache1i
#define NAME_(x)	CONC(CHIP, cache1i, x)
#define CONFIG_CODE
#ifdef CONFIG_CPU_L1I_SIZE
#define CONFIG_SIZE	CONFIG_CPU_L1I_SIZE
#define CONFIG_ASSOC	CONFIG_CPU_L1I_ASSOC
#endif
#include "arch_gen_cpu_x86_cache1.c"
#ifdef CONFIG_CPU_L1I_SIZE
#undef CONFIG_ASSOC
#undef CONFIG_SIZE
#endif
#undef CONFIG_CODE
#undef NAME_
#undef NAME

#define NAME		cache1d
#define NAME_(x)	CONC(CHIP, cache1d, x)
#define CONFIG_DATA
#ifdef CONFIG_CPU_L1D_SIZE
#define CONFIG_SIZE	CONFIG_CPU_L1D_SIZE
#define CONFIG_ASSOC	CONFIG_CPU_L1D_ASSOC
#endif
#include "arch_gen_cpu_x86_cache1.c"
#ifdef CONFIG_CPU_L1D_SIZE
#undef CONFIG_ASSOC
#undef CONFIG_SIZE
#endif
#undef CONFIG_DATA
#undef NAME_
#undef NAME

#undef STATE

	/* CPU is simulated by process. */
	struct process process;
};

#ifdef FAST
register struct cpssp *env asm(AREG0);
#endif

#endif /* __CPU_STATE_H_INCLUDED */
