/*
 * Derived from QEMU sources.
 *
 *  Copyright (c) 2005-2017 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */

/* FIXME */
#if defined CONFIG_MMU_TLB_SIZE
/* MMU with TLB for all sizes. */
#define SETS	CONFIG_MMU_TLB_ASSOC
#define LINES	(CONFIG_MMU_TLB_SIZE / SETS)

#elif defined CONFIG_MMU_TLB4M_SIZE && defined CONFIG_MMU_TLB4K_SIZE
/* MMU with distinct TLBs for 4K/4M sizes. */
#define SETS4M	CONFIG_MMU_TLB4M_ASSOC
#define LINES4M	(CONFIG_MMU_TLB4M_SIZE / SETS4M)
#define SETS4K	CONFIG_MMU_TLB4K_ASSOC
#define LINES4K	(CONFIG_MMU_TLB4K_SIZE / SETS4K)

#else
/* No MMU - FIXME */
#define SETS4M	4
#define LINES4M	4
#endif

#ifdef INCLUDE

#include "glue-lru.h"

#endif /* INCLUDE */
#ifdef STATE

struct {
#ifdef LINES
	struct {
		vaddr_t vaddr;
		paddr_t paddr;
		paddr_t psize;
		uint8_t gflag;
		uint8_t wflag;
		uint8_t uflag;
		uint8_t type; /* UC, WB, ... */
	} set[SETS];
	LRU_DECL(SETS, lru);
#endif
#ifdef LINES4M
	struct {
		struct {
			vaddr_t vaddr;
			paddr_t paddr;
			uint8_t gflag;
			uint8_t wflag;
			uint8_t uflag;
			uint8_t type; /* UC, WB, ... */
		} set[SETS4M];
		LRU_DECL(SETS4M, lru);
	} tlb4M[LINES4M];
#endif
#ifdef LINES4K
	struct {
		struct {
			vaddr_t vaddr;
			paddr_t paddr;
			uint8_t gflag;
			uint8_t wflag;
			uint8_t uflag;
			uint8_t type; /* UC, WB, ... */
		} set[SETS4K];
		LRU_DECL(SETS4K, lru);
	} tlb4K[LINES4K];
#endif
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static int
NAME_(map)(struct cpssp *cpssp, vaddr_t va, int wflag, int uflag, int xflag, paddr_t *paddrp, uint16_t *errp);
#if 80486 <= CONFIG_CPU
/*forward*/ static void
NAME_(invlpg)(struct cpssp *cpssp, vaddr_t page);
#endif
/*forward*/ static void
NAME_(tlb_flush)(struct cpssp *cpssp, int all);
/*forward*/ static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#ifndef FAST
//static void
//NAME_(create)(struct cpssp *cpssp);
//static void
//NAME_(destroy)(struct cpssp *cpssp);
#endif

#endif /* EXPORT */
#ifdef BEHAVIOR

#define PG_PRESENT_BIT  0
#define PG_RW_BIT       1
#define PG_USER_BIT     2
#define PG_PWT_BIT      3
#define PG_PCD_BIT      4
#define PG_ACCESSED_BIT 5
#define PG_DIRTY_BIT    6
#define PG_PSE_BIT      7
#define PG_GLOBAL_BIT   8
#define PG_NX_BIT       63

#define PG_PRESENT_MASK  (1 << PG_PRESENT_BIT)
#define PG_RW_MASK       (1 << PG_RW_BIT)
#define PG_USER_MASK     (1 << PG_USER_BIT)
#define PG_PWT_MASK      (1 << PG_PWT_BIT)
#define PG_PCD_MASK      (1 << PG_PCD_BIT)
#define PG_ACCESSED_MASK (1 << PG_ACCESSED_BIT)
#define PG_DIRTY_MASK    (1 << PG_DIRTY_BIT)
#define PG_PSE_MASK      (1 << PG_PSE_BIT)
#define PG_GLOBAL_MASK   (1 << PG_GLOBAL_BIT)
#define PG_NX_MASK       (1LL << PG_NX_BIT)

#define PG_ERROR_W_BIT     1
#define PG_ERROR_U_BIT     2

#define PG_ERROR_P_MASK    0x01
#define PG_ERROR_W_MASK    (1 << PG_ERROR_W_BIT)
#define PG_ERROR_U_MASK    (1 << PG_ERROR_U_BIT)
#define PG_ERROR_RSVD_MASK 0x08
#define PG_ERROR_I_D_MASK  0x10

#define PHYS_ADDR_MASK 0xfffff000

static uint32_t
NAME_(read_entry)(struct cpssp *cpssp, paddr_t pa)
{
	udata_t val;

	NAME_(mr)(cpssp, pa & ~(sizeof(udata_t) - 1),
			0b1111 << (pa & (sizeof(udata_t) - 1)), &val);

	return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xffffffff;
}

static void
NAME_(write_entry)(struct cpssp *cpssp, paddr_t pa, uint32_t val)
{
	udata_t valX;

	assert(! (pa & 3));

	valX = ((udata_t) val) << ((pa & (sizeof(udata_t) - 1)) * 8);

	NAME_(mw)(cpssp, pa & ~(pa & (sizeof(udata_t) - 1)),
			0b1111 << (pa & (sizeof(udata_t) - 1)), valX);
}

#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
static uint64_t
NAME_(read_entry64)(struct cpssp *cpssp, paddr_t pa)
{
	uint32_t val1;
	uint32_t val2;

	assert(! (pa & 3));
	val1 = NAME_(read_entry)(cpssp, pa + 0);
	val2 = NAME_(read_entry)(cpssp, pa + 4);

	return (((uint64_t) val2) << 32) | val1;
}
#endif /* defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT */

static int
NAME_(check)(struct cpssp *cpssp, uint32_t ptep, int want_wflag, int want_uflag)
{
	if (want_uflag) {
		if (! (ptep & PG_USER_MASK))
			return 1;
		if (want_wflag
		 && ! (ptep & PG_RW_MASK))
			return 1;
#if 80486 <= CONFIG_CPU
	} else {
		if (NAME_(cr0_wp_get(cpssp))
		 && want_wflag
		 && !(ptep & PG_RW_MASK))
			return 1;
#endif /* 80486 <= CONFIG_CPU */
	}
	return 0;
}

static void
NAME_(touch)(struct cpssp *cpssp, paddr_t pte_addr, uint32_t pte, int want_wflag)
{
	int is_dirty;

	is_dirty = want_wflag && ! (pte & PG_DIRTY_MASK);
	if (! (pte & PG_ACCESSED_MASK)
	 || is_dirty) {
		pte |= PG_ACCESSED_MASK;
		if (is_dirty)
			pte |= PG_DIRTY_MASK;
		NAME_(write_entry)(cpssp, pte_addr, pte);
	}
}

static uint8_t
NAME_(cdwr_to_type)(uint32_t ptep)
{
	static const uint8_t type[] = {
		/* 00: WB */	0b110,
		/* 01: WT */	0b100,
		/* 10: WC */	0b001,
		/* 11: UC */	0b000,
	};
	unsigned int i;

	i = ((ptep >> PG_PWT_BIT) & 1) << 0
		| ((ptep >> PG_PCD_BIT) & 1) << 1;
	return type[i];
}

/*
 * Return value:
 * 0 = nothing more to do.
 * 1 = generate page-fault.
 */
static int
NAME_(__fault)(
	struct cpssp *cpssp,
	vaddr_t addr,
	int want_wflag,
	int want_uflag,
	paddr_t *paddrp,
	paddr_t *psizep,
	uint8_t *gflagp,
	uint8_t *typep,
	uint16_t *errp
)
{
	uint32_t pde_addr;
	uint32_t pte_addr;
	uint64_t pte;
	uint64_t ptep;
	int error_code;
	int gflag;
	int uflag;
	int wflag;
	int dflag;
	int page_size;
	unsigned long paddr;

	if (! NAME_(cr0_pg_get(cpssp))) {
		pte = addr & ~0x3fffff;
		uflag = 0;
		wflag = 1;
		dflag = 1;
		page_size = 1024*4096;
		ptep = (0 << PG_GLOBAL_BIT)
			| (0 << PG_PCD_BIT)
			| (0 << PG_PWT_BIT)
			| (1 << PG_USER_BIT)
			| (1 << PG_RW_BIT)
			| (1 << PG_PRESENT_BIT);
		goto do_mapping;
	}

#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
	if (NAME_(cr4_pae_get)(cpssp)) { /* FIXME fox the whole if part needs reworking */
		uint64_t pde, pdpe;
		uint64_t pdpe_addr;

		/* XXX: we only use 32 bit physical addresses */
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
		if (NAME_(efer_lme_get)(cpssp)) {
			uint32_t pml4e_addr;
			uint64_t pml4e;
			int32_t sext;

			/* test virtual address sign extension */
			sext = (int64_t) addr >> 47;
			if (sext != 0 && sext != -1) {
				/* Protection Fault */
				*errp = 0;
				return 1;
			}

			pml4e_addr = ((NAME_(cr3_get)(cpssp) & ~0xfff) + (((addr >> 39) & 0x1ff) << 3));
			pml4e = NAME_(read_entry64)(cpssp, pml4e_addr);
			if (!(pml4e & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			if ((pml4e & PG_NX_MASK)
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
			 && ! NAME_(efer_nxe_get)(cpssp)
#endif
					) {
				error_code = PG_ERROR_RSVD_MASK;
				goto do_fault;
			}
			NAME_(touch)(cpssp, pml4e_addr, pml4e, 0);
			ptep = pml4e ^ PG_NX_MASK;
			pdpe_addr = ((pml4e & PHYS_ADDR_MASK) + (((addr >> 30) & 0x1ff) << 3));
			pdpe = NAME_(read_entry64)(cpssp, pdpe_addr);
			if (!(pdpe & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			if ((pdpe & PG_NX_MASK)
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
			 && ! NAME_(efer_nxe_get)(cpssp)
#endif
					) {
				error_code = PG_ERROR_RSVD_MASK;
				goto do_fault;
			}
			ptep &= pdpe ^ PG_NX_MASK;
			NAME_(touch)(cpssp, pdpe_addr, pdpe, 0);
		} else
#endif /* CONFIG_CPU_LM_SUPPORT */
		{
			/* XXX: load them when cr3 is loaded ? */
			pdpe_addr = ((NAME_(cr3_get)(cpssp) & ~0x1f) + ((addr >> 30) << 3));
			pdpe = NAME_(read_entry64)(cpssp, pdpe_addr);
			if (!(pdpe & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			ptep = PG_NX_MASK | PG_USER_MASK | PG_RW_MASK;
		}

		pde_addr = ((pdpe & PHYS_ADDR_MASK) + (((addr >> 21) & 0x1ff) << 3));
		pde = NAME_(read_entry64)(cpssp, pde_addr);
		if (!(pde & PG_PRESENT_MASK)) {
			error_code = 0;
			goto do_fault;
		}
		if ((pde & PG_NX_MASK)
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
		 && ! NAME_(efer_nxe_get)(cpssp)
#endif
				) {
			error_code = PG_ERROR_RSVD_MASK;
			goto do_fault;
		}
		ptep &= pde ^ PG_NX_MASK;
		if (pde & PG_PSE_MASK) {
			/* 2 MB page */
			page_size = 2048 * 1024;
			ptep ^= PG_NX_MASK;
			if (NAME_(check)(cpssp, ptep, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(touch)(cpssp, pde_addr, pde, want_wflag);
			/* align to page_size */
			pte = pde & ((PHYS_ADDR_MASK & ~(page_size - 1)) | 0xfff);
		} else {
			/* 4 KB page */
			NAME_(touch)(cpssp, pde_addr, pde, 0);
			pte_addr = ((pde & PHYS_ADDR_MASK) + (((addr >> 12) & 0x1ff) << 3));
			pte = NAME_(read_entry64)(cpssp, pte_addr);
			if (!(pte & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			if ((pte & PG_NX_MASK)
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
			 && ! NAME_(efer_nxe_get)(cpssp)
#endif
					) {
				error_code = PG_ERROR_RSVD_MASK;
				goto do_fault;
			}
			/* combine pde and pte nx, user and rw protections */
			ptep &= pte ^ PG_NX_MASK;
			ptep ^= PG_NX_MASK;
			if (NAME_(check)(cpssp, ptep, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(touch)(cpssp, pte_addr, pte, want_wflag);
			page_size = 4096;
			pte = pte & (PHYS_ADDR_MASK | 0xfff);
		}
	} else /* End of if that needs FIXME fox */
#endif /* defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT */
	{
		uint32_t pde;

		/* Page directory entry. */
		pde_addr = ((NAME_(cr3_get)(cpssp) & ~0xfff) + ((addr >> 20) & ~3));
		pde = NAME_(read_entry)(cpssp, pde_addr);
		if (! (pde & PG_PRESENT_MASK)) {
			error_code = 0;
			goto do_fault;
		}
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
		/* If PSE bit is set, then we use a 4MB page. */
		if ((pde & PG_PSE_MASK)
		 && (NAME_(cr4_pse_get)(cpssp))) {
			if (NAME_(check)(cpssp, pde, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(touch)(cpssp, pde_addr, pde, want_wflag);

			pte = pde & ~0x003ff000; /* align to 4MB */
			ptep = pte;
			page_size = 4096 * 1024;

		} else
#endif /* defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT */
		{
			NAME_(touch)(cpssp, pde_addr, pde, 0);

			/* Page table entry. */
			pte_addr = ((pde & ~0xfff) + ((addr >> 10) & 0xffc));
			pte = NAME_(read_entry)(cpssp, pte_addr);
			if (! (pte & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			/* Combine pde and pte user and rw protections. */
			ptep = pte & pde;
			if (NAME_(check)(cpssp, ptep, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(touch)(cpssp, pte_addr, pte, want_wflag);

			page_size = 4096;
		}
	}

do_mapping:
	paddr = pte & ~0xfff;
#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
	gflag = (ptep >> PG_GLOBAL_BIT) & NAME_(cr4_pge_get)(cpssp);
#else
	gflag = 0;
#endif

	*paddrp = paddr;
	*psizep = page_size;
	*gflagp = gflag;
	*typep = NAME_(cdwr_to_type)(ptep);
	return 0;

do_fault_protect:
	error_code = PG_ERROR_P_MASK;
do_fault:
	*errp = error_code | (want_wflag << PG_ERROR_W_BIT) | (want_uflag << PG_ERROR_U_BIT);
	return 1;
}

#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT
static uint8_t
NAME_(combine_type)(uint8_t pat, uint8_t mtrr)
{
	/* 9-16 */
	uint8_t comb[8][8] = {
	/*        UC,WC,--,--,WT,WP,WB,UC- */
	/*UC:0*/ { 0, 0,-0,-0, 0, 0, 0, 0 },
	/*WC:1*/ { 0, 1,-0,-0,-0,-0, 1, 1 },
	/*--:2*/ {-0,-0,-0,-0,-0,-0,-0,-0 },
	/*--:3*/ {-0,-0,-0,-0,-0,-0,-0,-0 },
	/*WT:4*/ { 0, 1,-0,-0, 4,-0, 4, 0 },
	/*WP:5*/ { 0, 1,-0,-0,-0, 5, 5,-0 },
	/*WB:6*/ { 0, 1,-0,-0, 4, 5, 6, 0 },
	/*UC-:7*/{-0,-0,-0,-0,-0,-0,-0,-0 },
	};

	return comb[mtrr][pat];
}
#endif

#ifdef LINES4M
static int
NAME_(hash4M)(vaddr_t vaddr)
{
	return (vaddr / 1024 / 4096) % LINES4M;
}
#endif /* LINES4M */

#ifdef LINES4K
static int
NAME_(hash4K)(vaddr_t vaddr)
{
	return (vaddr / 4096) % LINES4K;
}
#endif /* LINES4K */

static int
NAME_(_fault)(
	struct cpssp *cpssp,
	vaddr_t vaddr,
	int want_wflag,
	int want_uflag,
	paddr_t *paddrp,
	paddr_t *psizep,
	uint16_t *errp
)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;
	paddr_t paddr;
	paddr_t psize;
	uint8_t gflag;
	uint8_t type;

#ifdef LINES
	/* Search for page in common-TLB. */
	for (set = 0; set < SETS; set++) {
		if ((vaddr & ~(cpssp->NAME.set[set].psize - 1))
				== cpssp->NAME.set[set].vaddr) {
			if (cpssp->NAME.set[set].wflag < want_wflag
			 || cpssp->NAME.set[set].uflag < want_uflag) {
				if (NAME_(__fault)(cpssp, vaddr, want_wflag, want_uflag,
						&paddr, &psize, &gflag, &type, errp)) {
					return 1;
				}
				cpssp->NAME.set[set].wflag = want_wflag;
				cpssp->NAME.set[set].uflag = want_uflag;
			}
		found:	;
			lru_use(SETS, cpssp->NAME.lru, set);
			*paddrp = cpssp->NAME.set[set].paddr;
			*psizep = cpssp->NAME.set[set].psize;
			return 0;
		}
	}
#endif

#ifdef LINES4M
	/* Search for page in 4M-page-TLB. */
	line = NAME_(hash4M)(vaddr);
	for (set = 0; set < SETS4M; set++) {
		if ((vaddr & ~0x3fffff) == cpssp->NAME.tlb4M[line].set[set].vaddr) {
			if (cpssp->NAME.tlb4M[line].set[set].wflag < want_wflag
			 || cpssp->NAME.tlb4M[line].set[set].uflag < want_uflag) {
				if (NAME_(__fault)(cpssp, vaddr, want_wflag, want_uflag,
						&paddr, &psize, &gflag, &type, errp)) {
					return 1;
				}
				cpssp->NAME.tlb4M[line].set[set].wflag = want_wflag;
				cpssp->NAME.tlb4M[line].set[set].uflag = want_uflag;
			}
		found4M:;
			lru_use(SETS4M, cpssp->NAME.tlb4M[line].lru, set);
			*paddrp = cpssp->NAME.tlb4M[line].set[set].paddr;
			*psizep = 1024*4096;
			return 0;
		}
	}
#endif /* LINES4M */

#ifdef LINES4K
	/* Search for page in 4K-page-TLB. */
	line = NAME_(hash4K)(vaddr);
	for (set = 0; set < SETS4K; set++) {
		if ((vaddr & ~0xfff) == cpssp->NAME.tlb4K[line].set[set].vaddr) {
			if (cpssp->NAME.tlb4K[line].set[set].wflag < want_wflag
			 || cpssp->NAME.tlb4K[line].set[set].uflag < want_uflag) {
				if (NAME_(__fault)(cpssp, vaddr, want_wflag, want_uflag,
						&paddr, &psize, &gflag, &type, errp)) {
					return 1;
				}
				cpssp->NAME.tlb4K[line].set[set].wflag = want_wflag;
				cpssp->NAME.tlb4K[line].set[set].uflag = want_uflag;
			}
		found4K:;
			lru_use(SETS4K, cpssp->NAME.tlb4K[line].lru, set);
			*paddrp = cpssp->NAME.tlb4K[line].set[set].paddr;
			*psizep = 4096;
			return 0;
		}
	}
#endif /* LINES4K */

	/* Not found. */
	if (NAME_(__fault)(cpssp, vaddr, want_wflag, want_uflag,
			&paddr, &psize, &gflag, &type, errp)) {
		return 1;
	}
#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT
	type = NAME_(combine_type)(type, NAME_(type)(cpssp, paddr));
#endif

#ifdef LINES
	if (psize == 1024*4096
	 || psize ==  512*4096
	 || psize ==    1*4096) {
		set = lru_oldest(SETS, cpssp->NAME.lru);
		cpssp->NAME.set[set].vaddr = vaddr & ~(psize - 1);
		cpssp->NAME.set[set].paddr = paddr;
		cpssp->NAME.set[set].psize = psize;
		cpssp->NAME.set[set].gflag = gflag;
		cpssp->NAME.set[set].wflag = want_wflag;
		cpssp->NAME.set[set].uflag = want_uflag;
		cpssp->NAME.set[set].type = type;
		goto found;
	}
#endif /* LINES */
#ifdef LINES4M
	if (psize == 1024*4096) {
		line = NAME_(hash4M)(vaddr);
		set = lru_oldest(SETS4M, cpssp->NAME.tlb4M[line].lru);
		cpssp->NAME.tlb4M[line].set[set].vaddr = vaddr & ~(psize - 1);
		cpssp->NAME.tlb4M[line].set[set].paddr = paddr;
		cpssp->NAME.tlb4M[line].set[set].gflag = gflag;
		cpssp->NAME.tlb4M[line].set[set].wflag = want_wflag;
		cpssp->NAME.tlb4M[line].set[set].uflag = want_uflag;
		cpssp->NAME.tlb4M[line].set[set].type = type;
		goto found4M;

	} else
#endif /* LINES4M */
#ifdef LINES4K
	if (psize == 4096) {
		line = NAME_(hash4K)(vaddr);
		set = lru_oldest(SETS4K, cpssp->NAME.tlb4K[line].lru);
		cpssp->NAME.tlb4K[line].set[set].vaddr = vaddr & ~(psize - 1);
		cpssp->NAME.tlb4K[line].set[set].paddr = paddr;
		cpssp->NAME.tlb4K[line].set[set].gflag = gflag;
		cpssp->NAME.tlb4K[line].set[set].wflag = want_wflag;
		cpssp->NAME.tlb4K[line].set[set].uflag = want_uflag;
		cpssp->NAME.tlb4K[line].set[set].type = type;
		goto found4K;

	} else
#endif /* LINES4K */
	{
		assert(0);
	}
}

/*
 * Return value:
 * 0 = nothing more to do.
 * 1 = generate page-fault.
 */
static int
NAME_(map)(
	struct cpssp *cpssp,
	vaddr_t addr,
	int want_wflag,
	int want_uflag,
	int want_xflag,
	paddr_t *paddrp,
	uint16_t *errp
)
{
	paddr_t paddr;
	paddr_t psize;
	int ret;

	ret = NAME_(_fault)(cpssp, addr, want_wflag, want_uflag, &paddr, &psize, errp);
	if (! ret) {
		paddr |= addr & (psize - 1);
		*paddrp = paddr;
	}

	return ret;
}

#if 80486 <= CONFIG_CPU
static void
NAME_(invlpg)(struct cpssp *cpssp, vaddr_t vaddr)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;

#ifdef LINES
	for (set = 0; set < SETS; set++) {
		if ((vaddr & ~(cpssp->NAME.set[set].psize - 1))
				== cpssp->NAME.set[set].vaddr) {
			cpssp->NAME.set[set].vaddr = (vaddr_t) -1; /* Invalid */
		}
	}
#endif
#ifdef LINES4M
	line = NAME_(hash4M)(vaddr);
	for (set = 0; set < SETS4M; set++) {
		if ((vaddr & ~0x3fffff)
				== cpssp->NAME.tlb4M[line].set[set].vaddr) {
			cpssp->NAME.tlb4M[line].set[set].vaddr = (vaddr_t) -1; /* Invalid */
		}
	}
#endif
#ifdef LINES4K
	line = NAME_(hash4K)(vaddr);
	for (set = 0; set < SETS4K; set++) {
		if ((vaddr & ~0xfff)
				== cpssp->NAME.tlb4K[line].set[set].vaddr) {
			cpssp->NAME.tlb4K[line].set[set].vaddr = (vaddr_t) -1; /* Invalid */
		}
	}
#endif

	NAME_(cache1_invlpg)(cpssp, vaddr);
}
#endif /* 80486 <= CONFIG_CPU */

static void
NAME_(tlb_flush)(struct cpssp *cpssp, int all)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;

#ifdef LINES
	{
		for (set = 0; set < SETS; set++) {
			if (all
			 || ! cpssp->NAME.set[set].gflag) {
				cpssp->NAME.set[set].vaddr
						= (vaddr_t) -1; /* Invalid */
			}
		}
	}
#endif
#ifdef LINES4M
	for (line = 0; line < LINES4M; line++) {
		for (set = 0; set < SETS4M; set++) {
			if (all
			 || ! cpssp->NAME.tlb4M[line].set[set].gflag) {
				cpssp->NAME.tlb4M[line].set[set].vaddr
						= (vaddr_t) -1; /* Invalid */
			}
		}
	}
#endif /* LINES4M */
#ifdef LINES4K
	for (line = 0; line < LINES4K; line++) {
		for (set = 0; set < SETS4K; set++) {
			if (all
			 || ! cpssp->NAME.tlb4K[line].set[set].gflag) {
				cpssp->NAME.tlb4K[line].set[set].vaddr
						= (vaddr_t) -1; /* Invalid */
			}
		}
	}
#endif /* LINES4K */

	NAME_(cache1_tlb_flush)(cpssp, all);
}

static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int val)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;

#ifdef LINES
	{
		for (set = 0; set < SETS; set++) {
			cpssp->NAME.set[set].vaddr
					= (vaddr_t) -1; /* Invalid */
		}
		lru_reset(SETS, cpssp->NAME.lru);
	}
#endif
#ifdef LINES4M
	for (line = 0; line < LINES4M; line++) {
		for (set = 0; set < SETS4M; set++) {
			cpssp->NAME.tlb4M[line].set[set].vaddr
					= (vaddr_t) -1; /* Invalid */
		}
		lru_reset(SETS4M, cpssp->NAME.tlb4M[line].lru);
	}
#endif /* LINES4M */
#ifdef LINES4K
	for (line = 0; line < LINES4K; line++) {
		for (set = 0; set < SETS4K; set++) {
			cpssp->NAME.tlb4K[line].set[set].vaddr
					= (vaddr_t) -1; /* Invalid */
		}
		lru_reset(SETS4K, cpssp->NAME.tlb4K[line].lru);
	}
#endif /* LINES4K */
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef LINES
#undef SETS
#undef LINES4M
#undef SETS4M
#undef LINES4K
#undef SETS4K
