/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Serial Peripheral Interface (SPI) */

/*
 * For most of the comments, infos, ... see:
 * TI AM1808/AM1810 ARM Microprocessor - Technical Reference Manual
 * (SPRUH82A - December 2011), Chapter 29, pages 1373...
 */

#define DEBUG_CONTROL_FLOW	0

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	int state_somi;

	/* Global Control Register 0, 29.3.1, 1402 */
	int nreset;

	/* Global Control Register 1, 29.3.2, 1403 */
	int enable;
	int loopback;
	int powerdown;
	int clkmod;
	int master;

	/* Interrupt Register, 29.3.3, 1405 */
	int enablehighz;
	int dmareqen;
	int txintena;
	int rxintena;
	int ovrnintena;
	int biterrena;
	int desyncena;
	int parerrena;
	int timeoutena;
	int dlenerrena;

	/* Level Register, 29.3.4, 1407 */
	int txintlvl;
	int rxintlvl;
	int ovrnintlvl;
	int biterrlvl;
	int desynclvl;
	int parerrlvl;
	int timeoutlvl;
	int dlenerrlvl;

	/* Pin Control Register 0, 29.3.6, 1410 */
	int somifun;
	int simofun;
	int clkfun;
	int enafun;
	int scsofun[8];

	/* Transmit Data Register 0, 29.3.12, 1416 */
	uint16_t txdata;

	/* Transmit Data Register 1, 29.3.13, 1417 */
	int cshold;
	int wdel;
	uint8_t dfsel;
	uint8_t csnrN;
	/* uint16_t txdata; see above */

	/* Receive Buffer Register, 29.3.14, 1418 */
	uint8_t rxempty;
	uint8_t rxovr;
	uint8_t txfull;
	uint8_t biterr;
	uint8_t desync;
	uint8_t parerr;
	uint8_t timeout;
	uint8_t dlenerr;
	uint16_t rxdata;

	/* Receive Buffer Emulation Register, 29.3.15, 1420 */
	/* See above... */

	/* Delay Register, 29.3.16, 1421 */
	uint8_t c2tdelay;
	uint8_t t2cdelay;
	uint8_t t2edelay;
	uint8_t c2edelay;

	/* Chip Select Default Register, 29.3.17, 1424 */
	uint8_t csdefN;

	/* Format 0 - 3 Register, 29.3.18, page 1425 */
	uint8_t wdelay[4];
	uint8_t parpol[4];
	uint8_t parena[4];
	uint8_t waitena[4];
	uint8_t shiftdir[4];
	uint8_t discstimers[4];
	uint8_t polarity[4];
	uint8_t phase[4];
	uint8_t prescale[4];
	uint8_t charlen[4];

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

static void
NAME_(send)(struct cpssp *cpssp)
{
	uint8_t shiftdir;
	uint8_t charlen;
	int i;
	int bit;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: val=0x%04x\n", __FUNCTION__,
				cpssp->NAME.txdata);
	}

	if (! cpssp->NAME.enable) {
		return;
	}

	charlen = cpssp->NAME.charlen[cpssp->NAME.dfsel];
	shiftdir = cpssp->NAME.shiftdir[cpssp->NAME.dfsel];

	assert(shiftdir == 0); /* FIXME */

	/* HACK */
	for (i = 0; i < 8; i++) {
		bit = (cpssp->NAME.csnrN >> i) & 1;

		NAME_(csN_set)(cpssp, i,
				bit ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0);
	}
	cpssp->NAME.rxdata = 0x0000;
	for (i = 0; i < charlen; i++) {
		bit = cpssp->NAME.state_somi;
		cpssp->NAME.rxdata <<= 1;
		cpssp->NAME.rxdata |= bit << 0;

		bit = (cpssp->NAME.txdata >> (charlen - 1)) & 1;
		cpssp->NAME.txdata <<= 1;
		NAME_(simo_set)(cpssp,
				bit ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0);

		NAME_(clk_set)(cpssp, SIG_STD_LOGIC_1);
		NAME_(clk_set)(cpssp, SIG_STD_LOGIC_0);
	}
	if (! cpssp->NAME.cshold) {
		for (i = 0; i < 8; i++) {
			bit = (cpssp->NAME.csdefN >> i) & 1;

			NAME_(csN_set)(cpssp, i,
					bit ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0);
		}
	}

	cpssp->NAME.txfull = 0;
	cpssp->NAME.rxempty = 0;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	int n;

	/* Global Control Register 0, 29.3.1, 1402 */
	cpssp->NAME.nreset = 0;

	/* Global Control Register 1, 29.3.2, 1403 */
	cpssp->NAME.enable = 0;
	cpssp->NAME.loopback = 0;
	cpssp->NAME.powerdown = 0;
	cpssp->NAME.clkmod = 0;
	cpssp->NAME.master = 0;

	/* Interrupt Register, 29.3.3, 1405 */
	cpssp->NAME.enablehighz = 0;
	cpssp->NAME.dmareqen = 0;
	cpssp->NAME.txintena = 0;
	cpssp->NAME.rxintena = 0;
	cpssp->NAME.ovrnintena = 0;
	cpssp->NAME.biterrena = 0;
	cpssp->NAME.desyncena = 0;
	cpssp->NAME.parerrena = 0;
	cpssp->NAME.timeoutena = 0;
	cpssp->NAME.dlenerrena = 0;

	/* Level Register, 29.3.4, 1407 */
	cpssp->NAME.txintlvl = 0;
	cpssp->NAME.rxintlvl = 0;
	cpssp->NAME.ovrnintlvl = 0;
	cpssp->NAME.biterrlvl = 0;
	cpssp->NAME.desynclvl = 0;
	cpssp->NAME.parerrlvl = 0;
	cpssp->NAME.timeoutlvl = 0;
	cpssp->NAME.dlenerrlvl = 0;

	/* Pin Control Register 0, 29.3.6, 1410 */
	cpssp->NAME.somifun = 0;
	cpssp->NAME.simofun = 0;
	cpssp->NAME.clkfun = 0;
	cpssp->NAME.enafun = 0;
	for (n = 0; n < 8; n++) {
		cpssp->NAME.scsofun[n] = 0;
	}

	/* Transmit Data Register 0, 29.3.12, 1416 */
	cpssp->NAME.txdata = 0x0000;

	/* Transmit Data Register 1, 29.3.13, 1417 */
	cpssp->NAME.cshold = 0;
	cpssp->NAME.wdel = 0;
	cpssp->NAME.dfsel = 0;
	cpssp->NAME.csnrN = 0;
	/* cpssp->NAME.txdata = 0x0000; see above. */

	/* Receive Buffer Register, 29.3.14, 1418 */
	cpssp->NAME.rxempty = 1;
	cpssp->NAME.rxovr = 0;
	cpssp->NAME.txfull = 0;
	cpssp->NAME.biterr = 0;
	cpssp->NAME.desync = 0;
	cpssp->NAME.parerr = 0;
	cpssp->NAME.timeout = 0;
	cpssp->NAME.dlenerr = 0;
	cpssp->NAME.rxdata = 0x0000;

	/* Receive Buffer Emulation Register, 29.3.15, 1420 */
	/* See above... */

	/* Delay Register, 29.3.16, 1421 */
	cpssp->NAME.c2tdelay = 0;
	cpssp->NAME.t2cdelay = 0;
	cpssp->NAME.t2edelay = 0;
	cpssp->NAME.c2edelay = 0;

	/* Chip Select Default Register, 29.3.17, 1424 */
	cpssp->NAME.csdefN = 0xff;
	for (n = 0; n < 8; n++) {
		NAME_(csN_set)(cpssp, n, SIG_STD_LOGIC_1);
	}

	/* Format 0 - 3 Register, 29.3.18, page 1425 */
	for (n = 0; n < 4; n++) {
		cpssp->NAME.wdelay[n] = 0;
		cpssp->NAME.parpol[n] = 0;
		cpssp->NAME.parena[n] = 0;
		cpssp->NAME.waitena[n] = 0;
		cpssp->NAME.shiftdir[n] = 0;
		cpssp->NAME.discstimers[n] = 0;
		cpssp->NAME.polarity[n] = 0;
		cpssp->NAME.phase[n] = 0;
		cpssp->NAME.prescale[n] = 0;
		cpssp->NAME.charlen[n] = 0;
	}

	/* FIXME */
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	int n;

	assert(bs == 0xf); /* FIXME */

	addr &= 0xfff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		/* Global Control Register 0 */
		cpssp->NAME.nreset = (val >> 0) & 1;
		if (! cpssp->NAME.nreset) {
			NAME_(reset)(cpssp);
		}
		break;

	case 0x004:
		/* Global Control Register 1 */
		cpssp->NAME.enable = (val >> 24) & 1;
		cpssp->NAME.loopback = (val >> 16) & 1;
		cpssp->NAME.powerdown = (val >> 8) & 1;
		cpssp->NAME.clkmod = (val >> 1) & 1;
		cpssp->NAME.master = (val >> 0) & 1;
		break;

	case 0x008:
		/* Interrupt Register, 29.3.3, 1405 */
		cpssp->NAME.enablehighz = (val >> 24) & 1;
		cpssp->NAME.dmareqen = (val >> 16) & 1;
		cpssp->NAME.txintena = (val >> 9) & 1;
		cpssp->NAME.rxintena = (val >> 8) & 1;
		cpssp->NAME.ovrnintena = (val >> 6) & 1;
		cpssp->NAME.biterrena = (val >> 4) & 1;
		cpssp->NAME.desyncena = (val >> 3) & 1;
		cpssp->NAME.parerrena = (val >> 2) & 1;
		cpssp->NAME.timeoutena = (val >> 1) & 1;
		cpssp->NAME.dlenerrena = (val >> 0) & 1;
		break;

	case 0x00c:
		/* Level Register, 29.3.4, 1407 */
		cpssp->NAME.txintlvl = (val >> 9) & 1;
		cpssp->NAME.rxintlvl = (val >> 8) & 1;
		cpssp->NAME.ovrnintlvl = (val >> 6) & 1;
		cpssp->NAME.biterrlvl = (val >> 4) & 1;
		cpssp->NAME.desynclvl = (val >> 3) & 1;
		cpssp->NAME.parerrlvl = (val >> 2) & 1;
		cpssp->NAME.timeoutlvl = (val >> 1) & 1;
		cpssp->NAME.dlenerrlvl = (val >> 0) & 1;
		break;

	case 0x010:
		/* Flag Register */
		goto todo;

	case 0x014:
		/* Pin Control Register 0, 29.3.6, 1410 */
		cpssp->NAME.somifun = (val >> 11) & 1;
		cpssp->NAME.simofun = (val >> 10) & 1;
		cpssp->NAME.clkfun = (val >> 9) & 1;
		cpssp->NAME.enafun = (val >> 8) & 1;
		for (n = 0; n < 8; n++) {
			cpssp->NAME.scsofun[n] = (val >> n) & 1;
		}
		break;

	case 0x018:
		/* Pin Control Register 1 */
		goto todo;

	case 0x01c:
		/* Pin Control Register 2 */
		goto todo;

	case 0x020:
		/* Pin Control Register 3 */
		goto todo;

	case 0x024:
		/* Pin Control Register 4 */
		goto todo;

	case 0x028:
		/* Pin Control Register 5 */
		goto todo;

	case 0x038:
		/* Transmit Data Register 0, 29.3.12, 1416 */
		cpssp->NAME.txdata = (val >> 0) & 0xffff;
		NAME_(send)(cpssp);
		break;

	case 0x03c:
		/* Transmit Data Register 1, 29.3.13, 1417 */
		cpssp->NAME.cshold = (val >> 28) & 1;
		cpssp->NAME.wdel = (val >> 26) & 1;
		cpssp->NAME.dfsel = (val >> 24) & 0x3;
		cpssp->NAME.csnrN = (val >> 16) & 0xff;
		cpssp->NAME.txdata = (val >> 0) & 0xffff;
		NAME_(send)(cpssp);
		break;

	case 0x040:
		/* Receive Buffer Register, 29.3.14, 1418 */
		/* Bit 31-0: Read-Only */
		goto read_only;

	case 0x044:
		/* Receive Buffer Emulation Register, 29.3.15, 1420 */
		/* Bit 31-0: Read-Only */
		goto read_only;

	case 0x048:
		/* Delay Register, 29.3.16, 1421 */
		cpssp->NAME.c2tdelay = (val >> 24) & 0xff;
		cpssp->NAME.t2cdelay = (val >> 16) & 0xff;
		cpssp->NAME.t2edelay = (val >>  8) & 0xff;
		cpssp->NAME.c2edelay = (val >>  0) & 0xff;
		break;

	case 0x04c:
		/* Chip Select Default Register, 29.3.17, 1424 */
		/* Bit 31-8: Reserved */
		cpssp->NAME.csdefN = (val >> 0) & 0xff;
		break;

	case 0x050 ... 0x05c:
		/* Format 0 - 3 Register, 29.3.18, page 1425 */
		n = (addr - 0x050) >> 2;

		/* Bit 31-30: Reserved */
		cpssp->NAME.wdelay[n] = (val >> 24) & 0x3f;
		cpssp->NAME.parpol[n] = (val >> 23) & 1;
		cpssp->NAME.parena[n] = (val >> 22) & 1;
		cpssp->NAME.waitena[n] = (val >> 21) & 1;
		cpssp->NAME.shiftdir[n] = (val >> 20) & 1;
		/* Bit 19: Reserved */
		cpssp->NAME.discstimers[n] = (val >> 18) & 1;
		cpssp->NAME.polarity[n] = (val >> 17) & 1;
		cpssp->NAME.phase[n] = (val >> 16) & 1;
		cpssp->NAME.prescale[n] = (val >> 8) & 0xff;
		/* Bit 7-5: Reserved */
		cpssp->NAME.charlen[n] = (val >> 0) & 0x1f;
		break;

	case 0x064:
		/* Interrupt Vector Register 1 */
		goto todo;

	default:
	todo:	;
	read_only:;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	int n;

	assert(bs == 0xf); /* FIXME */

	addr &= 0xfff;

	switch (addr) {
	case 0x000:
		/* Global Control Register 0 */
		*valp = 0;
		*valp |= cpssp->NAME.nreset << 0;
		break;

	case 0x004:
		/* Global Control Register 1 */
		*valp = 0;
		*valp |= cpssp->NAME.enable << 24;
		*valp |= cpssp->NAME.loopback << 16;
		*valp |= cpssp->NAME.powerdown << 8;
		*valp |= cpssp->NAME.clkmod << 1;
		*valp |= cpssp->NAME.master << 0;
		break;

	case 0x008:
		/* Interrupt Register, 29.3.3, 1405 */
		*valp = 0;
		*valp |= cpssp->NAME.enablehighz << 24;
		*valp |= cpssp->NAME.dmareqen << 16;
		*valp |= cpssp->NAME.txintena << 9;
		*valp |= cpssp->NAME.rxintena << 8;
		*valp |= cpssp->NAME.ovrnintena << 6;
		*valp |= cpssp->NAME.biterrena << 4;
		*valp |= cpssp->NAME.desyncena << 3;
		*valp |= cpssp->NAME.parerrena << 2;
		*valp |= cpssp->NAME.timeoutena << 1;
		*valp |= cpssp->NAME.dlenerrena << 0;
		break;

	case 0x00c:
		/* Level Register, 29.3.4, 1407 */
		*valp = 0;
		*valp |= cpssp->NAME.txintlvl << 9;
		*valp |= cpssp->NAME.rxintlvl << 8;
		*valp |= cpssp->NAME.ovrnintlvl << 6;
		*valp |= cpssp->NAME.biterrlvl << 4;
		*valp |= cpssp->NAME.desynclvl << 3;
		*valp |= cpssp->NAME.parerrlvl << 2;
		*valp |= cpssp->NAME.timeoutlvl << 1;
		*valp |= cpssp->NAME.dlenerrlvl << 0;
		break;

	case 0x010:
		/* Flag Register */
		goto todo;

	case 0x014:
		/* Pin Control Register 0, 29.3.6, 1410 */
		*valp = 0;
		*valp |= cpssp->NAME.somifun << 11;
		*valp |= cpssp->NAME.simofun << 10;
		*valp |= cpssp->NAME.clkfun << 9;
		*valp |= cpssp->NAME.enafun << 8;
		for (n = 0; n < 8; n++) {
			*valp |= cpssp->NAME.scsofun[n] << n;
		}
		break;

	case 0x018:
		/* Pin Control Register 1 */
		goto todo;

	case 0x01c:
		/* Pin Control Register 2 */
		goto todo;

	case 0x020:
		/* Pin Control Register 3 */
		goto todo;

	case 0x024:
		/* Pin Control Register 4 */
		goto todo;

	case 0x028:
		/* Pin Control Register 5 */
		goto todo;

	case 0x038:
		/* Transmit Data Register 0, 29.3.12, 1416 */
		*valp = 0;
		*valp |= 0x0000 << 0;
		break;

	case 0x03c:
		/* Transmit Data Register 1, 29.3.13, 1417 */
		*valp = 0;
		*valp |= cpssp->NAME.cshold << 28;
		*valp |= cpssp->NAME.wdel << 26;
		*valp |= cpssp->NAME.dfsel << 24;
		*valp |= cpssp->NAME.csnrN << 16;
		*valp |= 0x0000 << 0;
		break;

	case 0x040:
		/* Receive Buffer Register, 29.3.14, 1418 */
		*valp = 0;
		*valp |= cpssp->NAME.rxempty << 31;
			cpssp->NAME.rxempty = 1;
		*valp |= cpssp->NAME.rxovr << 30;
			cpssp->NAME.rxovr = 0;
		*valp |= cpssp->NAME.txfull << 29;
			cpssp->NAME.txfull = 0;
		*valp |= cpssp->NAME.biterr << 28;
			cpssp->NAME.biterr = 0;
		*valp |= cpssp->NAME.desync << 27;
			cpssp->NAME.desync = 0;
		*valp |= cpssp->NAME.parerr << 26;
			cpssp->NAME.parerr = 0;
		*valp |= cpssp->NAME.timeout << 25;
			cpssp->NAME.timeout = 0;
		*valp |= cpssp->NAME.dlenerr << 24;
			cpssp->NAME.dlenerr = 0;
		/* Bit 32-16: Reserved */
		*valp |= cpssp->NAME.rxdata << 0;
		break;

	case 0x044:
		/* Receive Buffer Emulation Register, 29.3.15, 1420 */
		*valp = 0;
		/* Bit 31-16: Reserved */
		*valp |= cpssp->NAME.rxdata << 0;
		break;

	case 0x048:
		/* Delay Register, 29.3.16, 1421 */
		*valp = 0;
		*valp |= cpssp->NAME.c2tdelay << 24;
		*valp |= cpssp->NAME.t2cdelay << 16;
		*valp |= cpssp->NAME.t2edelay << 8;
		*valp |= cpssp->NAME.c2edelay << 0;
		break;

	case 0x04c:
		/* Chip Select Default Register, 29.3.17, 1424 */
		*valp = 0;
		/* Bit 31-8: Reserved */
		*valp |= cpssp->NAME.csdefN << 0;
		break;

	case 0x050 ... 0x05c:
		/* Format 0 - 3 Register, 29.3.18, page 1425 */
		n = (addr - 0x050) >> 2;

		*valp = 0;
		/* Bit 31-30: Reserved */
		*valp |= cpssp->NAME.wdelay[n] << 24;
		*valp |= cpssp->NAME.parpol[n] << 23;
		*valp |= cpssp->NAME.parena[n] << 22;
		*valp |= cpssp->NAME.waitena[n] << 21;
		*valp |= cpssp->NAME.shiftdir[n] << 20;
		/* Bit 19: Reserved */
		*valp |= cpssp->NAME.discstimers[n] << 18;
		*valp |= cpssp->NAME.polarity[n] << 17;
		*valp |= cpssp->NAME.phase[n] << 16;
		*valp |= cpssp->NAME.prescale[n] << 8;
		/* Bit 7-5: Reserved */
		*valp |= cpssp->NAME.charlen[n] << 0;
		break;

	case 0x064:
		/* Interrupt Vector Register 1 */
		goto todo;

	default:
	todo:	;
		*valp = cpssp->NAME.reg[addr >> 2];

		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(somi_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->NAME.state_somi = 1500 <= SIG_mV(val);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
