/*
 * Copyright (C) 2006-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_RTC		0

#ifdef INCLUDE

#include <sys/time.h>
#include <errno.h>

#include "conv_gen.h" /* FIXME */
#include "umutil.h" /* FIXME */

#endif /* INCLUDE */
#ifdef STATE

struct {
	void *media;

	uint8_t reg;
	uint8_t reg_ext;

	uint8_t freq_select;
	uint8_t control;
	uint8_t intr_flags;
	
	uint8_t sec;
	uint8_t sec_alarm;
	uint8_t min;
	uint8_t min_alarm;
	uint8_t hour;
	uint8_t hour_alarm;
	uint8_t day_of_week;
	uint8_t day_of_month;
	uint8_t month;
	uint8_t year;

	unsigned long long tsc_call;
	unsigned long long tsc_PI_timer;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

/* ------------------------------------------------ */
/* see <linux/mc146818rtc.h> and PC-Hardware p. 753 */
/* ------------------------------------------------ */

/*
 * Clock data registers
 */
#define RTC_SECONDS		0
#define RTC_SECONDS_ALARM	1
#define RTC_MINUTES		2
#define RTC_MINUTES_ALARM	3
#define RTC_HOURS		4
#define RTC_HOURS_ALARM		5
#define RTC_DAY_OF_WEEK		6
#define RTC_DAY_OF_MONTH	7
#define RTC_MONTH		8
#define RTC_YEAR		9

/*
 * Clock control registers
 */
#define RTC_FREQ_SELECT	10
/* update-in-progress  - set to "1" 244 microsecs before RTC goes off the bus,
 * reset after update (may take 1.984ms @ 32768Hz RefClock) is complete,
 * totalling to a max high interval of 2.228 ms.
 */
#define RTC_UIP		0x80
#define RTC_DIV_CTL	0x70
   /* divider control: refclock values 4.194 / 1.049 MHz / 32.768 kHz */
#define RTC_CLCK_4MHZ	0x00
#define RTC_CLCK_1MHZ	0x10
#define RTC_CLCK_32KHZ	0x20
   /* 2 values for divider stage reset, others for "testing purposes only" */
#define RTC_DIV_RESET1	0x60
#define RTC_DIV_RESET2	0x70
  /* Periodic intr. / Square wave rate select. 0=none, 1=32.8kHz,... 15=2Hz */
#define RTC_RATE_SELECT	0x0F

#define RTC_CONTROL	11
#define RTC_SET		0x80 /* disable updates for clock setting */
#define RTC_PIE		0x40 /* periodic interrupt enable */
#define RTC_AIE		0x20 /* alarm interrupt enable */
#define RTC_UIE		0x10 /* update-finished interrupt enable */
#define RTC_SQWE	0x08 /* enable square-wave output */
#define RTC_DM_BINARY	0x04 /* all time/date values are BCD if clear */
#define RTC_24H		0x02 /* 24 hour mode - else hours bit 7 means pm */
#define RTC_DST_EN	0x01 /* auto switch DST - works f. USA only */

#define RTC_INTR_FLAGS	12
			     /* caution - cleared by read */
#define RTC_IRQF	0x80 /* any of the following 3 is active */
#define RTC_PF		0x40
#define RTC_AF		0x20
#define RTC_UF		0x10

#define RTC_VALID	13
#define RTC_VRT		0x80 /* valid RAM and time */

static void
NAME_(umktime)(
	unsigned long long t,
	int *year,
	int *month,
	int *day,
	int *hour,
	int *min,
	int *sec
)
{
	const int secspermin = 60;
	const int secsperhour = 60 * secspermin;
	const int secsperday = 24 * secsperhour;
	unsigned char dayspermonth[12] = {
		31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
	};
	unsigned int daysperyear;

	*year = 1970;
	*month = 0;
	*day = 0;
	*hour = 0;
	*min = 0;
	*sec = 0;
	daysperyear = (*year % 4 == 0) ? 366 : 365;
	while (daysperyear * secsperday <= t) {
		assert(365 <= daysperyear && daysperyear <= 366);
		t -= daysperyear * secsperday;
		(*year)++;
		daysperyear = (*year % 4 == 0) ? 366 : 365;
	}
	assert(1970 <= *year);
	dayspermonth[1] = (*year % 4 == 0) ? 29 : 28;
	while (dayspermonth[*month] * secsperday <= t) {
		t -= dayspermonth[*month] * secsperday;
		(*month)++;
	}
	(*month)++;	/* 0->1, 1->2, ... */
	assert(1 <= *month && *month <= 12);
	while (24 * 60 * 60 <= t) {
		t -= 24 * 60 * 60;
		(*day)++;
	}
	(*day)++;	/* 0->1, 1->2, ... */
	assert(1 <= *day && *day <= 31);
	while (60 * 60 <= t) {
		t -= 60 * 60;
		(*hour)++;
	}
	assert(0 <= *hour && *hour <= 23);
	while (60 <= t) {
		t -= 60;
		(*min)++;
	}
	assert(0 <= *min && *min <= 59);
	*sec = t;
	assert(0 <= *sec && *sec <= 59);

#if DEBUG_RTC
	fprintf(stderr, "date=%d-%d-%d %02d:%02d:%02d\n",
			*year, *month, *day, *hour, *min, *sec);
#endif
}

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	if ((cpssp->NAME.intr_flags & cpssp->NAME.control) & (RTC_PIE | RTC_AIE | RTC_UIE)) {
		NAME_(irq_set)(cpssp, 1);
		cpssp->NAME.intr_flags |= RTC_IRQF;
	} else {
		NAME_(irq_set)(cpssp, 0);
		cpssp->NAME.intr_flags &= ~RTC_IRQF;
	}
}

static void
NAME_(timer)(void *_css)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	if (! cpssp->state_power)
		return;

	cpssp->NAME.intr_flags |= RTC_PF;
	NAME_(irq_update)(cpssp);

	if (cpssp->NAME.freq_select & RTC_RATE_SELECT) {
		unsigned int sel;

		sel = cpssp->NAME.freq_select & RTC_RATE_SELECT;
		cpssp->NAME.tsc_PI_timer += TIME_HZ >> (16 - sel);
		time_call_at(cpssp->NAME.tsc_PI_timer, NAME_(timer), cpssp);
	}
}

static uint8_t
NAME_(bcd_add)(
	unsigned int bcd,
	uint8_t val,
	uint8_t first,
	uint8_t last,
	unsigned int *carryp
)
{
	if (bcd) {
		/* Switch to binary mode. */
		val = ((val >> 4) & 0xf) * 10 + ((val >> 0) & 0xf);
	}

	/* Do calculation. */
	val += *carryp;
	if (last < val) {
		*carryp = 1;
		val = first;
	} else {
		*carryp = 0;
	}

	if (bcd) {
		/* Switch back to bcd mode. */
		val = ((val / 10) << 4) | ((val % 10) << 0);
	}

	return val;
}

/*
 * MC146818 needs a max high interval of 2.228 ms to update.
 */
static void
NAME_(clock)(void *_css)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	if (cpssp->NAME.freq_select & RTC_UIP) {
		/*
		 * Generating UIP -> -UIP change.
		 */
		cpssp->NAME.freq_select &= ~RTC_UIP;

		/*
		 * Interrupt if alarm time is actual time and interrupt
		 * is desired for this reason.
		 */
		if (((cpssp->NAME.sec_alarm & 0xc0) == 0xc0
				|| cpssp->NAME.sec_alarm == cpssp->NAME.sec)
		 && ((cpssp->NAME.min_alarm & 0xc0) == 0xc0
				|| cpssp->NAME.min_alarm == cpssp->NAME.min)
		 && ((cpssp->NAME.hour_alarm & 0xc0) == 0xc0
				|| cpssp->NAME.hour_alarm == cpssp->NAME.hour)) {
			cpssp->NAME.intr_flags |= RTC_AF;
			NAME_(irq_update)(cpssp);
		}

		/*
		 * Interrupt if UIP interrupt is wanted.
		 */
		cpssp->NAME.intr_flags |= RTC_UF;
		NAME_(irq_update)(cpssp);

		/* Set new wakeup timer. */
		cpssp->NAME.tsc_call += TIME_HZ - (TIME_HZ * 2228 / 1000000);
		time_call_at(cpssp->NAME.tsc_call, NAME_(clock), cpssp);

	} else {
		/*
		 * Generate -UIP -> UIP change.
		 */
		cpssp->NAME.freq_select |= RTC_UIP;

		if (! (cpssp->NAME.control & RTC_SET)) {
			static const int ndays[1 + 0x12] = {
				0, /* Not used. */
				31, 29, 31, 30, 31, 30,
				31, 31, 30,
				31, 30, 31,	/* In case of binary mode. */
				0, 0, 0,
				31, 30, 31	/* In case of bcd mode. */
			};
			unsigned int bcd;
			unsigned int carry;
			unsigned int carry2;

			bcd = (cpssp->NAME.control & RTC_DM_BINARY) ? 0 : 1;
			carry = 1;
			cpssp->NAME.sec = NAME_(bcd_add)(bcd, cpssp->NAME.sec, 0, 59, &carry);
			cpssp->NAME.min = NAME_(bcd_add)(bcd, cpssp->NAME.min, 0, 59, &carry);
			if (cpssp->NAME.control & RTC_24H) {
				cpssp->NAME.hour = NAME_(bcd_add)(bcd, cpssp->NAME.hour, 0, 23, &carry);
			} else {
				uint8_t pm;
				uint8_t hour;

				pm = (cpssp->NAME.hour >> 7) & 1;
				hour = (cpssp->NAME.hour >> 0) & 0x7f;

				hour = NAME_(bcd_add)(bcd, hour, 0, 11, &carry);
				pm = NAME_(bcd_add)(bcd, pm, 0, 1, &carry);

				cpssp->NAME.hour = (pm << 7) | (hour << 0);
			}

			carry2 = carry;
			cpssp->NAME.day_of_week = NAME_(bcd_add)(bcd, cpssp->NAME.day_of_week, 1, 7, &carry2);

			if (cpssp->NAME.month == 2 && cpssp->NAME.year % 4) {
				cpssp->NAME.day_of_month = NAME_(bcd_add)(bcd,
					cpssp->NAME.day_of_month,
					1, 28, &carry);
			} else {
				cpssp->NAME.day_of_month = NAME_(bcd_add)(bcd,
					cpssp->NAME.day_of_month,
					1, ndays[cpssp->NAME.month], &carry);
			}
			cpssp->NAME.year = NAME_(bcd_add)(bcd, cpssp->NAME.year, 0, 99, &carry);
		}

		/* Set new wakeup timer. */
		cpssp->NAME.tsc_call += TIME_HZ * 2228 / 1000000;
		time_call_at(cpssp->NAME.tsc_call, NAME_(clock), cpssp);
	}
}

static int
NAME_(inb)(struct cpssp *cpssp, unsigned char *valuep, unsigned short addr)
{
	int ret;

	switch (addr) {
	case 0:
		/* correct? FIXME VOSSI */
		*valuep = cpssp->NAME.reg;
		return 0;

	case 1:
		switch (cpssp->NAME.reg) {
		case RTC_SECONDS:
			*valuep = cpssp->NAME.sec;
			break;

		case RTC_SECONDS_ALARM:
			*valuep = cpssp->NAME.sec_alarm;
			break;

		case RTC_MINUTES:
			*valuep = cpssp->NAME.min;
			break;
			
		case RTC_MINUTES_ALARM:
			*valuep = cpssp->NAME.min_alarm;
			break;

		case RTC_HOURS:
			*valuep = cpssp->NAME.hour;
			break;
			
		case RTC_HOURS_ALARM:
			*valuep = cpssp->NAME.hour_alarm;
			break;

		case RTC_DAY_OF_WEEK:
			*valuep = cpssp->NAME.day_of_week;
			break;

		case RTC_DAY_OF_MONTH:
			*valuep = cpssp->NAME.day_of_month;
			break;

		case RTC_MONTH:
			*valuep = cpssp->NAME.month;
			break;

		case RTC_YEAR:
			*valuep = cpssp->NAME.year;
			break;

		case RTC_FREQ_SELECT:
			*valuep = cpssp->NAME.freq_select;
			break;

		case RTC_CONTROL:
			*valuep = cpssp->NAME.control;
			break;

		case RTC_INTR_FLAGS:
			/* cpssp->NAME.intr_flags register is cleared by read */
			/* and set by interrupt */
			*valuep = cpssp->NAME.intr_flags;
			cpssp->NAME.intr_flags = 0x00;
			NAME_(irq_update)(cpssp);
			break;

		case RTC_VALID:
			*valuep = RTC_VRT;
			break;

		default:
			/* Those are read from the cmosram-file. */
			assert(cpssp->NAME.reg < 128);
			ret = storage_read(cpssp->NAME.media,
					valuep, sizeof(*valuep),
					cpssp->NAME.reg);
			assert(ret == sizeof(*valuep));
			break;
		}
#if DEBUG_RTC
		fprintf(stderr, "%s: read 0x%02x from register 0x%02x\n",
				__FUNCTION__, *valuep, cpssp->NAME.reg);
#endif
		return 0;

	case 2:
		*valuep = cpssp->NAME.reg_ext;
		return 0;

	case 3:
		ret = storage_read(cpssp->NAME.media,
				valuep, sizeof(*valuep),
				cpssp->NAME.reg_ext + 128);
		assert(ret == sizeof(*valuep));
#if DEBUG_RTC
		fprintf(stderr, "%s: read 0x%02x from register 0x%02x\n",
				__FUNCTION__, *valuep, cpssp->NAME.reg_ext + 128);
#endif
		return 0;

	default:
		return -1;
	}
}

static int
NAME_(outb)(struct cpssp *cpssp, unsigned char value, unsigned short addr)
{
	int ret;

	switch (addr) {
	case 0:
		/* Ignore bit 7 (NMI masking bit) - FIXME VOSSI */
		cpssp->NAME.reg = value & 0x7f;
		return 0;

	case 1:
		switch (cpssp->NAME.reg) {
		case RTC_SECONDS:
			cpssp->NAME.sec = value;
			break;

		case RTC_SECONDS_ALARM:
			cpssp->NAME.sec_alarm = value;
			break;

		case RTC_MINUTES:
			cpssp->NAME.min = value;
			break;

		case RTC_MINUTES_ALARM:
			cpssp->NAME.min_alarm = value;
			break;

		case RTC_HOURS:
			cpssp->NAME.hour = value;
			break;
			
		case RTC_HOURS_ALARM:
			cpssp->NAME.hour_alarm = value;
			break;

		case RTC_DAY_OF_WEEK:
			cpssp->NAME.day_of_week = value;
			break;

		case RTC_DAY_OF_MONTH:
			cpssp->NAME.day_of_month = value;
			break;

		case RTC_MONTH:
			cpssp->NAME.month = value;
			break;

		case RTC_YEAR:
			cpssp->NAME.year = value;
			break;

		case RTC_FREQ_SELECT:
			/* Bit 7 is read-only. */
			value &= ~RTC_UIP;
			value |= cpssp->NAME.freq_select & ~RTC_UIP;

			/* Start/update/stop timer. */
			/* Timer update/stop not implemented - FIXME */
			if (((cpssp->NAME.freq_select & RTC_RATE_SELECT) == 0)
			 && (value & RTC_RATE_SELECT) != 0){
				unsigned int sel;

				sel = value & 0xf;
				cpssp->NAME.tsc_PI_timer = time_virt();
				cpssp->NAME.tsc_PI_timer += TIME_HZ >> (16 - sel);
				time_call_at(cpssp->NAME.tsc_PI_timer,
						NAME_(timer), cpssp);
			}

			cpssp->NAME.freq_select = value;
			break;

		case RTC_CONTROL:
			cpssp->NAME.control = value;
			break;

		case RTC_INTR_FLAGS:
		case RTC_VALID:
			faum_log(FAUM_LOG_WARNING, "RTC", "",
					"Writing 0x%02x to read-only register 0x%02x.\n",
					value, cpssp->NAME.reg);
			break;

		/* Those are read from the cmosram-file. */
		case 14 ... 127:
			ret = storage_write(cpssp->NAME.media,
					&value, sizeof(value),
					cpssp->NAME.reg);
			assert(ret == sizeof(value));
			break;

		default:
			assert(0);	/* Cannot happen. */
			/*NOTREACHED*/
		}

#if DEBUG_RTC
		fprintf(stderr, "%s: wrote 0x%02x to register 0x%02x\n", __FUNCTION__,
				value, cpssp->NAME.reg);
#endif
		return 0;

	case 2:
		/* Ignore bit 7 */
		cpssp->NAME.reg_ext = value & 0x7f;
		return 0;

	case 3:
		ret = storage_write(cpssp->NAME.media,
				&value, sizeof(value),
				cpssp->NAME.reg_ext + 128);
		assert(ret == sizeof(value));

#if DEBUG_RTC
		fprintf(stderr, "%s: wrote 0x%02x to register 0x%02x\n", __FUNCTION__,
				value, cpssp->NAME.reg_ext + 128);
#endif
		return 0;

	default:
		return -1;
	}
}

static uint8_t
NAME_(bin_to_bcd)(uint8_t val)
{
	val = ((val / 10) << 4) | ((val % 10) << 0);

	return val;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.freq_select = 0x00;
}

static void
NAME_(init)(struct cpssp *cpssp)
{
	cpssp->NAME.tsc_call = TIME_HZ;
	time_call_at(cpssp->NAME.tsc_call, NAME_(clock), cpssp);
}

static void
NAME_(create)(
	struct cpssp *cpssp,
	const char *name,
	const char *rtc_start,
	const char *cmos_file
)
{
	int year;
	int month;
	int day_of_month;
	int hour;
	int min;
	int sec;
	int ret;
	unsigned int seconds;
	char cmos[128 + 128];

	system_name_push(name);

	if (rtc_start
	 && strcmp(rtc_start, "-1") != 0) {
		/*
		 * Start time given.
		 */
		errno = 0;
		seconds = strtol(rtc_start, NULL, 0);
		if (errno) {
			fprintf(stderr, "ERROR: Bad starting time (\"%s\") given.\n",
					rtc_start);
			exit(1);
		}

	} else if (simsetup.deterministic) {
		/*
		 * Deterministic mode without a given time
		 */
		fprintf(stderr, "WARNING: Deterministic mode without a given starting time!\n");
		seconds = 1500*1000*1000;
		fprintf(stderr, "Continuing with time=%u.\n", seconds);
		
	} else {
		/*
		 * Use current time to initialize clock.
		 */
		struct timeval tv;

		ret = gettimeofday(&tv, (struct timezone *) 0);
		assert(0 <= ret);

		seconds = tv.tv_sec;
	}

	NAME_(umktime)(seconds,
			&year, &month, &day_of_month, &hour, &min, &sec);

	year %= 100;

	cpssp->NAME.sec = NAME_(bin_to_bcd)(sec);
	cpssp->NAME.min = NAME_(bin_to_bcd)(min);
	cpssp->NAME.hour = NAME_(bin_to_bcd)(hour);
	cpssp->NAME.day_of_month = NAME_(bin_to_bcd)(day_of_month);
	/* Note: 1970-01-01 was a thursday. */
	cpssp->NAME.day_of_week = ((seconds / 60 / 24) + 5) % 7 + 1;
	cpssp->NAME.month = NAME_(bin_to_bcd)(month);
	cpssp->NAME.year = NAME_(bin_to_bcd)(year);

	cpssp->NAME.freq_select = 0;
	cpssp->NAME.reg = -1;
	cpssp->NAME.control = RTC_24H;
	cpssp->NAME.intr_flags = 0x00;

	cpssp->NAME.media = storage_create(system_path(), sizeof(cmos),
			buildpath(ROMDIR, cmos_file),
			conv_gen_open, conv_gen_close, conv_gen_read);
	assert(cpssp->NAME.media);

	system_name_pop();
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
	int ret;

	ret = storage_destroy(cpssp->NAME.media);
	assert(0 <= ret);
}

#endif /* BEHAVIOR */

#undef DEBUG_RTC
