/*
 * $Id: chip_fau_pci_real.c,v 1.20 2010-11-22 08:31:02 vrsieh Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG	1

#define NVIDIA_HACK	1

#include "config.h"
#include <assert.h>
#include <fcntl.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "glue-io.h"
#include "../../real-dev/faum-dev.h"

#include "chip_fau_pci_real.h"

#define COMP_(x) chip_fau_pci_real_ ## x

struct cpssp {
	int fd;
	struct faum_dev_info real;

	struct sig_boolean_or *port_intA;

	int mem_enabled;
	int io_enabled;
	uint32_t virt_base[7];
	uint8_t interrupt_line;
};

static void
COMP_(irq_update)(struct cpssp *cpssp, int val)
{
	static int oldval = -1;

	if (val != oldval) {
#if DEBUG
		fprintf(stderr, "%s: val=%d\n", __FUNCTION__, val);
#endif

		sig_boolean_or_set(cpssp->port_intA, cpssp, val);

		oldval = val;
	}
}

static void
COMP_(isr)(int fd, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t val;
	int ret;

	ret = read(fd, &val, 1);
	assert(ret == 1);

	COMP_(irq_update)(cpssp, val);
}

static int
COMP_(cread0)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = _cpssp;
	uint32_t val32;

	addr &= 0xfc;

	switch (addr) {
	case 0x10: /* Base Address 0 */
	case 0x14: /* Base Address 1 */
	case 0x18: /* Base Address 2 */
	case 0x1c: /* Base Address 3 */
	case 0x20: /* Base Address 4 */
	case 0x24: /* Base Address 5 */
		val32 = cpssp->virt_base[(addr - 0x10) / 4];
		if (cpssp->real.region[(addr - 0x10) / 4].base & 1) {
			/* I/O */
			val32 |= cpssp->real.region[(addr - 0x10) / 4].base & 0x3;
		} else {
			/* Mem */
			val32 |= cpssp->real.region[(addr - 0x10) / 4].base & 0xf;
		}
		break;

	case 0x3c: /* Max_Lat, Min_Gnt, Interrupt Pin, Interrupt Line */
		/* FIXME */
		val32 = cpssp->interrupt_line << 0;
		val32 |= 1 << 8; /* INT A# */
		val32 |= 1 << 16;
		val32 |= 1 << 24;
		break;

	default: {
		struct faum_dev_param param;
		int ret;

		param.addr = addr;
		param.bs = bs;
		ret = ioctl(cpssp->fd, FAUM_DEV_C0R, &param);
		assert(0 <= ret);
		val32 = param.data;

		COMP_(irq_update)(cpssp, param.irq);
		break;
	    }
	}

	*valp = val32;

#if DEBUG
	fprintf(stderr, "%s: 0x%08x(0x%x) -> 0x%08x\n", __FUNCTION__,
			addr, bs, *valp);
#endif

	return 0;
}

static int
COMP_(cwrite0)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = _cpssp;
	struct faum_dev_param param;
	int ret;

#if DEBUG
	fprintf(stderr, "%s: 0x%08x(0x%x) <- 0x%08x\n", __FUNCTION__,
			addr, bs, val);
#endif

	addr &= 0xfc;

	switch (addr) {
	case 0x04: /* Status, Command */
		if ((bs >> 0) & 1) {
			cpssp->mem_enabled = (val >> 1) & 1;
			cpssp->io_enabled = (val >> 0) & 1;
		}
		goto def;

	case 0x10: /* Base Address 0 */
	case 0x14: /* Base Address 1 */
	case 0x18: /* Base Address 2 */
	case 0x1c: /* Base Address 3 */
	case 0x20: /* Base Address 4 */
	case 0x24: /* Base Address 5 */ {
		uint32_t base;

		base = cpssp->virt_base[(addr - 0x10) / 4];
		if ((bs >> 0) & 1) {
			base &= ~0x000000ff;
			base |= val & 0x000000ff;
		}
		if ((bs >> 1) & 1) {
			base &= ~0x0000ff00;
			base |= val & 0x0000ff00;
		}
		if ((bs >> 2) & 1) {
			base &= ~0x00ff0000;
			base |= val & 0x00ff0000;
		}
		if ((bs >> 3) & 1) {
			base &= ~0xff000000;
			base |= val & 0xff000000;
		}
		base &= ~(cpssp->real.region[(addr - 0x10) / 4].size - 1);
		cpssp->virt_base[(addr - 0x10) / 4] = base;

		goto def;
	    }
	case 0x3c: /* Max_Lat, Min_Gnt, Interrupt Pin, Interrupt Line */
		if ((bs >> 0) & 1) {
			cpssp->interrupt_line = (val >> 0) & 0xff;
		}

		bs &= ~(1 << 0);
		goto def;

	default:
	def:	;
		if (bs) {
			param.addr = addr;
			param.bs = bs;
			param.data = val;
			ret = ioctl(cpssp->fd, FAUM_DEV_C0W, &param);
			assert(0 <= ret);

			COMP_(irq_update)(cpssp, param.irq);
		}
		break;
	}

	return 0;
}

static int
COMP_(iolookup)(struct cpssp *cpssp, uint32_t addr)
{
	unsigned int region;

	if (! cpssp->io_enabled) {
		return -1;
	}
	for (region = 0; ; region++) {
		if (region == 7) {
			return -1;
		}
		if ((cpssp->real.region[region].base & 1)
		 && cpssp->virt_base[region] <= addr
		 && addr < cpssp->virt_base[region] + cpssp->real.region[region].size) {
			return region;
		}
	}
}

static int
COMP_(ior)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = _cpssp;
	struct faum_dev_param param;
	int ret;

	if (COMP_(iolookup)(cpssp, addr) < 0) {
		return -1;
	}

	param.addr = addr;
	param.bs = bs;
	ret = ioctl(cpssp->fd, FAUM_DEV_IOR, &param);
	assert(0 <= ret);
	*valp = param.data;

#if DEBUG
	fprintf(stderr, "%s: 0x%08x(0x%x) -> 0x%08x\n", __FUNCTION__,
			addr, bs, *valp);
#endif

	COMP_(irq_update)(cpssp, param.irq);

	return ret;
}

static int
COMP_(iow)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = _cpssp;
	struct faum_dev_param param;
	int ret;

	if (COMP_(iolookup)(cpssp, addr) < 0) {
		return -1;
	}

	param.addr = addr;
	param.bs = bs;
	param.data = val;
	ret = ioctl(cpssp->fd, FAUM_DEV_IOW, &param);
	assert(0 <= ret);

#if DEBUG
	fprintf(stderr, "%s: 0x%08x(0x%x) <- 0x%08x\n", __FUNCTION__,
			addr, bs, val);
#endif

	COMP_(irq_update)(cpssp, param.irq);

	return ret;
}

static int
COMP_(mlookup)(struct cpssp *cpssp, uint32_t addr)
{
	unsigned int region;

	if (! cpssp->mem_enabled) {
		return -1;
	}
	for (region = 0; ; region++) {
		if (region == 7) {
			return -1;
		}
		if (! (cpssp->real.region[region].base & 1)
		 && cpssp->virt_base[region] <= addr
		 && addr < cpssp->virt_base[region] + cpssp->real.region[region].size) {
			return region;
		}
	}
}

static int
COMP_(mr)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = _cpssp;
	struct faum_dev_param param;
	int region;
	int ret;

	region = COMP_(mlookup)(cpssp, addr);
	if (region < 0) {
		return -1;
	}

#if NVIDIA_HACK
	if (region == 0
	 && cpssp->virt_base[0] + 0x88000 <= addr
	 && addr < cpssp->virt_base[0] + 0x88100) {
		/* Accessing mapped PCI config space. */
		return COMP_(cread0)(cpssp, addr, bs, valp);
	}
#endif

	param.addr = addr;
	param.bs = bs;
	ret = ioctl(cpssp->fd, FAUM_DEV_MEMR, &param);
	assert(0 <= ret);
	*valp = param.data;

#if DEBUG
	fprintf(stderr, "%s: 0x%08x(0x%x) -> 0x%08x\n", __FUNCTION__,
			addr, bs, *valp);
#endif

	COMP_(irq_update)(cpssp, param.irq);

	return ret;
}

static int
COMP_(mw)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = _cpssp;
	struct faum_dev_param param;
	int region;
	int ret;

	region = COMP_(mlookup)(cpssp, addr);
	if (region < 0) {
		return -1;
	}

#if NVIDIA_HACK
	if (region == 0
	 && cpssp->virt_base[0] + 0x88000 <= addr
	 && addr < cpssp->virt_base[0] + 0x88100) {
		/* Accessing mapped PCI config space. */
		return COMP_(cwrite0)(cpssp, addr, bs, val);
	}
#endif

	param.addr = addr;
	param.bs = bs;
	param.data = val;
	ret = ioctl(cpssp->fd, FAUM_DEV_MEMW, &param);
	assert(0 <= ret);

#if DEBUG
	fprintf(stderr, "%s: 0x%08x(0x%x) <- 0x%08x\n", __FUNCTION__,
			addr, bs, val);
#endif

	COMP_(irq_update)(cpssp, param.irq);

	return ret;
}

void *
COMP_(create)(
	const char *name,
	const char *dev,
	struct sig_manage *manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_resetXhashX,
	struct sig_pci_bus_idsel *port_idsel,
	struct sig_pci_bus_main *port_pci_bus,
	struct sig_boolean_or *port_intA
)
{
	static const struct sig_pci_bus_idsel_funcs idsel_funcs = {
		.c0r = COMP_(cread0),
		.c0w = COMP_(cwrite0),
	};
	static const struct sig_pci_bus_main_funcs pci_bus_funcs = {
		.ior = COMP_(ior),
		.iow = COMP_(iow),
		.mr = COMP_(mr),
		.mw = COMP_(mw),
		.map_r = NULL,
		.map_w = NULL,
	};
	struct cpssp *cpssp;
	int ret;
	unsigned int i;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->fd = open(dev, O_RDWR);
	assert(0 <= cpssp->fd);

	ret = ioctl(cpssp->fd, FAUM_DEV_CREATE, &cpssp->real);
	assert(0 <= ret);

	for (i = 0; i < 7; i++) {
		fprintf(stderr, "base[%d]=0x%08x\n", i, cpssp->real.region[i].base);
		fprintf(stderr, "size[%d]=0x%08lx\n", i, cpssp->real.region[i].size);
	}

	for (i = 0; i < 7; i++) {
		cpssp->virt_base[i] = 0;
	}

	io_register(cpssp->fd, cpssp, COMP_(isr));

	/* Call */
	sig_pci_bus_idsel_connect(port_idsel, cpssp, &idsel_funcs);

	sig_pci_bus_main_connect(port_pci_bus, cpssp, &pci_bus_funcs);

	/* Out */
	cpssp->port_intA = port_intA;
	sig_boolean_or_connect_out(port_intA, cpssp, 0);

	/* In */
	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	ret = ioctl(cpssp->fd, FAUM_DEV_DESTROY, &cpssp->real);
	assert(0 <= ret);

	ret = close(cpssp->fd);
	assert(0 <= ret);

	free(cpssp);
}
