/* $Id: SimpleName.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __SIMPLE_NAME_HPP_INCLUDED
#define __SIMPLE_NAME_HPP_INCLUDED

#include "frontend/ast/Name.hpp"
#include "frontend/misc/Driver.hpp"

namespace ast {

//! a VHDL simple name
/** This class represents a VHDL simple name, either because it was parsed
 *  from only one identifier, or because it was an expanded name which is
 *  reduced to the meaning of a simple name.
 */
class SimpleName : public Name {
public:
	//! c'tor 
	/** @param n identifier of the name 
	 *  @param cands list of candidate symbols.
	 *  @param loc corresponding location
	 */
	SimpleName(
		std::string *n,
		std::list<Symbol*> cands,
		Location loc
		) : 	Name(n, cands, loc), driver(NULL) {}

	//! c'tor 
	/** @param n identifier of the name 
	 *  @param loc corresponding location
	 */
	SimpleName(
		std::string *n,
		Location loc
		) : 	Name(n, std::list<Symbol*>(), loc), driver(NULL) {}

	//! c'tor
	/** @param n identifier of the name
	 *  @param loc corresponding location
	 */
	SimpleName(
		const char *n,
		Location loc
		) :	Name(new std::string(n), std::list<Symbol*>(), loc), 
			driver(NULL) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor &visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		assert(this->name);
		std::list<std::string*> names = this->prefixStrings;
		names.push_back(this->name);
		util::MiscUtil::listPut(&names, stream, ".");
	}

	//! does the name refer to a signal?
	/** @return true, if the name refers to a Signal.
	 */
	bool isSignal(void) const;

	/** optional prefix for expanded names (only for error reporting
	 *  purposes)
	 */
	std::list<std::string*> prefixStrings;

	/** associated driver, if any */
	const Driver *driver;

protected:
	/** d'tor*/
	virtual ~SimpleName() {
		for (std::list<std::string*>::iterator i = 
			this->prefixStrings.begin(); 
			i != this->prefixStrings.end(); i++) {

			delete *i;
		}
	}
}; 

}; /* namespace ast */

#endif /* __SIMPLE_NAME_HPP_INCLUDED */
