#include <stdio.h>
#include <stdint.h>
#include "BPatch.h"
#include "BPatch_function.h"

#include "eztrace_dyninst_core.h"

// describe a function to instrument
struct eztrace_dyninst_func {
  char* fname;
  uint32_t code_entry;
  uint32_t code_exit;
  struct eztrace_dyninst_func *next;
};
typedef struct eztrace_dyninst_func *eztrace_dyninst_func_t;

// list of registered functions
eztrace_dyninst_func_t func_list = NULL;

static int __intercept_function(BPatch_addressSpace *app,
				 const char* function_name,
				 uint32_t code_entry,
				 uint32_t code_exit);

static int nb_func=0;

// register a function
extern "C" void eztrace_dyninst_register(const char* fname,
			      uint32_t code_entry,
			      uint32_t code_exit) {
  struct eztrace_dyninst_func *new_func = (struct eztrace_dyninst_func *) malloc(sizeof(struct eztrace_dyninst_func));

  asprintf(&new_func->fname, "%s", fname);
  new_func->code_entry = code_entry;
  new_func->code_exit = code_exit;

  if(func_list)
    new_func->next = func_list;
  else
    new_func->next = NULL;
  func_list = new_func;
}

// instrument all the registered functions
extern "C" int eztrace_dyninst_instrument(BPatch_addressSpace *app) {
  int ret = 0;
  eztrace_dyninst_func_t func = func_list;

  ret = app->loadLibrary(LIB_EZTRACE_SO);
  if(!ret) {
    fprintf(stderr, "warning: cannot load libeztrace\n");
    return -1;
  }

  ret = 0;
  while(func != NULL) {
    ret += __intercept_function(app,
			 func->fname,
			 func->code_entry,
			 func->code_exit);
    func = func->next;
  }
  return ret;
}

extern "C" int eztrace_dyninst_nb_function_to_register()
{
  eztrace_dyninst_func_t func = func_list;
  int ret = 0;

  while(func && func->next != func_list) {
    func = func->next;
    ret++;
  }
  return ret;
}

// Instrument a function: eztrace_code0(code_entry) is called at the
// beginning of the function and eztrace_code0(code_entry) is called
// at the end of the function.
// If code_entry or code_exit is null, the corresponding call to
// eztrace_code0 is skipped
static int __intercept_function(BPatch_addressSpace *app,
			const char* function_name,
			uint32_t code_entry,
			uint32_t code_exit)
{
  BPatch_image *appImage;
  BPatch_Vector<BPatch_point*> *points;
  BPatch_Vector<BPatch_function *> functions;

  appImage = app->getImage();
  void *ret = appImage->findFunction(function_name, functions, false);

  if(ret==NULL) {
    fprintf(stderr, "warning: cannot find function %s in executable\n", function_name);
    return 0;
  }

  // Instrument the entry of the function
  if(code_entry) {
    // We need to call eztrace_generic(code, nb_param, param1, param2, ...)
    points = functions[0]->findPoint(BPatch_entry);
    BPatch_Vector<BPatch_snippet*> dummyArgs;

    // Create the parameter (code_entry)
    BPatch_constExpr code(code_entry);
    dummyArgs.push_back(&code);

    // get the function parameters
    BPatch_function *f = functions[0];
    BPatch_Vector<BPatch_localVar*> *args = f->getParams();

    int i;
    BPatch_constExpr expr_nb_params(args->size());
    int nb_args = args->size();
    BPatch_paramExpr* local_args[6];

    // retrieve the parameters
    switch (nb_args) {
    case 4: local_args[3] = new BPatch_paramExpr(3);
    case 3: local_args[2] = new BPatch_paramExpr(2);
    case 2: local_args[1] = new BPatch_paramExpr(1);
    case 1: local_args[0] = new BPatch_paramExpr(0);
      break;
    default:
      // more than 5 args
      local_args[4] = new BPatch_paramExpr(4);
    }

    dummyArgs.push_back(&expr_nb_params);
    for(i=0;i<nb_args;i++)
      dummyArgs.push_back(local_args[i]);

    // Create the function call
    BPatch_Vector<BPatch_function *> funcs;
    appImage->findFunction("eztrace_generic", funcs);
    BPatch_function *dummyFunc = funcs[0];

    BPatch_funcCallExpr dummyCall(*dummyFunc, dummyArgs);

    //Insert the function call at the point
    app->insertSnippet(dummyCall, *points);
  }

  // Instrument the exit of the function
  if(code_exit) {
    // the function parameters are not available here, so we have to
    // call eztrace_code0(code)

    points = functions[0]->findPoint(BPatch_exit);
    // Create the parameter (code_entry)
    BPatch_Vector<BPatch_snippet*> dummyArgs;
    BPatch_constExpr code(code_exit);
    dummyArgs.push_back(&code);

    // Create the function call
    BPatch_Vector<BPatch_function *> funcs;
    appImage->findFunction("eztrace_code0", funcs);
    BPatch_function *dummyFunc = funcs[0];
    BPatch_funcCallExpr dummyCall(*dummyFunc, dummyArgs);

    //Insert the function call at the point
    app->insertSnippet(dummyCall, *points);
  }
  return 1;
}
