/* Copyright (c) 2008-2013 Tollef Fog Heen <tfheen@err.no>

 eweouz is free software; you can redistribute it and/or modify it
 under the terms of the GNU General Public License version 2 as
 published by the Free Software Foundation.

 eweouz is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 02110-1301 USA.
*/

#include <libebook/libebook.h>
#include <glib.h>
#include <locale.h>

void dump_as_vcard(EContact *contact, int dump_all)
{
	GList *attrs;
	int i;

	printf("BEGIN:VCARD\n");
	attrs = e_vcard_get_attributes(&contact->parent);
	for (i = 0; i < g_list_length(attrs); i++) {
		EVCardAttribute *attr = (EVCardAttribute *) 
			g_list_nth_data(attrs, i);
		int j;
		const char *attr_name = e_vcard_attribute_get_name(attr);

		if (!dump_all && 
		    strcmp(attr_name, "TEL") != 0 &&
		    strcmp(attr_name, "FN") != 0 &&
		    strcmp(attr_name, "BDAY") != 0 &&
		    strcmp(attr_name, "ORG") != 0 &&
		    strcmp(attr_name, "TEL") != 0 &&
		    strcmp(attr_name, "EMAIL") != 0 &&
		    strcmp(attr_name, "NOTE") != 0 &&
		    strcmp(attr_name, "NICKNAME") != 0
			) {
			continue;
		}

		if (e_vcard_attribute_is_single_valued(attr)) {
			char *av = e_vcard_attribute_get_value(attr);
			printf("%s: %s\n", 
			       attr_name,
			       av);
		} else {
			GList *vals = e_vcard_attribute_get_values(attr);

			for (j = 0; j < g_list_length(vals); j++) {
				const char *av = *(char**) g_list_nth(vals, j);

				printf("%s: %s\n", attr_name, av);
			}
		}
	}
	printf("END:VCARD\n");
}

static gchar **search_filter = NULL;

static GOptionEntry entries[] = 
{
	{ G_OPTION_REMAINING, 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_STRING_ARRAY, &search_filter, NULL, NULL},
	{ NULL }
};

int main(int argc, char **argv)
{
	EBookClient *client;
	ESourceRegistry *eds_source_registry = NULL;
	GList *sources;
	ESource *source;
	GSList *groups;
	EBookQuery *query;
	GSList *contacts;
	GList *g, *s;
	GSList *c;
	GError *error = NULL;
	GOptionContext *optioncontext;

	setlocale (LC_ALL, "");

	optioncontext = g_option_context_new ("- whack address book");
	g_option_context_add_main_entries (optioncontext, entries, NULL);
	g_option_context_parse (optioncontext, &argc, &argv, &error);

	if (error != NULL) {
		fprintf(stderr, "%s\n", error->message);
		return 1;
	}

	eds_source_registry = e_source_registry_new_sync (NULL, &error);
	if (error != NULL) {
		fprintf(stderr, "%s\n", error->message);
		return 1;
	}

	sources = e_source_registry_list_sources (eds_source_registry, E_SOURCE_EXTENSION_ADDRESS_BOOK);

	if (search_filter != NULL) {
		query = e_book_query_any_field_contains(*search_filter);
	} else {
		query = e_book_query_field_exists(E_CONTACT_FULL_NAME);
	}

	for (s = sources ; s; s = s->next) {
		source = E_SOURCE(s->data);
		client = E_BOOK_CLIENT(e_book_client_connect_sync(source, NULL, &error));

		if (error != NULL) {
			fprintf(stderr, "e_book_client_connect_sync: %s\n", error->message);
			g_error_free(error);
			error = NULL;
			continue;
		}

		/* XXX: leaks the query, but we're short running so doesn't really matter */
		e_book_client_get_contacts_sync(client, e_book_query_to_string(query), &contacts, NULL, &error);
		if (error != NULL) {
			fprintf(stderr, "%s\n", error->message);
			return 1;
		}

		for (c = contacts; c; c = c->next) {
			dump_as_vcard(E_CONTACT(c->data), 1);
		}
	}
	e_book_query_unref(query);
	g_list_free_full(sources, g_object_unref);

}
