/* ethos-manager.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdlib.h>
#include <gmodule.h>

#include "ethos-error.h"
#include "ethos-manager.h"
#include "ethos-plugin-loader.h"
#include "ethos-plugin-info-private.h"

/**
 * SECTION:ethos-manager
 * @title: EthosManager
 * @short_description: plugin management during runtime
 *
 * The #EthosManager is responsible for managing plugins during runtime.  It can be used to
 * load and unload plugins as well as configure where and how plugins should be loaded.
 *
 * During runtime, the app_name and plugin_dirs should both be set.  The app_name is used
 * to derive the plugin filenames and how to parse them.  The plugin_dirs are used to locate
 * plugins.
 */

struct _EthosManagerPrivate
{
	gboolean     initialized;
	gchar       *app_name;
	gchar      **plugin_dirs;
	GList       *plugin_loaders;
	GList       *plugin_info;
	GHashTable  *plugin_info_hash;
	GHashTable  *plugins;
	GHashTable  *deps_cache;
};

enum
{
	INITIALIZED,
	PLUGIN_LOADED,
	PLUGIN_UNLOADED,
	LAST_SIGNAL
};

G_DEFINE_TYPE (EthosManager, ethos_manager, G_TYPE_OBJECT);

static guint signals[LAST_SIGNAL] = {0,};

/**
 * ethos_manager_new:
 *
 * Creates a new instance of #EthosManager.  There should only be one of these
 * created per process.
 *
 * Return value: the newly created #EthosManager instance.
 */
EthosManager*
ethos_manager_new (void)
{
	return g_object_new (ETHOS_TYPE_MANAGER, NULL);
}

/**
 * ethos_manager_new_full:
 * @app_name: The name of the application (Capitalized)
 * @plugin_dirs: An array of strings containing directories to locate plugins
 *
 * Creates a new #EthosManager instance and sets the application name and
 * plugin directories to traverse to locate plugins.
 *
 * Return value: The newly created #EthosManager instance.
 */
EthosManager*
ethos_manager_new_full (const gchar  *app_name,
                        gchar       **plugin_dirs)
{
	EthosManager *manager;

	manager = ethos_manager_new ();
	ethos_manager_set_app_name (manager, app_name);
	ethos_manager_set_plugin_dirs (manager, plugin_dirs);

	return manager;
}

static EthosPluginLoader*
ethos_manager_create_plugin_loader (EthosManager *manager,
                                    const gchar  *filename)
{
	EthosPluginLoader* (*create_plugin_loader) (void) = NULL;
	EthosPluginLoader  *plugin_loader;
	GModule            *module;

	module = g_module_open (filename, G_MODULE_BIND_LAZY);

	if (!module) {
		g_warning ("%s: %s", filename, g_module_error ());
		return NULL;
	}

	if (!g_module_symbol (module, "ethos_plugin_loader_register",
	                      (gpointer*)&create_plugin_loader)) {
	        g_warning ("%s: %s", filename, g_module_error ());
	        if (!g_module_close (module))
	        	g_warning ("%s: %s", filename, g_module_error ());
	        return NULL;
	}

	if (create_plugin_loader == NULL) {
		g_warning ("%s: ethos_plugin_loader_register is NULL", filename);
	        if (!g_module_close (module))
	        	g_warning ("%s: %s", filename, g_module_error ());
	        return NULL;
	}

	plugin_loader = create_plugin_loader ();

	if (plugin_loader && !ETHOS_IS_PLUGIN_LOADER (plugin_loader)) {
		g_warning ("%s: ethos_plugin_loader_register returned "
		           "something other than an EthosPluginLoader",
		           filename);
		if (!g_module_close (module))
	        	g_warning ("%s: %s", filename, g_module_error ());
		return NULL;
	}

	ethos_plugin_loader_initialize (plugin_loader, manager);

	return plugin_loader;
}

/*
 * ethos_manager_discover_plugin_loaders:
 * @manager: An #EthosManager
 *
 * Discovers the available plugin loaders.
 *
 * Return value: A #GList containing #EthosPluginLoader instances.
 */
static GList*
ethos_manager_discover_plugin_loaders (EthosManager *manager)
{
	EthosManagerPrivate *priv;
	GDir                *dir;
	const gchar         *loaders_dir;
	const gchar         *filename;
	gchar               *abspath;
	GList               *loaders = NULL;
	EthosPluginLoader   *loader;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);

	priv = manager->priv;

	if (g_getenv ("ETHOS_PLUGIN_LOADERS_DIR") != NULL)
		loaders_dir = g_getenv ("ETHOS_PLUGIN_LOADERS_DIR");
	else
		loaders_dir = ETHOS_PLUGIN_LOADERS_DIR;

	if (!g_file_test (loaders_dir, G_FILE_TEST_IS_DIR)) {
		g_warning ("plugin-loaders directory not found: %s",
		           loaders_dir);
		return NULL;
	}

	if (!(dir = g_dir_open (loaders_dir, 0, NULL))) {
		g_warning ("plugin-loaders directory not accessable: %s",
		           loaders_dir);
		return NULL;
	}

	while (NULL != (filename = g_dir_read_name (dir))) {
		if (g_str_has_suffix (filename, "." G_MODULE_SUFFIX)) {
			abspath = g_build_filename (loaders_dir, filename, NULL);
			loader = ethos_manager_create_plugin_loader (manager, abspath);
			if (loader != NULL)
				loaders = g_list_prepend (loaders, loader);
			g_free (abspath);
		}
	}

	g_dir_close (dir);

	return loaders;
}

/*
 * ethos_manager_discover_plugins:
 * @manager: An #EthosManager
 *
 * Discovers all of the plugins found within the plugin directories.
 *
 * Return value: A #GList containing #EthosPluginInfo instances that were
 *   created from plugin files in the plugin directories.
 */
static GList*
ethos_manager_discover_plugins (EthosManager *manager)
{
	EthosManagerPrivate  *priv;
	EthosPluginInfo      *plugin_info;
	GList                *plugins = NULL;
	GDir                 *dir;
	gchar               **plugin_dirs;
	const gchar          *filename;
	gchar                *lower;
	gchar                *abspath;
	gchar                *suffix;
	gchar                *group;
	gint                  n_dirs;
	gint                  i;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);

	priv = manager->priv;

	plugin_dirs = g_strdupv (priv->plugin_dirs);
	n_dirs = plugin_dirs ? g_strv_length (plugin_dirs) : 0;
	lower = g_utf8_strdown (priv->app_name, -1);
	suffix = g_strconcat (".", lower, "-plugin", NULL);
	group = g_strconcat (priv->app_name, " Plugin", NULL);

	for (i = 0; i < n_dirs; i++) {
		if (!(dir = g_dir_open (plugin_dirs [i], 0, NULL))) {
			g_warning ("Could not access plugin directory: %s",
			           plugin_dirs [i]);
			continue;
		}

		while (NULL != (filename = g_dir_read_name (dir))) {
			if (!g_str_has_suffix (filename, suffix))
				continue;

			abspath = g_build_filename (plugin_dirs [i], filename, NULL);

			plugin_info = ethos_plugin_info_new ();
			ethos_plugin_info_set_filename (plugin_info, abspath);

			/* overwrite the last '.' in the filename and re-use
			 * the same string as the id. */
			if (g_strrstr (filename, "."))
				*g_strrstr (filename, ".") = '\0';

			ethos_plugin_info_set_id (plugin_info, filename);

			if (ethos_plugin_info_load_from_file (plugin_info, group, abspath, NULL))
				plugins = g_list_prepend (plugins, g_object_ref (plugin_info));

			g_object_unref (plugin_info);
			g_free (abspath);
		}

		g_dir_close (dir);
	}

	g_strfreev (plugin_dirs);
	g_free (suffix);
	g_free (lower);
	g_free (group);

	return plugins;
}

/**
 * ethos_manager_initialize:
 * @manager: An #EthosManager
 *
 * Initialize the plugin manager.  This will result in the manager looking for
 * all of the available plugins that it can find in the registered plugin
 * directories.
 */
void
ethos_manager_initialize (EthosManager *manager)
{
	EthosManagerPrivate *priv;
	GList               *iter;

	g_return_if_fail (ETHOS_IS_MANAGER (manager));

	priv = manager->priv;

	if (priv->initialized)
		return;

	priv->initialized = TRUE;
	priv->plugin_loaders = ethos_manager_discover_plugin_loaders (manager);
	priv->plugin_info = ethos_manager_discover_plugins (manager);

	for (iter = priv->plugin_info; iter; iter = iter->next) {
		g_hash_table_insert (priv->plugin_info_hash,
		                     g_strdup (ethos_plugin_info_get_id (iter->data)),
		                     g_object_ref (iter->data));
	}

	g_signal_emit (manager, signals [INITIALIZED], 0);
}

/**
 * ethos_manager_unload:
 * @manager: An #EthosManager
 *
 * Unloads all of the plugins.
 */
void
ethos_manager_unload (EthosManager *manager)
{
	g_warning ("%s is not yet implemented", __func__);
}

/**
 * ethos_manager_lookup_plugin_loader:
 * @manager: An #EthosManager
 * @name: The name of the #EthosPluginLoader
 *
 * Attempts to find the #EthosPluginLoader matching @name.
 *
 * Return value: An #EthosPluginLoader or %NULL.
 */
EthosPluginLoader*
ethos_manager_lookup_plugin_loader (EthosManager *manager,
                                    const gchar  *name)
{
	EthosManagerPrivate *priv;
	GList               *iter;
	const gchar         *loader_name;
	gchar               *loader_down = NULL,
	                    *name_down   = NULL;
	EthosPluginLoader   *loader      = NULL;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);

	priv = manager->priv;

	for (iter = priv->plugin_loaders; iter; iter = iter->next) {
		/* To make things a bit more resilient, we will check to see if the developer
		 * accidentally used Loader=C or Loader=Vala and replace that with NULL to
		 * indicate we should use the default shared library loader.
		 */

		loader_name = ethos_plugin_loader_get_name (iter->data);

		if (loader_name)
			loader_down = g_ascii_strdown (loader_name, -1);
		if (name)
			name_down = g_ascii_strdown (name, -1);

		if (g_strcmp0 (name_down, "c") == 0 || g_strcmp0 (name_down, "vala") == 0) {
			g_free (name_down);
			name_down = NULL;
		}

		if (g_strcmp0 (name_down, loader_down) == 0)
			loader = iter->data;

		g_free (loader_down);
		g_free (name_down);
		loader_down = NULL;
		name_down = NULL;

		if (loader)
			return loader;
	}

	return NULL;
}

typedef enum
{
	OP_NONE = 0,
	OP_LT   = 1 << 0,
	OP_GT   = 1 << 1,
	OP_EQ   = 1 << 2,
	OP_LTE  = OP_LT | OP_EQ,
	OP_GTE  = OP_GT | OP_EQ,
} Operator;

typedef struct
{
	gchar    *name;
	gchar    *version;
	Operator  oper;
} Dependency;

typedef struct
{
	gint major;
	gint minor;
	gint rev;
} Version;

static GList*
parse_deps (EthosPluginInfo *plugin_info)
{
	const gchar *depstr;
	const gchar *name;
	GScanner    *scanner;
	GList       *deps = NULL;
	Dependency  *dep;
	gboolean     has_oper;

	depstr = ethos_plugin_info_get_dependencies (plugin_info);
	if (!depstr || g_strcmp0 (depstr, "") == 0)
		return NULL;

	name = ethos_plugin_info_get_id (plugin_info);
	scanner = g_scanner_new (NULL);
	g_scanner_input_text (scanner, depstr, strlen (depstr));

	scanner->config->scan_float = 0;

	if (g_scanner_get_next_token (scanner) != G_TOKEN_IDENTIFIER) {
		g_warning ("%s: Invalid Dependencies", name);
		goto cleanup;
	}

	do {
		has_oper = TRUE;
		dep = g_slice_new0 (Dependency);
		dep->name = g_strdup (g_scanner_cur_value (scanner).v_string);

		/* check for first of two possible operator tokens */
		switch (g_scanner_peek_next_token (scanner)) {
		case '=':
			dep->oper = OP_EQ;
			break;
		case '<':
			dep->oper = OP_LT;
			break;
		case '>':
			dep->oper = OP_GT;
			break;
		default:
			has_oper = FALSE;
			break;
		}

		if (has_oper) {
			g_scanner_get_next_token (scanner);

			/* check for the second possible operator token */
			switch (g_scanner_peek_next_token (scanner)) {
			case '=':
				dep->oper |= OP_EQ;
				break;
			case '<':
				dep->oper |= OP_LT;
				break;
			case '>':
				dep->oper |= OP_GT;
				break;
			default:
				has_oper = FALSE;
				break;
			}

			if (has_oper)
				g_scanner_get_next_token (scanner);

			GTokenType t;
			int a = 0, b = 0, c = 0;
			gboolean has_a = FALSE, has_b = FALSE, has_c = FALSE;

			t = g_scanner_peek_next_token (scanner);

			if (t != G_TOKEN_INT) {
				g_warning ("Invalid dependency version string for %s", dep->name);
			}
			else
			{
				do {
					t = g_scanner_get_next_token (scanner);
					if (t == G_TOKEN_INT) {
						if (!has_a) {
							a = g_scanner_cur_value (scanner).v_int;
							has_a = TRUE;
						}
						else if (!has_b) {
							b = g_scanner_cur_value (scanner).v_int;
							has_b = TRUE;
						}
						else if (!has_c) {
							c = g_scanner_cur_value (scanner).v_int;
							has_c = TRUE;
						}
						else {
							g_warning ("Invalid version string for %s",
								   ethos_plugin_info_get_id (plugin_info));
						}
					}
					t = g_scanner_peek_next_token (scanner);
				} while (t == G_TOKEN_INT || t == '.');
			}

			dep->version = g_strdup_printf ("%d.%d.%d", a, b, c);
		}

		deps = g_list_append (deps, dep);
	}
	while (g_scanner_get_next_token (scanner) == G_TOKEN_IDENTIFIER);

cleanup:
	g_scanner_destroy (scanner);
	return deps;

}

static void
free_deps (GList *deps)
{
	GList *iter;
	if (deps == NULL)
		return;
	for (iter = deps; iter; iter = iter->next) {
		g_free (((Dependency*)iter->data)->name);
		g_slice_free (Dependency, iter->data);
	}
	g_list_free (deps);
}

/**
 * ethos_manager_lookup_plugin_info:
 * @manager: An #EthosManager
 * @name: the plugin name
 *
 * Retrieves the #EthosPluginInfo found for the plugin named @name.
 *
 * Return value: The #EthosPluginInfo or %NULL.
 */
EthosPluginInfo*
ethos_manager_lookup_plugin_info (EthosManager *manager,
                                  const gchar  *name)
{
	EthosManagerPrivate *priv;
	GList               *iter;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);

	priv = manager->priv;

	for (iter = priv->plugin_info; iter; iter = iter->next) {
		if (g_strcmp0 (name, ethos_plugin_info_get_id (iter->data)) == 0)
			return iter->data;
	}

	return NULL;
}

static gboolean
parse_version (const gchar *str,
               Version     *version)
{
	gchar **parts;

	g_return_val_if_fail (str != NULL, FALSE);
	g_return_val_if_fail (version != NULL, FALSE);

	parts = g_strsplit (str, ".", -1);

	version->major = 0;
	version->minor = 0;
	version->rev = 0;

	if (parts [0]) {
		version->major = atoi (parts [0]);
		if (parts [1]) {
			version->minor = atoi (parts [1]);
			if (parts [2])
				version->rev = atoi (parts [2]);
		}
	}

	g_strfreev (parts);

	return TRUE;
}

static gint
version_cmp (Version *a,
             Version *b)
{
	if (a == b)
		return 0;
	else if (a->major > b->major
	         || (a->major == b->major && a->minor > b->minor)
	         || (a->major == b->major && a->minor == b->minor && a->rev > b->rev))
		return 1;
	else if (a->major < b->major
	         || (a->major == b->major && a->minor < b->minor)
	         || (a->major == b->major && a->minor == b->minor && a->rev < b->rev))
		return -1;
	else if (a->major == b->major && a->minor == b->minor && a->rev == b->rev)
		return 0;
	else
		g_assert_not_reached ();
}

/**
 * ethos_manager_load_plugin:
 * @manager: An #EthosManager
 * @plugin_info: An #EthosPluginInfo
 * @error: A location for a #GError
 *
 * Attempts to load a plugin using the #EthosPluginInfo.
 *
 * Return value: %TRUE if the plugin was loaded. @error is set in case of
 *   failure.
 */
gboolean
ethos_manager_load_plugin (EthosManager     *manager,
                           EthosPluginInfo  *plugin_info,
                           GError          **error)
{
	EthosManagerPrivate *priv;
	EthosPluginLoader   *plugin_loader;
	EthosPlugin         *plugin;
	EthosPluginInfo     *dep_info;
	const gchar         *loader;
	GList               *deps,
			    *iter,
			    *dep_names = NULL;
	Dependency          *dep;
	Version              ver, depver;
	GError              *local_error = NULL;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), FALSE);
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);

	if (ethos_plugin_info_get_active (plugin_info))
		return TRUE;

	priv = manager->priv;

	loader = ethos_plugin_info_get_loader (plugin_info);
	plugin_loader = ethos_manager_lookup_plugin_loader (manager, loader);

	if (!plugin_loader) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "The plugin loader \"%s\" could not be found",
		             loader);
		return FALSE;
	}

	deps = parse_deps (plugin_info);

	for (iter = deps; iter; iter = iter->next) {
		dep = iter->data;
		if (!parse_version (dep->version, &depver)) {
			local_error = g_error_new (ETHOS_ERROR,
			                           ETHOS_ERROR_PLUGIN,
			                           "Invalid request for plugin version %s (%s",
			                           dep->name, dep->version);
			ethos_plugin_info_add_error (plugin_info, local_error);
			if (error && *error == NULL)
				*error = local_error;
			else
				g_error_free (local_error);
			free_deps (deps);
			return FALSE;
		}
		dep_info = ethos_manager_lookup_plugin_info (manager, dep->name);
		if (!dep_info) {
			g_warning ("%s: Could not locate plugin dependency %s",
			           ethos_plugin_info_get_id (plugin_info),
			           dep->name);
			local_error = g_error_new (ETHOS_ERROR,
			                           ETHOS_ERROR_PLUGIN,
			                           "Missing dependency %s",
			                           dep->name);
			ethos_plugin_info_add_error (plugin_info, local_error);
			if (error && *error == NULL)
				*error = local_error;
			else
				g_error_free (local_error);
			free_deps (deps);
			return FALSE;
		}
		if (!parse_version (ethos_plugin_info_get_version (dep_info), &ver)) {
			local_error = g_error_new (ETHOS_ERROR,
			                           ETHOS_ERROR_PLUGIN,
			                           "Invalid dependency plugin version %s (%s)",
			                           ethos_plugin_info_get_id (dep_info),
			                           ethos_plugin_info_get_version (dep_info));
			ethos_plugin_info_add_error (plugin_info, local_error);
			if (error && *error == NULL)
				*error = local_error;
			else
				g_error_free (local_error);
			free_deps (deps);
			return FALSE;
		}

		/* make sure that the dependency is met */
		switch (version_cmp (&ver, &depver)) {
		case -1:
			if ((dep->oper & OP_LT) == 0)
				goto unsatisfied;
			break;
		case 0:
			if ((dep->oper & OP_EQ) == 0)
				goto unsatisfied;
			break;
		case 1:
			if ((dep->oper & OP_GT) == 0)
				goto unsatisfied;
			break;
		default:
			g_assert_not_reached ();
		}

		if (!ethos_manager_load_plugin (manager, dep_info, &local_error))
			goto unsatisfied;

		continue;

unsatisfied:
		if (local_error) {
			ethos_plugin_info_add_error (dep_info, local_error);
			ethos_plugin_info_add_error (plugin_info, local_error);
			if (error && *error == NULL)
				*error = local_error;
			else
				g_error_free (local_error);
		}

		free_deps (deps);
		return FALSE;
	}

	/* cache the dep names for this plugin */
	for (iter = deps; iter; iter = iter->next) {
		dep = iter->data;
		dep_names = g_list_prepend (dep_names, g_strdup (dep->name));
	}
	g_hash_table_insert (priv->deps_cache,
	                     g_strdup (ethos_plugin_info_get_id (plugin_info)),
	                     dep_names);

	free_deps (deps);

	if (!(plugin = ethos_plugin_loader_load (plugin_loader, plugin_info, error)))
	        return FALSE;

	if (ETHOS_MANAGER_GET_CLASS (manager)->load_plugin (manager, plugin, error)) {
		g_hash_table_insert (priv->plugins,
		                     g_strdup (ethos_plugin_info_get_id (plugin_info)),
		                     g_object_ref (plugin));
		ethos_plugin_info_set_active (plugin_info, TRUE);
		g_signal_emit (manager, signals [PLUGIN_LOADED], 0, plugin_info);
		g_object_unref (plugin);
		return TRUE;
	}

	g_object_unref (plugin);

	return FALSE;
}

/**
 * ethos_manager_unload_plugin:
 * @manager: An #EthosManager
 * @plugin_info: An #EthosPluginInfo
 * @error: A location for a #GError or %NULL
 *
 * Attempts to unload the plugin that was created using @plugin_info.
 *
 * Return value: %TRUE if the plugin was unloaded. @error is set in case of
 *   failure.
 */
gboolean
ethos_manager_unload_plugin (EthosManager     *manager,
                             EthosPluginInfo  *plugin_info,
                             GError          **error)
{
	EthosManagerPrivate *priv;
	EthosPlugin         *plugin;
	const gchar         *plugin_id;
	GList               *list;
	GHashTableIter       iter;
	gpointer             key, value;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), FALSE);
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), FALSE);

	if (!ethos_plugin_info_get_active (plugin_info))
		return TRUE;

	priv = manager->priv;
	plugin_id = ethos_plugin_info_get_id (plugin_info);

	if (!(plugin = g_hash_table_lookup (priv->plugins, plugin_id)))
		return TRUE;

	/* if any other plugin is depending on this plugin, we cannot
	 * unload it right now. deps_cache is a hashtable containing
	 * a linked list of the names of dependencies */
	g_hash_table_iter_init (&iter, priv->deps_cache);
	while (g_hash_table_iter_next (&iter, &key, &value)) {
		for (list = value; list; list = list->next) {
			if (g_strcmp0 (plugin_id, list->data) == 0) {
				g_set_error (error,
				             ETHOS_ERROR,
				             ETHOS_ERROR_PLUGIN,
				             "Cannot unload the plugin"
				             " because the %s plugin is "
				             "depending on it.",
				             (gchar*)key);
				return FALSE;
			}
		}
	}

	if (ETHOS_MANAGER_GET_CLASS (manager)->unload_plugin (manager, plugin, error)) {
		ethos_plugin_info_set_active (plugin_info, FALSE);
		g_hash_table_remove (priv->deps_cache, plugin_id);
		g_hash_table_remove (priv->plugins, plugin_id);
		g_signal_emit (manager, signals [PLUGIN_UNLOADED], 0, plugin_info);
		return TRUE;
	}

	return FALSE;
}

/**
 * ethos_manager_get_plugin_loaders:
 * @manager: An #EthosManager
 *
 * Retreives a list of #EthosPluginLoader<!-- -->s that were discovered
 * when the application was initialized.
 *
 * Return value: A #GList of #EthosPluginLoader<!-- -->s.  The list should
 *   be freed using g_list_free().
 */
GList*
ethos_manager_get_plugin_loaders (EthosManager *manager)
{
	EthosManagerPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);

	priv = manager->priv;

	return g_list_copy (priv->plugin_loaders);
}

/**
 * ethos_manager_get_plugin_info:
 * @manager: An #EthosManager
 *
 * Retreives the list of #EthosPluginInfo that were discovered.
 *
 * Return value: A #GList of #EthosPluginInfo which should be freed using
 *   g_list_free().
 */
GList*
ethos_manager_get_plugin_info (EthosManager *manager)
{
	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);
	return g_list_copy (manager->priv->plugin_info);
}

/**
 * ethos_manager_get_plugin_dirs:
 * @manager: An #EthosManager
 *
 * Retrieves the list of plugin directories that are traversed when looking to load plugins.
 *
 * Return value: a %NULL terminated list of strings containing directories that will be
 *   traversed to locate plugins.  This reslut should never be modified or freed.
 */
G_CONST_RETURN gchar**
ethos_manager_get_plugin_dirs (EthosManager *manager)
{
	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);
	return (G_CONST_RETURN gchar**)manager->priv->plugin_dirs;
}

/**
 * ethos_manager_set_plugin_dirs:
 * @manager: An #EthosManager
 * @plugin_dirs: A %NULL terminated list of strings of directories
 *
 * Sets the list of directories to use for locating plugins.
 */
void
ethos_manager_set_plugin_dirs (EthosManager  *manager,
                               gchar        **plugin_dirs)
{
	EthosManagerPrivate *priv;

	g_return_if_fail (ETHOS_IS_MANAGER (manager));

	priv = manager->priv;

	g_strfreev (priv->plugin_dirs);
	priv->plugin_dirs = g_strdupv (plugin_dirs);
}

/**
 * ethos_manager_get_app_name:
 * @manager: An #EthosManager
 *
 * Retrieves the app-name that the #EthosManager should use.
 *
 * Return value: the configured app name used for plugins or %NULL
 */
G_CONST_RETURN gchar*
ethos_manager_get_app_name (EthosManager *manager)
{
	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);
	return manager->priv->app_name;
}

/**
 * ethos_manager_set_app_name:
 * @manager: An #EthosManager
 * @app_name: A single-worded name for the application
 *
 * Sets the app name used for the application.  This should be a single word
 * to simplify the plugin development process.  The app name is used inside
 * the plugin description files for the group prefix as well as in the name
 * of the file.
 *
 * For example, an application named "Ethos" would have the group
 * [Ethos Plugin] in the plugin description file which would be named
 * myplugin.ethos-plugin.
 */
void
ethos_manager_set_app_name (EthosManager *manager,
                            const gchar  *app_name)
{
	EthosManagerPrivate *priv;

	g_return_if_fail (ETHOS_IS_MANAGER (manager));
	g_return_if_fail (app_name != NULL);

	priv = manager->priv;

	g_free (priv->app_name);
	priv->app_name = g_strdup (app_name);
}

/**
 * ethos_manager_get_plugin:
 * @manager: An #EthosManager
 * @plugin_info: An #EthosPluginInfo
 *
 * Retrieves the created instance of the plugin defined by @plugin_info.  If
 * no instance of the plugin has been created, then %NULL is returned.
 *
 * Return value: The #EthosPlugin instance or %NULL
 */
EthosPlugin*
ethos_manager_get_plugin (EthosManager    *manager,
                          EthosPluginInfo *plugin_info)
{
	EthosManagerPrivate *priv;

	g_return_val_if_fail (ETHOS_IS_MANAGER (manager), NULL);
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = manager->priv;

	return g_hash_table_lookup (priv->plugins,
	                            ethos_plugin_info_get_id (plugin_info));
}

static gboolean
ethos_manager_real_load_plugin (EthosManager  *manager,
                                EthosPlugin   *plugin,
                                GError       **error)
{
	ethos_plugin_activate (plugin);
	return TRUE;
}

static gboolean
ethos_manager_real_unload_plugin (EthosManager  *manager,
                                  EthosPlugin   *plugin,
                                  GError       **error)
{
	ethos_plugin_deactivate (plugin);
	return TRUE;
}

static void
ethos_manager_real_initialized (EthosManager *manager)
{
	GList  *list,
	       *iter;
	GError *error = NULL;

	g_return_if_fail (ETHOS_IS_MANAGER (manager));

	list = ethos_manager_get_plugin_info (manager);
	for (iter = list; iter; iter = iter->next) {
		if (ethos_plugin_info_get_active (iter->data)) {
			continue;
		}
		else if (!ethos_manager_load_plugin (manager, iter->data, &error)) {
			g_warning ("%s: %s",
			           ethos_plugin_info_get_id (iter->data),
			           error ? error->message : "Error loading");

			if (error) {
				ethos_plugin_info_add_error (iter->data, error);
				g_error_free (error);
				error = NULL;
			}
		}
	}

	g_list_free (list);
}

static void
ethos_manager_finalize (GObject *object)
{
	EthosManagerPrivate *priv;

	priv = ETHOS_MANAGER (object)->priv;

	g_free (priv->app_name);
	g_strfreev (priv->plugin_dirs);
	g_hash_table_destroy (priv->plugin_info_hash);

	g_list_foreach (priv->plugin_loaders, (GFunc)g_object_unref, NULL);
	g_list_free (priv->plugin_loaders);

	g_list_foreach (priv->plugin_info, (GFunc)g_object_unref, NULL);
	g_list_free (priv->plugin_info);

	G_OBJECT_CLASS (ethos_manager_parent_class)->finalize (object);
}

static void
ethos_manager_real_plugin_loaded (EthosManager    *manager,
                                  EthosPluginInfo *plugin_info)
{
}

static void
ethos_manager_real_plugin_unloaded (EthosManager    *manager,
                                    EthosPluginInfo *plugin_info)
{
}

static void
ethos_manager_class_init (EthosManagerClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = ethos_manager_finalize;
	g_type_class_add_private (object_class, sizeof(EthosManagerPrivate));

	klass->load_plugin = ethos_manager_real_load_plugin;
	klass->unload_plugin = ethos_manager_real_unload_plugin;
	klass->initialized = ethos_manager_real_initialized;
	klass->plugin_loaded = ethos_manager_real_plugin_loaded;
	klass->plugin_unloaded = ethos_manager_real_plugin_unloaded;

	/**
	 * EthosManager::initialized:
	 * @manager: An #EthosManager
	 *
	 * The initialized signal is emmitted after the manager has completed
	 * initializing.  This is a good place to load the plugins you want
	 * loaded on startup.
	 */
	signals [INITIALIZED] =
		g_signal_new (g_intern_static_string ("initialized"),
		              G_TYPE_FROM_CLASS (klass),
		              G_SIGNAL_RUN_FIRST,
		              G_STRUCT_OFFSET (EthosManagerClass, initialized),
		              NULL, NULL,
		              g_cclosure_marshal_VOID__VOID,
		              G_TYPE_NONE,
		              0);

	/**
	 * EthosManager::plugin-loaded:
	 * @manager: An #EthosManager
	 * @plugin_info: An #EthosPluginInfo
	 *
	 * The plugin-loaded signal is emitted when a plugin has been
	 * successfully loaded.
	 */
	signals [PLUGIN_LOADED] =
		g_signal_new (g_intern_static_string ("plugin-loaded"),
		              G_TYPE_FROM_CLASS (klass),
		              G_SIGNAL_RUN_FIRST,
		              G_STRUCT_OFFSET (EthosManagerClass, plugin_loaded),
		              NULL, NULL,
		              g_cclosure_marshal_VOID__OBJECT,
		              G_TYPE_NONE,
		              1, ETHOS_TYPE_PLUGIN_INFO);

	/**
	 * EthosManager::plugin-unloaded:
	 * @manager: An #EthosManager
	 * @plugin_info: An #EthosPluginInfo
	 *
	 * The plugin-loaded signal is emitted when a plugin has been
	 * successfully unloaded.
	 */
	signals [PLUGIN_UNLOADED] =
		g_signal_new (g_intern_static_string ("plugin-unloaded"),
		              G_TYPE_FROM_CLASS (klass),
		              G_SIGNAL_RUN_FIRST,
		              G_STRUCT_OFFSET (EthosManagerClass, plugin_unloaded),
		              NULL, NULL,
		              g_cclosure_marshal_VOID__OBJECT,
		              G_TYPE_NONE,
		              1, ETHOS_TYPE_PLUGIN_INFO);
}

static void
free_dep_list (gpointer data)
{
	GList *list = data,
	      *iter;
	for (iter = list; iter; iter = iter->next)
		g_free (iter->data);
	g_list_free (list);
}

static void
ethos_manager_init (EthosManager *manager)
{
	manager->priv = G_TYPE_INSTANCE_GET_PRIVATE (manager,
	                                             ETHOS_TYPE_MANAGER,
	                                             EthosManagerPrivate);

	/* set the default app name, which is gotten from the executable
	 * but with an uppercase first letter.
	 */
	manager->priv->app_name = g_strdup (g_get_prgname ());
	if (manager->priv->app_name) {
		if (!g_ascii_isupper (manager->priv->app_name [0])) {
			manager->priv->app_name [0] =
				g_ascii_toupper (manager->priv->app_name [0]);
		}
	}
	else
		manager->priv->app_name = g_strdup ("Ethos");

	manager->priv->plugin_info_hash =
		g_hash_table_new_full (g_str_hash,
		                       g_str_equal,
		                       g_free,
		                       g_object_unref);
	manager->priv->plugins =
		g_hash_table_new_full (g_str_hash,
		                       g_str_equal,
		                       g_free,
		                       g_object_unref);
	manager->priv->deps_cache =
		g_hash_table_new_full (g_str_hash,
		                       g_str_equal,
		                       g_free,
		                       free_dep_list);
}
