# -*- coding: utf-8 -*-

# Copyright (c) 2002, 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the viewmanager base class.
"""

import os

from qt import *
from qtext import QextScintillaAPIs, QextScintilla

import Preferences
from QScintilla.Editor import Editor
from QScintilla.GotoDialog import GotoDialog
from QScintilla.SearchReplaceDialog import SearchReplaceDialog
from QScintilla.ZoomDialog import ZoomDialog
import Utilities
import UI.PixmapCache

class ViewManager:
    """
    Base class inherited by all specific viewmanager classes.
    
    It defines the interface to be implemented by specific
    viewmanager classes and all common methods.
    
    @signal lastEditorClosed emitted after the last editor window was closed
    @signal editorOpened(string) emitted after an editor window was opened
    @signal editorSaved(string) emitted after an editor window was saved
    @signal checkActions(editor) emitted when some actions should be checked
            for their status
    @signal cursorChanged(editor) emitted after the cursor position of the active
            window has changed
    @signal breakpointToggled(editor) emitted when a breakpoint is toggled.
    @signal bookmarkToggled(editor) emitted when a bookmark is toggled.
    """
    def __init__(self, ui, dbs):
        """
        Constructor
        
        @param ui reference to the main user interface
        @param dbs reference to the debug server object
        """
        # initialize the instance variables
        self.ui = ui
        self.editors = []
        self.currentEditor = None
        self.dbs = dbs
        self.untitledCount = 0
        self.srHistory = {"search" : QStringList(), "replace" : QStringList()}
        self.editorsCheckFocusIn = 1
        
        self.recent = QStringList()
        rp, ok = Preferences.Prefs.settings.readListEntry('/eric3/Recent/Sources')
        if ok:
            self.recent = rp
            
        self.searchDlg = SearchReplaceDialog(0, self, ui)
        self.replaceDlg = SearchReplaceDialog(1, self, ui)
        
        # initialize the central store for api information (used by
        # autocompletion and calltips)
        self.apis = {}
        for apiLang in Preferences.Prefs.editorAPIDefaults.keys():
            self.apis[apiLang] = None
        
    def setSbInfo(self, sbFile, sbLine, sbPos, sbWritable):
        """
        Public method to transfer statusbar info from the user interface to viewmanager.
        
        @param sbFile reference to the file part of the statusbar
        @param sbLine reference to the line number part of the statusbar
        @param sbPos reference to the character position part of the statusbar
        @param sbWritable reference to the writability indicator part of the statusbar
        """
        self.sbFile = sbFile
        self.sbLine = sbLine
        self.sbPos = sbPos
        self.sbWritable = sbWritable
        self.setSbFile()
        
    def canCascade(self):
        """
        Public method to signal if cascading of managed windows is available.
        
        @return flag indicating cascading of windows is available
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    def canTile(self):
        """
        Public method to signal if tiling of managed windows is available.
        
        @return flag indicating tiling of windows is available
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    def tile(self):
        """
        Public method to tile the managed windows.
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def cascade(self):
        """
        Public method to cascade the managed windows.
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'

    def removeAllViews(self):
        """
        Private method to remove all views (i.e. windows)
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def removeView(self, win):
        """
        Private method to remove a view (i.e. window)
        
        @param win editor window to be removed
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def addView(self, win, fn=None):
        """
        Private method to add a view (i.e. window)
        
        @param win editor window to be added
        @param fn filename of this editor
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def showView(self, win, fn=None):
        """
        Private method to show a view (i.e. window)
        
        @param win editor window to be shown
        @param fn filename of this editor
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def activeWindow(self):
        """
        Private method to return the active (i.e. current) window.
        
        @return reference to the active editor
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def handleShowWindowMenu(self, windowMenu):
        """
        Private method to set up the viewmanager part of the Window menu.
        
        @param windowMenu reference to the window menu
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def initWindowActions(self):
        """
        Define the user interface actions for window handling.
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'

    def setEditorName(self, editor, newName):
        """
        Change the displayed name of the editor.
        
        @param editor editor window to be changed
        @param newName new name to be shown (string or QString)
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    def handleModificationStatusChanged(self, m, editor):
        """
        Private slot to handle the modificationStatusChanged signal.
        
        @param m flag indicating the modification status (boolean)
        @param editor editor window changed
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    #####################################################################
    ## methods above need to be implemented by a subclass
    #####################################################################
    
    def canSplit(self):
        """
        public method to signal if splitting of the view is available.
        
        @return flag indicating splitting of the view is available.
        """
        return 0
        
    def addSplit(self):
        """
        Public method used to split the current view.
        """
        pass
        
    def removeSplit(self):
        """
        Public method used to remove the current split view.
        
        @return Flag indicating successful deletion
        """
        return 0
        
    def setSplitOrientation(self, orientation):
        """
        Public method used to set the orientation of the split view.
        
        @param orientation orientation of the split
                (QSplitter.Horizontal or QSplitter.Vertical)
        """
        pass
        
    def eventFilter(self, object, event):
        """
        Private method called to filter an event.
        
        @param object object, that generated the event (QObject)
        @param event the event, that was generated by object (QEvent)
        @return flag indicating if event was filtered out
        """
        return 0
        
    def focusInEvent(self, event):
        """
        Public method called when the viewmanager receives focus.
        
        @param event the event object (QFocusEvent)
        """
        self.editorActGrp.setEnabled(1)
        
    def focusOutEvent(self, event):
        """
        Public method called when the viewmanager loses focus.
        
        @param event the event object (QFocusEvent)
        """
        self.editorActGrp.setEnabled(0)
        
    #####################################################################
    ## methods above need to be implemented by a subclass, that supports
    ## splitting of the viewmanager area.
    #####################################################################
    
    def initActions(self):
        """
        Public method defining the user interface actions.
        """
        # list containing all edit actions
        self.editActions = []
        
        # list containing all file actions
        self.fileActions = []
        
        # list containing all search actions
        self.searchActions = []
        
        # list containing all view actions
        self.viewActions = []
        
        # list containing all window actions
        self.windowActions = []
        
        # list containing all macro actions
        self.macroActions = []
        
        # list containing all bookmark actions
        self.bookmarkActions = []
        
        self.initWindowActions()
        self.initFileActions()
        self.initEditActions()
        self.initSearchActions()
        self.initViewActions()
        self.initMacroActions()
        self.initBookmarkActions()
        
    ##################################################################
    ## Initialize the file related actions, file menu and toolbar
    ##################################################################
    
    def initFileActions(self):
        """
        Private method defining the user interface actions for file handling.
        """
        self.newAct = QAction(self.trUtf8('New'),
                QIconSet(UI.PixmapCache.getPixmap("new")),
                self.trUtf8('&New'),
                QKeySequence(self.trUtf8("CTRL+N","File|New")),self)
        self.newAct.setStatusTip(self.trUtf8('Open an empty editor window'))
        self.newAct.setWhatsThis(self.trUtf8(
            """<b>New</b>"""
            """<p>An empty editor window will be created.</p>"""
        ))
        self.connect(self.newAct,SIGNAL('activated()'),self.newEditor)
        self.fileActions.append(self.newAct)
        
        self.openAct = QAction(self.trUtf8('Open'),
                QIconSet(UI.PixmapCache.getPixmap("open")),
                self.trUtf8('&Open...'),
                QKeySequence(self.trUtf8("CTRL+O","File|Open")),self)
        self.openAct.setStatusTip(self.trUtf8('Open a Python file'))
        self.openAct.setWhatsThis(self.trUtf8(
            """<b>Open a Python file</b>"""
            """<p>You will be asked for the name of a Python file to be opened"""
            """ in an editor window.</p>"""
        ))
        self.connect(self.openAct,SIGNAL('activated()'),self.handleOpen)
        self.fileActions.append(self.openAct)
        
        self.closeActGrp = QActionGroup(self)
        
        self.closeAct = QAction(self.trUtf8('Close'),
                QIconSet(UI.PixmapCache.getPixmap("close")),
                self.trUtf8('&Close'),
                QKeySequence(self.trUtf8("CTRL+W","File|Close")),
                self.closeActGrp)
        self.closeAct.setStatusTip(self.trUtf8('Close the current window'))
        self.closeAct.setWhatsThis(self.trUtf8(
            """<b>Close Window</b>"""
            """<p>Close the current window.</p>"""
        ))
        self.connect(self.closeAct,SIGNAL('activated()'),self.handleClose)
        self.fileActions.append(self.closeAct)

        self.closeAllAct = QAction(self.trUtf8('Close All'),
                self.trUtf8('Clos&e All'),
                0,self.closeActGrp)
        self.closeAllAct.setStatusTip(self.trUtf8('Close all editor windows'))
        self.closeAllAct.setWhatsThis(self.trUtf8(
            """<b>Close All Windows</b>"""
            """<p>Close all editor windows.</p>"""
        ))
        self.connect(self.closeAllAct,SIGNAL('activated()'),self.handleCloseAll)
        self.fileActions.append(self.closeAllAct)
        
        self.closeActGrp.setEnabled(0)
        
        self.saveActGrp = QActionGroup(self)

        self.saveAct = QAction(self.trUtf8('Save'),
                QIconSet(UI.PixmapCache.getPixmap("fileSave")),
                self.trUtf8('&Save'),
                QKeySequence(self.trUtf8("CTRL+S","File|Save")),
                self.saveActGrp)
        self.saveAct.setStatusTip(self.trUtf8('Save the current file'))
        self.saveAct.setWhatsThis(self.trUtf8(
            """<b>Save File</b>"""
            """<p>Save the contents of current editor window.</p>"""
        ))
        self.connect(self.saveAct,SIGNAL('activated()'),self.saveCurrentEditor)
        self.fileActions.append(self.saveAct)

        self.saveAsAct = QAction(self.trUtf8('Save as'),
                QIconSet(UI.PixmapCache.getPixmap("fileSaveAs")),
                self.trUtf8('Save &as...'),
                QKeySequence(self.trUtf8("SHIFT+CTRL+S","File|Save As")),
                self.saveActGrp)
        self.saveAsAct.setStatusTip(self.trUtf8('Save the current file to a new one'))
        self.saveAsAct.setWhatsThis(self.trUtf8(
            """<b>Save File as</b>"""
            """<p>Save the contents of current editor window to a new file."""
            """ The file can be entered in a file selection dialog.</p>"""
        ))
        self.connect(self.saveAsAct,SIGNAL('activated()'),self.saveAsCurrentEditor)
        self.fileActions.append(self.saveAsAct)

        self.saveAllAct = QAction(self.trUtf8('Save all'),
                QIconSet(UI.PixmapCache.getPixmap("fileSaveAll")),
                self.trUtf8('Save a&ll...'),
                0,self.saveActGrp)
        self.saveAllAct.setStatusTip(self.trUtf8('Save all files'))
        self.saveAllAct.setWhatsThis(self.trUtf8(
            """<b>Save All Files</b>"""
            """<p>Save the contents of all editor windows.</p>"""
        ))
        self.connect(self.saveAllAct,SIGNAL('activated()'),self.saveAllEditors)
        self.fileActions.append(self.saveAllAct)
        
        self.saveActGrp.setEnabled(0)

        self.saveToProjectAct = QAction(self.trUtf8('Save to Project'),
                QIconSet(UI.PixmapCache.getPixmap("fileSaveProject")),
                self.trUtf8('Save to Pro&ject'),
                0,self)
        self.saveToProjectAct.setStatusTip(self.trUtf8('Save the current file to the current project'))
        self.saveToProjectAct.setWhatsThis(self.trUtf8(
            """<b>Save to Project</b>"""
            """<p>Save the contents of the current editor window to the"""
            """ current project. After the file has been saved, it is"""
            """ automatically added to the current project.</p>"""
        ))
        self.connect(self.saveToProjectAct,SIGNAL('activated()'),self.saveCurrentEditorToProject)
        self.saveToProjectAct.setEnabled(0)
        self.fileActions.append(self.saveToProjectAct)
        
        self.printAct = QAction(self.trUtf8('Print'),
                QIconSet(UI.PixmapCache.getPixmap("print")),
                self.trUtf8('&Print'),
                QKeySequence(self.trUtf8("CTRL+P","File|Print")),self)
        self.printAct.setStatusTip(self.trUtf8('Print the current file'))
        self.printAct.setWhatsThis(self.trUtf8(
            """<b>Print File</b>"""
            """<p>Print the contents of current editor window.</p>"""
        ))
        self.connect(self.printAct,SIGNAL('activated()'),self.printCurrentEditor)
        self.printAct.setEnabled(0)
        self.fileActions.append(self.printAct)

        self.printSelAct = QAction(self.trUtf8('Print Selection'),
                QIconSet(UI.PixmapCache.getPixmap("print")),
                self.trUtf8('Prin&t Selection'),
                QKeySequence(self.trUtf8("SHIFT+CTRL+P","File|Print")),self)
        self.printSelAct.setStatusTip(self.trUtf8('Print the selection of the current file'))
        self.printSelAct.setWhatsThis(self.trUtf8(
            """<b>Print Selection</b>"""
            """<p>Print the selection of the current editor window.</p>"""
        ))
        self.connect(self.printSelAct,SIGNAL('activated()'),self.printCurrentEditorSel)
        self.printSelAct.setEnabled(0)
        self.fileActions.append(self.printSelAct)

    def initFileMenu(self):
        """
        Public method to create the File menu.
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        self.recentMenu = QPopupMenu(menu)
        menu.insertTearOffHandle()
        self.newAct.addTo(menu)
        self.openAct.addTo(menu)
        menu.insertItem(self.trUtf8('Open &Recent Files'), self.recentMenu)
        self.connect(self.recentMenu,SIGNAL('aboutToShow()'),self.handleShowRecentMenu)
        menu.insertSeparator()
        self.closeAct.addTo(menu)
        self.closeAllAct.addTo(menu)
        menu.insertSeparator()
        self.saveAct.addTo(menu)
        self.saveAsAct.addTo(menu)
        self.saveAllAct.addTo(menu)
        self.saveToProjectAct.addTo(menu)
        menu.insertSeparator()
        self.printAct.addTo(menu)
        self.printSelAct.addTo(menu)

        return menu
        
    def initFileToolbar(self):
        """
        Public method to create the File toolbar.
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.newAct.addTo(tb)
        self.openAct.addTo(tb)
        self.closeAct.addTo(tb)
        tb.addSeparator()
        self.saveAct.addTo(tb)
        self.saveAsAct.addTo(tb)
        self.saveAllAct.addTo(tb)
        self.saveToProjectAct.addTo(tb)
        tb.addSeparator()
        self.printAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the edit related actions, edit menu and toolbar
    ##################################################################
    
    def initEditActions(self):
        """
        Private method defining the user interface actions for the edit commands.
        """
        self.editActGrp = QActionGroup(self)

        self.undoAct = QAction(self.trUtf8('Undo'),
                QIconSet(UI.PixmapCache.getPixmap("editUndo")),
                self.trUtf8('&Undo'),
                QKeySequence(self.trUtf8("CTRL+Z","Edit|Undo")),
                self.editActGrp)
        self.undoAct.setStatusTip(self.trUtf8('Undo the last change'))
        self.undoAct.setWhatsThis(self.trUtf8(
            """<b>Undo</b>"""
            """<p>Undo the last change done in the current editor.</p>"""
        ))
        self.connect(self.undoAct,SIGNAL('activated()'),self.handleEditUndo)
        self.editActions.append(self.undoAct)
        
        self.redoAct = QAction(self.trUtf8('Redo'),
                QIconSet(UI.PixmapCache.getPixmap("editRedo")),
                self.trUtf8('&Redo'),
                QKeySequence(self.trUtf8("CTRL+Y","Edit|Redo")),
                self.editActGrp)
        self.redoAct.setStatusTip(self.trUtf8('Redo the last change'))
        self.redoAct.setWhatsThis(self.trUtf8(
            """<b>Redo</b>"""
            """<p>Redo the last change done in the current editor.</p>"""
        ))
        self.connect(self.redoAct,SIGNAL('activated()'),self.handleEditRedo)
        self.editActions.append(self.redoAct)
        
        self.revertAct = QAction(self.trUtf8('Revert to last saved state'),
                self.trUtf8('Re&vert to last saved state'),
                QKeySequence(self.trUtf8("SHIFT+CTRL+Z","Edit|Revert")),
                self.editActGrp)
        self.revertAct.setStatusTip(self.trUtf8('Revert to last saved state'))
        self.revertAct.setWhatsThis(self.trUtf8(
            """<b>Revert to last saved state</b>"""
            """<p>Undo all changes up to the last saved state of the current editor.</p>"""
        ))
        self.connect(self.revertAct,SIGNAL('activated()'),self.handleEditRevert)
        self.editActions.append(self.revertAct)
        
        self.cutAct = QAction(self.trUtf8('Cut'),
                QIconSet(UI.PixmapCache.getPixmap("editCut")),
                self.trUtf8('Cu&t'),
                QKeySequence(self.trUtf8("CTRL+X","Edit|Cut")),
                self.editActGrp)
        self.cutAct.setStatusTip(self.trUtf8('Cut the selection'))
        self.cutAct.setWhatsThis(self.trUtf8(
            """<b>Cut</b>"""
            """<p>Cut the selected text of the current editor to the clipboard.</p>"""
        ))
        self.connect(self.cutAct,SIGNAL('activated()'),self.handleEditCut)
        self.editActions.append(self.cutAct)
        
        self.copyAct = QAction(self.trUtf8('Copy'),
                QIconSet(UI.PixmapCache.getPixmap("editCopy")),
                self.trUtf8('&Copy'),
                QKeySequence(self.trUtf8("CTRL+C","Edit|Copy")),
                self.editActGrp)
        self.copyAct.setStatusTip(self.trUtf8('Copy the selection'))
        self.copyAct.setWhatsThis(self.trUtf8(
            """<b>Copy</b>"""
            """<p>Copy the selected text of the current editor to the clipboard.</p>"""
        ))
        self.connect(self.copyAct,SIGNAL('activated()'),self.handleEditCopy)
        self.editActions.append(self.copyAct)
        
        self.pasteAct = QAction(self.trUtf8('Paste'),
                QIconSet(UI.PixmapCache.getPixmap("editPaste")),
                self.trUtf8('&Paste'),
                QKeySequence(self.trUtf8("CTRL+V","Edit|Paste")),
                self.editActGrp)
        self.pasteAct.setStatusTip(self.trUtf8('Paste the last cut/copied text'))
        self.pasteAct.setWhatsThis(self.trUtf8(
            """<b>Paste</b>"""
            """<p>Paste the last cut/copied text from the clipboard to"""
            """ the current editor.</p>"""
        ))
        self.connect(self.pasteAct,SIGNAL('activated()'),self.handleEditPaste)
        self.editActions.append(self.pasteAct)
        
        self.deleteAct = QAction(self.trUtf8('Clear'),
                QIconSet(UI.PixmapCache.getPixmap("editDelete")),
                self.trUtf8('Cl&ear'),
                0,self.editActGrp)
        self.deleteAct.setStatusTip(self.trUtf8('Clear all text'))
        self.deleteAct.setWhatsThis(self.trUtf8(
            """<b>Clear</b>"""
            """<p>Delete all text of the current editor.</p>"""
        ))
        self.connect(self.deleteAct,SIGNAL('activated()'),self.handleEditDelete)
        self.editActions.append(self.deleteAct)
        
        self.indentAct = QAction(self.trUtf8('Indent'),
                QIconSet(UI.PixmapCache.getPixmap("editIndent")),
                self.trUtf8('&Indent'),
                QKeySequence(self.trUtf8("CTRL+I","Edit|Indent")),
                self.editActGrp)
        self.indentAct.setStatusTip(self.trUtf8('Indent line'))
        self.indentAct.setWhatsThis(self.trUtf8(
            """<b>Indent</b>"""
            """<p>Indents the current line or the lines of the"""
            """ selection by one level.</p>"""
        ))
        self.connect(self.indentAct,SIGNAL('activated()'),self.handleEditIndent)
        self.editActions.append(self.indentAct)
        
        self.unindentAct = QAction(self.trUtf8('Unindent'),
                QIconSet(UI.PixmapCache.getPixmap("editUnindent")),
                self.trUtf8('U&nindent'),
                QKeySequence(self.trUtf8("CTRL+U","Edit|Unindent")),
                self.editActGrp)
        self.unindentAct.setStatusTip(self.trUtf8('Unindent line'))
        self.unindentAct.setWhatsThis(self.trUtf8(
            """<b>Unindent</b>"""
            """<p>Unindents the current line or the lines of the"""
            """ selection by one level.</p>"""
        ))
        self.connect(self.unindentAct,SIGNAL('activated()'),self.handleEditUnindent)
        self.editActions.append(self.unindentAct)
        
        self.commentAct = QAction(self.trUtf8('Comment'),
                QIconSet(UI.PixmapCache.getPixmap("editComment")),
                self.trUtf8('C&omment'),
                QKeySequence(self.trUtf8("CTRL+M","Edit|Comment")),
                self.editActGrp)
        self.commentAct.setStatusTip(self.trUtf8('Comment Line or Selection'))
        self.commentAct.setWhatsThis(self.trUtf8(
            """<b>Comment</b>"""
            """<p>Comments the current line or the lines of the"""
            """ current selection.</p>"""
        ))
        self.connect(self.commentAct,SIGNAL('activated()'),self.handleEditComment)
        self.editActions.append(self.commentAct)
        
        self.uncommentAct = QAction(self.trUtf8('Uncomment'),
                QIconSet(UI.PixmapCache.getPixmap("editUncomment")),
                self.trUtf8('Unco&mment'),
                QKeySequence(self.trUtf8("ALT+CTRL+M","Edit|Uncomment")),
                self.editActGrp)
        self.uncommentAct.setStatusTip(self.trUtf8('Uncomment Line or Selection'))
        self.uncommentAct.setWhatsThis(self.trUtf8(
            """<b>Uncomment</b>"""
            """<p>Uncomments the current line or the lines of the"""
            """ current selection.</p>"""
        ))
        self.connect(self.uncommentAct,SIGNAL('activated()'),self.handleEditUncomment)
        self.editActions.append(self.uncommentAct)
        
        self.streamCommentAct = QAction(self.trUtf8('Stream Comment'),
                self.trUtf8('Stream Comment'),
                0, self.editActGrp)
        self.streamCommentAct.setStatusTip(self.trUtf8('Stream Comment Line or Selection'))
        self.streamCommentAct.setWhatsThis(self.trUtf8(
            """<b>Stream Comment</b>"""
            """<p>Stream comments the current line or the current selection.</p>"""
        ))
        self.connect(self.streamCommentAct,SIGNAL('activated()'),self.handleEditStreamComment)
        self.editActions.append(self.streamCommentAct)
        
        self.boxCommentAct = QAction(self.trUtf8('Box Comment'),
                self.trUtf8('Box Comment'),
                0, self.editActGrp)
        self.boxCommentAct.setStatusTip(self.trUtf8('Box Comment Line or Selection'))
        self.boxCommentAct.setWhatsThis(self.trUtf8(
            """<b>Box Comment</b>"""
            """<p>Box comments the current line or the lines of the"""
            """ current selection.</p>"""
        ))
        self.connect(self.boxCommentAct,SIGNAL('activated()'),self.handleEditBoxComment)
        self.editActions.append(self.boxCommentAct)
        
        self.selectBraceAct = QAction(self.trUtf8('Select to brace'),
                self.trUtf8('Select to &brace'),
                QKeySequence(self.trUtf8("CTRL+E","Edit|Select to brace")),
                self.editActGrp)
        self.selectBraceAct.setStatusTip(self.trUtf8('Select text to the matching brace'))
        self.selectBraceAct.setWhatsThis(self.trUtf8(
            """<b>Selct to brace</b>"""
            """<p>Select text of the current editor to the matching brace.</p>"""
        ))
        self.connect(self.selectBraceAct,SIGNAL('activated()'),self.handleEditSelectBrace)
        self.editActions.append(self.selectBraceAct)
        
        self.selectAllAct = QAction(self.trUtf8('Select all'),
                self.trUtf8('&Select all'),
                QKeySequence(self.trUtf8("CTRL+A","Edit|Select all")),
                self.editActGrp)
        self.selectAllAct.setStatusTip(self.trUtf8('Select all text'))
        self.selectAllAct.setWhatsThis(self.trUtf8(
            """<b>Selct All</b>"""
            """<p>Select all text of the current editor.</p>"""
        ))
        self.connect(self.selectAllAct,SIGNAL('activated()'),self.handleEditSelectAll)
        self.editActions.append(self.selectAllAct)
        
        self.deselectAllAct = QAction(self.trUtf8('Deselect all'),
                self.trUtf8('&Deselect all'),
                QKeySequence(self.trUtf8("ALT+CTRL+A","Edit|Deselect all")),
                self.editActGrp)
        self.deselectAllAct.setStatusTip(self.trUtf8('Deselect all text'))
        self.deselectAllAct.setWhatsThis(self.trUtf8(
            """<b>Deselct All</b>"""
            """<p>Deselect all text of the current editor.</p>"""
        ))
        self.connect(self.deselectAllAct,SIGNAL('activated()'),self.handleEditDeselectAll)
        self.editActions.append(self.deselectAllAct)
        
        self.convertEOLAct = QAction(self.trUtf8('Convert Line End Characters'),
                self.trUtf8('Convert &Line End Characters'),
                0, self.editActGrp)
        self.convertEOLAct.setStatusTip(self.trUtf8('Convert Line End Characters'))
        self.convertEOLAct.setWhatsThis(self.trUtf8(
            """<b>Convert Line End Characters</b>"""
            """<p>Convert the line end characters to the currently set type.</p>"""
        ))
        self.connect(self.convertEOLAct,SIGNAL('activated()'),self.handleConvertEOL)
        self.editActions.append(self.convertEOLAct)
        
        self.autoCompleteAct = QAction(self.trUtf8('Autocomplete'),
                self.trUtf8('&Autocomplete'),
                QKeySequence(self.trUtf8("CTRL+Space","Edit|Autocomplete")),
                self.editActGrp)
        self.autoCompleteAct.setStatusTip(self.trUtf8('Autocomplete current word'))
        self.autoCompleteAct.setWhatsThis(self.trUtf8(
            """<b>Autocomplete</b>"""
            """<p>Performs an autocompletion of the word containing the cursor.</p>"""
        ))
        self.connect(self.autoCompleteAct,SIGNAL('activated()'),self.handleEditAutoComplete)
        self.editActions.append(self.autoCompleteAct)
        
        self.editActGrp.setEnabled(0)
        
        ####################################################################
        ## Below follow the actions for qscintilla standard commands.
        ####################################################################
        
        self.esm = QSignalMapper(self)
        self.connect(self.esm, SIGNAL('mapped(int)'), self.editorCommand)
        
        self.editorActGrp = QActionGroup(self.editActGrp)
        
        act = QAction(self.trUtf8('Move left one character'), 
                      self.trUtf8('Move left one character'), 
                      QKeySequence(self.trUtf8('Left')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_CHARLEFT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move right one character'), 
                      self.trUtf8('Move right one character'), 
                      QKeySequence(self.trUtf8('Right')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_CHARRIGHT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move up one line'), 
                      self.trUtf8('Move up one line'), 
                      QKeySequence(self.trUtf8('Up')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEUP)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move down one line'), 
                      self.trUtf8('Move down one line'), 
                      QKeySequence(self.trUtf8('Down')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEDOWN)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move left one word part'), 
                      self.trUtf8('Move left one word part'), 
                      QKeySequence(self.trUtf8('Alt+Left')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDPARTLEFT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move right one word part'), 
                      self.trUtf8('Move right one word part'), 
                      QKeySequence(self.trUtf8('Alt+Right')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDPARTRIGHT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move left one word'), 
                      self.trUtf8('Move left one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Left')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDLEFT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move right one word'), 
                      self.trUtf8('Move right one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Right')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDRIGHT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move to first visible character in line'), 
                      self.trUtf8('Move to first visible character in line'), 
                      QKeySequence(self.trUtf8('Home')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_VCHOME)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move to start of line'), 
                      self.trUtf8('Move to start of line'), 
                      QKeySequence(self.trUtf8('Alt+Home')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_HOMEDISPLAY)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move to end of line'), 
                      self.trUtf8('Move to end of line'), 
                      QKeySequence(self.trUtf8('End')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Scroll view down one line'), 
                      self.trUtf8('Scroll view down one line'), 
                      QKeySequence(self.trUtf8('Ctrl+Down')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINESCROLLDOWN)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Scroll view up one line'), 
                      self.trUtf8('Scroll view up one line'), 
                      QKeySequence(self.trUtf8('Ctrl+Up')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINESCROLLUP)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move up one paragraph'), 
                      self.trUtf8('Move up one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Up')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PARAUP)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move down one paragraph'), 
                      self.trUtf8('Move down one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Down')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PARADOWN)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move up one page'), 
                      self.trUtf8('Move up one page'), 
                      QKeySequence(self.trUtf8('PgUp')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PAGEUP)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move down one page'), 
                      self.trUtf8('Move down one page'), 
                      QKeySequence(self.trUtf8('PgDown')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PAGEDOWN)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move to start of text'), 
                      self.trUtf8('Move to start of text'), 
                      QKeySequence(self.trUtf8('Ctrl+Home')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DOCUMENTSTART)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Move to end of text'), 
                      self.trUtf8('Move to end of text'), 
                      QKeySequence(self.trUtf8('Ctrl+End')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DOCUMENTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Indent one level'), 
                      self.trUtf8('Indent one level'), 
                      QKeySequence(self.trUtf8('Tab')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_TAB)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Unindent one level'), 
                      self.trUtf8('Unindent one level'), 
                      QKeySequence(self.trUtf8('Shift+Tab')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_BACKTAB)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection left one character'), 
                      self.trUtf8('Extend selection left one character'), 
                      QKeySequence(self.trUtf8('Shift+Left')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_CHARLEFTEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection right one character'), 
                      self.trUtf8('Extend selection right one character'), 
                      QKeySequence(self.trUtf8('Shift+Right')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_CHARRIGHTEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection up one line'), 
                      self.trUtf8('Extend selection up one line'), 
                      QKeySequence(self.trUtf8('Shift+Up')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEUPEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection down one line'), 
                      self.trUtf8('Extend selection down one line'), 
                      QKeySequence(self.trUtf8('Shift+Down')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEDOWNEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection left one word part'), 
                      self.trUtf8('Extend selection left one word part'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Left')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDPARTLEFTEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection right one word part'), 
                      self.trUtf8('Extend selection right one word part'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Right')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDPARTRIGHTEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection left one word'), 
                      self.trUtf8('Extend selection left one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Left')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDLEFTEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection right one word'), 
                      self.trUtf8('Extend selection right one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Right')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_WORDRIGHTEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection to first visible character in line'), 
                      self.trUtf8('Extend selection to first visible character in line'), 
                      QKeySequence(self.trUtf8('Shift+Home')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_VCHOMEEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection to start of line'), 
                      self.trUtf8('Extend selection to start of line'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Home')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_HOMEDISPLAYEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection to end of line'), 
                      self.trUtf8('Extend selection to end of line'), 
                      QKeySequence(self.trUtf8('Shift+End')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEENDEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection up one paragraph'), 
                      self.trUtf8('Extend selection up one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Up')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PARAUPEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection down one paragraph'), 
                      self.trUtf8('Extend selection down one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Down')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PARADOWNEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection up one page'), 
                      self.trUtf8('Extend selection up one page'), 
                      QKeySequence(self.trUtf8('Shift+PgUp')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PAGEUPEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection down one page'), 
                      self.trUtf8('Extend selection down one page'), 
                      QKeySequence(self.trUtf8('Shift+PgDown')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_PAGEDOWNEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection to start of text'), 
                      self.trUtf8('Extend selection to start of text'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Home')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DOCUMENTSTARTEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Extend selection to end of text'), 
                      self.trUtf8('Extend selection to end of text'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+End')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DOCUMENTENDEXTEND)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete previous character'), 
                      self.trUtf8('Delete previous character'), 
                      QKeySequence(self.trUtf8('Backspace')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DELETEBACK)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete previous character if not at line start'), 
                      self.trUtf8('Delete previous character if not at line start'), 
                      0, self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DELETEBACKNOTLINE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete current character'), 
                      self.trUtf8('Delete current character'), 
                      QKeySequence(self.trUtf8('Del')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_CLEAR)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete word to left'), 
                      self.trUtf8('Delete word to left'), 
                      QKeySequence(self.trUtf8('Ctrl+Backspace')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DELWORDLEFT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete word to right'), 
                      self.trUtf8('Delete word to right'), 
                      QKeySequence(self.trUtf8('Ctrl+Del')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DELWORDRIGHT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete line to left'), 
                      self.trUtf8('Delete line to left'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Backspace')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DELLINELEFT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete line to right'), 
                      self.trUtf8('Delete line to right'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Del')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_DELLINERIGHT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Insert new line'), 
                      self.trUtf8('Insert new line'), 
                      QKeySequence(self.trUtf8('Return')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_NEWLINE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Delete current line'), 
                      self.trUtf8('Delete current line'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+L')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEDELETE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Duplicate current line'), 
                      self.trUtf8('Duplicate current line'), 
                      QKeySequence(self.trUtf8('Ctrl+D')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINEDUPLICATE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Swap current and previous lines'), 
                      self.trUtf8('Swap current and previous lines'), 
                      QKeySequence(self.trUtf8('Ctrl+T')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINETRANSPOSE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Cut current line'), 
                      self.trUtf8('Cut current line'), 
                      QKeySequence(self.trUtf8('Alt+Shift+L')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINECUT)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Copy current line'), 
                      self.trUtf8('Copy current line'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+T')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LINECOPY)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Toggle insert/overtype'), 
                      self.trUtf8('Toggle insert/overtype'), 
                      QKeySequence(self.trUtf8('Ins')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_EDITTOGGLEOVERTYPE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Convert selection to lower case'), 
                      self.trUtf8('Convert selection to lower case'), 
                      QKeySequence(self.trUtf8('Alt+Shift+U')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_LOWERCASE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = QAction(self.trUtf8('Convert selection to upper case'), 
                      self.trUtf8('Convert selection to upper case'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+U')), self.editorActGrp)
        self.esm.setMapping(act, QextScintilla.SCI_UPPERCASE)
        self.connect(act,SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
    def initEditMenu(self):
        """
        Public method to create the Edit menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        menu.insertTearOffHandle()
        self.undoAct.addTo(menu)
        self.redoAct.addTo(menu)
        self.revertAct.addTo(menu)
        menu.insertSeparator()
        self.cutAct.addTo(menu)
        self.copyAct.addTo(menu)
        self.pasteAct.addTo(menu)
        self.deleteAct.addTo(menu)
        menu.insertSeparator()
        self.indentAct.addTo(menu)
        self.unindentAct.addTo(menu)
        menu.insertSeparator()
        self.commentAct.addTo(menu)
        self.uncommentAct.addTo(menu)
        self.streamCommentAct.addTo(menu)
        self.boxCommentAct.addTo(menu)
        menu.insertSeparator()
        self.autoCompleteAct.addTo(menu)
        menu.insertSeparator()
        self.searchAct.addTo(menu)
        self.searchAgainAct.addTo(menu)
        self.replaceAct.addTo(menu)
        menu.insertSeparator()
        self.searchFilesAct.addTo(menu)
        menu.insertSeparator()
        self.gotoAct.addTo(menu)
        self.gotoBraceAct.addTo(menu)
        menu.insertSeparator()
        self.selectBraceAct.addTo(menu)
        self.selectAllAct.addTo(menu)
        self.deselectAllAct.addTo(menu)
        menu.insertSeparator()
        self.convertEOLAct.addTo(menu)
        
        return menu
        
    def initEditToolbar(self):
        """
        Public method to create the Edit toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.undoAct.addTo(tb)
        self.redoAct.addTo(tb)
        tb.addSeparator()
        self.cutAct.addTo(tb)
        self.copyAct.addTo(tb)
        self.pasteAct.addTo(tb)
        self.deleteAct.addTo(tb)
        tb.addSeparator()
        self.indentAct.addTo(tb)
        self.unindentAct.addTo(tb)
        tb.addSeparator()
        self.commentAct.addTo(tb)
        self.uncommentAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the search related actions, search menu and toolbar
    ##################################################################
    
    def initSearchActions(self):
        """
        Private method defining the user interface actions for the search commands.
        """
        self.searchActGrp = QActionGroup(self)
        
        self.searchAct = QAction(self.trUtf8('Search'),
                QIconSet(UI.PixmapCache.getPixmap("find")),
                self.trUtf8('&Search...'),
                QKeySequence(self.trUtf8("CTRL+F","Search|Search")),
                self.searchActGrp)
        self.searchAct.setStatusTip(self.trUtf8('Search for a text'))
        self.searchAct.setWhatsThis(self.trUtf8(
            """<b>Search</b>"""
            """<p>Search for some text in the current editor. A"""
            """ dialog is shown to enter the searchtext and options"""
            """ for the search.</p>"""
        ))
        self.connect(self.searchAct,SIGNAL('activated()'),self.handleSearch)
        self.searchActions.append(self.searchAct)
        
        self.searchAgainAct = QAction(self.trUtf8('Search again'),
                QIconSet(UI.PixmapCache.getPixmap("findNext")),
                self.trUtf8('Search &again'),
                Qt.Key_F3,self.searchActGrp)
        self.searchAgainAct.setStatusTip(self.trUtf8('Search again for text'))
        self.searchAgainAct.setWhatsThis(self.trUtf8(
            """<b>Search again</b>"""
            """<p>Search again for some text in the current editor."""
            """ The previously entered searchtext and options are reused.</p>"""
        ))
        self.connect(self.searchAgainAct,SIGNAL('activated()'),self.searchDlg.handleFindNext)
        self.searchActions.append(self.searchAgainAct)
        
        self.replaceAct = QAction(self.trUtf8('Replace'),
                self.trUtf8('&Replace...'),
                QKeySequence(self.trUtf8("CTRL+R","Search|Replace")),
                self.searchActGrp)
        self.replaceAct.setStatusTip(self.trUtf8('Replace some text'))
        self.replaceAct.setWhatsThis(self.trUtf8(
            """<b>Replace</b>"""
            """<p>Search for some text in the current editor and replace it. A"""
            """ dialog is shown to enter the searchtext, the replacement text"""
            """ and options for the search and replace.</p>"""
        ))
        self.connect(self.replaceAct,SIGNAL('activated()'),self.handleReplace)
        self.searchActions.append(self.replaceAct)
        
        self.gotoAct = QAction(self.trUtf8('Goto Line'),
                QIconSet(UI.PixmapCache.getPixmap("goto")),
                self.trUtf8('&Goto Line...'),
                QKeySequence(self.trUtf8("CTRL+G","Search|Goto Line")),
                self.searchActGrp)
        self.gotoAct.setStatusTip(self.trUtf8('Goto Line'))
        self.gotoAct.setWhatsThis(self.trUtf8(
            """<b>Goto Line</b>"""
            """<p>Go to a specific line of text in the current editor."""
            """ A dialog is shown to enter the linenumber.</p>"""
        ))
        self.connect(self.gotoAct,SIGNAL('activated()'),self.handleGoto)
        self.searchActions.append(self.gotoAct)
        
        self.gotoBraceAct = QAction(self.trUtf8('Goto Brace'),
                QIconSet(UI.PixmapCache.getPixmap("gotoBrace")),
                self.trUtf8('Goto &Brace'),
                QKeySequence(self.trUtf8("CTRL+L","Search|Goto Brace")),
                self.searchActGrp)
        self.gotoBraceAct.setStatusTip(self.trUtf8('Goto Brace'))
        self.gotoBraceAct.setWhatsThis(self.trUtf8(
            """<b>Goto Brace</b>"""
            """<p>Go to the matching brace in the current editor.</p>"""
        ))
        self.connect(self.gotoBraceAct,SIGNAL('activated()'),self.handleGotoBrace)
        self.searchActions.append(self.gotoBraceAct)
        
        self.searchActGrp.setEnabled(0)
        
        self.searchFilesAct = QAction(self.trUtf8('Search in Files'),
                QIconSet(UI.PixmapCache.getPixmap("projectFind")),
                self.trUtf8('Search in &Files...'),
                QKeySequence(self.trUtf8("SHIFT+CTRL+F","Search|Search Files")),
                self)
        self.searchFilesAct.setStatusTip(self.trUtf8('Search for a text in files'))
        self.searchFilesAct.setWhatsThis(self.trUtf8(
            """<b>Search in Files</b>"""
            """<p>Search for some text in the files of a directory tree"""
            """ or the project. A dialog is shown to enter the searchtext"""
            """ and options for the search and to display the result.</p>"""
        ))
        self.connect(self.searchFilesAct,SIGNAL('activated()'),self.handleSearchFiles)
        self.searchActions.append(self.searchFilesAct)
        
    def initSearchToolbar(self):
        """
        Public method to create the Search toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.searchAct.addTo(tb)
        self.searchAgainAct.addTo(tb)
        tb.addSeparator()
        self.searchFilesAct.addTo(tb)
        tb.addSeparator()
        self.gotoAct.addTo(tb)
        self.gotoBraceAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the view related actions, view menu and toolbar
    ##################################################################
    
    def initViewActions(self):
        """
        Protected method defining the user interface actions for the view commands.
        """
        self.viewActGrp = QActionGroup(self)
        self.viewFoldActGrp = QActionGroup(self)

        self.zoomInAct = QAction(self.trUtf8('Zoom in'),
                            QIconSet(UI.PixmapCache.getPixmap("zoomIn")),
                            self.trUtf8('Zoom &in'),
                            Qt.CTRL+Qt.Key_Plus, self.viewActGrp)
        self.zoomInAct.setStatusTip(self.trUtf8('Zoom in on the text'))
        self.zoomInAct.setWhatsThis(self.trUtf8(
                """<b>Zoom in</b>"""
                """<p>Zoom in on the text. This makes the text bigger.</p>"""
                ))
        self.connect(self.zoomInAct,SIGNAL('activated()'),self.handleZoomIn)
        self.viewActions.append(self.zoomInAct)
        
        self.zoomOutAct = QAction(self.trUtf8('Zoom out'),
                            QIconSet(UI.PixmapCache.getPixmap("zoomOut")),
                            self.trUtf8('Zoom &out'),
                            Qt.CTRL+Qt.Key_Minus, self.viewActGrp)
        self.zoomOutAct.setStatusTip(self.trUtf8('Zoom out on the text'))
        self.zoomOutAct.setWhatsThis(self.trUtf8(
                """<b>Zoom out</b>"""
                """<p>Zoom out on the text. This makes the text smaller.</p>"""
                ))
        self.connect(self.zoomOutAct,SIGNAL('activated()'),self.handleZoomOut)
        self.viewActions.append(self.zoomOutAct)
        
        self.zoomToAct = QAction(self.trUtf8('Zoom'),
                            QIconSet(UI.PixmapCache.getPixmap("zoomTo")),
                            self.trUtf8('&Zoom'),
                            0, self.viewActGrp)
        self.zoomToAct.setStatusTip(self.trUtf8('Zoom the text'))
        self.zoomToAct.setWhatsThis(self.trUtf8(
                """<b>Zoom</b>"""
                """<p>Zoom the text. This opens a dialog where the"""
                """ desired size can be entered.</p>"""
                ))
        self.connect(self.zoomToAct,SIGNAL('activated()'),self.handleZoom)
        self.viewActions.append(self.zoomToAct)
        
        self.toggleAllAct = QAction(self.trUtf8('Toggle all folds'),
                            self.trUtf8('Toggle &all folds'),
                            0, self.viewFoldActGrp)
        self.toggleAllAct.setStatusTip(self.trUtf8('Toggle all folds'))
        self.toggleAllAct.setWhatsThis(self.trUtf8(
                """<b>Toggle all folds</b>"""
                """<p>Toggle all folds of the current editor.</p>"""
                ))
        self.connect(self.toggleAllAct,SIGNAL('activated()'),self.handleToggleAll)
        self.viewActions.append(self.toggleAllAct)
        
        self.toggleCurrentAct = QAction(self.trUtf8('Toggle current fold'),
                            self.trUtf8('Toggle &current fold'),
                            0, self.viewFoldActGrp)
        self.toggleCurrentAct.setStatusTip(self.trUtf8('Toggle current fold'))
        self.toggleCurrentAct.setWhatsThis(self.trUtf8(
                """<b>Toggle current fold</b>"""
                """<p>Toggle the folds of the current line of the current editor.</p>"""
                ))
        self.connect(self.toggleCurrentAct,SIGNAL('activated()'),self.handleToggleCurrent)
        self.viewActions.append(self.toggleCurrentAct)
        
        self.unhighlightAct = QAction(self.trUtf8('Remove all highlights'),
                            QIconSet(UI.PixmapCache.getPixmap("unhighlight")),
                            self.trUtf8('Remove all highlights'),
                            0, self)
        self.unhighlightAct.setStatusTip(self.trUtf8('Remove all highlights'))
        self.unhighlightAct.setWhatsThis(self.trUtf8(
                """<b>Remove all highlights</b>"""
                """<p>Remove the highlights of all editors.</p>"""
                ))
        self.connect(self.unhighlightAct,SIGNAL('activated()'),self.unhighlight)
        self.viewActions.append(self.unhighlightAct)
        
        self.splitViewAct = QAction(self.trUtf8('Split view'),
                            QIconSet(UI.PixmapCache.getPixmap("splitVertical")),
                            self.trUtf8('&Split view'),
                            0, self)
        self.splitViewAct.setStatusTip(self.trUtf8('Add a split to the view'))
        self.splitViewAct.setWhatsThis(self.trUtf8(
                """<b>Split view</b>"""
                """<p>Add a split to the view.</p>"""
                ))
        self.connect(self.splitViewAct,SIGNAL('activated()'),self.handleSplitView)
        self.viewActions.append(self.splitViewAct)
        
        self.splitOrientationAct = QAction(self.trUtf8('Arrange horizontally'),
                            self.trUtf8('Arrange &horizontally'),
                            0, self, None, 1)
        self.splitOrientationAct.setStatusTip(self.trUtf8('Arrange the splitted views horizontally'))
        self.splitOrientationAct.setWhatsThis(self.trUtf8(
                """<b>Arrange horizontally</b>"""
                """<p>Arrange the splitted views horizontally.</p>"""
                ))
        self.splitOrientationAct.setOn(0)
        self.connect(self.splitOrientationAct,SIGNAL('activated()'),self.handleSplitOrientation)
        self.viewActions.append(self.splitOrientationAct)
        
        self.splitRemoveAct = QAction(self.trUtf8('Remove split'),
                            self.trUtf8('&Remove split'),
                            0, self)
        self.splitRemoveAct.setStatusTip(self.trUtf8('Remove the current split'))
        self.splitRemoveAct.setWhatsThis(self.trUtf8(
                """<b>Remove split</b>"""
                """<p>Remove the current split.</p>"""
                ))
        self.connect(self.splitRemoveAct,SIGNAL('activated()'),self.removeSplit)
        self.viewActions.append(self.splitRemoveAct)
        
        self.viewActGrp.setEnabled(0)
        self.viewFoldActGrp.setEnabled(0)
        self.unhighlightAct.setEnabled(0)
        self.splitViewAct.setEnabled(0)
        self.splitOrientationAct.setEnabled(0)
        self.splitRemoveAct.setEnabled(0)
        
    def initViewMenu(self):
        """
        Public method to create the View menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        menu.insertTearOffHandle()
        self.viewActGrp.addTo(menu)
        menu.insertSeparator()
        self.viewFoldActGrp.addTo(menu)
        menu.insertSeparator()
        self.unhighlightAct.addTo(menu)
        if self.canSplit():
            menu.insertSeparator()
            self.splitViewAct.addTo(menu)
            self.splitOrientationAct.addTo(menu)
            self.splitRemoveAct.addTo(menu)       
        return menu
        
    def initViewToolbar(self):
        """
        Public method to create the View toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.viewActGrp.addTo(tb)
        tb.addSeparator()
        self.unhighlightAct.addTo(tb)
        if self.canSplit():
            tb.addSeparator()
            self.splitViewAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the macro related actions and macro menu
    ##################################################################
    
    def initMacroActions(self):
        """
        Private method defining the user interface actions for the macro commands.
        """
        self.macroActGrp = QActionGroup(self)

        self.macroStartRecAct = QAction(self.trUtf8('Start Macro Recording'),
                            self.trUtf8('S&tart Macro Recording'),
                            0, self.macroActGrp)
        self.macroStartRecAct.setStatusTip(self.trUtf8('Start Macro Recording'))
        self.macroStartRecAct.setWhatsThis(self.trUtf8(
                """<b>Start Macro Recording</b>"""
                """<p>Start recording editor commands into a new macro.</p>"""
                ))
        self.connect(self.macroStartRecAct,SIGNAL('activated()'),self.handleMacroStartRecording)
        self.macroActions.append(self.macroStartRecAct)
        
        self.macroStopRecAct = QAction(self.trUtf8('Stop Macro Recording'),
                            self.trUtf8('Sto&p Macro Recording'),
                            0, self.macroActGrp)
        self.macroStopRecAct.setStatusTip(self.trUtf8('Stop Macro Recording'))
        self.macroStopRecAct.setWhatsThis(self.trUtf8(
                """<b>Stop Macro Recording</b>"""
                """<p>Stop recording editor commands into a new macro.</p>"""
                ))
        self.connect(self.macroStopRecAct,SIGNAL('activated()'),self.handleMacroStopRecording)
        self.macroActions.append(self.macroStopRecAct)
        
        self.macroRunAct = QAction(self.trUtf8('Run Macro'),
                            self.trUtf8('&Run Macro'),
                            0, self.macroActGrp)
        self.macroRunAct.setStatusTip(self.trUtf8('Run Macro'))
        self.macroRunAct.setWhatsThis(self.trUtf8(
                """<b>Run Macro</b>"""
                """<p>Run a previously recorded editor macro.</p>"""
                ))
        self.connect(self.macroRunAct,SIGNAL('activated()'),self.handleMacroRun)
        self.macroActions.append(self.macroRunAct)
        
        self.macroDeleteAct = QAction(self.trUtf8('Delete Macro'),
                            self.trUtf8('&Delete Macro'),
                            0, self.macroActGrp)
        self.macroDeleteAct.setStatusTip(self.trUtf8('Delete Macro'))
        self.macroDeleteAct.setWhatsThis(self.trUtf8(
                """<b>Delete Macro</b>"""
                """<p>Delete a previously recorded editor macro.</p>"""
                ))
        self.connect(self.macroDeleteAct,SIGNAL('activated()'),self.handleMacroDelete)
        self.macroActions.append(self.macroDeleteAct)
        
        self.macroLoadAct = QAction(self.trUtf8('Load Macro'),
                            self.trUtf8('&Load Macro'),
                            0, self.macroActGrp)
        self.macroLoadAct.setStatusTip(self.trUtf8('Load Macro'))
        self.macroLoadAct.setWhatsThis(self.trUtf8(
                """<b>Load Macro</b>"""
                """<p>Load an editor macro from a file.</p>"""
                ))
        self.connect(self.macroLoadAct,SIGNAL('activated()'),self.handleMacroLoad)
        self.macroActions.append(self.macroLoadAct)
        
        self.macroSaveAct = QAction(self.trUtf8('Save Macro'),
                            self.trUtf8('&Save Macro'),
                            0, self.macroActGrp)
        self.macroSaveAct.setStatusTip(self.trUtf8('Save Macro'))
        self.macroSaveAct.setWhatsThis(self.trUtf8(
                """<b>Save Macro</b>"""
                """<p>Save a previously recorded editor macro to a file.</p>"""
                ))
        self.connect(self.macroSaveAct,SIGNAL('activated()'),self.handleMacroSave)
        self.macroActions.append(self.macroSaveAct)
        
        self.macroActGrp.setEnabled(0)
        
    def initMacroMenu(self):
        """
        Public method to create the Macro menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        menu.insertTearOffHandle()
        self.macroActGrp.addTo(menu)
        return menu
    
    #####################################################################
    ## Initialize the bookmark related actions, bookmark menu and toolbar
    #####################################################################
    
    def initBookmarkActions(self):
        """
        Private method defining the user interface actions for the bookmarks commands.
        """
        self.bookmarkActGrp = QActionGroup(self)

        self.bookmarkToggleAct = QAction(self.trUtf8('Toggle Bookmark'),
                            QIconSet(UI.PixmapCache.getPixmap("bookmarkToggle")),
                            self.trUtf8('&Toggle Bookmark'),
                            QKeySequence(self.trUtf8("ALT+CTRL+T","Bookmark|Toggle")),
                            self.bookmarkActGrp)
        self.bookmarkToggleAct.setStatusTip(self.trUtf8('Toggle Bookmark'))
        self.bookmarkToggleAct.setWhatsThis(self.trUtf8(
                """<b>Toggle Bookmark</b>"""
                """<p>Toggle a bookmark at the current line of the current editor.</p>"""
                ))
        self.connect(self.bookmarkToggleAct,SIGNAL('activated()'),self.handleToggleBookmark)
        self.bookmarkActions.append(self.bookmarkToggleAct)
        
        
        self.bookmarkNextAct = QAction(self.trUtf8('Next Bookmark'),
                            QIconSet(UI.PixmapCache.getPixmap("bookmarkNext")),
                            self.trUtf8('&Next Bookmark'),
                            QKeySequence(self.trUtf8("CTRL+PgDown","Bookmark|Next")),
                            self.bookmarkActGrp)
        self.bookmarkNextAct.setStatusTip(self.trUtf8('Next Bookmark'))
        self.bookmarkNextAct.setWhatsThis(self.trUtf8(
                """<b>Next Bookmark</b>"""
                """<p>Go to next bookmark of the current editor.</p>"""
                ))
        self.connect(self.bookmarkNextAct,SIGNAL('activated()'),self.handleNextBookmark)
        self.bookmarkActions.append(self.bookmarkNextAct)
        
        self.bookmarkPreviousAct = QAction(self.trUtf8('Previous Bookmark'),
                            QIconSet(UI.PixmapCache.getPixmap("bookmarkPrevious")),
                            self.trUtf8('&Previous Bookmark'),
                            QKeySequence(self.trUtf8("CTRL+PgUp","Bookmark|Previous")),
                            self.bookmarkActGrp)
        self.bookmarkPreviousAct.setStatusTip(self.trUtf8('Previous Bookmark'))
        self.bookmarkPreviousAct.setWhatsThis(self.trUtf8(
                """<b>Previous Bookmark</b>"""
                """<p>Go to previous bookmark of the current editor.</p>"""
                ))
        self.connect(self.bookmarkPreviousAct,SIGNAL('activated()'),self.handlePreviousBookmark)
        self.bookmarkActions.append(self.bookmarkPreviousAct)
        
        self.bookmarkClearAct = QAction(self.trUtf8('Clear Bookmarks'),
                            self.trUtf8('&Clear Bookmarks'),
                            QKeySequence(self.trUtf8("ALT+CTRL+C","Bookmark|Clear")),
                            self.bookmarkActGrp)
        self.bookmarkClearAct.setStatusTip(self.trUtf8('Clear Bookmarks'))
        self.bookmarkClearAct.setWhatsThis(self.trUtf8(
                """<b>Clear Bookmarks</b>"""
                """<p>Clear bookmarks of all editors.</p>"""
                ))
        self.connect(self.bookmarkClearAct,SIGNAL('activated()'),self.handleClearAllBookmarks)
        self.bookmarkActions.append(self.bookmarkClearAct)
        
        self.bookmarkActGrp.setEnabled(0)
        
    def initBookmarkMenu(self):
        """
        Public method to create the Bookmark menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        self.bookmarksMenu = QPopupMenu(menu)
        
        menu.insertTearOffHandle()
        self.bookmarkActGrp.addTo(menu)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('&Bookmarks'), self.bookmarksMenu)
        self.connect(self.bookmarksMenu,SIGNAL('aboutToShow()'),self.handleShowBookmarksMenu)
        self.connect(self.bookmarksMenu,SIGNAL('activated(int)'),self.handleBookmarkSelected)
        
        return menu
        
    def initBookmarkToolbar(self):
        """
        Public method to create the Bookmark toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.bookmarkActGrp.addTo(tb)
        
        return tb
    
    ##################################################################
    ## Methods and slots that deal with file and window handling
    ##################################################################
    
    def handleOpen(self,prog=None):
        """
        Public slot to open a Python (or other) file.
        
        @param prog name of file to be opened (string or QString)
        """
        # Get the file name if one wasn't specified.
        if prog is None:
            # set the cwd of the dialog based on the following search criteria:
            #     1: Directory of currently active editor
            #     2: Directory of currently active Project
            #     3: CWD
            prog = QFileDialog.getOpenFileName(self._getOpenStartDir(),
                        self.trUtf8('Python Files (*.py);;'
                            'Pyrex Files (*.pyx);;'
                            'Quixote Template Files (*.ptl);;'
                            'IDL Files (*.idl);;'
                            'C Files (*.h *.c);;'
                            'C++ Files (*.h *.hpp *.hh *.cxx *.cpp *.cc);;'
                            'C# Files (*.cs);;'
                            'HTML Files (*.html *.htm *.asp *.shtml *.css);;'
                            'PHP Files (*.php *.php3 *.php4 *.phtml);;'
                            'XML Files (*.xml *.xsl *.xslt *.dtd);;'
                            'Java Files (*.java);;'
                            'JavaScript Files (*.js);;'
                            'SQL Files (*.sql);;'
                            'Docbook Files (*.docbook);;'
                            'All Files (*)'),
                        self.ui)

            if prog.isNull():
                return

        prog = Utilities.normabspath(str(prog))

        # Open up the new file.
        self.handlePythonFile(prog)

    def checkDirty(self, editor):
        """
        Private method to check dirty status and open a message window.
        
        @param editor editor window to check
        @return flag indicating successful reset of the dirty flag (boolean)
        """
        if editor.isModified():
            fn = editor.getFileName()
            if fn is None:
                fn = self.trUtf8('Noname')
            res = QMessageBox.warning(self.parent(), 
                self.trUtf8("File Modified"),
                self.trUtf8("The file <b>%1</b> has unsaved changes.")
                    .arg(fn),
                self.trUtf8("&Save"), self.trUtf8("&Discard changes"),
                self.trUtf8("&Abort"), 0, 2)
            if res == 0:
                (ok, newName) = editor.saveFile()
                if ok:
                    self.setEditorName(editor, newName)
                return ok
            elif res == 2:
                return 0
        
        return 1
        
    def checkAllDirty(self):
        """
        Public method to check the dirty status of all editors.
        
        @return flag indicating successful reset of all dirty flags (boolean)
        """
        for editor in self.editors:
            if not self.checkDirty(editor):
                return 0
                
        return 1
        
    def closeEditor(self, editor):
        """
        Private method to close an editor window.
        
        @param editor editor window to be closed
        @return flag indicating success (boolean)
        """
        # save file if necessary
        if not self.checkDirty(editor):
            return 0
            
        # remove the window
        self.removeView(editor)
        self.editors.remove(editor)
        if not len(self.editors):
            self.handleLastEditorClosed()
            self.emit(PYSIGNAL('lastEditorClosed'), ())
        return 1
    
    def handleClose(self):
        """
        Public method to close the current window.
        
        @return flag indicating success (boolean)
        """
        aw = self.activeWindow()
        if aw is None:
            return 0
            
        res = self.closeEditor(aw)
        if res and aw == self.currentEditor:
            self.currentEditor = None
            
        return res
            
    def handleCloseAll(self):
        """
        Private method to close all editor windows via file menu.
        """
        savedEditors = self.editors[:]
        for editor in savedEditors:
            self.closeEditor(editor)
            
    def handleCloseWindow(self, fn):
        """
        Public method to close an arbitrary source editor.
        
        @param fn filename of editor to be closed
        @return flag indicating success (boolean)
        """
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                break
        else:
            return 1
            
        res = self.closeEditor(editor)
        if res and editor == self.currentEditor:
            self.currentEditor = None
            
        return res
        
    def handleExit(self):
        """
        Public method to handle the debugged program terminating.
        """
        if self.currentEditor is not None:
            self.currentEditor.highlight()
            self.currentEditor = None
            
        self.setSbFile()

    def handlePythonFile(self,pyfn,lineno=None):
        """
        Public method to handle the user selecting a file for display.
        
        @param pyfn name of file to be opened
        @param lineno line number to place the cursor at
        """
        try:
            self.displayPythonFile(pyfn,lineno)
        except IOError:
            pass

    def displayPythonFile(self,fn,lineno=None):
        """
        Public slot to display a file in an editor.
        
        @param fn name of file to be opened
        @param lineno line number to place the cursor at
        """
        isPyFile = lineno and lineno < 0
        newWin, editor = self.getEditor(fn, isPythonFile=isPyFile)
        
        if newWin:
            self.addView(editor, fn)
            self.handleModificationStatusChanged(editor.isModified(), editor)
        else:
            self.showView(editor, fn)
        self.checkActions(editor)
            
        if lineno is not None and lineno >= 0:
            editor.ensureVisible(lineno)
            editor.gotoLine(lineno)
        
        # insert filename into list of recently opened projects
        self.addToRecentList(fn)
            
    def addToRecentList(self, fn):
        """
        Public slot to add a filename to the list of recently opened files.
        
        @param fn name of the file to be added
        """
        self.recent.remove(fn)
        self.recent.prepend(fn)
        if len(self.recent) > 9:
            self.recent = self.recent[:9]

    def toggleWindow(self,w):
        """
        Private method to toggle a workspace window.
        
        @param w editor window to be toggled
        """
        if w.isHidden():
            w.show()
        else:
            w.hide()

    def setFileLine(self,fn,line,error=0):
        """
        Public method to update the user interface when the current program or line changes.
        
        @param fn filename of editor to update (string)
        @param line line number to highlight (int)
        @param error flag indicating an error highlight (boolean)
        """
        self.setSbFile(fn,line)

        try:
            newWin, self.currentEditor = self.getEditor(fn)
        except IOError:
            return

        # Change the highlighted line.
        self.currentEditor.highlight(line,error)

        if newWin:
            self.addView(self.currentEditor, fn)
        else:
            self.showView(self.currentEditor, fn)
        self.currentEditor.highlightVisible()
        self.checkActions(self.currentEditor, 0)
            
    def setSbFile(self,fn=None,line=None,pos=None):
        """
        Private method to set the file info in the status bar.
        
        @param fn filename to display (string)
        @param line line number to display (int)
        @param pos character position to display (int)
        """
        if fn is None:
            fn = ''
            writ = '   '
        else:
            if QFileInfo(fn).isWritable():
                writ = ' rw'
            else:
                writ = ' ro'
        
        self.sbWritable.setText(writ)
        self.sbFile.setText(self.trUtf8('File: %1').arg(fn,-50))

        if line is None:
            line = ''

        self.sbLine.setText(self.trUtf8('Line: %1').arg(line,5))

        if pos is None:
            pos = ''
            
        self.sbPos.setText(self.trUtf8('Pos: %1').arg(pos, 5))
        
    def unhighlight(self, current=0):
        """
        Public method to switch off all highlights.
        
        @param current flag indicating only the current editor should be unhighlighted
                (boolean)
        """
        if current: 
            if self.currentEditor is not None:
                self.currentEditor.highlight()
        else:
            for editor in self.editors:
                editor.highlight()

    def getOpenFilenames(self):
        """
        Public method returning a list of the filenames of all editors.
        
        @return list of all opened filenames (list of strings)
        """
        filenames = []
        for editor in self.editors:
            fn = editor.getFileName()
            if fn is not None:
                filenames.append(fn)
                
        return filenames
        
    def getEditor(self, fn, isPythonFile=0):
        """
        Private method to return the editor displaying the given file.
        
        If there is no editor with the given file, a new editor window is
        created.
        
        @param fn filename to look for
        @param isPythonFile flag indicating that this is a Python file
                even if it doesn't have the .py extension (boolean)
        @return tuple of two values giving a flag indicating a new window creation and
            a reference to the editor displaying this file
        """
        newWin = 0
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                break
        else:
            editor = Editor(self.dbs,fn,self,isPythonFile=isPythonFile)
            self.editors.append(editor)
            self.connect(editor, PYSIGNAL('modificationStatusChanged'),
                self.handleModificationStatusChanged)
            self.connect(editor, PYSIGNAL('cursorChanged'), self.handleCursorChanged)
            self.connect(editor, PYSIGNAL('editorSaved'), self.handleEditorSaved)
            self.connect(editor, PYSIGNAL('breakpointToggled'), self.handleBreakpointToggled)
            self.connect(editor, PYSIGNAL('bookmarkToggled'), self.handleBookmarkToggled)
            self.handleEditorOpened()
            self.emit(PYSIGNAL('editorOpened'), (fn,))
            newWin = 1

        return (newWin, editor)
        
    def getOpenEditor(self, fn):
        """
        Public method to return the editor displaying the given file.
        
        @param fn filename to look for
        @return a reference to the editor displaying this file or None, if
            no editor was found
        """
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                return editor
                
        return None

    def getActiveName(self):
        """
        Public method to retrieve the filename of the active window.
        
        @return filename of active window (string)
        """
        aw = self.activeWindow()
        if aw:
            return aw.getFileName()
        else:
            return None

    def saveEditor(self, fn):
        """
        Public method to save a named editor file.
        
        @param fn filename of editor to be saved (string)
        @return flag indicating success (boolean)
        """
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                break
        else:
            return 1
            
        if not editor.isModified():
            return 1
        else:
            ok, dummy = editor.saveFile()
            return ok
        
    def saveCurrentEditor(self):
        """
        Public slot to save the contents of the current editor.
        """
        aw = self.activeWindow()
        if aw:
            ok, newName = aw.saveFile()
            if ok:
                self.setEditorName(aw, newName)
        else:
            return

    def saveAsCurrentEditor(self):
        """
        Public slot to save the contents of the current editor to a new file.
        """
        aw = self.activeWindow()
        if aw:
            ok, newName = aw.saveFileAs()
            if ok:
                self.setEditorName(aw, newName)
        else:
            return

    def saveAllEditors(self):
        """
        Public slot to save the contents of all editors.
        """
        for editor in self.editors:
            ok, newName = editor.saveFile()
            if ok:
                self.setEditorName(editor, newName)

    def saveCurrentEditorToProject(self):
        """
        Public slot to save the contents of the current editor to the current project.
        """
        pro = self.ui.getProject()
        path = pro.ppath
        aw = self.activeWindow()
        if aw:
            ok, newName = aw.saveFileAs(path)
            if ok:
                self.setEditorName(aw, newName)
                pro.appendFile(newName)
        else:
            return
        
    def newEditor(self):
        """
        Public slot to generate a new empty editor.
        """
        editor = Editor(self.dbs, None, self)
        self.editors.append(editor)
        self.connect(editor, PYSIGNAL('modificationStatusChanged'),
            self.handleModificationStatusChanged)
        self.connect(editor, PYSIGNAL('cursorChanged'), self.handleCursorChanged)
        self.connect(editor, PYSIGNAL('editorSaved'), self.handleEditorSaved)
        self.connect(editor, PYSIGNAL('breakpointToggled'), self.handleBreakpointToggled)
        self.connect(editor, PYSIGNAL('bookmarkToggled'), self.handleBookmarkToggled)
        self.addView(editor, None)
        self.handleEditorOpened()
        self.checkActions(editor)
        self.emit(PYSIGNAL('editorOpened'), (None,))
        
    def printCurrentEditor(self):
        """
        Public slot to print the contents of the current editor.
        """
        aw = self.activeWindow()
        if aw:
            aw.printFile()
        else:
            return

    def printCurrentEditorSel(self):
        """
        Public slot to print the selection of the current editor.
        """
        aw = self.activeWindow()
        if aw:
            aw.printSelection()
        else:
            return

    def handleShowRecentMenu(self):
        """
        Private method to set up recent files menu.
        """
        idx = 0
        self.recentMenu.clear()
        
        for rp in self.recent:
            id = self.recentMenu.insertItem('&%d. %s' % (idx+1, str(rp)),
                                            self.handleOpenRecent)
            self.recentMenu.setItemParameter(id,idx)
            
            idx = idx + 1
            
        self.recentMenu.insertSeparator()
        self.recentMenu.insertItem(self.trUtf8('Clear'), self.handleClearRecent)
        
    def handleOpenRecent(self, idx):
        """
        Private method to open a file from the list of rencently opened files.
        
        @param idx index of the selected entry (int)
        """
        self.handlePythonFile(str(self.recent[idx]))
        
    def handleClearRecent(self):
        """
        Private method to clear the recent files menu.
        """
        self.recent = QStringList()
        
    def handleNewProject(self):
        """
        Public slot to handle the NewProject signal.
        """
        self.saveToProjectAct.setEnabled(1)
        
    def handleProjectOpened(self):
        """
        Public slot to handle the projectOpened signal.
        """
        self.saveToProjectAct.setEnabled(1)
        
    def handleProjectClosed(self):
        """
        Public slot to handle the projectClosed signal.
        """
        self.saveToProjectAct.setEnabled(0)
        
    def enableEditorsCheckFocusIn(self, enabled):
        """
        Public method to set a flag enabling the editors to perform focus in checks.
        
        @param enabled flag indicating focus in checks should be performed (boolean)
        """
        self.editorsCheckFocusIn = enabled
        
    def editorsCheckFocusInEnabled(self):
        """
        Public method returning the flag indicating editors should perform focus in checks.
        
        @return flag indicating focus in checks should be performed (boolean)
        """
        return self.editorsCheckFocusIn

    ##################################################################
    ## Below are the action methods for the edit menu
    ##################################################################
    
    def handleEditUndo(self):
        """
        Private method to handle the undo action.
        """
        self.activeWindow().undo()
        
    def handleEditRedo(self):
        """
        Private method to handle the redo action.
        """
        self.activeWindow().redo()
        
    def handleEditRevert(self):
        """
        Private method to handle the revert action.
        """
        self.activeWindow().revertToUnmodified()
        
    def handleEditCut(self):
        """
        Private method to handle the cut action.
        """
        self.activeWindow().cut()
        
    def handleEditCopy(self):
        """
        Private method to handle the copy action.
        """
        if qApp.focusWidget() == self.ui.shell:
            self.ui.shell.copy()
        else:
            self.activeWindow().copy()
        
    def handleEditPaste(self):
        """
        Private method to handle the paste action.
        """
        if qApp.focusWidget() == self.ui.shell:
            self.ui.shell.paste()
        else:
            self.activeWindow().paste()
        
    def handleEditDelete(self):
        """
        Private method to handle the delete action.
        """
        self.activeWindow().clear()
        
    def handleEditIndent(self):
        """
        Private method to handle the indent action.
        """
        self.activeWindow().indentLineOrSelection()
        
    def handleEditUnindent(self):
        """
        Private method to handle the unindent action.
        """
        self.activeWindow().unindentLineOrSelection()
        
    def handleEditComment(self):
        """
        Private method to handle the comment action.
        """
        self.activeWindow().commentLineOrSelection()
        
    def handleEditUncomment(self):
        """
        Private method to handle the uncomment action.
        """
        self.activeWindow().uncommentLineOrSelection()
        
    def handleEditStreamComment(self):
        """
        Private method to handle the stream comment action.
        """
        self.activeWindow().streamCommentLineOrSelection()
        
    def handleEditBoxComment(self):
        """
        Private method to handle the box comment action.
        """
        self.activeWindow().boxCommentLineOrSelection()
        
    def handleEditSelectBrace(self):
        """
        Private method to handle the select to brace action.
        """
        self.activeWindow().selectToMatchingBrace()
        
    def handleEditSelectAll(self):
        """
        Private method to handle the select all action.
        """
        self.activeWindow().selectAll(1)
        
    def handleEditDeselectAll(self):
        """
        Private method to handle the select all action.
        """
        self.activeWindow().selectAll(0)
        
    def handleConvertEOL(self):
        """
        Private method to handle the convert line end characters action.
        """
        aw = self.activeWindow()
        aw.convertEols(aw.eolMode())
        
    def handleEditAutoComplete(self):
        """
        Private method to handle the autocomplete action.
        """
        aw = self.activeWindow()
        aw.autoComplete()
        
    ##################################################################
    ## Below are the action and utility methods for the search menu
    ##################################################################

    def getWord(self, text, index):
        """
        Private method to get the word at a position.
        
        @param text text to look at (string or QString)
        @param index position to look at (int)
        @return the word at that position
        """
        re = QRegExp('[^\w_]')
        start = text.findRev(re, index) + 1
        end = text.find(re, index)
        if end > start:
            word = text.mid(start, end-start)
        else:
            word = QString('')
        return word
        
    def textForFind(self):
        """
        Private method to determine the selection or the current word for the next find operation.
        
        @return selection or current word (QString)
        """
        aw = self.activeWindow()
        if aw is None:
            return ''
            
        if aw.hasSelectedText():
            text = aw.selectedText()
            if text.contains('\r') or text.contains('\n'):
                # the selection contains at least a newline, it is
                # unlikely to be the expression to search for
                return ''
                
            return text
            
        # no selected text, determine the word at the current position
        line, index = aw.getCursorPosition()
        return self.getWord(aw.text(line), index)
        
    def getSRHistory(self, key):
        """
        Private method to get the search or replace history list.
        
        @param key list to return (must be 'search' or 'replace')
        @return the requested history list (QStringList)
        """
        return self.srHistory[key]
        
    def handleSearch(self):
        """
        Private method to handle the search action.
        """
        self.searchDlg.showFind(self.textForFind())
        
    def handleReplace(self):
        """
        Private method to handle the replace action.
        """
        self.replaceDlg.showReplace(self.textForFind())
        
    def handleGoto(self):
        """
        Private method to handle the goto action.
        """
        aw = self.activeWindow()
        dlg = GotoDialog(self.ui, None, 1)
        dlg.setMaxValue(aw.lines())
        if dlg.exec_loop() == QDialog.Accepted:
            aw.gotoLine(dlg.getLinenumber())
        
    def handleGotoBrace(self):
        """
        Private method to handle the goto brace action.
        """
        self.activeWindow().moveToMatchingBrace()
        
    def handleSearchFiles(self):
        """
        Private method to handle the search in files action.
        """
        self.ui.findFilesDialog.show(self.textForFind())
        self.ui.findFilesDialog.raiseW()
        
    ##################################################################
    ## Below are the action methods for the view menu
    ##################################################################
    
    def handleZoomIn(self):
        """
        Private method to handle the zoom in action.
        """
        self.activeWindow().zoomIn()
        
    def handleZoomOut(self):
        """
        Private method to handle the zoom out action.
        """
        self.activeWindow().zoomOut()
        
    def handleZoom(self):
        """
        Private method to handle the zoom action.
        """
        aw = self.activeWindow()
        dlg = ZoomDialog(aw.getZoom(), self.ui, None, 1)
        if dlg.exec_loop() == QDialog.Accepted:
            aw.zoomTo(dlg.getZoomSize())
            
    def handleToggleAll(self):
        """
        Private method to handle the toggle all folds action.
        """
        self.activeWindow().foldAll()
        
    def handleToggleCurrent(self):
        """
        Private method to handle the toggle current fold action.
        """
        aw = self.activeWindow()
        line, index = aw.getCursorPosition()
        aw.foldLine(line)
        
    def handleSplitView(self):
        """
        Private method to handle the split view action.
        """
        self.addSplit()
        
    def handleSplitOrientation(self):
        """
        Private method to handle the split orientation action.
        """
        if self.splitOrientationAct.isOn():
            self.setSplitOrientation(QSplitter.Horizontal)
            self.splitViewAct.setIconSet(\
                QIconSet(UI.PixmapCache.getPixmap("splitHorizontal")))
        else:
            self.setSplitOrientation(QSplitter.Vertical)
            self.splitViewAct.setIconSet(\
                QIconSet(UI.PixmapCache.getPixmap("splitVertical")))
    
    ##################################################################
    ## Below are the action methods for the macro menu
    ##################################################################
    
    def handleMacroStartRecording(self):
        """
        Private method to handle the start macro recording action.
        """
        self.activeWindow().handleStartMacroRecording()
        
    def handleMacroStopRecording(self):
        """
        Private method to handle the stop macro recording action.
        """
        self.activeWindow().handleStopMacroRecording()
        
    def handleMacroRun(self):
        """
        Private method to handle the run macro action.
        """
        self.activeWindow().handleRunMacro()
        
    def handleMacroDelete(self):
        """
        Private method to handle the delete macro action.
        """
        self.activeWindow().handleDeleteMacro()
        
    def handleMacroLoad(self):
        """
        Private method to handle the load macro action.
        """
        self.activeWindow().handleLoadMacro()
        
    def handleMacroSave(self):
        """
        Private method to handle the save macro action.
        """
        self.activeWindow().handleSaveMacro()
    
    ##################################################################
    ## Below are the action methods for the bookmarks menu
    ##################################################################
    
    def handleToggleBookmark(self):
        """
        Private method to handle the toggle bookmark action.
        """
        self.activeWindow().handleToggleBookmark()
        
    def handleNextBookmark(self):
        """
        Private method to handle the next bookmark action.
        """
        self.activeWindow().handleNextBookmark()
    
    def handlePreviousBookmark(self):
        """
        Private method to handle the previous bookmark action.
        """
        self.activeWindow().handlePreviousBookmark()
    
    def handleClearAllBookmarks(self):
        """
        Private method to handle the clear all bookmarks action.
        """
        for editor in self.editors:
            editor.handleClearBookmarks()
    
    def handleShowBookmarksMenu(self):
        """
        Private method to handle the show bookmarks menu signal.
        """
        self.bookmarks = {}
        self.bookmarksMenu.clear()
        
        filenames = self.getOpenFilenames()
        filenames.sort()
        for filename in filenames:
            editor = self.getOpenEditor(filename)
            for bookmark in editor.getBookmarks():
                if len(filename) > 50:
                    dots = "..."
                else:
                    dots = ""
                id = self.bookmarksMenu.insertItem(\
                        "%s%s : %d" % (dots, filename[-50:], bookmark))
                self.bookmarks[id] = (filename, bookmark)
    
    def handleBookmarkSelected(self, id):
        """
        Private method to handle the bookmark selected signal.
        
        @param id index of the selected menu entry
                This acts as an index into the list of bookmarks
                that was created, when the bookmarks menu was built.
        """
        self.displayPythonFile(self.bookmarks[id][0], self.bookmarks[id][1])
        
    def handleBookmarkToggled(self, editor):
        """
        Private slot to handle the breakpointToggled signal.
        
        It checks some bookmark actions and reemits the signal.
        
        @param editor editor that sent the signal
        """
        if editor.hasBookmarks():
            self.bookmarkNextAct.setEnabled(1)
            self.bookmarkPreviousAct.setEnabled(1)
            self.bookmarkClearAct.setEnabled(1)
        else:
            self.bookmarkNextAct.setEnabled(0)
            self.bookmarkPreviousAct.setEnabled(0)
            self.bookmarkClearAct.setEnabled(0)
        self.emit(PYSIGNAL('bookmarkToggled'), (editor,))
    
    ##################################################################
    ## Below are general utility methods
    ##################################################################
    
    def handleResetUI(self):
        """
        Public slot to handle the resetUI signal.
        """
        editor = self.activeWindow()
        if editor is None:
            self.setSbFile()
        else:
            line, pos = editor.getCursorPosition()
            self.setSbFile(editor.getFileName(), line+1, pos)
        
    def closeViewManager(self):
        """
        Public method to shutdown the viewmanager. 
        
        If it cannot close all editor windows, it aborts the shutdown process.
        
        @return flag indicating success (boolean)
        """
        self.handleCloseAll()
            
        # save the list of recently opened projects
        ok = Preferences.Prefs.settings.writeEntry('/eric3/Recent/Sources', self.recent)
        
        if len(self.editors):
            return 0
        else:
            return 1

    def handleLastEditorClosed(self):
        """
        Private slot to handle the lastEditorClosed signal.
        """
        self.closeActGrp.setEnabled(0)
        self.saveActGrp.setEnabled(0)
        self.printAct.setEnabled(0)
        self.printSelAct.setEnabled(0)
        self.editActGrp.setEnabled(0)
        self.searchActGrp.setEnabled(0)
        self.viewActGrp.setEnabled(0)
        self.viewFoldActGrp.setEnabled(0)
        self.unhighlightAct.setEnabled(0)
        self.splitViewAct.setEnabled(0)
        self.splitOrientationAct.setEnabled(0)
        self.macroActGrp.setEnabled(0)
        self.bookmarkActGrp.setEnabled(0)
        self.setSbFile()
        
        # reinitialize the central store for api information (used by
        # autocompletion and calltips)
        for key in self.apis.keys():
            self.apis[key] = None
            
        # remove all split views, if this is supported
        if self.canSplit():
            while self.removeSplit(): pass
        
    def handleEditorOpened(self):
        """
        Private slot to handle the editorOpened signal.
        """
        self.closeActGrp.setEnabled(1)
        self.saveActGrp.setEnabled(1)
        self.printAct.setEnabled(1)
        self.printSelAct.setEnabled(1)
        self.editActGrp.setEnabled(1)
        self.searchActGrp.setEnabled(1)
        self.viewActGrp.setEnabled(1)
        self.viewFoldActGrp.setEnabled(1)
        self.unhighlightAct.setEnabled(1)
        self.splitViewAct.setEnabled(1)
        self.splitOrientationAct.setEnabled(1)
        self.macroActGrp.setEnabled(1)
        self.bookmarkActGrp.setEnabled(1)
        
    def checkActions(self, editor, setSb=1):
        """
        Private slot to check some actions for their enable/disable status and set the statusbar info.
        
        @param editor editor window
        @param setSb flag indicating an update of the status bar is wanted (boolean)
        """
        if editor is not None:
            self.saveAct.setEnabled(editor.isModified())
            self.revertAct.setEnabled(editor.isModified())
            
            lex = editor.getLexer()
            if lex is not None:
                self.commentAct.setEnabled(lex.canBlockComment())
                self.uncommentAct.setEnabled(lex.canBlockComment())
                self.streamCommentAct.setEnabled(lex.canStreamComment())
                self.boxCommentAct.setEnabled(lex.canBoxComment())
            else:
                self.commentAct.setEnabled(0)
                self.uncommentAct.setEnabled(0)
                self.streamCommentAct.setEnabled(0)
                self.boxCommentAct.setEnabled(0)
            
            if editor.hasBookmarks():
                self.bookmarkNextAct.setEnabled(1)
                self.bookmarkPreviousAct.setEnabled(1)
                self.bookmarkClearAct.setEnabled(1)
            else:
                self.bookmarkNextAct.setEnabled(0)
                self.bookmarkPreviousAct.setEnabled(0)
                self.bookmarkClearAct.setEnabled(0)
            
            if setSb:
                line, pos = editor.getCursorPosition()
                self.setSbFile(editor.getFileName(), line+1, pos)
                
            self.emit(PYSIGNAL('checkActions'), (editor,))
        
    def handlePreferencesChanged(self):
        """
        Public slot to handle the preferencesChanged signal.
        
        This method performs the following actions
            <ul>
            <li>reread the colours for the syntax highlighting</li>
            <li>reloads the already created API objetcs</li>
            <li>changes in viewmanager type are activated on an
              application restart.</li>
            </ul>
        """
        # reload api information
        for language, api in self.apis.items():
            if api is not None:
                apifiles = Preferences.getEditorAPI(language)
                if len(apifiles):
                    api.clear()
                    for apifile in apifiles:
                        api.load(apifile)
                else:
                    self.apis[language] = None
                    
        # reload editor settings
        for editor in self.editors:
            editor.readSettings()
            
    def handleEditorSaved(self, fn):
        """
        Public slot to handle the editorSaved signal.
        
        It simply reemits the signal.
        
        @param fn filename of the saved editor
        """
        self.emit(PYSIGNAL('editorSaved'), (fn,))
        
    def handleCursorChanged(self, fn, line, pos):
        """
        Private slot to handle the cursorChanged signal. 
        
        It emits the signal cursorChanged with parameter editor.
        
        @param fn filename (string)
        @param line line number of the cursor (int)
        @param pos position in line of the cursor (int)
        """
        self.setSbFile(fn, line, pos)
        self.emit(PYSIGNAL('cursorChanged'), (self.getOpenEditor(fn),))
        
    def handleBreakpointToggled(self, editor):
        """
        Private slot to handle the breakpointToggled signal.
        
        It simply reemits the signal.
        
        @param editor editor that sent the signal
        """
        self.emit(PYSIGNAL('breakpointToggled'), (editor,))
        
    def getAPIs(self, language):
        """
        Public method to get an apis object for autocompletion/calltips
        
        This method creates and loads the QextScintillaAPIs object
        dynamically upon request. This saves memory for languages,
        that might not be needed at the moment.
        
        @param language the language of the requested api object (string)
        @return the apis object (QextScintillaAPIs)
        """
        try:
            if self.apis[language] is None:
                # create the api object
                apifiles = Preferences.getEditorAPI(language)
                if len(apifiles):
                    self.apis[language] = QextScintillaAPIs()
                    for apifile in apifiles:
                        self.apis[language].load(apifile)
            return self.apis[language]
        except KeyError:
            return None
            
    def getProject(self):
        """
        Public method to get a reference to the Project object.
        
        @return Reference to the Project object (Project.Project)
        """
        return self.ui.getProject()
        
    def getActions(self, type):
        """
        Public method to get a list of all actions.
        
        @param type string denoting the action set to get.
                It must be one of "edit", "file", "search",
                "view" or "window"
        @return list of all actions (list of QAction)
        """
        try:
            exec 'actionList = self.%sActions[:]' % type
        except:
            actionList = []
                
        return actionList
        
    def editorCommand(self, cmd):
        """
        Private method to execute send an editor command to the active window.
        
        @param cmd the scintilla command to be sent
        """
        self.activeWindow().SendScintilla(cmd)
        
    ##################################################################
    ## Below are protected utility methods
    ##################################################################
    
    def _getOpenStartDir(self):
        """
        Protected method to return the starting directory for a file open dialog. 
        
        The appropriate starting directory is calculated
        using the following search order, until a match is found:<br />
            1: Directory of currently active editor<br />
            2: Directory of currently active Project<br />
            3: CWD

        @return String name of directory to start or None
        """
        # if we have an active source, return its path
        if self.activeWindow() is not None and \
           self.activeWindow().getFileName():
            return os.path.dirname(self.activeWindow().getFileName())
            
        # ok, try if there is an active project and return its path
        elif self.getProject().isOpen():
            return self.getProject().ppath
            
        else:
            # None will cause open dialog to start with cwd
            return None
