# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the svn log command process.
"""
import os

from qt import *

from LogForm import LogForm
from DiffDialog import SvnDiffDialog

class SvnLogDialog(LogForm):
    """
    Module implementing a dialog to show the output of the svn log command process.
    
    The dialog is nonmodal. Clicking a link in the upper text pane shows 
    a diff of the versions.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        LogForm.__init__(self, parent)
        
        self.saveButton.hide()
        
        QWhatsThis.add(self.contents,self.tr("<b>Subversion Log</b>\n"
            "<p>This shows the output of the svn log command."
            " By clicking on the links you may show the difference"
            " between versions.</p>"
        ))
        QWhatsThis.add(self.errors,self.tr("<b>Subversion log errors</b>\n"
            "<p>This shows possible error messages of the svn log"
            " command.</p>"
        ))
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess(self)
        self.vcs = vcs
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
        self.disconnect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.contents, SLOT('setSource(const QString&)'))
        self.connect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.handleLinkClicked)
            
        self.rx_sep = QRegExp('\\-+')
        self.rx_sep2 = QRegExp('=+')
        self.rx_rev = QRegExp('rev ([0-9]+):  ([^|]*) \| ([^|]*) \| ([0-9]+) lines?')
        # "rev" followed by one or more decimals followed by a colon followed
        # anything up to " | " (twice) followed by one or more decimals followed
        # by "line" and an optional "s"
        self.rx_flags = QRegExp('   ([ADM])( .*)')
        # three blanks followed by A or D or M
        self.rx_changed = QRegExp('Changed .*')
        
        self.flags = {
            'A' : self.trUtf8('Added'),
            'D' : self.trUtf8('Deleted'),
            'M' : self.trUtf8('Modified')
        }
        
        self.revisions = QStringList()
        self.revString = self.trUtf8('revision')
        
        self.buf = QStringList()    # buffer for stdout
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn):
        """
        Public slot to start the cvs log command.
        
        @param fn filename to be diffed (string)
        """
        self.filename = fn
        self.dname, self.fname = self.vcs.splitPath(fn)
        
        self.process.kill()
        
        self.process.clearArguments()
        self.process.addArgument('svn')
        self.process.addArgument('log')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.vcs.addArguments(self.process, self.vcs.options['log'])
        if os.path.isdir(fn):
            self.process.addArgument('--verbose')
        try:
            if self.vcs.svnGetReposName(self.dname).startswith('http'):
                self.vcs.addArguments(self.process, self.vcs.authData())
            self.setActiveWindow()
            self.raiseW()
        except:
            pass
        self.process.addArgument(self.fname)
        self.process.setWorkingDirectory(QDir(self.dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('Subversion Log %1').arg(self.filename))
        
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        After the process has exited, diff links are inserted into the contents pane.
        """
        self.contents.clear()
        lvers = 1
        for s in self.buf:
            if self.rx_rev.exactMatch(s):
                ver = self.rx_rev.cap(1)
                author = self.rx_rev.cap(2)
                date = self.rx_rev.cap(3)
                # number of lines is ignored
                
                dstr = QString('<b>%1 %2</b>').arg(self.revString).arg(ver)
                try:
                    lv = self.revisions[lvers]
                    lvers += 1
                except IndexError:
                    lv = None
                
                if lv is not None:
                    url = QUrl()
                    url.setProtocol("file")
                    url.setFileName(self.filename)
                    query = QString()
                    query.append(lv).append('_').append(ver)
                    url.setQuery(query)
                    dstr.append(' [<a href="')\
                        .append(url.toString())\
                        .append('">')\
                        .append(self.trUtf8('diff to %1').arg(lv))\
                        .append('</a>]')
                self.contents.append(dstr)
                
                dstr = self.trUtf8('<i>author: %1</i>').arg(author)
                self.contents.append(dstr)
                
                dstr = self.trUtf8('<i>date: %1</i>').arg(date)
                self.contents.append(dstr)
                
            elif self.rx_sep.exactMatch(s) or self.rx_sep2.exactMatch(s):
                self.contents.append('<hr>')
                
            elif self.rx_flags.exactMatch(s):
                dstr = QString(self.flags[str(self.rx_flags.cap(1))])
                dstr.append(self.rx_flags.cap(2))
                self.contents.append(dstr)
                
            elif self.rx_changed.exactMatch(s):
                dstr = QString('<br>%1').arg(s)
                self.contents.append(dstr)
                
            else:
                if s.isEmpty():
                    s = QString('<br>')
                self.contents.append(s)
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            self.buf.append(s)
            if self.rx_rev.exactMatch(s):
                ver = self.rx_rev.cap(1)
                
                # save revision number for later use
                self.revisions.append(ver)
                
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal. 
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        while self.process.canReadLineStderr():
            s = self.process.readLineStderr()
            self.errors.append(s)
            
    def handleLinkClicked(self, link):
        """
        Private slot to handle the linkClicked signal of the contents pane.
        
        @param link the link that was clicked (QString)
        """
        self.contents.setSource('')
        link = QUrl(link)
        filename = link.path()
        ver = link.query()
        v1 = ver.section('_', 0, 0)
        v2 = ver.section('_', 1, 1)
        if v1.isEmpty() or v2.isEmpty():
            return
            
        self.diff = SvnDiffDialog(self.vcs)
        self.diff.show()
        self.diff.start(filename, [v1, v2])
