# -*- coding: utf-8 -*-

# Copyright (c) 2004 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing functions dealing with keyboard shortcuts.

<b>Note</b>: These functions are imported into the Preferences
package and should be called like Preferences.readShortcuts(...).
"""

import os
import time

from qt import *

from ThirdParty.XMLFilter import parseStream

from Preferences import Prefs, getUser, syncPreferences
from ShortcutsHandler import ShortcutsHandler

def readShortcuts(ui, prefClass = Prefs):
    """
    Module function to read the keyboard shortcuts for the defined QActions.
    
    @param ui reference to the main widget (UI.UserInterface)
    @param prefClass preferences class used as the storage area
    """
    for act in ui.getProject().getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Project/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/General/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getDebugger().getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Debug/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getWizards().getActions():
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Wizards/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('edit'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Edit/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('file'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/File/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('search'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Search/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('view'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/View/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('macro'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Macro/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    for act in ui.getViewManager().getActions('bookmark'):
        accel, ok = prefClass.settings.readEntry(\
            QString("/eric3/Shortcuts/Bookmarks/%1").arg(act.text()))
        if ok:
            act.setAccel(QKeySequence(accel))
            
    actions = ui.getViewManager().getActions('window')
    if actions:
        for act in actions:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Window/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
                
    ref = ui.getRefactoring()
    if ref is not None:
        actions = ref.getActions()
        for act in actions:
            accel, ok = prefClass.settings.readEntry(\
                QString("/eric3/Shortcuts/Refactoring/%1").arg(act.text()))
            if ok:
                act.setAccel(QKeySequence(accel))
    
def saveShortcuts(prefClass = Prefs):
    """
    Module function to write the keyboard shortcuts for the defined QActions.
    
    @param prefClass preferences class used as the storage area
    """
    ui = qApp.mainWidget()
    
    # step 1: clear all previously saved shortcuts
    subkeyList = prefClass.settings.subkeyList("/eric3/Shortcuts")
    for subkey in subkeyList:
        entryList = prefClass.settings.entryList(QString("/eric3/Shortcuts/%1").arg(subkey))
        for entry in entryList:
            prefClass.settings.removeEntry(\
                QString("/eric3/Shortcuts/%1/%2").arg(subkey).arg(entry))
                
    # step 2: save the various shortcuts
    for act in ui.getProject().getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Project/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/General/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getDebugger().getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Debug/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getWizards().getActions():
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Wizards/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getViewManager().getActions('edit'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Edit/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getViewManager().getActions('file'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/File/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getViewManager().getActions('search'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Search/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getViewManager().getActions('view'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/View/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getViewManager().getActions('macro'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Macro/%1").arg(act.text()), QString(act.accel()))
        
    for act in ui.getViewManager().getActions('bookmark'):
        prefClass.settings.writeEntry(\
            QString("/eric3/Shortcuts/Bookmarks/%1").arg(act.text()), QString(act.accel()))
        
    actions = ui.getViewManager().getActions('window')
    if actions:
        for act in actions:
            prefClass.settings.writeEntry(\
                QString("/eric3/Shortcuts/Window/%1").arg(act.text()), QString(act.accel()))
                
    ref = ui.getRefactoring()
    if ref is not None:
        actions = ref.getActions()
        for act in actions:
            prefClass.settings.writeEntry(\
                QString("/eric3/Shortcuts/Refactoring/%1").arg(act.text()), QString(act.accel()))

def _write(f, s, newline=1):
    """
    Private method used to do the real write operation.
    
    @param f open file object of the XML file
    @param s string to be written to the XML file
    @param newline flag indicating a linebreak
    """
    f.write("%s%s" % (s.encode('utf-8'), 
        newline and os.linesep or ""))

def exportShortcuts(fn):
    """
    Module function to export the keyboard shortcuts for the defined QActions.
    
    @param fn filename of the export file (string)
    @return flag indicating success
    """
    ui = qApp.mainWidget()
    email = getUser('Email')
    
    try:
        if fn.lower().endswith("e3kz"):
            try:
                import gzip
            except ImportError:
                QMessageBox.critical(None,
                    qApp.translate("Shortcuts", "Export Keyboard Shortcuts"),
                    qApp.translate("Shortcuts", """Compressed keyboard shortcut files not supported. The compression library is missing."""),
                    qApp.translate("Shortcuts", "&Abort"),
                    None,
                    None,
                    0, -1)
                return 0
            f = gzip.open(fn, "wb")
        else:
            f = open(fn, "wb")
        
        # generate a new XML document
        _write(f, "<?xml version='1.0' encoding='UTF-8'?>")
        
        # add some generation comments
        _write(f, "<!--Eric3 keyboard shortcuts-->")
        _write(f, "<!--Saved: %s-->" % time.strftime('%Y-%m-%d, %H:%M:%S'))
        _write(f, "<!--Author: %s-->" % email)
        
        # add the main tag
        _write(f, "<Shortcuts>")
        
        for act in ui.getProject().getActions():
            _write(f, '  <Shortcut category="Project">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getActions():
            _write(f, '  <Shortcut category="General">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getDebugger().getActions():
            _write(f, '  <Shortcut category="Debug">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getWizards().getActions():
            _write(f, '  <Shortcut category="Wizards">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getViewManager().getActions('edit'):
            _write(f, '  <Shortcut category="Edit">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getViewManager().getActions('file'):
            _write(f, '  <Shortcut category="File">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getViewManager().getActions('search'):
            _write(f, '  <Shortcut category="Search">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getViewManager().getActions('view'):
            _write(f, '  <Shortcut category="View">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getViewManager().getActions('macro'):
            _write(f, '  <Shortcut category="Macro">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        for act in ui.getViewManager().getActions('bookmark'):
            _write(f, '  <Shortcut category="Bookmarks">')
            _write(f, '    <Name>%s</Name>' % act.text())
            _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
            _write(f, '  </Shortcut>')
        
        actions = ui.getViewManager().getActions('window')
        if actions:
            for act in actions:
                _write(f, '  <Shortcut category="Window">')
                _write(f, '    <Name>%s</Name>' % act.text())
                _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
                _write(f, '  </Shortcut>')
        
        ref = ui.getRefactoring()
        if ref is not None:
            actions = ref.getActions()
            for act in actions:
                _write(f, '  <Shortcut category="Refactoring">')
                _write(f, '    <Name>%s</Name>' % act.text())
                _write(f, '    <Accel>%s</Accel>' % QString(act.accel()))
                _write(f, '  </Shortcut>')
    
        # add the main end tag
        _write(f, "</Shortcuts>")
        
        f.close()
        return 1
    except IOError:
        return 0

def importShortcuts(fn):
    """
    Module function to import the keyboard shortcuts for the defined QActions.
    
    @param fn filename of the import file (string)
    @return flag indicating success
    """
    fn = str(fn)
    
    handler = ShortcutsHandler()
    
    try:
        if fn.lower().endswith("e3kz"):
            try:
                import gzip
            except ImportError:
                QMessageBox.critical(None,
                    qApp.translate("Shortcuts", "Import Keyboard Shortcuts"),
                    qApp.translate("Shortcuts", """Compressed keyboard shortcut files not supported. The compression library is missing."""),
                    qApp.translate("Shortcuts", "&Abort"),
                    None,
                    None,
                    0, -1)
                return 0
            f = gzip.open(fn, "rb")
        else:
            f = open(fn, "rb")
        try:
            parseStream(f, handler)
        finally:
            f.close()
    except IOError:
        return 0
        
    shortcuts = handler.getShortcuts()
    
    ui = qApp.mainWidget()
    
    if shortcuts.has_key("Project"):
        actions = ui.getProject().getActions()
        sdict = shortcuts["Project"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("General"):
        actions = ui.getActions()
        sdict = shortcuts["General"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Debug"):
        actions = ui.getDebugger().getActions()
        sdict = shortcuts["Debug"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Wizards"):
        actions = ui.getWizards().getActions()
        sdict = shortcuts["Wizards"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Edit"):
        actions = ui.getViewManager().getActions('edit')
        sdict = shortcuts["Edit"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("File"):
        actions = ui.getViewManager().getActions('file')
        sdict = shortcuts["File"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Search"):
        actions = ui.getViewManager().getActions('search')
        sdict = shortcuts["Search"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("View"):
        actions = ui.getViewManager().getActions('view')
        sdict = shortcuts["View"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Macro"):
        actions = ui.getViewManager().getActions('macro')
        sdict = shortcuts["Macro"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Bookmarks"):
        actions = ui.getViewManager().getActions('bookmark')
        sdict = shortcuts["Bookmarks"]
        for act in actions:
            try:
                accel = sdict[str(act.text())]
                act.setAccel(QKeySequence(accel))
            except KeyError:
                pass
    
    if shortcuts.has_key("Window"):
        actions = ui.getViewManager().getActions('window')
        if actions:
            sdict = shortcuts["Window"]
            for act in actions:
                try:
                    accel = sdict[str(act.text())]
                    act.setAccel(QKeySequence(accel))
                except KeyError:
                    pass
    
    if shortcuts.has_key("Refactoring"):
        ref = ui.getRefactoring()
        if ref is not None:
            actions = ref.getActions()
            sdict = shortcuts["Refactoring"]
            for act in actions:
                try:
                    accel = sdict[str(act.text())]
                    act.setAccel(QKeySequence(accel))
                except KeyError:
                    pass
    
    saveShortcuts()
    syncPreferences()
    
    return 1
