# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the builtin documentation generator.

The different parts of the module document are assembled from the parsed
Python file. The appearance is determined by several templates defined within
this module.
"""

import sys
import re

from TemplatesListsStyle import *

_signal = re.compile(r"""
    ^@signal [ \t]+ 
    (?P<SignalName1>
        [a-zA-Z_] \w* [ \t]* \( [^)]* \)
    )
    [ \t]* (?P<SignalDescription1> .*)
|
    ^@signal [ \t]+ 
    (?P<SignalName2>
        [a-zA-Z_] \w*
    )
    [ \t]+ (?P<SignalDescription2> .*)
""", re.VERBOSE | re.DOTALL | re.MULTILINE).search

class TagError(Exception):
    """
    Exception class raised, if an invalid documentation tag was found.
    """
    def __init__(self, args=None):
        """
        Constructor
        
        @param args The arguments.
        """
        self.args = args
        
class ModuleDocument:
    """
    Class implementing the builtin documentation generator.
    """
    def __init__(self, module):
        """
        Constructor
        
        @param module The information of the parsed Python file.
        """
        self.module = module
        self.empty = 1
        
    def isEmpty(self):
        """
        Method to determine, if the module contains any classes or functions.
        
        @return Flag indicating an empty module (i.e. __init__.py without
            any contents)
        """
        return self.empty
        
    def name(self):
        """
        Method used to get the module name.
        
        @return The name of the module. (string)
        """
        return self.module.name
        
    def description(self):
        """
        Method used to get the description of the module.
        
        @return The description of the module. (string)
        """
        return self.formatDescription(self.module.description)
        
    def shortDescription(self):
        """
        Method used to get the short description of the module.
        
        The short description is just the first line of the modules
        description.
        
        @return The short description of the module. (string)
        """
        return self.getShortDescription(self.module.description)
        
    def genDocument(self):
        """
        Method to generate the source code documentation.
        
        @return The source code documentation. (string)
        """
        doc = headerTemplate % {'Title' : self.module.name} + \
              self.genModuleSection() + \
              footerTemplate
        return doc
        
    def genModuleSection(self):
        """
        Private method to generate the body of the document.
        
        @return The body of the document. (string)
        """
        classList = self.genClassListSection()
        functionList = self.genFunctionListSection()
        try:
            modBody = moduleTemplate % { \
                'Module' : self.module.name,
                'ModuleDescription' : self.formatDescription(self.module.description),
                'ClassList' : classList,
                'FunctionList' : functionList,
            }
        except TagError, e:
            sys.stderr.write("Error in tags of description of module %s.\n" % \
                self.module.name)
            sys.stderr.write("%s\n" % e.args)
            return ""
            
        classesSection = self.genClassesSection()
        functionsSection = self.genFunctionsSection()
        return "%s%s%s" % (modBody, classesSection, functionsSection)
        
    def genListSection(self, names, dict, className=''):
        """
        Private method to generate a list section of the document.
        
        @param names The names to appear in the list. (list of strings)
        @param dict A dictionary containing all relevant information.
        @param className The class name containing the names.
        @return The list section. (string)
        """
        lst = []
        for name in names:
            lst.append(listEntryTemplate % { \
                'Link' : "%s%s" % (className, name),
                'Name' : name,
                'Description' : self.getShortDescription(dict[name].description),
                'Deprecated' : self.checkDeprecated(dict[name].description) and listEntryDeprecatedTemplate or "",
            })
        return ''.join(lst)
        
    def genClassListSection(self):
        """
        Private method to generate the section listing all classes of the module.
        
        @return The classes list section. (string)
        """
        names = self.module.classes.keys()
        names.sort()
        if names:
            self.empty = 0
            s = self.genListSection(names, self.module.classes)
        else:
            s = listEntryNoneTemplate
        return listTemplate % { \
            'Entries' : s,
        }
        
    def genFunctionListSection(self):
        """
        Private method to generate the section listing all functions of the module.
        
        @return The functions list section. (string)
        """
        names = self.module.functions.keys()
        names.sort()
        if names:
            self.empty = 0
            s = self.genListSection(names, self.module.functions)
        else:
            s = listEntryNoneTemplate
        return listTemplate % { \
            'Entries' : s,
        }
        
    def genClassesSection(self):
        """
        Private method to generate the document section with details about classes.
        
        @return The classes details section. (string)
        """
        classNames = self.module.classes.keys()
        classNames.sort()
        classes = []
        for className in classNames:
            supers = self.module.classes[className].super
            if len(supers) > 0:
                supers = ', '.join(supers)
            else:
                supers = ''
                
            methList, methBodies = self.genMethodSection(className)
            
            try:
                clsBody = classTemplate % { \
                    'Class' : className,
                    'ClassSuper' : supers,
                    'ClassDescription' : self.formatDescription(self.module.classes[className].description),
                    'MethodList' : methList,
                    'MethodDetails' : methBodies,
                }
            except TagError, e:
                sys.stderr.write("Error in tags of description of class %s.\n" % \
                    className)
                sys.stderr.write("%s\n" % e.args)
                clsBody = ""
            
            classes.append(clsBody)
            
        return ''.join(classes)
        
    def genMethodsListSection(self, names, dict, className=''):
        """
        Private method to generate the methods list section of a class.
        
        @param names The names to appear in the list. (list of strings)
        @param dict A dictionary containing all relevant information.
        @param className The class name containing the names.
        @return The list section. (string)
        """
        lst = []
        try:
            lst.append(listEntryTemplate % { \
                'Link' : "%s%s" % (className, '__init__'),
                'Name' : className,
                'Description' : self.getShortDescription(dict['__init__'].description),
                'Deprecated' : self.checkDeprecated(dict['__init__'].description) and listEntryDeprecatedTemplate or "",
            })
        except KeyError:
            pass
            
        for name in names:
            lst.append(listEntryTemplate % { \
                'Link' : "%s%s" % (className, name),
                'Name' : name,
                'Description' : self.getShortDescription(dict[name].description),
                'Deprecated' : self.checkDeprecated(dict[name].description) and listEntryDeprecatedTemplate or "",
            })
        return ''.join(lst)
        
    def genMethodSection(self, className):
        """
        Private method to generate the method details section.
        
        @param classname Name of the class containing the method. (string)
        @return The method details section. (string)
        """
        methList = []
        methBodies = []
        _class = self.module.classes[className]
        methods = _class.methods.keys()
        methods.sort()
        
        # first do the constructor
        if '__init__' in methods:
            methods.remove('__init__')
            try:
                methBody = constructorTemplate % { \
                    'Class' : className,
                    'Method' : '__init__',
                    'MethodDescription' : self.formatDescription(_class.methods['__init__'].description),
                    'Params' : ', '.join(_class.methods['__init__'].parameters[1:]),
                }
            except TagError, e:
                sys.stderr.write("Error in tags of description of method %s.%s.\n" % \
                    (className, method))
                sys.stderr.write("%s\n" % e.args)
                methBody = ""
            methBodies.append(methBody)
            
        for method in methods:
            try:
                methBody = methodTemplate % { \
                    'Class' : className,
                    'Method' : method,
                    'MethodDescription' : self.formatDescription(_class.methods[method].description),
                    'Params' : ', '.join(_class.methods[method].parameters[1:]),
                }
            except TagError, e:
                sys.stderr.write("Error in tags of description of method %s.%s.\n" % \
                    (className, method))
                sys.stderr.write("%s\n" % e.args)
                methBody = ""
            methBodies.append(methBody)
            
        methList = self.genMethodsListSection(methods, _class.methods, className)
        
        return listTemplate % { \
            'Entries' : methList,
            }, ''.join(methBodies)
        
    def genFunctionsSection(self):
        """
        Private method to generate the document section with details about functions.
        
        @return The functions details section. (string)
        """
        funcBodies = []
        funcNames = self.module.functions.keys()
        funcNames.sort()
        for funcName in funcNames:
            try:
                funcBody = functionTemplate % { \
                    'Function' : funcName,
                    'FunctionDescription' : self.formatDescription(self.module.functions[funcName].description),
                    'Params' : ', '.join(self.module.functions[funcName].parameters),
                }
            except TagError, e:
                sys.stderr.write("Error in tags of description of function %s.\n" % \
                    funcName)
                sys.stderr.write("%s\n" % e.args)
                funcBody = ""
            
            funcBodies.append(funcBody)
            
        return ''.join(funcBodies)
        
    def getShortDescription(self, desc):
        """
        Private method to determine the short description of an object.
        
        The short description is just the first non empty line of the
        documentation string.
        
        @param desc The documentation string. (string)
        @return The short description. (string)
        """
        dlist = desc.splitlines()
        for desc in dlist:
            if desc:
                return desc
        else:
            return ''
            
    def checkDeprecated(self, descr):
        """
        Private method to check, if the object to be documented contains a deprecated flag.
        
        @param desc The documentation string. (string)
        @return Flag indicating the deprecation status. (boolean)
        """
        dlist = descr.splitlines()
        for desc in dlist:
            desc = desc.strip()
            if desc.startswith("@deprecated"):
                return 1
        return 0
        
    def genParagraphs(self, lines):
        """
        Private method to assemble the descriptive paragraphs of a docstring.
        
        A paragraph is made up of a number of consecutive lines without
        an intermediate empty line. Empty lines are treated a paragraph
        delimiter.
        
        @param lines A list of individual lines. (list of strings)
        @return Ready formatted paragraphs. (string)
        """
        lst = []
        linelist = []
        for line in lines:
            if line:
                linelist.append(line)
            else:
                lst.append(paragraphTemplate % { \
                    'Lines' : '\n'.join(linelist)
                })
                linelist = []
        if linelist:
            lst.append(paragraphTemplate % { \
                'Lines' : '\n'.join(linelist)
            })
        return ''.join(lst)
        
    def genDescriptionListSection(self, dictionary, template):
        """
        Private method to generate the list section of a description.
        
        @param dictionary Dictionary containing the info for the
            list section.
        @param template The template to be used for the list. (string)
        @return The list section. (string)
        """
        lst = []
        keys = dictionary.keys()
        keys.sort()
        for key in keys:
            lst.append(template % { \
                'Name' : key,
                'Description' : '\n'.join(dictionary[key]),
            })
        return ''.join(lst)
        
    def formatDescription(self, descr):
        """
        Private method to format the contents of the documentation string.
        
        @param descr The contents of the documentation string. (string)
        @exception TagError A tag doesn't have the correct number
            of arguments.
        @return The formated contents of the documentation string. (string)
        """
        paragraphs = []
        paramDict = {}
        returns = []
        exceptionDict = {}
        signalDict = {}
        deprecated = []
        lastItem = paragraphs
        inTagSection = 0
        
        dlist = descr.splitlines()
        while dlist and not dlist[0]:
            del dlist[0]
        for desc in dlist:
            desc = desc.strip()
            if desc:
                if desc.startswith("@param") or desc.startswith("@keyparam"):
                    inTagSection = 1
                    parts = desc.split(None, 2)
                    if len(parts) < 2:
                        raise TagError, "Wrong format in %s line.\n" % parts[0]
                    paramName = parts[1]
                    if parts[0] == "@keyparam":
                        paramName += '='
                    try:
                        paramDict[paramName] = [parts[2]]
                    except IndexError:
                        paramDict[paramName] = []
                    lastItem = paramDict[paramName]
                    continue
                elif desc.startswith("@return"):
                    inTagSection = 1
                    parts = desc.split(None, 1)
                    if len(parts) < 2:
                        raise TagError, "Wrong format in %s line.\n" % parts[0]
                    returns = [parts[1]]
                    lastItem = returns
                    continue
                elif desc.startswith("@exception") or \
                     desc.startswith("@throws") or \
                     desc.startswith("@raise"):
                    inTagSection = 1
                    parts = desc.split(None, 2)
                    if len(parts) < 2:
                        raise TagError, "Wrong format in %s line.\n" % parts[0]
                    excName = parts[1]
                    try:
                        exceptionDict[excName] = [parts[2]]
                    except IndexError:
                        exceptionDict[excName] = []
                    lastItem = exceptionDict[excName]
                    continue
                elif desc.startswith("@signal"):
                    inTagSection = 1
                    m = _signal(desc,0)
                    if m is None:
                        raise TagError, "Wrong format in %s line.\n" % parts[0]
                    signalName = 1 and m.group("SignalName1") \
                                   or m.group("SignalName2")
                    signalDesc = 1 and m.group("SignalDescription1") \
                                   or m.group("SignalDescription2")
                    signalDict[signalName] = [signalDesc]
                    lastItem = signalDict[signalName]
                    continue
                elif desc.startswith("@deprecated"):
                    inTagSection = 1
                    parts = desc.split(None, 1)
                    if len(parts) < 2:
                        raise TagError, "Wrong format in %s line.\n" % parts[0]
                    deprecated = [parts[1]]
                    lastItem = deprecated
                    continue
                elif desc.startswith("@"):
                    tag = desc.split(None, 1)[0]
                    raise TagError, "Unknown tag encountered, %s.\n" % tag
                else:
                    lastItem.append(desc)
            elif not inTagSection:
                lastItem.append(desc)
                
        if paragraphs:
            description = self.genParagraphs(paragraphs)
        else:
            description = ""
        
        if paramDict:
            parameterSect = parametersListTemplate % { \
                'Parameters' : self.genDescriptionListSection(paramDict,
                               parametersListEntryTemplate)
            }
        else:
            parameterSect = ""
            
        if returns:
            returnSect = returnsTemplate %'\n'.join(returns)
        else:
            returnSect = ""
            
        if exceptionDict:
            exceptionSect = exceptionsListTemplate % { \
                'Exceptions' : self.genDescriptionListSection(exceptionDict,
                               exceptionsListEntryTemplate)
            }
        else:
            exceptionSect = ""
            
        if signalDict:
            signalSect = signalsListTemplate % { \
                'Signals' : self.genDescriptionListSection(signalDict,
                               signalsListEntryTemplate)
            }
        else:
            signalSect = ""
            
        if deprecated:
            deprecatedSect = deprecatedTemplate % { \
                'Lines' : '\n'.join(deprecated),
            }
        else:
            deprecatedSect = ""
        
        return "%s%s%s%s%s%s" % ( \
            deprecatedSect, description, parameterSect, returnSect,
            exceptionSect, signalSect,
        )
