# -*- coding: utf-8 -*-

# Copyright (c) 2002, 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the Start Program dialog.
"""

from qt import *

from StartForm import StartForm
from Cyclops.RunCyclops import CyclopsStats,CyclopsCycles,CyclopsCycleObjs, \
    CyclopsSCCS,CyclopsArcs,CyclopsIterate

class StartDialog(StartForm):
    """
    Class implementing the Start Program dialog.
    
    It implements a dialog that is used to start an
    application for debugging. It asks the user to enter
    the commandline parameters, the working directory and
    whether exception reporting should be disabled.
    """
    def __init__(self, caption, argvList, wdList, exceptions, parent=None,
        type=0,modfuncList=None,tracePython=0):
        """
        Constructor
        
        @param caption the caption to be displayed (QString)
        @param argvList history list of commandline arguments (QStringList)
        @param wdList history list of working directories (QStringList)
        @param exceptions exception reporting flag (boolean)
        @param parent parent widget of this dialog (QWidget)
        @param type type of the start dialog
                <ul>
                <li>0 = start debug dialog</li>
                <li>1 = start run dialog</li>
                <li>2 = start coverage dialog</li>
                <li>3 = start profile dialog</li>
                <li>4 = start cyclops dialog</li>
                </ul>
        @keyparam modfuncList history list of module functions (QStringList)
        @keyparam tracePython flag indicating if the Python library should 
            be traced as well (boolean)
        """
        StartForm.__init__(self, parent, None, 1)
        
        self.setCaption(caption)
        self.cmdlineCombo.clear()
        self.cmdlineCombo.insertStringList(argvList)
        if argvList.count() > 0:
            self.cmdlineCombo.setCurrentItem(0)
        self.workdirCombo.clear()
        self.workdirCombo.insertStringList(wdList)
        if wdList.count() > 0:
            self.workdirCombo.setCurrentItem(0)
        self.exceptionCheckBox.setChecked(exceptions)
        
        if type == 0:        # start debug dialog
            self.tracePythonCheckBox.setChecked(tracePython)
            self.tracePythonCheckBox.show()
        else:
            self.tracePythonCheckBox.hide()
            
        if type in [2, 3]:   # start coverage or profile dialog
            if type == 3:
                self.eraseCheckBox.setText(self.trUtf8('Erase timing data'))
                self.eraseCheckBox.setChecked(1)
            self.eraseCheckBox.show()
        else:
            self.eraseCheckBox.hide()
            
        if type == 4:       # start cyclops dialog
            self.cyclopsFrame.show()
            if modfuncList:
                self.modFuncCombo.clear()
                self.modFuncCombo.insertStringList(modfuncList)
                if modfuncList.count() > 0:
                    self.modFuncCombo.setCurrentItem(0)
            self.okButton.setDisabled(self.modFuncCombo.currentText().isEmpty())
        else:
            self.cyclopsFrame.hide()
            
        self.resize(self.width(), 1)
        
    def handleDir(self):
        """
        Private method used to open a directory selection dialog.
        """
        cwd = self.workdirCombo.currentText()
        d = QFileDialog.getExistingDirectory(cwd,
            self, None, self.trUtf8('Working directory'))
            
        if not d.isNull() and not d.isEmpty():
            self.workdirCombo.setCurrentText(QDir.convertSeparators(d))
        
    def getData(self):
        """
        Public method to retrieve the data entered into this dialog.
        
        @return a tuple of argv, workdir, exceptions
        """
        return (self.cmdlineCombo.currentText(),
                self.workdirCombo.currentText(),
                self.exceptionCheckBox.isChecked())
                
    def getDebugData(self):
        """
        Public method to retrieve the debug related data entered into this dialog.
        
        @return flag indicating if the Python library should be traced as well
        """
        return self.tracePythonCheckBox.isChecked()
        
    def getCoverageData(self):
        """
        Public method to retrieve the coverage related data entered into this dialog.
        
        @return flag indicating erasure of coverage info (boolean)
        """
        return self.eraseCheckBox.isChecked()
        
    def getCyclopsData(self):
        """
        Public method to retrieve the coverage related data entered into this dialog.
        
        @return tuple of module function that is the entry point (string) and
            bitmask of reports to be generated (integer)
        """
        mask = 0
        if self.cyclopsStatsCheckBox.isChecked():
            mask += CyclopsStats
        if self.cyclopsCyclesCheckBox.isChecked():
            mask += CyclopsCycles
        if self.cyclopsCycleObjsCheckBox.isChecked():
            mask += CyclopsCycleObjs
        if self.cyclopsSCCSCheckBox.isChecked():
            mask += CyclopsSCCS
        if self.cyclopsArcsCheckBox.isChecked():
            mask += CyclopsArcs
        if self.cyclopsIterateCheckBox.isChecked():
            mask += CyclopsIterate
        return (self.modFuncCombo.currentText(), mask)
        
    def enableOkButton(self):
        """
        Private slot to enable/disable the OK button.
        """
        self.okButton.setDisabled(self.modFuncCombo.currentText().isEmpty())
