#-------------------------------------------------------------------------------
#  
#  Defines the OMCanvas class of the Enable 'om' (Object Model) package. 
#
#  The OMCanvas class defines the visual container for the other om objects 
#  used to represent the underlying object model.
#  
#  Written by: David C. Morrill
#  
#  Date: 01/27/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from om_base      import om_handler
from om_traits    import StyleDelegate, SelectionBackgroundColor, ERGBAColor, \
                         SelectionBorderColor
from om_link      import OMLink
from om_component import OMComponent

from enthought.enable               import Container, Box, TOP, BOTTOM, LEFT, \
                                           RIGHT
from enthought.enable.base          import transparent_color, empty_rectangle, \
                                           intersect_bounds
from enthought.enable.enable_traits import grid_trait, border_size_trait

from enthought.traits.api               import HasStrictTraits, Instance, Str, \
                                           Property, List, Any, Event, true, \
                                           false
from enthought.traits.ui.api            import View, Group, Handler, UndoHistory

#-------------------------------------------------------------------------------
#  Trait aliases  (TEMPORARY):  
#-------------------------------------------------------------------------------

GridTrait       = grid_trait
BorderSizeTrait = border_size_trait

#-------------------------------------------------------------------------------
#  Data:
#-------------------------------------------------------------------------------

# The default, singleton, OMCanvasController object:
_default_canvas_controller = None

#-------------------------------------------------------------------------------
#  'OMCanvasStyle' class:  
#-------------------------------------------------------------------------------

class OMCanvasStyle ( HasStrictTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------

    # Canvas background color:
    bg_color     = ERGBAColor( ( 1.0, 1.0, 1.0, 1.0 ) )
    
    # Grid line color:
    grid_color   = ERGBAColor( ( .804, .761, .859, .75 ) )
    
    # Spacing between grid lines horizontally (in pixels):
    grid_width   = GridTrait
    
    # Spacing between grid lines vertically (in pixels):
    grid_height  = GridTrait
    
    # Thickness of the grid lines (in pixels):
    grid_size    = BorderSizeTrait( 1 )
    
    # Should the grid lines be displayed?
    grid_visible = true
    
    # Should components snap to the grad lines?
    snap_to_grid = false
    
    # Show tooltip information?
    show_tooltip = true
    
    # Show status information?
    show_status  = true
        
    # Color used to draw the background of a selected component:
    selection_bg_color     = SelectionBackgroundColor
        
    # Color used to draw the border of a selected component:
    selection_border_color = SelectionBorderColor
     
    # Context menu:
    menu = Property
    
#-- Property implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'menu' property (normally overridden):  
    #---------------------------------------------------------------------------
        
    def _get_menu ( self ):
        return self.get_menu()
        
    def get_menu ( self ):
        return None        
    
# Create a default canvas style:
default_canvas_style = OMCanvasStyle()

#-------------------------------------------------------------------------------
#  'OMCanvas' class:  
#-------------------------------------------------------------------------------

class OMCanvas ( Container ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
    
    # The controller for the canvas:
    controller   = Instance( 'enthought.enable.om.OMCanvasController' )

    # The style to use for drawing the canvas:
    style        = Instance( OMCanvasStyle, default_canvas_style )
    
    # Canvas background color:
    bg_color     = StyleDelegate
    
    # Grid line color:
    grid_color   = StyleDelegate
    
    # Spacing between grid lines horizontally (in pixels):
    grid_width   = StyleDelegate
    
    # Spacing between grid lines vertically (in pixels):
    grid_height  = StyleDelegate
    
    # Thickness of the grid lines (in pixels):
    grid_size    = StyleDelegate
    
    # Should the grid lines be displayed?
    grid_visible = StyleDelegate
    
    # Should components snap to the grid lines?
    snap_to_grid = StyleDelegate
    
    # Show tooltip information?
    show_tooltip = StyleDelegate
    
    # Show status information?
    show_status  = StyleDelegate
        
    # Color used to draw the background of a selected component:
    selection_bg_color     = StyleDelegate
        
    # Color used to draw the border of a selected component:
    selection_border_color = StyleDelegate

    # Context menu:
    menu                   = StyleDelegate
    
    # Tooltip text:
    tooltip      = Str
    
    # Status text:
    status       = Str
    
    # Current set of selected objects:
    selection    = List
    
    # Event fired when selection is modified:
    selection_modified = Event( bool )
    
    # List of all contacts available on the canvas:
    contacts     = Property
    
    # Canvas data (to help controller relate canvas back to object model):
    data         = Any
    
    # Event fired when canvas is updated in some significant manner:
    updated      = Event( bool )
    
    # Undo history of changes made to the canvas:
    history      = Instance( UndoHistory )
    
#-- Trait overrides ------------------------------------------------------------

    # Canvas accepts keyboard focus:
    accepts_focus = True

    #---------------------------------------------------------------------------
    #  Trait view definitions:
    #---------------------------------------------------------------------------
    
    traits_view = View( [ [ 'grid_visible{Is the grid visible?}', ' ', 
                            'snap_to_grid{Snap components to the grid?}',
                            '-[Options]>' ],
                          [ 'bg_color{Background Color}', '_', 
                            'grid_color{Grid Color}',
                            '|[Colors]@' ],
                          [ [ 'grid_width', '9', 'grid_height', '-' ],
                            [ '_', 'grid_size' ],
                            '|[Grid]@' ] ],
                        title   = 'Edit Canvas Properties', 
                        handler = om_handler )
    
    colorchip_map = {
        'fg_color': 'grid_color',
        'bg_color': 'bg_color'
    }
    
#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  'contacts' property:  
    #---------------------------------------------------------------------------

    def _get_contacts ( self ):
        contacts = []
        for component in self.components:
            if isinstance( component, OMComponent ):
                contacts.extend( component.contacts )
        return contacts
  
#-- Default Trait Value Handlers -----------------------------------------------

    #---------------------------------------------------------------------------
    #  Returns the default value for the 'controller' trait:  
    #---------------------------------------------------------------------------
    
    def _controller_default ( self ):
        global _default_canvas_controller
    
        if _default_canvas_controller is None:
            from om_canvas_controller import OMCanvasController
            _default_canvas_controller = OMCanvasController()
        return _default_canvas_controller
        
#-- Overridden Public Methods --------------------------------------------------
        
    #---------------------------------------------------------------------------
    #  Add one or more components to the container:
    #---------------------------------------------------------------------------
    
    def add ( self, *components ):
        the_components = self.components
        for component in components:
            if isinstance( component, OMLink ):
                self.add_at( 0, component )
            else:
                self.add_at( len( the_components ), component )
    
#-- Public Methods -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Selects the specified component:  
    #---------------------------------------------------------------------------

    def select_component ( self, component ):
        selection = self.selection
        if (len( selection ) != 1) or (component is not selection[0]):
            self.clear_selection( False )
            component.selected = True
            selection.append( component )
            self._selection_modified()
            self.redraw()
            
    #---------------------------------------------------------------------------
    #  Selects a list of components:  
    #---------------------------------------------------------------------------
                        
    def select_components ( self, components ):
        selection = self.selection
        if len( selection ) == len( components ):
            for component in components:
                if component not in selection:
                    break
            else:
                for component in selection:
                    if component not in components:
                        break
                else:
                    return False
        self.clear_selection( False )
        for component in components:
            component.selected = True
            selection.append( component )
        self._selection_modified()
        self.redraw()
        return True
        
    #---------------------------------------------------------------------------
    #  Toggles the selection status of a specified component:  
    #---------------------------------------------------------------------------
                
    def toggle_select_components ( self, components ):
        modified  = False
        selection = self.selection
        for component in components:
            component.selected = not component.selected
            if component.selected:
                if component not in selection:
                    modified = True
                    selection.append( component )
            else:
                if component in selection:
                    modified = True
                    selection.remove( component )
        if modified:
            self._selection_modified()
        self.redraw()
        
    #---------------------------------------------------------------------------
    #  Clears the current selection (if any):  
    #---------------------------------------------------------------------------
    
    def clear_selection ( self, update = True ):
        selection = self.selection
        if len( selection ) > 0:
            for item in selection:
                item.selected = False
            del selection[:]
            if update:
                self._selection_modified()
                self.redraw()
                
    #---------------------------------------------------------------------------
    #  Selects all components on the canvas:  
    #---------------------------------------------------------------------------
                                
    def select_all ( self ):
        """ Selects all components on the canvas.
        """
        update    = False
        selection = [ c for c in self.components 
                      if isinstance( c, OMComponent ) ]
        for c in selection:
            update    |= (c.selected == False)
            c.selected = True
        self.selection[:] = selection
        if update:
            self._selection_modified()
            self.redraw()
                
    #---------------------------------------------------------------------------
    #  Edit the properties of the canvas:  
    #---------------------------------------------------------------------------
    
    def edit ( self ):
        self.edit_traits()
        
    #---------------------------------------------------------------------------
    #  Force the component to be updated:    
    #---------------------------------------------------------------------------

    def update ( self ):
        self.redraw()     
        
    #---------------------------------------------------------------------------
    #  Undo the most recent change made to the canvas:  
    #---------------------------------------------------------------------------
                
    def undo ( self ):
        """ Undo the most recent change made to the canvas.
        """
        if self.history is not None:
            self.history.undo()
        
    #---------------------------------------------------------------------------
    #  Redo the most recently undone change made to the canvas:  
    #---------------------------------------------------------------------------
                
    def redo ( self ):
        """ Redo the most recently undone change made to the canvas.
        """
        if self.history is not None:
            self.history.redo()
            
    #---------------------------------------------------------------------------
    #  Adds an undo item to the undo history:    
    #---------------------------------------------------------------------------
                        
    def add_undo ( self, undo_item, extend = False ):
        """ Adds an undo item to the undo history.
        """
        if self.history is not None:
            self.history.add( undo_item, extend )
            
    #---------------------------------------------------------------------------
    #  Extends the most recent undo history entry with another undo item:
    #---------------------------------------------------------------------------
                        
    def extend_undo ( self, undo_item ):
        """ Extends the most recent undo history entry with another undo item.
        """
        if self.history is not None:
            self.history.extend( undo_item )
        
#-- Private Methods ------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Notifies the controller whenever the selection is changed:  
    #---------------------------------------------------------------------------
    
    def _selection_modified ( self ):
        self.selection_modified = True
        controller = self.controller
        if controller is not None:
            controller.selection_changed( self )

    #---------------------------------------------------------------------------
    #  Verify that the suggested bounds for a component match the current 
    #  snap to grid mode and grid size. If not, adjust them accordingly: 
    #---------------------------------------------------------------------------
    
    def _check_bounds ( self, component ):
        super( OMCanvas, self )._check_bounds( component )
        if self.snap_to_grid:
            component.bounds = self._check_snap( component, component.bounds )
    
    #---------------------------------------------------------------------------
    #  Makes sure that the specified component bounds match the current grid:  
    #---------------------------------------------------------------------------
                        
    def _check_snap ( self, component, bounds ):
        if not self.snap_to_grid:
            return bounds
            
        x, y, dx, dy     = self.bounds
        cx, cy, cdx, cdy = bounds
        gdx = self.grid_width
        if gdx > 1:
            cx  = max( 0, int( gdx * round( float( cx - x ) / gdx ) ) )
            cdx = gdx * ((cdx + gdx - 1) / gdx)
            if (cx + cdx) > dx:
                cx = dx - cdx
        gdy = self.grid_height
        if gdy > 1:
            cy  = max( 0, int( gdy * round( float( cy - y ) / gdy ) ) )
            cdy = gdy * ((cdy + gdy - 1) / gdy)
            if (cy + cdy) > dy:
                cy = dy - cdy
        return ( x + cx, y + cy, cdx, cdy )

    #---------------------------------------------------------------------------
    #  Draw the container background in a specified graphics context:
    #  (This method should normally be overridden by a subclass)
    #---------------------------------------------------------------------------
    
    def _draw_container ( self, gc ):
        gc.save_state()
        
        x, y, dx, dy = self.bounds
        
        # Fill the background region (if required);
        bg_color = self.bg_color_
        if bg_color is not transparent_color:
            gc.set_fill_color( bg_color )
            gc.begin_path()
            gc.rect( x, y, dx, dy ) 
            gc.fill_path()
            
        # Draw the grid (if required):
        if self.grid_visible:
            gs = self.grid_size
            if gs > 0:
                gsh = gs / 2.0
                gc.set_stroke_color( self.grid_color_ )
                gc.set_line_width( gs )
                gc.begin_path()
                yb  = y + gsh
                yt  = y + dy - gsh
                xl  = xc = x + gsh
                xr  = x + dx - gsh
                gdx = self.grid_width
                if gdx > 0:
                    while xc < xr:
                        gc.move_to( xc, yb )
                        gc.line_to( xc, yt )
                        xc += gdx
                gdy = self.grid_height
                if gdy > 0:
                    while yb < yt:
                        gc.move_to( xl, yb )
                        gc.line_to( xr, yb )
                        yb += gdy
                gc.stroke_path()

        gc.restore_state()

#-- Event Handlers -------------------------------------------------------------
 
    #---------------------------------------------------------------------------
    #  Handles a keyboard event for the canvas:  
    #---------------------------------------------------------------------------
    
    def _key_changed ( self, event ):
        self.controller.key_pressed( self, event )

    #---------------------------------------------------------------------------
    #  Handles the completion of a 'drag select' operation:  
    #---------------------------------------------------------------------------
    
    def _selection_done ( self, marker, old, event ):
        bounds        = marker.bounds
        new_selection = [ c for c in self.components
                          if isinstance( c, OMComponent ) and 
                             intersect_bounds( bounds, c.bounds ) is not 
                             empty_rectangle ]
        cur_selection = self.selection
        update        = (len( new_selection ) != len( cur_selection ))
        if not update:
            for component in new_selection:
                if component not in cur_selection:
                    update = True
                    break
        if update:
            self.controller.components_selected( self, new_selection )
        marker.bounds = ( 99999.0, 99999.0, 1.0, 1.0 )
        
    #---------------------------------------------------------------------------
    #  Handle mouse events: 
    #---------------------------------------------------------------------------
    
#-- 'normal' State -------------------------------------------------------------

    def normal_left_down ( self, event ):
        event.handled           = True
        self.window.mouse_owner = self
        self.event_state        = 'selection_pending'
        self._x_y               = ( event.x, event.y )
        
    def normal_left_dclick ( self, event ):
        event.handled = True
        self.controller.edit_canvas( self )

    def normal_right_up ( self, event ):
        event.handled = True
        self.controller.popup_menu( self, event )
        
#-- 'selection_pending' State -------------------------------------------------------

    def selection_pending_left_up ( self, event ):
        event.handled           = True
        self.window.mouse_owner = None
        self.event_state        = 'normal'
        if self.controller.can_clear_selection( self ):
            self.controller.components_selected( self, [] )
        
    def selection_pending_mouse_move ( self, event ):
        event.handled = True
        x, y          = self._x_y
        dx            = abs( x - event.x )
        dy            = abs( y - event.y )        
        if (dx + dy) > 2:
            self.event_state = 'normal'
            if self.controller.can_drag_select( self ): 
                self.window.mouse_owner = None
                marker = Box( container    = self,
                              color        = self.selection_bg_color,
                              border_color = self.selection_border_color,
                              border_size  = 1,
                              bounds       = ( min( x, event.x ), 
                                               min( y, event.y ),
                                               max( 1, dx ), max( 1, dy ) ) )
                marker.on_trait_change( self._selection_done, 'resized' )
                self.window.drag_resize( marker, self.bounds, event,
                                 unconstrain  = TOP | BOTTOM | LEFT | RIGHT, 
                                 overlay      = True )

    #---------------------------------------------------------------------------
    #  Handles an Enable 'dropped on' event:  
    #---------------------------------------------------------------------------
                                                                  
    def _dropped_on_changed ( self, event ):
        """ Handles an Enable 'dropped on' event.
        """
        super( OMCanvas, self )._dropped_on_changed( event )
        components = [ c for c in event.components 
                       if isinstance( c, OMComponent ) ]
        if components == self._drag_components:
            self.controller.components_dragged( self, self._drag_components,
                                                self._drag_bounds )
            self._drag_components = self._drag_original_bounds = None

    #---------------------------------------------------------------------------
    #  Handle an OMComponent object being dropped on the canvas: 
    #---------------------------------------------------------------------------
    
    def dropped_on_by_omcomponent ( self, component, event ):
        event.handled    = True
        x, y, dx, dy     = component.bounds
        component.bounds = self._check_snap( component, 
                               ( x + event.x - event.x0, y + event.y - event.y0,
                                 dx, dy ) )
        try:
            self.components.remove( component )
        except:
            pass
        self.components.append( component )
        
        for link in component.links:
            link.visible = True
                             
    #--------------------------------------------------------------------------
    #  Handle an object with traits being dragged/dropped on the canvas:
    #--------------------------------------------------------------------------
    
    def drag_over_by_hastraits ( self, component, event ):
        if self.controller.can_drop( self, component, event ):
            event.handled = True
    
    def dropped_on_by_hastraits ( self, component, event ):
        event.handled = True
        self.controller.dropped( self, component, event )
            
