#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <glib.h>

#include "edv_context.h"
#include "edv_id.h"
#include "edv_ids_list.h"


GList *EDVUIDGetSystem(void);
GList *EDVGIDGetSystem(void);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets a list of UIDs from the system.
 *
 *	Returns a GList of edv_uid_struct * or NULL on error. The
 *	calling function must delete the returned list and each
 *	uid.
 */
GList *EDVUIDGetSystem(void)
{
	struct passwd *pwent;
	GList *uids_list = NULL;
	edv_uid_struct *uid;

	/* Open the passwords file and get the first entry */
	pwent = getpwent();

	/* Iterate through each entry */
	while(pwent != NULL)
	{
	    /* Create a new UID */
	    uid = EDVUIDNew();
	    if(uid == NULL)
		break;

	    /* Get this UID */
	    uid->name = STRDUP(pwent->pw_name);
	    uid->password = STRDUP(pwent->pw_passwd);
	    uid->user_id = pwent->pw_uid;
	    uid->group_id = pwent->pw_gid;
	    uid->government_name = STRDUP(pwent->pw_gecos);
	    uid->home_directory = STRDUP(pwent->pw_dir);
	    uid->shell_program = STRDUP(pwent->pw_shell);

	    uids_list = g_list_append(uids_list, uid);

	    /* Get the next entry */
	    pwent = getpwent();
	}

	/* Close the passwords file */
	endpwent();

	return(uids_list);
}

/*
 *	Gets a list of GIDs from the system.
 *
 *	Returns a GList of edv_gid_struct * or NULL on error. The
 *	calling function must delete the returned list and each
 *	gid.
 */
GList *EDVGIDGetSystem(void)
{
	struct group *grent;
	GList *gids_list = NULL;
	edv_gid_struct *gid;

	/* Open the groups file and get the first entry */
	grent = getgrent();

	/* Iterate through each entry */
	while(grent != NULL)
	{
	    /* Create a new GID */
	    gid = EDVGIDNew();
	    if(gid == NULL)
		break;

	    /* Get this GID */
	    gid->name = STRDUP(grent->gr_name);
	    gid->password = STRDUP(grent->gr_passwd);
	    gid->group_id = grent->gr_gid;

	    /* Any group members to add? */
	    if(grent->gr_mem != NULL)
	    {
		gint i;
		for(i = 0; grent->gr_mem[i] != NULL; i++)
		    gid->group_members_list = g_list_append(
			gid->group_members_list,
			STRDUP(grent->gr_mem[i])
		    );
	    }

	    gids_list = g_list_append(gids_list, gid);

	    /* Get the next entry */
	    grent = getgrent();
	}

	/* Close the groups file */
	endgrent();

	return(gids_list);
}
