#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "csd.h"
#include "fsd.h"
#include "pulist.h"
#include "style_edit.h"
#include "stacklist.h"
#include "keymap_list.h"
#include "menucfg_list.h"

#include "edv_types.h"
#include "edv_generic_options_win.h"
#include "edv_generic_options_win_op.h"
#include "edv_utils_gtk.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"


/* Callbacks */
static gint EDVGenOptWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void OptWinSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
);
static void OptWinOKCB(GtkWidget *widget, gpointer data);
static void OptWinApplyCB(GtkWidget *widget, gpointer data);
static void OptWinSaveCB(GtkWidget *widget, gpointer data);
static void OptWinCancelCB(GtkWidget *widget, gpointer data);
static void OptWinAnyChangedCB(GtkWidget *widget, gpointer data);
static void OptWinCListChangedCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void OptWinPopupListBoxChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void OptWinStyleEditChangedCB(
	style_edit_struct *se, gpointer data
);
static void OptWinStackListChangedCB(
	stack_list_struct *slist, gpointer data
);
static void OptWinKeymapListChangedCB(
	keymap_list_struct *kmlist, gpointer data
);
static void OptWinMenuCfgListChangedCB(
	menucfg_list_struct *list, gpointer data
);


/* Widget References */
edv_gen_opt_wref_struct *EDVGenOptWinWRefNew(
	const edv_gen_opt_widget_type type,
	gpointer w,
	GtkWidget *browse,	/* Browse GtkButton */
	const gchar *cfg_parm,	/* Configuration item parameter */
	gpointer optwin
);
void EDVGenOptWinWRefDelete(edv_gen_opt_wref_struct *wref);
edv_gen_opt_wref_struct *EDVGenOptWinWRefAppend(
	edv_gen_opt_win_struct *optwin,
	const edv_gen_opt_widget_type type,
	gpointer w,
	GtkWidget *browse,	/* Browse button */
	const gchar *cfg_parm	/* Cfg item parameter */
);

/* Options Window */
edv_gen_opt_win_struct *EDVGenOptWinNew(
	edv_core_struct *core,
	const gchar *title, guint8 **icon_data
);
GtkWidget *EDVGenOptWinPageAppend(
	edv_gen_opt_win_struct *optwin, GtkWidget *w
);
void EDVGenOptWinUpdateMenus(edv_gen_opt_win_struct *optwin);
void EDVGenOptWinSetBusy(edv_gen_opt_win_struct *optwin, gboolean is_busy);
gboolean EDVGenOptWinIsMapped(edv_gen_opt_win_struct *optwin);
void EDVGenOptWinMap(edv_gen_opt_win_struct *optwin);
void EDVGenOptWinUnmap(edv_gen_opt_win_struct *optwin);
void EDVGenOptWinDelete(edv_gen_opt_win_struct *optwin);


#define OPTWIN_WIDTH		640
#define OPTWIN_HEIGHT		480

#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint EDVGenOptWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_gen_opt_win_struct *optwin = EDV_GEN_OPT_WIN(data);
	if(optwin == NULL)
	    return(TRUE);

	OptWinCancelCB(NULL, optwin);

	return(TRUE);
}

/*
 *	GtkNotebook "switch_page" signal callback.
 */
static void OptWinSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
)
{
	edv_gen_opt_win_struct *optwin = EDV_GEN_OPT_WIN(data);
	if(optwin == NULL)
	    return;

	if(optwin->freeze_count > 0)
	    return;

	optwin->freeze_count++;

	EDVGenOptWinUpdateMenus(optwin);

	optwin->freeze_count--;
}

/*
 *	OK callback.
 */
static void OptWinOKCB(GtkWidget *widget, gpointer data)
{
	edv_gen_opt_win_struct *optwin = EDV_GEN_OPT_WIN(data);
	if(optwin == NULL)
	    return;

	if(optwin->freeze_count > 0)
	    return;

	optwin->freeze_count++;

	EDVGenOptWinSetValues(optwin);
	EDVGenOptWinResetHasChanges(optwin, FALSE);
	EDVGenOptWinUnmap(optwin);

	optwin->freeze_count--;
}

/*
 *	Apply callback.
 */
static void OptWinApplyCB(GtkWidget *widget, gpointer data)
{
	edv_gen_opt_win_struct *optwin = EDV_GEN_OPT_WIN(data);
	if(optwin == NULL)
	    return;

	if(optwin->freeze_count > 0)
	    return;

	optwin->freeze_count++;

	EDVGenOptWinSetValues(optwin);
	EDVGenOptWinResetHasChanges(optwin, FALSE);
	EDVGenOptWinUpdateMenus(optwin);

	optwin->freeze_count--;
}

/*
 *	Save callback.
 */
static void OptWinSaveCB(GtkWidget *widget, gpointer data)
{
	edv_gen_opt_win_struct *optwin = EDV_GEN_OPT_WIN(data);
	if(optwin == NULL)
	    return;

	if(optwin->freeze_count > 0)
	    return;

	optwin->freeze_count++;

/* TODO */


	EDVGenOptWinUpdateMenus(optwin);

	optwin->freeze_count--;
}

/*
 *	Cancel callback.
 */
static void OptWinCancelCB(GtkWidget *widget, gpointer data)
{
	edv_gen_opt_win_struct *optwin = EDV_GEN_OPT_WIN(data);
	if(optwin == NULL)
	    return;

	if(optwin->freeze_count > 0)
	    return;

	optwin->freeze_count++;

	EDVGenOptWinUnmap(optwin);

	optwin->freeze_count--;
}

/*
 *	Any GtkWidget changed signal callback.
 *
 *	For widgets referenced in the Option Window's Widget References
 *	that have been changed, this will update the Widget References's
 *	has_changes marker and the Option Window's has_changes marker.
 *
 *	The given widget must be ignored, instead use wref->w.
 */
static void OptWinAnyChangedCB(GtkWidget *widget, gpointer data)
{
	edv_gen_opt_win_struct *optwin;
	edv_gen_opt_wref_struct *wref = EDV_GEN_OPT_WREF(data);
	if(wref == NULL)
	    return;

	/* Must use Widget Reference's widget (may be NULL) */
	widget = wref->w;

	/* Ignore changes? */
	if(wref->ignore_changes)
	    return;

	optwin = EDV_GEN_OPT_WIN(wref->optwin);
	if(optwin == NULL)
	    return;

	if(optwin->freeze_count > 0)
	    return;

	optwin->freeze_count++;

	if(!wref->has_changes)
	{
	    wref->has_changes = TRUE;
	    EDVGenOptWinUpdateMenus(optwin);
	}

	if(!optwin->has_changes)
	{
	    optwin->has_changes = TRUE;
	    EDVGenOptWinUpdateMenus(optwin);
	}

	optwin->freeze_count--;
}
static void OptWinCListChangedCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	OptWinAnyChangedCB(NULL, data);
}
static void OptWinPopupListBoxChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	OptWinAnyChangedCB(NULL, data);
}
static void OptWinStyleEditChangedCB(
	style_edit_struct *se, gpointer data
)
{
	OptWinAnyChangedCB(NULL, data);
}
static void OptWinStackListChangedCB(
	stack_list_struct *slist, gpointer data
)
{
	OptWinAnyChangedCB(NULL, data);
}
static void OptWinKeymapListChangedCB(
	keymap_list_struct *kmlist, gpointer data
)
{
	OptWinAnyChangedCB(NULL, data);
}
static void OptWinMenuCfgListChangedCB(
	menucfg_list_struct *list, gpointer data
)
{
	OptWinAnyChangedCB(NULL, data);
}

/*
 *	Creates a new Options Window Widget Reference.
 */
edv_gen_opt_wref_struct *EDVGenOptWinWRefNew(
	const edv_gen_opt_widget_type type,
	gpointer w,
	GtkWidget *browse,	/* Browse GtkButton */
	const gchar *cfg_parm,	/* Configuration item parameter */
	gpointer optwin
)
{
	edv_gen_opt_wref_struct *wref = EDV_GEN_OPT_WREF(
	    g_malloc0(sizeof(edv_gen_opt_wref_struct))
	);
	if(wref == NULL)
	    return(wref);

	wref->type = type;
	wref->w = w;
	wref->browse = browse;
	wref->cfg_parm = STRDUP(cfg_parm);
	wref->optwin = optwin;
	wref->has_changes = FALSE;
	wref->ignore_changes = FALSE;

	wref->radio_value = 0;

	return(wref);
}

/*
 *	Deletes the Options Window Widget Reference.
 */
void EDVGenOptWinWRefDelete(edv_gen_opt_wref_struct *wref)
{
	GtkWidget *toplevel = NULL;
	edv_gen_opt_win_struct *optwin;
	edv_core_struct *core = NULL;

	if(wref == NULL)
	    return;

	optwin = wref->optwin;
	if(optwin != NULL)
	{
	    toplevel = optwin->toplevel;
	    core = optwin->core;
	}

	/* Destroy the widget */
	switch(wref->type)
	{
	  case EDV_GEN_OPT_WIDGET_UNKNOWN:
	    if(wref->w != NULL)
	    {
		gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefDelete():\n\
optwin=%p\n\
wref->cfg_parm=\"%s\"\n\
wref->w=%p\n\
\n\
Unable to destroy of type EDV_GEN_OPT_WIDGET_UNKNOWN.",
		    optwin,
		    wref->cfg_parm,
		    wref->w
		);
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Internal Error",
		    msg,
		    NULL,
		    toplevel
		);
		g_free(msg);
	    }
	    break;

	  case EDV_GEN_OPT_WIDGET_DRAWING_AREA:
	  case EDV_GEN_OPT_WIDGET_BUTTON:
	  case EDV_GEN_OPT_WIDGET_TOGGLE_BUTTON:
	  case EDV_GEN_OPT_WIDGET_RADIO_BUTTON:
	  case EDV_GEN_OPT_WIDGET_RANGE:
	  case EDV_GEN_OPT_WIDGET_EDITABLE:
	  case EDV_GEN_OPT_WIDGET_SPIN_BUTTON:
	  case EDV_GEN_OPT_WIDGET_COMBO:
	  case EDV_GEN_OPT_WIDGET_CLIST:
	    if(wref->w != NULL)
	    {
		GtkWidget *w = (GtkWidget *)wref->w;
		GTK_WIDGET_DESTROY(w);
	    }
	    break;

	  case EDV_GEN_OPT_WIDGET_COLOR_BUTTON:
	    if(wref->w != NULL)
	    {
		GtkWidget *w = (GtkWidget *)wref->w;
		GTK_WIDGET_DESTROY(w);
	    }
	    break;
	  case EDV_GEN_OPT_WIDGET_POPUP_LIST_BOX:
	    PUListBoxDelete(PULISTBOX(wref->w));
	    break;
	  case EDV_GEN_OPT_WIDGET_STYLE_EDIT:
	    StyleEditDelete(STYLE_EDIT(wref->w));
	    break;
	  case EDV_GEN_OPT_WIDGET_STACK_LIST:
	    StackListDelete(STACK_LIST(wref->w));
	    break;
	  case EDV_GEN_OPT_WIDGET_KEYMAP_LIST:
	    KeymapListDelete(KEYMAP_LIST(wref->w));
	    break;
	  case EDV_GEN_OPT_WIDGET_MENUCFG_LIST:
	    MenuCfgListDelete(MENUCFG_LIST(wref->w));
	    break;
	}

	GTK_WIDGET_DESTROY(wref->browse);

	g_free(wref->cfg_parm);

	g_free(wref);
}

/*
 *	Appends a new Widget Reference to the Options Window.
 *
 *	The optwin specifies the Options Window.
 *
 *	The type specifies the what type of "widget" w is.
 *
 *	The w specifies the "widget".
 *
 *	The browse specifies the browse GtkButton.
 *
 *	The cfg_parm specifies the Cfg Item's parameter
 *
 *	Returns the new Widget Reference, the new Widget Reference
 *	must not be deleted since it has already been appended to
 *	the Options Window.
 */
edv_gen_opt_wref_struct *EDVGenOptWinWRefAppend(
	edv_gen_opt_win_struct *optwin,
	const edv_gen_opt_widget_type type,
	gpointer w,
	GtkWidget *browse,
	const gchar *cfg_parm
)
{
	GtkWidget *toplevel;
	edv_gen_opt_wref_struct *wref;
	edv_core_struct *core;

	if(optwin == NULL)
	    return(NULL);

	toplevel = optwin->toplevel;
	core = optwin->core;

	/* Create a new widget reference */
	wref = EDVGenOptWinWRefNew(
	    type, w, browse, cfg_parm, optwin
	);
	if(wref != NULL)
	{
	    void (*optwin_change_cb)(GtkWidget *, gpointer) =
		OptWinAnyChangedCB;
	    gpointer data = wref;
	    guint sig_id = 0;

	    /* Attach the "changed" signal callback based on the
	     * widget's type
	     */
	    switch(type)
	    {
	      case EDV_GEN_OPT_WIDGET_UNKNOWN:
		if(TRUE)
		{
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
Unable to append a widget of type EDV_GEN_OPT_WIDGET_UNKNOWN.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;

	      case EDV_GEN_OPT_WIDGET_DRAWING_AREA:
		if((w != NULL) ? GTK_IS_DRAWING_AREA(w) : FALSE)
		{
		    /* Does not generate a signal */
		}
		else
		{
		    /* Not a GtkDrawingArea */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_DRAWING_AREA\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkDrawingArea.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_BUTTON:
		if((w != NULL) ? GTK_IS_BUTTON(w) : FALSE)
		{
		    GtkButton *btn = (GtkButton *)w;
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(btn), "clicked",
			GTK_SIGNAL_FUNC(optwin_change_cb), data
		    );
		    if(GTK_IS_TOGGLE_BUTTON(w))
		    {
			gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget's type was specified as\n\
EDV_GEN_OPT_WIDGET_BUTTON but the widget was detected\n\
to be a GtkToggleButton, you should use specify the\n\
type as EDV_GEN_OPT_WIDGET_TOGGLE_BUTTON instead.",
			    optwin,
			    cfg_parm,
			    w
			);
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Internal Error",
			    msg,
			    NULL,
			    toplevel
			);
			g_free(msg);
		    }
		}
		else
		{
		    /* Not a GtkButton */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkButton.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_TOGGLE_BUTTON:
		if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		{
		    GtkToggleButton *tb = (GtkToggleButton *)w;
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(tb), "toggled",
			GTK_SIGNAL_FUNC(optwin_change_cb), data
		    );
		    if(GTK_IS_RADIO_BUTTON(w))
		    {
			gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_TOGGLE_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget's type was specified as\n\
EDV_GEN_OPT_WIDGET_TOGGLE_BUTTON but the widget was detected\n\
to be a GtkRadioButton, you should use specify the\n\
type as EDV_GEN_OPT_WIDGET_RADIO_BUTTON instead.",
			    optwin,
			    cfg_parm,
			    w
			);
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Internal Error",
			    msg,
			    NULL,
			    toplevel
			);
			g_free(msg);
		    }
		}
		else
		{
		    /* Not a GtkToggleButton */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_TOGGLE_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkToggleButton.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_RADIO_BUTTON:
		if((w != NULL) ? GTK_IS_RADIO_BUTTON(w) : FALSE)
		{
		    GtkToggleButton *tb = (GtkToggleButton *)w;
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(tb), "toggled",
			GTK_SIGNAL_FUNC(optwin_change_cb), data
		    );
		}
		else
		{
		    /* Not a GtkRadioButton */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_RADIO_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkRadioButton.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_RANGE:
		if((w != NULL) ? GTK_IS_RANGE(w) : FALSE)
		{
		    GtkRange *range = (GtkRange *)w;
		    GtkAdjustment *adj = range->adjustment;
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(adj), "value_changed",
			GTK_SIGNAL_FUNC(optwin_change_cb), data
		    );
		}
		else
		{
		    /* Not a GtkRange */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_RANGE\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkRange.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_EDITABLE:
		if((w != NULL) ? GTK_IS_EDITABLE(w) : FALSE)
		{
		    GtkEditable *editable = (GtkEditable *)w;
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(editable), "changed",
			GTK_SIGNAL_FUNC(optwin_change_cb), data
		    );
		    if(GTK_IS_SPIN_BUTTON(w))
		    {
			gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_EDITABLE\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget's type was specified as\n\
EDV_GEN_OPT_WIDGET_EDITABLE but the widget was detected\n\
to be a GtkSpinButton, you should use specify the\n\
type as EDV_GEN_OPT_WIDGET_SPIN_BUTTON instead.",
			    optwin,
			    cfg_parm,
			    w
			);
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Internal Error",
			    msg,
			    NULL,
			    toplevel
			);
			g_free(msg);
		    }
		}
		else
		{
		    /* Not a GtkEditable */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_EDITABLE\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkEditable.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_SPIN_BUTTON:
		if((w != NULL) ? GTK_IS_SPIN_BUTTON(w) : FALSE)
		{
		    GtkEditable *editable = (GtkEditable *)w;
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(editable), "changed",
			GTK_SIGNAL_FUNC(optwin_change_cb), data
		    );
		}
		else
		{
		    /* Not a GtkSpinButton */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_SPIN_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkSpinButton.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_COMBO:
		if((w != NULL) ? GTK_IS_COMBO(w) : FALSE)
		{
		    GtkCombo *combo = (GtkCombo *)w;
		    GtkEntry *entry = GTK_ENTRY(combo->entry);
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(entry), "changed",
			GTK_SIGNAL_FUNC(optwin_change_cb), data
		    );
		}
		else
		{
		    /* Not a GtkCombo */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_COMBO\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkCombo.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;
	      case EDV_GEN_OPT_WIDGET_CLIST:
		if((w != NULL) ? GTK_IS_CLIST(w) : FALSE)
		{
		    GtkCList *clist = (GtkCList *)w;
		    sig_id = gtk_signal_connect(
			GTK_OBJECT(clist), "select_row",
			GTK_SIGNAL_FUNC(OptWinCListChangedCB), data
		    );
		}
		else
		{
		    /* Not a GtkCList */
		    gchar *msg = g_strdup_printf(
"EDVGenOptWinWRefAppend():\n\
optwin=%p\n\
type=EDV_GEN_OPT_WIDGET_CLIST\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkCList.",
			optwin,
			cfg_parm,
			w
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Internal Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		break;

	      case EDV_GEN_OPT_WIDGET_COLOR_BUTTON:
		if(w != NULL)
		{
		    GtkButton *btn = (GtkButton *)CSDColorButtonGetButton((GtkWidget *)w);
		    gtk_signal_connect(
			GTK_OBJECT(btn), "clicked",
			GTK_SIGNAL_FUNC(OptWinAnyChangedCB), wref
		    );
		}
		break;
	      case EDV_GEN_OPT_WIDGET_POPUP_LIST_BOX:
		if(w != NULL)
		{
		    pulistbox_struct *pulistbox = PULISTBOX(w);
		    PUListBoxSetChangedCB(
			pulistbox,
			OptWinPopupListBoxChangedCB, wref
		    );
		}
		break;
	      case EDV_GEN_OPT_WIDGET_STYLE_EDIT:
		if(w != NULL)
		{
		    style_edit_struct *style_edit = STYLE_EDIT(w);
		    StyleEditSetChangedCB(
			style_edit,
			OptWinStyleEditChangedCB, wref
		    );
		}
		break;
	      case EDV_GEN_OPT_WIDGET_STACK_LIST:
		if(w != NULL)
		{
		    stack_list_struct *slist = STACK_LIST(w);
		    StackListSetChangedCB(
			slist,
			OptWinStackListChangedCB, wref
		    );
		}
		break;
	      case EDV_GEN_OPT_WIDGET_KEYMAP_LIST:
		if(w != NULL)
		{
		    keymap_list_struct *kmlist = KEYMAP_LIST(w);
		    KeymapListSetChangedCB(
			kmlist,
			OptWinKeymapListChangedCB, wref
		    );
		}
		break;
	      case EDV_GEN_OPT_WIDGET_MENUCFG_LIST:
		if(w != NULL)
		{
		    menucfg_list_struct *menucfg_list = MENUCFG_LIST(w);
		    MenuCfgListSetChangedCB(
			menucfg_list,
			OptWinMenuCfgListChangedCB, wref
		    );
		}
		break;
	    }

	    /* Append this widget reference to the list */
	    optwin->wrefs_list = g_list_append(
		optwin->wrefs_list, wref
	    );
	}

	return(wref);
}


/*
 *	Creates a new Options Window.
 */
edv_gen_opt_win_struct *EDVGenOptWinNew(
	edv_core_struct *core,
	const gchar *title, guint8 **icon_data
)
{
	const gint border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget	*w,
			*parent, *parent2,
			*toplevel;
	edv_gen_opt_win_struct *optwin;

	if(core == NULL)
	    return(NULL);

	optwin = EDV_GEN_OPT_WIN(g_malloc0(sizeof(edv_gen_opt_win_struct)));
	if(optwin == NULL)
	    return(NULL);

	optwin->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	optwin->accelgrp = accelgrp = gtk_accel_group_new();
	optwin->freeze_count = 0;
	optwin->busy_count = 0;
	optwin->has_changes = FALSE;
	optwin->title = STRDUP(title);
	optwin->icon_data = icon_data;
	optwin->core = core;

	optwin->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(w, OPTWIN_WIDTH, OPTWIN_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(w), title);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "options", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_data);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVGenOptWinDeleteEventCB), optwin
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	style = gtk_widget_get_style(w);
	parent = w;


	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Notebook GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Main GtkNotebook */
	optwin->notebook = w = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
/*	gtk_notebook_set_page(GTK_NOTEBOOK(w), 0); */
	gtk_signal_connect(
	    GTK_OBJECT(w), "switch_page",
	    GTK_SIGNAL_FUNC(OptWinSwitchPageCB), optwin
	);
	gtk_widget_show(w);
	parent2 = w;



	/* Horizontal GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	optwin->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm,
	    "OK",
	    NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(OptWinOKCB), optwin
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);

	/* Apply button */
	optwin->apply_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(OptWinApplyCB), optwin
	);
	gtk_accel_group_add(
	    accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);

	/* Save button */
	optwin->save_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_save_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Salve"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregistrer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Red"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Poupe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt"
#else
"Save"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(OptWinSaveCB), optwin
	);
	gtk_accel_group_add(
	    accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);

	/* Cancel button */
	optwin->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(OptWinCancelCB), optwin
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close button */
	optwin->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(OptWinCancelCB), optwin
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	optwin->freeze_count--;

	return(optwin);
}

/*
 *	Appends a new page to the Options Window.
 *
 *	The w specifies the GtkWidget to be used as the page tab,
 *	this widget should not be referenced again after this call.
 *	If w is NULL then a generic page label will be used instead.
 */
GtkWidget *EDVGenOptWinPageAppend(
	edv_gen_opt_win_struct *optwin, GtkWidget *w
)
{
	GtkWidget *main_vbox;
	GtkNotebook *notebook;

	if(optwin == NULL)
	{
	    GTK_WIDGET_DESTROY(w);
	    return(NULL);
	}

	notebook = (GtkNotebook *)optwin->notebook;
	if(notebook == NULL)
	{
	    GTK_WIDGET_DESTROY(w);
	    return(NULL);
	}

	/* If no label widget is given then create one */
	if(w == NULL)
	{
	    gchar *s = g_strdup_printf(
		"Page %i",
		g_list_length(optwin->page_toplevels_list) + 1
	    );
	    w = gtk_label_new(s);
	    g_free(s);
	}

	/* Create main GtkVBox for the new page */
	main_vbox = gtk_vbox_new(FALSE, 0);
	gtk_notebook_append_page(notebook, main_vbox, w);
	gtk_widget_show(main_vbox);

	/* Append the main GtkVBox to the page toplevels list */
	optwin->page_toplevels_list = g_list_append(
	    optwin->page_toplevels_list, main_vbox
	);

	/* Return the new main GtkVBox of the appended page */
	return(main_vbox);
}

/*
 *	Updates the Options Window widgets to reflect current values.
 */
void EDVGenOptWinUpdateMenus(edv_gen_opt_win_struct *optwin)
{
	gboolean	sensitive,
			has_changes;
	gchar *s;

	if(optwin == NULL)
	    return;

	optwin->freeze_count++;

	has_changes = optwin->has_changes;

	s = g_strconcat(
	    (optwin->title != NULL) ? optwin->title : "",
	    (has_changes) ? " (*)" : "",
	    NULL
	);
	gtk_window_set_title(GTK_WINDOW(optwin->toplevel), s);
	g_free(s);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = has_changes ? TRUE : FALSE;
	gtk_widget_set_sensitive(optwin->ok_btn, sensitive);
	gtk_widget_set_sensitive(optwin->apply_btn, sensitive);
	gtk_widget_set_sensitive(optwin->cancel_btn, sensitive);
	sensitive = has_changes ? FALSE : TRUE;
	gtk_widget_set_sensitive(optwin->close_btn, sensitive);
	if(has_changes)
	{
	    gtk_widget_show(optwin->ok_btn);
	    gtk_widget_show(optwin->apply_btn);
	    gtk_widget_show(optwin->cancel_btn);
	    gtk_widget_hide(optwin->close_btn);
	}
	else
	{
	    gtk_widget_show(optwin->ok_btn);
	    gtk_widget_show(optwin->apply_btn);
	    gtk_widget_hide(optwin->cancel_btn);
	    gtk_widget_show(optwin->close_btn);
	}

	optwin->freeze_count--;
}

/*
 *	Sets the Options Window as busy or ready.
 */
void EDVGenOptWinSetBusy(edv_gen_opt_win_struct *optwin, gboolean is_busy)
{
#if 0
	GdkCursor *cursor;
	GtkWidget *w;
	edv_core_struct *core;

	if(optwin == NULL)
	    return;

	core = optwin->core;

	w = optwin->toplevel;
	if(w != NULL)
	{
	    if(is_busy)
	    {
		/* Increase busy count */
		optwin->busy_count++;

		/* If already busy then don't change anything */
		if(optwin->busy_count > 1)
		    return;

		cursor = EDVGetCursor(core, EDV_CURSOR_CODE_BUSY);
	    }
	    else
	    {
		/* Reduce busy count */
		optwin->busy_count--;
		if(optwin->busy_count < 0)
		    optwin->busy_count = 0;

		/* If still busy do not change anything */
		if(optwin->busy_count > 0)
		    return;

		cursor = NULL;  /* Use default cursor */
	    }

	    /* Update toplevel window's cursor */
	    if(w->window != NULL)
	    {
		gdk_window_set_cursor(w->window, cursor);
		gdk_flush();
	    }
	}
#endif
}

/*
 *	Checks if the Options Window is mapped.
 */
gboolean EDVGenOptWinIsMapped(edv_gen_opt_win_struct *optwin)
{
	if(optwin == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(optwin->toplevel));
}

/*
 *	Maps the Options Window.
 */
void EDVGenOptWinMap(edv_gen_opt_win_struct *optwin)
{
	if(optwin == NULL)
	    return;

	gtk_widget_show_raise(optwin->toplevel);

	gtk_widget_grab_focus(optwin->ok_btn);
	gtk_widget_grab_default(optwin->ok_btn);
}

/*
 *	Unmaps the Options Window.
 */
void EDVGenOptWinUnmap(edv_gen_opt_win_struct *optwin)
{
	if(optwin == NULL)
	    return;

	gtk_widget_hide(optwin->toplevel);
}

/*
 *	Deletes the Options Window.
 */
void EDVGenOptWinDelete(edv_gen_opt_win_struct *optwin)
{
	if(optwin == NULL)
	    return;

	EDVGenOptWinUnmap(optwin);

	optwin->freeze_count++;

	/* Delete the widget references list */
	if(optwin->wrefs_list != NULL)
	{
	    g_list_foreach(
		optwin->wrefs_list, (GFunc)EDVGenOptWinWRefDelete, NULL
	    );
	    g_list_free(optwin->wrefs_list);
	    optwin->wrefs_list = NULL;
	}

	/* Delete the toplevel widgets of each page */
	if(optwin->page_toplevels_list != NULL)
	{
	    g_list_foreach(
		optwin->page_toplevels_list, (GFunc)gtk_widget_destroy, NULL
	    );
	    g_list_free(optwin->page_toplevels_list);
	    optwin->page_toplevels_list = NULL;
	}

	gtk_widget_destroy(optwin->notebook);
	gtk_widget_destroy(optwin->toplevel);
	gtk_accel_group_unref(optwin->accelgrp);

	g_free(optwin->title);

	optwin->freeze_count--;

	g_free(optwin);
}
