#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "pulist.h"
#include "fprompt.h"
#include "cdialog.h"
#include "fb.h"

#include "edv_types.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "devices_list_win.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_device_misc_48x48.xpm"


/* Callbacks */
static void EDVIconPickerUpdateDisplay(
	edv_device_icon_picker_struct *ei, const gint state
);
static void EDVIconPickerChangeIconCB(
	GtkWidget *widget, gpointer data
);
static void EDVIconPickerPreviousStateCB(
	GtkWidget *widget, gpointer data
);
static void EDVIconPickerNextStateCB(
	GtkWidget *widget, gpointer data
);

static gint EDVDevicesEditWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVDevicesEditWinAnyChangedCB(
	GtkWidget *widget, gpointer data
);

static void EDVDevicesEditWinBrowseDeviceCB(
	GtkWidget *widget, gpointer data
);
static void EDVDevicesEditWinBrowseMountPathCB(
	GtkWidget *widget, gpointer data
);
static void EDVDevicesEditWinBrowseProgramCB(
	GtkWidget *widget, gpointer data
);
static void EDVDeviceEditDlgFSTypeChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void EDVDevicesEditWinOKCB(
	GtkWidget *widget, gpointer data
);
static void EDVDevicesEditWinApplyCB(
	GtkWidget *widget, gpointer data
);
static void EDVDevicesEditWinCancelCB(
	GtkWidget *widget, gpointer data
);

void EDVDevicesEditWinFetchValues(
	edv_device_edit_dlg_struct *d, const gint dev_num
);
void EDVDevicesEditWinDeviceRemovedCB(
	edv_device_edit_dlg_struct *d, gint dev_num
);

static edv_device_icon_picker_struct *EDVDevicesEditIconNew(
	edv_device_edit_dlg_struct *d,
	GtkWidget *parent, const gchar *title,
	const gint icon_width, const gint icon_height,
	const gint btn_width, const gint btn_height
);
static void EDVDevicesEditIconDelete(edv_device_icon_picker_struct *ei);

edv_device_edit_dlg_struct *EDVDevicesEditWinNew(
	edv_core_struct *core, edv_devices_list_win_struct *lw
);
void EDVDevicesEditWinResetHasChanges(
	edv_device_edit_dlg_struct *d, const gboolean has_changes
);
void EDVDevicesEditWinUpdateMenus(edv_device_edit_dlg_struct *d);
gboolean EDVDevicesEditWinIsMapped(edv_device_edit_dlg_struct *d);
void EDVDevicesEditWinMap(edv_device_edit_dlg_struct *d);
void EDVDevicesEditWinUnmap(edv_device_edit_dlg_struct *d);
void EDVDevicesEditWinDelete(edv_device_edit_dlg_struct *d);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define EDITWIN_WIDTH		500
#define EDITWIN_HEIGHT		-1

#if defined(PROG_LANGUAGE_SPANISH)
# define EDITWIN_TITLE		"El Dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
# define EDITWIN_TITLE		"Appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
# define EDITWIN_TITLE		"Vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
# define EDITWIN_TITLE		"Il Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
# define EDITWIN_TITLE		"Apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
# define EDITWIN_TITLE		"O Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
# define EDITWIN_TITLE		"Innretning"
#else
# define EDITWIN_TITLE		"Device"
#endif


/*
 *	Updates the icon displayed in the Icon Picker based on the
 *	given state.
 */
static void EDVIconPickerUpdateDisplay(
	edv_device_icon_picker_struct *ei, const gint state
)
{
	const gchar *label_str = "";
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkWidget *w, *parent;

	if(ei == NULL)
	    return;

	if((state < 0) || (state >= EDV_DEVICE_TOTAL_ICON_STATES))
	    return;

	/* Get references to pixmap and mask pair specified by the
	 * given state
	 */
	pixmap = ei->pixmap[state];
	mask = ei->mask[state];

	/* Get button widget as the parent to create GtkPixmap in */
	parent = ei->button;
	if(parent == NULL)
	    return;

	/* Referenced pixmap valid? */
	if(pixmap != NULL)
	{
	    /* Referenced pixmap is valid, now check if the GtkPixmap
	     * that is to be used to display the pixmap has been
	     * created yet (if not then create a new one)
	     */
	    w = ei->gtk_pixmap;
	    if(w == NULL)
	    {
		ei->gtk_pixmap = w = gtk_pixmap_new(pixmap, mask);
		gtk_container_add(GTK_CONTAINER(parent), w);
		gtk_widget_show(w);
	    }
	    else
	    {
		/* GtkPixmap exists, so just update it */
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
		gtk_widget_show(w);
	    }
	}
	else
	{
	    /* No referenced pixmap, so unmap the the GtkPixmap */
	    w = ei->gtk_pixmap;
	    if(w != NULL)
	    {
		gtk_widget_hide(w);
	    }
	}


	/* Update state label, get the currect label string to display
	 * for the state label based on the given state
	 */
	switch(state)
	{
	  case EDV_DEVICE_ICON_STATE_UNMOUNTED:
	    label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"No Montado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Pas Mont"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nicht Aufgestellt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Non Montato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Niet Bestijgenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"No Montado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ikke Montert"
#else
"Unmounted"
#endif
	    ;
	    break;
	  case EDV_DEVICE_ICON_STATE_SELECTED:
	    label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"Escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ausgewhlt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scelto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Geselecteerde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecionado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Valgt Ut"
#else
"Selected"
#endif
	    ;
	    break;
	  case EDV_DEVICE_ICON_STATE_STANDARD:
	    label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"El Estndar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Norme"
#elif defined(PROG_LANGUAGE_GERMAN)
"Standard"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Norma"
#elif defined(PROG_LANGUAGE_DUTCH)
"Standaard"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Padro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Standard"
#else
"Standard"
#endif
	    ;
	}
	w = ei->state_label;
	if(w != NULL)
	    gtk_label_set_text(GTK_LABEL(w), label_str);

#if 0
	/* Update button sensitivities */
	gtk_widget_set_sensitive(
	    ei->left_btn,
	    (state > 0) ? TRUE : FALSE
	);
	gtk_widget_set_sensitive(
	    ei->right_btn,
	    (state < (EDV_DEVICE_TOTAL_ICON_STATES - 1)) ? TRUE : FALSE
	);
#endif
}

/*
 *	Devices Edit Window change icon callback.
 */
static void EDVIconPickerChangeIconCB(
	GtkWidget *widget, gpointer data
)
{
	gboolean response;
	gint state, nftypes = 0, npaths = 0;
	const gchar *old_path;
	gchar *old_parent_path, **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;
	edv_device_edit_dlg_struct *d;
	edv_device_icon_picker_struct *ei =
	    EDV_DEVICE_ICON_PICKER(data);
	if((ei == NULL) || FileBrowserIsQuery())
	    return;

	/* Get icon state */
	state = (gint)GTK_ADJUSTMENT_GET_VALUE(ei->adj);
	if((state < 0) || (state >= EDV_DEVICE_TOTAL_ICON_STATES))
	    return;

	d = EDV_DEVICE_EDIT_DLG(ei->d);
	toplevel = d->toplevel;
	core = d->core;

	if(d->freeze_count > 0)
	    return;

	/* Get current path to the icon file for this state */
	old_path = ei->path[state];
	if(!STRISEMPTY(old_path))
	    old_parent_path = g_dirname(old_path);
	else
	    old_parent_path = NULL;

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    ".xpm", "X Pixmap"
	);
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Current old_path must be considered invalid at this point */
	old_path = NULL;

	/* Query the user for an image */
	d->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Icono Selecto",
"Selecto",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Icne Privilgie",
"Privilgi",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Erlesenes Abbild",
"Erlesen",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scegliere L'Icona",
"Scegliere",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Uitgezochte Beeld",
"Uitgezocht",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecione Icone",
"Selecione",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvalgt Ikon",
"Utvalgt",
"Kanseller",
#else
"Select Icon",
"Select",
"Cancel",
#endif
	    old_parent_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	d->freeze_count--;

	/* Got user response? */
	if(response)
	{
	    const gchar *new_path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(new_path != NULL)
	    {
		/* Open the new pixmap and mask */
		GdkBitmap *mask;
		GdkPixmap *pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
		    &mask, new_path
		);
		if(pixmap != NULL)
		{
		    /* New pixmap and mask opened successfully */

		    /* Unref the old and set the new pixmap and mask */
		    GDK_PIXMAP_UNREF(ei->pixmap[state]);
		    GDK_BITMAP_UNREF(ei->mask[state]);
		    EDVResizePixmap(
			pixmap, mask,
			ei->icon_width, ei->icon_height,
			&ei->pixmap[state], &ei->mask[state]
		    );
		    GDK_PIXMAP_UNREF(pixmap);
		    GDK_BITMAP_UNREF(mask);

		    /* Set the new icon file path */
		    g_free(ei->path[state]);
		    ei->path[state] = STRDUP(new_path);
		}
		else
		{
                    GDK_PIXMAP_UNREF(ei->pixmap[state]);
		    ei->pixmap[state] = NULL;

                    GDK_BITMAP_UNREF(ei->mask[state]);
		    ei->mask[state] = NULL;

		    g_free(ei->path[state]);
		    ei->path[state] = NULL;
		}

		/* Update the displayed icon and mark has changes */
		EDVIconPickerUpdateDisplay(ei, state);
		EDVDevicesEditWinResetHasChanges(d, TRUE);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	g_free(old_parent_path);
}

/*
 *	Icon Picker previous state callback.
 */
static void EDVIconPickerPreviousStateCB(
	GtkWidget *widget, gpointer data
)
{
	GtkAdjustment *adj;
	edv_device_icon_picker_struct *ei =
	    EDV_DEVICE_ICON_PICKER(data);
	if(ei == NULL)
	    return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

	/* Increment adjustment position */
	if(adj->value > adj->lower)
	    adj->value -= adj->step_increment;
	if(adj->value < adj->lower)
	    adj->value = adj->lower;

	EDVIconPickerUpdateDisplay(ei, (gint)adj->value);
}

/*
 *	Icon Picker next state callback.
 */
static void EDVIconPickerNextStateCB(
	GtkWidget *widget, gpointer data
)
{
	GtkAdjustment *adj;
	edv_device_icon_picker_struct *ei =
	    EDV_DEVICE_ICON_PICKER(data);
	if(ei == NULL)
	    return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

	/* Increment adjustment position */
	if(adj->value < adj->upper)
	    adj->value += adj->step_increment;
	if(adj->value > adj->upper)
	    adj->value = adj->upper;

	EDVIconPickerUpdateDisplay(ei, (gint)adj->value);
}


/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint EDVDevicesEditWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return(TRUE);

	EDVDevicesEditWinCancelCB(widget, d);

	return(TRUE);
}

/*
 *	GtkWidget any "changed" signal callback.
 */
static void EDVDevicesEditWinAnyChangedCB(GtkWidget *widget, gpointer data)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(!d->has_changes)
	    EDVDevicesEditWinResetHasChanges(d, TRUE);

	d->freeze_count--;
}


/*
 *	Devices Edit Window browse device callback.
 */
static void EDVDevicesEditWinBrowseDeviceCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	GtkWidget *toplevel;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	GtkEntry *entry = (GtkEntry *)data;
	if((entry == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the device */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Device",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),		/* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(entry, path);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Devices Edit Window browse mount path callback.
 */
static void EDVDevicesEditWinBrowseMountPathCB(
	GtkWidget *widget, gpointer data
)
{
	gboolean response;
	GtkWidget *toplevel;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	GtkEntry *entry = (GtkEntry *)data;
	if((entry == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the mount path */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Mount Path",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),          /* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(entry, path);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Devices Edit Window browse program callback.
 */
static void EDVDevicesEditWinBrowseProgramCB(
	GtkWidget *widget, gpointer data
)
{
	gboolean response;
	GtkWidget *toplevel;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	GtkEntry *entry = (GtkEntry *)data;
	if((entry == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the program */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),          /* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(entry, path);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}


/*
 *	Filesystem type popup list box changed signal callback.
 */
static void EDVDeviceEditDlgFSTypeChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	EDVDevicesEditWinAnyChangedCB(NULL, data);
}

/*
 *	Devices Edit Window ok callback.
 */
static void EDVDevicesEditWinOKCB(
	GtkWidget *widget, gpointer data
)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	EDVDevicesEditWinApplyCB(NULL, d);
	EDVDevicesEditWinUnmap(d);
}

/*
 *      Devices Edit Window apply callback.
 */
static void EDVDevicesEditWinApplyCB(
	GtkWidget *widget, gpointer data
)
{
	gint dev_num;
	GtkEntry *entry;
	pulistbox_struct *pulistbox;
	edv_device_struct *dev;
	edv_devices_list_win_struct *lw;
	edv_device_icon_picker_struct *ei;
	edv_core_struct *core;
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	lw = d->list_win;
	core = d->core;
	if((lw == NULL) || (core == NULL))
	    return;

	/* Get device index and pointer to structure on the core's
	 * list
	 */
	dev_num = d->dev_num;
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;
	if(dev == NULL)
	    return;


	/* Begin applying values */

	/* Descriptive name */
	entry = (GtkEntry *)d->name_entry;
	if(entry != NULL)
	{
	    g_free(dev->name);
	    dev->name = STRDUP(gtk_entry_get_text(entry));
	}

	/* Device path */
	entry = (GtkEntry *)d->device_path_entry;
	if(entry != NULL)
	{
	    g_free(dev->device_path);
	    dev->device_path = STRDUP(gtk_entry_get_text(entry));
	}

	/* Mount path */
	entry = (GtkEntry *)d->mount_path_entry;
	if(entry != NULL)
	{
	    g_free(dev->mount_path);
	    dev->mount_path = STRDUP(gtk_entry_get_text(entry));
	}

	/* File system type */
	pulistbox = d->fs_type_pulistbox;
	if(pulistbox != NULL)
	{
	    pulist_struct *pulist = PUListBoxGetPUList(pulistbox);
	    const gint i = PUListBoxGetSelected(pulistbox);
	    dev->fs_type = (guint32)PUListGetItemData(pulist, i);
	}

	/* Mount command */
	entry = (GtkEntry *)d->command_mount_entry;
	if(entry != NULL)
	{
	    g_free(dev->command_mount);
	    dev->command_mount = STRDUP(gtk_entry_get_text(entry));
	}
	/* Unmount command */
	entry = (GtkEntry *)d->command_unmount_entry;
	if(entry != NULL)
	{
	    g_free(dev->command_unmount);
	    dev->command_unmount = STRDUP(gtk_entry_get_text(entry));
	}
	/* Eject command */
	entry = (GtkEntry *)d->command_eject_entry;
	if(entry != NULL)
	{
	    g_free(dev->command_eject);
	    dev->command_eject = STRDUP(gtk_entry_get_text(entry));
	}

	/* Check command */
	entry = (GtkEntry *)d->command_check_entry;
	if(entry != NULL)
	{
	    g_free(dev->command_check);
	    dev->command_check = STRDUP(gtk_entry_get_text(entry));
	}
	/* Tools command */
	entry = (GtkEntry *)d->command_tools_entry;
	if(entry != NULL)
	{
	    g_free(dev->command_tools);
	    dev->command_tools = STRDUP(gtk_entry_get_text(entry));
	}
	/* Format command */
	entry = (GtkEntry *)d->command_format_entry;
	if(entry != NULL)
	{
	    g_free(dev->command_format);
	    dev->command_format = STRDUP(gtk_entry_get_text(entry));
	}

	/* No Unmount */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->no_unmount_check))
	    dev->flags |= EDV_DEVICE_NO_UNMOUNT;
	else
	    dev->flags &= ~EDV_DEVICE_NO_UNMOUNT;
	/* Read-Only */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->read_only_check))
	    dev->flags |= EDV_DEVICE_READ_ONLY;
	else
	    dev->flags &= ~EDV_DEVICE_READ_ONLY;
	/* Unlisted */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->unlisted_check))
	    dev->flags |= EDV_DEVICE_UNLISTED;
	else
	    dev->flags &= ~EDV_DEVICE_UNLISTED;
	/* No Scan */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->no_scan_check))
	    dev->flags |= EDV_DEVICE_NO_SCAN;
	else
	    dev->flags &= ~EDV_DEVICE_NO_SCAN;

	/* Begin applying icons file paths */

	/* Small icon */
	ei = d->icon_small;
	if(ei != NULL)
	{
	    gint i;
	    for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	    {
		g_free(dev->small_icon_file[i]);
		dev->small_icon_file[i] = STRDUP(ei->path[i]);
	    }
	}

	/* Medium icon */
	ei = d->icon_medium;
	if(ei != NULL)
	{
	    gint i;
	    for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	    {
		g_free(dev->medium_icon_file[i]);
		dev->medium_icon_file[i] = STRDUP(ei->path[i]);
	    }
	}

	/* Large icon */
	ei = d->icon_large;
	if(ei != NULL)
	{
	    gint i;
	    for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	    {
		g_free(dev->large_icon_file[i]);
		dev->large_icon_file[i] = STRDUP(ei->path[i]);
	    }
	}


	/* Reset has changes marker */
	EDVDevicesEditWinResetHasChanges(d, FALSE);


	/* Re-realize the device */
	EDVDeviceRealize(dev, TRUE);

	/* Update all device mount states and stats */
	EDVDevicesListUpdateMountStates(
	    core->device, core->total_devices
	);
	EDVDevicesListUpdateStats(
	    core->device, core->total_devices
	);

	/* Send device modified signal to all of endeavour's resources */
	EDVDeviceModifiedEmit(core, dev_num, dev);
}

/*
 *	Devices Edit Window cancel or close callback.
 */
static void EDVDevicesEditWinCancelCB(
	GtkWidget *widget, gpointer data
)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVDevicesEditWinUnmap(d);
}

/*
 *	Loads the device values to the Devices Edit Window.
 */
void EDVDevicesEditWinFetchValues(
	edv_device_edit_dlg_struct *d, const gint dev_num
)
{
	GtkEntry *entry;
	pulistbox_struct *pulistbox;
	edv_device_struct *dev;
	edv_core_struct *core;
	edv_device_icon_picker_struct *ei;

	if(d == NULL)
	    return;

	core = d->core;
	if(core == NULL)
	    return;

	/* Get pointer to device structure on core's devices list */
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;
	if(dev == NULL)
	    return;


	/* Begin fetching values */

	/* Set device index on core structure from the given dev_num */
	d->dev_num = dev_num;

	/* Descriptive name */
	entry = (GtkEntry *)d->name_entry;
	if(entry != NULL)
	{
	    gtk_entry_set_text(
		entry,
		(dev->name != NULL) ? dev->name : ""
	    );
	}

	/* Device path */
	entry = (GtkEntry *)d->device_path_entry;
	if(entry != NULL)
	{
	    gtk_entry_set_text(
		entry,
		(dev->device_path != NULL) ? dev->device_path : ""
	    );
	}

	/* Mount path */
	entry = (GtkEntry *)d->mount_path_entry;
	if(entry != NULL)
	{
	    gtk_entry_set_text(
		entry,
		(dev->mount_path != NULL) ? dev->mount_path : ""
	    );
	}

	/* File system type */
	pulistbox = d->fs_type_pulistbox;
	if(pulistbox != NULL)
	{
	    const guint32 fs_type = dev->fs_type;
	    pulist_struct *pulist = PUListBoxGetPUList(pulistbox);
	    const gint m = PUListGetTotalItems(pulist);
	    gint i;
	    for(i = 0; i < m; i++)
	    {
		if(fs_type == (guint32)PUListGetItemData(pulist, i))
		{
		    PUListBoxSelect(pulistbox, i);
		    break;
		}
	    }
	}

	/* Mount command */
	entry = (GtkEntry *)d->command_mount_entry;
	if(entry != NULL)
	    gtk_entry_set_text(
		entry,
		(dev->command_mount != NULL) ? dev->command_mount : ""
	    );
	/* Unmount command */
	entry = (GtkEntry *)d->command_unmount_entry;
	if(entry != NULL)
	    gtk_entry_set_text(
		entry,
		(dev->command_unmount != NULL) ? dev->command_unmount : ""
	    );
	/* Eject command */
	entry = (GtkEntry *)d->command_eject_entry;
	if(entry != NULL)
	    gtk_entry_set_text(
		entry,
		(dev->command_eject != NULL) ? dev->command_eject : ""
	    );

	/* Check command */
	entry = (GtkEntry *)d->command_check_entry;
	if(entry != NULL)
	    gtk_entry_set_text(
		entry,
		(dev->command_check != NULL) ? dev->command_check : ""
	    );
	/* Tools command */
	entry = (GtkEntry *)d->command_tools_entry;
	if(entry != NULL)
	    gtk_entry_set_text(
		entry,
		(dev->command_tools != NULL) ? dev->command_tools : ""
	    );
	/* Format command */
	entry = (GtkEntry *)d->command_format_entry;
	if(entry != NULL)
	    gtk_entry_set_text(
		entry,
		(dev->command_format != NULL) ? dev->command_format : ""
	    );

	/* No Unmount */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->no_unmount_check,
	    EDV_DEVICE_IS_NO_UNMOUNT(dev)
	);
	/* Read-Only */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->read_only_check,
	    EDV_DEVICE_IS_READ_ONLY(dev)
	);
	/* Unlisted */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->unlisted_check,
	    EDV_DEVICE_IS_UNLISTED(dev)
	);
	/* No Scan */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->no_scan_check,
	    EDV_DEVICE_IS_NO_SCAN(dev)
	);

	/* Begin loading the icons */

#define DO_LOAD_ICON	{				\
 gint i;						\
 for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++) {	\
  /* Unref pixmap and mask on the Icon Picker */	\
  if(ei->pixmap[i] != NULL)				\
   gdk_pixmap_unref(ei->pixmap[i]);			\
  if(ei->mask[i] != NULL)				\
   gdk_bitmap_unref(ei->mask[i]);			\
							\
  /* Increase ref count on new pixmap and mask and set	\
   * if to the Icon Picker				\
   */							\
  if(pixmap[i] != NULL)					\
   gdk_pixmap_ref(pixmap[i]);				\
  if(mask[i] != NULL)					\
   gdk_bitmap_ref(mask[i]);				\
							\
  ei->pixmap[i] = pixmap[i];				\
  ei->mask[i] = mask[i];				\
							\
  /* Set icon path */					\
  g_free(ei->path[i]);					\
  ei->path[i] = STRDUP(path[i]);			\
 }							\
}
	/* Small icons */
	ei = d->icon_small;
	if(ei != NULL)
	{
	    GdkPixmap **pixmap = dev->small_pixmap;
	    GdkBitmap **mask = dev->small_mask;
	    gchar **path = dev->small_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0f;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVIconPickerUpdateDisplay(
		    ei, (gint)ei->adj->value
		);
	}

	/* Medium icons */
	ei = d->icon_medium;
	if(ei != NULL)
	{
	    GdkPixmap **pixmap = dev->medium_pixmap;
	    GdkBitmap **mask = dev->medium_mask;
	    gchar **path = dev->medium_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0f;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVIconPickerUpdateDisplay(
		    ei, (gint)ei->adj->value
		);
	}

	/* Large icons */
	ei = d->icon_large;
	if(ei != NULL)
	{
	    GdkPixmap **pixmap = dev->large_pixmap;
	    GdkBitmap **mask = dev->large_mask;
	    gchar **path = dev->large_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0f;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVIconPickerUpdateDisplay(
		    ei, (gint)ei->adj->value
		);
	}
#undef DO_LOAD_ICON

}


/*
 *	Device removed signal callback.
 */
void EDVDevicesEditWinDeviceRemovedCB(
	edv_device_edit_dlg_struct *d, gint dev_num
)
{
	if(d == NULL)
	    return;

	if(d->dev_num != dev_num)
	    return;

	d->dev_num = -1;
	EDVDevicesEditWinUnmap(d);
}


/*
 *	Creates a new Icon Picker.
 */
static edv_device_icon_picker_struct *EDVDevicesEditIconNew(
	edv_device_edit_dlg_struct *d,
	GtkWidget *parent, const gchar *title,
	const gint icon_width, const gint icon_height,
	const gint btn_width, const gint btn_height
)
{
	gint bw = 20, bh = 20;
	GtkAdjustment *adj;
	GtkWidget *w, *parent2, *parent3, *parent4;
	edv_device_icon_picker_struct *ei;

	if(d == NULL)
	    return(NULL);

	ei = EDV_DEVICE_ICON_PICKER(
	    g_malloc0(sizeof(edv_device_icon_picker_struct))
	);
	if(ei == NULL)
	    return(NULL);

	ei->d = d;
	ei->icon_width = icon_width;
	ei->icon_height = icon_height;

	/* Create adjustment for keeping track of displayed icon */
	ei->adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,		/* Value */
	    0.0f,		/* Lower */
	    (gfloat)EDV_DEVICE_TOTAL_ICON_STATES - 1.0f,	/* Upper */
	    1.0f, 1.0f,		/* Step and page increments */
	    1.0f		/* Page size */
	);

	/* Toplevel GtkVBox */
	ei->toplevel = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Title GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	w = gtk_label_new(title);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Button GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Create button */
	ei->button = w = gtk_button_new();
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVIconPickerChangeIconCB), ei
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
#if 0
	ei->gtk_pixmap = w = gtk_pixmap_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
#endif


	/* Hbox for buttons and labels */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Left button */
	ei->left_btn = w = gtk_button_new();
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVIconPickerPreviousStateCB), ei
	);
	gtk_widget_set_usize(w, bw, bh);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Arrow */
	w = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);

	/* Frame for state label */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* State label */
	ei->state_label = w = gtk_label_new("");
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);

	/* Right button */
	ei->right_btn = w = gtk_button_new();
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVIconPickerNextStateCB), ei
	);
	gtk_widget_set_usize(w, bw, bh);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Arrow */
	w = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);


	return(ei);
}

/*
 *	Deletes the Icon Picker.
 */
static void EDVDevicesEditIconDelete(edv_device_icon_picker_struct *ei)
{
	gint i;

	if(ei == NULL)
	    return;

	/* Unref all pixmap and mask pairs */
	for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	{
	    GDK_PIXMAP_UNREF(ei->pixmap[i])
	    GDK_BITMAP_UNREF(ei->mask[i])
	    g_free(ei->path[i]);
	}

	/* Begin destroying widgets */
	GTK_WIDGET_DESTROY(ei->gtk_pixmap)
	GTK_WIDGET_DESTROY(ei->button)
	GTK_WIDGET_DESTROY(ei->left_btn)
	GTK_WIDGET_DESTROY(ei->right_btn)
	GTK_WIDGET_DESTROY(ei->state_label)
	GTK_WIDGET_DESTROY(ei->toplevel)

	GTK_OBJECT_UNREF(ei->adj)

	g_free(ei);
}


/*
 *	Creates a new Devices Edit Window.
 */
edv_device_edit_dlg_struct *EDVDevicesEditWinNew(
	edv_core_struct *core, edv_devices_list_win_struct *lw
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint i, n;
	gpointer entry_rtn, browse_rtn;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*parent5,
			*toplevel;
	pulist_struct *pulist;
	pulistbox_struct *pulistbox;
	const edv_fs_type_struct *fs_type_ptr;
	const edv_fs_type_struct fs_type_list[] = EDV_FS_TYPE_LIST;
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(
	    g_malloc0(sizeof(edv_device_edit_dlg_struct))
	);
	if((d == NULL) || (core == NULL))
	{
	    g_free(d);
	    return(NULL);
	}

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->busy_count = 0;
	d->freeze_count = 0;
	d->has_changes = 0;
	d->title = STRDUP(EDITWIN_TITLE);
	d->core = core;
	d->list_win = lw;
	d->dev_num = -1;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(w, EDITWIN_WIDTH, EDITWIN_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(w), d->title);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_device_misc_48x48_xpm);
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* GtkHBox to make two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Name entry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Nombre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Nom"
#elif defined(PROG_LANGUAGE_GERMAN)
"Name"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Nome"
#elif defined(PROG_LANGUAGE_DUTCH)
"Naam"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Nome"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Navn"
#else
"Name"
#endif
	    ":", NULL, &entry_rtn
	);
	d->name_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el nombre de este artefacto (este valor hace slo sea usado\
 para propsitos de despliegue)"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le nom de cet appareil (cette valeur sera seulement utilise\
 pour les buts d'exposition)"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den namen dieser vorrichtung ein (dieser wert wird nur\
 fr ausstellung zwecke benutzt werden)"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il nome di questo congegno (questo valore sar soltanto usato\
 per gli scopi di mostra)"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de (deze enige waarde zal voor tentoonstelling doelen gebruikt\
 worden) naam van deze apparaat binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o nome deste artifcio (este valor s ser usado para propsitos\
 de exposio)"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i navnet av denne innretningen (denne verdien brukt bare for\
 utstillingsforml)"
#else
"Enter the name of this device (this value will only be used for\
 display purposes)"
#endif
	);


	/* Device path entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"Sendero Dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Sentier D'appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gertepfad"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Sentiero Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat Pad"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Caminho Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretnings Sti"
#else
"Device Path"
#endif
	    ":", NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->device_path_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	EDVEntrySetCompletePath(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el sendero al objeto del artefacto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le objeto d'al de sendero d'el de Entre del artefacto"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den pfad zum gerteobjekt ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il sentiero all'oggetto di congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het pad aan het apparaat voorwerp binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o caminho ao objeto de artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i stien til innretnings objektet"
#else
"Enter the path to the device node"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseDeviceCB), entry_rtn
	);


	/* Mount path entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"Monte Sendero"
#elif defined(PROG_LANGUAGE_FRENCH)
"Monter Sentier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Stellen Pfad"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montare Sentiero"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bestijg Pad"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Monte Caminho"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Monter Sti"
#else
"Mount Path"
#endif
	    ":", NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->mount_path_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	EDVEntrySetCompletePath(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el sendero al objeto de la gua que este artefacto\
 deber ser montado en"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le sentier  l'objet d'annuaire que cet appareil va\
 tre mont sur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den pfad zum verzeichnis objekt, das diese vorrichtung auf\
 aufgestellt werden soll ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il sentiero all'oggetto di elenco che questo congegno \
 essere montato su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het pad aan het gids voorwerp binnen dat deze apparaat op bestegen\
 worden zal"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o caminho ao objeto de guia que este artifcio  ser montado\
 em"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i stien til katalogobjektet som denne innretningen er montert\
 p"
#else
"Enter the path to the directory that this device is to\
 be mounted on"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseMountPathCB), entry_rtn
	);


	/* Filesystem GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Archivesistema"
#elif defined(PROG_LANGUAGE_FRENCH)
"Systmeclassement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dateisystem"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Schedaresistema"
#elif defined(PROG_LANGUAGE_DUTCH)
"Archiveersysteem"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivesistema"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiversystem"
#else
"Filesystem"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;


	/* FS Type GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Label */
	w = gtk_label_new("Filesystem Type:");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Popup list box */
	d->fs_type_pulistbox = pulistbox = PUListBoxNew(
	    parent5, -1, -1
	);
	pulist = PUListBoxGetPUList(pulistbox);
        if(core->run_flags & EDV_RUN_SAFE_MODE)
            PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	PUListBoxSetChangedCB(
	    pulistbox,
	    EDVDeviceEditDlgFSTypeChangedCB, d
	);
	PUListBoxMap(pulistbox);   
	for(i = 0; fs_type_list[i].fs_type; i++)
	{
	    fs_type_ptr = &fs_type_list[i];
	    n = PUListAddItem(
		pulist,
		fs_type_ptr->name
	    );
	    PUListSetItemData(
		pulist, n,
		(gpointer)fs_type_ptr->fs_type
	    );
	}
	GUISetWidgetTip(
	    pulistbox->da,
#if defined(PROG_LANGUAGE_SPANISH)
"Escoja el de tipo filesystem de los medios usados con este\
 artefacto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisir le type de filesystem du presse utilis avec cet\
 appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Whlen sie den filesystem typ der medien hat benutzt mit dieser\
 vorrichtung aus"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scegliere il tipo di filesystem della stampa usata con questo\
 congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Selecteer het filesystem type van de media gebruikte met deze\
 apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecione o filesystem tipo da imprensa usada com este\
 artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Velg ut filesystem type av mediene som brukt med denne\
 innretningen"
#else
"Select the filesystem type of the media used with this device"
#endif
	);


	/* Commands Frame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Las rdenes"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ordres"
#elif defined(PROG_LANGUAGE_GERMAN)
"Befehle"
#elif defined(PROG_LANGUAGE_ITALIAN)
"I Comandi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevelen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os Comandos"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommandoer"
#else
"Commands"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Mount command entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Monte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Mont"
#elif defined(PROG_LANGUAGE_GERMAN)
"Untersatz"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Monte"
#elif defined(PROG_LANGUAGE_DUTCH)
"Berg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Monte"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fotstykke"
#else
"Mount"
#endif
	    ":", NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_mount_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the command that will be executed to mount\
 this device (or leave it blank if it is not mountable)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre la orden que ser ejecutada al mount este\
 artefacto (o le sale blanco si no es mountable)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer l'ordre qui sera excut  moun cet\
 appareil (ou le part le vide si ne ce pas est mountable)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseProgramCB), entry_rtn
	);

	/* Unmount command entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Unmount:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "El Unmoute:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Unmont:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_unmount_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to unmount\
 this device (or leave it blank if it is not unmountable)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que ser ejecutada al unmount este\
 artefacto (o le sale blanco si no es unmountable)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrer l'ordre qui sera excut  unmount cet\
 appareil (ou le part le vide si ne ce pas est unmountable)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseProgramCB), entry_rtn
	);

	/* Eject command entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Eject:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Expulse:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Ejecter:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_eject_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to eject\
 the media from the device (or leave it blank if the device does not\
 support ejecting)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que ser ejecutada a eject los\
 medios del artefacto (o le sale blanco si el artefacto hace\
 expulsar de apoyo de not)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrer l'ordre qui sera excut  eject le presse\
 de l'appareil (ou le part le vide si l'appareil fait\
 l'jecter de soutien de not)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseProgramCB), entry_rtn
	);


	/* Check command entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Check:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Cheque:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Chque:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_check_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to run the\
 file system check program for this device (or leave it blank if\
 you do not have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que se ejecutar para correr el\
 programa de cheque de sistema de archivo para este artefacto\
 (o le sale blanco si usted no tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrez l'ordre qui sera excut courir le\
 programme de chque de systme de fichier pour cet appareil\
 (ou le part le vide si vous ne celui avez pas)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseProgramCB), entry_rtn
	);

	/* Tools command entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Tools:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Las Herramientas:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Outils:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_tools_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to run the\
 tools program for this device (or leave it blank if you do not\
 have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que se ejecutar para correr el\
 programa de herramientas para este artefacto (o le sale\
 blanco si usted no tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrer l'ordre qui sera excut courir le\
 programme d'outils pour cet appareil (ou le part le vide\
 si vous ne celui avez pas)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseProgramCB), entry_rtn
	);

	/* Format command entry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Format:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Formato:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Format:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_format_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the command that will be executed to run the\
 format program for this device (or leave it blank if you do not\
 have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre la orden que se ejecutar para correr the el\
 programa de formato para este artefacto (o le sale\
 blanco si usted hace not tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer l'ordre qui sera excut courir le programme\
 de format pour cet appareil (ou le part le vide si\
 vous ne celui avez pas)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinBrowseProgramCB), entry_rtn
	);


	/* Options GtkFrame */
	w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Options"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Las Opciones"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Options"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* No Unmount GtkCheckButton */
	d->no_unmount_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
	    "No Unmount"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Ningn Unmount"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Aucun Unmount"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to locally mark the device as \"fixed\", so that it cannot\
 be mounted or unmounted"
	);
	gtk_widget_show(w);

	/* Read-Only GtkCheckButton */
	d->read_only_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Read-Only"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Lea Slo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Lire Seulement"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to always mount the device as read only"
	);
	gtk_widget_show(w);

	/* Unlisted GtkCheckButton */
	d->unlisted_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Unlisted"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Unlisted"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Unlisted"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this if you do not want this device to be listed on any\
 of the \"brief\" device listings (such as the devices popup list)"
	);
	gtk_widget_show(w);

	/* No Scan GtkCheckButton */
	d->no_scan_check = w = gtk_check_button_new_with_label(
	    "No Scan"
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to avoid scanning the mount path for objects\
 unless you explicitly select it (useful for\
 conserving bandwidth on NFS filesystems)"
	);
	gtk_widget_show(w);



	/* Right side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icons frame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Iconos"
#elif defined(PROG_LANGUAGE_FRENCH)
"Icnes"
#elif defined(PROG_LANGUAGE_GERMAN)
"Abbilder"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le Icone"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beelden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os cones"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ikoner"
#else
"Icons"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;


	/* Create icon structures */
	d->icon_large = EDVDevicesEditIconNew(
	    d, parent4,
#if defined(PROG_LANGUAGE_SPANISH)
"Grande"
#elif defined(PROG_LANGUAGE_FRENCH)
"Grand"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gro"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Grande"
#elif defined(PROG_LANGUAGE_DUTCH)
"Groot"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Grande"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stor"
#else
"Large"
#endif
	    " 48x48:",
	    48, 48,
	    54, 54
	);
	d->icon_medium = EDVDevicesEditIconNew(
	    d, parent4,
#if defined(PROG_LANGUAGE_SPANISH)
"El Medio"
#elif defined(PROG_LANGUAGE_FRENCH)
"Milieu"
#elif defined(PROG_LANGUAGE_GERMAN)
"Medium"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Mezzo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Middel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mdio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Middel"
#else
"Medium"
#endif
	    " 32x32:",
	    32, 32,
	    38, 38
	);
	d->icon_small = EDVDevicesEditIconNew(
	    d, parent4,
#if defined(PROG_LANGUAGE_SPANISH)
"Pequeo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Petit"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Piccolo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klein"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pequeno"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Liten"
#else
"Small"
#endif
	    " 20x20:",
	    20, 20,
	    26, 26
	);


	/* Horizontal separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);

	/* Apply button */
	d->apply_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"S'Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinApplyCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);
	gtk_widget_show(w);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDevicesEditWinCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);


	EDVDevicesEditWinUpdateMenus(d);


	return(d);
}

/*
 *	Resets the Devices Edit Window's has changes marker.
 */
void EDVDevicesEditWinResetHasChanges(
	edv_device_edit_dlg_struct *d, const gboolean has_changes
)
{
	if(d == NULL)
	    return;

	if(d->has_changes == has_changes)
	    return;

	d->has_changes = has_changes;
	EDVDevicesEditWinUpdateMenus(d);
}

/*
 *	Updates the Devices Edit Window's widgets to reflect current
 *	values.
 */
void EDVDevicesEditWinUpdateMenus(edv_device_edit_dlg_struct *d)
{
	gboolean        sensitive,
			has_changes;
	gint dev_num;
	gchar *s;
	const edv_device_struct *dev;
	edv_core_struct *core;

	if(d == NULL)
	    return;

	d->freeze_count++;

	core = d->core;

	has_changes = d->has_changes;
	dev_num = d->dev_num;
	if((dev_num >= 0) && (dev_num < core->total_devices))
	    dev = core->device[dev_num];
	else
	    dev = NULL;

	/* Title */
	s = g_strconcat(
	    (d->title != NULL) ? d->title : "",
	    (has_changes) ? " (*)" : "",
	    NULL
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = has_changes ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->ok_btn, sensitive);
	gtk_widget_set_sensitive(d->apply_btn, sensitive);
	sensitive = has_changes;
	gtk_widget_set_sensitive(d->cancel_btn, sensitive);
	sensitive = has_changes ? FALSE : TRUE;      
	gtk_widget_set_sensitive(d->close_btn, sensitive);
	if(has_changes)                                  
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_show(d->cancel_btn);
	    gtk_widget_hide(d->close_btn);
	}
	else
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_hide(d->cancel_btn);
	    gtk_widget_show(d->close_btn);
	}

	d->freeze_count--;
}

/*
 *	Checks if the Devices Edit Window.
 */
gboolean EDVDevicesEditWinIsMapped(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
            return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the Devices Edit Window.
 */
void EDVDevicesEditWinMap(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);

	gtk_widget_grab_focus(d->name_entry);
}

/*
 *	Unmaps the Devices Edit Window.
 */
void EDVDevicesEditWinUnmap(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the Devices List Window.
 */
void EDVDevicesEditWinDelete(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	EDVDevicesEditWinUnmap(d);

	d->freeze_count++;

	EDVDevicesEditIconDelete(d->icon_small);
	EDVDevicesEditIconDelete(d->icon_medium);
	EDVDevicesEditIconDelete(d->icon_large);

	PUListBoxDelete(d->fs_type_pulistbox);

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	g_free(d->title);

	d->freeze_count--;

	g_free(d);
}
