#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "toolbar.h"
#include "cdialog.h"

#include "edv_types.h"
#include "edv_list_cb.h"
#include "cfg.h"
#include "edv_status_bar.h"
#include "archiver.h"
#include "archiver_cb.h"
#include "archiver_op_cb.h"
#include "archiver_contents_list.h"
#include "archiver_dnd.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_hsep_20x20.xpm"
#include "images/icon_new_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_trace_20x20.xpm"
#include "images/icon_run_20x20.xpm"
#include "images/icon_terminal2_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_archive_add_20x20.xpm"
#include "images/icon_archive_extract_20x20.xpm"
#include "images/icon_aid_20x20.xpm"
#include "images/icon_archive_check_20x20.xpm"
#include "images/icon_security_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_select_all_20x20.xpm"
#include "images/icon_unselect_all_20x20.xpm"
#include "images/icon_invert_selection_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_info_20x20.xpm"
#include "images/icon_mimetypes_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_properties2_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_wildcards_20x20.xpm"
#include "images/icon_options2_20x20.xpm"
#include "images/icon_customize_20x20.xpm"
#include "images/icon_insecure_20x20.xpm"
#include "images/icon_secure_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"

#include "images/endeavour_20x20.xpm"
#include "images/image_browser_20x20.xpm"
#include "images/archiver_20x20.xpm"
#include "images/icon_trash_20x20.xpm"
#include "images/icon_trash_empty_20x20.xpm"
#include "images/icon_purge_20x20.xpm"
#include "images/archiver_48x48.xpm"


/* Utilities */
edv_archiver_opid_struct *EDVArchiverMatchOPID(
	edv_archiver_struct *archiver, edv_archiver_op op
);

/* Title */
void EDVArchiverSetTitle(
	edv_archiver_struct *archiver, const gchar *path
);

/* Location */
gchar *EDVArchiverCurrentLocation(edv_archiver_struct *archiver);
void EDVArchiverSetLocation(
	edv_archiver_struct *archiver,
	const gchar *path, const gboolean record_history
);
void EDVArchiverUpdateLocationIcon(
	edv_archiver_struct *archiver, const gchar *path
);

/* Password */
gchar *EDVArchiverCurrentPassword(edv_archiver_struct *archiver);
void EDVArchiverSetPassword(
	edv_archiver_struct *archiver, const gchar *password
);

/* Regenerate Style, Menus, and Tool Bar */
void EDVArchiverAccelkeysRegenerate(edv_archiver_struct *archiver);
void EDVArchiverToolBarRegenerate(edv_archiver_struct *archiver);
void EDVArchiverContentsMenuRegenerate(edv_archiver_struct *archiver);

/* Sync Configuration */
void EDVArchiverSyncConfiguration(edv_archiver_struct *archiver);

/* Open Archive */
void EDVArchiverOpenArchive(
	edv_archiver_struct *archiver, const gchar *arch_obj,
	const gchar *password
);

/* Operation IDs */
GList *EDVArchiverOPIDListNew(edv_archiver_struct *archiver);
void EDVArchiverOPIDListDelete(GList *glist);

/* Accel Keys */
cfg_accelkey_list_struct *EDVArchiverAccelkeysListNewDefault(void);

/* Archiver */
static void EDVArchiverCreateMenuBar(
	edv_archiver_struct *archiver, GtkWidget *parent
);
edv_archiver_struct *EDVArchiverNew(edv_core_struct *core);
void EDVArchiverSyncData(edv_archiver_struct *archiver);
void EDVArchiverUpdateMenus(edv_archiver_struct *archiver);
void EDVArchiverUpdateContentsHeadings(
	edv_archiver_struct *archiver,
	const gint *contents_heading, gint total_contents_headings
);
void EDVArchiverSetBusy(edv_archiver_struct *archiver, gboolean busy);
gboolean EDVArchiverIsMapped(edv_archiver_struct *archiver);
void EDVArchiverMap(edv_archiver_struct *archiver);
void EDVArchiverUnmap(edv_archiver_struct *archiver);
void EDVArchiverDelete(edv_archiver_struct *archiver);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the Archiver Operation ID that matches the given
 *	operation.
 */
edv_archiver_opid_struct *EDVArchiverMatchOPID(
	edv_archiver_struct *archiver, edv_archiver_op op
)
{
	GList *glist;
	edv_archiver_opid_struct *opid;

	if(archiver == NULL)
	    return(NULL);

	for(glist = archiver->opid_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    opid = EDV_ARCHIVER_OPID(glist->data);
	    if(opid == NULL)
		continue;

	    if(opid->op == op)
		return(opid);
	}

	return(NULL);
}


/*
 *	Sets the Archiver's title.
 */
void EDVArchiverSetTitle(
	edv_archiver_struct *archiver, const gchar *path
)
{
	gboolean include_user_name, include_location;
	const gchar *prog_name = "Archiver";
	gchar *title;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	w = archiver->toplevel;
	core = archiver->core;
	if((w == NULL) || (core == NULL))
	    return;

	cfg_list = core->cfg_list;

	/* Use user defined title? */
	if(!STRISEMPTY(core->title))
	    prog_name = core->title;

	/* Get title bar customization */
	include_user_name = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_TITLE_WITH_USER);
	include_location = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_TITLE_WITH_LOCATION);

	/* Format new title */
	if(include_user_name && include_location &&
	   (path != NULL)
	)
	    title = g_strdup_printf(
		"%s: %s %s",
		prog_name, core->effective_user_id_host_str,
		path
	    );
	else if(include_location && (path != NULL))
	    title = g_strdup_printf(
		"%s: %s",
		prog_name, path
	    );
	else if(include_user_name)
	    title = g_strdup_printf(
		"%s: %s",
		prog_name, core->effective_user_id_host_str
	    );
	else
	    title = STRDUP(prog_name);

	/* Set new title */
	gtk_window_set_title(GTK_WINDOW(w), title);

	g_free(title);
}

/*
 *	Returns the Archiver's current location.
 */
gchar *EDVArchiverCurrentLocation(edv_archiver_struct *archiver)
{
	GtkCombo *combo = (GtkCombo *)((archiver != NULL) ?
	    archiver->location_combo : NULL
	);
	return((combo != NULL) ?
	    gtk_entry_get_text(GTK_ENTRY(combo->entry)) : NULL
	);
}

/*
 *	Sets the Archiver's location.
 *
 *	If record_history is TRUE then the given path will also be
 *	recorded on the locations history.
 */
void EDVArchiverSetLocation(
	edv_archiver_struct *archiver,
	const gchar *path, const gboolean record_history
)
{
	const gchar *s;
	gchar *new_path;
	GtkEntry *entry;
	GtkCombo *combo = (GtkCombo *)((archiver != NULL) ?
	    archiver->location_combo : NULL
	);
	if((combo == NULL) || STRISEMPTY(path))
	    return;

	entry = GTK_ENTRY(combo->entry);

	/* Get new path */
	new_path = STRDUP(path);

	/* Simplify the new path */
	EDVSimplifyPath(new_path);

	/* No change? */
	s = gtk_entry_get_text(entry);
	if((s != NULL) ? !strcmp(s, new_path) : FALSE)
	{
	    g_free(new_path);
	    return;
	}

	/* Record new location on the combo list? */
	if(record_history)
	    GUIComboAddItem(GTK_WIDGET(combo), new_path);

	/* Set new location */
	gtk_entry_set_text(entry, new_path);
	gtk_entry_set_position(entry, -1);

	g_free(new_path);
}

/*
 *	Updates the Archiver Location Bar's icon.
 *
 *	If the given path is NULL then the icon will be set as unknown.
 */
void EDVArchiverUpdateLocationIcon(
	edv_archiver_struct *archiver, const gchar *path
)
{
	edv_object_struct *obj = NULL;
	GdkPixmap *pixmap, *pixmap_hid;
	GdkBitmap *mask, *mask_hid;
	GtkWidget *parent, *w;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	core = archiver->core;
	parent = archiver->location_icon_fixed;
	if((core == NULL) || (parent == NULL))
	    return;

	if(!STRISEMPTY(path))
	{
	    struct stat lstat_buf;
	    if(!lstat(path, &lstat_buf))
	    {
		obj = EDVObjectNew();
		EDVObjectSetPath(obj, path);
		EDVObjectSetStat(obj, &lstat_buf);
		EDVObjectUpdateLinkFlags(obj);
	    }
	}
	if(obj == NULL)
	{
	    obj = EDVObjectNew();
	    if(obj != NULL)
	    {
		obj->type = EDV_OBJECT_TYPE_FILE;
		obj->permissions = 0x00000000;
		if(!STRISEMPTY(path))
		{
		    struct stat lstat_buf;
		    EDVObjectSetPath(obj, path);
		    if(!lstat(path, &lstat_buf))
		    {
			EDVObjectSetStat(obj, &lstat_buf);
			EDVObjectUpdateLinkFlags(obj);
		    }
		}
	    }
	}

	/* Get pixmap and mask */
	EDVMatchObjectIcon(
	    core->device, core->total_devices,
	    core->mimetype, core->total_mimetypes,
	    obj->type,
	    obj->full_path,
	    EDV_OBJECT_IS_LINK_VALID(obj),
	    obj->permissions,
	    0,			/* Small icons */
	    &pixmap, &mask,
	    NULL, NULL,
	    NULL, NULL,
	    &pixmap_hid, &mask_hid
	);
	/* Hidden */
	if(EDVIsObjectHidden(obj))
	{
	    if(pixmap_hid != NULL)
	    {
		pixmap = pixmap_hid;
		mask = mask_hid;
	    }
	}
	if(pixmap == NULL)
	    return;

	GDK_PIXMAP_REF(pixmap);
	GDK_BITMAP_REF(mask);

	/* Create new icon pixmap widget as needed */
	w = archiver->location_icon_pm;
	if(w == NULL)
	{
	    archiver->location_icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
	    gtk_widget_show(w);
	}
	else
	{
	    gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
	}

	EDVObjectDelete(obj);

	GDK_PIXMAP_UNREF(pixmap);
	GDK_BITMAP_UNREF(mask);
}

/*
 *	Returns the Archiver's current password.
 */
gchar *EDVArchiverCurrentPassword(edv_archiver_struct *archiver)
{
	GtkEntry *entry = (GtkEntry *)((archiver != NULL) ?
	    archiver->location_password_entry : NULL
	);
	return((entry != NULL) ? gtk_entry_get_text(entry) : NULL);
}

/*
 *	Sets the Archiver's password.
 */
void EDVArchiverSetPassword(
	edv_archiver_struct *archiver, const gchar *password
)
{
	GtkEntry *entry;
	gchar *new_password;

	if((archiver == NULL) || (password == NULL))
	    return;

	entry = (GtkEntry *)archiver->location_password_entry;
	if(entry == NULL)
	    return;

	/* Copy password */
	new_password = STRDUP(password);

	/* Set new password */
	gtk_entry_set_text(entry, new_password);
	gtk_entry_set_position(entry, -1);

	g_free(new_password);
}

/* 
 *	Resets the Archiver's accelkeys displayed on the widgets.
 */
void EDVArchiverAccelkeysRegenerate(edv_archiver_struct *archiver)
{
	GtkAccelGroup *accelgrp = NULL;
	cfg_accelkey_list_struct *ak_list;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	core = archiver->core;
	if(core == NULL)
	    return;
		   
	cfg_list = core->cfg_list;

	ak_list = EDV_GET_ACCELKEY_LIST(EDV_CFG_PARM_ARCHIVER_ACCELERATOR_KEYS);
	if(ak_list == NULL)
	    return;

#define UPDATE_ACCELKEY(_w_,_opid_)	{	\
 if(((_w_) != NULL) && ((_opid_) > 0)) {	\
  GList *glist;					\
  cfg_accelkey_struct *ak;			\
						\
  for(glist = ak_list->list;			\
      glist != NULL;				\
      glist = g_list_next(glist)		\
  ) {						\
   ak = CFG_ACCELKEY(glist->data);		\
   if(ak == NULL)				\
    continue;					\
						\
   if(ak->opid == (gint)(_opid_)) {		\
    GUIMenuItemSetAccelKey(			\
     (_w_), accelgrp, ak->key, ak->modifiers	\
    );						\
    break;					\
   }						\
  }						\
 }						\
}

	UPDATE_ACCELKEY(
	    archiver->new_mi,
	    EDV_ARCHIVER_OP_NEW
	);
	UPDATE_ACCELKEY(
	    archiver->open_mi,
	    EDV_ARCHIVER_OP_OPEN
	);
	UPDATE_ACCELKEY(
	    archiver->sync_disks_mi,
	    EDV_ARCHIVER_OP_SYNC_DISKS
	);
	UPDATE_ACCELKEY(
	    archiver->history_mi,
	    EDV_ARCHIVER_OP_HISTORY
	);
	UPDATE_ACCELKEY(
	    archiver->run_mi,
	    EDV_ARCHIVER_OP_RUN
	);
	UPDATE_ACCELKEY(
	    archiver->close_mi,
	    EDV_ARCHIVER_OP_CLOSE
	);

	UPDATE_ACCELKEY(
	    archiver->edit_add_mi,
	    EDV_ARCHIVER_OP_ADD
	);
	UPDATE_ACCELKEY(
	    archiver->edit_extract_mi,
	    EDV_ARCHIVER_OP_EXTRACT
	);
	UPDATE_ACCELKEY(
	    archiver->edit_extract_all_mi,
	    EDV_ARCHIVER_OP_EXTRACT_ALL
	);
	UPDATE_ACCELKEY(
	    archiver->edit_delete_mi,
	    EDV_ARCHIVER_OP_DELETE
	);

	UPDATE_ACCELKEY(
	    archiver->check_mi,
	    EDV_ARCHIVER_OP_CHECK
	);
	UPDATE_ACCELKEY(
	    archiver->fix_mi,
	    EDV_ARCHIVER_OP_FIX
	);

	UPDATE_ACCELKEY(
	    archiver->select_all_mi,
	    EDV_ARCHIVER_OP_SELECT_ALL
	);
	UPDATE_ACCELKEY(
	    archiver->unselect_all_mi,
	    EDV_ARCHIVER_OP_UNSELECT_ALL
	);
	UPDATE_ACCELKEY(
	    archiver->invert_selection_mi,
	    EDV_ARCHIVER_OP_INVERT_SELECTION
	);
	UPDATE_ACCELKEY(
	    archiver->find_mi,
	    EDV_ARCHIVER_OP_FIND
	);
	UPDATE_ACCELKEY(
	    archiver->properties_mi,
	    EDV_ARCHIVER_OP_PROPERTIES
	);

	UPDATE_ACCELKEY(
	    archiver->refresh_mi,
	    EDV_ARCHIVER_OP_REFRESH
	);
	UPDATE_ACCELKEY(
	    archiver->refresh_all_mi,
	    EDV_ARCHIVER_OP_REFRESH_ALL
	);
	UPDATE_ACCELKEY(
	    archiver->comment_and_statistics_mi,
	    EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS
	);
	UPDATE_ACCELKEY(
	    archiver->show_tool_bar_micheck,
	    EDV_ARCHIVER_OP_SHOW_TOOL_BAR
	);
	UPDATE_ACCELKEY(
	    archiver->show_location_bar_micheck,
	    EDV_ARCHIVER_OP_SHOW_LOCATION_BAR
	);
	UPDATE_ACCELKEY(
	    archiver->show_find_bar_micheck,
	    EDV_ARCHIVER_OP_SHOW_FIND_BAR
	);
	UPDATE_ACCELKEY(
	    archiver->show_status_bar_micheck,
	    EDV_ARCHIVER_OP_SHOW_STATUS_BAR
	);
	UPDATE_ACCELKEY(
	    archiver->contents_list_filter_mi,
	    EDV_ARCHIVER_OP_CONTENTS_LIST_FILTER
	);
	UPDATE_ACCELKEY(
	    archiver->mime_types_mi,
	    EDV_ARCHIVER_OP_MIME_TYPES
	);

	UPDATE_ACCELKEY(
	    archiver->write_protect_mi,
	    EDV_ARCHIVER_OP_WRITE_PROTECT
	);
	UPDATE_ACCELKEY(
	    archiver->delete_method_recycle_micheck,
	    EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE
	);
	UPDATE_ACCELKEY(
	    archiver->delete_method_purge_micheck,
	    EDV_ARCHIVER_OP_DELETE_METHOD_PURGE
	);
	UPDATE_ACCELKEY(
	    archiver->options_mi,
	    EDV_ARCHIVER_OP_OPTIONS
	);
	UPDATE_ACCELKEY(
	    archiver->customize_mi,
	    EDV_ARCHIVER_OP_CUSTOMIZE
	);

	UPDATE_ACCELKEY(
	    archiver->window_new_browser_mi,
	    EDV_ARCHIVER_OP_NEW_BROWSER
	);
	UPDATE_ACCELKEY(
	    archiver->window_new_imbr_mi,
	    EDV_ARCHIVER_OP_NEW_IMBR
	);
	UPDATE_ACCELKEY(
	    archiver->window_new_archiver_mi,
	    EDV_ARCHIVER_OP_NEW_ARCHIVER
	);
	UPDATE_ACCELKEY(
	    archiver->window_recycle_bin_mi,
	    EDV_ARCHIVER_OP_RECYCLE_BIN
	);

	UPDATE_ACCELKEY(
	    archiver->help_contents_mi,
	    EDV_ARCHIVER_OP_HELP_CONTENTS
	);
	UPDATE_ACCELKEY(
	    archiver->help_archiver_mi,
	    EDV_ARCHIVER_OP_HELP_ARCHIVER
	);
	UPDATE_ACCELKEY(
	    archiver->help_keys_list_mi,
	    EDV_ARCHIVER_OP_HELP_KEYS_LIST
	);
	UPDATE_ACCELKEY(
	    archiver->help_common_operations_mi,
	    EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS
	);
	UPDATE_ACCELKEY(
	    archiver->help_about_mi,
	    EDV_ARCHIVER_OP_HELP_ABOUT
	);

	UPDATE_ACCELKEY(
	    archiver->contents_clist_add_mi,
	    EDV_ARCHIVER_OP_ADD
	);
	UPDATE_ACCELKEY(
	    archiver->contents_clist_extract_mi,
	    EDV_ARCHIVER_OP_EXTRACT
	);
	UPDATE_ACCELKEY(
	    archiver->contents_clist_extract_all_mi,
	    EDV_ARCHIVER_OP_EXTRACT_ALL
	);
	UPDATE_ACCELKEY(
	    archiver->contents_clist_delete_mi,
	    EDV_ARCHIVER_OP_DELETE
	);
	UPDATE_ACCELKEY(
	    archiver->contents_clist_comment_and_statistics_mi,
	    EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS
	);
	UPDATE_ACCELKEY(
	    archiver->contents_clist_check_mi,
	    EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS
	);
	UPDATE_ACCELKEY(
	    archiver->contents_clist_fix_mi,
	    EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS
	);
	UPDATE_ACCELKEY(
	    archiver->contents_clist_properties_mi,
	    EDV_ARCHIVER_OP_PROPERTIES
	);

#undef UPDATE_ACCELKEY
}

/*
 *	Recreates the Archiver's Tool Bar.
 */
void EDVArchiverToolBarRegenerate(edv_archiver_struct *archiver)
{
	GList *glist, *items_list;
	GtkWidget *parent;
	toolbar_display tb_display;
	toolbar_relief tb_relief;
	toolbar_item_struct *item;
	toolbar_struct *toolbar;
	const cfg_intlist_struct *intlist;
	const cfg_item_struct *cfg_list;
	edv_archiver_opid_struct *opid;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	parent = archiver->tool_bar_handle;
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Get the Tool Bar configuration */
	tb_display = (toolbar_display)EDV_GET_I(
	    EDV_CFG_PARM_ARCHIVER_TOOL_BAR_DISPLAY
	);
	tb_relief = (toolbar_relief)EDV_GET_I(
	    EDV_CFG_PARM_ARCHIVER_TOOL_BAR_RELIEF
	);
	intlist = EDV_GET_INTLIST(EDV_CFG_PARM_ARCHIVER_TOOL_BAR);
	if(intlist == NULL)
	    return;

	/* Delete the existing Tool Bar */
	ToolBarDelete(archiver->tool_bar);
	archiver->tool_bar = NULL;

	/* Create the Tool Bar items list */
	items_list = NULL;
	for(glist = intlist->list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    /* Get the Operation ID */
	    opid = EDVArchiverMatchOPID(
		archiver,
		(edv_archiver_op)glist->data
	    );
	    if(opid == NULL)
		continue;

	    switch(opid->op)
	    {
	      case EDV_ARCHIVER_OP_NONE:
		break;

	      default:
		item = ToolBarItemNew(
		    opid->tb_item_type,
		    NULL,
		    opid->button_name,
		    opid->button_icon_data,
		    opid->tooltip,
		    opid->op,
		    EDVArchiverOPCB, opid,
		    EDVArchiverOPEnterCB, opid,
		    EDVArchiverOPLeaveCB, opid
		);
		items_list = g_list_append(items_list, item);
		break;
	    }
	}

	/* Create the Tool Bar */
	archiver->tool_bar = toolbar = ToolBarNew(
	    parent,
	    items_list,
	    tb_display, tb_relief, FALSE
	);
	ToolBarMap(toolbar);

	/* Delete the Tool Bar items list */
	if(items_list != NULL)
	{
	    g_list_foreach(items_list, (GFunc)ToolBarItemDelete, NULL);
	    g_list_free(items_list);
	}
}

/*
 *	Recreates the Archiver Contents List right click menu.
 */
void EDVArchiverContentsMenuRegenerate(edv_archiver_struct *archiver)
{
	edv_archiver_op op;
	guint8 **icon_data = NULL;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu;

	if(archiver == NULL)
	    return;

	/* Recreate menu */
	GTK_WIDGET_DESTROY(archiver->contents_clist_menu);
	archiver->contents_clist_menu = menu = GUIMenuCreate();

#define DO_ADD_MENU_ITEM_LABEL		{	\
 edv_archiver_opid_struct *opid =		\
  EDVArchiverMatchOPID(archiver, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (guint8 **)((icon_data != NULL) ?		\
    icon_data : opid->menu_icon_data),		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods, NULL,	\
   opid, EDVArchiverMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVArchiverMenuItemEnterCB, opid,	\
   (gpointer)EDVArchiverMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}

	op = EDV_ARCHIVER_OP_ADD;
	DO_ADD_MENU_ITEM_LABEL
	GUISetMenuItemDefault(w);
	archiver->contents_clist_add_mi = w;

	op = EDV_ARCHIVER_OP_EXTRACT;
	DO_ADD_MENU_ITEM_LABEL
	archiver->contents_clist_extract_mi = w;

	op = EDV_ARCHIVER_OP_EXTRACT_ALL;
	DO_ADD_MENU_ITEM_LABEL
	archiver->contents_clist_extract_all_mi = w;

	op = EDV_ARCHIVER_OP_DELETE;
	DO_ADD_MENU_ITEM_LABEL
	archiver->contents_clist_delete_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS;
	DO_ADD_MENU_ITEM_LABEL
	archiver->contents_clist_comment_and_statistics_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_ARCHIVER_OP_CHECK;
	DO_ADD_MENU_ITEM_LABEL
	archiver->contents_clist_check_mi = w;

	op = EDV_ARCHIVER_OP_FIX;
	DO_ADD_MENU_ITEM_LABEL
	archiver->contents_clist_fix_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_ARCHIVER_OP_PROPERTIES;
	DO_ADD_MENU_ITEM_LABEL
	archiver->contents_clist_properties_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
}


/*
 *	Sets the Archiver's current values to the configuration.
 */
void EDVArchiverSyncConfiguration(edv_archiver_struct *archiver)
{
	GtkWidget *w;
	edv_core_struct *core;
	cfg_item_struct *cfg_list;

	if(archiver == NULL)
	    return;

	core = archiver->core;
	if(core == NULL)
	    return;

	cfg_list = core->cfg_list;

	/* Location history */
	w = archiver->location_combo;
	if(w != NULL)
	    EDVSaveFileGList(
		EDV_GET_S(EDV_CFG_PARM_FILE_ARCHIVER_LOCATION_HISTORY),
		GUIComboGetList(w)
	    );

	/* Toplevel geometry */
	w = archiver->toplevel;
	if(w != NULL)
	{
	    gint x = 0, y = 0;
	    GdkWindow *window = w->window;
	    if(window != NULL)
		gdk_window_get_root_origin(window, &x, &y);

	    EDV_SET_I(EDV_CFG_PARM_ARCHIVER_X, x);
	    EDV_SET_I(EDV_CFG_PARM_ARCHIVER_Y, y);
	    EDV_SET_I(EDV_CFG_PARM_ARCHIVER_WIDTH, w->allocation.width);
	    EDV_SET_I(EDV_CFG_PARM_ARCHIVER_HEIGHT, w->allocation.height);
	}

	/* EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH parameter is updated
	 * as they change in EDVArchiverResizeColumnCB() or when the
	 * configuration list is changed directly
	 */

	/* EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN parameter is updated
	 * when the configuration list is changed directly
	 */

	/* Contents list filter */
	if(STRISEMPTY(archiver->contents_list_filter))
	{
	    EDV_SET_S(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_FILTER,
		"*"
	    );
	}
	else
	{
	    EDV_SET_S(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_FILTER,
		archiver->contents_list_filter
	    );
	}
}

/*
 *	Opens the archive.
 *
 *	The arch_obj specifies the archive to open.
 *
 *	The password specifies the password needed to open the archive
 *	or NULL if none is specified.
 *
 *	All the objects in the specified archive will be opened to the
 *	Archiver's Contents List, any existing items in the Contents
 *	List will be cleared first.
 *
 *	The Archiver's title, location, and password will be updated
 *	to reflect the opened archive.
 *
 *	The location's history will be recorded prior to setting the
 *	new location.
 *
 *	The Archive Comments & Statistics Dialog will be mapped as
 *	appropriate.
 */
void EDVArchiverOpenArchive(
	edv_archiver_struct *archiver, const gchar *arch_obj,
	const gchar *password
)
{
	gboolean show_info_on_open;
	gulong size;
	gchar *darch_obj, *dpassword, *get_listing_error_msg;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;
	struct stat stat_buf;

	if((archiver == NULL) || STRISEMPTY(arch_obj))
	    return;

	clist = (GtkCList *)archiver->contents_clist;
	core = archiver->core;
	if((clist == NULL) || (core == NULL))
	    return;

	EDVArchiverSetBusy(archiver, TRUE);
	GUIBlockInput(archiver->toplevel, TRUE);

	cfg_list = core->cfg_list;

	/* Get configuration */
	show_info_on_open = EDV_GET_B(
	    EDV_CFG_PARM_ARCHIVER_SHOW_ARCHIVE_INFO_ON_OPEN
	);

	darch_obj = STRDUP(arch_obj);
	dpassword = STRDUP(password);

	/* Check if archive exists and get its size */
	if(stat(darch_obj, &stat_buf))
	    size = 0l;
#ifdef S_ISREG
	else if(S_ISREG(stat_buf.st_mode))
	    size = (gulong)stat_buf.st_size;
#endif
	else
	    size = 0l;

	/* Update the title, location, and password */
	EDVArchiverSetTitle(archiver, darch_obj);
	EDVArchiverSetLocation(archiver, darch_obj, TRUE);
	EDVArchiverSetPassword(archiver, dpassword);
	EDVArchiverUpdateLocationIcon(archiver, darch_obj);

	/* Clear the Contents List and open the archive's objects
	 * to it
	 */
	gtk_clist_freeze(clist);
	EDVArchiverContentsGetListing(
	    archiver,
	    EDV_GET_B(EDV_CFG_PARM_LISTS_ANIMATE_UPDATES)
	);
	get_listing_error_msg = STRDUP(EDVArchiverContentsGetError(archiver));
	gtk_clist_thaw(clist);

	EDVArchiverUpdateMenus(archiver);

	GUIBlockInput(archiver->toplevel, FALSE);
	EDVArchiverSetBusy(archiver, FALSE);

	/* Does the archive have a non-zero size (implying it is not
	 * an empty new archive) and we should show the archive info
	 * on open?
	 */
	if((size > 0l) && show_info_on_open)
	    EDVArchiverOPCommentAndStatistics(archiver);

	if(TRUE)
	{
	    gchar *msg = g_strdup_printf(
"Archive \"%s\" opened (%s %s)",
		g_basename(darch_obj),
		EDVSizeStrDelim(size),
		(size == 1l) ? "byte" : "bytes"
	    );
	    EDVStatusBarMessage(archiver->status_bar, msg, FALSE);
	    g_free(msg);
	}

	/* Did an error occure while getting the listing? */
	if(!STRISEMPTY(get_listing_error_msg) && !CDialogIsQuery() &&
	   (size > 0l)
	)
	{
	    gchar *msg = g_strdup_printf(
"%s.",
		get_listing_error_msg
	    );
	    EDVPlaySoundWarning(core);
	    CDialogSetTransientFor(archiver->toplevel);
	    CDialogGetResponse(
		"Open Archive Warning",
		msg,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    g_free(msg);
	    CDialogSetTransientFor(NULL);
	}

	g_free(darch_obj);
	g_free(dpassword);
	g_free(get_listing_error_msg);
}


/*
 *	Creates a new Archiver Operation IDs list.
 */
GList *EDVArchiverOPIDListNew(edv_archiver_struct *archiver)
{
	const gchar *button_name, *menu_name, *tooltip;
	guint accel_key, accel_mods;
	guint8 **button_icon_data, **menu_icon_data;
	GList *opid_list = NULL;
	toolbar_item_type tb_item_type;
	edv_opid_flags flags;
	edv_archiver_op op;
	edv_archiver_opid_struct *opid;
	void (*func_cb)(toolbar_item_struct *, gint, gpointer);
	void (*enter_func_cb)(toolbar_item_struct *, gint, gpointer);

#define DO_APPEND_OPID	{				\
 opid = EDV_ARCHIVER_OPID(g_malloc0(			\
  sizeof(edv_archiver_opid_struct)			\
 ));							\
 if(opid != NULL) {					\
  opid->op = op;					\
  opid->flags = flags;					\
  opid->tb_item_type = tb_item_type;			\
  opid->func_cb = func_cb;				\
  opid->enter_func_cb = enter_func_cb;			\
  opid->button_name = STRDUP(button_name);		\
  opid->menu_name = STRDUP(menu_name);			\
  opid->tooltip = STRDUP(tooltip);			\
  opid->accel_key = accel_key;				\
  opid->accel_mods = accel_mods;			\
  opid->button_icon_data = button_icon_data;		\
  opid->menu_icon_data = menu_icon_data;		\
  opid->archiver = archiver;				\
  opid_list = g_list_append(opid_list, opid);		\
 }							\
}

	func_cb = EDVArchiverOPCB;
	enter_func_cb = EDVArchiverOPEnterCB;


	op = EDV_ARCHIVER_OP_SEPARATOR;
	flags = EDV_OPID_ALLOW_MULTIPLE | EDV_OPID_NO_OP;
	tb_item_type = TOOLBAR_ITEM_SEPARATOR;
	button_icon_data = (guint8 **)icon_hsep_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sparateur";
	menu_name = "Sparateur";
	tooltip = "Sparateur";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "Separatore";
	menu_name = "Separatore";
	tooltip = "Separatore";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#else
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_CLOSE;
	flags = 0;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	button_icon_data = (guint8 **)icon_close_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Cierre";
	menu_name = "Cierre";
	tooltip = "Cierre esta ventana";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Proche";
	menu_name = "Proche";
	tooltip = "Fermer cette fentre";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zamknij";
	menu_name = "Zamknij";
	tooltip = "Zamknij to okno";
#else
	button_name = "Close";
	menu_name = "Close";
	tooltip = "Close this window";
#endif

	accel_key = GDK_w;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_EXIT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_exit_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Salida";
	menu_name = "La Salida";
	tooltip = "Cierra todas ventanas en la aplicacin y la";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sortie";
	menu_name = "Sortie";
	tooltip = "Ferme toutes fentres dans l'application et sortie";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Wyjcie";
	menu_name = "Wyjcie";
	tooltip = "Zamknij wszystkie okna tej aplikacji i wyjd";
#else
	button_name = "Exit";
	menu_name = "Exit";
	tooltip = "Close all windows in the application and exit";
#endif
	accel_key = GDK_q;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_SYNC_DISKS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_save_20x20_xpm;
	menu_icon_data = button_icon_data;

#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Sincronizar Discos";
	menu_name = "Sincronizar Discos";
	tooltip = "Limpan cualquiera transferencias pendientes de datos al disco";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Disq Syn";
	menu_name = "Les Disques De Syn";
	tooltip = "Rincer des transferts de donnes en attente au disque";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Synchr. dyski";
	menu_name = "Synchronizuj dyski";
	tooltip = "Wykonaj wszystkie oczekujce transfery";
#else
	button_name = "Sync Disks";
	menu_name = "Sync Disks";
	tooltip = "Flush any pending data transfers to disk";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HISTORY;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_trace_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Historia";
	menu_name = "La Historia...";
	tooltip = "La Historia de rdenes procesadas por esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Histoire";
	menu_name = "L'Histoire...";
	tooltip = "L'Histoire d'ordres traits par cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Historia";
	menu_name = "Historia...";
	tooltip = "Historia komend wykonanych przez t aplikacj";
#else
	button_name = "History";
	menu_name = "History...";
	tooltip = "History of commands processed by this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_RUN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_run_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Corra";
	menu_name = "Corra...";
	tooltip = "Corra un programa";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Courir";
	menu_name = "Courir...";
	tooltip = "Excute un programme";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Uruchom";
	menu_name = "Uruchom...";
	tooltip = "Uruchom program";
#else
	button_name = "Run";
	menu_name = "Run...";
	tooltip = "Run a program";
#endif
	accel_key = GDK_r;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_RUN_TERMINAL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_terminal2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Terminal";
	menu_name = "La Terminal";
	tooltip = "Corre una terminal";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Terminal";
	menu_name = "Le Terminal";
	tooltip = "Courir un terminal";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Uruchom terminal";
#else
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Run a terminal";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_WRITE_PROTECT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_secure_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Proteja";
	menu_name = "Proteja contra escritura";
	tooltip = "Habilite/inhabilita la proteccion contra escritura";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Protge";
	menu_name = "Protge En criture";
	tooltip = "Rend capable/rend infirme protge en criture";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zabezpieczenie";
	menu_name = "Zabezpieczenie przed zapisem";
	tooltip = "Wcz/wycz zabezpieczenie przed zapisem";
#else
	button_name = "Protect";
	menu_name = "Write Protect";
	tooltip = "Enable/disable write protect";
#endif
	accel_key = GDK_y;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_trash_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Recycle";
	menu_name = "Delete Method: Recycle";
	tooltip = "Set the delete method to recycle";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_DELETE_METHOD_PURGE;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_purge_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Purge";
	menu_name = "Delete Method: Purge";
	tooltip = "Set the delete method to purge";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_NEW;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_new_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Nuevo";
	menu_name = "Nuevo...";
	tooltip = "Crea un archivo nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Nouveau";
	menu_name = "Nouveau...";
	tooltip = "Crer une nouvelle archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe";
	menu_name = "Nowe...";
	tooltip = "Utwrz nowe archiwum";
#else
	button_name = "New";
	menu_name = "New...";
	tooltip = "Create a new archive";
#endif
	accel_key = GDK_n;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_OPEN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_open_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Abrir";
	menu_name = "Abrir...";
	tooltip = "Abre un archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ouvert";
	menu_name = "Ouvert...";
	tooltip = "Ouvrir une archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Otwrz";
	menu_name = "Otwrz...";
	tooltip = "Otwrz archiwum";
#else
	button_name = "Open";
	menu_name = "Open...";
	tooltip = "Open an archive";
#endif
	accel_key = GDK_F3;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_ADD;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_archive_add_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Agregue";
	menu_name = "Agregue...";
	tooltip = "Agregue objetos al archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ajouter";
	menu_name = "Ajouter...";
	tooltip = "Ajoute des objets  l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Dodaj";
	menu_name = "Dodaj...";
	tooltip = "Dodaj obiekty do archiwum";
#else
	button_name = "Add";
	menu_name = "Add...";
	tooltip = "Add objects to the archive";
#endif
	accel_key = GDK_Insert;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_EXTRACT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_archive_extract_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Extraer";
	menu_name = "Extraer...";
	tooltip = "Extraer los objetos escogidos del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Extrait";
	menu_name = "L'Extrait...";
	tooltip = "L'Extrait a choisi l'objet (s) de l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Rozpakuj";
	menu_name = "Rozpakuj...";
	tooltip = "Rozpakuj zaznaczony(e) obiekt(y) z archiwum";
#else
	button_name = "Extract";
	menu_name = "Extract...";
	tooltip = "Extract the selected object(s) from the archive";
#endif
	accel_key = GDK_Return;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_EXTRACT_ALL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_archive_extract_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ExtrTodo";
	menu_name = "Extraiga Todo...";
	tooltip = "Extrae todo del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ExtrTout";
	menu_name = "Extraire Tout...";
	tooltip = "Extraire tous objets de l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Rozpakuj wszystko";
	menu_name = "Rozpakuj wszystko...";
	tooltip = "Rozpakuj wszystkie obiekty z archiwum";
#else
	button_name = "ExtractAll";
	menu_name = "Extract All...";
	tooltip = "Extract all objects from the archive";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_SHIFT_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_DELETE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_cancel_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Borre";
	menu_name = "Borre";
	tooltip = "Borre los objeto(s) escogido(s) del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Effacer";
	menu_name = "Effacer";
	tooltip = "Efface l'objet choisi (s) de l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Usu";
	menu_name = "Usu";
	tooltip = "Usu zaznaczony(e) obiekt(y) z archiwum";
#else
	button_name = "Delete";
	menu_name = "Delete";
	tooltip = "Delete the selected object(s) from the archive";
#endif
	accel_key = GDK_Delete;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_CHECK;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_archive_check_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Check";
	menu_name = "Check";
	tooltip = "Check the archive for problems";
	accel_key = GDK_c;
	accel_mods = GDK_SHIFT_MASK | GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_FIX;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_aid_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Fix";
	menu_name = "Fix";
	tooltip = "Attempt to fix any problems with the archive";
	accel_key = GDK_f;
	accel_mods = GDK_SHIFT_MASK | GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_PROPERTIES;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_properties2_20x20_xpm;;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Props";
	menu_name = "Las Propiedades";
	tooltip = "Modifican escogi las propiedades de objeto";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Props";
	menu_name = "Les Proprits";
	tooltip = "Modifier a choisi les proprits de l'objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Waciwoci";
	menu_name = "Waciwoci...";
	tooltip = "Modyfikuj waciwoci zaznaczonego obiektu";
#else
	button_name = "Properties";
	menu_name = "Properties...";
	tooltip = "Modify the archive's properties";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_SELECT_ALL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_select_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "SelAll";
	menu_name = "Escoja Todo";
	tooltip = "Escoge todo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "SelAll";
	menu_name = "Choisir Tout";
	tooltip = "Choisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ZaznWszyst";
	menu_name = "Zaznacz wszystko";
	tooltip = "Zaznacz wszystkie obiekty";
#else
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all objects";
#endif
	accel_key = GDK_a;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_UNSELECT_ALL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_unselect_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "UnSelAll";
	menu_name = "Unescoja Todo";
	tooltip = "Unescoge todo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "UnSelAll";
	menu_name = "Unchoisir Tout";
	tooltip = "Unchoisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdznWszyst";
	menu_name = "Odznacz wszystko";
	tooltip = "Usu zaznaczenie ze wszystkich obiektw";
#else
	button_name = "UnSelAll";
	menu_name = "Unselect All";
	tooltip = "Unselect all objects";
#endif
	accel_key = GDK_u;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_INVERT_SELECTION;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_invert_selection_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "InvSel";
	menu_name = "Invierta Seleccin";
	tooltip =
 "Selectos todos objetos y unselect no seleccionados todos objetos escogidos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "InvSel";
	menu_name = "Inverser Slection";
	tooltip =
 "Choisir tous objets dslections et deslectionner tous objets choisis";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdwrZazn";
	menu_name = "Odwr zaznaczenie";
	tooltip =
 "Zaznacza wszystkie niezaznaczone obiekty i usuwa zaznaczenie ze wszystkich zaznaczonych obiektw";
#else
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip =
 "Select all unselected objects and unselect all selected objects";
#endif
	accel_key = GDK_i;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_FIND;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_search_20x20_xpm;;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Buscar";
	menu_name = "Buscar";
	tooltip = "Buscar objeto(s) por atributos especificados";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Dcouverte";
	menu_name = "La Dcouverte";
	tooltip = "L'objet de Dcouverte (s) par les attributs spcifis ";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Znajd";
	menu_name = "Znajd...";
	tooltip = "Znajd obiekt(y) o podanych atrybutach";
#else
	button_name = "Find";
	menu_name = "Find...";
	tooltip = "Find object(s) by specific attributes";
#endif
	accel_key = GDK_f;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_REFRESH;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque";
	menu_name = "Refresque";
	tooltip = "Refresque objetos de archivo que listan";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir";
	menu_name = "Rafrachir";
	tooltip = "Rafrachit les objets d'archive numrants";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie";
	menu_name = "Odwie";
	tooltip = "Odwiea list obiektw archiwum";
#else
	button_name = "Refresh";
	menu_name = "Refresh";
	tooltip = "Refresh the archive objects listing";
#endif
	accel_key = GDK_F5;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_REFRESH_ALL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque Todo";
	menu_name = "Refresque Todo";
	tooltip = "Refresca todas listas";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir Tout";
	menu_name = "Rafrachir Tout";
	tooltip = "Rafrachir toutes listes";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie wszystko";
	menu_name = "Odwie wszystko";
	tooltip = "Odwiea wszystkie listy";
#else
	button_name = "Refresh All";
	menu_name = "Refresh All";
	tooltip = "Refresh all listings";
#endif

	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_info_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Info";
	menu_name = "El Comentario & La Estadstica";
	tooltip = "Vea el comentario & la estadstica";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Info";
	menu_name = "Commenter & La Statistique";
	tooltip = "Regarder le commentaire & la statistique";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "Info";
	menu_name = "Bemerkung & Statistik";
	tooltip = "Siehe bemerkung & statistik an";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "Info";
	menu_name = "Il Commento & La Statistica";
	tooltip = "Osservare il commento & la statistica";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "Info";
	menu_name = "Opmerking & Statistieken";
	tooltip = "Bekijk opmerking & statistieken";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "Info";
	menu_name = "O Comentrio & Estatstica";
	tooltip = "Veja comentrio & estatstica";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "Info";
	menu_name = "Kommentar & Statistics";
	tooltip = "Sikt kommentar & statistikk";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Info";
	menu_name = "Komentarz i statystyki";
	tooltip = "Obejrzyj komentarz i statystyki";
#else
	button_name = "Info";
	menu_name = "Comment & Statistics";
	tooltip = "View comment & statistics";
#endif
	accel_key = GDK_F9;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_SHOW_TOOL_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;

#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La Exposicin/la barra de herramienta de cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La barre d'outil de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasNarz";
	menu_name = "Pasek narzdzi";
	tooltip = "Poka/ukryj pasek narzdzi";
#else
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "Show/hide tool bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_SHOW_LOCATION_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La exposicin/la barra de la ubicacin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La barre d'emplacement de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pascie";
	menu_name = "Pasek cieki";
	tooltip = "Poka/ukryj pasek cieki";
#else
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "Show/hide location bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_SHOW_FIND_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La exposicin/la barra del hallazgo del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La barre de dcouverte de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasSzuk";
	menu_name = "Pasek szukania";
	tooltip = "Poka/ukryj pasek szukania";
#else
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "Show/hide find bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_SHOW_STATUS_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_status_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La exposicin/la barra de la posicin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La barre de statut de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasStatus";
	menu_name = "Pasek statusu";
	tooltip = "Poka/ukryj pasek statusu";
#else
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "Show/hide status bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_CONTENTS_LIST_FILTER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_wildcards_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Filter";
	menu_name = "Set Filter...";
	tooltip = "Set the name filter for listing objects";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_MIME_TYPES;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_mimetypes_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "El panorama/modifica MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "Vue/modifie MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "TypyMIME";
	menu_name = "Typy MIME...";
	tooltip = "Obejrzyj/zmodyfikuj typy MIME";
#else
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "View/modify MIME Types";
#endif
	accel_key = GDK_m;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_NEW_BROWSER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)endeavour_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FBrowser";
	menu_name = "Nuevo Examinador De Archivos";
	tooltip = "Muestra un nuevo examinador de archivos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FBrowser";
	menu_name = "Nouveau Browser de Fichier";
	tooltip = "Crer un nouveau browser de fichier";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglPlik";
	menu_name = "Nowa przegldarka plikw";
	tooltip = "Utwrz now przegldark plikw";
#else
	button_name = "FBrowser";
	menu_name = "New File Browser";
	tooltip = "Create a new file browser";
#endif
	accel_key = GDK_1;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_NEW_IMBR;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)image_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "IBrowser";
	menu_name = "Nuevo Examinador De Imagenes";
	tooltip = "Muestra un nuevo examinador de imagenes";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "IBrowser";
	menu_name = "Nouveau Browser D'Image";
	tooltip = "Crer un nouveau browser d'image";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglObr";
	menu_name = "Nowa przegldarka obrazw";
	tooltip = "Utwrz now przegldark obrazw";
#else
	button_name = "IBrowser";
	menu_name = "New Image Browser";
	tooltip = "Create a new image browser";
#endif
	accel_key = GDK_2;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_NEW_ARCHIVER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)archiver_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Archiver";
	menu_name = "Nuevo Compresor De Archivos";
	tooltip = "Muestra un archivo comprimido nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Archiver";
	menu_name = "Nouveau Archiver";
	tooltip = "Crer un nouveau archiver";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Archiwizator";
	menu_name = "Nowy archwizator";
	tooltip = "Utwrz nowego archwizatora";
#else
	button_name = "Archiver";
	menu_name = "New Archiver";
	tooltip = "Create a new archiver";
#endif
	accel_key = GDK_3;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_RECYCLE_BIN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_trash_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "RecBin";
	menu_name = "El Cajn De Reciclaje";
	tooltip = "Abrir Cajn de Reciclaje (basura)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "RecBin";
	menu_name = "Recycler L'Huche";
	tooltip = "Recycler l'huche (la poubelle)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kosz";
	menu_name = "Kosz";
	tooltip = "Otwrz kosz";
#else
	button_name = "RecBin";
	menu_name = "Recycle Bin";
	tooltip = "Open Recycle Bin";
#endif
	accel_key = GDK_4;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_OPTIONS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_options2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Las Opciones";
	menu_name = "Las Opciones...";
	tooltip = "Modifican estas opciones de la aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Les Options";
	menu_name = "Les Options";
	tooltip = "Modifier ces options de l'application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Opcje";
	menu_name = "Opcje...";
	tooltip = "Modyfikuj opcje tej aplikacji";
#else
	button_name = "Options";
	menu_name = "Options...";
	tooltip = "Modify this application's options";
#endif
	accel_key = GDK_o;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_CUSTOMIZE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_customize_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Personal";
	menu_name = "Personaliza...";
	tooltip = "Personaliza la mirada y se siente de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Personnal";
	menu_name = "Personnalise...";
	tooltip = "Personnalise le regard et sens de cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Dostosuj";
	menu_name = "Dostosuj...";
	tooltip = "Dostosuj wygld i zachowanie tej aplikacji";
#else
	button_name = "Custom";
	menu_name = "Customize...";
	tooltip = "Customize the look and feel of this application";
#endif
	accel_key = GDK_t;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_ARCHIVER_OP_HELP_CONTENTS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Ayuda";
	menu_name = "El Contenido";
	tooltip = "El Indice de toda ayuda pagina";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Aide";
	menu_name = "Les Contenus";
	tooltip = "L'Index de toutes pages d'aide";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc";
	menu_name = "Tematy pomocy";
	tooltip = "Spis treci wszystkich stron pomocy";
#else
	button_name = "Help";
	menu_name = "Contents";
	tooltip = "Index of all help pages";
#endif
	accel_key = GDK_F1;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HELP_FILE_BROWSER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "File Browser";
	tooltip = "Help on using the file browser";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HELP_IMAGE_BROWSER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Image Browser";
	tooltip = "Help on using the image browser";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HELP_ARCHIVER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Archiver";
	tooltip = "Help on using the archiver";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HELP_RECYCLE_BIN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Aide";
	menu_name = "Corbeille";
	tooltip = "Aide sur l'utilisation de la Corbeille";
#else
	button_name = "Help";
	menu_name = "Recycle Bin";
	tooltip = "Help on using the recycle bin";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HELP_KEYS_LIST;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpKeys";
	menu_name = "La Lista De Llaves";
	tooltip = "La Lista de todo teclado las operaciones claves";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpKeys";
	menu_name = "La Liste De Clefs";
	tooltip = "La Liste de tout clavier toutes oprations principales";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomocKlaw";
	menu_name = "Lista klawiszy";
	tooltip = "Lista skrtw klawiaturowych";
#else
	button_name = "HelpKeys";
	menu_name = "Keys List";
	tooltip = "List of all keyboard key operations";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Ayude a establecer MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Aider sur montant MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomocMIME";
	menu_name = "Typy MIME";
	tooltip = "Pomoc w ustawianiu typw MIME";
#else
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Help on setting up MIME Types";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_ARCHIVER_OP_HELP_ABOUT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_about_20x20_xpm;
	menu_icon_data = (guint8 **)icon_about_20x20_xpm;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Acerca";
	menu_name = "Acerca De";
	tooltip = "Acerca de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "De";
	menu_name = "De";
	tooltip = "De cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "O programie";
	menu_name = "O programie";
	tooltip = "Informacje o programie";
#else
	button_name = "About";
	menu_name = "About";
	tooltip = "About this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	return(opid_list);
#undef DO_APPEND_OPID
}

/*
 *	Deletes the Archiver Operation IDs list.
 */
void EDVArchiverOPIDListDelete(GList *glist)
{
	GList *opid_list = glist;
	edv_archiver_opid_struct *opid;

	for(; glist != NULL; glist = g_list_next(glist))
	{
	    opid = EDV_ARCHIVER_OPID(glist->data);
	    if(opid == NULL)
		continue;

	    g_free(opid->button_name);
	    g_free(opid->menu_name);
	    g_free(opid->tooltip);

	    g_free(opid);
	}
	g_list_free(opid_list);
}


/*
 *	Creates a new Accel Key List with default values.
 */
cfg_accelkey_list_struct *EDVArchiverAccelkeysListNewDefault(void)
{
	GList *glist, *opid_list, *ak_glist = NULL;
	cfg_accelkey_list_struct *ak_list;
	edv_archiver_opid_struct *opid;

	opid_list = EDVArchiverOPIDListNew(NULL);
	for(glist = opid_list; glist != NULL; glist = g_list_next(glist))
	{
	    opid = EDV_ARCHIVER_OPID(glist->data);
	    if(opid == NULL)
		continue;

	    if(opid->flags & EDV_OPID_NO_OP)
		continue;

	    ak_glist = g_list_append(
		ak_glist,
		CFGAccelkeyNew(
		    (gint)opid->op,
		    opid->accel_key,
		    opid->accel_mods
		)
	    );
	}
	EDVArchiverOPIDListDelete(opid_list);

	ak_list = CFGAccelkeyListNew(ak_glist);

	g_list_foreach(ak_glist, (GFunc)CFGAccelkeyDelete, NULL);
	g_list_free(ak_glist);

	return(ak_list);
}


/*
 *	Creates the Archiver's Menu Bar.
 *
 *	This function should be called by EDVArchiverNew().
 */
static void EDVArchiverCreateMenuBar(
	edv_archiver_struct *archiver, GtkWidget *parent
)
{
	edv_archiver_op op;
	guint8 **icon_data = NULL;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu, *menu_bar;

	/* Create menu bar */
	archiver->menu_bar = menu_bar = GUIMenuBarCreate(NULL);
	gtk_container_add(GTK_CONTAINER(parent), menu_bar);
	gtk_widget_show(menu_bar);

#define DO_ADD_MENU_ITEM_LABEL		{	\
 edv_archiver_opid_struct *opid =		\
  EDVArchiverMatchOPID(archiver, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (guint8 **)((icon_data != NULL) ?		\
    icon_data : opid->menu_icon_data),		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods, NULL,	\
   opid, EDVArchiverMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVArchiverMenuItemEnterCB, opid,	\
   (gpointer)EDVArchiverMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_ITEM_SUBMENU	{	\
 edv_archiver_opid_struct *opid =		\
  EDVArchiverMatchOPID(archiver, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_SUBMENU, accelgrp,	\
   (guint8 **)((icon_data != NULL) ?		\
    icon_data : opid->menu_icon_data),		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods, NULL,	\
   opid, NULL					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVArchiverMenuItemEnterCB, opid,	\
   (gpointer)EDVArchiverMenuItemLeaveCB, opid	\
  );						\
  if(w != NULL)					\
   GUIMenuItemSetSubMenu(w, submenu);		\
  if(w_record != NULL)				\
   *w_record = w;				\
 }						\
}

#define DO_ADD_MENU_ITEM_CHECK		{	\
 edv_archiver_opid_struct *opid =		\
  EDVArchiverMatchOPID(archiver, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_CHECK, accelgrp,	\
   (guint8 **)((icon_data != NULL) ?		\
    icon_data : opid->menu_icon_data),		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods, NULL,	\
   opid, EDVArchiverMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVArchiverMenuItemEnterCB, opid,	\
   (gpointer)EDVArchiverMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}

	/* Create Menus */
	/* File */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_ARCHIVER_OP_NEW;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->new_mi = w;

	    op = EDV_ARCHIVER_OP_OPEN;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->open_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_SYNC_DISKS;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->sync_disks_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_HISTORY;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->history_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_RUN;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->run_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_CLOSE;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->close_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#elif defined(PROG_LANGUAGE_POLISH)
"Plik"
#else
"File"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* Edit */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_ARCHIVER_OP_ADD;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->edit_add_mi = w;

	    op = EDV_ARCHIVER_OP_EXTRACT;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->edit_extract_mi = w;

	    op = EDV_ARCHIVER_OP_EXTRACT_ALL;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->edit_extract_all_mi = w;

	    op = EDV_ARCHIVER_OP_DELETE;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->edit_delete_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_CHECK;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->check_mi = w;

	    op = EDV_ARCHIVER_OP_FIX;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->fix_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_SELECT_ALL;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->select_all_mi = w;

	    op = EDV_ARCHIVER_OP_UNSELECT_ALL;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->unselect_all_mi = w;

	    op = EDV_ARCHIVER_OP_INVERT_SELECTION;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->invert_selection_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_FIND;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->find_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_PROPERTIES;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->properties_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Editar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#elif defined(PROG_LANGUAGE_POLISH)
"Edycja"
#else
"Edit"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* View */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_ARCHIVER_OP_REFRESH;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->refresh_mi = w;

	    op = EDV_ARCHIVER_OP_REFRESH_ALL;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->refresh_all_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->comment_and_statistics_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_SHOW_TOOL_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    archiver->show_tool_bar_micheck = w;

	    op = EDV_ARCHIVER_OP_SHOW_LOCATION_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    archiver->show_location_bar_micheck = w;

	    op = EDV_ARCHIVER_OP_SHOW_FIND_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    archiver->show_find_bar_micheck = w;

	    op = EDV_ARCHIVER_OP_SHOW_STATUS_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    archiver->show_status_bar_micheck = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_CONTENTS_LIST_FILTER;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->contents_list_filter_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_MIME_TYPES;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->mime_types_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Vista"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vue"
#elif defined(PROG_LANGUAGE_GERMAN)
"Blick"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Veduta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Overzicht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Vista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sikt"
#elif defined(PROG_LANGUAGE_POLISH)
"Widok"
#else
"View"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* Settings */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_ARCHIVER_OP_WRITE_PROTECT;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->write_protect_mi = w;

	    op = EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE;
	    DO_ADD_MENU_ITEM_CHECK
	    archiver->delete_method_recycle_micheck = w;

	    op = EDV_ARCHIVER_OP_DELETE_METHOD_PURGE;
	    DO_ADD_MENU_ITEM_CHECK
	    archiver->delete_method_purge_micheck = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_OPTIONS;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->options_mi = w;

	    op = EDV_ARCHIVER_OP_CUSTOMIZE;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->customize_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Configuracion"
#elif defined(PROG_LANGUAGE_FRENCH)
"Montages"
#elif defined(PROG_LANGUAGE_GERMAN)
"Setzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montaggi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Por"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innstillinger"
#elif defined(PROG_LANGUAGE_POLISH)
"Ustawienia"
#else
"Settings"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* Windows */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_ARCHIVER_OP_NEW_BROWSER;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->window_new_browser_mi = w;

	    op = EDV_ARCHIVER_OP_NEW_IMBR;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->window_new_imbr_mi = w;

	    op = EDV_ARCHIVER_OP_NEW_ARCHIVER;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->window_new_archiver_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_RECYCLE_BIN;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->window_recycle_bin_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ventanas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Windows"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Finestre"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ramen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Janelas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Vinduer"
#elif defined(PROG_LANGUAGE_POLISH)
"Okna"
#else
"Windows"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* Help */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_ARCHIVER_OP_HELP_CONTENTS;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->help_contents_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_HELP_ARCHIVER;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->help_archiver_mi = w;

	    op = EDV_ARCHIVER_OP_HELP_KEYS_LIST;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->help_keys_list_mi = w;

	    op = EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->help_common_operations_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_ARCHIVER_OP_HELP_ABOUT;
	    DO_ADD_MENU_ITEM_LABEL
	    archiver->help_about_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#elif defined(PROG_LANGUAGE_POLISH)
"Pomoc"
#else
"Help"
#endif
	    , GUI_MENU_BAR_ALIGN_RIGHT
	);

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_SUBMENU
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_SEP
}

/*
 *	Creates a new Archiver.
 */
edv_archiver_struct *EDVArchiverNew(edv_core_struct *core)
{
	const gint	border_major = 5;
	gboolean	tool_bar_map_state,
			location_bar_map_state,
			find_bar_map_state,
			status_bar_map_state;
	gint	toplevel_x, toplevel_y,
		toplevel_width, toplevel_height;
	gpointer entry_rtn, combo_rtn;
	const gchar	*wm_name, *wm_class,
			*location_history_file;
	GList *glist;
	GdkCursor *cursor;
	GdkWindow *window;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkStyle *style;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*toplevel;
	GtkEntry *entry;
	GtkCombo *combo;
	GtkCList *clist;
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	};
	const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_ARCHIVE_OBJECT,	0,	EDV_DND_INFO_ARCHIVE_OBJECT}
	};
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *status_bar;
	edv_archiver_struct *archiver;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	wm_name = core->wm_name;
	wm_class = core->wm_class;

	location_history_file = EDV_GET_S(
	    EDV_CFG_PARM_FILE_ARCHIVER_LOCATION_HISTORY
	);

	if(core->geometry_set)
	{
	    const GdkRectangle *rect = &core->geometry;
	    toplevel_x = rect->x;
	    toplevel_y = rect->y;
	    toplevel_width = rect->width;
	    toplevel_height = rect->height;
	}
	else
	{
	    toplevel_x = EDV_GET_I(EDV_CFG_PARM_ARCHIVER_X);
	    toplevel_y = EDV_GET_I(EDV_CFG_PARM_ARCHIVER_Y);
	    toplevel_width = EDV_GET_I(EDV_CFG_PARM_ARCHIVER_WIDTH);
	    toplevel_height = EDV_GET_I(EDV_CFG_PARM_ARCHIVER_HEIGHT);
	}

	tool_bar_map_state = EDV_GET_B(
	    EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR
	);
	location_bar_map_state = EDV_GET_B(
	    EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR
	);
	find_bar_map_state = EDV_GET_B(
	    EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR
	);
	status_bar_map_state = EDV_GET_B(
	    EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR
	);


	/* Create the Archiver */
	archiver = EDV_ARCHIVER(g_malloc0(sizeof(edv_archiver_struct)));
	if(archiver == NULL)
	    return(NULL);

	archiver->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	archiver->processing = FALSE;
	archiver->freeze_count = 0;
	archiver->busy_count = 0;
	archiver->core = core;

	archiver->opid_list = EDVArchiverOPIDListNew(archiver);

	archiver->tool_bar_map_state = tool_bar_map_state;
	archiver->location_bar_map_state = location_bar_map_state;
	archiver->find_bar_map_state = find_bar_map_state;
	archiver->status_bar_map_state = status_bar_map_state;
	archiver->contents_list_filter = STRDUP(EDV_GET_S(
	    EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_FILTER
	));
	archiver->contents_clist_selected_row = -1;
	archiver->last_recbin_items = -1;
	archiver->last_write_protect_state = -1;
	archiver->find_bar = NULL;
	archiver->status_bar = NULL;

	archiver->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_widget_set_name(w, EDV_ARCHIVER_TOPLEVEL_WIDGET_NAME);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_uposition(
	    w, toplevel_x, toplevel_y
	);
	gtk_widget_set_usize(
	    w, toplevel_width, toplevel_height
	);
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
	    gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
	    gtk_window_set_wmclass(
		GTK_WINDOW(w), "archiver", PROG_NAME
	    );
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    GdkGeometry geo;

	    geo.min_width = 100;
	    geo.min_height = 70;
	    geo.base_width = 0;
	    geo.base_height = 0;
	    geo.width_inc = 1;
	    geo.height_inc = 1;
	    gdk_window_set_geometry_hints(
		window, &geo,
		GDK_HINT_MIN_SIZE |
		GDK_HINT_BASE_SIZE |
		GDK_HINT_RESIZE_INC
	    );

	    GUISetWMIcon(window, (guint8 **)archiver_48x48_xpm);
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVArchiverDeleteEventCB), archiver
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), archiver
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), archiver
	);
	style = gtk_widget_get_style(w);
	parent = w;


	/* Main GtkVBox */
	archiver->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Tool Bars GtkVBox */
	archiver->bars_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Menu Bar GtkHandleBox */
	archiver->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
	);
	gtk_widget_show(w);
	/* Create menu bar */
	EDVArchiverCreateMenuBar(archiver, w);


	/* Tool Bar GtkHandleBox */
	archiver->tool_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
	);
	if(archiver->tool_bar_map_state)
	    gtk_widget_show(w);
	/* Build tool bar later */
	archiver->tool_bar = NULL;


	/* Location Bar GtkHandleBox */
	archiver->location_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(archiver->location_bar_map_state)
	    gtk_widget_show(w);
	parent2 = w;

	/* Hbox for location bar */
	archiver->location_bar_hbox = w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Hbox for location icon and combo (no spacing between icon
	 * and combo needed since combo already has spacing)
	 */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Vbox to orient location icon fixed parent */
	w = gtk_vbox_new(TRUE, 0);
	gtk_widget_set_usize(w, 20, 20);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Location icon fixed parent */
	archiver->location_icon_fixed = w = gtk_fixed_new();
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVArchiverLocBarIconCrossingCB), archiver
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVArchiverLocBarIconCrossingCB), archiver
	);
	gtk_widget_realize(w);
	cursor = EDVGetCursor(
	    core, EDV_CURSOR_CODE_HAND
	);
	gdk_window_set_cursor(w->window, cursor);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto crear una conexin a este archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cette archive"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schleppen sie dies, ein kettenglied zu diesem archiv zu schaffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trascinare questo per creare una maglia a quest'archivio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sleep dit om een schakel aan deze archief te creren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arraste isto criar um elo a este arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Slep dette skape et ledd til dette arkivet"
#elif defined(PROG_LANGUAGE_POLISH)
"Przecignij to by utworzy odnonik do tego archiwum"
#else
"Drag this to create a link to this archive"
#endif
	);
	gtk_widget_show(w);

	/* Reset location icon pixmap */
	archiver->location_icon_pm = NULL;

	/* Location GtkCombo */
	glist = EDVOpenFileGList(location_history_file, 0);
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Archive:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Archiv:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Archivio:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Archief:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv:"
#elif defined(PROG_LANGUAGE_POLISH)
"Archiwum:"
#else
"Archive:"
#endif
	    , "/",
	    glist,
	    20,		/* Maximum items */
	    &combo_rtn,
	    archiver,
	    EDVArchiverComboActivateCB,
	    NULL
	);
	if(glist != NULL)
	{
	    g_list_foreach(glist, (GFunc)g_free, NULL);
	    g_list_free(glist);
	    glist = NULL;
	}
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	archiver->location_combo = w = (GtkWidget *)combo_rtn;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);
	if(TRUE)
	{
	    /* Need to set different dnd_src_types for the drag source */
	    const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	    };
	    GUIDNDSetSrc(
		archiver->location_icon_fixed,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,		/* Actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		EDVLocBarIconDragDataGet,
		NULL,
		NULL,
		combo
	    );
	}
	w = combo->entry;
	EDVEntrySetDND(core, w);
	EDVEntrySetCompletePath(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Password GtkVBox */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent3 = w;
	/* Password prompt */
	w = GUIPromptBar(
	    (guint8 **)icon_security_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Contrasea:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Mot De Passe:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kennwort:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Parola D'Ordine:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Wachtwoord:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Senha:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Passord:"
#elif defined(PROG_LANGUAGE_POLISH)
"Haso:"
#else
"Password:"
#endif
	    , NULL, &entry_rtn
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	archiver->location_password_entry = w = (GtkWidget *)entry_rtn;
	entry = GTK_ENTRY(w);
	gtk_widget_set_usize(w, 100, -1);
	gtk_entry_set_visibility(entry, FALSE);
	gtk_entry_set_editable(entry, TRUE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(EDVArchiverComboActivateCB), archiver
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre la contrasea (si necesit) para el decrypting/codificado\
 el archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le mot de passe (si ncessaire) pour le decrypting\
 encrypting l'archive"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen das kennwort (wenn hat gebraucht) fr decrypting\
 verschlsseln des archivs zu ein dem sie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare la parola d'ordine (se bisogno avuto di) per decrypting\
 codificare l'archivio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het wachtwoord binnen (indien nodig) voor decrypting coderen\
 het archief had"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre a senha (se necessrio) para decrypting encrypting o arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i passordet (om trengt) for decrypting encrypting arkivet"
#elif defined(PROG_LANGUAGE_POLISH)
"Podaj haso (jeli potrzebne) by odszyfrowa/zaszyfrowa archiwum"
#else
"Enter the password (if needed) for decrypting/encrypting the archive"
#endif
	);

	/* GtkHandleBox for the Find Bar */
	archiver->find_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(archiver->find_bar_map_state)
	    gtk_widget_show(w);
	parent2 = w;

	/* Create the Find Bar */
	archiver->find_bar = EDVFindBarNew(
	    core, parent2,
	    EDV_FINDBAR_ROLE_ARCHIVE_OBJECT,
	    EDVArchiverFindBarLocationCB,
	    EDVArchiverFindBarStartCB,
	    EDVArchiverFindBarEndCB,
	    EDVArchiverFindBarMatchCB,
	    EDVArchiverStatusMessageCB,
	    EDVArchiverStatusProgressCB,
	    archiver
	);
	EDVFindBarMap(archiver->find_bar);


	/* GtkScrolledWindow for the GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Contents GtkCList */
	archiver->contents_clist = w = gtk_clist_new(
	    EDV_ARCHIVER_CONTENTS_CLIST_COLUMNS_MAX
	);
	gtk_widget_set_name(w, EDV_ARCHIVER_CLIST_WIDGET_NAME);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), archiver
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), archiver
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVArchiverButtonPressEventCB), archiver
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_signal_connect(
	    GTK_OBJECT(w), "resize_column",
	    GTK_SIGNAL_FUNC(EDVArchiverResizeColumnCB), archiver
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "click_column",
	    GTK_SIGNAL_FUNC(EDVArchiverClickColumnCB), archiver
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(EDVArchiverSelectRowCB), archiver
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(EDVArchiverUnselectRowCB), archiver
	);
	GUIDNDSetSrc(
	    w,
	    dnd_src_types,
	    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY,			/* Actions */
	    GDK_BUTTON1_MASK,			/* Buttons */
	    NULL,
	    EDVArchiverContentsDragDataGetCB,
	    EDVArchiverContentsDragDataDeleteCB,
	    NULL,
	    archiver
	);
	GUIDNDSetTar(
	    w,
	    dnd_tar_types,
	    sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY,			/* Actions */
	    GDK_ACTION_COPY,			/* Default action if same */
	    GDK_ACTION_COPY,			/* Default action */
	    EDVArchiverContentsDragDataReceivedCB,
	    archiver
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "drag_motion",
	    GTK_SIGNAL_FUNC(EDVArchiverContentsDragMotionCB), archiver
	);
	gtk_widget_show(w);


	/* Create the Tool Bar */
	EDVArchiverToolBarRegenerate(archiver);

	/* Create the Contents List popup menu */
	EDVArchiverContentsMenuRegenerate(archiver);


	/* Create the Status Bar */
	archiver->status_bar = status_bar = EDVStatusBarNew(
	    core, archiver->main_vbox
	);
	if(status_bar != NULL)
	{
	    if(archiver->status_bar_map_state)
		EDVStatusBarMap(status_bar);
	    else
		EDVStatusBarUnmap(status_bar);
	}


	/* The Archive Info Dialog is created later when needed */
	archiver->arch_info_dlg = NULL;

	/* Set the initial RC styles */
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		archiver->toplevel, standard_rcstyle
	    );
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		archiver->contents_clist, lists_rcstyle
	    );
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		archiver->contents_clist_menu, standard_rcstyle
	    );

	/* Set the initial values */
	EDVArchiverSetTitle(archiver, NULL);
	EDVArchiverAccelkeysRegenerate(archiver);
	EDVArchiverUpdateMenus(archiver);

	archiver->freeze_count--;

	return(archiver);
}

/*
 *	Syncronizes all data in the archiver's widgets to internal data
 *	on the archiver.
 */
void EDVArchiverSyncData(edv_archiver_struct *archiver)
{
	if(archiver == NULL)
	    return;

/* This function may be removed in the future, since there really is
 * no floating data from widgets to be synced to disk on the archiver
 */
}

/*
 *	Updates all widgets on the given archiver to reflect the current
 *	data.
 */
void EDVArchiverUpdateMenus(edv_archiver_struct *archiver)
{
	gboolean write_protect, write_protect_changed = FALSE;
	edv_delete_method delete_method;
	gboolean sensitive, state;
	guint8 **icon_data, **recbin_icon_data;
	const gchar *location;
	gint selected_obj_row = -1;
	GtkWidget *w;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	toolbar_struct *tb;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	tb = archiver->tool_bar;
	core = archiver->core;
	cfg_list = core->cfg_list;

	archiver->freeze_count++;

#define DO_SHOW				\
{ if(w != NULL) gtk_widget_show(w); }
#define DO_HIDE				\
{ if(w != NULL) gtk_widget_hide(w); }
#define DO_SET_CHECK_STATE		\
{ GUIMenuItemSetCheck(w, state, FALSE); }

	/* Get the global write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(archiver->last_write_protect_state < 0)
	{
	    write_protect_changed = TRUE;
	}
	else
	{
	    if((write_protect && !archiver->last_write_protect_state) ||
	       (!write_protect && archiver->last_write_protect_state)
	    )
		write_protect_changed = TRUE;
	}
	archiver->last_write_protect_state = write_protect ? 1 : 0;

	/* Delete method */
	delete_method = (edv_delete_method)EDV_GET_I(
	    EDV_CFG_PARM_DELETE_METHOD
	);

	/* Get last selected row on the contents clist */
	clist = GTK_CLIST(archiver->contents_clist);
	selected_obj_row = EDVCListGetSelectedLast(clist, NULL);

	/* Update recycle bin icon data, setting it to not NULL if it
	 * needs to be changed and update the recycle bin's last_recbin_items
	 * to match the value on the core structure
	 */
	icon_data = NULL;
	if(archiver->last_recbin_items < 0)
	{
	    if(core->last_recbin_items > 0)
		icon_data = (guint8 **)icon_trash_20x20_xpm;
	    else
		icon_data = (guint8 **)icon_trash_empty_20x20_xpm;
	}
	else
	{
	    if(!archiver->last_recbin_items &&
	       core->last_recbin_items
	    )
		icon_data = (guint8 **)icon_trash_20x20_xpm;
	    else if(archiver->last_recbin_items &&
		    !core->last_recbin_items
	    )
		icon_data = (guint8 **)icon_trash_empty_20x20_xpm;
	}
	archiver->last_recbin_items = core->last_recbin_items;
	recbin_icon_data = icon_data;

	/* Get the current location */
	location = EDVArchiverCurrentLocation(archiver);


        /* Update the find bar */
        EDVFindBarUpdateMenus(archiver->find_bar);



	/* Begin updating menu items and tool bar items */

	/* Edit */
	/* Add */
	sensitive = !STRISEMPTY(location);
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_add_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_ARCHIVER_OP_ADD, sensitive
	);
	/* Extract all */
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_extract_all_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_ARCHIVER_OP_EXTRACT_ALL, sensitive
	);

	/* Extract */
	sensitive = (selected_obj_row < 0) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_extract_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_ARCHIVER_OP_EXTRACT, sensitive
	);
	/* Delete */
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_delete_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_ARCHIVER_OP_DELETE, sensitive
	);

	/* Properties */
	sensitive = !STRISEMPTY(location);
	GTK_WIDGET_SET_SENSITIVE(archiver->properties_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_ARCHIVER_OP_PROPERTIES, sensitive
	);


	/* View */
	/* Archive Info */
	sensitive = !STRISEMPTY(location);
	GTK_WIDGET_SET_SENSITIVE(
	    archiver->comment_and_statistics_mi, sensitive
	);
	ToolBarItemSetSensitiveID(
	    tb, EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS, sensitive
	);
	/* Show Tool Bar */
	w = archiver->show_tool_bar_micheck;
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR);
	DO_SET_CHECK_STATE
	ToolBarItemSetToggleID(
	    tb, EDV_ARCHIVER_OP_SHOW_TOOL_BAR, state
	);
	/* Show Location Bar */
	w = archiver->show_location_bar_micheck;
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR);
	DO_SET_CHECK_STATE
	ToolBarItemSetToggleID(
	    tb, EDV_ARCHIVER_OP_SHOW_LOCATION_BAR, state
	);
	/* Show Find Bar */
	w = archiver->show_find_bar_micheck;
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR);
	DO_SET_CHECK_STATE
	ToolBarItemSetToggleID(
	    tb, EDV_ARCHIVER_OP_SHOW_FIND_BAR, state
	);
	/* Show Status Bar */
	w = archiver->show_status_bar_micheck;
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR);
	DO_SET_CHECK_STATE
	ToolBarItemSetToggleID(
	    tb, EDV_ARCHIVER_OP_SHOW_STATUS_BAR, state
	);


	/* Settings */
	if(write_protect_changed)
	{
	    w = archiver->write_protect_mi;
	    if(w != NULL)
		GUIMenuItemSetPixmap(
		    w,
		    (guint8 **)(write_protect ?
			icon_secure_20x20_xpm : icon_insecure_20x20_xpm)
		);
	    ToolBarItemUpdateByID(
		tb, EDV_ARCHIVER_OP_WRITE_PROTECT,
		NULL,
		(guint8 **)(write_protect ?
		    icon_secure_20x20_xpm : icon_insecure_20x20_xpm),
		NULL
	    );
	}

	/* Delete Method: Recycle */
	w = archiver->delete_method_recycle_micheck;
	GUIMenuItemSetPixmap(w, recbin_icon_data);
	state = (delete_method == EDV_DELETE_METHOD_RECYCLE) ?
	    TRUE : FALSE;
	DO_SET_CHECK_STATE
	ToolBarItemUpdateByID(
	    tb, EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE,
	    NULL, recbin_icon_data, NULL
	);
	ToolBarItemSetToggleID(
	    tb, EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE, state
	);

	/* Delete Method: Purge */
	w = archiver->delete_method_purge_micheck;
	state = (delete_method == EDV_DELETE_METHOD_PURGE) ?
	    TRUE : FALSE;
	DO_SET_CHECK_STATE
	ToolBarItemSetToggleID(
	    tb, EDV_ARCHIVER_OP_DELETE_METHOD_PURGE, state
	);


	/* Windows */
	if(recbin_icon_data != NULL)
	{
	    GUIMenuItemSetPixmap(
		archiver->window_recycle_bin_mi, recbin_icon_data
	    );
	    ToolBarItemUpdateByID(
		tb, EDV_ARCHIVER_OP_RECYCLE_BIN,
		NULL, recbin_icon_data, NULL
	    );
	}


	/* Contents List popup menu */
	sensitive = (selected_obj_row < 0) ?
	    FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(
	    archiver->contents_clist_extract_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    archiver->contents_clist_delete_mi, sensitive
	);

	sensitive = !STRISEMPTY(location);
	GTK_WIDGET_SET_SENSITIVE(
	    archiver->contents_clist_comment_and_statistics_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    archiver->contents_clist_properties_mi, sensitive
	);


	/* Status Bar */
	EDVStatusBarUpdate(archiver->status_bar);

#undef DO_SHOW
#undef DO_HIDE
#undef DO_SET_CHECK_STATE

	archiver->freeze_count--;
}

/*
 *	Marks the archiver as busy or ready.
 */
void EDVArchiverSetBusy(edv_archiver_struct *archiver, gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	w = archiver->toplevel;
	core = archiver->core;

	if(busy)
	{
	    archiver->busy_count++;
	    if(archiver->busy_count > 1)
		return;

	    cur = EDVGetCursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
	    archiver->busy_count--;
	    if(archiver->busy_count < 0)
		archiver->busy_count = 0;
	    if(archiver->busy_count > 0)
		return;
	    cur = NULL;
	}

	if(w->window != NULL)
	{
	    gdk_window_set_cursor(w->window, cur);
	    gdk_flush();
	}
}


/*
 *	Checks if the Archiver is mapped.
 */
gboolean EDVArchiverIsMapped(edv_archiver_struct *archiver)
{
	if(archiver == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(archiver->toplevel));
}

/*
 *	Maps the Archiver.
 */
void EDVArchiverMap(edv_archiver_struct *archiver)
{
	if(archiver == NULL)
	    return;

	gtk_widget_show_raise(archiver->toplevel);
}

/*
 *	Unmaps the Archiver.
 */
void EDVArchiverUnmap(edv_archiver_struct *archiver)
{
        if(archiver == NULL)
            return;

	gtk_widget_hide(archiver->toplevel);
}

/*
 *	Deletes the Archiver.
 */
void EDVArchiverDelete(edv_archiver_struct *archiver)
{
	if(archiver == NULL)
	    return;

	/* Clear the Contents List */
	EDVArchiverContentsClear(archiver);

	/* Reset the last selected item references */
	archiver->contents_clist_selected_row = -1;

	/* Delete the Archive Info Dialog */
	EDVArchiveInfoDelete(archiver->arch_info_dlg);
	archiver->arch_info_dlg = NULL;

	EDVArchiverUnmap(archiver);

	archiver->freeze_count++;

	GTK_WIDGET_DESTROY(archiver->contents_clist_menu);
	EDVStatusBarDelete(archiver->status_bar);
	EDVFindBarDelete(archiver->find_bar);
	ToolBarDelete(archiver->tool_bar);
	GTK_WIDGET_DESTROY(archiver->toplevel);

	/* Delete the Operation IDs */
	EDVArchiverOPIDListDelete(archiver->opid_list);

	g_free(archiver->contents_list_filter);

	archiver->freeze_count--;

	g_free(archiver);
}
