#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#ifdef __SOLARIS__
# include "../include/os.h"
#endif

#include "guiutils.h"
#include "pie_chart.h"
#include "cdialog.h"
#include "fb.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_archive_stat.h"
#include "edv_archive_cmt.h"
#include "archive_info_dlg.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_browse_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"


static gint EDVArchiveInfoDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint EDVArchiveInfoTextEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVArchiveInfoAnyChangedCB(GtkWidget *widget, gpointer data);

static void EDVArchiveInfoPasteCommentCB(GtkWidget *widget, gpointer data);
static void EDVArchiveInfoInsertCommentFileCB(GtkWidget *widget, gpointer data);
static void EDVArchiveInfoSetCommentCB(GtkWidget *widget, gpointer data);

static void EDVArchiveInfoOKCB(GtkWidget *widget, gpointer data);
static void EDVArchiveInfoCancelCB(GtkWidget *widget, gpointer data);

static void EDVArchiveInfoInsertCommentFile(
	edv_archive_info_dlg_struct *d, const gchar *path
);

edv_archive_info_dlg_struct *EDVArchiveInfoNew(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *ref_toplevel
);
void EDVArchiveInfoSetHasChanges(
	edv_archive_info_dlg_struct *d, const gboolean has_changes
);
void EDVArchiveInfoUpdateMenus(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoSetBusy(edv_archive_info_dlg_struct *d, const gboolean is_busy);
gboolean EDVArchiveInfoIsMapped(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoMap(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoUnmap(edv_archive_info_dlg_struct *d);
void EDVArchiveInfoDelete(edv_archive_info_dlg_struct *d);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Archive Info toplevel GtkWindow "delete_event" signal callback.
 */
static gint EDVArchiveInfoDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVArchiveInfoCancelCB(widget, data);
	return(TRUE);
}

/*
 *	GtkText event signal callback.
 */
static gint EDVArchiveInfoTextEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	GtkText *text;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	text = GTK_TEXT(widget);

	switch((gint)event->type)
	{
	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON4:
		/* Scroll up */
		if(text->vadj != NULL)
		{
		    GtkAdjustment *adj = text->vadj;
		    const gfloat inc = MAX(
			(0.25f * adj->page_size),
			adj->step_increment
		    );
		    gfloat v = adj->value - inc;
		    if(v > (adj->upper - adj->page_size))
			v = adj->upper - adj->page_size;
		    if(v < adj->lower)
			v = adj->lower;
		    gtk_adjustment_set_value(adj, v);
		}
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_press_event"
		);
		status = TRUE;
		break;
	      case GDK_BUTTON5:
		/* Scroll down */
		if(text->vadj != NULL)
		{
		    GtkAdjustment *adj = text->vadj;
		    const gfloat inc = MAX(
			(0.25f * adj->page_size),
			adj->step_increment
		    );
		    gfloat v = adj->value + inc;
		    if(v > (adj->upper - adj->page_size))
			v = adj->upper - adj->page_size;
		    if(v < adj->lower)
			v = adj->lower;
		    gtk_adjustment_set_value(adj, v);
		}
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_press_event"
		);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON4:
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_release_event"
		);
		status = TRUE;
		break;
	      case GDK_BUTTON5:
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_release_event"
		);
		status = TRUE;
		break;
	    }
	    break;
	}

	return(status);
}

/*
 *	Any GtkWidget "changed" signal callback.
 */
static void EDVArchiveInfoAnyChangedCB(GtkWidget *widget, gpointer data)
{
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(!d->has_changes)
	    EDVArchiveInfoSetHasChanges(d, TRUE);

	d->freeze_count--;
}

/*
 *	Paste comment callback.
 */
static void EDVArchiveInfoPasteCommentCB(GtkWidget *widget, gpointer data)
{
	gchar *s;
	GtkText *text;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	EDVArchiveInfoSetBusy(d, TRUE);

	text = GTK_TEXT(d->comment_text);

	/* Paste */
	s = GUIDDEGetString(
	    GTK_WIDGET(text),		/* Widget */
	    GDK_SELECTION_PRIMARY,	/* Selection */
	    GDK_CURRENT_TIME		/* Time */
	);
	if(s != NULL)
	{
#if 0
/* The GtkEditable widget already has a handler to receive the paste
 * so we only need to make a request and not actually paste
 */
	    gtk_text_freeze(text);
	    gtk_text_insert(
		text, NULL, NULL, NULL,
		s, -1
	    );
	    gtk_text_thaw(text);
#endif
	    g_free(s);
	}

	EDVArchiveInfoSetHasChanges(d, TRUE);

	EDVArchiveInfoSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Insert comment file callback.
 */
static void EDVArchiveInfoInsertCommentFileCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar *parent_path, **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	EDVArchiveInfoSetBusy(d, TRUE);

	toplevel = d->toplevel;

	/* Get startup path from archive path */
	parent_path = (d->arch_path != NULL) ?
	    g_dirname(d->arch_path) : NULL;

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftype, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the file to insert */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Insert File",
	    "Insert", "Cancel",
	    parent_path,	/* Startup path */
	    ftype, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	    EDVArchiveInfoInsertCommentFile(d, paths_list[0]);

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, nftypes);

	/* Delete startup path */
	g_free(parent_path);

	EDVArchiveInfoSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Set Comment Button callback.
 */
static void EDVArchiveInfoSetCommentCB(GtkWidget *widget, gpointer data)
{
	gint status, strc;
	gchar *s, **strv;
	const gchar *arch_path;
	GtkWidget *toplevel;
	GtkEditable *editable;
	GtkText *text;
	edv_core_struct *core;
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	toplevel = d->toplevel;
	editable = GTK_EDITABLE(d->comment_text);
	text = GTK_TEXT(editable);
	core = d->core;
	arch_path = d->arch_path;
	if(STRISEMPTY(arch_path))
	    return;

	EDVArchiveInfoSetBusy(d, TRUE);

	d->freeze_count++;

	/* Get the comment and explode it into multiple lines */
	s = gtk_editable_get_chars(editable, 0, -1);
	strv = g_strsplit(s, "\n", -1);
	g_free(s);
	for(strc = 0; strv[strc] != NULL; strc++);

	/* Set the archive's comment */
	status = EDVArchCommentSet(
	    core, arch_path, strv, strc
	);

	/* Delete the comment lines */
	g_strfreev(strv);

	d->freeze_count--;

	if(status == 0)
	{
	    struct stat lstat_buf;

	    /* Reset has changes */
	    EDVArchiveInfoSetHasChanges(d, FALSE);

	    /* Notify about the archive being modified */
	    if(lstat((const char *)arch_path, &lstat_buf))
	    {
		const gint error_code = (gint)errno;
#ifdef ENOENT
		if(error_code == ENOENT)
		    EDVObjectRemovedEmit(core, arch_path);
#endif
	    }
	    else
	    {
		EDVObjectModifiedEmit(core, arch_path, arch_path, &lstat_buf);
	    }
	}

	/* Error setting archive comment? */
	if(status != 0)
	{
	    const gchar *msg;

	    switch(status)
	    {
	      case -3:
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"El error de sistemas."
#elif defined(PROG_LANGUAGE_FRENCH)
"Erreur de systmes."
#elif defined(PROG_LANGUAGE_GERMAN)
"Systeme fehler."
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'errore di sistemi."
#elif defined(PROG_LANGUAGE_DUTCH)
"Systemen fout."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O erro de sistemas."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Systemfeil."
#elif defined(PROG_LANGUAGE_POLISH)
"Bd systemu."
#else
"Systems error."
#endif
		;
		break;
	      case -2:
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Este formato del archivo no sostiene los comentarios."
#elif defined(PROG_LANGUAGE_FRENCH)
"Ce format de l'archive pas soutien commente."
#elif defined(PROG_LANGUAGE_GERMAN)
"Das format dieses archivs untersttzt bemerkungen nicht."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Questo formato dell'archivio non sostiene i commenti."
#elif defined(PROG_LANGUAGE_DUTCH)
"Het formaat van deze archief steunt opmerkingen niet."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Este formato do arquivo nao apoia comentrios."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dette arkivs format sttter ikke kommentarer."
#elif defined(PROG_LANGUAGE_POLISH)
"Format tego archiwum nie obsuguje komentarzy."
#else
"This archive's format does not support comments."
#endif
		;
		break;
	      default:
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"El comentario del archivo de poner de error."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le commentaire d'archive de montage d'erreur."
#elif defined(PROG_LANGUAGE_GERMAN)
"Fehler setzen archiv bemerkung."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il commento di archivio di montaggio di errore."
#elif defined(PROG_LANGUAGE_DUTCH)
"Fout zetten archief opmerking."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Erro por o comentrio de arquivo."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"En feil forekommet whiles etting arkivkommentaren."
#elif defined(PROG_LANGUAGE_POLISH)
"Bd przy ustawianiu komentarza archiwum."
#else
"Error setting archive comment."
#endif
		;
		break;
	    }
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
#if defined(PROG_LANGUAGE_SPANISH)
"El Comentario Fijo Fall"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rgler Le Commentaire Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Feste Bemerkung Hat Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha Regolato Il Commento Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vast Opmerking Verzuimde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comentrio Fixo Fracassou"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fast Comment Failed"
#elif defined(PROG_LANGUAGE_POLISH)
"Bd ustawiania komentarza"
#else
"Set Comment Failed"
#endif
		, msg,
		NULL,
		toplevel
	    );
	}

	EDVArchiveInfoSetBusy(d, FALSE);
}

/*
 *	OK callback.
 */
static void EDVArchiveInfoOKCB(GtkWidget *widget, gpointer data)
{
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	EDVArchiveInfoUnmap(d);
	EDVArchiveInfoSetHasChanges(d, FALSE);

	d->freeze_count--;
}

/*
 *	Cancel callback.
 */
static void EDVArchiveInfoCancelCB(GtkWidget *widget, gpointer data)
{
	edv_archive_info_dlg_struct *d = EDV_ARCHIVE_INFO_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	EDVArchiveInfoUnmap(d);
	EDVArchiveInfoSetHasChanges(d, FALSE);

	d->freeze_count--;
}


/*
 *	Inserts the contents in the file specified by path into the
 *	comment text.
 */
static void EDVArchiveInfoInsertCommentFile(
	edv_archive_info_dlg_struct *d, const gchar *path
)
{
	FILE *fp;
	GtkWidget *toplevel;
	GtkText *text;

	if((d == NULL) || STRISEMPTY(path))
	    return;

	toplevel = d->toplevel;
	text = GTK_TEXT(d->comment_text);

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp != NULL)
	{
	    struct stat stat_buf;
	    gulong block_size;
	    guint8 *buf;

	    gtk_text_freeze(text);

	    if(fstat(fileno(fp), &stat_buf))
	    {
		block_size = 0l;
	    }
	    else
	    {
#if defined(_WIN32)
		block_size = 1024l;
#else
		block_size = (gulong)stat_buf.st_blksize;
#endif
	    }

	    /* Allocate the read buffer */
	    buf = (block_size > 0l) ?
		(guint8 *)g_malloc(block_size * sizeof(guint8)) : NULL;
	    if(buf != NULL)
	    {
		gint units_read;
		gchar *s, *s_end;

		/* Begin reading the file one block at a time */
		while(!feof(fp))
		{
		    /* Read the next block from the file */
		    units_read = (gint)fread(
			buf, sizeof(guint8), (size_t)block_size, fp
		    );
		    if(units_read <= 0)
			break;

		    /* Convert non-printable characters into spaces */
		    s = (gchar *)buf;
		    s_end = s + units_read;
		    while(s < s_end)
		    {
			/* Not a printable character? */
			if(!isprint((int)(*s)))
			{
			    /* Exclude newlines */
			    if(*s != '\n')
				*s = ' ';
			}
			s++;
		    }

		    /* Insert this block to the comment text */
		    gtk_text_insert(
			text, NULL, NULL, NULL,
			(const gchar *)buf, units_read
		    );
		}

		/* Delete the read buffer */
		g_free(buf);
	    }
	    else
	    {
		gint c;
		guint8 buf[1];

		while(!feof(fp))
		{
		    /* Read the next character from the file */
		    c = (gint)fgetc(fp);
		    if((int)c == EOF)
			break;

		    /* Convert non-printable characters into spaces */
		    if(!isprint((int)c) && (c != '\n'))
			c = ' ';

		    buf[0] = c;

		    /* Insert this character to the comment text */
		    gtk_text_insert(
			text, NULL, NULL, NULL,
			(const gchar *)buf, 1
		    );
		}
	    }

	    /* Close the file */
	    fclose(fp);

	    gtk_text_thaw(text);

	    EDVArchiveInfoSetHasChanges(d, TRUE);
	}
	else
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
		g_strerror(error_code), path
	    );
	    EDVPlaySoundWarning(d->core);
	    EDVMessageWarning(
		"Insert File Failed",
                msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	}
}


/*
 *	Creates a new Archive Info Dialog.
 *
 *	If arch_path is not NULL then the archive object information will
 *	be loaded.
 */
edv_archive_info_dlg_struct *EDVArchiveInfoNew(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *ref_toplevel
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *s, *s2, *comment = NULL;
	gint	total_objects = 0,
		file_objects = 0,
		directory_objects = 0,
		link_objects = 0,
		other_objects = 0;
	gulong	files_size = 0l,
		directories_size = 0l,
		links_size = 0l,
		others_size = 0l,
		uncompressed_size = 0l;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GdkWindow *window;
	GtkRcStyle *rcstyle;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4, *parent5,
			*toplevel;
	GtkEditable *editable;
	GtkText *text;
	pie_chart_struct *pc;
	edv_object_struct *obj;
	edv_archive_info_dlg_struct *d;

	if(core == NULL)
	    return(NULL);

	d = EDV_ARCHIVE_INFO_DLG(g_malloc0(
	    sizeof(edv_archive_info_dlg_struct)
	));
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->has_changes = FALSE;
	d->freeze_count = 0;
	d->busy_count = 0;
	d->core = core;

	d->arch_path = STRDUP(arch_path);

	d->busy_cur = gdk_cursor_new(GDK_WATCH);

	d->freeze_count++;

	/* Get the archive's statistics */
	if(STRISEMPTY(arch_path))
	{
	    obj = NULL;
	}
	else
	{
	    GList *obj_list;

	    obj = EDVObjectNew();
	    if(obj != NULL)
	    {
		struct stat lstat_buf;

		EDVObjectSetPath(obj, arch_path);
		if(!lstat((const char *)arch_path, &lstat_buf))
		    EDVObjectSetStat(obj, &lstat_buf);
		EDVObjectUpdateLinkFlags(obj);

		/* Get the icon for the archive */
		EDVMatchObjectIcon(
		    core->device, core->total_devices,
		    core->mimetype, core->total_mimetypes,
		    obj->type,
		    obj->full_path,
		    EDV_OBJECT_IS_LINK_VALID(obj),
		    obj->permissions,
		    1,			/* Medium sized icons */
		    &pixmap, &mask,
		    NULL, NULL,
		    NULL, NULL,
		    NULL, NULL
		);

		/* Get the number of objects and total size */
		obj_list = EDVArchStatList(
		    core, arch_path,
		    NULL,		/* Get all the objects */
		    NULL,		/* No filter */
		    password,
		    NULL, NULL		/* No progress callback */
		);
		if(obj_list != NULL)
		{
		    gulong this_obj_size;
		    GList *glist;
		    edv_archive_object_struct *arch_obj;

		    for(glist = obj_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			arch_obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(arch_obj == NULL)
			    continue;

			total_objects++;
			switch(arch_obj->type)
			{
			  case EDV_OBJECT_TYPE_UNKNOWN:
			    break;
			  case EDV_OBJECT_TYPE_FILE:
			    file_objects++;
			    this_obj_size = arch_obj->size + (
				(arch_obj->full_path != NULL) ?
				STRLEN(arch_obj->full_path) : STRLEN(arch_obj->name)
			    );
			    files_size += this_obj_size;
			    uncompressed_size += this_obj_size;
			    break;
			  case EDV_OBJECT_TYPE_DIRECTORY:
			    directory_objects++;
			    this_obj_size = arch_obj->size + (
				(arch_obj->full_path != NULL) ?
				STRLEN(arch_obj->full_path) : STRLEN(arch_obj->name)
			    );
			    directories_size += this_obj_size;
			    uncompressed_size += this_obj_size;
			    break;
			  case EDV_OBJECT_TYPE_LINK:
			    link_objects++;
			    this_obj_size = arch_obj->size + (
				(arch_obj->full_path != NULL) ?
				STRLEN(arch_obj->full_path) : STRLEN(arch_obj->name)
			    ) + STRLEN(arch_obj->link_target);
			    links_size += this_obj_size;
			    uncompressed_size += this_obj_size;
			    break;
			  default:
			    other_objects++;
			    this_obj_size = arch_obj->size + (
				(arch_obj->full_path != NULL) ?
				STRLEN(arch_obj->full_path) : STRLEN(arch_obj->name)
			    );
			    others_size += this_obj_size;
			    uncompressed_size += this_obj_size;
			    break;
			}

			EDVArchObjectDelete(arch_obj);
		    }
		    g_list_free(obj_list);
		}
	    }

	    /* Get the comment */
	    comment = EDVArchCommentGet(core, arch_path);
	    uncompressed_size += STRLEN(comment);
	}

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
/*	    GUISetWMIcon(window, (guint8 **)archiver_48x48_xpm); */
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : FALSE
	)
	{
	    gtk_window_set_transient_for(
		GTK_WINDOW(w), GTK_WINDOW(ref_toplevel)
	    );
	}
	parent = w;

	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Information & Statistics GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;


	/* GtkHBox to make two columns */
	w = gtk_hbox_new(FALSE, 20);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Icon & Name GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Icon */
	if(pixmap != NULL)
	{
	    d->icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}
	/* Label */
	if(obj != NULL)
	    s = EDVShortenPath(obj->name, 40);
	else
	    s = STRDUP("");
	d->name_label = w = gtk_label_new(s);
	g_free(s);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1"
#endif
	);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);


	/* Location GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#elif defined(PROG_LANGUAGE_POLISH)
"cieka"
#else
"Location"
#endif
	    ":"
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);
	/* Location Label */
	if(obj != NULL)
	    s2 = g_dirname(obj->full_path);
	else
	    s2 = STRDUP("");
	s = EDVShortenPath(s2, 45);
	d->location_label = w = gtk_label_new(s);
	g_free(s);
	g_free(s2);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);


	/* GtkHBox to separate columns for object and size break-down */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Table for object and size break-down labels */
	w = gtk_table_new(5, 3, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(w), border_minor);
	gtk_table_set_col_spacings(GTK_TABLE(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

#define CREATE_LABEL(r,n,i,b)	{				\
 GtkAttachOptions	x_attach_opt = 0,			\
			y_attach_opt = 0;			\
 guint	x_pad = 0, y_pad = 0;					\
								\
 /* Alignment for name label */					\
 w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);			\
 gtk_widget_set_usize(w, 100, -1);				\
 gtk_table_attach(						\
  GTK_TABLE(parent4), w,					\
  0, 1, (r), (r) + 1,						\
  x_attach_opt, y_attach_opt, x_pad, y_pad			\
 );								\
 gtk_widget_show(w);						\
 parent5 = w;							\
 /* Name label */						\
 w = gtk_label_new((n));					\
 gtk_container_add(GTK_CONTAINER(parent5), w);			\
 gtk_widget_modify_style(w, rcstyle);				\
 gtk_widget_show(w);						\
								\
 /* Alignment for quantity label */				\
 w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);			\
 gtk_widget_set_usize(w, 50, -1);				\
 gtk_table_attach(						\
  GTK_TABLE(parent4), w,					\
  1, 2, (r), (r) + 1,						\
  x_attach_opt, y_attach_opt, x_pad, y_pad			\
 );								\
 gtk_widget_show(w);						\
 parent5 = w;							\
 /* Quantity label */						\
 s = g_strdup_printf("%i", (i));				\
 w = gtk_label_new(s);						\
 g_free(s);							\
 gtk_container_add(GTK_CONTAINER(parent5), w);			\
 gtk_widget_modify_style(w, rcstyle);				\
 gtk_widget_show(w);						\
								\
 /* Alignmet for size label */					\
 w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);			\
 gtk_widget_set_usize(w, 150, -1);				\
 gtk_table_attach(						\
  GTK_TABLE(parent4), w,					\
  2, 3, (r), (r) + 1,						\
  x_attach_opt, y_attach_opt, x_pad, y_pad			\
 );								\
 gtk_widget_show(w);						\
 parent5 = w;							\
 /* Size label */						\
 s = g_strdup_printf(						\
  "%s %s",							\
  EDVSizeStrDelim(b),						\
  ((b) == 1) ? "byte" : "bytes"					\
 );								\
 w = gtk_label_new(s);						\
 g_free(s);							\
 gtk_container_add(GTK_CONTAINER(parent5), w);			\
 gtk_widget_modify_style(w, rcstyle);				\
 gtk_widget_show(w);						\
}

	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
#if defined(PROG_LANGUAGE_SPANISH)
	CREATE_LABEL(0, "Los Archivos:", file_objects, files_size);
	CREATE_LABEL(1, "Las Guas:", directory_objects, directories_size);
	CREATE_LABEL(2, "Las Conexiones:", link_objects, links_size);
	CREATE_LABEL(3, "Otros Objetos:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_FRENCH)
	CREATE_LABEL(0, "Fichiers:", file_objects, files_size);
	CREATE_LABEL(1, "Annuaires:", directory_objects, directories_size);
	CREATE_LABEL(2, "Liens:", link_objects, links_size);
	CREATE_LABEL(3, "Autres Objets:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_GERMAN)
	CREATE_LABEL(0, "Akten:", file_objects, files_size);
	CREATE_LABEL(1, "Verzeichnisse:", directory_objects, directories_size);
	CREATE_LABEL(2, "Kettenglieder:", link_objects, links_size);
	CREATE_LABEL(3, "Andere Objekte:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_ITALIAN)
	CREATE_LABEL(0, "I File:", file_objects, files_size);
	CREATE_LABEL(1, "Gli Elenchi:", directory_objects, directories_size);
	CREATE_LABEL(2, "Le Maglie:", link_objects, links_size);
	CREATE_LABEL(3, "L'Altro Obbietta:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_DUTCH)
	CREATE_LABEL(0, "Dossiers:", file_objects, files_size);
	CREATE_LABEL(1, "Gidzen:", directory_objects, directories_size);
	CREATE_LABEL(2, "Schakels:", link_objects, links_size);
	CREATE_LABEL(3, "Anderze Voorwerpen:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	CREATE_LABEL(0, "Os Arquivos:", file_objects, files_size);
	CREATE_LABEL(1, "Os Guias:", directory_objects, directories_size);
	CREATE_LABEL(2, "Os Elos:", link_objects, links_size);
	CREATE_LABEL(3, "Outro Ope-se:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	CREATE_LABEL(0, "Arkiver:", file_objects, files_size);
	CREATE_LABEL(1, "Kataloger:", directory_objects, directories_size);
	CREATE_LABEL(2, "Ledd:", link_objects, links_size);
	CREATE_LABEL(3, "Annen Object:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_POLISH)
	CREATE_LABEL(0, "Pliki:", file_objects, files_size);
	CREATE_LABEL(1, "Katalogi:", directory_objects, directories_size);
	CREATE_LABEL(2, "Odnoniki:", link_objects, links_size);
	CREATE_LABEL(3, "Inne obiekty:", other_objects, others_size);
#else
	CREATE_LABEL(0, "Files:", file_objects, files_size);
	CREATE_LABEL(1, "Directories:", directory_objects, directories_size);
	CREATE_LABEL(2, "Links:", link_objects, links_size);
	CREATE_LABEL(3, "Misc Objects:", other_objects, others_size);
#endif
	GTK_RC_STYLE_UNREF(rcstyle);

	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-helvetica-bold-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-helvetica-bold-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
#if defined(PROG_LANGUAGE_SPANISH)
	CREATE_LABEL(4, "El Suma:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_FRENCH)
	CREATE_LABEL(4, "Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_GERMAN)
	CREATE_LABEL(4, "Summe:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_ITALIAN)
	CREATE_LABEL(4, "Il Totale:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_DUTCH)
	CREATE_LABEL(4, "Totaal:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	CREATE_LABEL(4, "O Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	CREATE_LABEL(4, "Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_POLISH)
	CREATE_LABEL(4, "Razem:", total_objects, uncompressed_size);
#else
	CREATE_LABEL(4, "Total:", total_objects, uncompressed_size);
#endif
	GTK_RC_STYLE_UNREF(rcstyle);

#undef CREATE_LABEL



	/* Right column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Size pie chart */
	if(obj != NULL)
	{
	    gulong v;
	    gchar *s2;
	    GdkColor *c, uncompressed_size_color, current_size_color;

	    /* Uncompressed Size */
	    GtkAdjustment *adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, (gfloat)MAX(uncompressed_size, obj->size),
		0.0f, 0.0f, 0.0f
	    );
	    c = &uncompressed_size_color;
	    GDK_COLOR_SET_COEFF(c, 0.0f, 0.0f, 1.0f);
	    v = MAX(uncompressed_size, obj->size);
	    s = g_strdup_printf(
		"%s %s",
		EDVSizeStrDelim(v),
		(v == 1l) ? "byte" : "bytes"
	    );
	    s2 = g_strdup_printf(
		"%.0f%% Compressed",
		CLIP(100.0f - (((uncompressed_size > 0) ?
		    (gfloat)obj->size / (gfloat)uncompressed_size : 1.0f
		) * 100.0f), 0.0f, 100.0f)
	    );

	    /* Pie Chart */
	    d->piechart = pc = PieChartNew(
		110, 70,
		adj, c,
#if defined(PROG_LANGUAGE_SPANISH)
		"El Tamao", s2,
		"Ensanchado:", s
#elif defined(PROG_LANGUAGE_FRENCH)
		"Taille", s2,
		"Augment:", s
#elif defined(PROG_LANGUAGE_GERMAN)
		"Gre", s2,
		"Ausgebreitet:", s
#elif defined(PROG_LANGUAGE_ITALIAN)
		"La Misura", s2,
		"Allargato:", s
#elif defined(PROG_LANGUAGE_DUTCH)
		"Maat", s2,
		"Uitgebreide:", s
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"O Tamanho", s2,
		"Expandido:", s
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Strrelse", s2,
		"Utvidet:", s
#elif defined(PROG_LANGUAGE_POLISH)
		"Rozmiar", s2,
		"Niezkompresowany:", s
#else
		"Size", s2,		/* Header & Footer */
		"Uncompressed:", s
#endif
	    );
	    g_free(s);
	    g_free(s2);
	    gtk_object_unref(GTK_OBJECT(adj));
	    w = pc->toplevel;
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    PieChartMap(pc);

	    /* Current Size */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, (gfloat)obj->size,
		0.0f, 0.0f, 0.0f
	    );
	    c = &current_size_color;
	    GDK_COLOR_SET_COEFF(c, 1.0f, 0.0f, 1.0f);
	    s = g_strdup_printf(
		"%s %s",
		EDVSizeStrDelim(obj->size),
		(obj->size == 1l) ? "byte" : "bytes"
	    );
	    PieChartValueAdd(
		pc, adj, c,
#if defined(PROG_LANGUAGE_SPANISH)
"Comprimido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Serr"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zusammengedrckt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Compresso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Samengedrukte"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comprimido"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Komprimert"
#elif defined(PROG_LANGUAGE_POLISH)
"Zkompresowany"
#else
"Compressed"
#endif
		":", s
	    );
	    g_free(s);
	    gtk_object_unref(GTK_OBJECT(adj));
	}


	/* Comments GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Comentario"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commentaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bemerkung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Commento"
#elif defined(PROG_LANGUAGE_DUTCH)
"Opmerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Comentrio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommentar"
#elif defined(PROG_LANGUAGE_POLISH)
"Komentarz"
#else
"Comment"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Comments text and scroll bars GtkTable */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Comments GtkText */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	d->comment_text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
	gtk_widget_set_usize(w, 520, 150);
	text->default_tab_width = 8;
	gtk_text_set_editable(text, TRUE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
	    GTK_TABLE(parent3), w,
	    0, 1, 0, 1,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    0, 0
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoTextEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoTextEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoAnyChangedCB), d
	);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_UNDO);
	if(!STRISEMPTY(comment))
	{
	    gtk_text_freeze(text);
	    gtk_text_insert(text, NULL, NULL, NULL, comment, -1);
	    gtk_text_thaw(text);
	}
	gtk_widget_show(w);
	/* Vertical GtkScrollBar */
	w = gtk_vscrollbar_new(text->vadj);
	gtk_table_attach(
	    GTK_TABLE(parent3), w,
	    1, 2, 0, 1,
	    GTK_FILL,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    0, 0
	);
	gtk_widget_show(w);

	/* Set comment button GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Set Comment Button */
	d->comment_set_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"El Conjunto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Srie"
#elif defined(PROG_LANGUAGE_GERMAN)
"Satz"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Serie"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Jogo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sett"
#elif defined(PROG_LANGUAGE_POLISH)
"Ustaw"
#else
"Set"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoSetCommentCB), d
	);
	gtk_widget_show(w);


	/* Comment utils buttons GtkHBox */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Paste Comment Button */
	d->comment_paste_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_paste_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoPasteCommentCB), d
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"La Pasta"
#elif defined(PROG_LANGUAGE_FRENCH)
"Pte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Paste"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Pasta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plakmiddel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Pasta"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Masse"
#elif defined(PROG_LANGUAGE_POLISH)
"Wklej"
#else
"Paste"
#endif
	);
	gtk_widget_show(w);
	/* Insert Comment File Button */
	d->comment_insert_file_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_browse_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoInsertCommentFileCB), d
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"El Archivo De La Adicin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier D'Insertion"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen Sie Akte Ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il File Di Inserzione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Voeg Dossier In"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Insira Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy Arkiv"
#elif defined(PROG_LANGUAGE_POLISH)
"Wstaw plik"
#else
"Insert File"
#endif
	);
	gtk_widget_show(w);


	/* Delete the archive's statistics */
	EDVObjectDelete(obj);

	/* Delete the comment */
	g_free(comment);


	parent = d->main_vbox;

	/* Separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm, "Cancel", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVArchiveInfoCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	EDVArchiveInfoUpdateMenus(d);

	d->freeze_count--;

	return(d);
}

/*
 *	Resets the Archive Info Dialog has_changes marker.
 */
void EDVArchiveInfoSetHasChanges(
	edv_archive_info_dlg_struct *d, const gboolean has_changes
)
{
	if(d == NULL)
	    return;

	if(d->has_changes == has_changes)
	    return;

	d->has_changes = has_changes;
	EDVArchiveInfoUpdateMenus(d);
}

/*
 *	Updates the Archive Info Dialog's widgets to reflect current
 *	values.
 */
void EDVArchiveInfoUpdateMenus(edv_archive_info_dlg_struct *d)
{
	gboolean	sensitive,
			has_changes;
	gchar *s;

	if(d == NULL)
	    return;

	has_changes = d->has_changes;

	/* Title */
	s = g_strconcat(
#if defined(PROG_LANGUAGE_SPANISH)
"El Comentario & La Estadstica"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commenter & La Statistique"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bemerkung & Statistik"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Commento & La Statistica"
#elif defined(PROG_LANGUAGE_DUTCH)
"Opmerking & Statistieken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Comentrio & Estatstica"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommentar & Statistics"
#elif defined(PROG_LANGUAGE_POLISH)
"Komentarz i statystyki archiwum"
#else
"Archive Comment & Statistics"
#endif
	    , ": ",
	    (d->arch_path != NULL) ?
		g_basename(d->arch_path) : "Untitled",
	    (has_changes) ? " (*)" : "",
	    NULL
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	/* Set Comment Button */
	sensitive = has_changes;
	GTK_WIDGET_SET_SENSITIVE(d->comment_set_btn, sensitive);

	if(has_changes)
	{
	    gtk_widget_hide(d->ok_btn);
	    gtk_widget_show(d->cancel_btn);
	}
	else
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_hide(d->cancel_btn);
	}
}

/*
 *	Sets the Archive Info Dialog as busy or ready.
 */
void EDVArchiveInfoSetBusy(edv_archive_info_dlg_struct *d, const gboolean is_busy)
{
	GdkCursor *cur;
	GtkWidget *w;

	if(d == NULL)
	    return;

	w = d->toplevel;

	if(is_busy)
	{
	    d->busy_count++;
	    if(d->busy_count > 1)
		return;

	    cur = d->busy_cur;
	}
	else
	{
	    d->busy_count--;
	    if(d->busy_count < 0)
		d->busy_count = 0;

	    if(d->busy_count > 0)
		return;

	    cur = NULL;
	}

	gdk_window_set_cursor(w->window, cur);

	gdk_flush();
}

/*
 *	Checks if the Archive Info Dialog is mapped.
 */
gboolean EDVArchiveInfoIsMapped(edv_archive_info_dlg_struct *d)
{
	if(d == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the Archive Info Dialog.
 */
void EDVArchiveInfoMap(edv_archive_info_dlg_struct *d)
{
	GtkWidget *w;

	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);

	/* Grab focus and default on the OK button */
	w = d->ok_btn;
	if(w != NULL)
	{
	    gtk_widget_grab_focus(w);
	    gtk_widget_grab_default(w);
	}
}

/*
 *	Unmap the Archive Info Dialog.
 */
void EDVArchiveInfoUnmap(edv_archive_info_dlg_struct *d)
{
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the Archive Info Dialog.
 */
void EDVArchiveInfoDelete(edv_archive_info_dlg_struct *d)
{
	if(d == NULL)
	    return;

	EDVArchiveInfoUnmap(d);

	gtk_window_set_modal(GTK_WINDOW(d->toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(d->toplevel), NULL);

	d->freeze_count++;

	PieChartDelete(d->piechart);
	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	GDK_CURSOR_DESTROY(d->busy_cur);

	g_free(d->arch_path);

	d->freeze_count--;

	g_free(d);
}
