#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "cfg.h"
#include "edvobj.h"
#include "edvdevices.h"
#include "edvstatusbar.h"
#include "imbr.h"
#include "imbrcb.h"
#include "imbropcb.h"
#include "imbrtlist.h"
#include "imbrimgview.h"
#include "imbrdnd.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvlistseek.h"
#include "edvcfglist.h"
#include "config.h"


void EDVImbrTListItemDestroyCB(gpointer data);

gint EDVImbrDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint EDVImbrKeyEventCB(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint EDVImbrButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVImbrHandleChildAttachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void EDVImbrHandleChildDetachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void EDVImbrTListSelectCB(
	gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
);
void EDVImbrTListUnselectCB(
	gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
);

void EDVImbrComboActivateCB(GtkWidget *widget, gpointer data);

gint EDVImbrLoadingTOCB(gpointer data);

void EDVImbrWriteProtectChangedCB(
	edv_imbr_struct *imbr, gboolean state
);
void EDVImbrObjectAddedNotifyCB(
	edv_imbr_struct *imbr, const gchar *path,
	const struct stat *lstat_buf
);
void EDVimbrObjectModifiedNotifyCB(
	edv_imbr_struct *imbr, const gchar *path,
	const gchar *new_path,
	const struct stat *lstat_buf
);
void EDVImbrObjectRemovedNotifyCB(
	edv_imbr_struct *imbr, const gchar *path
);

void EDVImbrMountNotifyCB(
	edv_imbr_struct *imbr,
	gint dev_num, edv_device_struct *dev_ptr,
	gboolean is_mounted
);

void EDVImbrRecycledObjectAddedNotifyCB(
	edv_imbr_struct *imbr, guint index
);
void EDVImbrRecycledObjectRemovedNotifyCB(
	edv_imbr_struct *imbr, guint index
);

void EDVImbrReconfiguredNotifyCB(edv_imbr_struct *imbr);

void EDVImbrMimeTypeAddedCB(
	edv_imbr_struct *imbr,
	gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVImbrMimeTypeModifiedCB(
	edv_imbr_struct *imbr,
	gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVImbrMimeTypeRemovedCB(
	edv_imbr_struct *imbr, gint mt_num
);

void EDVImbrDeviceAddedCB(
	edv_imbr_struct *imbr,
	gint dev_num, edv_device_struct *dev_ptr
);
void EDVImbrDeviceModifiedCB(
	edv_imbr_struct *imbr,
	gint dev_num, edv_device_struct *dev_ptr
);
void EDVImbrDeviceRemovedCB(
	edv_imbr_struct *imbr, gint dev_num
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Image Browser thumbs list "destroy" signal callback.
 */
void EDVImbrTListItemDestroyCB(gpointer data)
{
	EDVObjectDelete(EDV_OBJECT(data));
}


/*
 *	Image Browser toplevel GtkWidget "delete_event" signal callback.
 */
gint EDVImbrDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if(imbr == NULL)
	    return(TRUE);

	if(imbr->processing)
	    return(TRUE);

	EDVImbrOPClose(imbr);

	return(TRUE);
}

/*
 *	Image Browser "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint EDVImbrKeyEventCB(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	guint keyval, state;
	gboolean press;
	tlist_struct *tlist;
	edv_core_struct *core_ptr;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((widget == NULL) || (key == NULL) || (imbr == NULL))
	    return(status);

	if(imbr->processing)
	    return(status);

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return(status);

	/* Get event type */
	etype = key->type;

	/* Get other key event values */
	press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return(status);

/* Stop emit of signal */
#define DO_STOP_KEY_SIGNAL_EMIT	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* Handle by key value */
	switch(keyval)
	{
#if 0
	  case GDK_Return:
	  case GDK_KP_Enter:
	  case GDK_ISO_Enter:
	  case GDK_3270_Enter:
	    /* Handle only if control key modifier is not held, so
	     * that accelerator keys will get handled properly
	     */
	    if(!(state & GDK_CONTROL_MASK))
	    {
		if(press)
		    EDVImbrTListDoOpenObject(
			imbr, imbr->tlist_selected_thumb, state
		    );
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
	    }
	    break;
#endif
	  case GDK_Delete:
	    if(press)
		EDVImbrOPDelete(imbr);
	    DO_STOP_KEY_SIGNAL_EMIT
	    status = TRUE;
	    break;

	  case GDK_Insert:
	    if(state & GDK_CONTROL_MASK)
	    {
		if(press)
		    EDVImbrOPCopyPath(imbr);
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
	    }
	    break;

	  default:
	    /* For all other alphanumeric character keys and while
	     * no modifier keys are held, attempt to seek to the
	     * item who's name starts with the letter of the key
	     * that was pressed
	     */
	    if(isalnum((char)keyval) && press &&
	       ((state == 0x00000000) || (state == GDK_SHIFT_MASK))
	    )
	    {
		TListFreeze(tlist);
		EDVTListSeekCharacter(
		    tlist, 0,
		    (gchar)((state & GDK_SHIFT_MASK) ?
			toupper((char)keyval) : keyval
		    )
		);
		TListThaw(tlist);

		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
	    }
	    break;
	}

	return(status);
#undef DO_STOP_KEY_SIGNAL_EMIT
}

/*
 *	Image Browser GtkWidget "button_press_event" signal callback.
 */
gint EDVImbrButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	guint state;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((widget == NULL) || (button == NULL) || (imbr == NULL))
	    return(status);

	if(imbr->processing)
	    return(status);

	tlist = imbr->tlist;
	core_ptr = EDV_CORE(imbr->core_ptr);
	if((tlist == NULL) || (core_ptr == NULL))
	    return(status);

	cfg_list = core_ptr->cfg_list;

	/* Get event type */
	etype = button->type;

	state = button->state;

	/* Event occured on the thumbs list? */
	if(widget == tlist->list_da)
	{
	    gint	thumb_num = -1, total_selected_thumbs = 0,
			single_selected_thumb = -1;
	    GList *glist = tlist->selection;


	    /* Get total number of selected thumbs and set
	     * single_selected_thumb to be the only single selected
	     * thumb or -1 of there are none or multiple thumbs
	     * selected
	     */
	    while(glist != NULL)
	    {
		single_selected_thumb = (gint)glist->data;
		total_selected_thumbs++;
		glist = g_list_next(glist);
	    }
	    if(total_selected_thumbs != 1)
		single_selected_thumb = -1;

	    /* Get thumb that button event occured over */
	    TListGetSelection(
		tlist, button->x, button->y, &thumb_num, NULL, NULL
	    );

	    /* Handle by button number */
	    switch(button->button)
	    {
	      case 3:
		if(etype == GDK_BUTTON_PRESS)
		{
		    GtkMenu *menu;

		    /* Select item before mapping menu? */
		    if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
		       (thumb_num > -1)
		    )
		    {
			/* Select the thumb that the button was pressed over */
			TListFreeze(tlist);
			if(!(button->state & GDK_CONTROL_MASK) &&
			   !(button->state & GDK_SHIFT_MASK) &&
			   (thumb_num != single_selected_thumb)
			)
			    TListUnselectAll(tlist);
			tlist->focus_thumb = thumb_num;
			TListSelectThumb(tlist, thumb_num);
			TListThaw(tlist);
		    }

		    /* Update all menus and map right click menu */
		    EDVImbrUpdateMenus(imbr);
		    menu = (GtkMenu *)imbr->tlist_menu;
		    if(menu != NULL)
			gtk_menu_popup(
			    menu, NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
		}
		status = TRUE;
		break;

	      case 2:
		if(etype == GDK_BUTTON_PRESS)
		{
		    if(thumb_num > -1)
			EDVImbrTListDoFPromptRename(imbr, thumb_num);
		}
		status = TRUE;
		break;

	      case 1:
		/* Double click? */
		if(etype == GDK_2BUTTON_PRESS)
		{
		    if(thumb_num > -1)
		    {
			EDVImbrTListDoOpenObject(imbr, thumb_num, state);
			status = TRUE;
		    }
		}
		break;
	    }
	}
	/* Event occured on the image viewer? */
	else if(widget == (GtkWidget *)ImgViewGetViewWidget(imbr->imgview))
	{
	    switch(button->button)
	    {
	      case 3:
		if(etype == GDK_BUTTON_PRESS)
		{
#if 0
		    GtkMenu *menu;

		    /* Update all menus and map right click menu */
		    EDVImbrUpdateMenus(imbr);
		    menu = (GtkMenu *)imbr->tlist_menu;
		    if(menu != NULL)
			gtk_menu_popup(
			    menu, NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
#endif
		}
		status = TRUE;
		break;

	      case 1:
		/* Double click? */
		if(etype == GDK_2BUTTON_PRESS)
		{
		    gint thumb_num = imbr->tlist_selected_thumb;
		    if(thumb_num > -1)
		    {
			EDVImbrTListDoOpenObject(imbr, thumb_num, state);
			status = TRUE;
		    }
		}
		break;
	    }
	}

	return(status);
}

/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void EDVImbrHandleChildAttachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((handle_box == NULL) || (imbr == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *      GtkHandleBox "child_detached" signal callback.
 */
void EDVImbrHandleChildDetachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((handle_box == NULL) || (imbr == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}


/*
 *	Image Browser thumbs list select thumb callback.
 */
void EDVImbrTListSelectCB(
	gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
)
{
	tlist_thumb_struct *thumb;
	tlist_struct *tlist;
	edv_core_struct *core_ptr;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((tl == NULL) || (imbr == NULL))
	    return;

	if(imbr->processing)
	    return;

	tlist = imbr->tlist;
	core_ptr = EDV_CORE(imbr->core_ptr);
	if((tlist == NULL) || (core_ptr == NULL))
	    return;

	thumb = ((thumb_num >= 0) && (thumb_num < tlist->total_thumbs)) ?
	    tlist->thumb[thumb_num] : NULL;

	if(tl == (gpointer)tlist)
	{
	    /* Change in selection? */
	    if(imbr->tlist_selected_thumb != thumb_num)
	    {
		const gint total_selected = g_list_length(tlist->selection);
		edv_object_struct *obj = EDV_OBJECT(
		    TListGetThumbData(tlist, thumb_num)
		);

		EDVImbrSetBusy(imbr, TRUE);
		GUIBlockInput(imbr->toplevel, TRUE);
		imbr->processing = TRUE;

		/* Scroll if selected thumb is not visible */
		if(TListIsThumbVisible(tlist, thumb_num) !=
		    GTK_VISIBILITY_FULL
		)
		    TListMoveTo(tlist, thumb_num, 0.5f);

		/* Update current selected thumb on the tlist */
		imbr->tlist_selected_thumb = thumb_num;

		/* Match device index number from core structure's list
		 * of devices who's mount path matches the selected
		 * object's path
		 */
		if(obj != NULL)
		    EDVDeviceListMatchMountPath(
			core_ptr->device, core_ptr->total_devices,
			&imbr->selected_dev_num,
			obj->full_path
		    );

		/* Update DND icon for thumbs list */
		EDVImbrTListDNDSetIcon(imbr, thumb_num);

		/* Update title and status bar if selected object is
		 * valid
		 */
		if((obj != NULL) ? !STRISEMPTY(obj->name) : FALSE)
		{
		    gchar *buf, *size_str = NULL;
		    const gchar	*type_str = NULL,
				*full_path = obj->full_path;

		    /* If only one thumb is selected then display it
		     * on the Image Browser's ImgView
		     */
		    if((total_selected == 1) &&
		       (thumb != NULL)
		    )
		    {
			gboolean need_reload_thumb = FALSE;

			if(thumb->load_state != TLIST_LOAD_STATE_LOADED)
			{
			    need_reload_thumb = TRUE;
			}
			else if(!STRISEMPTY(full_path))
			{
			    struct stat stat_buf;
			    if(!stat(full_path, &stat_buf))
			    {
				if((gulong)stat_buf.st_mtime != obj->modify_time)
				{
				    obj->modify_time = (gulong)stat_buf.st_mtime;
				    need_reload_thumb = TRUE;
				}
				if((gulong)stat_buf.st_size != obj->size)
				{
				    obj->size = (gulong)stat_buf.st_size;
				    need_reload_thumb = TRUE;
				}
			    }
			}
			EDVImbrImgViewLoad(
			    imbr,
			    full_path,
			    need_reload_thumb ? thumb_num : -1
			);
		    }

		    /* Get object type string and size string */
		    switch(obj->type)
		    {
		      case EDV_OBJECT_TYPE_UNKNOWN:
			break;
		      case EDV_OBJECT_TYPE_FILE:
			type_str = "File";
			size_str = g_strdup_printf(
			    " (%s byte%s)",
			    EDVGetObjectSizeStr(
				EDV_CORE(imbr->core_ptr),
				obj->size
			    ),
			    (obj->size == 1) ? "" : "s"
			);
			break;
		      case EDV_OBJECT_TYPE_DIRECTORY:
			type_str = "Directory";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_LINK:
			type_str = "Link";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			type_str = "Block device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			type_str = "Character device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_FIFO:
			type_str = "FIFO Pipe";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_SOCKET:
			type_str = "Socket";
			size_str = NULL;
			break;
		    }

		    /* Set status bar message */
		    if(total_selected > 1)
			buf = g_strdup_printf(
			    "%i objects selected",
			    total_selected
			);
		    else if(!strcmp(obj->name, ".."))
			buf = g_strdup_printf(
			    "Parent directory selected"
			);
		    else
			buf = g_strdup_printf(
			    "%s \"%s\" selected%s",
			    type_str, obj->name,
			    (size_str != NULL) ? size_str : ""
			);
		    EDVStatusBarMessage(
			imbr->status_bar, buf, FALSE
		    );
		    g_free(buf);
		    g_free(size_str);
		}

		imbr->processing = FALSE;
		GUIBlockInput(imbr->toplevel, FALSE);
		EDVImbrSetBusy(imbr, FALSE);

		EDVImbrUpdateMenus(imbr);
	    }
	}
}

/*
 *	Image Browser thumbs list unselect thumb callback.
 */
void EDVImbrTListUnselectCB(
	gpointer tl, GdkEventButton *button, gint thumb_num, gpointer data
)
{
	edv_core_struct *core_ptr;
	tlist_struct *tlist;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((tl == NULL) || (imbr == NULL))
	    return;

	if(imbr->processing)
	    return;

	tlist = imbr->tlist;
	core_ptr = EDV_CORE(imbr->core_ptr);
	if((tlist == NULL) || (core_ptr == NULL))
	    return;

	if(tl == (gpointer)tlist)
	{
	    /* Get last selected thumb (if any) */
	    const gint total_selected = g_list_length(tlist->selection);
	    GList *glist = tlist->selection_end;
	    const gint thumb_num = (glist != NULL) ? (gint)glist->data : -1;
	    if(thumb_num > -1)
	    {
		/* Last selected thumb changed? */
		if(imbr->tlist_selected_thumb != thumb_num)
		{
		    const edv_object_struct *obj = EDV_OBJECT(
			TListGetThumbData(tlist, thumb_num)
		    );
		    if(obj != NULL)
		    {
			/* Update last selected thumb and device */
			imbr->tlist_selected_thumb = thumb_num;
			EDVDeviceListMatchMountPath(
			    core_ptr->device, core_ptr->total_devices,
			    &imbr->selected_dev_num,
			    obj->full_path
			);
		    }
		}

		if(total_selected == 1)
		{

/* TODO */

		}
	    }
	    else
	    {
		/* Mark that no thumb and device are selected */
		imbr->tlist_selected_thumb = -1;
		imbr->selected_dev_num = -1;
	    }

	    EDVImbrUpdateMenus(imbr);
	}
}

/*
 *	Image Browser Location Combo activate callback.
 */
void EDVImbrComboActivateCB(GtkWidget *widget, gpointer data)
{
	GtkCombo *combo = (GtkCombo *)widget;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((widget == NULL) || (imbr == NULL))
	    return;

	if(imbr->processing)
	    return;

	/* Check which widget was activated */
	if(widget == imbr->location_combo)
	{
	    gchar *new_path = EDVCopyEvaluateInputPath(
		NULL,           /* No parent path, imply use toplevel */
		gtk_entry_get_text(GTK_ENTRY(combo->entry))
	    );
	    if(new_path != NULL)
	    {
		EDVImbrSetBusy(imbr, TRUE);
		GUIBlockInput(imbr->toplevel, TRUE);

		/* Update location combo, this is really just to record
		 * the new value as history. The combo will be updated
		 * again (without recording history) if the new path is
		 * valid and selected in the callbacks triggered by
		 * EDVImbrSelectPath() farther below
		 */
		EDVImbrSetLocation(imbr, new_path, TRUE);

		/* Reset loading procedure values and get new directory
		 * listing, then initialize loading procedure values for
		 * the loading of images from the directory specified by
		 * new_path
		 */
		EDVImbrSelectPath(imbr, new_path);

		GUIBlockInput(imbr->toplevel, FALSE);
		EDVImbrSetBusy(imbr, FALSE);


		EDVImbrUpdateMenus(imbr);


		g_free(new_path);
	    }
	}
}


/*
 *	Loading procedure timeout callback.
 */
gint EDVImbrLoadingTOCB(gpointer data)
{
	gint status;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core_ptr;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if(imbr == NULL)
	    return(FALSE);

	if(gtk_events_pending() > 0)
	    return(TRUE);

	if(imbr->processing)
	    return(TRUE);

#define DO_FINAL_RETURN		{	\
					\
 /* Reset loading timeout id */		\
 imbr->loading_tocb = 0;		\
					\
 return(FALSE);				\
}

#define DO_RESCHEDUAL_RETURN	{	\
					\
 /* Set new loading timeout callback */	\
 imbr->loading_tocb = gtk_timeout_add(	\
  100,					\
  (GtkFunction)EDVImbrLoadingTOCB,	\
  imbr					\
 );					\
					\
 /* Return FALSE since we reschedualed	\
  * a timeout callback to this function	\
  */					\
 return(FALSE);				\
}

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	{
	    DO_FINAL_RETURN
	}

	cfg_list = core_ptr->cfg_list;

	/* Stop loading? */
	if(imbr->stop_count > 0)
	{
	    imbr->stop_count = 0;
	    imbr->loading_tocb = 0;

	    EDVStatusBarMessage(
		imbr->status_bar, "Loading interrupted", FALSE
	    );
	    EDVStatusBarProgress(imbr->status_bar, 0.0f, FALSE);

	    EDVImbrUpdateMenus(imbr);

	    return(FALSE);
	}

	/* Load the next thumb */
	status = EDVImbrTListLoadIterate(
	    imbr, TRUE,
	    EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_NO_ENLARGE)
	);
#if 0
g_print(
 "EDVImbrTListLoadIterate(): status=%i\n",
 status
);
#endif
	switch(status)
	{
	  case 0:	/* All thumb loaded */
	    imbr->loading_tocb = 0;
	    EDVImbrUpdateMenus(imbr);
	    return(FALSE);
	    break;

	  case 1:
	    /* Thumb loaded successfully, but there still may be more
	     * thumbs that need to be loaded
	     */
	    DO_RESCHEDUAL_RETURN
	    break;

	  case -3:	/* System error */
	    DO_FINAL_RETURN
	    break;

	  default:	/* General error */
	    /* It is safe to continue again after this error since
	     * this thumb that failed to load will not be attempted
	     * to be loaded again
	     */
	    DO_RESCHEDUAL_RETURN
	    break;
	}
#undef DO_FINAL_RETURN
#undef DO_RESCHEDUAL_RETURN
}



/*
 *	Called whenever the global write protect has changed.
 *
 *	The new state is given as state.
 */
void EDVImbrWriteProtectChangedCB(
	edv_imbr_struct *imbr, gboolean state
)
{
	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	EDVImbrUpdateMenus(imbr);
}

/*
 *	Called whenever a disk object has been added.
 */
void EDVImbrObjectAddedNotifyCB(
	edv_imbr_struct *imbr, const gchar *path,
	const struct stat *lstat_buf
)
{
	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	EDVImbrTListObjectAddedNotify(
	    imbr, path, lstat_buf
	);

	/* Need to update menus because a new loading process may be
	 * queued.
	 */
	EDVImbrUpdateMenus(imbr);
}

/*
 *      Called whenever a disk object has had its properties modified.
 */
void EDVImbrObjectModifiedNotifyCB(
	edv_imbr_struct *imbr, const gchar *path,
	const gchar *new_path,
	const struct stat *lstat_buf
)
{
	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	/* Check current location first, if the old path matches the
	 * current location then the current location needs to be updated
	 * to reflect the new path before notifying the contents clist.
	 */
	if(!STRISEMPTY(path))
	{
	    const gchar *cur_location = EDVImbrCurrentLocation(imbr);

	    if(new_path == NULL)
		new_path = path;

	    /* Check if the old path matches the current location, if it
	     * does then the current location needs to be updated to the
	     * new path
	     */
	    if((cur_location != NULL) ?
		!strcmp(cur_location, path) : FALSE
	    )
	    {
		/* Old path matches current location, change values
		 * reflecting the current location to the value of
		 * new_path
		 */
		EDVImbrSetTitle(imbr, new_path);
		EDVImbrSetLocation(imbr, new_path, FALSE);
		EDVImbrUpdateLocationIcon(imbr, new_path);
	    }
	}

	EDVImbrTListObjectModifiedNotify(
	    imbr, path, new_path, lstat_buf
	);

/*	EDVImbrUpdateMenus(imbr); */
}

/*
 *	Called whenever a disk object has been removed.
 */
void EDVImbrObjectRemovedNotifyCB(
	edv_imbr_struct *imbr, const gchar *path
)
{
	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	EDVImbrTListObjectRemovedNotify(imbr, path);

	/* Check if the path that was removed is the current location
	 * (after letting contents clist and directory ctree check on it
	 * first), if the path that was removed is the location then
	 * set the current location to be the parent of the path that
	 * was removed
	 */
	if(!STRISEMPTY(path))
	{
	    const gchar *cur_location = EDVImbrCurrentLocation(imbr);

	    /* Check if the removed object path matches the current
	     * location. If it matches then the current location needs to
	     * be updated to reflect that
	     */
	    if((cur_location != NULL) ?
		!strcmp(cur_location, path) : FALSE
	    )
	    {
		/* Removed object path matches the current location, so
		 * technically the current location needs to be cleared
		 * however we'll instead just change the current location
		 * value to reflect the value of the parent path of the
		 * removed object path
		 */
		gchar *parent_path = g_dirname(path);
		if(parent_path != NULL)
		{
		    EDVImbrSetTitle(imbr, parent_path);
		    EDVImbrSetLocation(imbr, parent_path, FALSE);
		    EDVImbrUpdateLocationIcon(imbr, parent_path);
		    g_free(parent_path);
		}
	    }
	}

/*	EDVImbrUpdateMenus(imbr); */
}


/*
 *      Called whenever a device has been mounted or unmounted.
 */
void EDVImbrMountNotifyCB(
	edv_imbr_struct *imbr,
	gint dev_num, edv_device_struct *dev_ptr,
	gboolean is_mounted
)
{
	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	EDVImbrTListMountNotify(imbr, dev_ptr, is_mounted);

	EDVImbrUpdateMenus(imbr);
}


/*
 *	Called whenever an object has been added to the recycle bin.
 */
void EDVImbrRecycledObjectAddedNotifyCB(
	edv_imbr_struct *imbr, guint index
)
{
	edv_core_struct *core_ptr;


	if(imbr == NULL)
	    return;

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return;


	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects.
	 */
	if(core_ptr->last_recbin_items != imbr->last_recbin_items)
	    EDVImbrUpdateMenus(imbr);
}

/*
 *	Called whenever an object has been removed from the recycle bin.
 */
void EDVImbrRecycledObjectRemovedNotifyCB(
	edv_imbr_struct *imbr, guint index
)
{
	edv_core_struct *core_ptr;


	if(imbr == NULL)
	    return;

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return;


	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects.
	 */
	if(core_ptr->last_recbin_items != imbr->last_recbin_items)
	    EDVImbrUpdateMenus(imbr);
}


/*
 *	Reconfigured notify callback.
 */
void EDVImbrReconfiguredNotifyCB(edv_imbr_struct *imbr)
{
	gchar *cur_path;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *status_bar;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;
	imgview_struct *iv;


	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;
	standard_rcstyle = core_ptr->standard_rcstyle;
	lists_rcstyle = core_ptr->lists_rcstyle;


	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfigure
	 */
	imbr->last_recbin_items = -1;
	imbr->last_write_protect_state = -1;


	/* Get copy of current location path */
	cur_path = STRDUP(EDVImbrCurrentLocation(imbr));


	/* Begin updating */

	/* Update Title */
	EDVImbrSetTitle(imbr, cur_path);

	/* Recreate Tool Bar */
	EDVImbrToolbarRegenerate(imbr);

	/* Show/hide Tool Bar */
	w = imbr->tool_bar_handle;
	if(w != NULL)
	{
	    imbr->tool_bar_map_state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_TOOL_BAR);
	    if(imbr->tool_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show/hide Location Bar */
	w = imbr->location_bar_handle;
	if(w != NULL)
	{
	    imbr->location_bar_map_state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_LOCATION_BAR);
	    if(imbr->location_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show/hide Mount Bar */
	w = imbr->mount_bar_handle;
	if(w != NULL)
	{
	    imbr->mount_bar_map_state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_MOUNT_BAR);
	    if(imbr->mount_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show/hide Find Bar */
	w = imbr->find_bar_handle;
	if(w != NULL)
	{
	    imbr->find_bar_map_state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_FIND_BAR);
	    if(imbr->find_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show/hide Status Bar */
	status_bar = imbr->status_bar;
	if(status_bar != NULL)
	{
	    imbr->status_bar_map_state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_STATUS_BAR);
	    if(imbr->status_bar_map_state)
		EDVStatusBarMap(status_bar);
	    else
		EDVStatusBarUnmap(status_bar);
	}

	/* Thumbs List */
	tlist = imbr->tlist;
	if(tlist != NULL)
	{
	    TListFreeze(tlist);

	    TListThumbGeometry(
		tlist,
		EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_WIDTH),
		EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_HEIGHT),
		EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_BORDER)
	    );
	    TListDoubleBuffer(
		tlist, EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER)
	    );
	    TListOrientation(
		tlist, EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_LIST_HORIZONTAL)
	    );
	    TListShowThumbFrames(
		tlist, EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_SHOW_FRAMES)
	    );
	    TListShowThumbLabels(
		tlist, EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_SHOW_LABELS)
	    );
	    TListShowTextTips(
		tlist, EDV_GET_B(EDV_CFG_PARM_SHOW_TEXTTIPS)
	    );

	    /* Update geometry of thumb size */
	    TListResize(tlist, -1, -1);

	    /* Update all thumbs */
	    EDVImbrTListResetThumbs(imbr);

	    TListThaw(tlist);
	}

	/* Image Viewer */
	iv = imbr->imgview;
	if(iv != NULL)
	{
	    const cfg_color_struct *color;

	    /* Image quality */
	    switch(EDV_GET_I(EDV_CFG_PARM_IMAGE_QUALITY))
	    {
	      case 0:
		iv->quality = 0;
		break;
	      case 1:
		iv->quality = 1;
		break;
	      case 2:
		iv->quality = 2;
		break;
	      default:
		iv->quality = 1;
		break;
	    }

	    /* Background color */
	    color = EDV_GET_COLOR(EDV_CFG_PARM_IMBR_COLOR_BG);
	    if(color != NULL)
	    {
		GdkColor gdk_color[5];
		gint i, m = sizeof(gdk_color) / sizeof(GdkColor);
		const cfg_color_struct *sc = color;

		for(i = 0; i < m; i++)
		{
		    GDK_COLOR_SET_COEFF(
			&gdk_color[i],
			sc->r, sc->g, sc->b
		    )
		}
		ImgViewSetViewBG(iv, gdk_color);
	    }

	    /* Redraw image viewer and update menus */
	    ImgViewDraw(iv);
	    ImgViewUpdateMenus(iv);
	}

	/* Recreate New Object Submenu */
	EDVImbrNewObjectMenuRegenerate(imbr);

	/* Update RC styles */
	w = imbr->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);
	tlist = imbr->tlist;
	w = (tlist != NULL) ? tlist->list_da : NULL;
	if((w != NULL) && (lists_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = ImgViewGetToplevelWidget(imbr->imgview);
	if((w != NULL) && (lists_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = (GtkWidget *)ImgViewGetMenuWidget(imbr->imgview);
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(
		w, standard_rcstyle
	    );
	w = imbr->tlist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);


	/* Update menus */
	EDVImbrUpdateMenus(imbr);


	/* Notify Image Browser's toplevel widget to resize */
	w = imbr->toplevel;
	if(w != NULL)
	    gtk_widget_queue_resize(w);


	/* Delete copy of current location path */
	g_free(cur_path);
}


/*
 *      Called whenever a MIME Type has been added to the core
 *      structure's list of MIME Types.
 */
void EDVImbrMimeTypeAddedCB(
	edv_imbr_struct *imbr,
	gint mt_num, edv_mimetype_struct *mt_ptr
)
{
	/* Treat a MIME Type added the same as it would be for a MIME Type
	 * modified, forward signal to the MIME Type modified callback.
	 */
	EDVImbrMimeTypeModifiedCB(imbr, mt_num, mt_ptr);
}

/*
 *      Called whenever a MIME Type has been modified on the core
 *      structure's list of MIME Types.
 */
void EDVImbrMimeTypeModifiedCB(
	edv_imbr_struct *imbr,
	gint mt_num, edv_mimetype_struct *mt_ptr
)
{
	gchar *cur_path;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;


	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;


	/* Get copy of current location path */
	cur_path = STRDUP(EDVImbrCurrentLocation(imbr));


	/* Reset displayed thumb values */
	TListFreeze(tlist);
	EDVImbrTListResetThumbs(imbr);
	TListThaw(tlist);

	/* Delete copy of current location path */
	g_free(cur_path);
}

/*
 *      Called whenever a MIME Type has been removed from the core
 *      structure's list of MIME Types.
 */
void EDVImbrMimeTypeRemovedCB(
	edv_imbr_struct *imbr, gint mt_num
)
{
	gchar *cur_path;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;


	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;


	/* Get copy of current location path */
	cur_path = STRDUP(EDVImbrCurrentLocation(imbr));



	/* Reset displayed thumb values */
	TListFreeze(tlist);
	EDVImbrTListResetThumbs(imbr);
	TListThaw(tlist);

	/* Delete copy of current location path */
	g_free(cur_path);
}


/*
 *      Called whenever a device has been added to the core
 *      structure's list of devices.
 */
void EDVImbrDeviceAddedCB(
	edv_imbr_struct *imbr,
	gint dev_num, edv_device_struct *dev_ptr
)
{
	/* Treat a device added the same as it would be for a device
	 * modified, forward signal to the device modified callback.
	 */
	EDVImbrDeviceModifiedCB(imbr, dev_num, dev_ptr);
}

/*
 *	Called whenever a device has been modified on the core
 *	structure's list of devices.
 */
void EDVImbrDeviceModifiedCB(
	edv_imbr_struct *imbr,
	gint dev_num, edv_device_struct *dev_ptr
)
{
	gchar *cur_path;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;


	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;


	/* Get copy of current location path */
	cur_path = STRDUP(EDVImbrCurrentLocation(imbr));



	/* Reset displayed thumb values */
	TListFreeze(tlist);
	EDVImbrTListResetThumbs(imbr);
	TListThaw(tlist);

	/* Update menus */
	EDVImbrUpdateMenus(imbr);

	/* Delete copy of current location path */
	g_free(cur_path);
}

/*
 *      Called whenever a device has been removed from the core
 *      structure's list of devices.
 */
void EDVImbrDeviceRemovedCB(
	edv_imbr_struct *imbr, gint dev_num
)
{
	gchar *cur_path;
	edv_mountbar_struct *mb;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;


	if(imbr == NULL)
	    return;

	if(imbr->processing)
	    return;

	core_ptr = EDV_CORE(imbr->core_ptr);
	if(core_ptr == NULL)
	    return;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;


	/* Get copy of current location path */
	cur_path = STRDUP(EDVImbrCurrentLocation(imbr));


	/* Check if mount bar is currently referencing this device, if it
	 * is then its selected device needs to be set to -1.
	 */
	mb = imbr->mountbar;
	if(mb != NULL)
	{
	    if(mb->selected_dev_num == dev_num)
		mb->selected_dev_num = -1;
	    /* Mount bar will be updated further below when menus are
	     * updated.
	     */
	}


	/* Reset displayed thumb values */
	TListFreeze(tlist);
	EDVImbrTListResetThumbs(imbr);
	TListThaw(tlist);

	/* Update menus */
	EDVImbrUpdateMenus(imbr);

	/* Delete copy of current location path */
	g_free(cur_path);
}
