#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <glib.h>

#include "../../include/fio.h"
#include "../../include/disk.h"
#include "../../include/prochandle.h"

#include "../edvtypes.h"
#include "../edvdevices.h"

#include "ziptoolio.h"
#include "config.h"


#define PROG_ZIPTOOL	"/usr/bin/ziptool"


gint ZipToolsMount(edv_device_struct *dev_ptr);
gint ZipToolsUnmount(edv_device_struct *dev_ptr);

gint ZipToolsDeviceIsProtected(edv_device_struct *dev_ptr);
gint ZipToolsLock(
        edv_device_struct *dev_ptr, const gchar *password
);
gint ZipToolsUnlock(
        edv_device_struct *dev_ptr, const gchar *password
);

gint ZipToolsSpinDown(edv_device_struct *dev_ptr);
gint ZipToolsEject(edv_device_struct *dev_ptr);



/*
 *	Mounts the device.
 *
 *      Returns:
 *
 *      0       Success (or device is already mounted).
 *      -1      General error
 *      -2      Bad or ambiguous value
 *      -3      Systems error or unable to execute command
 */
gint ZipToolsMount(edv_device_struct *dev_ptr)
{
        pid_t p;
        gchar *cmd;

        if(dev_ptr == NULL)
            return(-2);

        if(dev_ptr->device_path == NULL)
            return(-2);
        if(dev_ptr->mount_path == NULL)
            return(-2);

        /* Format and execute eject command. */
        cmd = g_strdup_printf(
            "%s -m %s %s",
            PROG_ZIPTOOL,
            dev_ptr->device_path, dev_ptr->mount_path
        );
        p = ExecB(cmd);

        /* Deallocate locals. */
        g_free(cmd);

        return((p == 0) ? -3 : 0);
}

/*
 *	Unmounts the device.
 *
 *      Returns:
 *
 *      0       Success (or device is already unmounted).
 *      -1      General error
 *      -2      Bad or ambiguous value
 *      -3      Systems error or unable to execute command
 */
gint ZipToolsUnmount(edv_device_struct *dev_ptr)
{
        pid_t p;
        gchar *cmd;

        if(dev_ptr == NULL)
            return(-2);

        if(dev_ptr->device_path == NULL)
            return(-2);

        /* Format and execute eject command. */
        cmd = g_strdup_printf(
            "%s -u %s",
            PROG_ZIPTOOL,
            dev_ptr->device_path
        );
        p = ExecB(cmd);

        /* Deallocate locals. */
        g_free(cmd);

        return((p == 0) ? -3 : 0);
}

/*
 *	Checks if the device is unprotected, write protected, or
 *	write protected with password.
 *
 *	Returns:
 *
 *	0	Unprotected (read/write)
 *	1	Protected (read only)
 *	2	Protected with password
 */
gint ZipToolsDeviceIsProtected(edv_device_struct *dev_ptr)
{
	gint status = 0;
	pid_t p;
	gchar *cmd, *stdout_path;


	if(dev_ptr == NULL)
	    return(status);

	if(dev_ptr->device_path == NULL)
	    return(status);

	/* Generate tempory file path for output. */
	stdout_path = tempnam(NULL, NULL);

	/* Format and execute device status command. */
	cmd = g_strdup_printf(
	    "%s -s %s",
	    PROG_ZIPTOOL,
	    dev_ptr->device_path
	);
	p = ExecBO(cmd, stdout_path);
	if(p == 0)
	{
	    /* Error. */
	    unlink(stdout_path);
	    g_free(stdout_path);
	    g_free(cmd);
	    return(status);
	}
	else
	{
	    /* Success, read first line of output file. */
	    FILE *stdout_fp = FOpen(stdout_path, "rb");
	    gchar *buf = FGetStringLiteral(stdout_fp);
	    FClose(stdout_fp);

	    /* Got first line of output file? */
	    if(buf != NULL)
	    {
		if(strstr(
		    buf,
		    "password write-protected"
		) != NULL)
		    status = 2;
		else if(strstr(
                    buf,
                    "write-protected"
                ) != NULL)
		    status = 1;
		else if(strstr(
		    buf,
		    "not protected"
		) != NULL)
		    status = 0;

		/* All else assume not protected. */

		g_free(buf);
	    }
	}

	/* Delete output file and deallocate locals. */
	unlink(stdout_path);
	g_free(stdout_path);
	g_free(cmd);

	return(status);
}


/*
 *	Locks the device (must be currently unlocked). If password
 *	is not NULL then the device will be locked with password.
 *
 *	Returns:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad or ambiguous value
 *	-3	Systems error or unable to execute command
 *	-4	Unable to lock
 *	-5	Device is already locked (with or without password),
 *		must unlock before locking again
 */
gint ZipToolsLock(
        edv_device_struct *dev_ptr, const gchar *password
)
{
        if(dev_ptr == NULL)
            return(-2);

        if(dev_ptr->device_path == NULL)
            return(-2);

        /* Device is already locked? */
        if(ZipToolsDeviceIsProtected(dev_ptr))
            return(-5);

	/* Lock with password? */
	if(password != NULL)
	{
	    /* Format and execute lock device with password command. */
	    gchar *cmd = g_strdup_printf(
		"%s -rp %s",
		PROG_ZIPTOOL,
		dev_ptr->device_path
	    );
	    FILE *fp = popen(cmd, "w");
	    if(fp == NULL)
	    {
		/* Unable to execute command. */
		g_free(cmd);
		return(-3);
	    }

	    /* Send password. */
	    fprintf(fp, "%s\n", password);

	    /* Close process output stream and deallocate locals. */
	    pclose(fp);
	    g_free(cmd);
	}
	else
	{
            /* Format and execute lock device command. */
            gchar *cmd = g_strdup_printf(
                "%s -ro %s",
                PROG_ZIPTOOL,
                dev_ptr->device_path
            );
            FILE *fp = popen(cmd, "w");
            if(fp == NULL)
            {
                /* Unable to execute command. */
                g_free(cmd);
                return(-3);
            }

	    /* Do not send any password. */

            /* Close process output stream and deallocate locals. */
            pclose(fp);
            g_free(cmd);
	}

        /* Device was not successfully locked? */
/* This does not work quite well, because the device is ejected after
 * locking and there is no way to see if it was locked correctly.
 */
/*
        if(ZipToolsDeviceIsProtected(dev_ptr))
            return(0);
        else
            return(-4);
 */
	return(0);
}


/*
 *	Unlocks the device (if it is already locked) and uses the
 *	given password.
 *
 *	Returns:
 *
 *	0	Success (or device is already unlocked)
 *	-1	General error
 *	-2	Bad or ambiguous value
 *	-3	Systems error or unable to execute command
 *	-4	Bad password or unable to unlock
 */
gint ZipToolsUnlock(
        edv_device_struct *dev_ptr, const gchar *password
)
{
	gchar *cmd;
	FILE *fp;

	if(dev_ptr == NULL)
	    return(-2);

        if(dev_ptr->device_path == NULL)
            return(-2);

	/* Device is not locked? */
	if(!ZipToolsDeviceIsProtected(dev_ptr))
	    return(0);

	if(password == NULL)
	    password = "";


        /* Format and execute unlock device command. */
        cmd = g_strdup_printf(
            "%s -rw %s",
            PROG_ZIPTOOL,
            dev_ptr->device_path
        );
        fp = popen(cmd, "w");
	if(fp == NULL)
	{
	    /* Unable to execute command. */
	    g_free(cmd);
	    return(-3);
	}

	/* Send password. */
	fprintf(fp, "%s\n", password);

	/* Close process output stream and deallocate locals. */
	pclose(fp);
	g_free(cmd);

        /* Device was not successfully unlocked? */
        if(ZipToolsDeviceIsProtected(dev_ptr))
            return(-4);
	else
	    return(0);
}

/*
 *	Spins down the device
 *
 *	Returns:
 *
 *	0	Success (or device already spinned down)
 *	-1	General error
 *	-2	Bad or ambiguous value
 *	-3	Systems error or unable to execute command
 */
gint ZipToolsSpinDown(edv_device_struct *dev_ptr)
{
        pid_t p;
        gchar *cmd;

        if(dev_ptr == NULL)
            return(-2);

        if(dev_ptr->device_path == NULL)
            return(-2);

        /* Format and execute spin down command. */
        cmd = g_strdup_printf(
            "%s -p %s",
            PROG_ZIPTOOL,
            dev_ptr->device_path
        );
        p = ExecB(cmd);

        /* Deallocate locals. */
        g_free(cmd);

        return((p == 0) ? -3 : 0);
}

/*
 *	Ejects the media from the device
 *
 *      Returns:
 *
 *      0       Success (or media is already ejected)
 *      -1      General error
 *      -2      Bad or ambiguous value
 *      -3      Systems error or unable to execute command
 *      -4      Device is currently mounted
 */
gint ZipToolsEject(edv_device_struct *dev_ptr)
{
	pid_t p;
        gchar *cmd;

        if(dev_ptr == NULL)
            return(-2);

        if(dev_ptr->device_path == NULL)
            return(-2);

        /* Format and execute eject command. */
        cmd = g_strdup_printf(
            "%s -e %s",
            PROG_ZIPTOOL,
            dev_ptr->device_path
        );
        p = ExecB(cmd);

        /* Deallocate locals. */
	g_free(cmd);

	return((p == 0) ? -3 : 0);
}

