#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "rundlg.h"
#include "endeavour.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/icon_executable_20x20.xpm"
#include "images/icon_executable_48x48.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_browse_20x20.xpm"
#include "images/icon_clear_20x20.xpm"


static gint EDVRunDlgDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVRunDlgRunCB(
        GtkWidget *widget, gpointer data
);
static void EDVRunDlgBrowseCB(
        GtkWidget *widget, gpointer data
);
static void EDVRunDlgClearCB(
        GtkWidget *widget, gpointer data
);
static void EDVRunDlgGetWorkingDirFromCmdCB(
        GtkWidget *widget, gpointer data
);
static void EDVRunDlgCloseCB(GtkWidget *widget, gpointer data);

void EDVRunDlgFetchValues(edv_run_dlg_struct *dlg);

edv_run_dlg_struct *EDVRunDlgNew(gpointer core_ptr);
void EDVRunDlgMap(edv_run_dlg_struct *dlg);
void EDVRunDlgUnmap(edv_run_dlg_struct *dlg);
void EDVRunDlgDelete(edv_run_dlg_struct *dlg);


#define RUNDLG_WIDTH	500
#define RUNDLG_HEIGHT	-1

#define RUNDLG_BTN_WIDTH	(100 + (2 * 3))
#define RUNDLG_BTN_HEIGHT	(30 + (2 * 3))

#define RUNDLG_ARROW_WIDTH	20
#define RUNDLG_ARROW_HEIGHT	20

#define RUNDLG_TITLE		"Run"



/*
 *	Run dialog "delete_event" signal callback.
 */
static gint EDVRunDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_run_dlg_struct *dlg = (edv_run_dlg_struct *)data;
	if(dlg == NULL)
	    return(TRUE);

	EDVRunDlgCloseCB(NULL, dlg);

	return(TRUE);
}

/*
 *      Run dialog run callback.
 */
static void EDVRunDlgRunCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
	GtkToggleButton *tb;
	GtkEntry *entry;
	GtkCombo *combo;
        edv_core_struct *core_ptr;
	const gchar *cmd_src, *working_dir;
	gchar orig_working_dir[PATH_MAX];
        edv_run_dlg_struct *dlg = (edv_run_dlg_struct *)data;
        if(dlg == NULL)
            return;

        core_ptr = (edv_core_struct *)dlg->core_ptr;
        if(core_ptr == NULL)
            return;

	combo = (GtkCombo *)dlg->run_combo;
	if(combo == NULL)
	    return;

	entry = (GtkEntry *)dlg->working_dir_entry;
	if(entry == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


	/* Get working directory and original working directory. */
	working_dir = gtk_entry_get_text(entry);
	if(getcwd(orig_working_dir, PATH_MAX) != NULL)
	    orig_working_dir[PATH_MAX - 1] = '\0';
	else
	    strcpy(orig_working_dir, "/");


	/* Get user typed command from run combo. */
	cmd_src = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if(cmd_src != NULL)
	{
	    gchar *cmd = NULL;

	    /* Record user typed command. */
	    GUIComboAddItem(combo, cmd_src);

	    /* Begin formatting target command string, first check if we
	     * need to prepend terminal command?
	     */
	    tb = (GtkToggleButton *)dlg->run_in_terminal_check;
	    if((tb != NULL) ? tb->active : FALSE)
	    {
		const gchar *cmd_terminal_run = EDVCFGItemListGetValueS(
		    core_ptr->cfg_list, EDV_CFG_PARM_PROG_TERMINAL_RUN
		);
		if(cmd_terminal_run != NULL)
		    cmd = g_strdup_printf(
			"%s \"%s\" &",
			cmd_terminal_run,
			cmd_src
		    );
	    }
	    else
	    {
		cmd = g_strdup_printf(
		    "%s &",
		    cmd_src
		);
	    }

	    /* Generated command string successfully? */
	    if(cmd != NULL)
	    {
		gint status;
		gchar *buf;

		/* Change working directory before executing command. */
		if((working_dir != NULL) ? (*working_dir != '\0') : FALSE)
		    chdir(working_dir);

		/* Execute command. */
		status = system(cmd);

		/* Restore original working directory. */
		if(*orig_working_dir != '\0')
		    chdir(orig_working_dir);

		/* Check if there was an error executing the command. */
		switch(status)
		{
		  case 127:
		  case -1:
		    buf = g_strdup_printf(
			"Failed to run:\n\n    %s",
			cmd_src
		    );
		    CDialogSetTransientFor(dlg->toplevel);
		    CDialogGetResponse(
			"Run Failed",
			buf,
"Please verify that the program you are\n\
trying to run exists and that the arguments\n\
(if any) are correct.",
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		    g_free(buf);
		    break;
		}

		/* Deallocate command string. */
		g_free(cmd);
	    }
	}


	/* Check keep dialog check state before closing. */
	tb = (GtkToggleButton *)dlg->keep_dialog_check;
	if((tb != NULL) ? !tb->active : TRUE)
	    EDVRunDlgCloseCB(NULL, dlg);

	reenterent = FALSE;
}


/*
 *	Run dialog browse button callback.
 *
 *	The data must be a GtkEntry.
 */
static void EDVRunDlgBrowseCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        gbool status;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
	GtkWidget *toplevel, *w;
	GtkEntry *entry;

	w = (GtkWidget *)data;
        if((w != NULL) ? GTK_IS_ENTRY(w) : FALSE)
	    entry = GTK_ENTRY(w);
	else
            return;

	if(FileBrowserIsQuery())
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	toplevel = gtk_widget_get_toplevel(w);

	/* Create file types list. */
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for program to run. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
            "Select Program",
            "Select", "Cancel",
            NULL,               /* Use last path. */
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
        {
            const gchar *new_path = (total_path_rtns > 0) ?
                path_rtn[0] : NULL;
	    if(new_path != NULL)
	    {
		gtk_entry_set_text(entry, new_path);
		gtk_entry_set_position(entry, -1);
	    }
	}	/* Got user response? */

	/* Deallocate file types list. */
	FileBrowserDeleteTypeList(ftype, total_ftypes);
	ftype = NULL;
	total_ftypes = 0;

        reenterent = FALSE;
}

/*
 *	Run dialog clear callback.
 *
 *      The data must be a GtkEntry.
 */
static void EDVRunDlgClearCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
	GtkWidget *w;
	GtkEntry *entry;

        w = (GtkWidget *)data;
        if((w != NULL) ? GTK_IS_ENTRY(w) : FALSE)
            entry = GTK_ENTRY(w);
        else
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	gtk_entry_set_text(entry, "");
	gtk_entry_set_position(entry, 0);

        reenterent = FALSE;
}

/*
 *	Run dialog get working directory from current run command value
 *	callback.
 */
static void EDVRunDlgGetWorkingDirFromCmdCB(
        GtkWidget *widget, gpointer data
)
{
	const gchar *cmd;
        GtkWidget *w;
	GtkEntry *entry;
	GtkCombo *combo;
        edv_run_dlg_struct *dlg = (edv_run_dlg_struct *)data;
        if(dlg == NULL)
            return;

	w = dlg->run_combo;
	if(w != NULL)
	    combo = GTK_COMBO(w);
	else
	    return;

        w = dlg->working_dir_entry;
        if(w != NULL)
            entry = GTK_ENTRY(w);
        else
            return;

	cmd = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if(cmd != NULL)
	{
	    while(ISBLANK(*cmd))
		cmd++;

	    /* Run command starts with an absolute path? */
	    if(*cmd == DIR_DELIMINATOR)
	    {
		gchar	*strptr,
			*new_working_dir = g_strdup(cmd);

		/* Seek strptr to first space (if any) in
		 * new_working_dir.
		 */
		strptr = new_working_dir;
		while(!ISBLANK(*strptr) && (*strptr != '\0'))
		    strptr++;
		/* Cap new_working_dir at first space. */
		*strptr = '\0';

		gtk_entry_set_text(
		    entry, GetParentDir(new_working_dir)
		);
		gtk_entry_set_position(entry, -1);

		g_free(new_working_dir);
	    }
	}
}


/*
 *	Run dialog close button callback.
 */
static void EDVRunDlgCloseCB(
	GtkWidget *widget, gpointer data
)
{
	GtkWidget *w;
	edv_core_struct *core_ptr;
        edv_run_dlg_struct *dlg = (edv_run_dlg_struct *)data;
        if(dlg == NULL)
            return;

        core_ptr = (edv_core_struct *)dlg->core_ptr;

	/* Begin recording current widget values to global
	 * configuration.
	 */

	/* Run history. */
        if(core_ptr != NULL)
	{
	    GList *glist = (GList *)GUIComboGetList(dlg->run_combo);
	    const gchar *path = EDVCFGItemListGetValueS(
		core_ptr->cfg_list, EDV_CFG_PARM_FILE_RUNDLG_HISTORY
	    );
	    if((path != NULL) && (glist != NULL))
	    {
		const gchar *line_ptr;
		FILE *fp = FOpen(path, "wb");
		if(fp != NULL)
		{
		    while(glist != NULL)
		    {
			line_ptr = (const gchar *)glist->data;
			if((line_ptr != NULL) ? (*line_ptr != '\0') : FALSE)
			{
			    fputs(line_ptr, fp);
			    fputc('\n', fp);
			}

			glist = glist->next;
		    }

		    FClose(fp);
		    fp = NULL;
		}
	    }
	}

	/* Toplevel geometry. */
	w = dlg->toplevel;
	if((core_ptr != NULL) && (w != NULL))
	{
	    gint x = 0, y = 0;
	    GdkWindow *window = w->window;
            if(window != NULL)
                gdk_window_get_root_origin(window, &x, &y);

	    EDVCFGItemListSetValueI(
		core_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_X,
		x, FALSE
	    );
	    EDVCFGItemListSetValueI(
		core_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_Y,
		y, FALSE
	    );

	    gtk_widget_set_uposition(w, x, y);
	}

	/* Last working directory. */
        w = dlg->working_dir_entry;
        if((core_ptr != NULL) && (w != NULL))
            EDVCFGItemListSetValueS(
                core_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_LAST_WORKING_DIR,
                gtk_entry_get_text(GTK_ENTRY(w)), FALSE
            );

	/* Run in terminal. */
        w = dlg->run_in_terminal_check;
	if((core_ptr != NULL) && (w != NULL))
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_RUN_IN_TERMINAL,
                GTK_TOGGLE_BUTTON(w)->active, FALSE
            );

	/* Keep dialog. */
        w = dlg->keep_dialog_check;
        if((core_ptr != NULL) && (w != NULL))
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_KEEP_DIALOG,
                GTK_TOGGLE_BUTTON(w)->active, FALSE
            );


	/* Now close dialog. */
	EDVRunDlgUnmap(dlg);
}

/*
 *	Regets list of past runned objects and updates the widgets to
 *	match the current configuration.
 */
void EDVRunDlgFetchValues(edv_run_dlg_struct *dlg)
{
	GtkToggleButton *tb;
	GtkCombo *combo;
	edv_core_struct *core_ptr;


	if(dlg == NULL)
	    return;

	core_ptr = (edv_core_struct *)dlg->core_ptr;
	if(core_ptr == NULL)
	    return;

	/* Begin fetching values from global configuration and updating
	 * the run dialog's widgets.
	 */

	combo = (GtkCombo *)dlg->run_combo;
	if(combo != NULL)
	{
	    FILE *fp;
	    GList *glist = NULL;
	    const gchar *path = EDVCFGItemListGetValueS(
		core_ptr->cfg_list, EDV_CFG_PARM_FILE_RUNDLG_HISTORY
            );

	    /* Open run history file and load each line to the glist
	     * as a list of strings.
	     */
	    fp = FOpen(path, "rb");
	    if(fp != NULL)
	    {
		gchar *buf;

		while(1)
		{
		    buf = FGetStringLiteral(fp);
		    if(buf == NULL)
			break;

		    glist = g_list_append(glist, g_strdup(buf));
		    g_free(buf);
		}

		FClose(fp);
		fp = NULL;
	    }

	    /* Check if we got a glist of strings from the run
	     * history file.
	     */
	    if(glist != NULL)
	    {
		/* Set the new glist to the run combo's history, this
		 * glist will be managed by the call and should not
		 * be referenced afterwards.
		 */
		GUIComboSetList(combo, glist);
		glist = NULL;
	    }
	}

	tb = (GtkToggleButton *)dlg->run_in_terminal_check;
	if(tb != NULL)
	{
	    tb->active = EDVCFGItemListGetValueI(
		core_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_RUN_IN_TERMINAL
	    );
	}

        tb = (GtkToggleButton *)dlg->keep_dialog_check;
        if(tb != NULL)
        {
            tb->active = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_KEEP_DIALOG
            );
        }
}


/*
 *	Creates a new run dialog.
 */
edv_run_dlg_struct *EDVRunDlgNew(gpointer core_ptr)
{
        gint border_minor = 2, border_major = 5;
	gint x = 0, y = 0;
        GdkWindow *window;
        GtkAccelGroup *accelgrp;
	GtkStyle *style;
	gpointer combo_rtn;
	GtkWidget *w, *parent, *parent2;
	GtkCombo *combo;
	edv_core_struct *c_ptr = (edv_core_struct *)core_ptr;
        edv_run_dlg_struct *dlg = (edv_run_dlg_struct *)g_malloc0(
            sizeof(edv_run_dlg_struct)
        );
        if(dlg == NULL)
            return(dlg);

	if(c_ptr != NULL)
	{
            x = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_X
            );
            y = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_Y
            );
	}

        /* Reset values. */
        dlg->initialized = TRUE;
        dlg->map_state = FALSE;
	dlg->busy_count = 0;
        dlg->core_ptr = core_ptr;


        /* Keyboard accelerator group. */
        dlg->accelgrp = accelgrp = gtk_accel_group_new();


        /* Toplevel. */
        dlg->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
        gtk_widget_set_usize(w, RUNDLG_WIDTH, RUNDLG_HEIGHT);
	gtk_widget_set_uposition(w, x, y);
        gtk_widget_realize(w);
        gtk_window_set_title(GTK_WINDOW(w), RUNDLG_TITLE);
	style = gtk_widget_get_style(w);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
            GUISetWMIcon(window, (u_int8_t **)icon_executable_48x48_xpm);
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(EDVRunDlgDeleteEventCB), dlg
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;


        /* Main vbox. */
        dlg->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


	/* Hbox for run combo, browse button, and clear button. */
	w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Run:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Corra:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Courir:",
#endif
            "",
            NULL,	/* Set glist later. */
            10,		/* Maximum items. */
            &combo_rtn,
            dlg,
            EDVRunDlgRunCB,
            NULL
        );
        dlg->run_combo = (GtkWidget *)combo_rtn;
	combo = (GtkCombo *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);

	if(combo != NULL)
	{
	    w = combo->entry;
	    EDVEntrySetDND((edv_core_struct *)core_ptr, w);
	    GUISetWidgetTip(
		w,
#ifdef PROG_LANGUAGE_ENGLISH
		"Enter the program's name and any arguments here"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Entre el nombre del programa y cualquier argumento aqu"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Entre le programme et n'importe quels arguments ici"
#endif
	    );
	}

	/* Vbox to align buttons. */
	w = gtk_vbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

	/* Browse button. */
        dlg->browse_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_browse_20x20_xpm
        );
        gtk_widget_set_usize(w, 20 + 2, 20 + 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	combo = (GtkCombo *)dlg->run_combo;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVRunDlgBrowseCB), combo->entry
        );
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Browse"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Hojee"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Brouter"
#endif
	);
        gtk_widget_show(w);

        /* Clear button. */
        dlg->clear_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_clear_20x20_xpm
        );
        gtk_widget_set_usize(w, 20 + 2, 20 + 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVRunDlgClearCB), combo->entry
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Clear"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Claro"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Clair"
#endif
        );
        gtk_widget_show(w);


        /* Hbox for working directory entry and get from command
	 * button.
	 */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

	/* Working directory entry. */
	w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Working Directory:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Gua Trabajo:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Annuaire Travaillant:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	dlg->working_dir_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_entry_set_text(
            GTK_ENTRY(w),
            EDVCFGItemListGetValueS(
                c_ptr->cfg_list, EDV_CFG_PARM_RUNDLG_LAST_WORKING_DIR
            )
        );
        EDVEntrySetDND((edv_core_struct *)core_ptr, w);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the working directory that will be used when running the program"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre la gua de trabajo que se usar cundo correr el programa"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer l'annuaire qui travaillant qui sera utilis en courant le programme"
#endif
        );
        gtk_widget_show(w);


        /* Vbox to align buttons. */
        w = gtk_vbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

        /* Browse button. */
        dlg->working_dir_browse_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_browse_20x20_xpm
        );
        gtk_widget_set_usize(w, 20 + 2, 20 + 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        combo = (GtkCombo *)dlg->run_combo;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVRunDlgBrowseCB), dlg->working_dir_entry
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Browse"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Hojee"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Brouter"
#endif
        );
        gtk_widget_show(w);

        /* Clear button. */
        dlg->working_dir_clear_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_clear_20x20_xpm
        );
        gtk_widget_set_usize(w, 20 + 2, 20 + 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVRunDlgClearCB), dlg->working_dir_entry
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Clear"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Claro"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Clair"
#endif
        );
        gtk_widget_show(w);

        /* Get from command button. */
        dlg->working_dir_get_from_cmd_btn = w = gtk_button_new_with_label(
	    "./"
	);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVRunDlgGetWorkingDirFromCmdCB), dlg
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Get working directory from the current command value"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Obtenga trabajando gua del valor actual de la orden"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Recevoir l'annuaire qui travaillant de la valeur d'ordre actuelle"
#endif
        );
        gtk_widget_show(w);


	/* Hbox for run in terminal and keep dialog checks. */
	w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

	dlg->run_in_terminal_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Run In Terminal"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Corra En la Terminal"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Courir Dans le Terminal"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Run the specified program in a terminal, this is often\
 needed for text-based programs"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Corra el programa especificado en una terminal, esto a\
 menudo se necesita para programas basados de texto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Courir le programme spcifi dans un terminal, ceci\
 est souvent eu besoin de pour les programmes texte-bass"
#endif
        );
        gtk_widget_show(w);

        dlg->keep_dialog_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Keep Dialog"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Mantenga El Dilogo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Garder Le Dialogue"
#endif
        );
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Keep this dialog mapped after running each program"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Mantenga este dilogo trazado despus de correr un programa"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Garder ce mapped de dialogue aprs avoir excutant un programme"
#endif
        );
        gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for buttons. */
	w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

        /* Run button. */
        dlg->run_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_executable_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Run",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Corra",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Courir",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, RUNDLG_BTN_WIDTH, RUNDLG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVRunDlgRunCB), dlg
        );
        gtk_accel_group_add(
            accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);

        /* Cancel button. */
        dlg->cancel_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Cancel",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cancele",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Annuler",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, RUNDLG_BTN_WIDTH, RUNDLG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVRunDlgCloseCB), dlg
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);



	return(dlg);
}

/*
 *	Maps the run dialog.
 */
void EDVRunDlgMap(edv_run_dlg_struct *dlg)
{
        GtkWidget *w;

        if(dlg == NULL)
            return;

	/* Set run button as the default on map. */
        w = dlg->run_btn;
        if(w != NULL)
        {
            gtk_widget_grab_focus(w);
            gtk_widget_grab_default(w);
        }

        w = dlg->toplevel;
	gtk_widget_show_raise(w);
	dlg->map_state = TRUE;
}

/*
 *	Unmaps the run dialog.
 */
void EDVRunDlgUnmap(edv_run_dlg_struct *dlg)
{
	GtkWidget *w;

	if(dlg == NULL)
	    return;

	w = dlg->toplevel;
	if(w == NULL)
	    return;

	if(dlg->map_state)
	{
	    gtk_widget_hide(w);
	    dlg->map_state = FALSE;
	}
}

/*
 *	Deallocates all resources of the given run dialog and
 *	deallocates the structure itself.
 */
void EDVRunDlgDelete(edv_run_dlg_struct *dlg)
{
        GtkWidget **w;


        if(dlg == NULL)
            return;

        if(dlg->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}


	    /* Begin destroying widgets. */
            w = &dlg->run_combo;
            DO_DESTROY_WIDGET
            w = &dlg->browse_btn;
            DO_DESTROY_WIDGET
            w = &dlg->clear_btn;
            DO_DESTROY_WIDGET

            w = &dlg->working_dir_entry;
            DO_DESTROY_WIDGET
            w = &dlg->working_dir_browse_btn;
            DO_DESTROY_WIDGET
            w = &dlg->working_dir_clear_btn;
            DO_DESTROY_WIDGET
            w = &dlg->working_dir_get_from_cmd_btn;
            DO_DESTROY_WIDGET

            w = &dlg->run_in_terminal_check;
            DO_DESTROY_WIDGET
            w = &dlg->keep_dialog_check;
            DO_DESTROY_WIDGET

            w = &dlg->run_btn;
            DO_DESTROY_WIDGET
            w = &dlg->cancel_btn;
            DO_DESTROY_WIDGET

            w = &dlg->toplevel;
            DO_DESTROY_WIDGET

            if(dlg->accelgrp != NULL)
            {
                gtk_accel_group_unref(dlg->accelgrp);
                dlg->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
        }

        /* Deallocate structure itself. */
	g_free(dlg);
}
