#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvobj.h"
#include "edvcfg.h"
#include "edvconfirm.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvrecbinfop.h"
#include "recbin.h"
#include "recbindnd.h"
#include "edvdde.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


static gint EDVRecBinDNDConfirm(
        edv_core_struct *core_ptr,
	gint gdk_action, guint info,
        GtkWidget *toplevel,
        edv_dnd_object_struct **dnd_object, gint total_dnd_objects
);
static void EDVRecBinDNDUpdateStatusBar(
        edv_recbin_struct *recbin,
	gint gdk_action, guint info,
	gint total_src_objects,
	gint total_objects_processed, gint status
);

static void EDVRecBinDragDataReceivedNexus(
	edv_core_struct *core_ptr,
	edv_recbin_struct *recbin,	/* Can be NULL. */
        GdkDragContext *dc, guint info, GtkSelectionData *selection_data
);


void EDVRecBinContentsDNDSetIcon(
        edv_recbin_struct *recbin, gint row, gint column
);
gboolean EDVRecBinContentsDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y, guint t,
        gpointer data
);
gboolean EDVRecBinDeskIconDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y, guint t,
        gpointer data
);
void EDVRecBinContentsDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVRecBinContentsDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVRecBinDeskIconDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVRecBinContentsDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
);



/*
 *      All purpose initial operation confirmation procedure, prompts
 *      the user for response for confirmation of pending DND operation
 *      specified by the given inputs.
 *
 *      If the configuration will be checked first to se if confirmation
 *      is needed. If it is not needed then CDIALOG_RESPONSE_YES will
 *      be returned.
 *
 *      Returns a CDIALOG_RESPONSE_* code.
 */
static gint EDVRecBinDNDConfirm(
        edv_core_struct *core_ptr,
	gint gdk_action, guint info,
        GtkWidget *toplevel,
        edv_dnd_object_struct **dnd_object, gint total_dnd_objects
)
{
        const gchar *src_path = NULL;


        if(total_dnd_objects <= 0)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        /* If exactly one source object then get source path of that
         * object. Otherwise leave src_path as NULL.
         */
        if(total_dnd_objects == 1)
        {
            edv_dnd_object_struct *dnd_obj = dnd_object[0];
            if(dnd_obj != NULL)
                src_path = dnd_obj->full_path;
        }

        /* Do confirmation, handle by dnd target type. */
        if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
           (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
           (info == EDV_DND_TYPE_INFO_STRING)
        )
        {
            switch(gdk_action)
            {
              case GDK_ACTION_COPY:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_LINK:
		return(EDVConfirmDelete(
		    core_ptr, toplevel,
		    src_path, total_dnd_objects
		));
                break;
            }
        }
        else if(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        {
	    /* Skip this, a recycled object should never be dragged back
	     * into the recycle bin.
	     */
	}

        /* Unsupported DND target type or drag action, return response not
         * available just to be on the safe side so that the operation
         * does not continue.
         */
        return(CDIALOG_RESPONSE_NOT_AVAILABLE);
}

/*
 *      Updates the status bar to indicate the result of the dnd operation
 *      on the target recycle bin.
 */
static void EDVRecBinDNDUpdateStatusBar(
        edv_recbin_struct *recbin,
	gint gdk_action, guint info,
	gint total_src_objects,
        gint total_objects_processed, gint status
)
{
        gchar *buf = NULL;
        edv_status_bar_struct *sb;


        if(recbin == NULL)
            return;

        /* Get status bar. */
        sb = recbin->status_bar;
        if(sb == NULL)
            return;

        /* Begin formatting status message, handle by dnd target type. */
        if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
           (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
           (info == EDV_DND_TYPE_INFO_STRING)
        )
        {
            switch(gdk_action)
            {
              case GDK_ACTION_COPY:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_LINK:
		switch(status)
                {
                  case 0: case -5:
                    buf = g_strdup_printf(
                        "Deleted %i object%s",
                        total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
                    );
                    break;

                  case -4:  /* Cancel. */
		    buf = g_strdup_printf(
			"Delete operation canceled"
		    );
		    break;

		  default:  /* Error. */
		    buf = g_strdup_printf(
                        "Unable to delete object%s",
                        (total_src_objects == 1) ? "" : "s"
                    );
                    break;
                }
                break;
	    }
        }
        else if(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        {
            /* Recycled objects cannot be dragged back into the recycle
             * bin.
             */
	}
        else if(info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
        {
            /* Archive objects cannot cannot be `recycled'. */
        }
        else
        {
            /* Unsupported target type. */
        }

        /* Set status bar message. */
        EDVStatusBarMessage(sb, buf, FALSE);

        /* Deallocate status bar message buffer. */
        g_free(buf);
	buf = NULL;
}


/*
 *	Data received nexus, all inputs assumed valid.
 */
static void EDVRecBinDragDataReceivedNexus(
        edv_core_struct *core_ptr,
	edv_recbin_struct *recbin,	/* Can be NULL. */
        GdkDragContext *dc, guint info, GtkSelectionData *selection_data
)
{
	gint initial_confirmation_result;
	GtkWidget *toplevel = (recbin != NULL) ? recbin->toplevel : NULL;
	edv_dnd_object_struct **dnd_object, *dnd_obj;
	gint total_dnd_objects;


#define DO_FREE_LOCALS	\
{ \
 gint i; \
\
 /* Deallocate dnd recycled object reference list. */ \
 for(i = 0; i < total_dnd_objects; i++) \
  EDVDNDObjectDelete(dnd_object[i]); \
 g_free(dnd_object); \
 dnd_object = NULL; \
 total_dnd_objects = 0; \
}

        /* Parse the given selection buffer into an array of dnd object
         * structures.
         */
        dnd_object = EDVDNDBufferToObject(
            (const guint8 *)selection_data->data,
            selection_data->length,
            &total_dnd_objects
        );

	/* Make initial user confirmation querying to proceed with this
	 * operation.
	 */
	initial_confirmation_result = EDVRecBinDNDConfirm(
	    core_ptr, (gint)dc->action, info,
	    toplevel,
	    dnd_object, total_dnd_objects
	);
        /* User confirmed and dnd target type is a disk object? */
        if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
           ((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
            (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
            (info == EDV_DND_TYPE_INFO_STRING)
           )
        )
        {
            gint i, status = -1;
            gbool yes_to_all = FALSE;
	    gint objects_deleted = 0;
            guint *index;
            gint total_indexes;
            const gchar *error_mesg;


	    if(recbin != NULL)
	    {
		EDVRecBinSetBusy(recbin, TRUE);
		recbin->processing = TRUE;
	    }

            /* Iterate through list of dnd recycled object references. */
            for(i = 0; i < total_dnd_objects; i++)
            {
                dnd_obj = dnd_object[i];
                if(dnd_obj == NULL)
                    continue;

                if(dnd_obj->full_path == NULL)
                    continue;

                /* Handle by drag action type. */
                status = -1;
                index = NULL;
                total_indexes = 0;
		error_mesg = NULL;
                switch((gint)dc->action)
                {
                  case GDK_ACTION_COPY:
                    error_mesg = "Cannot copy an object to the recycle bin";
                    break;
                  case GDK_ACTION_MOVE:
                    status = EDVRecBinFOPDelete(
                        core_ptr,
                        dnd_obj->full_path,
                        &index, &total_indexes,
                        toplevel,
                        TRUE, TRUE, &yes_to_all
                    );
                    break;
                  case GDK_ACTION_LINK:
                    error_mesg = "Cannot link an object to the recycle bin";
                    break;
                  default:
                    error_mesg = "Unsupported drag operation";
                    break;
                }

                /* Get error message (if any) that might have occured
                 * in the above operation.
                 */
		if(error_mesg == NULL)
		    error_mesg = EDVRecBinFOPGetError();
                if((error_mesg != NULL) && (status != -4))
                {
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }

                /* Object was delected successfully? */
                if((index != NULL) && !status)
                {
                    gint n;

		    if(recbin != NULL)
			recbin->processing = FALSE;

		    /* Report recycle object added for all non-zero
		     * indexes.
		     */
		    for(n = 0; n < total_indexes; n++)
		    {
			if(index[n] != 0)
			    EDVRecycledObjectAddedEmit(core_ptr, index[n]);
		    }

		    if(recbin != NULL)
			recbin->processing = TRUE;

                    objects_deleted += total_indexes;
                }

                /* Deallocate returned recycle objects index array. */
                g_free(index);
                index = NULL;
                total_indexes = 0;

                /* Error occured (not 0) and it was not a user `no' or
                 * `not available' (-5) response?
                 */
                if(status && (status != -5))
                {
                    /* Skip rest of the DND object references due to error. */
                    break;
                }
            }

            /* Update status bar message. */
	    if(recbin != NULL)
		EDVRecBinDNDUpdateStatusBar(
		    recbin, (gint)dc->action, info,
		    total_dnd_objects, objects_deleted, status
		);

            /* Unmap progress dialog, it may have been mapped if any
             * operations occured in the above loop.
             */
            ProgressDialogBreakQuery(TRUE);
            ProgressDialogSetTransientFor(NULL);

	    if(recbin != NULL)
	    {
		recbin->processing = FALSE;
		EDVRecBinSetBusy(recbin, FALSE);
	    }
        }

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}


/*
 *      Sets the drag and drop icon based on the coordinates of the
 *      contents clist's node given by row.
 */
void EDVRecBinContentsDNDSetIcon(
        edv_recbin_struct *recbin, gint row, gint column
)
{
        GtkCList *clist;
        edv_recbin_object_struct *obj;


        if(recbin == NULL)
            return;

        if(!recbin->initialized || !recbin->map_state)
            return;

        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

        /* Get recycled object structure referenced by matched row. */
        obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
            clist, row
        );
        if(obj != NULL)
        {
            gint i, status;
            GdkPixmap *pixmap = NULL;
            GdkBitmap *mask = NULL;


            /* Iterate through each cell of this row, looking for a
             * useable pixmap.
             */
            for(i = 0; i < clist->columns; i++)
            {
                status = (gint)gtk_clist_get_cell_type(
                    clist, row, i
                );
                if(status == GTK_CELL_PIXMAP)
                {
                    gtk_clist_get_pixmap(
                        clist, row, i,
                        &pixmap, &mask
                    );
                    break;
                }
                else if(status == GTK_CELL_PIXTEXT)
                {
                    gchar *text = NULL;
                    guint8 spacing = 0;

                    gtk_clist_get_pixtext(
                        clist, row, i,
                        &text, &spacing, &pixmap, &mask
                    );
                    break;
                }
            }

            /* Is icon pixmap available? If so then set the new DND icon. */
            if(pixmap != NULL)
            {
                gint w = 15, h = 15;

                gdk_window_get_size((GdkWindow *)pixmap, &w, &h);

                GUIDNDSetDragIcon(
                    pixmap, mask,
                    (w / 2), (h / 2)
                );
            }
        }
}

/*
 *	Recycle bin contents clist DND "drag_motion" signal callback.
 *
 *	This is used to constrain all drags (regardless of its type
 *	or source data type) to be a drag action of move.
 */
gboolean EDVRecBinContentsDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y, guint t,
        gpointer data
)
{
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((dc == NULL) || (recbin == NULL))
            return(FALSE);

        if(dc->actions & GDK_ACTION_MOVE)
            gdk_drag_status(dc, GDK_ACTION_MOVE, t);
        else
            gdk_drag_status(dc, 0, t);

	return(TRUE);
}

/*
 *      Recycle bin desktop icon DND "drag_motion" signal callback.
 *
 *      This is used to constrain all drags (regardless of its type
 *      or source data type) to be a drag action of move.
 */
gboolean EDVRecBinDeskIconDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y, guint t,
        gpointer data
)
{
        edv_recbin_deskicon_struct *rbdi = (edv_recbin_deskicon_struct *)data;
        if((dc == NULL) || (rbdi == NULL))
            return(FALSE);

	if(dc->actions & GDK_ACTION_MOVE)
	    gdk_drag_status(dc, GDK_ACTION_MOVE, t);
	else
	    gdk_drag_status(dc, 0, t);

        return(TRUE);
}

/*
 *      Recycle bin contents clist DND "drag_data_get" signal callback.
 */
void EDVRecBinContentsDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
        gbool data_sent = FALSE;
        gint i;
        edv_core_struct *core_ptr;
        GList *glist;
        GtkWidget *w;
        GtkCList *clist;
        gint row;
        edv_dnd_object_struct **dnd_object, *dnd_obj;
        gint total_dnd_objects;
        edv_recbin_object_struct *obj;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((dc == NULL) || (recbin == NULL))
            return;

        if(!recbin->initialized || recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;
        /* Get contents clist and make sure it matches the given widget. */
        w = recbin->contents_clist;
        if((w == NULL) || (w != widget))
            return;

        clist = GTK_CLIST(w);


        /* Sync data on recycle bin so as to ensure we have the most up to
         * date information to send out.
         */
        EDVRecBinSyncData(recbin);


        /* Iterate through selected rows, generating a list of DND disk
         * object reference structures based on each selected row's disk
         * object structure.
         */
        total_dnd_objects = 0;
        dnd_object = NULL;
        glist = clist->selection;
        while(glist != NULL)
        {
            /* Get pointer to selected row. */
            row = (gint)glist->data;

            /* Get recycled object structure on selected row. */
            obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            if(obj != NULL)
            {
                /* Allocate more pointers for a new dnd recycled object
                 * reference structure.
                 */
                i = total_dnd_objects;
                total_dnd_objects = i + 1;
                dnd_object = (edv_dnd_object_struct **)g_realloc(
                    dnd_object,
                    total_dnd_objects * sizeof(edv_dnd_object_struct *)
                );
                if(dnd_object == NULL)
                {
                    total_dnd_objects = 0;
                    break;
                }
                /* Allocate a new dnd recycled object reference structure. */
                dnd_object[i] = dnd_obj = EDVDNDObjectNew();
                if(dnd_obj != NULL)
                {
                    dnd_obj->full_path = g_strdup_printf(
			"%i",
			obj->index
		    );
                }
            }

            /* Get next selected row. */
            glist = glist->next;
        }

        /* Any DND recycled object reference structures to send out? */
        if(total_dnd_objects > 0)
        {
            guint8 *buf;
            gint buf_len;


            /* Allocate a serial buffer for DND data sending based on the
             * list of DND recycled object reference structures.
             */
            buf = EDVDNDObjectToBuffer(
                (const edv_dnd_object_struct **)dnd_object,
                total_dnd_objects, &buf_len
            );
            if(buf != NULL)
            {
                /* Send out DND data buffer. */
                gtk_selection_data_set(
                    selection_data,
                    GDK_SELECTION_TYPE_STRING,
                    8,          /* 8 bits per character. */
                    buf, buf_len
                );
                data_sent = TRUE;

                /* Deallocate DND data buffer. */
                g_free(buf);
                buf = NULL;
                buf_len = 0;
            }
        }

        /* Deallocate all dnd recycled object reference structures. */
        for(i = 0; i < total_dnd_objects; i++)
            EDVDNDObjectDelete(dnd_object[i]);
        g_free(dnd_object);
        dnd_object = NULL;
        total_dnd_objects = 0;


        /* Failed to send out data? */
        if(!data_sent)
        {
            /* Then send out an error response. */
            const char *strptr = "Error";
            gtk_selection_data_set(
                selection_data,
                GDK_SELECTION_TYPE_STRING,
                8,      /* 8 bits per character. */
                strptr, strlen(strptr)
            );
            data_sent = TRUE;
        }
}

/*
 *	Recycle bin DND "drag_data_received" signal callback.
 */
void EDVRecBinContentsDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
	static gbool reenterent = FALSE;
	edv_core_struct *core_ptr;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((dc == NULL) || (recbin == NULL))
            return;

        if(!recbin->initialized || recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Check if received data is not empty. */
        if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
	{
	    reenterent = FALSE;
            return;
	}

        /* Sync data on recycle bin so as to ensure we have the most up to
         * date information to apply the new drag data to.
         */
        EDVRecBinSyncData(recbin);

        /* Handle received drag data. */
	EDVRecBinDragDataReceivedNexus(
	    core_ptr, recbin, dc, info, selection_data
	);

	reenterent = FALSE;
}

/*
 *      Recycle bin desktop icon DND "drag_data_received" signal callback.
 */
void EDVRecBinDeskIconDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_core_struct *core_ptr;
        edv_recbin_deskicon_struct *rbdi = (edv_recbin_deskicon_struct *)data;
        if((dc == NULL) || (rbdi == NULL))
            return;

        core_ptr = (edv_core_struct *)rbdi->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Check if received data is not empty. */
        if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
        {
            reenterent = FALSE;
            return;
        }

        /* Handle received drag data. */
        EDVRecBinDragDataReceivedNexus(
            core_ptr, NULL, dc, info, selection_data
        );

        reenterent = FALSE;
}


/*
 *      Recycle bin contents clist DND "drag_data_delete" signal callback.
 */
void EDVRecBinContentsDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	guint *index;
        gint i, total_indexes;
        edv_core_struct *core_ptr;
        GList *glist;
        GtkWidget *w;
        GtkCList *clist;
        gint row;
	const gchar *recycled_index_file;
        edv_recbin_object_struct *obj;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((dc == NULL) || (recbin == NULL))
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist and make sure it matches the given widget. */
        w = recbin->contents_clist;
        if((w == NULL) || (w != widget))
            return;

        clist = GTK_CLIST(w);

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
            return;


        /* Iterate through selected rows, getting a list of paths. */
	index = NULL;
        total_indexes = 0;
        glist = clist->selection;
        while(glist != NULL)
        {
            /* Get pointer to selected row. */
            row = (gint)glist->data;

            /* Get recycled object structure on selected row. */
            obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            if(obj != NULL)
            {
                i = total_indexes;
                total_indexes = i + 1;
                index = (guint *)g_realloc(
                    index, total_indexes * sizeof(guint)
                );
                if(index == NULL)
                {
                    total_indexes = 0;
                    break;
                }
                index[i] = obj->index;
            }

            /* Get next selected row. */
            glist = glist->next;
        }

        /* Emit signal indicating object removed for each path and
         * deallocate each path.
         */
        for(i = 0; i < total_indexes; i++)
        {
	    /* Get recycled object statistics, to check if the recycled
	     * object has actually been removed.
	     */
	    obj = EDVRecBinObjectStat(recycled_index_file, index[i]);

	    /* Recycled object actually no longer exists? */
	    if(obj == NULL)
		EDVRecycledObjectRemovedEmit(core_ptr, index[i]);

	    /* Deallocate recycled object structure. */
	    EDVRecBinObjectDelete(obj);
	    obj = NULL;
        }
        /* Deallocate selected recycled objects index list. */
        g_free(index);
        index = NULL;
        total_indexes = 0;
}
