#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvcfg.h"
#include "edvid.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvmimetypes.h"
#include "recbin.h"
#include "recbincb.h"
#include "recbincontents.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"



static void EDVRecBinContentsResetColumns(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist
);
static void EDVRecBinContentsClear(GtkCList *clist);

static void EDVRecBinContentsSetCellName(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellSize(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column,
        gbool hide_dir_size, gbool hide_link_size
);
static void EDVRecBinContentsSetCellType(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellPermissions(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column, gbool hide_link_permissions
);
static void EDVRecBinContentsSetCellOwner(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellGroup(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellDateAccess(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellDateModified(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellDateChanged(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellDateDeleted(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellLinkedTo(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellOriginalPath(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetCellIndex(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
);
static void EDVRecBinContentsSetRow(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row
);

static gint EDVRecBinContentsAppendObject(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object
);
static void EDVRecBinContentsGetListing(
        edv_recbin_struct *recbin, gbool update_status_bar
);

gint EDVRecBinContentsFindRowByIndex(
        edv_recbin_struct *recbin, guint index
);

void EDVRecBinContentsResetRows(edv_recbin_struct *recbin);
void EDVRecBinContentsDoUpdate(
        edv_recbin_struct *recbin, gbool update_status_bar
);

void EDVRecBinContentsObjectAddedNotify(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinContentsObjectModifiedNotify(
        edv_recbin_struct *recbin, guint index
);
void EDVRecBinContentsObjectRemovedNotify(
        edv_recbin_struct *recbin, guint index
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      Clears the titles of each column on the given clist.
 */
static void EDVRecBinContentsResetColumns(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist
)
{
        gint i, m, column_type, width;
        const gchar *title;
        edv_intlist_struct *column_types_intlist, *column_width_intlist;
        gint justify;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL))
            return;

        /* Get column types mapping. */
        column_types_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
        );
        if(column_types_intlist == NULL)
            return;

        column_width_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
        );
        if(column_width_intlist == NULL)
            return;


        /* Update clist column settings. */
        gtk_clist_column_titles_active(clist);
        gtk_clist_column_titles_show(clist);
	gtk_clist_set_auto_sort(clist, FALSE);
        gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING);


        /* Change clist selection mode to GTK_SELECTION_EXTENDED.
         * The selection mode can change whenever the contents list is
         * updated.
         */
/* Already set. */
/*        gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED); */

        /* Iterate through each column but not exceeding the number of
         * columns specified by the column types or column widths
         * intlists.
         */
        m = MIN(
            MIN(clist->columns, column_types_intlist->total),
            column_width_intlist->total
        );
        for(i = 0; i < m; i++)
        {
            /* Get column's type which determines what will be listed on
             * the current column i.
             */
            column_type = column_types_intlist->i[i];

            /* Get width of the column displaying this type. */
            if((column_type >= 0) && (column_type < column_width_intlist->total))
                width = column_width_intlist->i[column_type];
            else
                width = 0;

            /* Title based on column type. */
            switch(column_type)
            {
              case EDV_RECBIN_COLUMN_TYPE_NAME:
                title = "Name";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_SIZE:
                title = "Size";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_TYPE:
                title = "Type";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_PERMISSIONS:
                title = "Permissions";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_OWNER:
                title = "Owner";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_GROUP:
                title = "Group";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_ACCESS:
                title = "Date Access";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_MODIFIED:
                title = "Date Modified";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_CHANGED:
                title = "Date Changed";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_DELETED:
                title = "Date Deleted";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_LINKED_TO:
                title = "Linked To";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_ORIGINAL_PATH:
                title = "Original Path";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_RECBIN_COLUMN_TYPE_INDEX:
                title = "Index";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              default:
                title = "";
                justify = GTK_JUSTIFY_LEFT;
                break;
            }

            gtk_clist_set_column_auto_resize(
                clist, i, FALSE
            );
            gtk_clist_set_column_title(clist, i, title);
            gtk_clist_set_column_width(clist, i, width);
            gtk_clist_set_column_justification(clist, i, justify);
        }
        /* Reset the rest of the columns incase the number of columns in
         * the intlists were less than the columns specified by the clist.
         */
        for(; i < clist->columns; i++)
        {
            gtk_clist_set_column_auto_resize(
                clist, i, FALSE
            );
            gtk_clist_set_column_title(clist, i, "");
            gtk_clist_set_column_width(clist, i, 0);
        }
}

/*
 *      Deletes all items in the given clist.
 */
static void EDVRecBinContentsClear(GtkCList *clist)
{
        if(clist == NULL)
            return;

        gtk_clist_clear(clist);
}


/*
 *	Sets the recycled object name for the specified cell on the given
 *	clist.
 */
static void EDVRecBinContentsSetCellName(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        GdkPixmap *pixmap;
        GdkBitmap *mask;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get pixmap and mask for the given object. */
        EDVMatchObjectIcon(
            NULL, 0,
            core_ptr->mimetype, core_ptr->total_mimetypes,
            object->type,
            object->name,
	    TRUE, object->permissions,
            0,                  /* Small icons. */
            &pixmap, &mask,
            NULL, NULL, NULL, NULL
        );

        if(pixmap != NULL)
            gtk_clist_set_pixtext(
                clist, row, column,
                (object->name != NULL) ?
                    object->name : "(null)",
                EDV_LIST_PIXMAP_TEXT_SPACING,
                pixmap, mask
            );
        else
            gtk_clist_set_text(
                clist, row, column,
                (object->name != NULL) ?
                    object->name : "(null)"
            );
}

/*
 *      Sets the recycled object size for the specified cell on the given
 *      clist.
 */
static void EDVRecBinContentsSetCellSize(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column,
        gbool hide_dir_size, gbool hide_link_size
)
{
	const gchar *s;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	if(hide_dir_size && (object->type == EDV_OBJECT_TYPE_DIRECTORY))
	    s = "";
        else if(hide_link_size && (object->type == EDV_OBJECT_TYPE_LINK))
            s = "";
	else
	    s = EDVGetObjectSizeStr(core_ptr, object->size);
        gtk_clist_set_text(
            clist, row, column, s
        );
}

/*
 *      Sets the recycled object type for the specified cell on the given
 *      clist.
 */
static void EDVRecBinContentsSetCellType(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
	gchar *type_str;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get MIME Type type string for the given object. */
        EDVMatchObjectTypeString(
            core_ptr->mimetype, core_ptr->total_mimetypes,
            object->type, object->permissions,
            object->name,
            &type_str
        );

        gtk_clist_set_text(
            clist, row, column, (type_str != NULL) ? type_str : ""
        );
}

/*
 *      Sets the recycled object permissions for the specified cell on the given
 *      clist.
 */
static void EDVRecBinContentsSetCellPermissions(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column, gbool hide_link_permissions
)
{
        gchar s[80];


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        if(hide_link_permissions && (object->type == EDV_OBJECT_TYPE_LINK))
            *s = '\0';
	else
            sprintf(
                s,
                "%c%c%c%c%c%c%c%c%c",
            (object->permissions & EDV_PERMISSION_UREAD) ? 'r' : '-',
            (object->permissions & EDV_PERMISSION_UWRITE) ? 'w' : '-',
            (object->permissions & EDV_PERMISSION_SETUID) ?
                'S' :
                ((object->permissions & EDV_PERMISSION_UEXECUTE) ? 'x' : '-'),
            (object->permissions & EDV_PERMISSION_GREAD) ? 'r' : '-',
            (object->permissions & EDV_PERMISSION_GWRITE) ? 'w' : '-',
            (object->permissions & EDV_PERMISSION_SETGID) ?
                'G' :
                ((object->permissions & EDV_PERMISSION_GEXECUTE) ? 'x' : '-'),
            (object->permissions & EDV_PERMISSION_AREAD) ? 'r' : '-',
            (object->permissions & EDV_PERMISSION_AWRITE) ? 'w' : '-',
            (object->permissions & EDV_PERMISSION_STICKY) ?
                'T' :
                ((object->permissions & EDV_PERMISSION_AEXECUTE) ? 'x' : '-')
            );

        gtk_clist_set_text(
            clist, row, column, s
        );
}

/*
 *      Sets the recycled object owner for the specified cell on the given
 *      clist.
 */
static void EDVRecBinContentsSetCellOwner(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        const gchar *owner_name;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get owner name from object's user id. */
        owner_name = EDVUIDGetNameFromUID(
            core_ptr->uid, core_ptr->total_uids,
            object->owner_id, NULL
        );
        if(owner_name == NULL)
            owner_name = "";

        gtk_clist_set_text(
            clist, row, column, owner_name
        );
}

/*
 *      Sets the recycled object owner for the specified cell on the given
 *      clist.
 */
static void EDVRecBinContentsSetCellGroup(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        const gchar *group_name;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get group name from object's group id. */
        group_name = EDVGIDGetNameFromGID(
            core_ptr->gid, core_ptr->total_gids,
            object->group_id, NULL
        );
        if(group_name == NULL)
            group_name = "";

        gtk_clist_set_text(
            clist, row, column, group_name
        );
}

/*
 *	Sets the recycled object last access date for the specified cell
 *	on the given clist.
 */
static void EDVRecBinContentsSetCellDateAccess(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->access_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the recycled object last modified date for the specified cell
 *      on the given clist.
 */
static void EDVRecBinContentsSetCellDateModified(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->modify_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the recycled object last changed date for the specified cell
 *      on the given clist.
 */
static void EDVRecBinContentsSetCellDateChanged(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->change_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the recycled object delete date for the specified cell
 *      on the given clist.
 */
static void EDVRecBinContentsSetCellDateDeleted(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->date_deleted, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}


/*
 *      Sets the recycled object linked to path for the specified cell
 *      on the given clist.
 */
static void EDVRecBinContentsSetCellLinkedTo(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        const gchar *link_dest;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get link destination. */
/*
        link_dest = (object->linked_to != NULL) ? object->linked_to : "";
 */
link_dest = "";

        gtk_clist_set_text(
            clist, row, column, link_dest
        );
}

/*
 *      Sets the recycled object linked to path for the specified cell
 *      on the given clist.
 */
static void EDVRecBinContentsSetCellOriginalPath(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
        const gchar *original_path;


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get original path. */
	original_path = (object->original_path != NULL) ?
	    object->original_path : "";

        gtk_clist_set_text(
            clist, row, column, original_path
        );
}

/*
 *      Sets the recycled object index to path for the specified cell
 *      on the given clist.
 */
static void EDVRecBinContentsSetCellIndex(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row, gint column
)
{
 	gchar num_str[80];


        if((core_ptr == NULL) || (recbin == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get index string. */
	sprintf(num_str, "%i", object->index);

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}


/*
 *      Sets all cells of the specified row of the given contents clist
 *      with the values of the given object.
 *
 *      The clist row's client data will not be updated.
 */
static void EDVRecBinContentsSetRow(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object,
        gint row
)
{
        gint i, column_type, columns;
        gbool hide_dir_size, hide_link_size, hide_link_permissions;
        edv_intlist_struct *column_types_intlist;


        if((core_ptr == NULL) || (recbin == NULL) ||
           (clist == NULL) || (object == NULL)
        )
            return;


        /* Get column types mapping. */
        column_types_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
        );
        if(column_types_intlist == NULL)
            return;

        /* Get additional display options. */
        hide_dir_size = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
            EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_DIR_SIZE
        );
        hide_link_size = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
            EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_LINK_SIZE
        );
        hide_link_permissions = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
            EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_LINK_PERMISSIONS
        );


        /* Get total number of columns on the contents clist. */
        columns = MIN(clist->columns, column_types_intlist->total);

        /* Iterate through each column but not more than the columns
         * specified by the column types intlist.
         */
        for(i = 0; i < columns; i++)
        {
            /* Check column references on configuration list, see which
             * one this column should show.
             */
            column_type = column_types_intlist->i[i];
            switch(column_type)
            {
              case EDV_RECBIN_COLUMN_TYPE_NAME:
                EDVRecBinContentsSetCellName(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_SIZE:
                EDVRecBinContentsSetCellSize(
                    core_ptr, recbin, clist, object,
                    row, i, hide_dir_size, hide_link_size
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_TYPE:
                EDVRecBinContentsSetCellType(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_PERMISSIONS:
                EDVRecBinContentsSetCellPermissions(
                    core_ptr, recbin, clist, object,
                    row, i, hide_link_permissions
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_OWNER:
                EDVRecBinContentsSetCellOwner(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_GROUP:
                EDVRecBinContentsSetCellGroup(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_ACCESS:
                EDVRecBinContentsSetCellDateAccess(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_MODIFIED:
                EDVRecBinContentsSetCellDateModified(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_CHANGED:
                EDVRecBinContentsSetCellDateChanged(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_DATE_DELETED:
		EDVRecBinContentsSetCellDateDeleted(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_LINKED_TO:
                EDVRecBinContentsSetCellLinkedTo(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_ORIGINAL_PATH:
                EDVRecBinContentsSetCellOriginalPath(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;

              case EDV_RECBIN_COLUMN_TYPE_INDEX:
                EDVRecBinContentsSetCellIndex(
                    core_ptr, recbin, clist, object,
                    row, i
                );
                break;
	    }
	}
}

/*
 *	Appends the recycled object to the contents clist. The given
 *	recycled object structure will be transfered to the new row and
 *	should not be referenced again after this call.
 *
 *      Returns the row number that it was appended to or -1 on failure.
 */
static gint EDVRecBinContentsAppendObject(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin,
        GtkCList *clist, edv_recbin_object_struct *object
)
{
        gint row, columns;
        gchar **row_text;


        if((core_ptr == NULL) || (recbin == NULL) ||
           (clist == NULL) || (object == NULL)
        )
        {
            EDVRecBinObjectDelete(object);
            return(-1);
        }

        /* Get total number of columns on the contents clist. */
        columns = MAX(clist->columns, 1);


        /* Allocate row text. */
        row_text = (gchar **)g_malloc0(columns * sizeof(gchar *));
        if(row_text != NULL)
        {
            gint i;

            /* Allocate blank strings for each cell. */
            for(i = 0; i < columns; i++)
                row_text[i] = g_strdup("");

            /* Append a new row. */
            row = gtk_clist_append(clist, row_text);

            /* Deallocate row text. */
            for(i = 0; i < columns; i++)
                g_free(row_text[i]);
            g_free(row_text);
            row_text = NULL;
        }
        else
        {
            row = -1;
        }
        /* Successfully appended row? */
        if(row < 0)
        {
            EDVRecBinObjectDelete(object);
            return(-1);
        }

        /* Update each cell of the new row with the values from the
         * disk object structure.
         */
        EDVRecBinContentsSetRow(
            core_ptr, recbin, clist, object, row
        );

	/* Set given recycled object structure as the row data for this
	 * new row. The recycled object structure should not be referenced
	 * again after this point.
         */
        gtk_clist_set_row_data_full(
            clist, row,
            object, (GtkDestroyNotify)EDVRecBinContentsItemDestroyCB
        );
        /* Recycled object pointer has now been transfered to the new row's
         * data. Mark it NULL so we do not reference it again.
         */
        object = NULL;

        return(row);
}

/*
 *	Gets all recycled objects in the recycled objects directory and
 *	appends them to the contents clist.
 */
static void EDVRecBinContentsGetListing(
        edv_recbin_struct *recbin, gbool update_status_bar
)
{
        gint last_progress_percent, progress_percent;
	gint recycled_objects_loaded, total_recycled_objects;
        GtkCList *clist;
        edv_core_struct *core_ptr;
	const gchar *recycled_index_file;
	edv_recbin_object_struct *obj;
	edv_recbin_index_struct *rbi_ptr;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

	/* Get path to recycled objects index file. */
	recycled_index_file = EDVCFGItemListGetValueS(
	    core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
            return;

	/* Get total number of recycled objects. */
	total_recycled_objects = EDVRecBinFIOTotalItems(recycled_index_file);


	last_progress_percent = -1;     /* None/undefined. */
	progress_percent = 0;
	recycled_objects_loaded = 0;

        /* Get listing of contents in the directory specified by path. */
	rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
        while(!EDVRecBinIndexNext(rbi_ptr))
        {
            obj = rbi_ptr->obj;
            if(obj != NULL)
            {
		gint new_row;
		edv_recbin_object_struct *tar_obj;


		/* Make a copy of the recycled object obtained from the
		 * index file.
		 */
		tar_obj = EDVRecBinObjectCopy(obj);
		if(tar_obj != NULL)
		{
		    /* Append row, transfering the recycled object
		     * structure tar_obj to the new row. The tar_obj
		     * should not be referenced after this point.
		     */
		    new_row = EDVRecBinContentsAppendObject(
			core_ptr, recbin, clist, tar_obj
		    );
		    tar_obj = NULL;

		    recycled_objects_loaded++;
		}

		/* Update progress? */
		if(update_status_bar && (total_recycled_objects > 0))
		{
		    progress_percent = recycled_objects_loaded * 100 /
			total_recycled_objects;
		    if(progress_percent > last_progress_percent)
		    {
			EDVStatusBarProgress(
			    recbin->status_bar,
			    (gfloat)recycled_objects_loaded /
				(gfloat)total_recycled_objects,
			    TRUE
                        );
                        progress_percent = last_progress_percent;
                    }
		}
	    }
	}

	/* Close recycled index file. */
	EDVRecBinIndexClose(rbi_ptr);
        rbi_ptr = NULL;
}

/*
 *      Returns the row index of the contents clist that contains a
 *	recycled object structure who's index matches the given index.
 *
 *      Can return -1 on failed match.
 */
gint EDVRecBinContentsFindRowByIndex(
        edv_recbin_struct *recbin, guint index
)
{
        gint i;
        GtkCList *clist;
        edv_recbin_object_struct *obj;


        if(recbin == NULL)
            return(-1);

        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return(-1);

        /* Iterate through all rows. */
        for(i = 0; i < clist->rows; i++)
        {
            obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
                clist, i
            );
            if(obj == NULL)
                continue;

            /* Full path not specified on disk object structure? */
            if(obj->index == index)
                return(i);
        }

        return(-1);
}

/*
 *      Updates all rows on the contents clist by getting the row data
 *      and updates the clist's rows.
 *
 *      This is designed to be called whenever the displayed values
 *      of each row need to be set again from internal data, usually
 *      when a MIME Type has been added/modified/removed. This function
 *      should not be used to `refresh' the list (get new values of
 *      disk object structures), use EDVBrowserContentsDoUpdate()
 *      instead.
 */
void EDVRecBinContentsResetRows(edv_recbin_struct *recbin)
{
        gint i;
        GtkWidget *w;
        GtkCList *clist;
        edv_core_struct *core_ptr;
        edv_recbin_object_struct *object;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        w = recbin->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

        for(i = 0; i < clist->rows; i++)
        {
            object = (edv_recbin_object_struct *)gtk_clist_get_row_data(
                clist, i
            );
            if(object != NULL)
                EDVRecBinContentsSetRow(
                    core_ptr, recbin, clist, object, i
                );
        }
}

/*
 *	Procedure to clear and reupdate the contents list on the recycle
 *	bin with the contents found in the recycled objects directory.
 */
void EDVRecBinContentsDoUpdate(
        edv_recbin_struct *recbin, gbool update_status_bar
)
{
        GtkWidget *w;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        w = recbin->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);


        if(update_status_bar)
        {
            /* Reset progress. */
            EDVStatusBarMessage(
                recbin->status_bar,
                "Loading recycled objects...",
                FALSE
            );
            EDVStatusBarProgress(
                recbin->status_bar, 0.0, TRUE
            );
        }


        /* Delete all entries. */
        EDVRecBinContentsClear(clist);


        /* Update columns. */
        EDVRecBinContentsResetColumns(
            core_ptr, recbin, clist
        );

        /* Get listing of recycled objects. */
        EDVRecBinContentsGetListing(recbin, update_status_bar);


        if(update_status_bar)
        {
            /* Reset progress. */
            EDVStatusBarMessage(
                recbin->status_bar,
                "Recycled objects loaded",
                FALSE
            );
            EDVStatusBarProgress(
                recbin->status_bar, 0.0, TRUE
            );
        }
}


/*
 *	This should be called whenever a new object has been added to the
 *	recycled objects directory.
 */
void EDVRecBinContentsObjectAddedNotify(
        edv_recbin_struct *recbin, guint index
)
{
        GtkWidget *w;
        GtkCList *clist;
        gint row;
        edv_core_struct *core_ptr;
	const gchar *recycled_index_file;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        w = recbin->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
            return;

	/* Check if the recycled object specified by the given index
	 * already exists in the list.
	 */
	row = EDVRecBinContentsFindRowByIndex(recbin, index);
	if((row >= 0) && (row < clist->rows))
	{
	    /* Recycled object already in list, do nothing (no update). */
	}
	else
	{
	    /* Recycled object not in list, so add it. */
	    edv_recbin_object_struct *obj = EDVRecBinObjectStat(
		recycled_index_file, index
	    );
	    if(obj != NULL)
	    {
		/* Append row, transfering the recycled object structure
		 * tar_obj to the new row. The tar_obj should not be
		 * referenced after this point.
		 */
		row = EDVRecBinContentsAppendObject(
		    core_ptr, recbin, clist, obj
		);
		obj = NULL;
	    }
	}
}

/*
 *	This should be called whenever an existing recycled object has been
 *	modified in the recycled objects directory.
 */
void EDVRecBinContentsObjectModifiedNotify(
        edv_recbin_struct *recbin, guint index
)
{
        GtkWidget *w;
        GtkCList *clist;
        gint row;
        edv_core_struct *core_ptr;
        const gchar *recycled_index_file;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        w = recbin->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
            return;

        /* Check if the recycled object specified by the given index
         * already exists in the list.
         */
        row = EDVRecBinContentsFindRowByIndex(recbin, index);
        if((row >= 0) && (row < clist->rows))
        {
            /* Recycled object already in list, update it. */
	    edv_recbin_object_struct *obj;


	    /* Get existing recycled object structure from the matched
	     * row data and delete it.
	     */
	    obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
		clist, row
	    );
	    if(obj != NULL)
	    {
		gtk_clist_set_row_data_full(clist, row, NULL, NULL);
		EDVRecBinObjectDelete(obj);
		obj = NULL;
	    }

	    /* Create a new recycled object structure from the given index
	     * and update this row.
	     */
	    obj = EDVRecBinObjectStat(
                recycled_index_file, index
            );
	    if(obj != NULL)
	    {
                gtk_clist_set_row_data_full(
                    clist, row,
                    obj, (GtkDestroyNotify)EDVRecBinContentsItemDestroyCB
                );

		EDVRecBinContentsSetRow(
		    core_ptr, recbin, clist, obj, row
		);
	    }
	}
        else
        {
            /* Recycled object not in list, so add it. */
            edv_recbin_object_struct *obj = EDVRecBinObjectStat(
                recycled_index_file, index
            );
            if(obj != NULL)
            {
                /* Append row, transfering the recycled object structure
                 * tar_obj to the new row. The tar_obj should not be
                 * referenced after this point.
                 */
                row = EDVRecBinContentsAppendObject(
                    core_ptr, recbin, clist, obj
                );
                obj = NULL;
            }
        }
}

/*
 *      This should be called whenever an existing recycled object has been
 *      removed from the recycled objects directory.
 */
void EDVRecBinContentsObjectRemovedNotify(
        edv_recbin_struct *recbin, guint index
)
{
        GtkWidget *w;
        GtkCList *clist;
        gint row;
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        w = recbin->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

        /* Check if the recycled object specified by the given index
         * already exists in the list.
         */
        row = EDVRecBinContentsFindRowByIndex(recbin, index);
	if((row >= 0) && (row < clist->rows))
	{
	    /* Remove matched row. */
	    gtk_clist_remove(clist, row);
	}
}
