#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvstatusbar.h"
#include "recbin.h"
#include "recbincb.h"
#include "recbinopcb.h"
#include "recbindnd.h"
#include "recbincontents.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvlistseek.h"
#include "edvcfglist.h"
#include "config.h"



void EDVRecBinContentsItemDestroyCB(gpointer data);

static gint EDVRecBinCListColumnSortDateNexus(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
        gint sort_code
);
static gint EDVRecBinCListColumnSortDateAccessCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateModifyCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateChangeCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateDeleteCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);

gint EDVRecBinDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
void EDVRecBinDestroyCB(GtkObject *object, gpointer data);

gint EDVRecBinMapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
);
gint EDVRecBinUnmapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
);

gint EDVRecBinKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint EDVRecBinButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVRecBinHandleChildAttachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void EDVRecBinHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void EDVRecBinResizeColumnCB(
        GtkCList *clist, gint column, gint width, gpointer data
);
void EDVRecBinClickColumnCB(
        GtkCList *clist, gint column, gpointer data
);
void EDVRecBinSelectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);
void EDVRecBinUnselectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);

void EDVRecBinWriteProtectChangedCB(
        edv_recbin_struct *recbin, gbool state
);
void EDVRecBinRecycledObjectAddedNotifyCB(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinRecycledObjectModifiedNotifyCB(
        edv_recbin_struct *recbin, guint index
);
void EDVRecBinRecycledObjectRemovedNotifyCB(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinReconfiguredNotifyCB(edv_recbin_struct *recbin);
void EDVRecBinMimeTypeAddedCB(
        edv_recbin_struct *recbin,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVRecBinMimeTypeModifiedCB(
        edv_recbin_struct *recbin,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVRecBinMimeTypeRemovedCB(
        edv_recbin_struct *recbin, gint mt_num
);



#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Recycle bin contents GtkCList item "destroy" signal callback.
 */
void EDVRecBinContentsItemDestroyCB(gpointer data)
{
	EDVRecBinObjectDelete((edv_recbin_object_struct *)data);
}


/*
 *      Returns the sort code for the rows sorted by date.
 *
 *      The recycled object structures are obtained from each row and the
 *      dates are compared.
 */
static gint EDVRecBinCListColumnSortDateNexus(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
        gint sort_code
)
{
        gint sort_column;
        const GtkCListRow *row1 = (const GtkCListRow *)ptr1;
        const GtkCListRow *row2 = (const GtkCListRow *)ptr2;
        edv_recbin_object_struct *obj1, *obj2;


        if((clist == NULL) || (row1 == NULL) || (row2 == NULL))
            return(-1);

        sort_column = clist->sort_column;
        if((sort_column < 0) || (sort_column >= clist->columns))
            return(-1);

        obj1 = (edv_recbin_object_struct *)row1->data;
        obj2 = (edv_recbin_object_struct *)row2->data;
        if((obj1 == NULL) || (obj2 == NULL))
            return(-1);

        switch(sort_code)
        {
          case 0:       /* Access time. */
            if(obj1->access_time <= obj2->access_time)
                    return((gint)(obj1->access_time < obj2->access_time));
                else
                    return(-1);
            break;

          case 1:       /* Modify time. */
            if(obj1->modify_time <= obj2->modify_time)
                    return((gint)(obj1->modify_time < obj2->modify_time));
                else
                    return(-1);
            break;

          case 2:       /* Change time. */
            if(obj1->change_time <= obj2->change_time)
                    return((gint)(obj1->change_time < obj2->change_time));
                else
                    return(-1);
            break;

          case 3:	/* Delete time. */
            if(obj1->date_deleted <= obj2->date_deleted)
                    return((gint)(obj1->date_deleted < obj2->date_deleted));
                else
                    return(-1);
            break;
        }

        return(-1);
}

static gint EDVRecBinCListColumnSortDateAccessCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
            clist, ptr1, ptr2, 0
        ));
}

static gint EDVRecBinCListColumnSortDateModifyCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVRecBinCListColumnSortDateNexus(
            clist, ptr1, ptr2, 1
        ));
}

static gint EDVRecBinCListColumnSortDateChangeCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVRecBinCListColumnSortDateNexus(
            clist, ptr1, ptr2, 2
        ));
}

static gint EDVRecBinCListColumnSortDateDeleteCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVRecBinCListColumnSortDateNexus(
            clist, ptr1, ptr2, 3
        ));
}


/*
 *	Recycle bin toplevel GtkWidget "delete_event" signal callback.
 */
gint EDVRecBinDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if(recbin == NULL)
            return(TRUE);

        if(recbin->processing)
            return(TRUE);

        EDVRecBinOPClose(recbin);

        return(TRUE);
}

/*
 *	Recycle bin toplevel GtkWidget "destroy" signal callback.
 */
void EDVRecBinDestroyCB(GtkObject *object, gpointer data)
{
        return;
}


/*
 *	Recycle bin toplevel "map" signal callback.
 */
gint EDVRecBinMapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
)
{
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if(recbin == NULL)
	    return(FALSE);

	return(FALSE);
}

/*
 *      Recycle bin toplevel "unmap" signal callback.
 */
gint EDVRecBinUnmapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
)
{
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if(recbin == NULL)
            return(FALSE);

	return(FALSE);
}


/*
 *	Recycle bin "key_press_event" or "key_release_event" signal callback.
 */
gint EDVRecBinKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
        static gbool reenterent = FALSE;
        gint status = FALSE;
        gint etype;
        guint keyval, state;
        gbool press;
        edv_core_struct *core_ptr;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((widget == NULL) || (key == NULL) || (recbin == NULL))
            return(status);

        if(recbin->processing)
            return(status);

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

        /* Get event type. */
        etype = key->type;

        /* Get other key event values. */
        press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
        keyval = key->keyval;
        state = key->state;

/* Macro to emit a signal stop for a key press or release depending
 * on the current event's type.
 */
#define DO_STOP_KEY_SIGNAL_EMIT \
{ \
 gtk_signal_emit_stop_by_name( \
  GTK_OBJECT(widget), \
  press ? "key_press_event" : "key_release_event" \
 ); \
}

/* Macro to clamp the GtkAdjustment adj and emit a "value_changed"
 * signal.
 */
#define DO_ADJ_CLAMP_EMIT       \
{ \
 if(adj->value > (adj->upper - adj->page_size)) \
  adj->value = adj->upper - adj->page_size; \
\
 if(adj->value < adj->lower) \
  adj->value = adj->lower; \
\
 gtk_signal_emit_by_name( \
  GTK_OBJECT(adj), "value_changed" \
 ); \
}

/* Macro to clamp the GtkCList clist's focus_row. */
#define DO_CLIST_FOCUS_ROW_CLAMP	\
{ \
 if(clist->focus_row >= clist->rows) \
  clist->focus_row = clist->rows - 1; \
 if(clist->focus_row < 0) \
  clist->focus_row = 0; \
}


        /* Check which ctree this signal is for. */

	/* Contents clist. */
	if(widget == recbin->contents_clist)
        {
            gint row;
	    GList *glist;
            GtkCList *clist = GTK_CLIST(widget);


            /* Get last selected row. */
	    row = EDVCListGetSelectedLast(clist, NULL);

            /* Handle by key value. */
            switch(keyval)
            {
              case GDK_Return:
              case GDK_KP_Enter:
              case GDK_ISO_Enter:
              case GDK_3270_Enter:
                /* Handle only if control key modifier is not held, so
                 * that accelerator keys will get handled properly.
                 */
                if(!(state & GDK_CONTROL_MASK))
                {
		    if(press)
			EDVRecBinOPRecover(recbin);

                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

	      case GDK_Delete:
                if(press)
                {
		    if((state & GDK_CONTROL_MASK) &&
		       (state & GDK_SHIFT_MASK)
		    )
			EDVRecBinOPPurgeAll(recbin);
		    else
			EDVRecBinOPPurge(recbin);
                }
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
                break;

              case GDK_space:
              case GDK_KP_Space:
                row = clist->focus_row;
                if((row >= 0) && (row < clist->rows) && press)
                {
                    gbool already_selected = FALSE;

                    /* Check if this row is already selected. */
                    glist = clist->selection;
                    while(glist != NULL)
                    {
                        if(row == (gint)glist->data)
                        {
                            already_selected = TRUE;
                            break;
                        }
                        glist = glist->next;
                    }

                    gtk_clist_freeze(clist);
                    if(already_selected)
                        gtk_clist_unselect_row(clist, row, 0);
                    else
                        gtk_clist_select_row(clist, row, 0);
                    gtk_clist_thaw(clist);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Up:
              case GDK_KP_Up:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                else
                {
                    if(press)
                    {
                        gint prev_focus_row = clist->focus_row;

                        gtk_clist_freeze(clist);
                        clist->focus_row--;
                        DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
                            gtk_clist_moveto(
                                clist,
                                clist->focus_row, -1,   /* Row, column. */
                                0.5, 0.0                /* Row, column. */
                            );

                        if(state & GDK_SHIFT_MASK)
                        {
                            gtk_clist_select_row(
                                clist, prev_focus_row, 0
                            );
                            gtk_clist_select_row(
                                clist, clist->focus_row, 0
                            );
                        }
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

              case GDK_Down:
              case GDK_KP_Down:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                else
                {
                    if(press)
                    {
                        gint prev_focus_row = clist->focus_row;

                        gtk_clist_freeze(clist);
                        clist->focus_row++;
                        DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
                            gtk_clist_moveto(
                                clist,
                                clist->focus_row, -1,   /* Row, column. */
                                0.5, 0.0                /* Row, column. */
                            );

                        if(state & GDK_SHIFT_MASK)
                        {
                            gtk_clist_select_row(
                                clist, prev_focus_row, 0
                            );
                            gtk_clist_select_row(
                                clist, clist->focus_row, 0
                            );
                        }
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

              case GDK_Left:
              case GDK_KP_Left:
                if(1)
                {
                    /* Get adjustment and scroll left. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        if(state & GDK_CONTROL_MASK)
                            adj->value -= adj->step_increment * 4;
                        else if(state & GDK_SHIFT_MASK)
                            adj->value = adj->lower;
                        else
                            adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Right:
              case GDK_KP_Right:
                if(1)
                {
                    /* Get adjustment and scroll right. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        if(state & GDK_CONTROL_MASK)
                            adj->value += adj->step_increment * 4;
                        else if(state & GDK_SHIFT_MASK)
                            adj->value = adj->upper - adj->page_size;
                        else
                            adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Up:
              case GDK_KP_Page_Up:
                if(1)
                {
                    /* Get adjustment and scroll up one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Down:
              case GDK_KP_Page_Down:
                if(1)
                {
                    /* Get adjustment and scroll down one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Home:
              case GDK_KP_Home:
                if(1)
                {
                    /* Get adjustment and scroll all the way up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->lower;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_End:
              case GDK_KP_End:
                if(1)
                {
                    /* Get adjustment and scroll all the way down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->upper - adj->page_size;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              default:
                /* For all other alphanumeric character keys and while
                 * no modifier keys are held, attempt to seek to the
                 * item who's name starts with the letter of the key
                 * that was pressed.
                 */
                if(isalnum((char)keyval) &&
                   ((state == 0x00000000) || (state == GDK_SHIFT_MASK))
                )
                {
                    /* Get the column that is displaying the object
                     * name as column_type_name.
                     */
                    gint i, column_type_name = -1;
                    edv_intlist_struct *column_type_intlist =
                        EDVCFGItemListGetValueIntList(
                            core_ptr->cfg_list,
                            EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
                        );
                    if(column_type_intlist != NULL)
                    {
                        /* Iterate through column type intlist. */
                        for(i = 0; i < column_type_intlist->total; i++)
                        {
                            /* Is column i displaying the name? */
                            if(column_type_intlist->i[i] == EDV_RECBIN_COLUMN_TYPE_NAME)
                            {
                                column_type_name = i;
                                break;
                            }
                        }
                    }

                    gtk_clist_freeze(clist);
                    EDVCListSeekCharacter(
                        clist, column_type_name, 0,
                        (gchar)((state & GDK_SHIFT_MASK) ?
                            toupper((char)keyval) : keyval
                        )
                    );
                    gtk_clist_thaw(clist);

                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

            }
        }

#undef DO_CLIST_FOCUS_ROW_CLAMP
#undef DO_ADJ_CLAMP_EMIT
#undef DO_STOP_KEY_SIGNAL_EMIT

        reenterent = FALSE;
        return(status);
}

/*
 *	Recycle bin GtkWidget "button_press_event" signal callback.
 */
gint EDVRecBinButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
        static gbool reenterent = FALSE;
	gint status = FALSE;
        gint etype;
        edv_core_struct *core_ptr;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((widget == NULL) || (recbin == NULL))
            return(status);

        if(recbin->processing)
            return(status);

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

	/* Get event type. */
        etype = button->type;

	/* Check which widget this signal is for. */
	if(widget == recbin->contents_clist)
        {
            gint row, column;
            gint rows_selected = 0, selected_row = -1;
            GList *glist;
            GtkCList *clist = GTK_CLIST(widget);


            /* Find row and column based on given coordinates. */
            if(!gtk_clist_get_selection_info(
                clist, button->x, button->y, &row, &column
            ))
            {
                row = -1;
                column = 0;
            }

            /* Get number of selected rows and highest selected row. */
            glist = clist->selection;
            while(glist != NULL)
            {
                rows_selected++;
                selected_row = (gint)glist->data;
                glist = glist->next;
            }

            /* Handle by button number. */
            switch(button->button)
            {
              case 3:
                if(etype == GDK_BUTTON_PRESS)
                {
                    GtkMenu *menu;

                    /* Select item before mapping menu? */
                    if(EDVCFGItemListGetValueI(
                        core_ptr->cfg_list, EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS
                    ) && (row >= 0) && (row < clist->rows))
                    {
                        /* Select the row that the button was pressed over.
                         * if no key modifiers are held then this will also
                         * unselect all previously selected rows.
                         */
                        gtk_clist_freeze(clist);
                        if(!(button->state & GDK_CONTROL_MASK) &&
                           !(button->state & GDK_SHIFT_MASK)
                        )
                            gtk_clist_unselect_all(clist);
                        clist->focus_row = row;
                        gtk_clist_select_row(clist, row, 0);
                        gtk_clist_thaw(clist);
                    }

                    /* Update all menus and map right click menu. */
                    EDVRecBinUpdateMenus(recbin);
                    menu = (GtkMenu *)recbin->contents_clist_menu;
                    if(menu != NULL)
                        gtk_menu_popup(
                            menu, NULL, NULL,
                            NULL, NULL,
                            button->button, button->time
                        );
                }
		status = TRUE;
                break;

              case 2:
                if(etype == GDK_BUTTON_PRESS)
                {
                    if((row >= 0) && (row < clist->rows))
		    {
/* Renaming not allowed. */
		    }
                }
                break;

              case 1:
		/* Double click? */
                if(etype == GDK_2BUTTON_PRESS)
                {
                    if((row >= 0) && (row < clist->rows))
		    {
			/* Do recover recycled object. */
			EDVRecBinOPRecover(recbin);
			status = TRUE;
		    }
                }
                break;
            }

            if(etype == GDK_BUTTON_PRESS)
                gtk_widget_grab_focus(widget);
        }

        reenterent = FALSE;
        return(status);
}


/*
 *      GtkHandleBox "child_attached" signal callback.
 */
void EDVRecBinHandleChildAttachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((handle_box == NULL) || (recbin == NULL))
            return;

        gtk_widget_queue_resize(
            gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
        );
}

/*
 *      GtkHandleBox "child_detached" signal callback.
 */
void EDVRecBinHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((handle_box == NULL) || (recbin == NULL))
            return;

        gtk_widget_queue_resize(
            gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
        );
}


/*
 *      Recycle bin GtkCList "resize_column" signal callback.
 */
void EDVRecBinResizeColumnCB(
        GtkCList *clist, gint column, gint width, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_core_struct *core_ptr;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((clist == NULL) || (recbin == NULL))
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == recbin->contents_clist)
        {
            gint column_type = -1;
            edv_intlist_struct *column_types_intlist, *column_width_intlist;


            /* Get column_type from the given column index. */
            column_types_intlist = EDVCFGItemListGetValueIntList(
                core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
            );
            if(column_types_intlist != NULL)
            {
                if((column >= 0) && (column < column_types_intlist->total))
                    column_type = column_types_intlist->i[column];
            }

            /* Get column widths intlist. */
            column_width_intlist = EDVCFGItemListGetValueIntList(
                core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
            );
            if(column_width_intlist != NULL)
            {
                /* Need to increase column_width_intlist allocation? */
                if((column_type >= column_width_intlist->total) &&
                   (column_type >= 0)
                )
                {
                    gint n;
                    gint prev_total = column_width_intlist->total;

                    /* Increase array allocation. */
                    column_width_intlist->total = column_type + 1;
                    column_width_intlist->i = (gint *)g_realloc(
                        column_width_intlist->i,
                        column_width_intlist->total * sizeof(gint)
                    );
                    /* Reset newly allocated indexes. */
                    for(n = prev_total; n < column_width_intlist->total; n++)
                        column_width_intlist->i[n] = 0;
                }

                /* Got column type in bounds as index on column width
                 * intlist?
                 */
                if((column_type >= 0) &&
                   (column_type < column_width_intlist->total)
                )
                {
                    /* Record new width of column on the column width
                     * intlist.
                     */
                    column_width_intlist->i[column_type] = width;
                }
            }
        }

        reenterent = FALSE;
}

/*
 *      Recycle bin GtkCList "click_column" signal callback.
 */
void EDVRecBinClickColumnCB(
        GtkCList *clist, gint column, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_core_struct *core_ptr;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((clist == NULL) || (recbin == NULL))
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == recbin->contents_clist)
        {
            edv_intlist_struct *column_types_intlist;
            GtkCListCompareFunc cmp_func = NULL;
            GtkCListCompareFunc cmp_func_str =
                 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
            GtkCListCompareFunc cmp_func_num =
                (GtkCListCompareFunc)EDVCListColumnSortNumericCB;


            EDVRecBinSetBusy(recbin, TRUE);
	    GUIBlockInput(recbin->toplevel, TRUE);
            recbin->processing = TRUE;


            /* Get column types mapping. */
            column_types_intlist = EDVCFGItemListGetValueIntList(
                core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
            );
            if(column_types_intlist != NULL)
            {
                if((column >= 0) && (column < column_types_intlist->total))
                {
                    gint column_type = column_types_intlist->i[column];
                    switch(column_type)
                    {
                      case EDV_RECBIN_COLUMN_TYPE_NAME:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_SIZE:
                        cmp_func = cmp_func_num;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_TYPE:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_PERMISSIONS:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_OWNER:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_GROUP:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_DATE_ACCESS:
                        cmp_func = EDVRecBinCListColumnSortDateAccessCB;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_DATE_MODIFIED:
                        cmp_func = EDVRecBinCListColumnSortDateModifyCB;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_DATE_CHANGED:
                        cmp_func = EDVRecBinCListColumnSortDateChangeCB;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_DATE_DELETED:
                        cmp_func = EDVRecBinCListColumnSortDateDeleteCB;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_LINKED_TO:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_ORIGINAL_PATH:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_RECBIN_COLUMN_TYPE_INDEX:
                        cmp_func = cmp_func_num;
                        break;
		    }
		}
	    }


            gtk_clist_freeze(clist);

            /* Set sort column settings on the clist. */
            if(column != clist->sort_column)
                gtk_clist_set_sort_column(clist, column);
            else
                gtk_clist_set_sort_type(
                    clist,
                    (clist->sort_type == GTK_SORT_ASCENDING) ?
                        GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
                );
            if(cmp_func != NULL)
                gtk_clist_set_compare_func(clist, cmp_func);

            /* Sort rows. */
            gtk_clist_sort(clist);

            gtk_clist_thaw(clist);

            recbin->processing = FALSE;
            GUIBlockInput(recbin->toplevel, FALSE);
            EDVRecBinSetBusy(recbin, FALSE);

            EDVRecBinUpdateMenus(recbin);
        }

        reenterent = FALSE;
}

/*
 *      Recycle bin GtkCList "select_row" signal callback.
 */
void EDVRecBinSelectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((clist == NULL) || (recbin == NULL))
            return;

        if(recbin->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == recbin->contents_clist)
        {
            gint total_selected = 0;
            GList *glist;
            edv_recbin_object_struct *obj;


            /* Get total number of objects selected. */
            glist = clist->selection;
            while(glist != NULL)
            {
                total_selected++;
                glist = glist->next;
            }


            /* Update selected row. */
            recbin->contents_clist_selected_row = row;

            /* Update DND icon for contents clist. */
	    EDVRecBinContentsDNDSetIcon(recbin, row, column);

            /* Get disk object structure from selected row. */
            obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            if(obj != NULL)
            {
                /* Update status bar message. */
                if(obj->name != NULL)
                {
                    gchar *buf;
                    const gchar *type_str = "Object";
                    gchar size_str[80];


                    /* Get object type string and size string. */
                    *size_str = '\0';
                    switch(obj->type)
                    {
                      case EDV_OBJECT_TYPE_FILE:
                        type_str = "File";
                        sprintf(size_str, " (%s bytes)",
                            EDVGetObjectSizeStr(
                                (edv_core_struct *)recbin->core_ptr,
                                obj->size
                            )
			);
                        break;
                      case EDV_OBJECT_TYPE_DIRECTORY:
                        type_str = "Directory";
                        break;
                      case EDV_OBJECT_TYPE_LINK:
                        type_str = "Link";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
                        type_str = "Block device";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
                        type_str = "Character device";
                        break;
                      case EDV_OBJECT_TYPE_FIFO:
                        type_str = "FIFO Pipe";
                        break;
                      case EDV_OBJECT_TYPE_SOCKET:
                        type_str = "Socket";
                        break;
                    }

                    /* Format status bar message. */
                    if(total_selected > 1)
                        buf = g_strdup_printf(
                            "%i objects selected",
                            total_selected
                        );
                    else if(!strcmp(obj->name, ".."))
                        buf = g_strdup_printf(
                            "Parent directory selected"
                        );
                    else
                        buf = g_strdup_printf(
                            "%s \"%s\" selected%s",
                            type_str, obj->name, size_str
                        );

                    /* Set status bar message. */
                    EDVStatusBarMessage(
                        recbin->status_bar, buf, FALSE
                    );

                    /* Deallocate status bar message. */
                    g_free(buf);
                }
                else
                {
                    EDVStatusBarMessage(
                        recbin->status_bar,
                        "Object with NULL name selected",
                        FALSE
                    );
                }
            }

            /* Check if selected row is fully visible, if not then adjust
             * scroll position to try and make it visible.
             */
            if(gtk_clist_row_is_visible(clist, row) !=
                GTK_VISIBILITY_FULL
            )
                gtk_clist_moveto(
                    clist,
                    row, -1,	/* Row, column. */
                    0.5, 0.0	/* Row, column. */
                );

/*	    EDVRecBinSetTitle(recbin); */
            EDVRecBinUpdateMenus(recbin);
        }

        reenterent = FALSE;
}

/*
 *      Recycle bin GtkCList "unselect_row" signal callback.
 */
void EDVRecBinUnselectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((clist == NULL) || (recbin == NULL))
            return;

        if(recbin->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == recbin->contents_clist)
        {


/*	    EDVRecBinSetTitle(recbin); */
            EDVRecBinUpdateMenus(recbin);
        }

        reenterent = FALSE;
}


/*
 *      Called whenever the global write protect has changed.
 *
 *      The new state is given as state.
 */
void EDVRecBinWriteProtectChangedCB(
        edv_recbin_struct *recbin, gbool state
)
{
        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        EDVRecBinUpdateMenus(recbin);
}

/*
 *      Called whenever a recycled object has been added to the recycled
 *	objects directory.
 */
void EDVRecBinRecycledObjectAddedNotifyCB(
        edv_recbin_struct *recbin, guint index
)
{
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Notify contents clist about added object. */
        EDVRecBinContentsObjectAddedNotify(recbin, index);

	/* Update menus and recycle bin toplevel window's WM icon. */
	EDVRecBinUpdateMenus(recbin);

        /* Update menus for the purpose of updating the recycle bin
         * toplevel window's WM icon. Do this only if there is a change
         * in the number of recycled objects from 0.
         */
        if(core_ptr->last_recbin_items != recbin->last_recbin_items)
            EDVRecBinUpdateMenus(recbin);
}

/*
 *      Called whenever a recycled object has been modified in the recycled
 *      objects directory.
 */
void EDVRecBinRecycledObjectModifiedNotifyCB(
        edv_recbin_struct *recbin, guint index
)
{
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Notify contents clist about modified object. */
        EDVRecBinContentsObjectModifiedNotify(recbin, index);

        /* Update menus for the purpose of updating the recycle bin
         * toplevel window's WM icon. Do this only if there is a change
         * in the number of recycled objects from 0.
         */
        if(core_ptr->last_recbin_items != recbin->last_recbin_items)
            EDVRecBinUpdateMenus(recbin);
}

/*
 *      Called whenever a recycled object has been removed from the recycled
 *      objects directory.
 */
void EDVRecBinRecycledObjectRemovedNotifyCB(
        edv_recbin_struct *recbin, guint index
)
{
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Notify contents clist and directory ctree about removed object. */
	EDVRecBinContentsObjectRemovedNotify(recbin, index);

        /* Update menus for the purpose of updating the recycle bin
	 * toplevel window's WM icon. Do this only if there is a change
	 * in the number of recycled objects from 0.
	 */
        if(core_ptr->last_recbin_items != recbin->last_recbin_items)
	    EDVRecBinUpdateMenus(recbin);
}


/*
 *      Called whenever the global configuration has changed.
 */
void EDVRecBinReconfiguredNotifyCB(edv_recbin_struct *recbin)
{
        GtkWidget *w;
        GtkCList *clist;
        edv_status_bar_struct *status_bar;
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;


	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfiguring.
	 */
	recbin->last_recbin_items = -1;
	recbin->last_write_protect_state = -1;


        /* Update title. */
        EDVRecBinSetTitle(recbin);

        /* Regenerate tool bar. */
        EDVRecBinToolbarRegenerate(recbin);

        /* Show tool bar? */
        w = recbin->tool_bar_handle;
        if(w != NULL)
        {
            recbin->tool_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR
            );
            if(recbin->tool_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show find bar? */
        w = recbin->find_bar_handle;
        if(w != NULL)
        {
            recbin->find_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR
            );
            if(recbin->find_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show status bar? */
        status_bar = recbin->status_bar;
        if(status_bar != NULL)
        {
            recbin->status_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR
            );
            if(recbin->status_bar_map_state)
                EDVStatusBarMap(status_bar);
            else
                EDVStatusBarUnmap(status_bar);
        }


        /* Reget listing for contents clist. */
        clist = (GtkCList *)recbin->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
	    EDVRecBinContentsResetRows(recbin);
            gtk_clist_thaw(clist);
        }


        /* Update menus. */
        EDVRecBinUpdateMenus(recbin);


        /* Notify recbin's toplevel widget to resize. */
        w = recbin->toplevel;
        if(w != NULL)
            gtk_widget_queue_resize(w);
}


/*
 *      Called whenever a MIME Type has been added to the core
 *      structure's list of MIME Types.
 */
void EDVRecBinMimeTypeAddedCB(
        edv_recbin_struct *recbin,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        /* Treat a MIME Type added the same as it would be for a MIME Type
         * modified, forward signal to the MIME Type modified callback.
         */
	EDVRecBinMimeTypeModifiedCB(recbin, mt_num, mt_ptr);
}

/*
 *      Called whenever a MIME Type has been modified on the core
 *      structure's list of MIME Types.
 */
void EDVRecBinMimeTypeModifiedCB(
        edv_recbin_struct *recbin,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Reget listings. */

        /* Reset displayed row values of contents clist. */
        clist = (GtkCList *)recbin->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
	    EDVRecBinContentsResetRows(recbin);
            gtk_clist_thaw(clist);
        }

/*      EDVRecBinUpdateMenus(recbin); */
}

/*
 *      Called whenever a MIME Type has been removed from the core
 *      structure's list of MIME Types.
 */
void EDVRecBinMimeTypeRemovedCB(
        edv_recbin_struct *recbin, gint mt_num
)
{
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;


        /* Reget listings. */

        /* Reset displayed row values of contents clist. */
        clist = (GtkCList *)recbin->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
            EDVRecBinContentsResetRows(recbin);
            gtk_clist_thaw(clist);
        }

/*	EDVRecBinUpdateMenus(recbin); */
}
