#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <utime.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"
#include "piechart.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvid.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "edvdevices.h"
#include "edvmimetypes.h"
#include "propwin.h"
#include "endeavour.h"
#include "edvcfglist.h"
#include "edvfop.h"
#include "edvcb.h"
#include "edvop.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "config.h"


#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_fsck_48x48.xpm"
#include "images/icon_tools_48x48.xpm"
#include "images/icon_floppy_48x48.xpm"
#include "images/icon_properties2_20x20.xpm"
#include "images/icon_properties2_48x48.xpm"



static gint EDVPropWinDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVPropWinSwitchPageCB(
        GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
        gpointer data
);

static void EDVPropWinOKCB(GtkWidget *widget, gpointer data);
static void EDVPropWinApplyCB(GtkWidget *widget, gpointer data);
static void EDVPropWinCancelCB(GtkWidget *widget, gpointer data);

static void EDVPropWinDateTouchCB(GtkWidget *widget, gpointer data);
static void EDVPropWinRunCheckCB(GtkWidget *widget, gpointer data);
static void EDVPropWinRunToolsCB(GtkWidget *widget, gpointer data);
static void EDVPropWinRunFormatCB(GtkWidget *widget, gpointer data);

static void EDVPropWinOwnerMapPUListCB(GtkWidget *widget, gpointer data);
static void EDVPropWinGroupMapPUListCB(GtkWidget *widget, gpointer data);

static void EDVPropWinLinkDestinationChangeCB(GtkWidget *widget, gpointer data);
static void EDVPropWinDestinationPropertiesCB(GtkWidget *widget, gpointer data);

static void EDVPropWinDoApply(edv_propwin_struct *pw);


static void EDVPropWinCreateGeneralPage(
	edv_propwin_struct *pw, GtkWidget *parent,
	edv_core_struct *core_ptr, edv_object_struct *obj
);
static void EDVPropWinCreateLinkPage(
        edv_propwin_struct *pw, GtkWidget *parent,
        edv_core_struct *core_ptr, edv_object_struct *obj
);
static void EDVPropWinCreateDevicePage(
        edv_propwin_struct *pw, GtkWidget *parent,
        edv_core_struct *core_ptr, edv_object_struct *obj,
	gint dev_num, edv_device_struct *dev_ptr
);
static void EDVPropWinCreateDeviceNodePage(
        edv_propwin_struct *pw, GtkWidget *parent,
        edv_core_struct *core_ptr, edv_object_struct *obj
);

void EDVWriteProtectChangedCB(edv_propwin_struct *pw, gbool state);
void EDVPropWinReconfiguredNotifyCB(edv_propwin_struct *pw);

edv_propwin_struct *EDVPropWinNew(
	gpointer core_ptr, edv_object_struct *obj
);
void EDVPropWinMap(edv_propwin_struct *pw);
void EDVPropWinUnmap(edv_propwin_struct *pw);
void EDVPropWinDelete(edv_propwin_struct *pw);


#define PROPWIN_WIDTH	380
#define PROPWIN_HEIGHT	440

#define PROPWIN_BTN_WIDTH	(100 + (2 * 3))
#define PROPWIN_BTN_HEIGHT	(30 + (2 * 3))



/*
 *      "delete_event" signal callback.
 */
static gint EDVPropWinDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return(TRUE);

        EDVPropWinCancelCB(NULL, pw);

        return(TRUE);
}

/*
 *      GtkNotebook "switch_page" signal callback.
 */
static void EDVPropWinSwitchPageCB(
        GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;




        reenterent = FALSE;
}

/*
 *      OK button callback.
 */
static void EDVPropWinOKCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVPropWinDoApply(pw);
        EDVPropWinUnmap(pw);

        reenterent = FALSE;
}

/*
 *      Apply button callback.
 */
static void EDVPropWinApplyCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVPropWinDoApply(pw);

        reenterent = FALSE;
}

/*
 *      Cancel button callback.
 */
static void EDVPropWinCancelCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVPropWinUnmap(pw);

        reenterent = FALSE;
}


/*
 *	Updates the time stamps of the object to current.
 */
static void EDVPropWinDateTouchCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
	int relativity;
	const gchar *format;
	const gchar *path;
	GtkWidget *w;
	edv_object_struct *obj;
	edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

	path = pw->path;
	if(path == NULL)
	    return;

        core_ptr = (edv_core_struct *)pw->core_ptr;
        if(core_ptr == NULL)
            return;

	/* If the touch button is not sensitive, then it means that
	 * the date cannot be modified (and this function should never
	 * have been called).
	 */
	w = pw->gen_date_touch_btn;
	if((w != NULL) ? !GTK_WIDGET_SENSITIVE(w) : TRUE)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );


	/* Create a new tempory disk object structure and obtain
	 * current stats.
	 */
	obj = EDVObjectNew();
	if(obj != NULL)
	{
	    gulong cur_time;
	    struct utimbuf utime_buf;
	    struct stat lstat_buf;

	    /* Get stats of the disk object. */
	    if(!lstat(path, &lstat_buf))
	    {
		EDVObjectSetPath(obj, path);
		EDVObjectSetStat(obj, &lstat_buf);
/*		EDVObjectValidateLink(obj); */

		/* Update the time stamps to current time. */
		cur_time = time(NULL);
		obj->access_time = cur_time;
		obj->modify_time = cur_time;
                obj->change_time = cur_time;

		/* Actually modify object's time stamps. */
		utime_buf.actime = obj->access_time;
		utime_buf.modtime = obj->modify_time;
		if(utime(path, &utime_buf))
		{
		    const gchar *error_mesg =
"Unable to modify the object's time stamps";

		    /* Failed. */
		    switch(errno)
		    {
		      case EACCES:
			error_mesg =
"Insufficient permission to modify the object's time stamps";
			break;

		      case ENOENT:
			error_mesg =
"The object no longer exists";
		    }
		    CDialogSetTransientFor(pw->toplevel);
		    CDialogGetResponse(
			"Operation Error",
			error_mesg,
			NULL,
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		}
		else
		{
		    /* Success, now Update time stamp labels on the
		     * properties window.
		     */
                    w = pw->gen_date_access_label;
                    if(w != NULL)
                        gtk_label_set_text(
                            GTK_LABEL(w),
                            EDVDateFormatString(
                                obj->access_time, format, relativity
                            )
                        );

                    w = pw->gen_date_modified_label;
                    if(w != NULL)
                        gtk_label_set_text(
                            GTK_LABEL(w),
                            EDVDateFormatString(
                                obj->modify_time, format, relativity
                            )
                        );

		    w = pw->gen_date_changed_label;
		    if(w != NULL)
			gtk_label_set_text(
			    GTK_LABEL(w),
			    EDVDateFormatString(
				obj->change_time, format, relativity
			    )
			);

		    /* Try to get new local object stats, if successful
		     * then emit an object modified signal to all of
		     * Endeavour's resources.
		     */
		    if(!lstat(path, &lstat_buf))
			EDVObjectModifiedEmit(
			    core_ptr, path, path, &lstat_buf
			);
		}

	    }	/* Get stats of the disk object. */

	    EDVObjectDelete(obj);
	    obj = NULL;
	}



        reenterent = FALSE;
}

/*
 *      Runs the file system check command.
 */
static void EDVPropWinRunCheckCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
	gint dev_num;
	edv_device_struct *dev_ptr;
        edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        core_ptr = (edv_core_struct *)pw->core_ptr;
        if(core_ptr == NULL)
            return;

	dev_num = pw->dev_num;
	if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
	    dev_ptr = core_ptr->device[dev_num];
	else
	    dev_ptr = NULL;
	if(dev_ptr == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	EDVDoRunDeviceCheck(core_ptr, dev_ptr, pw->toplevel);

        reenterent = FALSE;
}

/*
 *	Runs the device tools command.
 */
static void EDVPropWinRunToolsCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        gint dev_num;
        edv_device_struct *dev_ptr;
        edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        core_ptr = (edv_core_struct *)pw->core_ptr;
        if(core_ptr == NULL)
            return;

        dev_num = pw->dev_num;
        if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
            dev_ptr = core_ptr->device[dev_num];
        else
            dev_ptr = NULL;
        if(dev_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVDoRunDeviceTools(core_ptr, dev_ptr, pw->toplevel);

        reenterent = FALSE;
}

/*
 *      Runs the device format command.
 */
static void EDVPropWinRunFormatCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        gint dev_num;
        edv_device_struct *dev_ptr;
        edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        core_ptr = (edv_core_struct *)pw->core_ptr;
        if(core_ptr == NULL)
            return;

        dev_num = pw->dev_num;
        if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
            dev_ptr = core_ptr->device[dev_num];
        else
            dev_ptr = NULL;
        if(dev_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVDoRunDeviceFormat(core_ptr, dev_ptr, pw->toplevel);

        reenterent = FALSE;
}


/*
 *	Maps the popup list for the owner entry.
 */
static void EDVPropWinOwnerMapPUListCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
	const gchar *value;
	GtkEntry *entry;
	edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

	core_ptr = (edv_core_struct *)pw->core_ptr;
	if(core_ptr == NULL)
	    return;

	entry = (GtkEntry *)pw->gen_owner_entry;
	if(entry == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	/* Block input and get value. */
	value = PUListMapQuery(
	    core_ptr->users_pulist,
	    gtk_entry_get_text(entry),
	    15,
	    PULIST_RELATIVE_CENTER,
	    GTK_WIDGET(entry), widget
	);
	/* Got value? */
	if(value != NULL)
	    gtk_entry_set_text(entry, value);

        reenterent = FALSE;
}

/*
 *      Maps the popup list for the group entry.
 */
static void EDVPropWinGroupMapPUListCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        const gchar *value;
        GtkEntry *entry;
        edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        core_ptr = (edv_core_struct *)pw->core_ptr;
        if(core_ptr == NULL)
            return;

        entry = (GtkEntry *)pw->gen_group_entry;
        if(entry == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Block input and get value. */
        value = PUListMapQuery(
            core_ptr->groups_pulist,
            gtk_entry_get_text(entry),
            15,
            PULIST_RELATIVE_CENTER,
            GTK_WIDGET(entry), widget
        );
        /* Got value? */
        if(value != NULL)
            gtk_entry_set_text(entry, value);

        reenterent = FALSE;
}

/*
 *	Link destination entry "changed" callback.
 */
static void EDVPropWinLinkDestinationChangeCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        const gchar *value;
        GtkEntry *entry;
        edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        core_ptr = (edv_core_struct *)pw->core_ptr;
        if(core_ptr == NULL)
            return;

        entry = (GtkEntry *)pw->link_dest_entry;
        if(entry == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	/* Get link destination value. */
	value = gtk_entry_get_text(entry);
	if(value != NULL)
	{
	    const gchar *dest_path;

	    /* Is link destination value an absolute path? */
	    if(ISPATHABSOLUTE(value))
	    {
		dest_path = value;
	    }
	    else
	    {
		/* Link destination is not an absolute path, so prefix
		 * the parent path of the link to the link destination 
		 * path.
		 */
		dest_path = GetParentDir(pw->path);
		dest_path = PrefixPaths(dest_path, value);
	    }
	    /* Got link destination path? */
	    if(dest_path != NULL)
	    {
		gbool sensitive;
		GtkWidget *w;
		struct stat lstat_buf;


		/* Check if link destination path exists locally. */
		if(lstat(dest_path, &lstat_buf))
		    sensitive = FALSE;
		else
		    sensitive = TRUE;

		/* Update link destination properties button sensitivity
		 * based on existance of link destination path object.
		 */
		w = pw->link_dest_properties_btn;
		if(w != NULL)
		    gtk_widget_set_sensitive(w, sensitive);
	    }
	}

        reenterent = FALSE;
}

/*
 *	Destination properties callback.
 */
static void EDVPropWinDestinationPropertiesCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
	const gchar *value;
        GtkEntry *entry;
        edv_core_struct *core_ptr;
        edv_propwin_struct *pw = (edv_propwin_struct *)data;
        if(pw == NULL)
            return;

        core_ptr = (edv_core_struct *)pw->core_ptr;
        if(core_ptr == NULL)
            return;

        entry = (GtkEntry *)pw->link_dest_entry;
        if(entry == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Get link destination value. */
        value = gtk_entry_get_text(entry);
        if(value != NULL)
        {
            const gchar *dest_path;

            /* Is link destination value an absolute path? */
            if(ISPATHABSOLUTE(value))
            {
                dest_path = value;
            }
            else
            {
                /* Link destination is not an absolute path, so prefix
                 * the parent path of the link to the link destination
                 * path.
                 */
                dest_path = GetParentDir(pw->path);
                dest_path = PrefixPaths(dest_path, value);
            }
            /* Got link destination path? */
            if(dest_path != NULL)
            {
		gchar *new_dest_path = g_strdup(dest_path);
		edv_object_struct *obj;
                struct stat lstat_buf;


                /* Check if link destination path exists locally. */
                if(!lstat(new_dest_path, &lstat_buf))
		{
		    obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, new_dest_path);
			EDVObjectSetStat(obj, &lstat_buf);
			EDVObjectValidateLink(obj);

			/* Create a new properties window on the core 
			 * structure and instruct it to load the destination
			 * disk object.
			 */
			EDVDoPropWin(core_ptr, obj, pw->toplevel);
		    }
		    EDVObjectDelete(obj);
		    obj = NULL;
		}

		/* Deallocate copy of destination path. */
		g_free(new_dest_path);
            }
        }

        reenterent = FALSE;
}


/*
 *	Applies values from the property window to the disk object
 *	specified on the property window.
 */
static void EDVPropWinDoApply(edv_propwin_struct *pw)
{
	GtkWidget *w;
	const gchar *path;
	edv_core_struct *core_ptr;
	struct stat lstat_buf;


	if(pw == NULL)
	    return;

	core_ptr = (edv_core_struct *)pw->core_ptr;
	if(core_ptr == NULL)
	    return;

	path = pw->path;
	if(path == NULL)
	    return;

	/* Check if global write protect state is on. */
	if(EDVCheckWriteProtect(core_ptr, TRUE))
	    return;


	/* Check if device node major and minor number widgets are valid,
	 * implying the device node needs to be recreated.
	 */
	w = pw->dev_node_major_entry;
	if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	{
	    const gchar *cstrptr;
	    gint new_major = 0, new_minor = 0, major = 0, minor = 0;


	    /* Get new major and minor numbers. */
            w = pw->dev_node_major_entry;
            if(w != NULL)
            {
                cstrptr = gtk_entry_get_text(GTK_ENTRY(w));
                new_major = (cstrptr != NULL) ? atoi(cstrptr) : 0;
            }
	    w = pw->dev_node_minor_entry;
	    if(w != NULL)
	    {
		cstrptr = gtk_entry_get_text(GTK_ENTRY(w));
	        new_minor = (cstrptr != NULL) ? atoi(cstrptr) : 0;
	    }

	    /* Get current major and minor numbers of actual device node. */
	    if(!lstat(path, &lstat_buf))
		EDVGetDeviceNumbers(lstat_buf.st_rdev, &major, &minor);

	    /* Original device node numbers different from new device
	     * node numbers?
	     */
	    if((new_major != major) || (new_minor != minor))
	    {
		/* Change in device node numbers which implies the device
		 * node needs to be recreated.
		 */
		mode_t m = lstat_buf.st_mode;
		dev_t rdev = EDVFormatDeviceNumbers(new_major, new_minor);

		unlink(path);
		mknod(path, m, rdev);
	    }
	}

	/* Check if link destination entry is valid, implying there is
	 * a new link destination to set.
	 */
	w = pw->link_dest_entry;
	if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
        {
            gbool yes_to_all = FALSE;
	    gint status;
            const gchar *cstrptr, *error_mesg;
	    gchar *new_dest;


	    /* Get copy of the new destination value. */
	    cstrptr = gtk_entry_get_text(GTK_ENTRY(w));
	    new_dest = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;

	    /* Relink. */
	    status = EDVFOPRelink(
		core_ptr,
		new_dest, path,
		pw->toplevel,
		TRUE, TRUE, &yes_to_all
	    );

	    /* Deallocate copy of new destination value. */
	    g_free(new_dest);
	    new_dest = NULL;

            /* Unmap progress dialog, it may have been mapped in the above
             * operation.
             */
            ProgressDialogBreakQuery(FALSE);
            ProgressDialogSetTransientFor(NULL);

            /* Get error message if any that might have occured in the
             * above operation.
             */
            error_mesg = EDVFOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(pw->toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }
	}

	/* Check if permissions frame is sensitive, implying there are
	 * permissions to be set.
	 */
        w = pw->gen_permissions_frame;
        if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
        {
            gbool yes_to_all = FALSE;
            guint permissions = 0;
            gint status;
            const gchar *error_mesg;


	    w = pw->gen_uread_check;
	    if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
		permissions |= EDV_PERMISSION_UREAD;
            w = pw->gen_uwrite_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_UWRITE;
            w = pw->gen_uexecute_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_UEXECUTE;

            w = pw->gen_gread_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_GREAD;
            w = pw->gen_gwrite_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_GWRITE;
            w = pw->gen_gexecute_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_GEXECUTE;

            w = pw->gen_aread_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_AREAD;
            w = pw->gen_awrite_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_AWRITE;
            w = pw->gen_aexecute_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_AEXECUTE;

            w = pw->gen_setuid_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_SETUID;
            w = pw->gen_setgid_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_SETGID;
            w = pw->gen_sticky_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_STICKY;

            /* Do chmod procedure. */
            status = EDVFOPChmod(
                core_ptr, path, permissions,
                pw->toplevel,
                TRUE, TRUE, &yes_to_all
            );

            /* Unmap progress dialog, it may have been mapped in the above
             * operation.
             */
            ProgressDialogBreakQuery(FALSE);
            ProgressDialogSetTransientFor(NULL);

            /* Get error message if any that might have occured in the
             * above operation.
             */
            error_mesg = EDVFOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(pw->toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }
	}

	/* Begin setting owner. */
	w = pw->gen_ownership_frame;
	if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	{
	    gbool yes_to_all = FALSE;
	    gint owner_id = 0, group_id = 0;
	    gint status;
	    const gchar *error_mesg;


            /* Get user id from user entry. */
            w = pw->gen_owner_entry;
            if(w != NULL)
            {
                owner_id = EDVUIDNameToUID(
                    core_ptr->uid, core_ptr->total_uids,
                    gtk_entry_get_text(GTK_ENTRY(w))
                );
            }

	    /* Get group id from group entry. */
	    w = pw->gen_group_entry;
	    if(w != NULL)
	    {
		group_id = EDVGIDNameToGID(
		    core_ptr->gid, core_ptr->total_gids,
		    gtk_entry_get_text(GTK_ENTRY(w))
		);
	    }

            /* Get original stats before changing owners. */
	    if(!lstat(path, &lstat_buf))
	    {
		/* Original owner or group different from new owner or group? */
		if((owner_id != lstat_buf.st_uid) ||
		   (group_id != lstat_buf.st_gid)
		)
		{
		    /* Do chown procedure. */
		    status = EDVFOPChown(
			core_ptr, path, owner_id, group_id,
			pw->toplevel,
			TRUE, TRUE, &yes_to_all
		    );

		    /* Unmap progress dialog, it may have been mapped in
		     * the above operation.
		     */
		    ProgressDialogBreakQuery(FALSE);
		    ProgressDialogSetTransientFor(NULL);

		    /* Get error message if any that might have occured
		     * in the above operation.
		     */
		    error_mesg = EDVFOPGetError();
		    if(error_mesg != NULL)
		    {
			CDialogSetTransientFor(pw->toplevel);
			CDialogGetResponse(
			    "Operation Error",
			    error_mesg,
			    NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		    }
		}
	    }
	}

        /* Try to get new local object stats, if successful then emit an
	 * object modified signal to all of endeavour's resources.
	 */
        if(!lstat(path, &lstat_buf))
        {
            EDVObjectModifiedEmit(
                core_ptr, path, path, &lstat_buf
            );
        }

}


/*
 * 	Creates the general page parented to the given GtkVBox parent
 *	with respect to the given disk object structure values.
 */
static void EDVPropWinCreateGeneralPage(
        edv_propwin_struct *pw, GtkWidget *parent,
        edv_core_struct *core_ptr, edv_object_struct *obj
)
{
	gint border_major = 5, border_minor = 2;
	gint relativity;
	const gchar *format;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	GtkWidget *w, *parent2, *parent3, *parent4, *parent5;


        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );


	if(obj == NULL)
	    return;


	/* Get medium pixmap and mask pair for this object. */
	EDVMatchObjectIcon(
	    core_ptr->device, core_ptr->total_devices,
	    core_ptr->mimetype, core_ptr->total_mimetypes,
	    obj->type,
	    obj->full_path,
	    TRUE, obj->permissions,
	    1,			/* Medium icon size. */
	    &pixmap, &mask,
	    NULL, NULL, NULL, NULL
	);


        /* Identification frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Identification"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Identificacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Identification"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

	/* Name hbox. */
	w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
	/* Icon. */
	if(pixmap != NULL)
	{
	    pw->gen_icon_pm = w = gtk_pixmap_new(pixmap, mask);
            gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
	}
	/* Name label. */
	pw->gen_name_label = w = gtk_label_new(
	    (obj->name != NULL) ? obj->name : ""
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Location hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
	/* Prefix label. */
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Location:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "La Ubicacin:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Emplacement:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Location label. */
        pw->gen_name_label = w = gtk_label_new(
            GetParentDir(obj->full_path)
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);



        /* Time stamps frame. */
	pw->gen_timestamps_frame = w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Time Stamps"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El tiempo Estampa"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Le Temps Timbre"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

	/* Hbox to separate date labels and touch button. */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

	/* Vbox for date labels. */
        w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Last accessed hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
	/* Prefix label. */
	w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
	gtk_widget_set_usize(w, 100, -1);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Last Accessed:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Dure Accesado:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Dernier Accessed:"
#endif
	);
	gtk_container_add(GTK_CONTAINER(parent5), w);
        gtk_widget_show(w);
	/* Date label. */
	pw->gen_date_access_label = w = gtk_label_new(
	    EDVDateFormatString(
		obj->access_time, format, relativity
	    )
	);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Last modified hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Prefix label. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_widget_set_usize(w, 100, -1);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
        parent5 = w;
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Last Modified:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Dure Modificado:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Durer Modifi:"
#endif
	);
        gtk_container_add(GTK_CONTAINER(parent5), w);
        gtk_widget_show(w);
        /* Date label. */
        pw->gen_date_modified_label = w = gtk_label_new(
            EDVDateFormatString(
                obj->modify_time, format, relativity
            )
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Last changed hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Prefix label. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_widget_set_usize(w, 100, -1);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
        parent5 = w;
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Last Changed:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Dure Cambiado:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Durer Chang:"
#endif
	);
        gtk_container_add(GTK_CONTAINER(parent5), w);
        gtk_widget_show(w);
        /* Date label. */
        pw->gen_date_changed_label = w = gtk_label_new(
            EDVDateFormatString(
                obj->change_time, format, relativity
            )
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Alignment for touch button. */
        w = gtk_alignment_new(1.0, 0.0, 0.0, 0.0);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
	parent3 = w;
	/* Touch button. */
	pw->gen_date_touch_btn = w = gtk_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Touch"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "El Toque"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Touche"
#endif
	);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_widget_set_usize(w, -1, PROPWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinDateTouchCB), pw
        );
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
"Update object's time stamps to the current time"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Actualice sellos de tiempo de objeto al tiempo actual"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Met  jour les timbres de temps de l'objet au temps actuel"
#endif
	);
        gtk_widget_show(w);



        /* Ownership frame. */
        pw->gen_ownership_frame = w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Ownership"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "La propiedad"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Possession"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Owner entry hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
        /* Prefix label. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_widget_set_usize(w, 60, -1);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Owner:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Dueo:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Propritaire:"
#endif
	);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
	/* Owner entry. */
	pw->gen_owner_entry = w = gtk_entry_new();
	gtk_entry_set_text(
	    GTK_ENTRY(w),
	    EDVUIDGetNameFromUID(
		core_ptr->uid, core_ptr->total_uids,
		obj->owner_id, NULL
	    )
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
	/* Popup list map button. */
        pw->gen_owner_btn = w = PUListNewMapButton(
	    EDVPropWinOwnerMapPUListCB, pw
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Group entry hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
        /* Prefix label. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_widget_set_usize(w, 60, -1);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Group:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Grupo:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Groupe:"
#endif
	);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        /* Group entry. */
        pw->gen_group_entry = w = gtk_entry_new();
        gtk_entry_set_text(
            GTK_ENTRY(w),
            EDVGIDGetNameFromGID(
                core_ptr->gid, core_ptr->total_gids,
                obj->group_id, NULL
            )
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
	/* Popup list map button. */
	pw->gen_group_btn = w = PUListNewMapButton(
            EDVPropWinGroupMapPUListCB, pw
        );
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Permissions frame. */
	pw->gen_permissions_frame = w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Permissions"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Los Permisos"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Permissions"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

	/* Set permissions frame insensitive for object types that do not
	 * have permissions.
	 */
	if(obj->type == EDV_OBJECT_TYPE_LINK)
	    gtk_widget_set_sensitive(w, FALSE);

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Vbox for standard permission labels and check buttons. */
        w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

	/* Labels hbox. */
	w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	parent4 = w;
	w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Owner"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Dueo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Propritaire"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Group"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Grupo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Groupe"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Other"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Otro"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Autre"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_widget_show(w);

        /* Check buttons hbox. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

	/* Owner check button set hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        parent5 = w;
	/* Read check button. */
	pw->gen_uread_check = w = gtk_check_button_new_with_label("R");
	GTK_TOGGLE_BUTTON(w)->active = (
	    (obj->permissions & EDV_PERMISSION_UREAD) ? TRUE : FALSE
	);
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	/* Write check button. */
        pw->gen_uwrite_check = w = gtk_check_button_new_with_label("W");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_UWRITE) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Execute check button. */
        pw->gen_uexecute_check = w = gtk_check_button_new_with_label("X");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_UEXECUTE) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Group check button set hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        parent5 = w;
        /* Read check button. */
        pw->gen_gread_check = w = gtk_check_button_new_with_label("R");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_GREAD) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Write check button. */
        pw->gen_gwrite_check = w = gtk_check_button_new_with_label("W");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_GWRITE) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Execute check button. */
        pw->gen_gexecute_check = w = gtk_check_button_new_with_label("X");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_GEXECUTE) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Other check button set hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        parent5 = w;
        /* Read check button. */
        pw->gen_aread_check = w = gtk_check_button_new_with_label("R");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_AREAD) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Write check button. */
        pw->gen_awrite_check = w = gtk_check_button_new_with_label("W");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_AWRITE) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Execute check button. */
        pw->gen_aexecute_check = w = gtk_check_button_new_with_label("X");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_AEXECUTE) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);



        /* Hbox for setuid, setgid, and sticky permission check buttons. */
        w = gtk_hbox_new(TRUE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Setuid check button. */
        pw->gen_setuid_check = w = gtk_check_button_new_with_label("SetUID");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_SETUID) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_widget_show(w);

        /* Setgid check button. */
        pw->gen_setgid_check = w = gtk_check_button_new_with_label("SetGID");
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_SETGID) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_widget_show(w);

        /* Sticky check button. */
        pw->gen_sticky_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Sticky"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Pegajoso"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Collant"
#endif
	);
        GTK_TOGGLE_BUTTON(w)->active = (
            (obj->permissions & EDV_PERMISSION_STICKY) ? TRUE : FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_widget_show(w);


}

/*
 *      Creates the link page parented to the given GtkVBox parent
 *      with respect to the given disk object structure values.
 */
static void EDVPropWinCreateLinkPage(
        edv_propwin_struct *pw, GtkWidget *parent,
        edv_core_struct *core_ptr, edv_object_struct *obj
)
{
        gint border_major = 5, border_minor = 2;
        GtkWidget *w, *parent2, *parent3;


        /* Destination frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Destination"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Destino"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Destination"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Destination hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
	/* Label. */
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Path:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Sendero:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Sentier:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	/* Entry. */
	pw->link_dest_entry = w = gtk_entry_new();
	if(obj->linked_to != NULL)
	    gtk_entry_set_text(GTK_ENTRY(w), obj->linked_to);
        gtk_signal_connect(
            GTK_OBJECT(w), "changed",
            GTK_SIGNAL_FUNC(EDVPropWinLinkDestinationChangeCB), pw
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);


	/* Hbox for destination properties. */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

	w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
        parent3 = w;

        w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_widget_show(w);
        parent3 = w;


        pw->link_dest_properties_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_properties2_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Destination Properties...",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Las Propiedades Del Destino...",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Proprits De Destination...",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, -1, PROPWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinDestinationPropertiesCB), pw
        );
        gtk_widget_show(w);


}

/*
 *      Creates the device page parented to the given GtkVBox parent
 *      with respect to the given disk object structure values.
 *
 *	Note that this is for objects who are directories and are mount
 *	paths for the given device. For device nodes see
 *	EDVPropWinCreateDeviceNodePage().
 */
static void EDVPropWinCreateDevicePage(
        edv_propwin_struct *pw, GtkWidget *parent,
        edv_core_struct *core_ptr, edv_object_struct *obj,
	gint dev_num, edv_device_struct *dev_ptr
)
{
        gint i, border_major = 5, border_minor = 2;
	gulong adj_division = 10;
	gchar *strptr;
	GdkColor c;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	GtkAdjustment *adj;
        GtkWidget *w, *parent2, *parent3, *parent4, *parent5;
	pie_chart_struct *pc;
	gchar text[256], text2[256];


        if(obj == NULL)
            return;

	/* Record device reference number. */
	pw->dev_num = dev_num;

        /* Get icon for this device (if any). */
        if(1)
        {
            for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
            {
                if(pixmap != NULL)
                    break;

                pixmap = dev_ptr->medium_pixmap[i];
                mask = dev_ptr->medium_mask[i];
            }
            for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
            {
                if(pixmap != NULL)
                    break;

                pixmap = dev_ptr->large_pixmap[i];
                mask = dev_ptr->large_mask[i];
            }
            for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
            {
                if(pixmap != NULL)
                    break;

                pixmap = dev_ptr->small_pixmap[i];
                mask = dev_ptr->small_mask[i];
            }
        }


        /* Identification frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Identification"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Identificacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Identification"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Device path, mount path, and fs type hbox. */
        w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Icon. */
        if(pixmap != NULL)
        {
	    w = gtk_vbox_new(TRUE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

            pw->dev_icon_pm = w = gtk_pixmap_new(pixmap, mask);
            gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
            gtk_widget_show(w);
        }

        /* Device path and mount path vbox. */
        w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

	/* Device path hbox and label. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent5 = w;

	if(dev_ptr->name != NULL)
	    strptr = g_strdup_printf("%s (%s)",
		dev_ptr->name, 
		(dev_ptr->device_path != NULL) ?
		    dev_ptr->device_path : ""
	    );
	else
	    strptr = g_strdup(
		(dev_ptr->device_path != NULL) ?
		    dev_ptr->device_path : ""
	    );
        pw->dev_device_path_label = w = gtk_label_new(strptr);
	g_free(strptr);
	strptr = NULL;
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Mount path hbox and label. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent5 = w;
	w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Mount Path:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Monte Sendero:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Monter Le Sentier:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        pw->dev_mount_path_label = w = gtk_label_new(
            (dev_ptr->mount_path != NULL) ?
                dev_ptr->mount_path : ""
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Filesystem type hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

	w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "File System Type:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Archive El Tipo De Sistema:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Classer Le Type De Systme:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        w = gtk_label_new(
	    EDVDeviceGetFSStringFromNumber(dev_ptr->fs_type)
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);



	/* Do not continue past this point if device is not mounted or
	 * that it is an infinate capacity device.
	 */
/*
	if(!dev_ptr->is_mounted || (dev_ptr->blocks_total <= 0))
	    return;
 */

        /* Usage frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Usage"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Uso"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Usage"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_vbox_new(FALSE, border_minor);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;


        /* Usage hbox. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Usage pie chart. */
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0, 0.0,
	    (gfloat)(dev_ptr->blocks_total / adj_division),
	    0.0, 0.0, 0.0
	);
        c.red = 0.0 * (guint16)-1;
        c.green = 0.0 * (guint16)-1;
        c.blue = 1.0 * (guint16)-1;
	sprintf(
	    text, "%s: %s kb",
#ifdef PROG_LANGUAGE_ENGLISH
            "Total Capacity",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "La Capacidad Total",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Capacit Totale",
#endif
	    EDVGetObjectSizeStr(core_ptr, dev_ptr->blocks_total)
	);
        sprintf(
	    text2, "%s kb",
	    EDVGetObjectSizeStr(core_ptr, dev_ptr->blocks_free)
	);
	pw->dev_usage_pc = pc = PieChartNew(
	    adj, &c, 110, 70,
	    NULL, text,
#ifdef PROG_LANGUAGE_ENGLISH
            "Free:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Libre:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Libre:",
#endif
	    text2
	);
	gtk_object_unref(GTK_OBJECT(adj));
	w = pc->toplevel;
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);

        c.red = 0.0 * (guint16)-1;
        c.green = 1.0 * (guint16)-1;
        c.blue = 1.0 * (guint16)-1;
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0, 0.0,
            (gfloat)(
		(dev_ptr->blocks_available) / adj_division),
            0.0, 0.0, 0.0
        );
        sprintf(
            text, "%s kb",
	    EDVGetObjectSizeStr(core_ptr, dev_ptr->blocks_available)
        );
        PieChartValueAdd(
	    pc, adj, &c,
#ifdef PROG_LANGUAGE_ENGLISH
            "Free & Available:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Libre & Disponible:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Libre & Disponible:",
#endif
	    text
	);
        gtk_object_unref(GTK_OBJECT(adj));

        c.red = 1.0 * (guint16)-1;
        c.green = 0.0 * (guint16)-1;
        c.blue = 1.0 * (guint16)-1;
        adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0, 0.0,
            (gfloat)(
(dev_ptr->blocks_total - dev_ptr->blocks_free) / adj_division),
            0.0, 0.0, 0.0
        );
        sprintf(
            text, "%s kb",
	    EDVGetObjectSizeStr(
		core_ptr,
		dev_ptr->blocks_total - dev_ptr->blocks_free
	    )
        );
        PieChartValueAdd(
	    pc, adj, &c,
#ifdef PROG_LANGUAGE_ENGLISH
            "Used:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Usado:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Utilis:",
#endif
	    text
	);
        gtk_object_unref(GTK_OBJECT(adj));



        /* Tools frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Maintainance"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Mantenga"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Maintenir"
#endif
        );
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_hbox_new(TRUE, 0);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;


        /* File system check button. */
        pw->dev_check_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_fsck_48x48_xpm
        );
        gtk_widget_set_usize(w, 48 + 2, 48 + 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinRunCheckCB), pw
        );
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
            "File system check"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Archive cheque de sistema"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Classer le chque de systme"
#endif
	);
        gtk_widget_show(w);
        if((dev_ptr->command_check != NULL) ?
            (*dev_ptr->command_check == '\0') : TRUE
	)
            gtk_widget_set_sensitive(w, FALSE);


        /* Tools button. */
        pw->dev_tools_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_tools_48x48_xpm
        );
        gtk_widget_set_usize(w, 48 + 2, 48 + 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinRunToolsCB), pw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Tools"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Las herramientas"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Outils"
#endif
        );
	gtk_widget_show(w);
	if((dev_ptr->command_tools != NULL) ?
	    (*dev_ptr->command_tools == '\0') : TRUE
	)
	    gtk_widget_set_sensitive(w, FALSE);


        /* Format button. */
        pw->dev_format_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_floppy_48x48_xpm
        );
        gtk_widget_set_usize(w, 48 + 2, 48 + 2);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinRunFormatCB), pw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Format media"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Formatear medios"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Met en format le presse"
#endif
        );
        gtk_widget_show(w);
        if((dev_ptr->command_format != NULL) ?
            (*dev_ptr->command_format == '\0') : TRUE
        )
            gtk_widget_set_sensitive(w, FALSE);


}

/*
 *      Creates the device node page parented to the given GtkVBox parent
 *      with respect to the given disk object structure values.
 */
static void EDVPropWinCreateDeviceNodePage(
        edv_propwin_struct *pw, GtkWidget *parent,
        edv_core_struct *core_ptr, edv_object_struct *obj
)
{
        gint border_major = 5, border_minor = 2;
	gint major, minor;
        GtkWidget *w, *parent2, *parent3;
        gchar num_str[80];


	/* Get device node's major and minor numbers. */
	EDVGetDeviceNumbers(
	    (dev_t)obj->device_type, &major, &minor
	);

        /* Device numbers frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Device Numbers"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Artefacto Numeros"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Appareil Numro"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent2 = w;

        w = gtk_hbox_new(FALSE, border_major);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Major number hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
        /* Label. */
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Major:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Mayor:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Majeur:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Entry. */
        pw->dev_node_major_entry = w = gtk_entry_new();
	sprintf(num_str, "%i", major);
	gtk_entry_set_text(GTK_ENTRY(w), num_str);
	gtk_widget_set_usize(w, 80, -1);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Minor number hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
        /* Label. */
        w = gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Minor:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Menor:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Mineur:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Entry. */
        pw->dev_node_minor_entry = w = gtk_entry_new();
        sprintf(num_str, "%i", minor);
        gtk_entry_set_text(GTK_ENTRY(w), num_str);
        gtk_widget_set_usize(w, 80, -1);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);




}


/*
 *      Properties window write protect changed signal callback.
 */
void EDVWriteProtectChangedCB(edv_propwin_struct *pw, gbool state)
{
	gbool sensitive;
	GtkWidget *w;


        if(pw == NULL)
            return;

	sensitive = !state;
        w = pw->gen_date_touch_btn;
        if(w != NULL)
            gtk_widget_set_sensitive(w, sensitive);
	w = pw->ok_btn;
	if(w != NULL)
	    gtk_widget_set_sensitive(w, sensitive);
        w = pw->apply_btn;
        if(w != NULL)
            gtk_widget_set_sensitive(w, sensitive);
}

/*
 *	Properties window reconfigure signal callback.
 */
void EDVPropWinReconfiguredNotifyCB(edv_propwin_struct *pw)
{
	if(pw == NULL)
	    return;

	/* Nothing to update. */
}


/*
 *	Creates a new properties window based on the given disk
 *	object structure.
 */
edv_propwin_struct *EDVPropWinNew(
        gpointer core_ptr, edv_object_struct *obj
)
{
        gint /* border_minor = 2, */ border_major = 5;
	gbool write_protect = FALSE;
	gint dev_num;
	edv_device_struct *dev_ptr;
        GdkWindow *window;
        GtkAccelGroup *accelgrp;
        GtkWidget *w, *parent, *parent2;
	edv_core_struct *c_ptr = (edv_core_struct *)core_ptr;
	edv_propwin_struct *pw = (edv_propwin_struct *)g_malloc0(
	    sizeof(edv_propwin_struct)
	);
	if(pw == NULL)
	    return(pw);

	if(c_ptr == NULL)
	{
	    g_free(pw);
	    pw = NULL;
	    return(pw);
	}

	/* Reset values. */
	pw->initialized = TRUE;
	pw->map_state = FALSE;
	pw->core_ptr = core_ptr;
	pw->path = NULL;


	/* Check if given disk object is valid. */
	if(obj != NULL)
	{
	    /* Record full path. */
	    pw->path = (obj->full_path != NULL) ?
		g_strdup(obj->full_path) : NULL;
	}

	/* Get global write protect state. */
        write_protect = EDVCFGItemListGetValueI(
            c_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT
        );

	/* Refresh all device mount states and device stats, then get
	 * device path or mount path that matches the given disk
	 * object's path.
	 */
	EDVDevicesListUpdateMountStates(
	    c_ptr->device, c_ptr->total_devices
	);
	EDVDevicesListUpdateStats(
	    c_ptr->device, c_ptr->total_devices
        );
	dev_ptr = EDVeviceMatchListByMountPath(
	   c_ptr->device, c_ptr->total_devices,
	   &dev_num, pw->path
	);
	if(dev_ptr == NULL)
	   dev_ptr = EDVDeviceMatchListByDevicePath(
		c_ptr->device, c_ptr->total_devices,
		&dev_num, pw->path
	   );


        /* Keyboard accelerator group. */
        pw->accelgrp = accelgrp = gtk_accel_group_new();

        /* Toplevel. */
	pw->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
        gtk_widget_set_usize(w, PROPWIN_WIDTH, PROPWIN_HEIGHT);
        gtk_widget_realize(w);
        gtk_window_set_title(GTK_WINDOW(w), "Properties");
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
            GUISetWMIcon(window, (u_int8_t **)icon_properties2_48x48_xpm);
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(EDVPropWinDeleteEventCB), pw
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;


        /* Main vbox. */
        w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


        /* Vbox for main notebook. */
        w = gtk_vbox_new(FALSE, 0);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent2 = w;

        /* Main notebook. */
        pw->notebook = w = gtk_notebook_new();
        gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
        gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
        gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
/*      gtk_notebook_set_page(GTK_NOTEBOOK(w), 0); */
        gtk_signal_connect(
            GTK_OBJECT(w), "switch_page",
            GTK_SIGNAL_FUNC(EDVPropWinSwitchPageCB), pw
        );
        gtk_widget_show(w);
        parent2 = w;


	/* Begin creating pages on notebook. */

	/* General page. */
        w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_notebook_append_page(
            GTK_NOTEBOOK(parent2), w, gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
		"General"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "General"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Gnral"
#endif
	    )
        );
        gtk_widget_show(w);
	EDVPropWinCreateGeneralPage(
	    pw, w, c_ptr, obj
	);


	/* Check if object is a link. */
	if((obj != NULL) ? (obj->type == EDV_OBJECT_TYPE_LINK) : FALSE)
	{
            /* Link page. */
            w = gtk_vbox_new(FALSE, border_major);
            gtk_container_border_width(GTK_CONTAINER(w), border_major);
            gtk_notebook_append_page(
                GTK_NOTEBOOK(parent2), w, gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
		    "Link"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		    "El Eslabn"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		    "Lien"
#endif
		)
            );
            gtk_widget_show(w);
            EDVPropWinCreateLinkPage(
                pw, w, c_ptr, obj
            );
	}


        /* Check if a device structure for this object was matched. */
	if(dev_ptr != NULL)
	{
	    /* Devices page. */
            w = gtk_vbox_new(FALSE, border_major);
            gtk_container_border_width(GTK_CONTAINER(w), border_major);
            gtk_notebook_append_page(
                GTK_NOTEBOOK(parent2), w, gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
                    "Device"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                    "El Artefacto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                    "Appareil"
#endif
		)
            );
            gtk_widget_show(w);
            EDVPropWinCreateDevicePage(
                pw, w, c_ptr, obj, dev_num, dev_ptr
            );
	}


	/* Check if object is a device node. */
        if((obj != NULL) ?
	    ((obj->type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
             (obj->type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)) : FALSE
	)
        {
            /* Device node page. */
            w = gtk_vbox_new(FALSE, border_major);
            gtk_container_border_width(GTK_CONTAINER(w), border_major);
            gtk_notebook_append_page(
                GTK_NOTEBOOK(parent2), w, gtk_label_new(
#ifdef PROG_LANGUAGE_ENGLISH
                    "Device Node"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                    "El Artefacto Puerto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                    "Appareil Port"
#endif
		)
            );
            gtk_widget_show(w);
            EDVPropWinCreateDeviceNodePage(
                pw, w, c_ptr, obj
            );
        }






        /* Separator. */
        w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Buttons hbox. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* OK button. */
        pw->ok_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_ok_20x20_xpm, "OK", NULL
        );
        gtk_widget_set_usize(w, PROPWIN_BTN_WIDTH, PROPWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinOKCB), pw
        );
        gtk_accel_group_add(
            accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_3270_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_KP_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_ISO_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);

        /* Apply button. */
        pw->apply_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_select_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Apply",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Aplique",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "S'appliquer",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, PROPWIN_BTN_WIDTH, PROPWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinApplyCB), pw
        );
        gtk_widget_show(w);

        /* Cancel button. */
        pw->cancel_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Cancel",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cancele",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Annuler",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, PROPWIN_BTN_WIDTH, PROPWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVPropWinCancelCB), pw
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);



	/* Emit a synthetic write protect changed signal to update
	 * widgets sensitivity to the global write protect state.
	 */
	EDVWriteProtectChangedCB(pw, write_protect);




	return(pw);
}


/*
 *	Maps the properties window.
 */
void EDVPropWinMap(edv_propwin_struct *pw)
{
        GtkWidget *w;

        if(pw == NULL)
            return;

        w = pw->toplevel;
        if(w == NULL)
            return;

        if(!pw->map_state)
        {
            gtk_widget_show_raise(w);
            pw->map_state = TRUE;
        }

        w = pw->ok_btn;
        if(w != NULL)
        {
            gtk_widget_grab_focus(w);
            gtk_widget_grab_default(w);
        }
}

/*
 *	Unmaps the properties window.
 */
void EDVPropWinUnmap(edv_propwin_struct *pw)
{
        GtkWidget *w;

        if(pw == NULL)
            return;

        w = pw->toplevel;
        if(w == NULL)
            return;

        if(pw->map_state)
        {
            gtk_widget_hide(w);
            pw->map_state = FALSE;
        }
}

/*
 *	Deallocates the properties window and all its resources.
 */
void EDVPropWinDelete(edv_propwin_struct *pw)
{
        GtkWidget **w;


	if(pw == NULL)
	    return;

        if(pw->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Destroy widgets on general page. */
            w = &pw->gen_icon_pm;
            DO_DESTROY_WIDGET
            w = &pw->gen_name_label;
            DO_DESTROY_WIDGET
            w = &pw->gen_location_label;
            DO_DESTROY_WIDGET

	    pw->gen_timestamps_frame = NULL;
            w = &pw->gen_date_access_label;
            DO_DESTROY_WIDGET
            w = &pw->gen_date_modified_label;
            DO_DESTROY_WIDGET
            w = &pw->gen_date_changed_label;
            DO_DESTROY_WIDGET
	    w = &pw->gen_date_touch_btn;
            DO_DESTROY_WIDGET

	    pw->gen_ownership_frame = NULL;
            w = &pw->gen_owner_entry;
            DO_DESTROY_WIDGET
            w = &pw->gen_owner_btn;
            DO_DESTROY_WIDGET
            w = &pw->gen_group_entry;
            DO_DESTROY_WIDGET
            w = &pw->gen_group_btn;
            DO_DESTROY_WIDGET

	    pw->gen_permissions_frame = NULL;
            w = &pw->gen_uread_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_uwrite_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_uexecute_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_gread_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_gwrite_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_gexecute_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_aread_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_awrite_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_aexecute_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_setuid_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_setgid_check;
            DO_DESTROY_WIDGET
            w = &pw->gen_sticky_check;
            DO_DESTROY_WIDGET


	    /* Destroy widgets on link page. */
	    w = &pw->link_dest_entry;
	    DO_DESTROY_WIDGET
            w = &pw->link_check_btn;
            DO_DESTROY_WIDGET
            w = &pw->link_dest_properties_btn;
            DO_DESTROY_WIDGET


	    /* Destroy widgets on device page. */
            w = &pw->dev_icon_pm;
            DO_DESTROY_WIDGET
            w = &pw->dev_device_path_label;
            DO_DESTROY_WIDGET
            w = &pw->dev_mount_path_label;
            DO_DESTROY_WIDGET
            w = &pw->dev_fs_type_label;
            DO_DESTROY_WIDGET
	    PieChartDelete(pw->dev_usage_pc);
	    pw->dev_usage_pc = NULL;
	    w = &pw->dev_check_btn;
            DO_DESTROY_WIDGET
            w = &pw->dev_tools_btn;
            DO_DESTROY_WIDGET
            w = &pw->dev_format_btn;
            DO_DESTROY_WIDGET

	    /* Destroy widgets on device node page. */
	    w = &pw->dev_node_major_entry;
            DO_DESTROY_WIDGET
            w = &pw->dev_node_minor_entry;
            DO_DESTROY_WIDGET


	    /* Destroy all other widgets. */
            w = &pw->notebook;
            DO_DESTROY_WIDGET

            w = &pw->ok_btn;
            DO_DESTROY_WIDGET
            w = &pw->apply_btn;
            DO_DESTROY_WIDGET
            w = &pw->cancel_btn;
            DO_DESTROY_WIDGET

            w = &pw->toplevel;
            DO_DESTROY_WIDGET


            if(pw->accelgrp != NULL)
            {
                gtk_accel_group_unref(pw->accelgrp);
                pw->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
        }

	/* Deallocate full path to disk object. */
	g_free(pw->path);
	pw->path = NULL;

        /* Deallocate structure itself. */
	g_free(pw);
}
