#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "piechart.h"


static void PieChartDrawArc(
        GdkPixmap *pixmap, GdkGC *gc,
        gint x, gint y, gint width, gint height,
        gfloat start_angle,     /* 0.0 to 1.0. */
        gfloat end_angle,       /* 0.0 to 1.0. */
        gbool outline_circumferance,
        gbool outline_radius, gbool outline_height
);
static void PieChartDrawPieIterate(
        pie_chart_struct *pc, GtkStyle *style, GdkPixmap *pixmap,
        gint state, gint x, gint y,
        gint width, gint height,
        gbool outline_circumference, gbool outline_radius,
        gbool outline_height,
        gbool shaded, gbool shadow
);
static void PieChartCreateValueLabel(
	pie_chart_struct *pc,
	GdkColor *c, const gchar *type_label, const gchar *value_label,
	GtkWidget **type_label_hbox_rtn, GtkWidget **value_label_hbox_rtn,
        GtkWidget **drawing_area_rtn
);
static void PieChartDoRealize(pie_chart_struct *pc);
static void PieChartRedrawPixmap(pie_chart_struct *pc);
static gint PieChartValueLabelExposeCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
static gint PieChartExposeCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
);

gint PieChartValueAdd(
        pie_chart_struct *pc,
	GtkAdjustment *adj, GdkColor *c,
	const gchar *type_label, const gchar *value_label
);
void PieChartValueSet(
        pie_chart_struct *pc, gint value_num,
	GtkAdjustment *adj, GdkColor *c,
        const gchar *type_label, const gchar *value_label
);
void PieChartValueRemove(
        pie_chart_struct *pc, gint value_num
);
void PieChartValueRemoveAll(pie_chart_struct *pc);

pie_chart_struct *PieChartNew(
	GtkAdjustment *adj, GdkColor *c, gint width, gint height,
	const gchar *title, const gchar *footer,
        const gchar *base_type_label, const gchar *base_value_label
);
void PieChartMap(pie_chart_struct *pc);
void PieChartUnmap(pie_chart_struct *pc);
void PieChartDelete(pie_chart_struct *pc);


#ifndef PI
# define PI     3.14159265
#endif

#define PIE_CHART_PIE_HEIGHT	10


/*
 *	Draws a segment of a pie arc.
 *
 *	Angles are in units from 0.0 to 1.0 starting at the 12 o'clock
 *	position being 0.0.
 */
static void PieChartDrawArc(
	GdkPixmap *pixmap, GdkGC *gc,
	gint x, gint y, gint width, gint height,
	gfloat start_angle,	/* 0.0 to 1.0. */
	gfloat end_angle,	/* 0.0 to 1.0. */
	gbool outline_circumferance,
	gbool outline_radius, gbool outline_height
)
{
	gfloat delta_angle;


	if((pixmap == NULL) || (gc == NULL))
	    return;

	if(start_angle == end_angle)
	    return;


	/* Outline radius? */
	if(outline_radius &&
	   ((start_angle != 0.0) || (end_angle != 1.0))
	)
	{
	    gfloat	ostart_angle = start_angle,
			oend_angle = end_angle;
	    gint	rx = (width / 2),
			ry = (height / 2),
			cx = x + rx,
			cy = y + ry;

            /* Sanitize outline start and end angles. */
            while(ostart_angle > 1.0)
                ostart_angle -= 1.0;
            while(ostart_angle < 0.0)
                ostart_angle += 1.0;

            while(oend_angle > 1.0)
                oend_angle -= 1.0;
            while(oend_angle < 0.0)
                oend_angle += 1.0;

	    gdk_draw_line(
		(GdkDrawable *)pixmap, gc,
		cx, cy,
		cx + (rx * sin(ostart_angle * 2.0 * PI)),
		cy + (ry * -cos(ostart_angle * 2.0 * PI))
	    );
	    gdk_draw_line(
		(GdkDrawable *)pixmap, gc,
		cx, cy,
		cx + (rx * sin(oend_angle * 2.0 * PI)),
		cy + (ry * -cos(oend_angle * 2.0 * PI))
	    );
	}

	/* Outline height? */
	if(outline_height)
	{
            gfloat      ostart_angle = start_angle,
                        oend_angle = end_angle;
            gint        rx = (width / 2),
                        ry = (height / 2),
                        cx = x + rx,
                        cy = y + ry;

            /* Sanitize outline start and end angles. */
            while(ostart_angle > 1.0)
                ostart_angle -= 1.0;
            while(ostart_angle < 0.0)
                ostart_angle += 1.0;

            while(oend_angle > 1.0)
                oend_angle -= 1.0;
            while(oend_angle < 0.0)
                oend_angle += 1.0;

	    /* Draw height lines dividing pie segments. */
	    if((ostart_angle > 0.25) && (ostart_angle < 0.75))
		gdk_draw_line(
                    (GdkDrawable *)pixmap, gc,
                    cx + (rx * sin(ostart_angle * 2.0 * PI)),
                    cy + (ry * -cos(ostart_angle * 2.0 * PI)),
                    cx + (rx * sin(ostart_angle * 2.0 * PI)),
                    cy + (ry * -cos(ostart_angle * 2.0 * PI)) +
			PIE_CHART_PIE_HEIGHT
                );
	    if((oend_angle > 0.25) && (oend_angle < 0.75))
                gdk_draw_line(
                    (GdkDrawable *)pixmap, gc,
                    cx + (rx * sin(oend_angle * 2.0 * PI)),
                    cy + (ry * -cos(oend_angle * 2.0 * PI)),
                    cx + (rx * sin(oend_angle * 2.0 * PI)),
                    cy + (ry * -cos(oend_angle * 2.0 * PI)) +
			PIE_CHART_PIE_HEIGHT
                );

	    /* Draw height lines at far left and right edges. */
	    gdk_draw_line(
		(GdkDrawable *)pixmap, gc,
		x, y + ry,
		x, y + ry + PIE_CHART_PIE_HEIGHT - 1
	    );
	    gdk_draw_line(
		(GdkDrawable *)pixmap, gc,
		x + width, y + ry,
		x + width, y + ry + PIE_CHART_PIE_HEIGHT - 1
	    );
	}

	/* Calculate delta angle. */
	delta_angle = start_angle - end_angle;

	/* Convert angles to degrees and make it relative to the 3 o'clock
	 * position.
	 */
	start_angle = 90.0 - (start_angle * 360.0);
	while(start_angle < 0.0)
	    start_angle += 360.0;
	while(start_angle >= 360.0)
	    start_angle -= 360.0;

        delta_angle = (delta_angle * 360.0);

	gdk_draw_arc(
	    (GdkDrawable *)pixmap, gc, !outline_circumferance,
	    x, y, width, height,
	    (gint)(start_angle * 64.0),
	    (gint)(delta_angle * 64.0)
	);
}

/*
 *	Creates a new value label and parents it to the given pie chart's
 *	value_labels_vbox.
 *
 *	Returns the pointer to the hbox containing the new label.
 */
static void PieChartCreateValueLabel(
	pie_chart_struct *pc,
	GdkColor *c, const gchar *type_label, const gchar *value_label,
	GtkWidget **type_label_hbox_rtn, GtkWidget **value_label_hbox_rtn,
	GtkWidget **drawing_area_rtn
)
{
	gint border_minor = 2, da_width = 10, da_height = 10;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *type_labels_vbox, *value_labels_vbox;
	


	if(type_label_hbox_rtn != NULL)
	    *type_label_hbox_rtn = NULL;
	if(value_label_hbox_rtn != NULL)
	    *value_label_hbox_rtn = NULL;
	if(drawing_area_rtn != NULL)
	    *drawing_area_rtn = NULL;

	if((pc == NULL) || (type_label == NULL))
	    return;

	type_labels_vbox = parent = pc->type_labels_vbox;
	if(parent == NULL)
	    return;

	/* Create type label hbox. */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;
	if(type_label_hbox_rtn != NULL)
	    *type_label_hbox_rtn = w;

	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Create drawing area to display color. */
	w = gtk_drawing_area_new();
        gtk_widget_set_usize(w, da_width, da_height);
        gtk_drawing_area_size(GTK_DRAWING_AREA(w), da_width, da_height);
        gtk_widget_add_events(w, GDK_EXPOSURE_MASK);
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(PieChartValueLabelExposeCB), pc
        );
	gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_widget_show(w);
	if(drawing_area_rtn != NULL)
	    *drawing_area_rtn = w;

	/* Create a new label widget based on the given label string. */
	w = gtk_label_new(type_label);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Create value label? */
        value_labels_vbox = parent = pc->value_labels_vbox;
	if((value_label != NULL) && (parent != NULL))
	{
            w = gtk_hbox_new(FALSE, border_minor);
            gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent2 = w;
            if(value_label_hbox_rtn != NULL)
                *value_label_hbox_rtn = w;

	    w = gtk_alignment_new(0.0, 0.5, 0.0, 0.0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

            w = gtk_label_new(value_label);
	    gtk_container_add(GTK_CONTAINER(parent2), w);
            gtk_widget_show(w);
	}
}

/*
 *	Checks if the given pie chart is not realized, if it is not
 *	realized then all colors and pixmap will be created and the pie
 *	chart will be marked as realized.
 */
static void PieChartDoRealize(pie_chart_struct *pc)
{
	gint i, width, height;
	GdkColor *c;
	GdkColormap *colormap;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GtkWidget *w;
	pie_chart_value_struct *value_ptr;


	if(pc == NULL)
	    return;

	/* Already realized? */
	if(pc->realized)
	    return;

	/* Get drawing area widget. */
	w = pc->drawing_area;
	if(w == NULL)
	    return;
	if(!GTK_WIDGET_REALIZED(w))
	    return;

	window = w->window;
	if(window == NULL)
	    return;

	gdk_window_get_size(window, &width, &height);


	/* Get colormap. */
	colormap = pc->colormap;
	if(colormap == NULL)
	{
	    pc->colormap = colormap = gtk_widget_get_colormap(w);
	    if(colormap != NULL)
		gdk_colormap_ref(colormap);
	}
	if(colormap == NULL)
	{
	    pc->colormap = colormap = gtk_widget_get_default_colormap();
            if(colormap != NULL)
                gdk_colormap_ref(colormap);
	}
	if(colormap == NULL)
	    return;


	/* Create graphics context. */
	if(pc->gc == NULL)
	{
	    pc->gc = gdk_gc_new(window);
	}


	/* Create off screen buffer pixmap for drawing area. */
	pixmap = pc->pixmap;
	if((pixmap == NULL) && (width > 0) && (height > 0))
	    pc->pixmap = pixmap = gdk_pixmap_new(
		window, width, height, -1
	    );

	/* Set pie base colors. */
	c = &pc->c;
	gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
        c = &pc->c_shade;
        gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
        c = &pc->c_shadow;
        gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
	/* Create base value label. */
	PieChartCreateValueLabel(
	    pc, &pc->c, pc->base_type_label, pc->base_value_label,
	    NULL, NULL, NULL
	);

	/* Realize all values. */
	for(i = 0; i < pc->total_values; i++)
	{
	    value_ptr = pc->value[i];
	    if(value_ptr == NULL)
		continue;

	    /* Allocate color of this value. */
	    c = &value_ptr->c;
	    gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
            c = &value_ptr->c_shade;
            gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);

	    /* Create label widget for this value. */
	    PieChartCreateValueLabel(
		pc, &value_ptr->c, value_ptr->type_label, value_ptr->value_label,
		&value_ptr->type_label_hbox, &value_ptr->value_label_hbox,
		&value_ptr->drawing_area
	    );
	}


	/* Mark as realized. */
	pc->realized = TRUE;
}


/*
 *	Draws one itteration of a complete pie disk at the given
 *	coordinates.
 */
static void PieChartDrawPieIterate(
	pie_chart_struct *pc, GtkStyle *style, GdkPixmap *pixmap,
	gint state, gint x, gint y,
	gint width, gint height,
	gbool outline_circumference, gbool outline_radius,
	gbool outline_height,
	gbool shaded, gbool shadow
)
{
	gint i;
	gbool outline = (outline_circumference || outline_radius || outline_height) ?
	    TRUE : FALSE;
	gfloat values_range, total_range, values_range_coeff;
	GdkGC *gc;
	GtkAdjustment *adj;
	pie_chart_value_struct *value_ptr;


	if((pc == NULL) || (style == NULL) || (pixmap == NULL) ||
           (width < 1) || (height < 1)
	)
	    return;

	if(outline)
	    gc = style->fg_gc[state];
	else
	    gc = pc->gc;
	if(gc == NULL)
	    return;

	/* Set foreground color only if not drawing for an outline. */
	if(!outline)
	{
	    if(shaded)
		gdk_gc_set_foreground(gc, &pc->c_shade);
	    else if(shadow)
		gdk_gc_set_foreground(gc, &pc->c_shadow);
	    else
		gdk_gc_set_foreground(gc, &pc->c);
            gdk_gc_set_fill(gc, GDK_SOLID);
	}

	/* Calculate total range of all values. */
	values_range = 0.0;
	for(i = 0; i < pc->total_values; i++)
	{
	    value_ptr = pc->value[i];
	    if(value_ptr == NULL)
		continue;

	    adj = value_ptr->adj;
	    values_range += (adj != NULL) ? (adj->upper - adj->lower) : 0.0;
	}

	/* Calculate absolute total range. */
	adj = pc->adj;
	total_range = (adj != NULL) ? (adj->upper - adj->lower) : 0.0;

	/* Check if more of the values range is visible on the `lower'
	 * side.
	 *
	 * Note that angles are in units from 0.0 to 1.0 starting at the
	 * 12 o'clock position being 0.0 going clockwise.
	 */
	values_range_coeff = (total_range > 0.0) ?
	    (values_range / total_range) : 0.0;
	if(values_range_coeff < 0.5)
	{
	    gfloat value_pos_coeff = 0.5 + (values_range_coeff / 2.0);
	    gfloat value_pos_delta;


	    /* Draw base disk. */
	    PieChartDrawArc(
		pixmap, gc, x, y, width, height,
		0.0, 1.0,
		outline_circumference, outline_radius,
		outline_height
	    );
	    /* Do not draw beyond this point if this disk is to be a
	     * shadow.
	     */
	    if(shadow)
		return;

	    /* Draw each value. */
	    for(i = 0; i < pc->total_values; i++)
	    {
                value_ptr = pc->value[i];
                if(value_ptr == NULL)
                    continue;

		adj = value_ptr->adj;
		if((adj != NULL) && (total_range > 0.0))
		    value_pos_delta = (adj->upper - adj->lower) /
			total_range;
		else
		    value_pos_delta = 0.0;

		if(!outline)
		{
		    if(shaded)
			gdk_gc_set_foreground(gc, &value_ptr->c_shade);
		    else
			gdk_gc_set_foreground(gc, &value_ptr->c);
		    gdk_gc_set_fill(gc, GDK_SOLID);
		}

		PieChartDrawArc(
		    pixmap, gc, x, y, width, height,
		    value_pos_coeff, value_pos_coeff - value_pos_delta,
                    outline_circumference, outline_radius,
                    outline_height
		);

		value_pos_coeff -= value_pos_delta;
	    }
	}
	else
	{
            gfloat value_pos_coeff = 1.0 - (values_range_coeff / 2.0);
            gfloat value_pos_delta;


            /* Draw base disk. */
            PieChartDrawArc(
                pixmap, gc, x, y, width, height,
		0.0, 1.0,
                outline_circumference, outline_radius,
                outline_height
            );
            /* Do not draw beyond this point if this disk is to be a
             * shadow.
             */
            if(shadow)
                return;

            /* Draw each value. */
            for(i = 0; i < pc->total_values; i++)
            {
                value_ptr = pc->value[i];
                if(value_ptr == NULL)
                    continue;

                adj = value_ptr->adj;
                if((adj != NULL) && (total_range > 0.0))
                    value_pos_delta = (adj->upper - adj->lower) /
                        total_range;
                else
                    value_pos_delta = 0.0;

                if(!outline)
                {
                    if(shaded)
                        gdk_gc_set_foreground(gc, &value_ptr->c_shade);
                    else
                        gdk_gc_set_foreground(gc, &value_ptr->c);
		    gdk_gc_set_fill(gc, GDK_SOLID);
                }

                PieChartDrawArc(
                    pixmap, gc, x, y, width, height,
                    value_pos_coeff, value_pos_coeff + value_pos_delta,
                    outline_circumference, outline_radius,
                    outline_height
                );

                value_pos_coeff += value_pos_delta;
            }
	}
}

/*
 *	Redraws the off screen buffer pixmap on the given pie chart.
 *
 *	The pie chart will be realized as needed.
 */
static void PieChartRedrawPixmap(pie_chart_struct *pc)
{
	gint state, x, y, width, height, pwidth, pheight;
        GdkGC *gc;
	GdkWindow *window;
	GdkPixmap *pixmap, *bg_pixmap;
        GtkStyle *style;
	GtkWidget *w;


	if(pc == NULL)
	    return;

	/* Get drawing area widget and its style. */
	w = pc->drawing_area;
	if(w == NULL)
	    return;
	window = w->window;
        style = gtk_widget_get_style(w);
        if((window == NULL) || (style == NULL))
            return;

	/* Drawing area widget not realized yet? */
	if(!GTK_WIDGET_REALIZED(w))
	    return;

	if(!pc->realized)
	    PieChartDoRealize(pc);

	/* Get offscreen buffer pixmap that we will be drawing to,
	 * this pixmap should be created in PieChartDoRealize().
	 */
	pixmap = pc->pixmap;
	if(pixmap == NULL)
	    return;

	state = GTK_WIDGET_STATE(w);
	gdk_window_get_size((GdkWindow *)pixmap, &width, &height);


	/* Get graphics context. */
	gc = pc->gc;
        bg_pixmap = style->bg_pixmap[state];
	if(gc == NULL)
	    return;


        /* Begin drawing. */

        /* Set background pixmap if the style has one. */
        if(bg_pixmap != NULL)
        {
	    gint tx, ty;

	    gdk_gc_set_tile(gc, bg_pixmap);
            gdk_gc_set_fill(gc, GDK_TILED);
	    gdk_window_get_position(window, &tx, &ty);
            gdk_gc_set_ts_origin(gc, -tx, -ty);
        }
	else
	{
            gdk_gc_set_fill(gc, GDK_SOLID);
	}
        gdk_gc_set_clip_mask(gc, NULL);

        /* Draw background to cover entire pixmap buffer. */
	gdk_gc_set_foreground(gc, &style->bg[state]);
        gdk_draw_rectangle(
            (GdkDrawable *)pixmap, gc, TRUE,
            0, 0, width, height
        );

	gdk_gc_set_fill(gc, GDK_SOLID);
        gdk_gc_set_ts_origin(gc, 0, 0);


	/* Begin drawing pie chart. */

	/* Shadow. */
	x = 4;
	y = PIE_CHART_PIE_HEIGHT + 2;
	pwidth = width - 6;
	pheight = height - PIE_CHART_PIE_HEIGHT - 4;
        if(pc->shadows)
            PieChartDrawPieIterate(
                pc, style, pixmap,
                state, x, y, pwidth, pheight,
                FALSE,		/* Outline circumference. */
                FALSE, FALSE,	/* Outline radius and height. */
                FALSE,		/* Is shaded. */
                TRUE		/* Is shadow. */
            );

	/* Base outline. */
	x = 0;
	y = PIE_CHART_PIE_HEIGHT;
	if(pc->outline)
	{
	    PieChartDrawPieIterate(
		pc, style, pixmap,
		state, x, y, pwidth, pheight,
                FALSE,		/* Outline circumference. */
                FALSE, FALSE,	/* Outline radius and height. */
                TRUE,		/* Is shaded. */
                FALSE		/* Is shadow. */
            );
            PieChartDrawPieIterate(
                pc, style, pixmap,
                state, x, y, pwidth, pheight,
                TRUE,		/* Outline circumference. */
                FALSE, FALSE,	/* Outline radius and height. */
                TRUE,		/* Is shaded. */
                FALSE		/* Is shadow. */
            );
	    y--;
	}

	while(y > 0)
	{
            PieChartDrawPieIterate(
                pc, style, pixmap,
                state, x, y, pwidth, pheight,
                FALSE, 		/* Outline circumference. */
                FALSE, FALSE,	/* Outline radius and height. */
                TRUE,		/* Is shaded. */
                FALSE		/* Is shadow. */
            );
	    y--;
	}

	y = 0;
        PieChartDrawPieIterate(
            pc, style, pixmap,
            state, x, y, pwidth, pheight,
	    FALSE,		/* Outline circumference. */
	    FALSE, FALSE,	/* Outline radius and height. */
	    FALSE,		/* Is shaded. */
	    FALSE		/* Is shadow. */
        );

        if(pc->outline)
            PieChartDrawPieIterate(
                pc, style, pixmap,
                state, x, y, pwidth, pheight,
                TRUE,		/* Outline circumference. */
		TRUE, TRUE,	/* Outline radius and height. */
                FALSE,		/* Is shaded. */
                FALSE		/* Is shadow. */
            );


}

/*
 *	Value label drawing area "expose_event" signal callback.
 */
static gint PieChartValueLabelExposeCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	gint i, state, width, height;
	GdkColor *c_bg;
	GdkGC *gc;
	GdkWindow *window;
	GtkStyle *style;
	pie_chart_value_struct *value_ptr;
	pie_chart_struct *pc = PIE_CHART(data);
	if((widget == NULL) || (pc == NULL))
	    return(FALSE);


	/* Find value structure that contains this drawing area. */
	value_ptr = NULL;
	for(i = 0; i < pc->total_values; i++)
	{
	    value_ptr = pc->value[i];
	    if(value_ptr == NULL)
		continue;

	    if(value_ptr->drawing_area == widget)
		break;
	}
	if(i >= pc->total_values)
	    value_ptr = NULL;
	if(value_ptr != NULL)
	{
	    c_bg = &value_ptr->c;
	}
	else
	{
	    c_bg = &pc->c;
	}


	state = GTK_WIDGET_STATE(widget);
	window = widget->window;
	style = gtk_widget_get_style(widget);
	if((window == NULL) || (style == NULL))
	    return(FALSE);


	gdk_window_get_size(window, &width, &height);
	
	gc = pc->gc;
	if(gc != NULL)
	{
	    gdk_gc_set_foreground(gc, c_bg);
            gdk_gc_set_fill(gc, GDK_SOLID);
	    gdk_draw_rectangle(
		(GdkDrawable *)window, gc, TRUE,
		0, 0, width, height
	    );
	}
/*
	gc = style->fg_gc[state];
	if(gc != NULL)
	{
            gdk_draw_rectangle(
                (GdkDrawable *)window, gc, FALSE,
                0, 0, width - 1, height - 1
            );
	}
 */

	return(TRUE);
}

/*
 *	Drawing area "expose_event" signal callback.
 */
static gint PieChartExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	gint state, width, height;
	GdkGC *gc;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GtkStyle *style;
	pie_chart_struct *pc = PIE_CHART(data);
	if((widget == NULL) || (pc == NULL))
	    return(FALSE);

	window = widget->window;
	if(window == NULL)
	    return(FALSE);

	style = gtk_widget_get_style(widget);
	if(style == NULL)
	    return(FALSE);

	state = GTK_WIDGET_STATE(widget);


	/* Get pixmap for drawing area, if the pixmap is NULL then
	 * it implies it has not been realized or redrawn, so realize
	 * and redraw it as needed.
	 */
	pixmap = pc->pixmap;
	if(pixmap == NULL)
	{
	    PieChartRedrawPixmap(pc);
	    pixmap = pc->pixmap;
	}
	if(pixmap == NULL)
	    return(TRUE);

        gdk_window_get_size((GdkWindow *)pixmap, &width, &height);

	gc = style->bg_gc[state];
	if(gc != NULL)
	    gdk_window_copy_area(
		window, gc,
		0, 0,
		(GdkWindow *)pixmap,
		0, 0,
		width, height
	    );

	return(TRUE);
}

/*
 *	Adds a new value to the pie chart, returns an index id to that
 *	value or -1 on failure.
 */
gint PieChartValueAdd(
	pie_chart_struct *pc,
	GtkAdjustment *adj, GdkColor *c,
        const gchar *type_label, const gchar *value_label
)
{
	int i;
	GdkColormap *colormap;
	pie_chart_value_struct *value_ptr;


	if((pc == NULL) || (adj == NULL) || (c == NULL))
	    return(-1);

	/* Sanitize total. */
	if(pc->total_values < 0)
	    pc->total_values = 0;

	/* Allocate more pointers. */
	i = pc->total_values;
	pc->total_values = i + 1;
	pc->value = (pie_chart_value_struct **)g_realloc(
	    pc->value,
	    pc->total_values * sizeof(pie_chart_value_struct *)
	);
	if(pc->value == NULL)
	{
	    pc->total_values = 0;
	    return(-1);
	}

	/* Allocate a new value structure. */
	pc->value[i] = value_ptr = (pie_chart_value_struct *)g_malloc0(
	    sizeof(pie_chart_value_struct)
	);
	if(value_ptr != NULL)
	{
	    GdkColor *tc;


	    /* Add a refcount to the given adjustment and assign it. */
	    gtk_object_ref(GTK_OBJECT(adj));
	    value_ptr->adj = adj;


	    /* Set given color, allocate it if the colormap is set. */
	    tc = &value_ptr->c;
	    memcpy(tc, c, sizeof(GdkColor));

            tc = &value_ptr->c_shade;
            tc->red = c->red / 2;
            tc->green = c->green / 2;
            tc->blue = c->blue / 2;

	    colormap = pc->colormap;
	    if(colormap != NULL)
	    {
		gdk_colormap_alloc_color(
		    colormap, &value_ptr->c, TRUE, TRUE
		);
                gdk_colormap_alloc_color(
		    colormap, &value_ptr->c_shade, TRUE, TRUE
		);
	    }


	    value_ptr->type_label = (type_label != NULL) ?
		g_strdup(type_label) : NULL;
            value_ptr->value_label = (value_label != NULL) ?
                g_strdup(value_label) : NULL;

	    return(i);
	}
	else
	{
	    return(-1);
	}
}

/*
 *	Sets the value specified by index value_num on the given pie 
 *	chart.
 */
void PieChartValueSet(
        pie_chart_struct *pc, gint value_num,
	GtkAdjustment *adj, GdkColor *c,
        const gchar *type_label, const gchar *value_label
)
{
	GdkColormap *colormap;
        pie_chart_value_struct *value_ptr;


	if((pc == NULL) || (adj == NULL) || (c == NULL))
	    return;

	if((value_num < 0) || (value_num >= pc->total_values))
	    return;

	value_ptr = pc->value[value_num];
	if(value_ptr == NULL)
	    return;

	/* Add refcount to given adjustment, remove refcount from existing
	 * adjustment and assign given adjustment.
	 */
	gtk_object_ref(GTK_OBJECT(adj));
	if(value_ptr->adj != NULL)
	    gtk_object_unref(GTK_OBJECT(value_ptr->adj));
	value_ptr->adj = adj;

	/* Deallocate old color if colormap is available, set new color
	 * and allocate it if colormap is available.
	 */
	colormap = pc->colormap;
	if(colormap != NULL)
	{
	    GdkColor *tc;

	    tc = &value_ptr->c;
	    gdk_colormap_free_colors(colormap, tc, 1);
	    memcpy(tc, c, sizeof(GdkColor));
	    gdk_colormap_alloc_color(colormap, tc, TRUE, TRUE);

	    tc = &value_ptr->c_shade;
            gdk_colormap_free_colors(colormap, tc, 1);
	    tc->red = c->red / 2;
            tc->green = c->green / 2;
            tc->blue = c->blue / 2;
            gdk_colormap_alloc_color(colormap, tc, TRUE, TRUE);
	}
	else
	{
            GdkColor *tc;

            tc = &value_ptr->c;
	    memcpy(tc, c, sizeof(GdkColor));

	    tc = &value_ptr->c_shade;
            tc->red = c->red / 2;
            tc->green = c->green / 2;
            tc->blue = c->blue / 2;
	}


	/* Set new label text. */
	g_free(value_ptr->type_label);
	value_ptr->type_label = (type_label != NULL) ?
	    g_strdup(type_label) : NULL;

        g_free(value_ptr->value_label);
        value_ptr->value_label = (value_label != NULL) ?
	    g_strdup(value_label) : NULL;

/* Need to recreate pixmap. */
}

/*
 *	Removes the value specified by index value_num on the given pie
 *	chart.
 */
void PieChartValueRemove(
        pie_chart_struct *pc, gint value_num
)
{
	pie_chart_value_struct *value_ptr;
	GdkColormap *colormap;


        if(pc == NULL)
            return;

	if((value_num < 0) || (value_num >= pc->total_values))
	    return;

	value_ptr = pc->value[value_num];
	if(value_ptr == NULL)
	    return;


	/* Unref adjustment. */
	if(value_ptr->adj != NULL)
	{
	    gtk_object_unref(GTK_OBJECT(value_ptr->adj));
	    value_ptr->adj = NULL;
	}

	/* Free color. */
	colormap = pc->colormap;
	if(colormap != NULL)
	{
	    gdk_colormap_free_colors(colormap, &value_ptr->c, 1);
	    memset(&value_ptr->c, 0x00, sizeof(GdkColor));

            gdk_colormap_free_colors(colormap, &value_ptr->c_shade, 1);
            memset(&value_ptr->c_shade, 0x00, sizeof(GdkColor));
	}

	/* Label hbox. */
	if(value_ptr->type_label_hbox != NULL)
	{
	    gtk_widget_destroy(value_ptr->type_label_hbox);
	    value_ptr->type_label_hbox = NULL;
	}
        if(value_ptr->value_label_hbox != NULL)
        {
            gtk_widget_destroy(value_ptr->value_label_hbox);
            value_ptr->value_label_hbox = NULL;
        }
	value_ptr->drawing_area = NULL;

	/* Labels. */
	g_free(value_ptr->type_label);
	value_ptr->type_label = NULL;
        g_free(value_ptr->value_label);
        value_ptr->value_label = NULL;


	/* Deallocate structure itself and mark it as having been 
	 * deallocated.
	 */
	g_free(value_ptr);
	pc->value[value_num] = value_ptr = NULL;
}

/*
 *	Removes all values on the given pie chart.
 */
void PieChartValueRemoveAll(pie_chart_struct *pc)
{
	gint i;


	if(pc == NULL)
	    return;

	for(i = 0; i < pc->total_values; i++)
	    PieChartValueRemove(pc, i);

	g_free(pc->value);
	pc->value = NULL;
	pc->total_values = 0;
}



/*
 *	Creates a new pie chart widget.
 */
pie_chart_struct *PieChartNew(
	GtkAdjustment *adj, GdkColor *c, gint width, gint height,
	const gchar *title, const gchar *footer,
        const gchar *base_type_label, const gchar *base_value_label
)
{
	gint border_minor = 2 /*, border_major = 5 */;
	GtkWidget *w, *parent, *parent2;
	pie_chart_struct *pc = PIE_CHART(g_malloc0(
	    sizeof(pie_chart_struct)
	));
	if(pc == NULL)
	    return(pc);


	/* Reset values. */
	pc->initialized = TRUE;
	pc->map_state = FALSE;
	pc->width = width;
	pc->height = height;
	pc->pixmap = NULL;
	if(c != NULL)
	{
	    GdkColor *tc;
	    tc = &pc->c;
	    memcpy(tc, c, sizeof(GdkColor));
	    tc = &pc->c_shade;
	    tc->red = c->red / 2;
            tc->green = c->green / 2;
            tc->blue = c->blue / 2;
            tc = &pc->c_shadow;
            tc->red = 0.5 * (guint16)-1;
            tc->green = 0.5 * (guint16)-1;
            tc->blue = 0.5 * (guint16)-1;
	}
	pc->base_type_label = (base_type_label != NULL) ?
	    g_strdup(base_type_label) : NULL;
        pc->base_value_label = (base_value_label != NULL) ?
            g_strdup(base_value_label) : NULL;
	pc->gc = NULL;
	pc->colormap = NULL;
	pc->realized = FALSE;
	pc->value = NULL;
	pc->total_values = 0;

	pc->show_labels = TRUE;
	pc->shadows = TRUE;
	pc->outline = TRUE;


	/* Transfer given adjustment over and add a refcount for 
	 * ourselves.
	 */
	pc->adj = adj;
	if(adj != NULL)
	    gtk_object_ref(GTK_OBJECT(adj));


	/* Create toplevel vbox. */
	pc->toplevel = w = gtk_vbox_new(FALSE, border_minor);
        gtk_widget_show(w);
	parent = w;

	/* Create title label if title string is given. */
	if(title != NULL)
	{
	    pc->title_label = w = gtk_label_new(title);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}


	/* Create hbox for columnizing type and value labels. */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	parent2 = w;

        /* Create vbox for type labels. */
        pc->type_labels_vbox = w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Hbox to center right column vbox. */
	w = gtk_hbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
	parent2 = w;
	/* Create vbox for value labels. */
        pc->value_labels_vbox = w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_widget_show(w);


	/* Create hbox to center pie chart drawing area. */
	w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

	/* Drawing area. */
	pc->drawing_area = w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, width, height);
	gtk_drawing_area_size(GTK_DRAWING_AREA(w), width, height);
	gtk_widget_add_events(w, GDK_EXPOSURE_MASK);
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(PieChartExposeCB), pc
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

        /* Create footer label if title string is given. */
        if(footer != NULL)
        {
            pc->footer_label = w = gtk_label_new(footer);
            gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
        }



	return(pc);
}

/*
 *	Maps the pie chart.
 */
void PieChartMap(pie_chart_struct *pc)
{
	GtkWidget *w;


	if(pc == NULL)
	    return;

	w = pc->toplevel;
	if(w == NULL)
	    return;

	if(!pc->map_state)
	{
	    gtk_widget_show(w);
	    pc->map_state = TRUE;
	}
}

/*
 *	Unmaps the pie chart.
 */
void PieChartUnmap(pie_chart_struct *pc)
{
        GtkWidget *w;


        if(pc == NULL)
            return;

        w = pc->toplevel;
        if(w == NULL)
            return;

        if(pc->map_state)
        {
            gtk_widget_hide(w);
            pc->map_state = FALSE;
        }
}

/*
 *	Deallocates the given pie chart and all its resources.
 */
void PieChartDelete(pie_chart_struct *pc)
{
	GdkColormap *colormap;
	GtkWidget **w;


	if(pc == NULL)
	    return;

	colormap = pc->colormap;

	/* Remove all value structures from the pie chart. */
	PieChartValueRemoveAll(pc);

	if(pc->initialized)
	{
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Free color. */
	    if(colormap != NULL)
	    {
		gdk_colormap_free_colors(colormap, &pc->c, 1);
		memset(&pc->c, 0x00, sizeof(GdkColor));
                gdk_colormap_free_colors(colormap, &pc->c_shade, 1);
                memset(&pc->c_shade, 0x00, sizeof(GdkColor));
                gdk_colormap_free_colors(colormap, &pc->c_shadow, 1);
                memset(&pc->c_shade, 0x00, sizeof(GdkColor));
	    }

	    /* Unref adjustment. */
	    if(pc->adj != NULL)
	    {
		gtk_object_unref(GTK_OBJECT(pc->adj));
		pc->adj = NULL;
	    }

	    /* Unref pixmap. */
	    if(pc->pixmap != NULL)
	    {
		gdk_pixmap_unref(pc->pixmap);
		pc->pixmap = NULL;
	    }

            /* Begin destroying widgets. */
            w = &pc->footer_label;
            DO_DESTROY_WIDGET
            w = &pc->title_label;
            DO_DESTROY_WIDGET
            w = &pc->type_labels_vbox;
            DO_DESTROY_WIDGET
            w = &pc->value_labels_vbox;
            DO_DESTROY_WIDGET
            w = &pc->drawing_area;
            DO_DESTROY_WIDGET
	    w = &pc->toplevel;
	    DO_DESTROY_WIDGET

	    /* Unref graphic context. */
	    if(pc->gc != NULL)
	    {
		gdk_gc_unref(pc->gc);
                pc->gc = NULL;
            }

	    /* Unref colormap. */
	    if(pc->colormap != NULL)
	    {
		gdk_colormap_unref(pc->colormap);
		pc->colormap = NULL;
	    }

	    pc->realized = FALSE;

#undef DO_DESTROY_WIDGET
        }


	/* Deallocate other memory. */
        g_free(pc->base_type_label);
        pc->base_type_label = NULL;

	g_free(pc->base_value_label);
	pc->base_value_label = NULL;


	/* Deallocate structure itself. */
	g_free(pc);
}
