#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "optwin.h"
#include "optwinop.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"


static gint OptWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void OptWinSwitchPageCB(
        GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
        gpointer data
);
static void OptWinOKCB(GtkWidget *widget, gpointer data);
static void OptWinApplyCB(GtkWidget *widget, gpointer data);
static void OptWinSaveCB(GtkWidget *widget, gpointer data);
static void OptWinCancelCB(GtkWidget *widget, gpointer data);

edv_cfg_wref_struct *OptWinWRefNew(
        GtkWidget *w,           /* Widget that holds the value. */
        GtkWidget *browse,      /* Browse button. */
        const gchar *cfg_parm   /* Configuration parameter name. */
);
void OptWinWRefDelete(edv_cfg_wref_struct *wref);
edv_cfg_wref_struct *OptWinWRefAppend(
	edv_optwin_struct *optwin,
        GtkWidget *w,           /* Widget that holds the value. */
        GtkWidget *browse,      /* Browse button. */
	const gchar *cfg_parm	/* Configuration parameter name. */
);

edv_optwin_struct *OptWinNew(
        gpointer core_ptr, const gchar *title, guint8 **icon_data
);
GtkWidget *OptWinPageAppend(
	edv_optwin_struct *optwin, GtkWidget *w
);
void OptWinMap(edv_optwin_struct *optwin);
void OptWinUnmap(edv_optwin_struct *optwin);
void OptWinDelete(edv_optwin_struct *optwin);



#define OPTWIN_WIDTH		640
#define OPTWIN_HEIGHT		480

#define OPTWIN_BTN_WIDTH	(100 + (2 * 3))
#define OPTWIN_BTN_HEIGHT	(30 + (2 * 3))



/*
 *	"delete_event" signal callback.
 */
static gint OptWinDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_optwin_struct *optwin = (edv_optwin_struct *)data;
	if(optwin == NULL)
	    return(TRUE);

	OptWinCancelCB(NULL, optwin);

	return(TRUE);
}

/*
 *	GtkNotebook "switch_page" signal callback.
 */
static void OptWinSwitchPageCB(
        GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
        gpointer data
)
{
	static gbool reenterent = FALSE;
        edv_optwin_struct *optwin = (edv_optwin_struct *)data;
        if(optwin == NULL)
            return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;




	reenterent = FALSE;
}

/*
 *	OK button callback.
 */
static void OptWinOKCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_optwin_struct *optwin = (edv_optwin_struct *)data;
        if(optwin == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	OptWinDoApply(optwin);
	OptWinUnmap(optwin);

        reenterent = FALSE;
}

/*
 *	Apply button callback.
 */
static void OptWinApplyCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_optwin_struct *optwin = (edv_optwin_struct *)data;
        if(optwin == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        OptWinDoApply(optwin);

        reenterent = FALSE;
}

/*
 *	Save button callback.
 */
static void OptWinSaveCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_optwin_struct *optwin = (edv_optwin_struct *)data;
        if(optwin == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

/* Need to work on this. */

        reenterent = FALSE;
}

/*
 *	Cancel button callback.
 */
static void OptWinCancelCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_optwin_struct *optwin = (edv_optwin_struct *)data;
        if(optwin == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	OptWinUnmap(optwin);

        reenterent = FALSE;
}


/*
 *	Allocates a new edv_cfg_wref_struct.
 */
edv_cfg_wref_struct *OptWinWRefNew(
        GtkWidget *w,           /* Widget that holds the value. */
        GtkWidget *browse,      /* Browse button. */
        const gchar *cfg_parm   /* Configuration parameter name. */
)
{
	edv_cfg_wref_struct *wref = (edv_cfg_wref_struct *)g_malloc0(
	    sizeof(edv_cfg_wref_struct)
	);
	if(wref == NULL)
	    return(wref);

	wref->w = w;
	wref->browse = browse;
	wref->cfg_parm = (cfg_parm != NULL) ? g_strdup(cfg_parm) : NULL;

	/* Reset special values. */
	wref->radio_value = 0;
	wref->slist = NULL;

	return(wref);
}

/*
 *	Deallocates all resources referenced on the given wref and the
 *	structure itself.
 */
void OptWinWRefDelete(edv_cfg_wref_struct *wref)
{
	GtkWidget **w;


	if(wref == NULL)
	    return;

#define DO_DESTROY_WIDGET       \
{ \
 if(*w != NULL) \
 { \
  GtkWidget *tw = *w; \
  *w = NULL; \
  gtk_widget_destroy(tw); \
 } \
}

	/* Deallocate standard referenced widgets (if any). */
	w = &wref->w;
	DO_DESTROY_WIDGET

	w = &wref->browse;
	DO_DESTROY_WIDGET


	/* Deallocate other resources referenced under special
	 * conditions.
	 */
	StackListDelete(wref->slist);
	wref->slist = NULL;

#undef DO_DESTROY_WIDGET

	g_free(wref->cfg_parm);
	wref->cfg_parm = NULL;

	g_free(wref);
}

/*
 *	Appends a new wref structure to the optwin, returns the pointer to
 *	the new wref structure.
 */
edv_cfg_wref_struct *OptWinWRefAppend(
        edv_optwin_struct *optwin,
        GtkWidget *w,           /* Widget that holds the value. */
        GtkWidget *browse,      /* Browse button. */
	const gchar *cfg_parm	/* Configuration parameter name. */
)
{
	gint i;
	edv_cfg_wref_struct *wref;


	if(optwin == NULL)
	    return(NULL);

	/* Sanitize total. */
	if(optwin->total_wrefs < 0)
	    optwin->total_wrefs = 0;

	/* Allocate more pointers. */
	i = optwin->total_wrefs;
	optwin->total_wrefs = i + 1;

	optwin->wref = (edv_cfg_wref_struct **)g_realloc(
	    optwin->wref,
	    optwin->total_wrefs * sizeof(edv_cfg_wref_struct *)
	);
	if(optwin->wref == NULL)
	{
	    optwin->total_wrefs = 0;
	    return(NULL);
	}

	/* Allocate a new structure. */
	optwin->wref[i] = wref = OptWinWRefNew(
	    w, browse, cfg_parm
	);

	return(wref);
}


/*
 *	Creates a new edv_optwin_struct.
 */
edv_optwin_struct *OptWinNew(
	gpointer core_ptr, const gchar *title, guint8 **icon_data
)
{
	gint /*border_minor = 2,*/ border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	edv_optwin_struct *optwin = (edv_optwin_struct *)g_malloc0(
	    sizeof(edv_optwin_struct)
	);
	if(optwin == NULL)
	    return(optwin);


	/* Reset values. */
	optwin->initialized = TRUE;
	optwin->map_state = FALSE;
	optwin->processing = FALSE;
	optwin->core_ptr = core_ptr;


	/* Keyboard accelerator group. */
	optwin->accelgrp = accelgrp = gtk_accel_group_new();


        /* Toplevel. */
	optwin->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, OPTWIN_WIDTH, OPTWIN_HEIGHT);
        gtk_widget_realize(w);
        gtk_window_set_title(GTK_WINDOW(w), title);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
	    GUISetWMIcon(window, (u_int8_t **)icon_data);
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(OptWinDeleteEventCB), optwin
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;


	/* Main vbox. */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Vbox for main notebook. */
        w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent2 = w;

	/* Main notebook. */
	optwin->notebook = w = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
/*	gtk_notebook_set_page(GTK_NOTEBOOK(w), 0); */
	gtk_signal_connect(
	    GTK_OBJECT(w), "switch_page",
	    GTK_SIGNAL_FUNC(OptWinSwitchPageCB), optwin
	);
	gtk_widget_show(w);
	parent2 = w;



        /* Separator. */
        w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Buttons hbox. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* OK button. */
        optwin->ok_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_ok_20x20_xpm, "OK", NULL
        );
        gtk_widget_set_usize(w, OPTWIN_BTN_WIDTH, OPTWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(OptWinOKCB), optwin
        );
        gtk_accel_group_add(
            accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_3270_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_KP_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_ISO_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);

        /* Apply button. */
        optwin->apply_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_select_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Apply",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Aplique",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "S'appliquer",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, OPTWIN_BTN_WIDTH, OPTWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(OptWinApplyCB), optwin
        );
        gtk_widget_show(w);

        /* Save button. */
        optwin->save_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_save_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Save",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Salve",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Epargner",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, OPTWIN_BTN_WIDTH, OPTWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(OptWinSaveCB), optwin
        );
/*	gtk_widget_show(w); */

        /* Cancel button. */
        optwin->cancel_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Cancel",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cancele",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Annuler",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, OPTWIN_BTN_WIDTH, OPTWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(OptWinCancelCB), optwin
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);



	return(optwin);
}

/*
 *	Appends a new page to the given optwin, using the given widget
 *	as the tab widget.
 *
 *	The given widget should not be referenced again after this
 *	call.
 */
GtkWidget *OptWinPageAppend(
        edv_optwin_struct *optwin, GtkWidget *w
)
{
	gint i;
	GtkWidget *parent, *parent2;


	if(w == NULL)
	    return(NULL);

	if(optwin == NULL)
	{
	    gtk_widget_destroy(w);
	    return(NULL);
	}

	parent = optwin->notebook;

	/* Create a new vbox as parent2 and append it to the notebook
	 * as a page using the given widget w as the tab widget.
	 */
	parent2 = gtk_vbox_new(FALSE, 0);
	gtk_notebook_append_page(
	    GTK_NOTEBOOK(parent), parent2, w
	);
	gtk_widget_show(parent2);

	/* Widget w should not be referenced again after this point. */
	w = NULL;


	/* Record new toplevel vbox of the appended page. */
	if(optwin->total_pages < 0)
	    optwin->total_pages = 0;

	i = optwin->total_pages;
	optwin->total_pages = i + 1;

	optwin->page = (GtkWidget **)g_realloc(
	    optwin->page,
	    optwin->total_pages * sizeof(GtkWidget *)
	);
	if(optwin->page == NULL)
	{
	    optwin->total_pages = 0;
	}
	else
	{
	    optwin->page[i] = parent2;
	}

	/* Return the newly created vbox of the appended page. */
	return(parent2);
}

/*
 *	Maps the given optwin.
 */
void OptWinMap(edv_optwin_struct *optwin)
{
	GtkWidget *w;

        if(optwin == NULL)
            return;

        w = optwin->toplevel;
	gtk_widget_show_raise(w);
	optwin->map_state = TRUE;

	w = optwin->ok_btn;
	if(w != NULL)
	{
	    gtk_widget_grab_focus(w);
	    gtk_widget_grab_default(w);
	}
}

/*
 *	Unmaps the given optwin.
 */
void OptWinUnmap(edv_optwin_struct *optwin)
{
        GtkWidget *w;

        if(optwin == NULL)
            return;

	w = optwin->toplevel;
	if(w == NULL)
	    return;

	if(optwin->map_state)
	{
	    gtk_widget_hide(w);
	    optwin->map_state = FALSE;
	}
}

/*
 *	Deallocates all resources on the given optwin and deallocates the
 *	structure itself.
 */
void OptWinDelete(edv_optwin_struct *optwin)
{
	gint i;
	GtkWidget **w;


	if(optwin == NULL)
	    return;

	if(optwin->initialized)
	{
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Deallocate all wrefs, this will destroy all widgets 
	     * that we referenced as instructed by the calling function.
	     */
	    for(i = 0; i < optwin->total_wrefs; i++)
		OptWinWRefDelete(optwin->wref[i]);
	    g_free(optwin->wref);
	    optwin->wref = NULL;
	    optwin->total_wrefs = 0;

	    /* Deallocate toplevel widgets on all notebook pages. */
	    for(i = 0; i < optwin->total_pages; i++)
	    {
		w = &optwin->page[i];
		DO_DESTROY_WIDGET
	    }
            g_free(optwin->page);
            optwin->page = NULL;
            optwin->total_pages = 0;


            w = &optwin->notebook;
            DO_DESTROY_WIDGET

	    w = &optwin->ok_btn;
	    DO_DESTROY_WIDGET
	    w = &optwin->apply_btn;
	    DO_DESTROY_WIDGET
            w = &optwin->save_btn;
	    DO_DESTROY_WIDGET
            w = &optwin->cancel_btn;
	    DO_DESTROY_WIDGET

            w = &optwin->toplevel;
            DO_DESTROY_WIDGET


            if(optwin->accelgrp != NULL)
            {
                gtk_accel_group_unref(optwin->accelgrp);
                optwin->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
	}

	/* Deallocate structure itself. */
	g_free(optwin);
}
