#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pulist.h"

#include "edvtypes.h"
#include "edvdevices.h"
#include "edvmount.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvop.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "mountbar.h"
#include "config.h"

#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm"
#include "images/icon_goto_20x20.xpm"
#include "images/icon_reload_20x20.xpm"



static gint EDVMountBarDeviceExposeCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
static gint EDVMountBarDeviceFocusInCB(
        GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static gint EDVMountBarDeviceFocusOutCB(
        GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static gint EDVMountBarCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static gint EDVMountBarDeviceKeyCB(
        GtkWidget *widget, GdkEventKey *key, gpointer data
);
static gint EDVMountBarDeviceButtonCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static void EDVMountBarMapPUListCB(GtkWidget *widget, gpointer data);

static void EDVMountBarMountCB(GtkWidget *widget, gpointer data);
static void EDVMountBarEjectCB(GtkWidget *widget, gpointer data);
static void EDVMountBarGoToCB(GtkWidget *widget, gpointer data);
static void EDVMountBarRefreshStatsCB(GtkWidget *widget, gpointer data);

static void EDVMountBarSetStats(edv_mountbar_struct *mb);

edv_mountbar_struct *EDVMountBarNew(
	gpointer core_ptr, GtkWidget *parent,
        void (*mount_cb)(gpointer, gint, edv_device_struct *, gpointer),
        void (*eject_cb)(gpointer, gint, edv_device_struct *, gpointer),
        void (*goto_cb)(gpointer, gint, edv_device_struct *, gpointer),
        void (*status_message_cb)(gpointer, const gchar *, gpointer),
        gpointer client_data
);
void EDVMountBarUpdateMenus(edv_mountbar_struct *mb);
void EDVMountBarMap(edv_mountbar_struct *mb);
void EDVMountBarUnmap(edv_mountbar_struct *mb);
void EDVMountBarDelete(edv_mountbar_struct *mb);



/*
 *	Device icon and label drawing area "expose_event" signal callback.
 */
static gint EDVMountBarDeviceExposeCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	static gbool reenterent = FALSE;
	gint status = FALSE;
	gint state, width, height;
	gint icon_width = 0, icon_height = 0;
	GdkGC *gc;
	GdkWindow *window;
	GtkStyle *style;
	GtkWidget *w;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if(mb == NULL)
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

	w = mb->dev_da;
	if(w == NULL)
	{
	    reenterent = FALSE;
	    return(status);
	}


        state = GTK_WIDGET_STATE(w);
	window = w->window;
        style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
        {
            reenterent = FALSE;
            return(status);
        }

        gdk_window_get_size(window, &width, &height);


        /* Create graphic context as needed. */
        gc = mb->gc;
        if(gc == NULL)
	{
	    mb->gc = gc = gdk_gc_new(window);
	    if(gc == NULL)
	    {
		reenterent = FALSE;
		return(status);
	    }
            gdk_gc_set_function(gc, GDK_COPY);
            gdk_gc_set_fill(gc, GDK_SOLID);
            gdk_gc_set_foreground(gc, &style->bg[state]);
        }


        /* Draw background to cover entire pixmap buffer. */
	gdk_gc_set_fill(gc, GDK_SOLID);
	gdk_gc_set_line_attributes(
	    gc, 1, GDK_LINE_SOLID, GDK_CAP_NOT_LAST, GDK_JOIN_MITER
	);
        gdk_gc_set_foreground(gc, &style->base[state]);
        gdk_gc_set_clip_mask(gc, NULL);

        gdk_draw_rectangle(
            (GdkDrawable *)window, gc, TRUE,
            0, 0, width, height
        );


	/* Draw icon. */
	if(mb->dev_pixmap != NULL)
	{
	    gint	x = 2 + 2,
			y = 2;

	    gdk_window_get_size(
		(GdkWindow *)mb->dev_pixmap, &icon_width, &icon_height
	    );
	    gdk_gc_set_clip_mask(gc, mb->dev_mask);
	    gdk_gc_set_clip_origin(gc, x, y);

	    gdk_draw_pixmap(
		(GdkDrawable *)window, gc, mb->dev_pixmap,
		0, 0, x, y,
		icon_width, icon_height
	    );

	    gdk_gc_set_clip_mask(gc, NULL);
	}

        /* Draw label */
        if((mb->dev_text != NULL) && (style->font != NULL))
        {
            gint        x = 2 + 2 + icon_width + 2,
                        y = 0;
            gint lbearing, rbearing, swidth, ascent, descent;


            gdk_string_extents(
                style->font, mb->dev_text,
                &lbearing, &rbearing, &swidth, &ascent, &descent
            );
            gdk_gc_set_foreground(gc, &style->text[state]);

	    gdk_draw_string(
		(GdkDrawable *)window,
		style->font, gc,
		x - lbearing,
		y + (height / 2) - ((ascent + descent) / 2) + ascent,
		mb->dev_text
	    );
        }

	/* Draw shadow in frame. */
	gtk_draw_shadow(
	    style, window, state, GTK_SHADOW_IN,
	    0, 0, width - 1, height - 1
	);

	/* Draw focus if widget is focused. */
	if(GTK_WIDGET_HAS_FOCUS(widget))
	    gtk_draw_focus(
		style, window,
		0, 0, width - 1, height - 1
	    );


	status = TRUE;
        reenterent = FALSE;
	return(status);
}

/*
 *      Device icon and label drawing area "focus_out_event" signal
 *      callback.
 */
static gint EDVMountBarDeviceFocusInCB(
        GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
	GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
	EDVMountBarDeviceExposeCB(widget, NULL, data);
        return(TRUE);
}

/*
 *	Device icon and label drawing area "focus_out_event" signal
 *	callback.
 */
static gint EDVMountBarDeviceFocusOutCB(
	GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
	GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
        EDVMountBarDeviceExposeCB(widget, NULL, data);
	return(TRUE);
}

/*
 *	Mount bar widget "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
static gint EDVMountBarCrossingCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
        static gbool reenterent = FALSE;
	gint status = FALSE;
	gint etype;
	GtkWidget *w;
	const gchar *message = NULL;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if((widget == NULL) || (crossing == NULL) || (mb == NULL))
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

	/* Get event type. */
	etype = crossing->type;

	w = widget;


	/* Handle by event type. */
	switch(etype)
	{
	  case GDK_ENTER_NOTIFY:
	    if(w == mb->mount_btn)
	    {
#ifdef PROG_LANGUAGE_ENGLISH
		message = "Mount/unmount device";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                message = "El monte/artefacto de unmount";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                message = "Appareil de monte/unmount";
#endif
	    }
            else if(w == mb->eject_btn)
            {
#ifdef PROG_LANGUAGE_ENGLISH
                message = "Eject media from device";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                message = "Expulse medios del artefacto";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                message = "Ejecter le presse de l'appareil";
#endif
            }
	    else if(w == mb->goto_btn)
	    {
#ifdef PROG_LANGUAGE_ENGLISH
                message = "Go to mount path";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                message = "Vaya a montar sendero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                message = "Aller monter le sentier";
#endif
            }
            else if(w == mb->refresh_stats_btn)
            {
#ifdef PROG_LANGUAGE_ENGLISH
		message = "Refresh all device statistics";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		message = "Refresque toda estadstica de artefacto";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		message = "Rafrachir toute statistique d'appareil";
#endif
            }
	    else if(w == mb->capacity_pbar)
	    {
		/* Ignore this. */
	    }
	    break;
	}

	if(mb->status_message_cb != NULL)
	{
	    mb->status_message_cb(
		mb,
		message,
		mb->client_data
	    );
	}

	reenterent = FALSE;
	return(status);
}


/*
 *      Device icon and label drawing area "key_press_event" or
 *      "key_release_event" signal callback.
 */
static gint EDVMountBarDeviceKeyCB(
        GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
        static gbool reenterent = FALSE;
        gint status = FALSE;
        gint etype;
        guint keyval, state;
        gbool press;
	edv_core_struct *core_ptr;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if((key == NULL) || (mb == NULL))
            return(status);

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

        /* Get event type. */
        etype = key->type;

        /* Get other key event values. */
        press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
        keyval = key->keyval;
        state = key->state;

/* Macro to emit a signal stop for a key press or release depending
 * on the current event's type.
 */
#define DO_STOP_KEY_SIGNAL_EMIT \
{ \
 gtk_signal_emit_stop_by_name( \
  GTK_OBJECT(widget), \
  press ? "key_press_event" : "key_release_event" \
 ); \
}
	if(1)
	{
            /* Handle by key value. */
            switch(keyval)
            {
              case GDK_Return:
              case GDK_KP_Enter:
              case GDK_ISO_Enter:
              case GDK_3270_Enter:
              case GDK_space:
              case GDK_KP_Space:
                if(press)
                {
		    EDVMountBarMountCB(NULL, mb);
		}
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Up:
              case GDK_KP_Up:
              case GDK_Page_Up:
              case GDK_KP_Page_Up:
		if(press)
		{
		    gint i = mb->selected_dev_num;
		    edv_device_struct *dev_ptr;

		    /* Current selection in bounds? */
		    if((i >= 0) && (i < core_ptr->total_devices))
		    {
                        /* Check previous device and see if it is selectable.
                         * Iterate on to previous device if the current is
                         * not selectable.
                         */
			for(i--; i >= 0; i--)
			{
			    dev_ptr = core_ptr->device[i];
                            if(dev_ptr == NULL)
                                continue;
                            if(dev_ptr->unlisted)
                                continue;
			    break;	/* This device is selectable. */
			}
                        /* Found a selectable device index? */
			if(i >= 0)
			    mb->selected_dev_num = i;
		    }
		    else
		    {
			/* Current selection is out of bounds, so select
			 * the first valid device.
			 */
			for(i = 0; i < core_ptr->total_devices; i++)
			{
			    dev_ptr = core_ptr->device[i];
			    if(dev_ptr == NULL)
				continue;
			    if(dev_ptr->unlisted)
				continue;

			    mb->selected_dev_num = i;
			}
		    }
		    EDVMountBarUpdateMenus(mb);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Down:
              case GDK_KP_Down:
              case GDK_Page_Down:
              case GDK_KP_Page_Down:
                if(press)
                {
                    gint i = mb->selected_dev_num;
                    edv_device_struct *dev_ptr;

                    /* Current selection in bounds? */
                    if((i >= 0) && (i < core_ptr->total_devices))
                    {
			/* Check next device and see if it is selectable.
			 * Iterate on to next device if the current is
			 * not selectable.
			 */
                        for(i++; i < core_ptr->total_devices; i++)
                        {
                            dev_ptr = core_ptr->device[i];
                            if(dev_ptr == NULL)
                                continue;
                            if(dev_ptr->unlisted)
                                continue;
                            break;	/* This device is selectable. */
                        }
			/* Found a selectable device index? */
                        if(i < core_ptr->total_devices)
                            mb->selected_dev_num = i;
                    }
                    else
                    {
                        /* Current selection is out of bounds, so select
                         * the first valid device.
                         */
                        for(i = 0; i < core_ptr->total_devices; i++)
                        {
                            dev_ptr = core_ptr->device[i];
                            if(dev_ptr == NULL)
                                continue;
                            if(dev_ptr->unlisted)
                                continue;

                            mb->selected_dev_num = i;
                        }
                    }
                    EDVMountBarUpdateMenus(mb);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Home:
                if(press)
                {
                    gint i;
                    edv_device_struct *dev_ptr;

		    /* Look for the first device that is selectable. */
                    for(i = 0; i < core_ptr->total_devices; i++)
                    {
			dev_ptr = core_ptr->device[i];
                        if(dev_ptr == NULL)
                            continue;
                        if(dev_ptr->unlisted)
                            continue;
                        break;		/* This device is selectable. */
		    }
		    /* Found a selectable device index? */
                    if(i < core_ptr->total_devices)
                    {
                        mb->selected_dev_num = i;
                        EDVMountBarUpdateMenus(mb);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_End:
                if(press)
                {
                    gint i;
                    edv_device_struct *dev_ptr;

                    /* Look for the last device that is selectable. */
                    for(i = core_ptr->total_devices - 1; i >= 0; i--)
                    {
                        dev_ptr = core_ptr->device[i];
                        if(dev_ptr == NULL)
                            continue;
                        if(dev_ptr->unlisted)
                            continue;
                        break;          /* This device is selectable. */
                    }
                    /* Found a selectable device index? */
                    if(i >= 0)
                    {
                        mb->selected_dev_num = i;
                        EDVMountBarUpdateMenus(mb);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;
	    }
	}


        reenterent = FALSE;
        return(status);
}


/*
 *      Device icon and label drawing area "button_press_event" or
 *	"button_release_event" signal callback.
 */
static gint EDVMountBarDeviceButtonCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	static gbool reenterent = FALSE;
	gint status = FALSE;
	gint etype;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
	if((button == NULL) || (mb == NULL))
	    return(status);

	if(reenterent)
	    return(status);
	else
	    reenterent = TRUE;

	/* Get event type. */
	etype = button->type;

	/* Handle by event type. */
	switch(etype)
	{
	  case GDK_BUTTON_PRESS:
	    switch(button->button)
	    {
	      case 1:
		EDVMountBarMapPUListCB(mb->map_btn, mb);
		status = TRUE;
		break;
	    }
	    /* Grab focus on any button press on this widget. */
	    if(!GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);
	    break;

	}

	reenterent = FALSE;
	return(status);
}

/*
 *	Map devices popup list callback.
 */
static void EDVMountBarMapPUListCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        const gchar *value;
	gint dev_num;
	edv_device_struct *dev_ptr;
	pulist_struct *pulist;
        edv_core_struct *core_ptr;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if(mb == NULL)
            return;

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return;

	pulist = core_ptr->devices_pulist;
	if(pulist == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Block input and get value. */
        value = PUListMapQuery(
            pulist, mb->dev_text,
            10,
            PULIST_RELATIVE_BELOW,
	    mb->dev_da, widget
        );
	/* User canceled? */
	if(value == NULL)
	{
	    reenterent = FALSE;
	    return;
	}

	/* Get device index number from the selected value. */
	dev_num = (gint)PUListGetDataFromValue(pulist, value);
	if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
	    dev_ptr = core_ptr->device[dev_num];
	else
	    dev_ptr = NULL;

        /* Got valid device? */
        if(dev_ptr != NULL)
	{
	    /* Set new selected device on the mount bar structure. */
	    mb->selected_dev_num = dev_num;

	    EDVMountBarUpdateMenus(mb);
	}

        reenterent = FALSE;
}

/*
 *	Mount/unmount callback.
 */
static void EDVMountBarMountCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        gint dev_num;
        edv_core_struct *core_ptr;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if(mb == NULL)
            return;

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        dev_num = mb->selected_dev_num;
        if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
        {
            edv_device_struct *dev_ptr = core_ptr->device[dev_num];
            if((dev_ptr != NULL) && (mb->mount_cb != NULL))
                mb->mount_cb(
                    mb,
                    dev_num, dev_ptr,
                    mb->client_data
                );
        }

        reenterent = FALSE;
}

/*
 *	Eject callback.
 */
static void EDVMountBarEjectCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        gint dev_num;
        edv_core_struct *core_ptr;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if(mb == NULL)
            return;

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        dev_num = mb->selected_dev_num;
        if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
        {
            edv_device_struct *dev_ptr = core_ptr->device[dev_num];
            if((dev_ptr != NULL) && (mb->eject_cb != NULL))
                mb->eject_cb(
                    mb,
                    dev_num, dev_ptr,
                    mb->client_data
                );
        }

        reenterent = FALSE;
}

/*
 *	Go to mount path callback.
 */
static void EDVMountBarGoToCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
	gint dev_num;
	edv_core_struct *core_ptr;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if(mb == NULL)
            return;

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	dev_num = mb->selected_dev_num;
	if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
	{
	    edv_device_struct *dev_ptr = core_ptr->device[dev_num];
	    if((dev_ptr != NULL) && (mb->goto_cb != NULL))
		mb->goto_cb(
		    mb,
		    dev_num, dev_ptr,
		    mb->client_data
		);
	}

	reenterent = FALSE;
}

/*
 *	Refresh stats callback.
 *
 *	Updates the device stats on the core structure.
 */
static void EDVMountBarRefreshStatsCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        edv_core_struct *core_ptr;
        edv_mountbar_struct *mb = (edv_mountbar_struct *)data;
        if(mb == NULL)
            return;

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Refresh all device mount states and device stats, then get
         * device path or mount path that matches the given disk
         * object's path.
         */
        EDVDevicesListUpdateMountStates(
            core_ptr->device, core_ptr->total_devices
        );
        EDVDevicesListUpdateStats(
            core_ptr->device, core_ptr->total_devices
        );

	EDVMountBarUpdateMenus(mb);

        reenterent = FALSE;
}

/*
 *	Sets the stats label of the given mount bar to reflect the values
 *	of the selected device.
 *
 *	The devices list on the core structure is not updated in this call.
 */
static void EDVMountBarSetStats(edv_mountbar_struct *mb)
{
	const gchar *cstrptr;
        gint dev_num;
	edv_device_struct *dev_ptr;
        edv_core_struct *core_ptr;
	GtkWidget *w;


        if(mb == NULL)
            return;

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return;

	/* Get selected device index number and pointer. */
        dev_num = mb->selected_dev_num;
        if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
	    dev_ptr = core_ptr->device[dev_num];
	else
	    dev_ptr = NULL;

        /* Begin updating capacity progress bar. */
        w = mb->capacity_pbar;
        if((dev_ptr != NULL) && (w != NULL))
        {
            gulong used = dev_ptr->blocks_total - dev_ptr->blocks_free;
	    GtkProgress *pr = GTK_PROGRESS(w);
            GtkProgressBar *pb = GTK_PROGRESS_BAR(w);

            gtk_progress_set_activity_mode(pr, FALSE);
            gtk_progress_set_format_string(pr, "%p%%");
            gtk_progress_set_show_text(pr, FALSE);

            gtk_progress_bar_update(
		pb,
		(dev_ptr->blocks_total > 0) ?
		    (gfloat)used /
			(gfloat)dev_ptr->blocks_total : 0.0
	    );
	}

	/* Begin updating stats label. */
	w = mb->stats_label;
	if((dev_ptr != NULL) && (w != NULL))
	{
	    gulong used = dev_ptr->blocks_total - dev_ptr->blocks_free;
	    gchar text[256];


	    if(dev_ptr->is_mounted)
	    {
		gchar	used_s[256], total_s[256], free_avail_s[256],
			free_s[256];

		strcpy(
		    used_s,
		    EDVGetObjectSizeStr(core_ptr, used)
		);
                strcpy(
                    total_s,
                    EDVGetObjectSizeStr(core_ptr, dev_ptr->blocks_total)
                );
                strcpy(
                    free_avail_s,
                    EDVGetObjectSizeStr(core_ptr, dev_ptr->blocks_available)
                );
                strcpy(
                    free_s,
                    EDVGetObjectSizeStr(core_ptr, dev_ptr->blocks_free)
                );

#ifdef PROG_LANGUAGE_ENGLISH
		cstrptr = "Used: %s kb  Total: %s kb  %.0f%%  \
Free & Available: %s kb  Free: %s kb";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		cstrptr = "Usado: %s kb  El Suma: %s kb %.0f%%  \
Free & Available: %s kb  Free: %s kb";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		cstrptr = "Utilis: %s kb  Total: %s kb  %.0f%%  \
Free & Available: %s kb  Free: %s kb";
#endif
		if(dev_ptr->blocks_total > 0)
		{
		    sprintf(
			text, cstrptr,
			used_s,
			total_s,
			(gfloat)used * 100.0 /
			    (gfloat)dev_ptr->blocks_total,
			free_avail_s,
			free_s
		    );
		}
		else
		{
#ifdef PROG_LANGUAGE_ENGLISH
		    cstrptr = "Undefined Capacity";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		    cstrptr = "La Capacidad Indefinida";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		    cstrptr = "Capacit Non Dfinie";
#endif
		    strcpy(text, cstrptr);
		}
	    }
	    else
	    {
#ifdef PROG_LANGUAGE_ENGLISH
		cstrptr = "Device Not Mounted";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		cstrptr = "El Artefacto No Mont";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		cstrptr = "L'appareil n'A Pas Mont";
#endif
		strcpy(text, cstrptr);
	    }

	    gtk_label_set_text(GTK_LABEL(w), text);
	}
}


/*
 *	Create a new mount bar.
 */
edv_mountbar_struct *EDVMountBarNew(
        gpointer core_ptr, GtkWidget *parent,
	void (*mount_cb)(gpointer, gint, edv_device_struct *, gpointer),
        void (*eject_cb)(gpointer, gint, edv_device_struct *, gpointer),
        void (*goto_cb)(gpointer, gint, edv_device_struct *, gpointer),
        void (*status_message_cb)(gpointer, const gchar *, gpointer),
	gpointer client_data
)
{
        gint border_minor = 2;
	GtkAdjustment *adj;
        GtkWidget *w, *parent2;
	edv_mountbar_struct *mb = (edv_mountbar_struct *)g_malloc0(
            sizeof(edv_mountbar_struct)
        );
        if(mb == NULL)
            return(mb);

        /* Reset values. */
        mb->initialized = TRUE;
        mb->map_state = FALSE;
	mb->core_ptr = core_ptr;
	mb->last_mount_btn_state = -1;
	mb->dev_text = NULL;
	mb->dev_pixmap = NULL;
	mb->dev_mask = NULL;
	mb->selected_dev_num = 0;
	mb->mount_cb = mount_cb;
	mb->eject_cb = eject_cb;
        mb->goto_cb = goto_cb;
	mb->status_message_cb = status_message_cb;
	mb->client_data = client_data;

	mb->gc = NULL;


	/* Set first device from the core structure as the initially
	 * selected device. This first device is not always device
	 * index 0, but rather the first one that is not marked as
	 * unlisted.
	 *
	 */
	if(core_ptr != NULL)
	{
	    gint i;
	    edv_device_struct *dev_ptr;
	    edv_core_struct *c_ptr = (edv_core_struct *)core_ptr;

	    for(i = 0; i < c_ptr->total_devices; i++)
	    {
		dev_ptr = c_ptr->device[i];
		if(dev_ptr == NULL)
		    continue;
		if(dev_ptr->unlisted)
		    continue;

		mb->selected_dev_num = i;
		break;
	    }
	}

        /* Create toplevel. */
	mb->toplevel = w = gtk_hbox_new(FALSE, border_minor);
        gtk_container_border_width(GTK_CONTAINER(w), 2);
        gtk_container_add(GTK_CONTAINER(parent), w);
        parent = w;


	/* Hbox for device icon and label drawing area and map button. */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	parent2 = w;

	/* Device drawing area. */
        mb->dev_da = w = gtk_drawing_area_new();
        gtk_widget_set_usize(w, 200, 20 + (2 * 2));
        gtk_drawing_area_size(GTK_DRAWING_AREA(w), 200, 20 + (2 * 2));
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
        gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK | GDK_FOCUS_CHANGE_MASK |
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(EDVMountBarDeviceExposeCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "focus_in_event",
            GTK_SIGNAL_FUNC(EDVMountBarDeviceFocusInCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "focus_out_event",
            GTK_SIGNAL_FUNC(EDVMountBarDeviceFocusOutCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_press_event",
            GTK_SIGNAL_FUNC(EDVMountBarDeviceKeyCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_release_event",
            GTK_SIGNAL_FUNC(EDVMountBarDeviceKeyCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(EDVMountBarDeviceButtonCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_release_event",
            GTK_SIGNAL_FUNC(EDVMountBarDeviceButtonCB), mb
        );
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Popup list map button. */
        mb->map_btn = w = PUListNewMapButton(
            EDVMountBarMapPUListCB, mb
        );
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);



	/* Mount/unmount button. */
	mb->mount_btn = w = (GtkWidget *)GUIButtonPixmap(
	    (u_int8_t **)icon_mount_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMountBarMountCB), mb
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Mount/unmount device"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "El monte/artefacto de unmount"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Appareil de monte/unmount"
#endif
	);
        gtk_widget_show(w);

        /* Eject button. */
        mb->eject_btn = w = (GtkWidget *)GUIButtonPixmap(
            (u_int8_t **)icon_eject_20x20_xpm
        );
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMountBarEjectCB), mb
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Eject media from device"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Expulse medios del artefacto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Ejecter le presse de l'appareil"
#endif
        );
        gtk_widget_show(w);

	w = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Go to button. */
        mb->goto_btn = w = (GtkWidget *)GUIButtonPixmap(
            (u_int8_t **)icon_goto_20x20_xpm
        );
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMountBarGoToCB), mb
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Go to mount path"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Vaya a montar sendero"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Aller monter le sentier"
#endif
        );
        gtk_widget_show(w);

        /* Refresh stats button. */
        mb->refresh_stats_btn = w = (GtkWidget *)GUIButtonPixmap(
            (u_int8_t **)icon_reload_20x20_xpm
        );
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVMountBarCrossingCB), mb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMountBarRefreshStatsCB), mb
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Refresh all device statistics"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Refresque toda estadstica de artefacto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Rafrachir toute statistique d'appareil"
#endif
        );
        gtk_widget_show(w);


	/* Capacity progress bar. */
        adj = (GtkAdjustment *)gtk_adjustment_new(0, 1, 100, 0, 5, 5);
        mb->capacity_pbar = w = gtk_progress_bar_new_with_adjustment(adj);
        gtk_widget_set_usize(w, 100, -1);
        gtk_progress_bar_set_orientation(
            GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
        );
        gtk_progress_bar_set_bar_style(
            GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
        );
        gtk_progress_set_activity_mode(
            GTK_PROGRESS(w), FALSE
        );
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


	/* Frame for stats label. */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
        parent2 = w;

	/* Stats label. */
	mb->stats_label = w = gtk_label_new("");
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_minor);
        gtk_widget_show(w);


	EDVMountBarUpdateMenus(mb);

	return(mb);
}


/*
 *	Updates all menus and widgets on the mount bar.
 *
 *	This should be called whenever a device has been mounted/unmounted
 *	or when one or more devices have been modified, added, or removed.
 */
void EDVMountBarUpdateMenus(edv_mountbar_struct *mb)
{
	gint dev_num;
	GtkWidget *w;
	edv_device_struct *dev_ptr;
	edv_core_struct *core_ptr;


	if(mb == NULL)
	    return;

        core_ptr = (edv_core_struct *)mb->core_ptr;
        if(core_ptr == NULL)
            return;

	/* Get selected device index number and pointer (if any). */
	dev_num = mb->selected_dev_num;
	if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
	    dev_ptr = core_ptr->device[dev_num];
	else
	    dev_ptr = NULL;

        /* Update device icon. */
        if(dev_ptr != NULL)
	{
	    gint i;
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;


	    EDVDeviceRealize(dev_ptr, FALSE);

	    /* Get device icon pixmap and mask pair from selected device
	     * structure.
	     */
	    for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	    {
		pixmap = dev_ptr->small_pixmap[i];
                mask = dev_ptr->small_mask[i];
		if(pixmap != NULL)
		    break;
	    }
	    /* Got device icon pixmap and mask pair? */
	    if(pixmap != NULL)
	    {
		GdkPixmap *old_pixmap = mb->dev_pixmap;
		GdkBitmap *old_mask = mb->dev_mask;

		/* Set new pixmap and mask pair and add a refcount. */
		mb->dev_pixmap = pixmap;
		mb->dev_mask = mask;
		if(pixmap != NULL)
		    gdk_pixmap_ref(pixmap);
		if(mask != NULL)
		    gdk_bitmap_ref(mask);

		/* Unref old pixmap and mask pair (if any). */
		if(old_pixmap != NULL)
		    gdk_pixmap_unref(old_pixmap);
		if(old_mask != NULL)
                    gdk_bitmap_unref(old_mask);
	    }
	}

        /* Update device label. */
        if(dev_ptr != NULL)
        {
	    const gchar *text = dev_ptr->name;
	    if((text == NULL) && (dev_ptr->device_path != NULL))
	    {
		text = strrchr(dev_ptr->device_path, DIR_DELIMINATOR);
		if(text == NULL)
		    text = dev_ptr->device_path;
		else
		    text++;
	    }
	    if(text == NULL)
		text = "(null)";

	    g_free(mb->dev_text);
	    mb->dev_text = g_strdup(text);
	}

	/* Update mount button. */
	w = mb->mount_btn;
	if(w != NULL)
	{
	    if(dev_ptr == NULL)
	    {
		/* No device selected, so set insensitive. */
		gtk_widget_set_sensitive(w, FALSE);
	    }
	    else
	    {
		u_int8_t **icon_data = NULL;


		/* Update sensitivity based on selected device's
		 * no_unmount setting.
		 */
                gtk_widget_set_sensitive(w, !dev_ptr->no_unmount);

		/* Current mount button icon not defined? */
		if(mb->last_mount_btn_state < 0)
		{
		    /* Go ahead and explicitly set icon then. */
		    icon_data = dev_ptr->is_mounted ?
			(u_int8_t **)icon_unmount_20x20_xpm :
			(u_int8_t **)icon_mount_20x20_xpm;
		}
		/* Set unmount icon? */
		else if(dev_ptr->is_mounted &&
                        (mb->last_mount_btn_state == 0)
		)
		{
		    icon_data = (u_int8_t **)icon_unmount_20x20_xpm;
		}
		/* Set mount icon? */
                else if(!dev_ptr->is_mounted &&
                        (mb->last_mount_btn_state == 1)
                )
                {
                    icon_data = (u_int8_t **)icon_mount_20x20_xpm;
                }
		if(icon_data != NULL)
		    GUIButtonPixmapUpdate(w, icon_data, NULL);

                /* Update last mount button icon state. */
                mb->last_mount_btn_state = dev_ptr->is_mounted ? 1 : 0;
	    }
	}

        /* Update eject button. */
        w = mb->eject_btn;
        if(w != NULL)
	{
	    if(dev_ptr == NULL)
		gtk_widget_set_sensitive(w, FALSE);
	    else
		gtk_widget_set_sensitive(
		    w,
		    (dev_ptr->command_eject != NULL) ?
			(*dev_ptr->command_eject != '\0') : FALSE
		);
	}

	/* Redraw device icon and label. */
	EDVMountBarDeviceExposeCB(mb->dev_da, NULL, mb);

	/* Update stats label. */
	EDVMountBarSetStats(mb);
}

/*
 *	Maps the mount bar.
 */
void EDVMountBarMap(edv_mountbar_struct *mb)
{
	GtkWidget *w;

        if(mb == NULL)
            return;

	if(!mb->map_state)
	{
	    w = mb->toplevel;
	    if(w != NULL)
		gtk_widget_show(w);

	    mb->map_state = TRUE;
	}
}

/*
 *	Unmaps the mount bar.
 */
void EDVMountBarUnmap(edv_mountbar_struct *mb)
{
        GtkWidget *w;

        if(mb == NULL)
            return;

        if(mb->map_state)
        {
            w = mb->toplevel;
            if(w != NULL)
                gtk_widget_hide(w);

            mb->map_state = FALSE;
        }
}

/*
 *	Deallocates all resources on the given mount bar structure and
 *	deallocates the mount bar structure itself.
 */
void EDVMountBarDelete(edv_mountbar_struct *mb)
{
        GtkWidget **w;


        if(mb == NULL)
            return;

        if(mb->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
	    w = &mb->dev_da;
            DO_DESTROY_WIDGET
	    w = &mb->map_btn;
            DO_DESTROY_WIDGET
	    w = &mb->mount_btn;
            DO_DESTROY_WIDGET
	    w = &mb->eject_btn;
            DO_DESTROY_WIDGET
	    w = &mb->goto_btn;
            DO_DESTROY_WIDGET
            w = &mb->refresh_stats_btn;
            DO_DESTROY_WIDGET
	    w = &mb->capacity_pbar;
	    DO_DESTROY_WIDGET
            w = &mb->stats_label;
            DO_DESTROY_WIDGET

	    w = &mb->toplevel;
	    DO_DESTROY_WIDGET

	    if(mb->dev_pixmap != NULL)
	    {
		gdk_pixmap_unref(mb->dev_pixmap);
		mb->dev_pixmap = NULL;
	    }
            if(mb->dev_mask != NULL)
            {
                gdk_bitmap_unref(mb->dev_mask);
                mb->dev_mask = NULL;
            }

	    if(mb->gc != NULL)
	    {
		gdk_gc_unref(mb->gc);
		mb->gc = NULL;
	    }

	    g_free(mb->dev_text);
	    mb->dev_text = NULL;

#undef DO_DESTROY_WIDGET
	}

        /* Deallocate mount bar structure itself. */
        g_free(mb);
}
